/*
    dft.c - Funktionen fuer die diskrete Fourier-Transformation
*/

# include <stdio.h>
# include <math.h>

# define MAX_N 10000	/* maximale Anzahl von Stuetzstellen */
# define REL_TOL 1e-6	/* Aufloesung fuer Re und Im bei Berechnung der Phase */

static double Cos[MAX_N], Sin[MAX_N];
static int N = 0;


	/*---------- Hilfsfunktionen ----------*/

int dft_init(int n)			/* cos() und sin() Feld anlegen */
 {
  register int i;
  register double freq;

  if (n > MAX_N)
   {
    fprintf(stderr, "dft_init() failed: Too many samples (>%d).\n", MAX_N);
    return(1);
   }

  freq = 2*M_PI/n;
  for (i=0; i<n; i++)
   {
    Cos[i] = cos(freq*i);
    Sin[i] = sin(freq*i);
   }
  return(0);
 }

double get_magnitude(double re, double im)	/* Amplitude berechnen */
 {
  return(sqrt(re*re+im*im));
 }

double get_phase(double re, double im)		/* Phase berechnen */
 {
  if (fabs(im) < REL_TOL)
    return((re < 0)?M_PI:0);
  if (fabs(re) < REL_TOL)
    return(((im < 0)?-M_PI:M_PI)/2.0);
  return(atan2(im, re));
 }

double limit_deg(double phase)
 {
  if (phase > 180)
    return(phase - 360);
  else if (phase <= -180)
    return(phase + 360);
  else
    return(phase);
 }

double distortion(double magnitude[], int n, int f0)  /* f0 ist INDEX der Grundschwingung! */
 {
  int i;
  double sum, mag_f0;

  if (f0 < 1)
    return(0.0);

  mag_f0 = magnitude[f0];
  sum = 0.0;
  for (i=2; i*f0<n; i++)
    if (magnitude[i*f0] > mag_f0/1e3)
      sum += magnitude[i*f0]*magnitude[i*f0];

  return(sqrt(sum/(mag_f0*mag_f0+sum)));
 }

void calc_bode(double mag[], double pha[], double gain[], double deg[], int n)
 {
  int i;

  for (i=0; i<n; i++)
   {
    gain[i] = 20*log10(mag[i]);
    deg[i] = 180.0/M_PI*pha[i];
   }
  return;
 }

void calc_gain(double u0_mag[], double u0_pha[], double u1_mag[], double u1_pha[], double gain[], double deg[], int n)
 {
  int i;

  for (i=0; i<n; i++)
   {
    gain[i] = 20*log10(u1_mag[i]/u0_mag[i]);
    deg[i] = limit_deg(180.0/M_PI*(u1_pha[i]-u0_pha[i]));
   }
  return;
 }

void calc_Z(double u0_mag[], double u0_pha[], double u1_mag[], double u1_pha[], double Z_mag[], double Z_pha[], int n, double R0)
 {
  int i;
  double u0_re, u0_im, u1_re, u1_im, u_diff_mag, u_diff_pha;

  for (i=0; i<n; i++)
   {
    u0_re = u0_mag[i]*cos(u0_pha[i]);
    u0_im = u0_mag[i]*sin(u0_pha[i]);
    u1_re = u1_mag[i]*cos(u1_pha[i]);
    u1_im = u1_mag[i]*sin(u1_pha[i]);
    u_diff_mag = get_magnitude(u0_re-u1_re, u0_im-u1_im);
    u_diff_pha = get_phase(u0_re-u1_re, u0_im-u1_im);
    Z_mag[i] = R0*(u1_mag[i]/u_diff_mag);
    Z_pha[i] = limit_deg(180.0/M_PI*(u1_pha[i]-u_diff_pha));
   }
  return;
 }

	/*---------- diskrete Fourier-Transformation ----------*/

int dft(double sequence[], double mag[], double pha[], int n)
 {
  register int i, j, k;
  int n2;
  double sum_c, sum_s, *data;

  if (n < 2)
   {
    fprintf(stderr, "dft(): Bad 'N' (%d).\n", n);
    return(0);
   }

  if (n != N)
   {
    if (dft_init(n))
      return(1);
    else
      N = n;
   }

  n2 = (n+1)/2;			/* Anzahl der Frequenz-Stuetzstellen */

  sum_c = 0.0;			/* Berechnung fuer f=0 */
  data = sequence;
  for(j=0; j<n; j++)
    sum_c += *data++;
  mag[0] = fabs(sum_c)/n;
  if (sum_c >= 0)
    pha[0] = 0.0;
  else
    pha[0] = M_PI;

  for(i=1; i<n2; i++)		/* Berechnung fuer 0 < f < fs/2 */
   {
    sum_c = sum_s = 0.0;
    k = 0;
    data = sequence;
    for (j=0; j<n; j++)
     {
      sum_c += (*data)*Cos[k];
      sum_s -= (*data++)*Sin[k];
      if ((k+=i) >= n)
        k -= n;
     }
    mag[i] = 2.0*get_magnitude(sum_c, sum_s)/n;
    pha[i] = get_phase(sum_c, sum_s);
   }

  return(n2);
 }
