/*
    audio.c - Funktionen zur Ansteuerung der Soundkarte

    linker Kanal: Chirp-Signal
    rechter Kanal: Sync-Signal
*/

# include <stdio.h>
# include <errno.h>
# include <stdlib.h>
# include <math.h>
# include <unistd.h>
# include <fcntl.h>
# include <sys/wait.h>
# include <sys/ioctl.h>
# include <linux/soundcard.h>
# include "chirp.h"

# define DSP_DEVICE "/dev/dsp"
# define MIXER_DEVICE "/dev/mixer"

# define MASTER_VOLUME 80

# define CHIRP_CYCLES 6
# define SEQUENCE_LENGTH (4*CHIRP_LENGTH*CHIRP_CYCLES)
# define RECORD_LENGTH (4*CHIRP_LENGTH*(CHIRP_CYCLES+4))

static char record_data[RECORD_LENGTH], testsequence[SEQUENCE_LENGTH],
            buffer[256], zero_data[4*CHIRP_LENGTH];


/*--------------------------------------------------------------*
 |	error = audio_open(&snd_in, &snd_out, &mixer)		|
 |	Devices oeffnen.					|
 *--------------------------------------------------------------*/
int audio_open(int *snd_in, int *snd_out, int *mixer)
 {
  int error;

  *snd_in = *mixer = -1;	/* fuer den Fehlerfall */

  if ((*snd_out = open(DSP_DEVICE, O_WRONLY)) == -1)
   {
    error = errno;
    perror("Can't open DSP device for output");
    return(error);
   }

  if ((*snd_in = open(DSP_DEVICE, O_RDONLY)) == -1)
   {
    error = errno;
    perror("Can't open DSP device for input");
    return(error);
   }

  if ((*mixer = open(MIXER_DEVICE, O_RDONLY)) == -1)
   {
    error = errno;
    perror("Can't open mixer device");
    return(error);
   }

  return(0);
 }


/*--------------------------------------------------------------*
 |	audio_close(&snd_in, &snd_out, &mixer)			|
 |	Devices schliessen.					|
 *--------------------------------------------------------------*/
void audio_close(int *snd_in, int *snd_out, int *mixer)
 {
  if (*snd_in > -1)
   {
    if (*snd_in != *snd_out)
      close(*snd_in);
    *snd_in = -1;
   }
  if (*snd_out > -1)
   {
    close(*snd_out);
    *snd_out = -1;
   }
  if (*mixer > -1)
   {
    close(*mixer);
    *mixer = -1;
   }
  return;
 }


/*--------------------------------------------------------------*
 |	error = audio_init(snd_in, snd_out, mixer)		|
 |	Sound-Karte initialisieren.				|
 |	Parameter: File-handle von PCM in, PCM out und Mixer	|
 *--------------------------------------------------------------*/
int audio_init(int snd_in, int snd_out, int mixer)
 {
  int value, error;

	/*---- Datenformat einstellen ----*/

  value = AFMT_S16_LE;
  if ((ioctl(snd_out, SNDCTL_DSP_SETFMT, &value) == -1) || (value != AFMT_S16_LE))
   {
    error = errno;
    perror("Can't set audio format");
    return(error);
   }
  value = AFMT_S16_LE;
  if ((ioctl(snd_in, SNDCTL_DSP_SETFMT, &value) == -1) || (value != AFMT_S16_LE))
   {
    error = errno;
    perror("Can't set audio format");
    return(error);
   }

	/*---- Anzahl Kanaele einstellen ----*/

  value = 1;
  if (ioctl(snd_out, SNDCTL_DSP_STEREO, &value) == -1)
   {
    error = errno;
    perror("Can't set to stereo");
    return(error);
   }
  value = 1;
  if (ioctl(snd_in, SNDCTL_DSP_STEREO, &value) == -1)
   {
    error = errno;
    perror("Can't set to stereo");
    return(error);
   }
  value = 2;
  if (ioctl(snd_out, SNDCTL_DSP_CHANNELS, &value) == -1)
   {
    error = errno;
    perror("Can't set to stereo");
    return(error);
   }
  value = 2;
  if (ioctl(snd_in, SNDCTL_DSP_CHANNELS, &value) == -1)
   {
    error = errno;
    perror("Can't set to stereo");
    return(error);
   }

	/*---- Sampling-Rate einstellen ----*/

  value = SAMPLE_FREQ;
  if ((ioctl(snd_out, SNDCTL_DSP_SPEED, &value) == -1) || (value != SAMPLE_FREQ))
   {
    error = errno;
    perror("Can't set sample frequency");
    return(error);
   }
  value = SAMPLE_FREQ;
  if ((ioctl(snd_in, SNDCTL_DSP_SPEED, &value) == -1) || (value != SAMPLE_FREQ))
   {
    error = errno;
    perror("Can't set sample frequency");
    return(error);
   }

	/*---- Aufnahmequelle einstellen ----*/

  value = SOUND_MASK_LINE;
  if (ioctl(mixer, MIXER_WRITE(SOUND_MIXER_RECSRC), &value) == -1)
   {
    error = errno;
    perror("Can't set recording source");
    return(error);
   }

	/*---- Lautstaerken einstellen ----*/

  value = MASTER_VOLUME + (MASTER_VOLUME << 8);
  if (ioctl(mixer, MIXER_WRITE(SOUND_MIXER_VOLUME), &value) == -1)
   {
    error = errno;
    perror("Can't set master volume");
    return(error);
   }

  value = 0;
  if (ioctl(mixer, MIXER_WRITE(SOUND_MIXER_MIC), &value) == -1)
    perror("Can't set microphone volume");

  value = 0;
  if (ioctl(mixer, MIXER_WRITE(SOUND_MIXER_LINE), &value) == -1)
    perror("Can't set line volume");

	/*---- Daten lesen und schreiben ----*/

  write(snd_out, "\0\0\0\0\0\0\0\0", 8);
  do					/* Einschaltknacken abwarten */
    read(snd_in, buffer, 256);
  while ((buffer[1] > 10) || (buffer[1] < -10));

  return(0);
 }


/*--------------------------------------------------------------*
 |	error = set_out_volume(mixer, left, right)		|
 |	Lautstaerke fuer Wiedergabe einstellen.			|
 |	Parameter: File-handle von Mixer, Lautst. links, rechts	|
 |	Wertebereich: 0..100					|
 *--------------------------------------------------------------*/
int set_out_volume(int mixer, int left, int right)
 {
  int volume, error;

  volume = left + (right << 8);
  if (ioctl(mixer, MIXER_WRITE(SOUND_MIXER_PCM), &volume) == -1)
   {
    error = errno;
    perror("Can't set PCM volume");
    return(error);
   }

  return(0);
 }


/*--------------------------------------------------------------*
 |	error = set_in_volume(mixer, left, right)		|
 |	Lautstaerke fuer Aufnahme einstellen.			|
 |	Parameter: File-handle von Mixer, Lautst. links, rechts	|
 |	Wertebereich: 0..100					|
 *--------------------------------------------------------------*/
int set_in_volume(int mixer, int left, int right)
 {
  int volume, error;

  volume = left + (right << 8);
  if (ioctl(mixer, MIXER_WRITE(SOUND_MIXER_IGAIN), &volume) == -1)
   {
    error = errno;
    perror("Can't set input gain");
    return(error);
   }

  return(0);
 }


/*--------------------------------------------------------------*
 |	next_sample = write_sample(value, &sample)		|
 |	Amplitudenwert (double) in S16_LE-Sample wandeln.	|
 *--------------------------------------------------------------*/
char *write_sample(double value, char *buffer)
 {
  int sample;
  short *short_ptr;

  sample = rint(value*32767);
  short_ptr = (short *)buffer;
  *short_ptr++ = sample;
  return((char *)short_ptr);
 }


/*--------------------------------------------------------------*
 |	value = sample2double(buffer, pos)			|
 |	S16_LE-Sample in double umwandeln.			|
 *--------------------------------------------------------------*/
double sample2double(char *buffer, int pos)
 {
  short *short_ptr;
  double value;

  buffer = &(buffer[pos]);
  short_ptr = (short *)buffer;
  value = *short_ptr;
  return(value/32767.0);
 }


/*--------------------------------------------------------------*
 |	create_testsequence()					|
 |	Berechnung des Testsignals inkl. Sync-Puls.		|
 *--------------------------------------------------------------*/
void create_testsequence(void)
 {
  int i;
  char *buffer;

  buffer = testsequence;
  for (i=0; i<SEQUENCE_LENGTH/4; i++)
   {
    buffer = write_sample(chirp[i % CHIRP_LENGTH], buffer);
    if (i < 12)
      buffer = write_sample(sin(M_PI*i/4), buffer);
    else
      buffer = write_sample(0.0, buffer);
   }

  for (i=0; i<CHIRP_LENGTH*4; i++)
    zero_data[i] = 0;

  return;
 }


/*--------------------------------------------------------------*
 |	create_sine(freq)					|
 |	Berechnung des Testsignals inkl. Sync-Puls.		|
 *--------------------------------------------------------------*/
void create_sine(double freq)
 {
  int i;
  char *buffer;

  buffer = testsequence;
  for (i=0; i<SEQUENCE_LENGTH/4; i++)
   {
    buffer = write_sample(sin(2*M_PI/SAMPLE_FREQ*freq*i), buffer);
    if (i < 12)
      buffer = write_sample(sin(M_PI*i/4), buffer);
    else
      buffer = write_sample(0.0, buffer);
   }
  return;
 }


/*--------------------------------------------------------------*
 |	play_testsequence(snd_out)				|
 |	Testsignal ausgeben					|
 *--------------------------------------------------------------*/
void play_testsequence(int snd_out)
 {
  if (write(snd_out, zero_data, 4*CHIRP_LENGTH) != 4*CHIRP_LENGTH)
    perror("play_testsequence(): write() failed");
  if (write(snd_out, testsequence, SEQUENCE_LENGTH) != SEQUENCE_LENGTH)
    perror("play_testsequence(): write() failed");
  if (write(snd_out, zero_data, 4*CHIRP_LENGTH) != 4*CHIRP_LENGTH)
    perror("play_testsequence(): write() failed");
  return;
 }


/*--------------------------------------------------------------*
 |	pos = find_startpos(buffer, max_pos)			|
 |	Sync-Signal in Aufnahme suchen.				|
 |	kann das Sync-Signal nicht gefunden werden, ist pos=-1	|
 *--------------------------------------------------------------*/
int find_startpos(char *buffer, int max_pos)
 {
  int i;
  double y, peak;

  i = 2;				/* linken Kanal ueberspringen */

  while(sample2double(buffer, i) < 0.05)	/* 1. pos. Halbwelle suchen */
   {
    if (i >= max_pos)
     {
      fprintf(stderr, "find_startpos(): Can't find sync.\n");
      return(-1);
     }
    i += 4;
   }

  peak = 0.0;
  while((y = sample2double(buffer, i)) > 0.0) /* 1. Nulldurchgang suchen */
   {
    if (y > peak)
      peak = y;
    if (i >= max_pos)
     {
      fprintf(stderr, "find_startpos(): Can't find sync.\n");
      return(-1);
     }
    i += 4;
   }

  while(sample2double(buffer, i) < -peak/10)	/* 2. Nulldurchgang suchen */
   {
    if (i >= max_pos)
     {
      fprintf(stderr, "find_startpos(): Can't find sync.\n");
      return(-1);
     }
    i += 4;
   }
  return(i-8*4-2);	/* Position des linken Kanals (Chirp) */
 }			/* Sync-Laenge=8, 4 Bytes pro Stereo-Sample */


/*--------------------------------------------------------------*
 |	length = record_sequence(snd_in, buffer, buf_size)	|
 |	Signal aufzeichnen, Sync suchen, Mittelwerte bilden	|
 |	und als double-Werte in buffer schreiben.		|
 *--------------------------------------------------------------*/
int record_sequence(int snd_in, double buffer[], int buf_size)
 {
  int pos, i, k, n;

  if (read(snd_in, record_data, RECORD_LENGTH) != RECORD_LENGTH)
   {
    perror("record_sequence(): read()-error");
    return(-1);
   }

  pos = find_startpos(record_data, RECORD_LENGTH-SEQUENCE_LENGTH);
  if (pos < 0)
    return(pos);

  pos += 8*CHIRP_LENGTH;	/* ersten zwei Zyklen zum Einschwingen */
  n = CHIRP_LENGTH;
  if (n > buf_size)
    n = buf_size;
  for (i=0; i<n; i++)
   {
    buffer[i] = 0;
    for (k=0; k<CHIRP_CYCLES-2; k++)
      buffer[i] += sample2double(record_data, pos+CHIRP_LENGTH*4*k+4*i);
    buffer[i] *= 1.0/(CHIRP_CYCLES-2);
   }
  return(CHIRP_LENGTH);
 }


/*--------------------------------------------------------------*
 |	length = acquire(snd_in, snd_out, buffer, buf_size)	|
 |	Messung durchfuehren und Daten (Zeitbereich) in buffer	|
 |	ablegen.						|
 *--------------------------------------------------------------*/
int acquire(int snd_in, int snd_out, double buffer[], int buf_size)
 {
  int length;

  read(snd_in, record_data, RECORD_LENGTH);	/* Aufnahmepuffer leeren */

  switch(fork())
   {
    case -1:
      perror("acquire(): fork() failed");
      return(-1);
    case 0:
//      usleep(1000);
      play_testsequence(snd_out);
      exit(0);
    default:
      usleep(10000);
      length = record_sequence(snd_in, buffer, buf_size);
      wait(NULL);
   }
  return(length);
 }


/*--------------------------------------------------------------*
 |	write_record_data(out_stream)				|
 |	Sample-Werte als ASCII-Datei schreiben.			|
 *--------------------------------------------------------------*/
void write_record_data(FILE *stream)
 {
  int i;
  short *data;

  data = (short *)record_data;
  fprintf(stream, "# time     left    right\n");
  for (i=0; i<RECORD_LENGTH/4; i++)
    fprintf(stream, "%.6f  %7.4f  %7.4f\n", 1.0/SAMPLE_FREQ*i,
            sample2double(record_data, 4*i), sample2double(record_data, 4*i+2));
  return;
 }
