namespace eval tclweather {
    variable LEGAL_FONTS "courier helvetica times"
    variable LEGAL_SIZES "5 7 8 10 12 14 16 18 20 24"
    variable OBS_NAMES {{pres "air pressure"} {cloud "cloud coverage"} \
            {dew "dew point"} {time "observation time"} \
            {relhum "relative humidity"} {cond "special conditions"} \
            {temp "temperature"} {vis "visibility"} \
            {wind "wind"}}
    variable temp_tw
    variable button_result
}

proc tclweather::configure {} {
    global tw Tree
    global tcl_platform
    variable temp_tw
    variable button_result
    
    if {[winfo exists .twc]} {
        # configuration window already open
        raise .twc
        return
    }
    toplevel .twc -class Dialog
    wm title .twc "TclWeather Configuration"
    wm group .twc $tw(p)

    # Tk on Unix uses a larger font than Windows, so if TclWeather is
    # running under Unix, make the configuration dialog box larger
    set config_height 320
    if {[string equal $tcl_platform(platform) "unix"]} {
        incr config_height 30
    }
    Notebook:create .twc.n \
            -pages {Display Color/Browser Format Network Location Misc} \
            -width 370 -height $config_height

    set temp_tw(toplevel) $tw(toplevel)
    create_display_page [Notebook:frame .twc.n "Display"]
    create_colors_page [Notebook:frame .twc.n "Color/Browser"]
    create_format_page [Notebook:frame .twc.n "Format"]
    create_network_page [Notebook:frame .twc.n "Network"]
    create_location_page [Notebook:frame .twc.n "Location"]
    create_misc_page [Notebook:frame .twc.n "Misc"]
    
    frame .twc.b
    button .twc.b.ok  -text " Ok " \
            -command {set tclweather::button_result "ok"}
    button .twc.b.apply -text " Apply " \
            -command {set tclweather::button_result "apply"}
    button .twc.b.can -text " Cancel " \
            -command {set tclweather::button_result "cancel"}
    pack .twc.b.ok .twc.b.apply .twc.b.can -side left -padx 10 -pady 5
    pack .twc.b -side bottom

    pack .twc.n -side top -fill both -expand 1

    set button_result "cancel"
    raise .twc
    wm protocol .twc WM_DELETE_WINDOW {set tclweather::button_result "cancel"}
    bind .twc <Escape> {set tclweather::button_result "cancel"}
    update
    set done 0
    while {!$done} {
        vwait tclweather::button_result
        if {[string equal $button_result "ok"] || \
                [string equal $button_result "apply"]} {
            ok_display_page
            ok_colors_page
            if {[ok_format_page]} {
                format_weather_data
            }
            if {[ok_network_page] || [ok_location_page]} {
                main_weather_loop restart
            }
            ok_misc_page
            update_weather_ticker_format
        }
        if {[string equal $button_result "ok"] || \
                [string equal $button_result "cancel"]} {
            set done 1
        }
    }

    ;# remove all traces associated with the temp variable
    foreach key [array names temp_tw] {
        set commands [trace vinfo tclweather::temp_tw($key)]
        foreach command $commands {
            trace vdelete tclweather::temp_tw($key) w [lindex $command 1]
        }
    }
    trace vdelete tw(titlebar) w tclweather::update_titlebar
    set tree_widget [Notebook:frame .twc.n "Location"].f.t
    trace vdelete Tree($tree_widget:selection) w \
            "tclweather::update_tree_selection $tree_widget"
    Tree:closeall $tree_widget
    destroy .twc
}

proc tclweather::create_display_page {page} {
    global tw
    variable temp_tw
    variable LEGAL_SIZES
    variable LEGAL_FONTS

    foreach key { \
            ticker_speed scroll_quality ticker_font ticker_size \
            scroll_left titlebar scroll_buttons enter_raise leave_lower} {
        set temp_tw($key) $tw($key)
    }
    
    label $page.l -text "Display Settings"
    pack $page.l -side top -anchor n

    scale $page.s -label "ticker scroll speed: " -from 10 -to 400 \
            -orient horizontal -variable tclweather::temp_tw(ticker_speed)
    scale $page.q -label "ticker animation quality: (smaller is smoother but moves slower)" -from 1 -to 20 \
            -orient horizontal -variable tclweather::temp_tw(scroll_quality)
    pack $page.s $page.q -side top -anchor w -fill x
    
    frame $page.f
    label $page.f.fl -text "ticker font:"
    ;# the following is a hack to force the option menu to be a certain size
    frame $page.f.f
    eval tk_optionMenu $page.f.f.fm tclweather::temp_tw(ticker_font) \
            $LEGAL_FONTS
    set temp_tw(ticker_font) "helvetica"
    pack $page.f.f.fm -anchor w
    update idletasks
    $page.f.f configure -width [winfo reqwidth $page.f.f.fm] \
            -height [winfo reqheight $page.f.f.fm]
    pack propagate $page.f.f 0
    set temp_tw(ticker_font) $tw(ticker_font)
    label $page.f.sl -text "size:"
    eval tk_optionMenu $page.f.sm tclweather::temp_tw(ticker_size) $LEGAL_SIZES
    pack $page.f.fl $page.f.f $page.f.sl $page.f.sm -side left -padx 4 -expand 1
    pack $page.f -side top -anchor w

    checkbutton $page.sl -text "scroll towards left edge" \
            -variable tclweather::temp_tw(scroll_left)
    checkbutton $page.sb -text "show scroll buttons" \
            -variable tclweather::temp_tw(scroll_buttons)
    pack $page.sl $page.sb -side top -anchor w
    if {$temp_tw(toplevel)} {
        checkbutton $page.t -text "show titlebar (hint: toggle with <spacebar>)" \
                -variable tclweather::temp_tw(titlebar)
        checkbutton $page.er -text "point at ticker to raise window" \
                -variable tclweather::temp_tw(enter_raise)
        checkbutton $page.ll -text "point away from ticker to lower window" \
            -variable tclweather::temp_tw(leave_lower)
        pack $page.t $page.er $page.ll -side top -anchor w
    }
    
    trace variable tw(titlebar) w tclweather::update_titlebar
}

proc tclweather::update_titlebar {name1 name2 op} {
    global tw
    variable temp_tw
    set temp_tw(titlebar) $tw(titlebar)
}

proc tclweather::ok_display_page {} {
    global tw
    variable temp_tw

    foreach key { \
            ticker_speed scroll_quality ticker_font ticker_size \
            scroll_left titlebar scroll_buttons enter_raise leave_lower} {
        set tw($key) $temp_tw($key)
    }
}

proc tclweather::create_colors_page {page} {
    global tw
    variable temp_tw
    
    foreach key { \
            cloud_tag_color cond_tag_color dew_tag_color pres_tag_color \
            relhum_tag_color temp_tag_color time_tag_color vis_tag_color \
            wind_tag_color bg_color button_fg_color button_bg_color \
            server_color misc_color browser_type browser} {
        set temp_tw($key) $tw($key)
    }
    
    label $page.l -text "Color Settings"
    pack $page.l -side top -anchor n
    
    set i 0
    frame $page.c
    foreach {line} { \
        {"air pressure:" pres_tag_color "cloud coverage:" cloud_tag_color } \
        {"dew point:" dew_tag_color "observation time:" time_tag_color} \
        {"relative humidity:" relhum_tag_color "special conditions:" cond_tag_color } \
        {"temperature:" temp_tag_color "visibility:" vis_tag_color} \
        {"wind:" wind_tag_color "background:" bg_color} \
        {"button foreground:" button_fg_color "button background:" button_bg_color} \
        {"server messages:" server_color "other texts:" misc_color}} {
        for {set j 0} {$j < 4} {incr j 2} {
            label $page.c.${i}${j}l -text [lindex $line $j]
            set tag_name [lindex $line [expr $j + 1]]
            set button_name $page.c.${i}${j}b
            button $button_name -background $temp_tw($tag_name) \
                    -command "tclweather::pick_color_button $button_name $tag_name" \
                    -relief ridge -text "        "
            trace variable tclweather::temp_tw($tag_name) w \
                    "tclweather::update_color_button $button_name $tag_name"
        }
        grid $page.c.${i}0l -row $i -column 0 -sticky e
        grid $page.c.${i}0b -row $i -column 1 -sticky w
        grid $page.c.${i}2l -row $i -column 2 -sticky e
        grid $page.c.${i}2b -row $i -column 3 -sticky w
        incr i
    }
    grid columnconfigure $page.c 1 -weight 1
    grid columnconfigure $page.c 3 -weight 1
    grid columnconfigure $page.c 0 -weight 3
    grid columnconfigure $page.c 2 -weight 3
    pack $page.c -side top -anchor n -expand yes -fill x -pady 4

    label $page.bl -text "External Browser Settings"
    pack $page.bl -side top -anchor n
    frame $page.b
    radiobutton $page.b.n -text "no browser (disables left-clicks)" \
            -variable tclweather::temp_tw(browser_type) -value "" \
            -command "tclweather::set_named_state $page.b.b disabled"
    radiobutton $page.b.w -text "use windows default" \
            -variable tclweather::temp_tw(browser_type) -value "win32" \
            -command "tclweather::set_named_state $page.b.b disabled"
    frame $page.b.b
    radiobutton $page.b.b.b -text "command:" \
            -variable tclweather::temp_tw(browser_type) -value "named" \
            -command "tclweather::set_named_state $page.b.b normal"
    entry $page.b.b.fn -textvariable tclweather::temp_tw(browser) -width 30
    button $page.b.b.fb -text "..." -command {
        set new_name [tk_getOpenFile -initialfile $tclweather::temp_tw(browser) \
                -initialdir [file rootname $tclweather::temp_tw(browser)] \
                -parent .twc -title "External Browser"]
        if {![string equal $new_name ""]} {
            set tclweather::temp_tw(browser) $new_name
        }
    }
    pack $page.b.b.b -side left
    pack $page.b.b.fn $page.b.b.fb -side left -anchor w -padx 3
    global tcl_platform
    if {![string equal $tcl_platform(platform) "windows"]} {
        $page.b.w configure -state disabled
    }
    if {![string equal $temp_tw(browser_type) "named"]} {
        set_named_state $page.b.b disabled
    }
    pack $page.b.n $page.b.w $page.b.b -side top -anchor w
    pack $page.b -side top -anchor n -expand yes -fill x
}

proc tclweather::pick_color_button {name tag_name} {
    variable temp_tw
    ;# don't allow more than one color palette at a time
    if {[winfo exists .twcolor]} {
        return
    }
    scan [string range $temp_tw($tag_name) 1 2] %x r
    scan [string range $temp_tw($tag_name) 3 4] %x g
    scan [string range $temp_tw($tag_name) 5 6] %x b
    set new_color \
            [pick_color [expr $r / 255.0] [expr $g / 255.0] [expr $b / 255.0]]
    if {![string equal $new_color ""]} {
        set temp_tw($tag_name) $new_color
    }
}

proc tclweather::update_color_button {name tag_name name1 name2 op} {
    variable temp_tw
    $name configure -background $temp_tw($tag_name)
}

proc tclweather::set_named_state {p state} {
    ${p}.fn configure -state $state
    ${p}.fb configure -state $state
}

proc tclweather::ok_colors_page {} {
    global tw
    variable temp_tw
    
    foreach key { \
            cloud_tag_color cond_tag_color dew_tag_color pres_tag_color \
            relhum_tag_color temp_tag_color time_tag_color vis_tag_color \
            wind_tag_color bg_color button_fg_color button_bg_color \
            server_color misc_color browser_type browser} {
        set tw($key) $temp_tw($key)
    }
}

proc tclweather::create_format_page {page} {
    global tw
    variable temp_tw
    variable OBS_NAMES

    foreach key { \
            report_order units calc_change show_24hr localtz} {
        set temp_tw($key) $tw($key)
    }
    
    label $page.l -text "Format Settings"
    pack $page.l -side top -anchor n

    frame $page.f
    frame $page.f.l
    label $page.f.l.l -text "observation types"
    pack $page.f.l.l -side top
    listbox $page.f.l.b -selectmode single -height 9 \
            -yscrollcommand "$page.f.l.s set"
    foreach item_choice $OBS_NAMES {
        $page.f.l.b insert end [lindex $item_choice 1]
    }
    bind $page.f.l.b <Double-Button-1> \
            "tclweather::left_box_event $page.f.l.b $page.f.r.b"
    pack $page.f.l.b -side left
    scrollbar $page.f.l.s -command "$page.f.l.b yview"
    pack $page.f.l.s -side right -fill y
    pack $page.f.l -side left

    frame $page.f.r
    label $page.f.r.l -text "displayed"
    pack $page.f.r.l -side top
    listbox $page.f.r.b -selectmode single -height 9 \
            -yscrollcommand "$page.f.r.s set"
    trace variable tclweather::temp_tw(report_order) w \
            "tclweather::update_right_box $page.f.r.b"
    update_right_box $page.f.r.b "" "" ""
    bind $page.f.r.b <Double-Button-1> \
            "tclweather::right_box_event $page.f.l.b $page.f.r.b"
    pack $page.f.r.b -side left
    scrollbar $page.f.r.s -command "$page.f.r.b yview"
    pack $page.f.r.s -side right -fill y
    pack $page.f.r -side right
    pack $page.f -side top -fill none -anchor n

    label $page.i -text "(double click left pane to add an observation to ticker;\ndouble click right pane to remove an entry)"
    pack $page.i -side top
    
    checkbutton $page.um -text "use metric system"\
            -variable tclweather::temp_tw(units)
    checkbutton $page.cc -text "show changes in values" \
            -variable tclweather::temp_tw(calc_change)
    checkbutton $page.ap -text "use 24-hour system" \
            -variable tclweather::temp_tw(show_24hr)
    checkbutton $page.ltz -text "show time relative to local time zone" \
            -variable tclweather::temp_tw(localtz)
    pack $page.um $page.cc $page.ap $page.ltz -side top -anchor w
}

proc tclweather::left_box_event {lb rb} {
    variable temp_tw
    variable OBS_NAMES
    set selection [$lb curselection]
    if {![string equal $selection ""]} {
        set item_selection [lindex $OBS_NAMES $selection]
        lappend temp_tw(report_order) [lindex $item_selection 0]
    }
}

proc tclweather::right_box_event {lb rb} {
    variable temp_tw
    set selection [$rb curselection]
    if {![string equal $selection ""]} {
        set temp_tw(report_order) \
                [lreplace $temp_tw(report_order) $selection $selection]
    }
}

proc tclweather::update_right_box {name name1 name2 op} {
    variable temp_tw
    variable OBS_NAMES
    $name delete 0 end
    foreach report $temp_tw(report_order) {
        set report_name "<unknown>"
        foreach item_choice $OBS_NAMES {
            if {[string equal $report [lindex $item_choice 0]]} {
                set report_name [lindex $item_choice 1]
                break
            }
        }
        $name insert end $report_name
    }
}

proc tclweather::ok_format_page {} {
    global tw
    variable temp_tw
    set reformat_needed 0
    foreach key { \
            report_order units calc_change show_24hr localtz} {
        incr reformat_needed [expr ! [string equal $tw($key) $temp_tw($key)]]
        set tw($key) $temp_tw($key)
    }
    return $reformat_needed
}

proc tclweather::create_network_page {page} {
    global tw
    variable temp_tw
    foreach key { \
            use_proxy \
            proxy_host proxy_port proxy_name proxy_password proxy_savepass \
            refresh notify_download notify_complete notify_error \
            print_obsdata} {
        set temp_tw($key) $tw($key)
    }

    label $page.l -text "Network Settings"
    pack $page.l -side top -anchor n

    checkbutton $page.up -text "use proxy server" \
            -variable tclweather::temp_tw(use_proxy)
    pack $page.up -side top -anchor w
    
    frame $page.p
    label $page.p.hl -text "http proxy host:"
    entry $page.p.he -width 20 -textvariable tclweather::temp_tw(proxy_host)
    label $page.p.pl -text " port:"
    entry $page.p.pe -width 4 -textvariable tclweather::temp_tw(proxy_port)
    pack $page.p.hl $page.p.he $page.p.pl $page.p.pe \
            -side left -padx 3 -pady 4
    
    frame $page.a
    label $page.a.ul -text "proxy user name:"
    entry $page.a.ue -width 8 -textvariable tclweather::temp_tw(proxy_name)
    label $page.a.pl -text " password:"
    entry $page.a.pe -width 8 \
            -textvariable tclweather::temp_tw(proxy_password) -show "*"
    pack $page.a.ul $page.a.ue $page.a.pl $page.a.pe \
            -side left -padx 3 -pady 4
    
    checkbutton $page.sp -text "save password (WARNING: only marginally encrypted)" \
            -variable tclweather::temp_tw(proxy_savepass)
    pack $page.p $page.a $page.sp -side top -anchor w

    set proxy_fields "$page.p.hl $page.p.he $page.p.pl $page.p.pe \
            $page.a.ul $page.a.ue $page.a.pl $page.a.pe $page.sp"
    trace variable tclweather::temp_tw(use_proxy) w \
            "tclweather::update_use_proxy [list $proxy_fields]"
    update_use_proxy $proxy_fields "" "" ""
    
    scale $page.r -label "data refresh (minutes):" -from 1 -to 90 \
            -orient horizontal -variable tclweather::temp_tw(refresh)
    pack $page.r -side top -anchor w -fill x

    checkbutton $page.nd -text "notify when download begins" \
            -variable tclweather::temp_tw(notify_download)
    checkbutton $page.nc -text "notify when download completes" \
            -variable tclweather::temp_tw(notify_complete)
    checkbutton $page.ne -text "notify upon download error" \
            -variable tclweather::temp_tw(notify_error)
    checkbutton $page.po -text "print downloaded METAR data to console" \
            -variable tclweather::temp_tw(print_obsdata)
    pack $page.nd $page.nc $page.ne $page.po -side top -anchor w
}

proc tclweather::update_use_proxy {proxy_fields name1 name2 op} {
    variable temp_tw
    foreach field $proxy_fields {
        if {$temp_tw(use_proxy)} {
            $field configure -state normal
        } else {
            $field configure -state disabled
        }
    }
}

proc tclweather::ok_network_page {} {
    global tw
    variable temp_tw
    set network_changed 0
    foreach key { \
            use_proxy proxy_host proxy_port proxy_name proxy_password \
            refresh} {
        if {![string equal $temp_tw($key) $tw($key)]} {
            set network_changed 1
            break
        }
    }
    set tw(proxy_host) [string trim $temp_tw(proxy_host)]
    scan $temp_tw(proxy_port) %d tw(proxy_port)
    
    foreach key { \
            use_proxy proxy_name proxy_password proxy_savepass \
            refresh notify_download notify_complete notify_error \
            print_obsdata} {
        set tw($key) $temp_tw($key)
    }
    return $network_changed
}

proc tclweather::create_location_page {page} {
    global tw locations Tree
    variable temp_tw

    foreach {key} { \
            location_filename location_name location_code} {
        set temp_tw($key) $tw($key)
    }
    
    label $page.l -text "Location Settings"
    pack $page.l -side top -anchor n

    frame $page.f
    if {[info exists Tree($page.f.t:reuse)] && $Tree($page.f.t:reuse)} {
        ;# reuse old tree -- to save time
        Tree:reuse $page.f.t -width 280 -height 180 \
                -yscrollcommand "$page.f.sb set" -relief sunken -borderwidth 2
    } else {
        ;# build a new tree
        Tree:create $page.f.t -width 280 -height 200 \
                -yscrollcommand "$page.f.sb set" -relief sunken -borderwidth 2
        Tree:setreuse $page.f.t 1
        foreach {key} [lsort -dictionary [array names locations]] {
            Tree:newitem $page.f.t $key
        }
    }
    Tree:setselection $page.f.t $temp_tw(location_name)
    Tree:openall $page.f.t $temp_tw(location_name)
    $page.f.t bind x <Button-1> {
        Tree:setselection %W [Tree:labelat %W %x %y]
    }
    $page.f.t bind x <Double-Button-1> {
        Tree:open %W [Tree:labelat %W %x %y]
    }
    pack $page.f.t -side left
    scrollbar $page.f.sb -orien vertical -command "$page.f.t yview"
    pack $page.f.sb -side right -fill y
    pack $page.f -side top -anchor n -pady 4

    set temp_tw(location_text) "current location:\n$temp_tw(location_name) ($temp_tw(location_code))"
    label $page.cl -textvariable tclweather::temp_tw(location_text) \
            -justify left
    pack $page.cl -side top -anchor n

    frame $page.lf
    label $page.lf.ll -text "database:"
    entry $page.lf.le -textvariable tclweather::temp_tw(location_filename) \
            -width 24
    button $page.lf.b -text "..." -command {
        set new_name [tk_getOpenFile \
                -initialdir [file rootname $tclweather::temp_tw(location_filename)] \
                -parent .twc -title "Location Database Filename"]
        if {![string equal $new_name ""]} {
            set tclweather::temp_tw(location_filename) $new_name
        }
    }
    button $page.lf.r -text "reload" \
            -command "tclweather::reload_database $page.f.t"
    pack $page.lf.ll $page.lf.le $page.lf.b $page.lf.r -side left -padx 3
    pack $page.lf -side top -anchor center -pady 4
    
    ;# set a trace on the selected item
    trace variable Tree($page.f.t:selection) w \
            "tclweather::update_tree_selection $page.f.t"
}

proc tclweather::update_tree_selection {tree_name name1 name2 op} {
    global locations Tree
    variable temp_tw
    set selected [Tree:getselection $tree_name]
    if {[info exists locations($selected)] && \
            [string length $locations($selected)] > 0} {
        set temp_tw(location_name) $selected
        set temp_tw(location_code) $locations($selected)
        set temp_tw(location_text) \
                "current location:\n$temp_tw(location_name) ($temp_tw(location_code))"
    }
}

proc tclweather::reload_database {tree} {
    global tw locations Tree
    variable temp_tw
    ;# reload the tree
    while {![load_location_cache $temp_tw(location_filename)]} {
        if {[string equal [tk_messageBox -title "TclWeather Startup"\
                -message "Invalid locations database $temp_tw(location_filename)." \
                -icon error -parent $tw(p) -type retrycancel] "cancel"]} {
            return
        }
    }
    trace vdelete Tree($tree:selection) w \
            "tclweather::update_tree_selection $tree"
    Tree:flush $tree
    foreach {key} [lsort -dictionary [array names locations]] {
        Tree:newitem $tree $key
    }
    Tree:setselection $tree $temp_tw(location_name)
    Tree:openall $tree $temp_tw(location_name)
    trace variable Tree($tree:selection) w \
            "tclweather::update_tree_selection $tree"
}

proc tclweather::ok_location_page {} {
    global tw
    variable temp_tw
    set location_changed \
            [expr ![string equal $temp_tw(location_name) $tw(location_name)]]
    foreach key { \
            location_filename location_name location_code} {
        set tw($key) $temp_tw($key)
    }
    return $location_changed
}

proc tclweather::create_misc_page {page} {
    global tw
    variable temp_tw

    foreach key { \
            filename save_on_exit save_geometry} {
        set temp_tw($key) $tw($key)
    }
    
    label $page.l -text "Miscellaneous Settings"
    pack $page.l -side top -anchor n

    frame $page.f
    label $page.f.fl -text "settings filename:"
    entry $page.f.fe -textvariable tclweather::temp_tw(filename) -width 24
    button $page.f.b -text "..." -command {
        set new_name [tk_getSaveFile \
                -initialdir [file rootname $tclweather::temp_tw(filename)] \
                -parent .twc -title "Settings Filename"]
        if {![string equal $new_name ""]} {
            set temp_tw(filename) $new_name
        }
    }
    pack $page.f.fl $page.f.fe $page.f.b -side left -padx 3
    pack $page.f -side top -anchor w -pady 4
    
    checkbutton $page.soe -text "save settings on exit" \
            -variable tclweather::temp_tw(save_on_exit)
    pack $page.soe -side top -anchor w
    if {$temp_tw(toplevel)} {
        checkbutton $page.swl -text "save window size and position" \
                -variable tclweather::temp_tw(save_geometry)
        pack $page.swl -side top -anchor w
    }

    frame $page.s
    button $page.s.ss -text "Save Settings Now" -command {
        set done 0
        while {!$done} {
            if {[tclweather::save_settings $tclweather::temp_tw(filename) tclweather::temp_tw]} {
                tk_messageBox -title "TclWeather" \
                        -message "Settings saved." \
                        -parent .twc -type "ok"
                set done 1
            } else {
                set answer [tk_messageBox -title "TclWeather" -icon error \
                        -message "Error while saving to $tclweather::temp_tw(filename)." \
                        -parent .twc -type "retrycancel"]
                if {[string equal $answer "cancel"]} {
                    set done 1
                }
            }
        }
    }
    button $page.s.ls -text "Load Settings" -command {
        set done 0
        while {!$done} {
            if {[load_settings $tclweather::temp_tw(filename) tclweather::temp_tw]} {
                tk_messageBox -title "TclWeather" \
                        -message "Settings have been adjusted." \
                        -parent .twc -type "ok"
                set done 1
            } else {
                set answer [tk_messageBox -title "TclWeather" -icon error \
                        -message "Error while reading from $tclweather::temp_tw(filename)." \
                        -parent .twc -type "retrycancel"]
                if {[string equal $answer "cancel"]} {
                    set done 1
                }
            }
        }
    }
    pack $page.s.ss -side left
    pack $page.s.ls -side left -padx 4
    
    button $page.d -text "Reset to Default Settings" -command {
        tclweather::set_defaults tclweather::temp_tw
        tk_messageBox -title "TclWeather" \
                -message "Settings have been reset." -parent .twc -type ok
    }
    button $page.r -text "Force Data Refresh Now" \
            -command {main_weather_loop restart}
    pack $page.s $page.d $page.r -side top -anchor w -padx 4 -pady 4

    frame $page.etc
    button $page.etc.a -text "About TclWeather" \
            -command {tclweather::show_about_dialog}
    button $page.etc.ex -text "Exit TclWeather" \
            -command {exit_tclweather} \
            -background red
    pack $page.etc.ex $page.etc.a -side right -padx 4
    pack $page.etc -side bottom -anchor e -pady 4
}

proc tclweather::show_about_dialog {} {
    global TCLWEATHER_VERSION
    tk_messageBox -title "About TclWeather" \
            -message "TclWeather $TCLWEATHER_VERSION by Jason Tang (tang@jtang.org).

Weather data from NOAA (noaa.gov) and
  Weather Underground (wunderground.com).

Left-click to open external browser.
Right-click to open configuration window
Spacebar to toggle titlebar.
Press flanking buttons to advance ticker.

Please read the README file." \
            -parent .twc -type ok
}

proc tclweather::ok_misc_page {} {
    global tw
    variable temp_tw
    foreach key { \
            filename save_on_exit save_geometry} {
        if {[string equal $key "filename"]} {
            set tw($key) [string trim $temp_tw(filename)]
        } else {
            set tw($key) $temp_tw($key)
        }
    }
}

proc tclweather::pick_color {{r ""} {g ""} {b ""}} {
    variable color_button_result
    
    toplevel .twcolor -class Dialog
    wm title .twcolor "Pick a Color"
    wm group .twcolor .twc
    colordial_create .twcolor.d
    pack .twcolor.d
    
    frame .twcolor.b
    button .twcolor.b.ok  -text " Ok " \
            -command {set tclweather::color_button_result "ok"}
    button .twcolor.b.can -text " Cancel " \
            -command {set tclweather::color_button_result "cancel"}
    pack .twcolor.b.ok .twcolor.b.can -side left -padx 10 -pady 5
    pack .twcolor.b -side top

    set color_button_result "cancel"
    raise .twcolor.b
    wm protocol .twcolor WM_DELETE_WINDOW\
            {set tclweather::color_button_result "cancel"}
    if {! [string equal $r ""]} {
        colordial_set_rgb .twcolor.d $r $g $b
    }
    update
    vwait tclweather::color_button_result

    set new_color ""
    if {[string equal $color_button_result "ok"]} {
        set new_color [colordial_get .twcolor.d]
    }

    destroy .twcolor
    return $new_color
}

proc tclweather::save_settings {filename {varname "tw"}} {
    upvar $varname var
    if {[catch {open $filename w} dest]} {
        return 0
    }
    foreach key [lsort -dictionary [array names var]] {
        if {[string equal $key "proxy_password"]} {
            ;# special case -- saving password
            if {$var(proxy_savepass)} {
                set password $var(proxy_password)
            } else {
                set password ""
            }
            puts $dest "proxy_password [base64_encode $password]"
        } elseif {[string equal $key "geometry"]} {
            ;# save the window geometry, if requested
            if {$var(save_geometry)} {
                puts $dest "geometry [winfo geometry .]"
            } else {
                puts $dest "geometry "
            }
        } elseif {![string equal $key "filename"]} {
            ;# also don't bother saving filename
            puts $dest "$key $var($key)"
        }
    }
    close $dest
    return 1
}

proc tclweather::load_settings {filename {varname "tw"}} {
    global TCLWEATHER_VERSION
    upvar $varname var
    
    ;# if no TCLWEATHER_VERSION is listed in the file, assume that it's from the
    ;# 0.90 release -- the last TCLWEATHER_VERSION of TclWeather that didn't
    ;# include TCLWEATHER_VERSION information in the settings file
    set var(TCLWEATHER_VERSION) 0.90
    
    if {[catch {open $filename r} src]} {
        return 0
    }
    while {[gets $src line] >= 0} {
        if {[regexp {^([^ ]+) (.*)} $line foo key value] == 0} {
            ;# invalid format
            return -1
        }
        set var($key) $value
    }
    close $src
    ;# compare the TCLWEATHER_VERSION information
    set file_TCLWEATHER_VERSION $var(TCLWEATHER_VERSION)
    if {$file_TCLWEATHER_VERSION > $TCLWEATHER_VERSION} {
        return -1
    } else {
        ;# upgrade the settings file to the current TCLWEATHER_VERSION
        set var(TCLWEATHER_VERSION) $TCLWEATHER_VERSION
    }
    ;# special case -- handle password decoding
    set fake_pw $var(proxy_password)
    if {[catch {base64_decode $fake_pw} real_pw]} {
        set real_pw ""
    }
    set var(proxy_password) $real_pw
    ;# file loaded successfully, so set the filename
    set var(filename) $filename
    return 1
}

proc tclweather::set_defaults {{varname "tw"}} {
    global TCLWEATHER_VERSION
    upvar $varname var

    ;# initialize display globals
    array set var {
        ticker_font "helvetica" ticker_size 12 ticker_speed 150
        scroll_quality 2 scroll_left 1 titlebar 1 scroll_buttons 1
        enter_raise 0 leave_lower 0
    }

    ;# initialize color globals
    array set var {
        bg_color "#000000"
        button_bg_color "#000000"
        button_fg_color "#ffffff"
        cloud_tag_color "#ffff80"
        cond_tag_color "#ff0000"
        dew_tag_color "#80ffff"
        misc_color "#ffffff"
        pres_tag_color "#ffa500"
        relhum_tag_color "#d2b48c"
        server_color "#808080"
        temp_tag_color "#00ff00"
        time_tag_color "#f8f8ff"
        vis_tag_color "#ee7ae9"
        wind_tag_color "#87cefa"
    }
    ;# clouds - yellow, conditions - red, dew point - aquamarine
    ;# pressure - orange, relative humidity - tan, temperature - green
    ;# time - ghost white, visibility - orchid2, wind - light sky blue
    ;# background - black, server - gray, misc - white

    ;# initialize format globals
    set var(report_order) \
            [list "time" "cond" "temp" "cloud" "wind" "pres" "dew" "relhum" "vis"]
    array set var {
        units 1 calc_change 1 localtz 0 show_24hr 1
    }

    ;# initialize network globals
    array set var {
        use_proxy 0
        proxy_host "" proxy_port "" proxy_name "" proxy_password ""
        proxy_savepass 0 refresh 60
        notify_download 1 notify_complete 1 notify_error 1
        print_obsdata 1
    }

    ;# intialize default location to be Baltimore, and other location defaults
    set var(location_name) "/United States/Maryland/Baltimore-Glen Burnie"
    set var(location_code) "KBWI"
    global env
    set var(location_filename) "[file join $env(TCLWEATHER_LIB) locations.cache]"

    ;# initialize other default values
    set var(filename) "[file join [pwd] tclweather.rc]"
    global tcl_platform
    if {[string equal $tcl_platform(platform) "windows"]} {
        set var(browser_type) "win32"
    } else {
        set var(browser_type) ""
    }
    set var(browser) ""    
    set var(save_on_exit) 1
    set var(save_geometry) 1
    set var(geometry) "400x50+20+20"
    set var(TCLWEATHER_VERSION) $TCLWEATHER_VERSION
}
