/****************************************************************************
 Telefonbuch, v0.1 (07042000)                                
 Copyright (C) 2000  Rafael Peregrino da Silva <rafael@bmt1.kf.TU-Berlin.DE>
 
 This program is free software. You may modify and/or redistribute the
 program as long as the conditions in the COPYING file (in the top level
 directory) are met.
 
 This program is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 or FITNESS FOR A PARTICULAR PURPOSE.
*****************************************************************************/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>

#include "callbacks.h"
#include "interface.h"
#include "support.h"

/* Aufzhlungstyp zur Behandlung von Warnungen (oder Fehlern) */

enum warnings {no_saved_data,
	       create_new_file,
	       load_new_file,
	       no_data_to_delete,
	       no_line_selected,
               no_data_entered,
	       no_data_to_sort,
               no_data_to_save} Warnings;


/* GLOBALE VARIABLEN (die wir auf jedem Kosten vermeiden sollten :-) */


/* Ein globaler Zeiger zu der clist des Hauptfensters (Main Window = mw)
 * und globale Variablen zu der Zeilennummer bzw. Spaltennummer. 
 */

static GtkCList *mw_clist = NULL;
static gint mw_clist_row = -1;
static gint mw_clist_column = -1;


/* Boole'sche Variablen, die anzeigen, ob das entsprechende Dialog-Fenster
 * bereits offen ist.
 */

static gboolean about_dialog_shown = FALSE;
static gboolean open_file_dialog_shown = FALSE;
static gboolean save_as_dialog_shown = FALSE;
static gboolean input_dialog_shown = FALSE;

/* Boole'sche Variabel zur Steuerung von einigen Interface-Aufgaben
 * (um anzuzeigen, ob die Daten neu sortiert wurden oder ob eine
 * bestimmte Message nicht eingeblendet werden soll)
 */

static gboolean data_resorted = FALSE;
static gboolean message_veto = TRUE;


/* Komplexe Dialog-Fenster knnen langsame/trgerische Erstellung aufweisen.
 * Aus diesem Grunde erstellen wir sie ein Mal und legen Zeiger in statischen
 * Variabeln zu ihnen, sodass wir sie wieder verwenden knnen.
 */

static GtkWidget *open_file_dialog = NULL;
static GtkWidget *save_file_dialog = NULL;
static GtkWidget *input_dialog = NULL;
static GtkWidget *warning_dialog = NULL;


/* Das ist der Name der Telefonbuchdatei, die aktuell bearbeitet wird */

static gchar *current_filename = NULL;


/* Dieses Flag wird als TRUE gesetzt, falls die Telefonbuchdatei neue
   Eintrge bekommt. Das wird verwendet, um ein Warnung-Dialog-Fenster
   ("Sind Sie sicher?" :-) zu erstellen.*/

static gboolean file_changed = FALSE;


/* Ein Schlssel zum Speichern von Zeigern zum Hauptfenster */

static const gchar *MainWindowKey = "MainWindowKey";


/* PROTOTYPEN VON FUNKTIONEN, DIE NICHT IN DEN OBENSTEHENDEN HEADER
 * DATEIEN ENTHALTEN SIND UND NICHT VON SIGNALEN GESTARTET WERDEN
 */

static void new_file (GtkWidget *main_window);
static void open_file (GtkWidget *main_window);
static void open_file_selection (GtkWidget *main_window);
static void real_open_file (GtkWidget *main_window,
			    gchar *filename);
static void save_as (GtkWidget *main_window);
static void real_save_file (GtkWidget *main_window,
			    gchar *filename);
static void add_data (GtkWidget *main_window);
static void delete_data (GtkWidget *main_window);
static void reset_env (GtkWidget *main_window);
static void set_window_title (GtkWidget *main_window);
static void quit_program (void);


/***************************************************************************
 *                     INITIALISIERUNG DES HAUPTFENSTERS                   *
 ***************************************************************************/

void
on_telefonbuch_show                    (GtkWidget       *widget,
                                        gpointer         user_data)
{  
  /* Die Erklrung zur untenstehenden Anweisung folgt weiter unten */
  reset_env((GtkWidget *) user_data);
}


/***************************************************************************
 *                           MENLEISTE-FUNKTIONEN                         *
 ***************************************************************************/

void
on_new_file_activate                   (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  /* Die Erklrung zur untenstehenden Anweisung folgt weiter unten */
  new_file((GtkWidget *) user_data);
}


void
on_open_file_activate                  (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget *main_window;

  /* In der nchsten Zeile befindet sich eine der berhaupt wichtigsten 
   * Support-Funktionen, die in der Datei "support.c" zu finden sind:
   *
   *       GtkWidget*  lookup_widget (GtkWidget   *widget,
   *                                  const gchar *widget_name)
   *
   * Mit dieser Funktion kann man ein beliebiges Widget in einem
   * Vater-Widget orten/finden.
   */
  main_window = lookup_widget (GTK_WIDGET (menuitem), "telefonbuch");
  open_file (main_window);
}


void
on_save_file_activate                  (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget *main_window;
  GtkLabel *warning_message_field;
  gchar *warning_message = "Sie kennen wohl\n nicht viele Leute,\noder? :-)";

  main_window = lookup_widget (GTK_WIDGET (menuitem), "telefonbuch");

  /* Warnung, falls die Telefonbuchliste leer ist */
  if((mw_clist == NULL) || (mw_clist->rows == 0))
    {
      Warnings = no_data_to_save;
      file_changed = FALSE;
      warning_dialog = create_warning_dialog ();
      gtk_widget_realize(warning_dialog);
      warning_message_field = GTK_LABEL(lookup_widget(GTK_WIDGET(warning_dialog),"warning_message_field"));
      gtk_label_set_text (warning_message_field,warning_message);
      gtk_widget_show (GTK_WIDGET(warning_message_field));
      gtk_widget_show (warning_dialog);
      gdk_beep();
    }
  else
    {
      /* Falls das aktuelle Dokument noch keinen Namen besitzt, rufen
       * wir die Funktion save_as() auf, um das Dialogfenster zum
       * Speichern von Dateien zu zeigen.
       */
      if (current_filename == NULL)
	save_as (main_window);
      else
	real_save_file (main_window,current_filename);
    }
}


void
on_save_as_activate                    (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget *main_window;

  main_window = lookup_widget (GTK_WIDGET (menuitem), "telefonbuch");
  save_as (main_window);
}


void
on_quit_activate                       (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  quit_program();
}


void
on_new_entry_activate                  (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  /* Hier folgt die angekndigte Erklrung (siehe die zwei ersten
   * Funktionen dieses Programmes oben):
   *
   * Das ist eine andere Variante, wie man den Namen eines Widget
   * einer Funktion bergeben kann. Die untenstehende Zeile hat die
   * selbe Aufgabe der funktion lookup_widget(...). Um die Variabel
   * user_data mit einem Widget zu verbinden, muss man allerdings
   * in Glade den Namen des zu verbindenden Widget in der 
   * Eigenschaftsfenster informieren. Bitte ffnen Sie die
   * Glade-Interface um diesen Vorgang besser zu verstehen. Dieser
   * Weg soll brigens bevorzugt werden, da die lookup_widget(...)-Funktion 
   * nicht mehr lang aufrecht erhalten werden soll...
   *
   * In diesem Fall ist die Adresse des Hauptfensters in user_data
   */
  add_data((GtkWidget *) user_data);
}


void
on_entry_delete_activate               (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  /* Beachten Sie den Unterschied zwischen der nchsten Zeile
   * und die entsprechende Zeile in der Funktion 
   *
   *    on_delete_data_button_clicked  (GtkButton       *button,
   *                                    gpointer         user_data)
   *
   * Sie erfllen die selbe Aufgabe, aber hier informiert man
   * ber Glade (genauer gesagt, ber den Glade-Eigenschaftsdialog,
   * Rubrik "Signale"), dass die Daten in user_data, die jenigen des
   * Hauptfensters (das den Namen "telefonbuch" in unserer Applikation
   * hat) sind.
   */
  delete_data((GtkWidget *) user_data);
}


void
on_about_activate                      (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget *about_dialog;

  if (about_dialog_shown)
    return;
  about_dialog_shown = TRUE;
  about_dialog = create_about_dialog ();
  gtk_widget_show (about_dialog);
}


/***************************************************************************
 *       FUNKTIONEN ZUR BEARBEITUNG DER BUTTONS DER KONTROLL-LEISTE        *
 ***************************************************************************/

void
on_new_file_button_clicked             (GtkButton       *button,
                                        gpointer         user_data)
{
  GtkWidget *main_window;

  main_window = lookup_widget (GTK_WIDGET (button), "telefonbuch");
  new_file(main_window);
}


void
on_file_open_button_clicked            (GtkButton       *button,
                                        gpointer         user_data)
{
  GtkWidget *main_window;

  main_window = lookup_widget (GTK_WIDGET (button), "telefonbuch");
  open_file (main_window);
}


void
on_file_save_button_clicked            (GtkButton       *button,
                                        gpointer         user_data)
{
  GtkWidget *main_window;
  GtkLabel *warning_message_field;
  gchar *warning_message = "Sie kennen wohl\n nicht viele Leute,\noder? :-)";

  main_window = lookup_widget (GTK_WIDGET (button), "telefonbuch");

  /* Warnung, falls die Telefonbuchliste leer ist */
  if((mw_clist == NULL) || (mw_clist->rows == 0))
    {
      Warnings = no_data_to_save;
      file_changed = FALSE;
      warning_dialog = create_warning_dialog ();
      gtk_widget_realize(warning_dialog);
      warning_message_field = GTK_LABEL(lookup_widget(GTK_WIDGET(warning_dialog),"warning_message_field"));
      gtk_label_set_text (warning_message_field,warning_message);
      gtk_widget_show (GTK_WIDGET(warning_message_field));
      gtk_widget_show (warning_dialog);
      gdk_beep();
    }
  else
    {
      /* Falls das aktuelle Dokument noch keinen Namen besitzt, rufen
       * wir die Funktion save_as() auf, um das Dialogfenster zum
       * Speichern von Dateien zu zeigen.
       */
      if (current_filename == NULL)
	save_as (main_window);
      else
	real_save_file (main_window,current_filename);
    }
}


void
on_add_new_data_clicked                (GtkButton       *button,
                                        gpointer         user_data)
{
  add_data((GtkWidget *) user_data);
}


void
on_delete_data_button_clicked          (GtkButton       *button,
                                        gpointer         user_data)
{
  delete_data(gtk_widget_get_toplevel(GTK_WIDGET(button)));
}


/***************************************************************************
 *                   FUNKTIONEN ZUR BEARBEITUNG DER CLIST                  *
 ***************************************************************************/

void
on_clist_select_row                    (GtkCList        *clist,
                                        gint             row,
                                        gint             column,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
  gchar *text;
  GtkEntry *entry;
  
  mw_clist_row = row;
  mw_clist_column = column;
  gtk_clist_get_text(GTK_CLIST(clist),row,column,&text);
  entry = GTK_ENTRY(lookup_widget (GTK_WIDGET (gtk_widget_get_toplevel (GTK_WIDGET (clist))),"clist_cell_entry"));
  gtk_widget_set_sensitive(GTK_WIDGET(entry),TRUE);
  gtk_entry_set_text(entry,text);
}


void
on_clist_cell_entry_changed            (GtkEditable     *editable,
                                        gpointer         user_data)
{
  GtkWidget *statusbar;

  if(!message_veto)
    {
      /* Die nchsten Zeilen schreiben eine Meldung in die Statuszeile */
      statusbar = lookup_widget (gtk_widget_get_toplevel(GTK_WIDGET(editable)), "statusbar");
      gtk_statusbar_pop (GTK_STATUSBAR (statusbar), 1);
      gtk_statusbar_push (GTK_STATUSBAR (statusbar), 1, "Daten werden bearbeitet.");
    }

  /* Die nchste Anweisung zeigt an, dass das Telefonbuch gendert wurde,
   * falls die Zeilenanzahl anders als 0 ist (wenn die Zeilenanzahl 0 ist,
   * macht es keinen Sinn, die Telefonbuchdatei beim Verlassen des
   * Programmes zu speichern, und dann wird file_changed = FALSE). 
   */
  if(mw_clist)
    {
      if(mw_clist->rows > 0)
	file_changed = TRUE;
      else
	file_changed = FALSE;

      /* ndert die entsprechenden Daten einer Zelle */
      gtk_clist_set_text(GTK_CLIST(mw_clist),
			 mw_clist_row,mw_clist_column,
			 gtk_entry_get_text(GTK_ENTRY(editable)));
    }

  /* Resetet message_veto zu FALSE */
  message_veto = FALSE;
}


void
on_clist_click_column                  (GtkCList        *clist,
                                        gint             column,
                                        gpointer         user_data)
{
  GtkWidget *main_window = (GtkWidget *) user_data;
  GtkWidget *statusbar;
  GtkLabel *warning_message_field;
  GtkEntry *entry;
  gchar *warning_message = "Oh, oh! Was soll\ndenn bitte schn\nsortiert werden?\nSie haben ja\nkeine Daten. :-)";


  if((mw_clist == NULL) || (mw_clist->rows == 0))
    {
      Warnings = no_data_to_sort;
      file_changed = FALSE;
      warning_dialog = create_warning_dialog ();
      gtk_widget_realize(warning_dialog);
      warning_message_field = GTK_LABEL(lookup_widget(GTK_WIDGET(warning_dialog),"warning_message_field"));
      gtk_label_set_text (warning_message_field,warning_message);
      gtk_widget_show (GTK_WIDGET(warning_message_field));
      gtk_widget_show (warning_dialog);
      gdk_beep();
    }
  else
    {
      /* nderung der Sortierungsmethode */
      gtk_clist_set_sort_column(mw_clist,column);
      gtk_clist_set_sort_type(mw_clist,GTK_SORT_ASCENDING);
      gtk_clist_set_auto_sort(mw_clist,TRUE);
      gtk_clist_sort (mw_clist);

      /* Resetet die Zeilen- bzw. Spaltenselektierung. Ausserdem
       * Lscht den Inhalt des Eingabefeldes und macht ihn 
       * nicht bearbeitbar (grau)
       */
      gtk_clist_unselect_all (mw_clist);
      mw_clist_row = -1;
      mw_clist_column = -1;
      entry = GTK_ENTRY(lookup_widget (user_data,"clist_cell_entry"));
      gtk_entry_set_text(entry,"");
      gtk_widget_set_sensitive(GTK_WIDGET(entry),FALSE);

      /* Die nchsten Zeilen schreiben verschiedene Meldungen
       * in die Statuszeile, je nach der Sortierungsmethode
       */
      statusbar = lookup_widget (GTK_WIDGET (main_window), "statusbar");
      switch(column)
	{
	case 0:
	  gtk_statusbar_pop (GTK_STATUSBAR (statusbar), 1);
	  gtk_statusbar_push (GTK_STATUSBAR (statusbar), 1, 
			      "Daten nach Namen sortiert.");
	  break;
	case 1:
	  gtk_statusbar_pop (GTK_STATUSBAR (statusbar), 1);
	  gtk_statusbar_push (GTK_STATUSBAR (statusbar), 1, 
			      "Daten nach Anschrift sortiert.");
	  break;
	case 2:
	  gtk_statusbar_pop (GTK_STATUSBAR (statusbar), 1);
	  gtk_statusbar_push (GTK_STATUSBAR (statusbar), 1, 
			      "Daten nach Telefonnummer sortiert.");
	  break;
	}

      /* Aktualisiert den file_changed-Flag und den data_resorted-Flag */
      file_changed = TRUE;
      data_resorted = TRUE;
    }
}


/***************************************************************************
 *          FUNKTIONEN ZUR BEARBEITUNG DER DIALOG-FENSTER-BUTTONS          *
 ***************************************************************************/

void
input_dialog_ok_button_clicked         (GtkButton       *button,
                                        gpointer         user_data)
{
  GtkWidget *main_window,*statusbar;
  GtkEntry *entry;
  GtkLabel *warning_message_field;
  gchar *data[3];
  gchar *warning_message = "Mindestens eines der\nEingabefelder ist leer.\nIch hoffe, Sie wissen,\nwas Sie da machen...";

  /* Die nchste Anweisung zeigt an, dass das Telefonbuch gendert wurde */
  file_changed = TRUE;

  /* Die nchste Zeile stellt die Adresse des Hauptfensters wieder her,
   * das vorher mit dem Dateneintrag-Dialogfenster verbunden worden war.
   */
  main_window = gtk_object_get_data (GTK_OBJECT (input_dialog),
				     MainWindowKey);

  /* Die nchsten drei Anweisungen speichern die vom Benutzer eingegebenen
   * Eintrge (Name, Anschrift und Telefonnummer) in den Zeichenkettenvektor
   * *data[3]
   */
  data[0] = gtk_entry_get_text(GTK_ENTRY(lookup_widget (GTK_WIDGET (gtk_widget_get_toplevel (GTK_WIDGET (button))),"name_entry")));
  data[1] = gtk_entry_get_text(GTK_ENTRY(lookup_widget (GTK_WIDGET (gtk_widget_get_toplevel (GTK_WIDGET (button))),"address_entry")));
  data[2] = gtk_entry_get_text(GTK_ENTRY(lookup_widget (GTK_WIDGET (gtk_widget_get_toplevel (GTK_WIDGET (button))),"phone_entry")));

  /* Warnung, falls mindestens eines der Eingabefelder leer ist
   * (die Eingabe wird sowieso akzeptiert).
   */
  if(!strcmp(data[0],"") || !strcmp(data[1],"") || !strcmp(data[2],""))
    {
      Warnings = no_data_entered;
      warning_dialog = create_warning_dialog ();
      gtk_widget_realize(warning_dialog);
      warning_message_field = GTK_LABEL(lookup_widget(GTK_WIDGET(warning_dialog),"warning_message_field"));
      gtk_label_set_text (warning_message_field,warning_message);
      gtk_widget_show (GTK_WIDGET(warning_message_field));
      gtk_widget_show (warning_dialog);
      gdk_beep();
    }

  /* Die folgenden Anweisungen erstellen eine neue clist, falls sie
   * bereits noch nicht kreiert wurde, und stellen ihre Eigenschaften
   * ein: Sortierungsspalte und -methode und ob diese automatisch sein
   * soll (bei der automatischen Sortierung werden neue Eintrge in
   * die Liste in der entsprechenden Position erfolgen).
   */
  if(mw_clist == NULL)
    {
      mw_clist = GTK_CLIST(lookup_widget(GTK_WIDGET(main_window),"clist"));
      gtk_clist_set_sort_column(mw_clist,0);
      gtk_clist_set_sort_type(mw_clist,GTK_SORT_ASCENDING);
      gtk_clist_set_auto_sort(mw_clist,TRUE);
    }

  /* Die folgende Zeile hngt die Eintrge an die clist auf dem 
   * Hauptfenster an 
   */
  gtk_clist_append(mw_clist,data);

  /* Die nchsten Zeilen schreiben eine Meldung in die Statuszeile */
  statusbar = lookup_widget (GTK_WIDGET (main_window), "statusbar");
  gtk_statusbar_pop (GTK_STATUSBAR (statusbar),1);
  gtk_statusbar_push (GTK_STATUSBAR (statusbar),1,
		      "Neue Daten hinzugefgt.");

  /* Aktualisiert den file_changed-Flag und den data_added-Flag */
  file_changed = TRUE;
  message_veto = TRUE;

  /* */
  input_dialog_shown = FALSE;

  /* Resetet die Zeilen- bzw. Spaltenselektierung. Ausserdem
   * lscht den Inhalt des Eingabefeldes und macht ihn nicht
   * bearbeitbar (grau)
   */
  gtk_clist_unselect_all (mw_clist);
  mw_clist_row = -1;
  mw_clist_column = -1;
  entry = GTK_ENTRY(lookup_widget (GTK_WIDGET(main_window),
				   "clist_cell_entry"));
  gtk_entry_set_text(entry,"");
  gtk_widget_set_sensitive(GTK_WIDGET(entry),FALSE);

  /* Dies zerstrt das Dateneintrag-Dialogfenster */ 
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET(button)));
}


void
input_dialog_cancel_button_clicked     (GtkButton       *button,
                                        gpointer         user_data)
{
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET(button)));
  input_dialog_shown = FALSE;
}


void
about_ok_button_clicked                (GtkButton       *button,
                                        gpointer         user_data)
{
  about_dialog_shown = FALSE;
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
}


void
open_file_ok_button_clicked            (GtkButton       *button,
                                        gpointer         user_data)
{
  GtkWidget *file_select_dialog, *main_window;
  gchar *filename;

  file_select_dialog = gtk_widget_get_toplevel (GTK_WIDGET (button));
  main_window = gtk_object_get_data (GTK_OBJECT (file_select_dialog),
				     MainWindowKey);
  gtk_widget_hide (file_select_dialog);
  filename = gtk_file_selection_get_filename (GTK_FILE_SELECTION (file_select_dialog));
  real_open_file (main_window,filename);
}


void
open_file_cancel_button_clicked        (GtkButton       *button,
                                        gpointer         user_data)
{
  gtk_widget_hide (gtk_widget_get_toplevel (GTK_WIDGET(button)));
  open_file_dialog_shown = FALSE;
}


void
save_as_ok_button_clicked              (GtkButton       *button,
                                        gpointer         user_data)
{
  GtkWidget *save_file_dialog, *main_window;
  gchar *filename;

  save_file_dialog = gtk_widget_get_toplevel (GTK_WIDGET (button));
  main_window = gtk_object_get_data (GTK_OBJECT (save_file_dialog),
				     MainWindowKey);
  gtk_widget_hide (save_file_dialog);
  filename = gtk_file_selection_get_filename (GTK_FILE_SELECTION (save_file_dialog));
  real_save_file (main_window,filename);
}


void
save_as_cancel_button_clicked          (GtkButton       *button,
                                        gpointer         user_data)
{
  gtk_widget_hide (gtk_widget_get_toplevel (GTK_WIDGET(button)));
  save_as_dialog_shown = FALSE;
}


void
on_warning_dialog_ok_button_clicked      (GtkButton       *button,
					  gpointer         user_data)
{
  GtkWidget *main_window;

  switch(Warnings)
    {
    case no_saved_data:
      gtk_exit(0);

    case load_new_file:

      /* Die nchste Zeile stellt die Adresse des Hauptfensters wieder her,
       * das vorher mit dem Warnung-Dialogfenster verbunden worden war.
       */
      main_window = gtk_object_get_data (GTK_OBJECT (warning_dialog),
					 MainWindowKey);
      open_file_selection(main_window);

    case create_new_file:

      /* Die nchste Zeile stellt die Adresse des Hauptfensters wieder her,
       * das vorher mit dem Warnung-Dialogfenster verbunden worden war.
       */
      main_window = gtk_object_get_data (GTK_OBJECT (warning_dialog),
					 MainWindowKey);
      reset_env(main_window);

    case no_data_to_delete:
    case no_line_selected:
    case no_data_entered:
    case no_data_to_sort:
    case no_data_to_save:
      gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET(button)));
    }
}


void
on_warning_dialog_cancel_button_clicked  (GtkButton       *button,
                                        gpointer         user_data)
{
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET(button)));
}



/***************************************************************************
 *          FUNKTIONEN ZUR BEARBEITUNG DER "DELETE-EVENT" SIGNALE          *
 ***************************************************************************/

/* Diese Funktion wird aufgerufen, falls das Hauptfenster ein so genanntes
 * "delete_event"-Signal erhlt, was normalerweise passiert, wenn der
 * Benutzer das X-Symbol (Schliesssymbol) auf der Fenster-Titelleiste
 * anklickt. Wenn wir das nicht bercksichtigen, wrde das Hauptfenster
 * zugemacht, aber unsere Applikation nicht geschlossen.
 */

gboolean
on_telefonbuch_delete_event            (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
  if(input_dialog_shown)
    return TRUE;
  quit_program();

  /* TRUE ist notwendig, ansonsten wrde das Hauptfenster zugemacht,
   * und man knnte es nicht mehr erreichen...
   */
  return TRUE;
}


/* Das selbe gilt fr alle Dialog-Fenster, die entsprechend bearbeitet
 * werden mssen...
 */

gboolean
on_input_dialog_delete_event           (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
  gtk_widget_destroy (gtk_widget_get_toplevel (widget));
  input_dialog_shown = FALSE;
  return TRUE;
}


gboolean
on_about_dialog_delete_event           (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
  gtk_widget_destroy (gtk_widget_get_toplevel (widget));
  about_dialog_shown = FALSE;
  return TRUE;
}


gboolean
on_open_file_dialog_delete_event       (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
  gtk_widget_hide (gtk_widget_get_toplevel (widget));
  open_file_dialog_shown = FALSE;
  return TRUE;
}

gboolean
on_save_as_dialog_delete_event         (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
  gtk_widget_hide (gtk_widget_get_toplevel (widget));
  save_as_dialog_shown = FALSE;
  return TRUE;
}

gboolean
on_warning_dialog_delete_event           (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
  gtk_widget_destroy (gtk_widget_get_toplevel (widget));
  return TRUE;
}


/***************************************************************************
 *           FUNKTIONEN, DIE NICHT VON SIGNALEN GESTARTET WERDEN           *
 ***************************************************************************/

/* Diese Funktion initialisiert alle Variablen, um eine neue Datei
 * zu kreieren.
 */

static void
new_file (GtkWidget *main_window)
{
  GtkLabel *warning_message_field;
  gchar *warning_message = "Telefonbuch nicht\ngespeichert! Ihre\nnderungen werden\nverlorengehen! Neue Datei\ntrotzdem erstellen?";

  /* Warnung, falls eine nicht gespeicherte Datei bearbeitet wird */
  if(file_changed == TRUE)
    {
      Warnings = create_new_file;
      warning_dialog = create_warning_dialog ();

      /* Die nchste Zeile verbindet den Hauptfenster-Zeiger mit
       * dem Warnung-Dialogfenster durch den Schlssel MainWindowKey */
      gtk_object_set_data (GTK_OBJECT (warning_dialog),
			   MainWindowKey,main_window);
      gtk_widget_realize(warning_dialog);
      warning_message_field = GTK_LABEL(lookup_widget(GTK_WIDGET(warning_dialog),"warning_message_field"));
      gtk_label_set_text (warning_message_field,warning_message);
      gtk_widget_show (GTK_WIDGET(warning_message_field));
      gtk_widget_show (warning_dialog);
      gdk_beep();
    }
  else
    {
      reset_env(main_window);
    }
}


static void open_file (GtkWidget *main_window)
{
  GtkLabel *warning_message_field;
  gchar *warning_message = "Telefonbuch nicht\ngespeichert! Ihre\nnderungen werden\nverlorengehen! Neue\nDatei trotzdem laden?";

  /* Warnung, falls eine nicht gespeicherte Datei bearbeitet wird */
  if(file_changed == TRUE)
    {
      Warnings = load_new_file;
      warning_dialog = create_warning_dialog ();

      /* Die nchste Zeile verbindet den Hauptfenster-Zeiger mit
       * dem Warnung-Dialogfenster durch den Schlssel MainWindowKey */
      gtk_object_set_data (GTK_OBJECT (warning_dialog),
			   MainWindowKey,main_window);
      gtk_widget_realize(warning_dialog);
      warning_message_field = GTK_LABEL(lookup_widget(GTK_WIDGET(warning_dialog),"warning_message_field"));
      gtk_label_set_text (warning_message_field,warning_message);
      gtk_widget_show (GTK_WIDGET(warning_message_field));
      gtk_widget_show (warning_dialog);
      gdk_beep();
    }
  else
    {
      open_file_selection(main_window);
    }
}


/* Diese Funktion zeigt das Dialog-Fenster zum ffnen einer
 * Telefonbuchdatei.
 */

static void
open_file_selection (GtkWidget *main_window)
{
  /* Wir werden immer das selbe Dialog-Fenster zum ffnen von Dateien
   * verwenden und deswegen muss man es zuallererst erstellen, falls 
   * das noch nicht gemacht wurde.
   */ 
  if (open_file_dialog == NULL)
    open_file_dialog = create_open_file_dialog ();

  /* Wir speichern einen Zeiger zum Hauptfenster innerhalb der Datenliste
   * des Dialog-Fensters, sodass man leicht auf das Hauptfenster durch
   * "Callback"-Funktionen zugreifen kann.
   */
  gtk_object_set_data (GTK_OBJECT (open_file_dialog), 
		       MainWindowKey, main_window);

  gtk_widget_show (open_file_dialog);
  gdk_window_raise (open_file_dialog->window);
}


/* Diese Funktion ldt eine Telefonbuchdatei */

static void
real_open_file (GtkWidget *main_window, gchar *filename)
{
  GtkWidget *statusbar;
  FILE *fp;
  gchar *data[3],*text,*buffer;
  gint bytes_read,i,j;

  buffer = (gchar *) g_malloc0(65536 * sizeof(gchar));

  statusbar = lookup_widget (main_window, "statusbar");
  gtk_statusbar_pop (GTK_STATUSBAR (statusbar), 1);
  g_free (current_filename);
  current_filename = NULL;
  file_changed = FALSE;

  /* Die nchste Anweisung rumt eine eventuell vorhandene Liste */
  if(mw_clist != NULL)
    gtk_clist_clear(mw_clist);

  fp = fopen (filename,"r");
  if (fp == NULL)
    {
      gtk_statusbar_push (GTK_STATUSBAR (statusbar), 1, "Datei ffnen schlug fehl!");
      return;
    }

  for (;;)
    {
      /* Lesen der Telefonbuchdatei und deren Aufnahme in eine 
       * Zeichenkette.
       */
      bytes_read = fread (buffer, sizeof (gchar), 65536, fp);

      /* Aufteilung der gelesenen Zeichenkette in CList-Feldern */
      if (bytes_read > 0)
	{
	  for(i=0;strcmp(buffer,"\0");i++)
	    {
	      for(j=0;j<3;j++)
		{
		  if(j<2)
		    text = strchr(buffer,'\t');
		  else
		    text = strchr(buffer,'\n');
		  *text = '\0';
 		  data[j] = g_strdup(buffer);
		  buffer = ++text;
		}

	      /* Die folgenden Anweisungen erstellen ein neue clist,
	       * falls sie bereits noch nicht kreiert wurde, und stellen
	       * ihre Eigenschaften ein: Sortierungsspalte und -methode
	       * und ob diese automatisch sein soll (bei der automatischen
	       * Sortierung werden neue Eintrge in die Liste in der
	       * entsprechenden Position erfolgen).
	       */
	      if(mw_clist == NULL)
		{
		  mw_clist = GTK_CLIST(lookup_widget(GTK_WIDGET(main_window),"clist"));
		  gtk_clist_set_sort_column(mw_clist,0);
		  gtk_clist_set_sort_type(mw_clist,GTK_SORT_ASCENDING);
		  gtk_clist_set_auto_sort(mw_clist,TRUE);
		}
	      gtk_clist_append(mw_clist,data);
	    }	  
	}

      /* Test ob EOF (Dateiendemarkierung) gefunden wurde */
      if (bytes_read != 65536 && (feof (fp) || ferror (fp)))
	break;
    }

  /* Falls ein Fehler auftritt, bringen wir alles zurck in einen 
   * "gesunden" Zustand.
   */
  if (ferror (fp))
    {
      fclose (fp);

      /* Die nchste Anweisung zeigt an, dass eine neue ungenderte
       * Telefonbuch-Liste angelegt wurde */
      if(mw_clist != NULL)
	gtk_clist_clear(mw_clist);

      set_window_title (main_window);
      gtk_statusbar_push (GTK_STATUSBAR (statusbar),1,
			  "Dateiladen schlug fehl!");
      return;
    }

  fclose (fp);

  current_filename = g_strdup (filename);
  set_window_title (main_window);
  gtk_statusbar_pop (GTK_STATUSBAR (statusbar),1);
  gtk_statusbar_push (GTK_STATUSBAR (statusbar),1,
		      "Datei wurde geladen.");
}


/* Diese Funktion zeigt das Dialog-Fenster zum Speichern einer 
 * Telefonbuchdatei.
 */

static void
save_as (GtkWidget *main_window)
{
  if (save_file_dialog == NULL)
    save_file_dialog = create_save_as_dialog ();
  gtk_object_set_data (GTK_OBJECT (save_file_dialog),
		       MainWindowKey, main_window);

  /* Falls die aktuelle Telefonbuchdatei bereits einen Namen besitzt,
   * verwenden wir ihn als Voreinstellung.
   */
  if (current_filename)
    gtk_file_selection_set_filename (GTK_FILE_SELECTION (save_file_dialog),
				     current_filename);
  gtk_widget_show (save_file_dialog);
  gdk_window_raise (save_file_dialog->window);
}


/* Diese Funktion speichert das Telefonbuch in eine Datei */

static void
real_save_file (GtkWidget *main_window, gchar *filename)
{
  GtkWidget *statusbar;
  gchar *data,*text,*buffer;
  FILE *fp;
  gint bytes_written,len,i,j;

  if (current_filename == NULL || strcmp (current_filename, filename))
    {
      g_free (current_filename);
      current_filename = g_strdup (filename);
      set_window_title (main_window);
    }

  data = (gchar *) calloc(65536,sizeof(gchar));
  buffer = (gchar *) calloc(1024,sizeof(gchar));
  statusbar = lookup_widget (main_window, "statusbar");

  /* Baut eine Zeichenkette, die als Datei abgespeichert wird */
  for(i=0;i<=(mw_clist->rows-1);i++)
    {
      for(j=0;j<3;j++)
	{
	  gtk_clist_get_text(GTK_CLIST(mw_clist),i,j,&text);
	  if(j<2)
	    sprintf(buffer,"%s\t",text);
	  else
	    sprintf(buffer,"%s",text);
	  strcat(data,buffer);
	}
      strcat(data,"\n");
    }
  strcat(data,"\0");
  fp = fopen (filename, "w");
  if (fp == NULL)
    {
      g_free (data);
      g_free (buffer);
      return;
    }

  len = strlen (data);
  bytes_written = fwrite (data, sizeof (gchar), len, fp);
  fclose (fp);

  g_free (data);
  g_free (buffer);

  /* Falls ein Fehler beim Speichern auftritt, wird eine Meldung in die
   * Statuszeile geschrieben.
   */
  gtk_statusbar_pop (GTK_STATUSBAR (statusbar), 1);
  if (len != bytes_written)
    {
      gtk_statusbar_push (GTK_STATUSBAR (statusbar), 1, "Fehler beim Speicher!");
      return;
    }

  file_changed = FALSE;

  /* Meldung eines erfolgreichen Speicherns der Telefonbuchdatei */
  gtk_statusbar_push (GTK_STATUSBAR (statusbar), 1, "Datei erfolgreich gespeichert!");
}


static void
add_data (GtkWidget *main_window)
{
  input_dialog_shown = TRUE;
  input_dialog = create_input_dialog ();

  /* Die nchste Zeile verbindet den Hauptfenster-Zeiger mit
   * dem Dateneintrag-Dialogfenster durch den Schlssel MainWindowKey
   */
  gtk_object_set_data (GTK_OBJECT (input_dialog),
		       MainWindowKey, main_window);
  gtk_widget_show (input_dialog);
}


static void
delete_data (GtkWidget *main_window)
{
  GtkWidget *statusbar;
  GtkEntry *entry;
  GtkLabel *warning_message_field;
  gchar *warning_message1 = "Oh, oh! Was soll\ndenn bitte schn\ngelscht werden?\nSie haben ja\nkeine Daten. :-)";
  gchar *warning_message2 = "Sind Sie sicher, dass\nSie nichts vergessen\nhaben (wie z.B. eine\nZeile zu whlen...:-)?";

  /* Fehlerbehandlung */
  if((mw_clist == NULL) || (mw_clist->rows == 0))
    {
      Warnings = no_data_to_delete;
      file_changed = FALSE;
      warning_dialog = create_warning_dialog ();
      gtk_widget_realize(warning_dialog);
      warning_message_field = GTK_LABEL(lookup_widget(GTK_WIDGET(warning_dialog),"warning_message_field"));
      gtk_label_set_text (warning_message_field,warning_message1);
      gtk_widget_show (GTK_WIDGET(warning_message_field));
      gtk_widget_show (warning_dialog);
      gdk_beep();
    }
  else
    {
      if(mw_clist_row == -1)
	{
	  Warnings = no_line_selected;
	  file_changed = FALSE;
	  warning_dialog = create_warning_dialog ();
	  gtk_widget_realize(warning_dialog);
	  warning_message_field = GTK_LABEL(lookup_widget(GTK_WIDGET(warning_dialog),"warning_message_field"));
	  gtk_label_set_text (warning_message_field,warning_message2);
	  gtk_widget_show (GTK_WIDGET(warning_message_field));
	  gtk_widget_show (warning_dialog);
	  gdk_beep();
	}
      else
	{
	  /* Die nchsten Zeilen schreiben eine Meldung in 
	   * die Statuszeile
	   */
	  statusbar = lookup_widget (main_window,"statusbar");
	  gtk_statusbar_pop (GTK_STATUSBAR (statusbar), 1);
	  gtk_statusbar_push (GTK_STATUSBAR (statusbar), 1, 
                              "Upsalla! Eine Zeile wurde gelscht! :-)"); 

	  /* Lscht eine Zeile */
	  gtk_clist_remove(mw_clist,mw_clist_row);

	  /* Zeigt an, dass das statusbar nicht wieder geschrieben
	   * werden soll, wenn der Inhalt des Eingabefeldes des 
	   * Hauptfensters gelscht wird (in diesem Fall soll ein Test
	   * in der Funktion, die das Signal "changed" behandelt, 
	   * mittels der "message_veto"-Variabeln testen, ob das
	   * statusbar geschrieben werden soll)
	   */
	  message_veto = TRUE;

	  /* Resetet die Zeilen- bzw. Spaltenselektierung. Ausserdem
	   * lscht es den Inhalt des Eingabefeldes und macht ihn 
	   * nicht bearbeitbar (grau).
	   */
	  gtk_clist_unselect_all (mw_clist);
	  mw_clist_row = -1;
	  mw_clist_column = -1;
	  entry = GTK_ENTRY(lookup_widget (GTK_WIDGET(main_window),
					   "clist_cell_entry"));
	  gtk_entry_set_text(entry,"");
	  gtk_widget_set_sensitive(GTK_WIDGET(entry),FALSE);
	}
    }
}


static void reset_env (GtkWidget *main_window)
{
  GtkWidget *statusbar;
  GtkEntry *entry;

  /* Reinitialisierung globaler Variabel */
  current_filename = NULL;
  mw_clist_row = -1;
  mw_clist_column = -1;
  file_changed = FALSE;
  message_veto = TRUE;

  /* Die nchsten Zeilen schreiben eine Meldung in die Statuszeile */
  statusbar = lookup_widget (GTK_WIDGET(main_window),"statusbar");
  gtk_statusbar_pop (GTK_STATUSBAR (statusbar),1);
  gtk_statusbar_push (GTK_STATUSBAR (statusbar),1,"Neue Datei");

  /* Die nchste Anweisung zeigt an, dass eine neue ungenderte
   * Telefonbuch-Liste angelegt wurde.
   */
  if(mw_clist != NULL)
    gtk_clist_clear(mw_clist);

  /* Anzeigen im Titel des Hauptfensters, dass eventuelle Daten in der
   * Clist des Hauptfensters noch nicht in eine Datei gespeichert wurden.
   */
  set_window_title (main_window);

  /* Inhalt des Eingabefeldes des Hauptfensters lschen und dieses
   * nicht bearbeitbar (grau) machen
   */
  entry = GTK_ENTRY(lookup_widget (GTK_WIDGET(main_window),
				   "clist_cell_entry"));
  gtk_entry_set_text(entry,"");
  gtk_widget_set_sensitive(GTK_WIDGET(entry),FALSE);
}


/* Diese Funktion gibt dem Hauptfenster einen Titel nach dem
 * aktuellen Dateinamen.
 */

void
set_window_title (GtkWidget *main_window)
{
  gchar *title;

  title = g_strdup_printf ("Telefonbuch: %s ",
			   current_filename ? g_basename (current_filename)
			   : "nicht gespeichert");

  gtk_window_set_title (GTK_WINDOW (main_window), title);
  g_free (title);
}


static void
quit_program(void)
{
  GtkLabel *warning_message_field;
  gchar *warning_message = "Man o man! Sie sind\nkurz vor einer Katas-\ntrophe! Hoffentlich\nhaben Sie ein gutes\nGedchnis und knnen\nsich die paar Namen\nauch ohne das Telefon-\nbuch merken...";

  if(file_changed == TRUE)
    {
      Warnings = no_saved_data;
      warning_dialog = create_warning_dialog ();
      gtk_widget_realize(warning_dialog);
      warning_message_field = GTK_LABEL(lookup_widget(GTK_WIDGET(warning_dialog),"warning_message_field"));
      gtk_label_set_text (warning_message_field,warning_message);
      gtk_widget_show (GTK_WIDGET(warning_message_field));
      gtk_widget_show (warning_dialog);
      gdk_beep();
    }
  else
    gtk_exit (0);
}

