/*
 * Adapted from Powertweak Linux. (C) 2000 Dave Jones, Arjan van de Ven.
 * Converted to standalone use by Leigh L. Klotz, Jr. <klotz@graflex.org>
 * Aug 10, 2001 Added read functionality: Harald Milz <hm@seneca.muc.de> 
 *
 * 	Licensed under the terms of the GNU GPL License version 2.
 *
 *
 */

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/io.h>
#include <fcntl.h>

void usage_devcpu();

static int wrmsr(int cpu, unsigned long index, unsigned long long *val)
{
	char cpuname[16];
	int fh, ret;

	snprintf (cpuname,15, "/dev/cpu/%d/msr", cpu);

	fh = open (cpuname, O_WRONLY);

	if (fh==-1)
		ret = -1;
	else {
		lseek (fh, index, SEEK_CUR);
		ret = (write (fh, val, 8) == 8);
		close (fh);
	}
	return (ret);
}


/* Clock ratio multiplied by 10 */
static int ClockRatio[8] = {	45,  /* 000 -> 4.5x */
				50,  /* 001 -> 5.0x */
				40,  /* 010 -> 4.0x */
				55,  /* 011 -> 5.5x */
				20,  /* 100 -> 2.0x */
				30,  /* 101 -> 3.0x */
				60,  /* 110 -> 6.0x */
				35   /* 111 -> 3.5x */
};




void set_cpu_mult_k6(int multbits) {
  /* "multbits" now contains the bitpattern of the new multiplier.
     we now need to transform it to the BVC format, see AMD#23446 */
  unsigned long outvalue=0,invalue=0;
  unsigned long long msrval;
  int errcode;

  /* Get IO permissions */
  if (iopl(3)!=0) {
    printf("error: You must be root or have raw IO permission to run this program.\n");
    return;
  }

  outvalue = (1<<12) | (1<<10) | (1<<9) | (multbits<<5);

  msrval = 0xFFF1;  	/* FIXME!! we should check if 0xfff0 is available */
  errcode = wrmsr(0,0xC0000086, &msrval); /* enable the PowerNow port */
  if (errcode < 0) {
    perror("Could not enable PowerNow");
    if (errno == ENOENT) usage_devcpu();
    return;
  }
  invalue=inl(0xfff8);
  invalue = invalue & 15;
  outvalue = outvalue | invalue;
  outl(outvalue, 0xFFF8);
  msrval = 0xFFF0;
  wrmsr(0,0xC0000086, &msrval); /* disable it again */
}


int get_cpu_mult_k6(void) {

  unsigned long invalue=0;
  unsigned long long msrval;
  int errcode;
  int multbits;

  /* Get IO permissions */
  if (iopl(3)!=0) {
    printf("error: You must be root or have raw IO permission to run this program.\n");
    return(-1);
  }

  msrval = 0xFFF1;      /* FIXME!! we should check if 0xfff0 is available */
  errcode = wrmsr(0,0xC0000086, &msrval); /* enable the PowerNow port */
  if (errcode < 0) {
    perror("Could not enable PowerNow");
    if (errno == ENOENT) usage_devcpu();
    return(-1);
  }
  invalue=inl(0xfff8);
  multbits = (invalue >> 5) & 7;
  msrval = 0xFFF0;
  wrmsr(0,0xC0000086, &msrval); /* disable it again */
  return (multbits);

}


void usage(char *name) {
  int i;
  fprintf(stderr, "usage: %s mult\n\twhere mult is one of", name);
  for (i=0;i<8;i++)
    fprintf(stderr, " %2.1f", ClockRatio[i]/10.0);
  fprintf(stderr, " r \n");
  fprintf(stderr, "\twhere \"r\" means \"read current setting\"\n");
}

void usage_devcpu() {
  fprintf(stderr,"\n"
"1. This program works only with K6-III+/K6-2+ (processors with AMD PowerNow(TM) CPU Family 6, Model 13)\n"
" To check, do this:\n"
"   $ cat /proc/cpuinfo | grep 'cpu family'\n"
"   cpu family	: 5\n"
"   $ cat /proc/cpuinfo | grep 'model'\n"
"   model	: 13\n"
"\n"
"2. Your system must be running a K6 kernel with MSR support.\n"
" To check, do this\n"
"   $ arch\n"
"   i586\n"
"   If you compiled the kernel, you can do this as well:\n"
"   $ grep MSR /usr/src/linux-2.4.2/.config\n"
"   CONFIG_X86_MSR=m\n"
"   \n"
"3. You must have the msr module installed.  If not, do this:\n"
"   $ su\n"
"   # /sbin/modprobe msr\n"
"   # exit\n"
"4. You must have /dev/cpu/0/msr created.  If not do this:\n"
"   $ su\n"
"   # mkdir -p /dev/cpu/0\n"
"   # mknod /dev/cpu/0/msr 0 202 c\n"
"   # exit\n"
"\n"
);
}

int main(int argc, char *argv[]) {
  int multno;
  int mult;
  if (argc != 2) {
    usage(argv[0]);
    exit(-1);
  }

  /* now this isn't exactly structured programming :-( */
  if (!strncmp("r", argv[1], 1)) {
    printf ("current clock ratio: %1.1f\n", 
      ClockRatio[get_cpu_mult_k6()] / 10.0);
    exit (0);
  }

  mult = 10*atof(argv[1]);
  for (multno=0;multno<8;multno++)
    if (mult == ClockRatio[multno])
      break;
  if (multno == 8) {
    fprintf(stderr, "multiplier %s not recognized:\n", argv[1]);
    usage(argv[0]);
    exit(-1);
  }
  
  // fprintf(stderr, "set mult=%d, bits=%d\n", ClockRatio[multno], multno);
  set_cpu_mult_k6(multno);
  exit(0);
}

