# TkGui.py
# Written by David Allen <s2mdalle@titan.vcu.edu>
# Released under the terms of the GNU General Public License
#
# This handles the GUI of the program, displaying, bookmarks, etc.
# This module is usually called by the starter script.  All other modules
# are usually invoked through this one as the GUI needs them.
############################################################################
# Python-wide modules not specific to this program
import os
import string
import regsub
import thread
import time
import pickle
import socket   # Just for errors 

# Data structures, connections, and the like.
import Options
import GopherThingy
import GopherObject
import ResourceInformation
import GopherResource
import GopherResponse
import GopherConnection
import Bookmark
import List
import ListNode
import Associations
import Cache
from gopher import *

# GUI specific modules.
from Tkinter            import *
#import DirCanvas
import ErrorDialog
import AssociationsEditor
import GUIDirectory
import GUIFile
import GUISearch
import GUISaveFile
import FindDialog
import SplashFrame
import Pmw

class TkGui(Tk, GopherThingy.GopherThingy):
    VERSION = '0.03'
    verbose = None
    def __init__(self, URL=None):
        """Create an instance of the gopher program."""
        Options.program_options = Options.Options()
        print "Prefs are ", self.getPrefsDirectory()
        self.loadOptions()              # Load program options from disk
        # This will create bookmarks if they don't exist.
        self.loadBookmarks()            # Load program bookmarks from disk

        if self.verbose:
            print "OPTIONS:\n%s" % Options.program_options.toString()
        
        Tk.__init__(self)

        self.saveFileDialog        = None
        self.DOWNLOAD_ME           = None     # Thread control variable
        self.CURRENTLY_DOWNLOADING = None     # Are we downloading right now?
        self.LAUNCH_ME             = None     # Thread control variable
        self.WIDGET_ENTRY          = 0
        self.RESOURCE_ENTRY        = 1
        self.RESPONSE_ENTRY        = 2

        self.mainBox = Frame(self)
        self.mainBox.pack(fill='both', expand=1)
        self.make_menus()
        self.config(menu=self.menu)
        self.buttonBar = Frame(self.mainBox)
        self.navBox    = Frame(self.mainBox)
        self.buttonBar.pack(fill='x', expand=0)
        self.navBox.pack(fill='x', expand=0)

        self.FORWARD  = Button(self.buttonBar, text='Forward',
                               command=self.goForward)
        self.BACKWARD = Button(self.buttonBar, text='Back',
                               command=self.goBackward)
        self.STOP     = Button(self.buttonBar, text='Stop',
                               command=self.notYetImplemented)
        self.RELOAD   = Button(self.buttonBar, text='Reload',
                               command=self.reload)
        self.BACKWARD.pack(side='left')
        self.FORWARD.pack(side='left')
        self.STOP.pack(side='left')
        self.RELOAD.pack(side='left')
        
        self.hostLabel = Label(self.navBox, text='Host:')
        self.hostEntry = Entry(self.navBox)
        self.resourceLabel = Label(self.navBox, text='Resource:')
        self.resourceEntry = Entry(self.navBox, text="/")
        self.resourceEntry.insert('end', '/')
        self.portLabel = Label(self.navBox, text='Port:')
        self.portEntry = Entry(self.navBox, width=5)
        self.gobutton  = Button(self.navBox, text='Go', command=self.go)
        
        # self.hostEntry.insert('end', 'gopher.ptloma.edu')
        self.portEntry.insert('end', '70')

        self.hostLabel.grid(row=0, col=0, sticky=W)
        self.hostEntry.grid(row=0, col=1, columnspan=2, sticky=W)
        self.resourceLabel.grid(row=0, col=3, sticky=W)
        self.resourceEntry.grid(row=0, col=4, columnspan=2, sticky=W)
        self.portLabel.grid(row=0, col=6, sticky=W)
        self.portEntry.grid(row=0, col=7, sticky=W) # No colspan: short box
        self.gobutton.grid(row=0, col=8, sticky=W)
        # self.hostLabel.pack(side='left')
        # self.hostEntry.pack(side='left', expand=0, fill='x')
        # self.resourceLabel.pack(side='left')
        # self.resourceEntry.pack(side='left', fill='x', expand=0)
        # self.portLabel.pack(side='left')
        # self.portEntry.pack(side='left')
        # self.gobutton.pack(side='right')

        self.CONTENT_BOX = Frame(self.mainBox)
        self.CONTENT_BOX.pack(expand=1, fill='both')

        self.messageBar = Pmw.MessageBar(self.mainBox,
                                         entry_width = 80,
                                         entry_relief='groove',
                                         labelpos = 'w',
                                         label_text = 'Status:')
        
        self.messageBar.pack(expand=0, fill='x')
        self.messageBar.message('state', "Ready")
        
        # Set up data structures
        # self.navList contains ONLY ListNode() objects
        self.locations      = List.List()
        self.navList        = List.List()
        # This is what is currently displayed.
        self.currentContent = None
        # Cached pages.
        self.cache          = Cache.Cache()

        self.createAssociations()
        self.createPopup()

        # self.changeContent(SplashFrame.SplashFrame(self.CONTENT_BOX,
        #                                            self, None, None, None))

        # The data for ListNodes in this list is a three-item list:
        # [widget, resource, response]
        # widget is the actual tkinter widget that is packed into the screen.
        # resource is the GopherResource object used to identify a connection.
        # response is the GopherRespones object used to cache the output of
        # a qeury with the server.

        if URL != None:
            resource = GopherResource.GopherResource()
            resource.setURL(URL)
            self.goElsewhere(resource)

        # Start the download thread and the external launcher thread.
        # These threads handle the launching of files saved in the cache,
        # and fetching documents from the network so that we don't seize the
        # program by waiting for data on a socket.
        thread.start_new_thread(self.downloadThread, ())
        thread.start_new_thread(self.externalLauncherThread, ())

        def fn(self=self, *args):
            print "Running DESTROY function"
            self.destroy()
            return None
            
        # Call fn when the window is destroyed.
        self.protocol("WM_DELETE_WINDOW", fn)
        
        return None

    def stats(self, *args):
        buffer = "%s item(s) in the queue\n" % self.navList.countNodes()
        buffer = buffer + self.cache.getCacheStats() + "\n"
        self.genericError(buffer)

    def destroy(self, *args):
        self.DOWNLOAD_ME = "DIE"    # Kill the two other threads with this
        self.LAUNCH_ME   = "DIE"    # 'message' to them.

        if Options.program_options.getOption('save_options_on_exit'):
            self.messageBar.message('state', "Saving program options...")
            self.saveOptions()
        
        Tk.destroy(self)
        print "I'm out."

    def editAssociations(self, *args):
        """Callback: opens the associations editor and lets the users change
        which programs launch on which filetypes."""
        x = AssociationsEditor.AssociationsEditor(self, self.associations)
        self.associations = x.getAssociations()
        print "Got new associations."
        return None

    def notYetImplemented(self, *args):
        """Opens a dialog for the user to see that the feature hasn't been
        implemented yet."""
        self.genericError("Contribute to the Programmer Pizza Fund today!\n" +
                          "Maybe that will help get this feature put into\n" +
                          "the program!")
        return None
    
    def setAssociations(self, assocList):
        """Modifier: associates a list of associations usually from the
        AssociationsEditor to this object."""
        self.associations = assocList
        print "GUI got list of associations."
        return None
    
    def createAssociations(self, *args):
        """Creates a set of default associations.  These are mostly UNIX
        centric unless I change them and forget to update this docstring.  :)
        """
        self.associations = Associations.Associations()
        images = [".gif", ".jpg", ".bmp", ".xpm", ".xbm",
                  ".png", ".jpeg", ".tif", ".tiff" ]

        for item in images:
            self.associations.addAssociation(item, "eeyes $1")

        browser_stuff = [".html", ".htm", ".css"]

        for item in browser_stuff:
            self.associations.addAssociation(item, "netscape $1")

    def genericError(self, errstr, title=''):
        """Given an error string, pops up a dialog with the same title and
        text, alerting the user to something or other.  This is not limited
        to errors, but could be warnings and messages too."""
        x = ErrorDialog.ErrorDialog(self, errstr, title)
        return None

    def find(self, *args):
        listNode = self.navList.getCurrent()
        widget = listNode.getData()[self.WIDGET_ENTRY]
        print "Will forward to ", widget
        dialog = FindDialog.FindDialog(self, widget, self)
        return None

    def addBookmark(self, *args):
        """Callback: take the current content frame and add the resource
        associated to it to the bookmark list.  The new bookmark is appended
        at the end of the bookmarks menu"""

        listNode = self.navList.getCurrent()
        resource = listNode.getData()[self.RESOURCE_ENTRY]

        print "Bookmarking with resource\n%s" % resource.toString()
        foo = Bookmark.Bookmark(resource)

        # Save the bookmark...
        self.bookmarks.insert(ListNode.ListNode(foo))
        
        def fn(self=self, bookmark=foo, *args):
            self.goElsewhere(bookmark)
            return None
        
        self.bookmarksmenu.add_command(label=foo.toString(),
                                       command=fn)
        self.saveBookmarks()
        return None

    def getPrefsDirectory(self):
        return Options.program_options.getOption('prefs_directory')

    def loadOptions(self, filename=None):
        if filename is None:
            filename = self.getPrefsDirectory() + os.sep + "forgrc"

        try:
            Options.program_options.parseFile(filename)
        except IOError, errstr:
            print "**** Couldn't parse options at %s: %s" % (filename, errstr)
            return None
        print "****Successfully loaded options from disk."
        return 1
    
    def saveOptions(self, *args):
        """Saves the user options to a file in their home directory.  Who knows
        what happens on windows boxen."""

        filename = self.getPrefsDirectory() + os.sep + "forg-assocations"
        
        # p = pickle.Pickler(filename)
        str = pickle.dumps(self.associations)

        try:
            fp = open(filename, "w")
            fp.write(str)
            fp.flush()
            fp.close()
        except IOException, str:
            self.genericError("Couldn't write options to\n%s:\n%s" % (filename,
                                                                      str))

        ofilename = self.getPrefsDirectory() + os.sep + "forgrc"
        
        try:
            fp = open(ofilename, "w")
            fp.write(Options.program_options.toString())
            fp.flush()
            fp.close()
        except IOException, str:
            self.genericError("Couldn't write options to\n%s:\n%s" (ofilename,
                                                                    str))

        self.genericError("Finished saving options.")
        return None

    def loadBookmarks(self, *args):
        self.bookmarks = List.List() # Create a new empty list
        filename = self.getPrefsDirectory() + os.sep + "bookmarks"
        
        try:
            fp = open(filename, 'r')
            line = fp.readline()
            while line != '':
                # Bookmarks are one per line.
                bmark = Bookmark.parseBookmark(line)
                # Read the next line in the file.
                line = fp.readline()   # Next line in file.
                if bmark is None:
                    print "****Bookmark Syntax error: \"%s\"" % line
                    continue
                # Add bookmark to the list.
                print "Inserting bookmark:  %s" % bmark.toData()
                self.bookmarks.insert(ListNode.ListNode(bmark), None)
        except IOError, errstr:
            print "****Couldn't load bookmarks at %s: %s" % (filename, errstr)
            return None
        
        print "****Bookmarks successfully loaded from disk."
        return 1
    
    def saveBookmarks(self, *args):
        filename = self.getPrefsDirectory() + os.sep + "bookmarks"

        try:
            fp = open(filename, "w")

            def traverse_function(data_item):
                return data_item.getData().toData()
            fp.write(string.join(self.bookmarks.traverse(traverse_function),
                                 "\n"))
            fp.flush()
            fp.close()
        except IOError, errstr:
            print "****Couldn't save bookmarks to %s: %s" % (filename, errstr)
            return None
        print "****Bookmarks successfully saved to %s" % filename
        return 1

    def externalLauncherThread(self, *args):
        """This sits and does nothing until a certain variable is set.  It
        then launches the file associated with that variable with the
        application specified."""
        while 1:
            time.sleep(0.5)
            # print "I'm not sleepy."

            if self.LAUNCH_ME == "DIE":
                print "I'm outta here.  External Launcher Thread."
                return
                
            if self.LAUNCH_ME != None:
                print "externalLauncherThread to launch: %s" % (string.join(self.LAUNCH_ME, ":"))
                [filename, assoc] = self.LAUNCH_ME
                self.associations.applyAssociation(filename, assoc)
                self.LAUNCH_ME = None
                
        return None
    
    def downloadThread(self, *args):
        """This sits there doing nothing until a certain variable is set.  It
        then fetches that resource through the network.  This is done in a
        different thread, because otherwise the GUI locks while the data is
        being downloaded.  When it is done, it will insert the new content
        into the window, and add the completed node to the list of active
        nodes."""
        while 1:
            time.sleep(0.5)

            if self.DOWNLOAD_ME == "DIE":
                print "I'm outta here.  Download Thread."
                return

            if self.DOWNLOAD_ME != None:            # We should do something...
                if not self.CURRENTLY_DOWNLOADING:  # Go download this.
                    thread.start_new_thread(self.download_function,
                                            (self.DOWNLOAD_ME, None))
                    request = self.DOWNLOAD_ME
                    self.DOWNLOAD_ME = None
                else:
                    r = self.DOWNLOAD_ME
                    name = "%s:%s %s" % (r.getHost(), r.getPort(), r.getName())
                    self.genericError("Cannot load\n%s:\nAlready working" %
                                      name)
                    self.DOWNLOAD_ME = None
        return None

    def download_function(self, resource, *args):
        request = resource
        self.CURRENTLY_DOWNLOADING = 1
        # try:
        self.messageBar.message('state', "Got request...")
        contentKey = self.createContentFrame(request)
        self.changeContent(contentKey[0])
        self.navList.insert(ListNode.ListNode(contentKey))
        # except Exception, es:
        #    self.genericError("Could not download resource:\n%s" % es)
        self.CURRENTLY_DOWNLOADING = 0
            
    def go(self, *rest):
        """This is what happens when the 'Go' button is clicked.  Information
        about the host, port, locator is fetched from the entry boxes, and
        the program goes there.  (Or tries, anyway)"""
        host    = self.hostEntry.get()
        port    = self.portEntry.get()
        locator = self.resourceEntry.get()

        if locator == '':
            locator = "/"

        if port == '' or port < 0:
            port = 70
        
        res  = GopherResource.GopherResource('1', host, port,
                                             locator, "%s Root" % host)
        self.goElsewhere(res)

    def goElsewhere(self, resource, usecache=1, *args):
        """This is a wrapper for dealing with the download thread.  Pass it
        one resource as an argument, and that resource will be downloaded in
        the background.  Optionally, if the file exists in the cache, it will
        be loaded from there if usecache is true."""

        response = None
        # Attempt to load the response object out of the cache.
        if usecache:
            self.messageBar.message('state',
                                    "Looking for document in cache...")
            response = self.cache.uncache(resource)

        # This will be true if usecache is false and we should reload the
        # document in question.
        if not response:
            self.messageBar.message('state',
                                    "Document not in cache.  Fetching from " +
                                    "network.")
            # We need to fetch this document from the network.
            # Signal the download thread to go to work, and get out.
            self.DOWNLOAD_ME = resource
            return
        else:
            self.messageBar.message('state',
                                    "Loading document from cache to display.")
            wid = self.responseToWidget(response, resource)
            # print "goElsewhere: got RTW"
            insertable = [wid, resource, response]

            self.navList.insert(ListNode.ListNode(insertable))
            self.changeContent(wid)
            self.messageBar.message('state', "Done") 
        return None

    def createContentFrame(self, resource):
        """Given a resource, this fetches it from the network and sets up
        the necessary widgets."""
        conn = GopherConnection.GopherConnection()
        msg = "Done"
        try:
            self.messageBar.message('state',
                                    "Creating connection and getting data...")
            resp = conn.getResource(resource, self.messageBar)
        except socket.error, err:
            try:
                if len(err) >= 2:
                    msg = "Can't fetch: errcode %d: %s" % (err[0], err[1])
                else:
                    msg = "Can't fetch: error %s" % err[0]
            except AttributeError:  # This is really weird, don't ask.
                msg =  "Can't fetch - unknown error."

            return [None, None, None]

        if resp.getError() != None:
            self.genericError(resp.getError())
            self.messageBar.message('state',
                                    'Error fetching: %s' % resp.getError())
            return [None, None, None]
            
        wid  = self.responseToWidget(resp, resource)
        # print "CreateContentFrame got RTW"
        self.messageBar.message('state', msg)
        return [wid, resource, resp]

    def saveFileFilename(self, bclicked):
        """Callback: this gets called when the file entry dialog button gets
        clicked.  A filename may be retrieved from self.saveFileDialog and if
        so, the content of the current frame is saved into that filename.  In
        any case, the dialog is destroyed."""
        
        if bclicked == 'OK' or bclicked != 'Cancel':
            filename = self.saveFileDialog.get()
            print "Saving data into %s" % filename

            listNode = self.navList.getCurrent()
            response = listNode.getData()[self.RESPONSE_ENTRY]

            data = response.getData()
            if data == None:
                resps = response.getResponses()
                for resp in resps:
                    data = "%s%s" % (data, resp.toProtocolString())

            try:
                fp = open(filename, "w")
                fp.write(data)
                fp.flush()
                fp.close()
            except IOError, errstr:
                self.genericError("Couldn't write data to\n%s:\n%s" %
                                  (filename, errstr))

        # In any case...
        self.saveFileDialog.destroy()
        self.saveFileDialog = None

        return None
    
    def saveFile(self, *args):
        """Pop up a dialog box telling the user to choose a filename to save
        the file as."""

        if self.saveFileDialog != None:
            return None
        
        self.saveFileDialog = Pmw.PromptDialog(self,
                                               title='Enter Filename:',
                                               label_text='Enter Filename:',
                                               entryfield_labelpos='n',
                                               buttons=['OK', 'Cancel'],
                                               command=self.saveFileFilename)
        Label(self.saveFileDialog.interior(),
              text="Sure sucks that I don't know how").pack()
        Label(self.saveFileDialog.interior(),
              text="to do browse boxes yet with Tk.").pack()
        return None

    def popupMenu(self, event):
        """Display pop-up menu on right click on a message"""
        self.popup.tk_popup(event.x_root, event.y_root)
        
    def createPopup(self):
        """Pop-up menu on right click on a message"""
        self.popup = Menu(self)
        self.popup['tearoff'] = FALSE
        self.popup.add_command(label='Save', command=self.saveFile)
        self.popup.add_command(label='Back', command=self.goBackward)
        self.popup.add_command(label='Forward', command=self.goForward)

    def setLocation(self, resource):
        """Takes a resource, and sets the location information at the top
        of the screen to the information in the resource.  If you're going to
        a certain location, this gets called to update what the user sees as
        the location."""
        self.hostEntry.delete(0, 'end')
        self.resourceEntry.delete(0, 'end')
        self.portEntry.delete(0, 'end')

        self.hostEntry.insert('end', resource.getHost())
        self.resourceEntry.insert('end', resource.getLocator())
        self.portEntry.insert('end', resource.getPort())
        return None

    def responseToWidget(self, resp, resource):
        """Takes a response object and turns it into a graphical box."""
        if self.verbose:
            print "Resource is %s" % resource

        self.setLocation(resource)

        # Truncate the name of the locator to a max of 50 chars.

        self.messageBar.message('state', 'Updating display...')

        try:
            filename = self.cache.cache(resp, resource)
        except CacheException, exceptionstr:
            self.genericError(exceptionstr)

        if resp.getTypeCode() == RESPONSE_INDEXS:
          foo = GUISearch.GUISearch(self.CONTENT_BOX, self,
                                    resp, resource, filename)
          return foo
        elif resp.getData() != None:   # There is data, display a scrolled text
            # print "DATA IS"
            # print resp.getData()

            if filename != None:
                assoc = self.associations.getAssociation(filename)
                
                if assoc != None:
                    self.LAUNCH_ME = [filename, assoc]

            if resp.getTypeCode() == RESPONSE_FILE:
                filebox = GUIFile.GUIFile(self.CONTENT_BOX, self,
                                          resp, resource, filename)
            else:
                filebox = GUISaveFile.GUISaveFile(self.CONTENT_BOX,
                                                  self, resp, resource,
                                                  filename)
                
            return filebox
        else:  # There is no data, display a directory entry
            menuAssocs = {
                "Backward"   : self.goBackward,
                "Forward"    : self.goForward,
                "About FORG" : self.about,
                "Quit"       : self.quit }

            dirbox  = GUIDirectory.GUIDirectory(self.CONTENT_BOX, self, 
                                                resp, resource, filename,
                                                menuAssocs)
            
            return dirbox

    def changeContent(self, newwid):
        """Changes the current main content of the window to newwid"""
        if self.currentContent != None:
            # print "CC: Destroying current..."
            # self.currentContent.withdraw()
            # print "CC: hidden."
            self.currentContent.destroy()
            # print "CC: destroyed"

        if newwid != None:
            # print "CC: packing"
            newwid.pack(expand=1, fill='both')
            # print "CC: packed."

        # print "CC: Reassinging current"
        self.currentContent = newwid
        # print "CC: returning"
        return self.currentContent

    def reload(self, *rest):
        listNode = self.navList.getCurrent()
        resource = listNode.getData()[self.RESOURCE_ENTRY]

        # Load this page onto the screen, the None is to avoid loading the
        # file from cache (even if it exists)
        self.goElsewhere(resource, None)
        
        return None

    def goForward(self, *rest):
        """Navigate forward"""
        try:
            node = self.navList.getNext()
            data = node.getData()
            # print "response is %s" % data[self.RESPONSE_ENTRY]
            # print "resource is %s" % data[self.RESOURCE_ENTRY]
            wid  = self.responseToWidget(data[self.RESPONSE_ENTRY],
                                         data[self.RESOURCE_ENTRY])
            print "goForward got RTW"
            self.changeContent(wid)
        except List.ListException, errstr:
            print "Couldn't get next: %s" % errstr
        return None
    
    def goBackward(self, *rest):
        """Navigate backward."""
        try:
            node = self.navList.getPrev()
            data = node.getData()
            # print "response is %s" % data[self.RESPONSE_ENTRY]
            # print "resource is %s" % data[self.RESOURCE_ENTRY]
            wid  = self.responseToWidget(data[self.RESPONSE_ENTRY],
                                         data[self.RESOURCE_ENTRY])
            print "goBackward: got RTW"
            node.setData([wid,
                          data[self.RESOURCE_ENTRY],
                          data[self.RESPONSE_ENTRY]])
            self.changeContent(wid)
        except List.ListException, errstr:
            print "Couldn't get prev: %s" % errstr

    def about(self, *args):
        """Display the about box."""
        print "About Gopher"
        Pmw.aboutversion(self.VERSION)
        Pmw.aboutcopyright('Copyright 2000')
        Pmw.aboutcontact(
            'This program is licensed under the GNU General Public License\n' +
            'For more information, please see\n' +
            'http://www.gnu.org/')
        dialog = Pmw.AboutDialog(self, applicationname='FORG')
        # PhotoImage(dialog.component('dialogchildsite'),
        #            file='forg.ppm', format='ppm').pack()
        dialog.show()
        return None

    def quit(self, *args):
        """Quit the entire program.  Caller may have something after this."""
        self.destroy()

    def addEssentials(self, pmenu):
        emenu = Menu(pmenu)

        floodgate = GopherResource.GopherResource(RESPONSE_DIR,
                                                  "gopher.floodgap.com",
                                                  70, "/",
                                                  "Floodgap Home",
                                                  [])
        heatdeath = GopherResource.GopherResource(RESPONSE_DIR,
                                                  "gopher.heatdeath.org",
                                                  70, "/",
                                                  "Heatdeath Organization",
                                                  [])
        quux      = GopherResource.GopherResource(RESPONSE_DIR,
                                                  "gopher.quux.org",
                                                  70, "/",
                                                  "Quux.org", [])
        umn       = GopherResource.GopherResource(RESPONSE_DIR,
                                                  "gopher.tc.umn.edu",
                                                  70, "/",
                                                  "UMN Home", [])
        veronica  = GopherResource.GopherResource(RESPONSE_INDEXS,
                                                  "gopher.floodgap.com",
                                                  70, "/v2/vs",
                                                  "Veronica-2 Search", [])
        
        # Function to use as a callback.
        for item in [floodgate, heatdeath, quux, umn, veronica]:
            bmark = Bookmark.Bookmark(item)
            def fn(self=self, bookmark=bmark, *args):
                self.goElsewhere(bookmark)
                return None
            
            emenu.add_command(label=bmark.toString(),
                              command=fn)

        pmenu.add_cascade(label="Essential Bookmarks", menu=emenu)
    
    def make_menus(self):
        """Create the menuing system"""
        self.menu = Menu(self.mainBox)
        self.filemenu  = Menu(self.menu)
        self.filemenu.add_command(label='Save', command=self.saveFile)
        self.filemenu.add_command(label='Quit', command=self.quit)

        self.editmenu = Menu(self.menu)
        self.editmenu.add_command(label="Find", command=self.find)

        self.navmenu   = Menu(self.menu)
        self.navmenu.add_command(label="Forward",  command=self.goForward)
        self.navmenu.add_command(label="Backward", command=self.goBackward)
        self.navmenu.add_command(label="Reload",   command=self.reload)

        self.optionsmenu = Menu(self.menu)
        self.optionsmenu.add_command(label='Associations',
                                     command=self.editAssociations)
        self.optionsmenu.add_command(label="Save Options/Associations",
                                     command=self.saveOptions)
        self.optionsmenu.add_command(label="Reload Options/Associations",
                                     command=self.loadOptions)

        self.bookmarksmenu = Menu(self.menu)
        self.bookmarksmenu.add_command(label="Bookmark this page",
                                       command=self.addBookmark)
        
        def doNothing(*args):
            return None

        self.bookmarksmenu.insert_separator('end')

        self.addEssentials(self.bookmarksmenu)
        
        # Add the bookmarks in the correct data structure
        def traverse_function(data_item, self=self):
            bmark = data_item.getData()

            # Function to use as a callback.
            def fn(self=self, bookmark=bmark, *args):
                self.goElsewhere(bookmark)
                return None

            self.bookmarksmenu.add_command(label=bmark.toString(),
                                           command=fn)

        self.bookmarks.traverse(traverse_function)
        
        self.helpmenu = Menu(self.menu)
        self.helpmenu.add_command(label='About', command=self.about)
        self.helpmenu.add_command(label='Statistics', command=self.stats)
        
        self.menu.add_cascade(label="File", menu=self.filemenu)
        self.menu.add_cascade(label="Edit", menu=self.editmenu)
        self.menu.add_cascade(label="Navigation", menu=self.navmenu)
        self.menu.add_cascade(label="Bookmarks", menu=self.bookmarksmenu)
        self.menu.add_cascade(label="Options", menu=self.optionsmenu)
        self.menu.add_cascade(label="Help", menu=self.helpmenu)


# END TkGui class

