# GopherResponse.py
# Contains GopherResource and GopherResponse objects
# Written by David Allen <s2mdalle@titan.vcu.edu>
# Released under the terms of the GNU General Public License
#
# This object holds the data corresponding to how a gopher server responded
# to a given request.  It holds one of two things in general, either a pile
# of data corresponding to text, a gif file, whatever, or it holds a list of
# GopherResource objects.  (This is when it's a directory entry that's being
# stored.
##############################################################################
import regsub
from string              import *
from urlparse            import *
from gopher              import *
import Connection
import GopherConnection
import GopherThingy
import GopherObject
import GopherResource
import ResourceInformation
import Options

GopherResponseException = "Gopher response error"

class GopherResponse(GopherObject.GopherObject):
    verbose = None
    def __init__(self, type=None, host=None, port=None, loc=None, name=None):
        GopherObject.GopherObject.__init__(self, type, host, port, loc, name)
        self.__class = "GopherResponse"
        self.data = None
        self.responses = []
        return None
    def getResponses(self):
        """Return a list of responses.  This is only really good when the
        response was a directory and set of entries."""
        if self.getData() != None:
            raise GopherResponseException, "Get data instead."
        return self.responses
    def getData(self):
        """Return the data associated with the response.  This is usually all
        of the data off of the socket except the trailing closer."""
        return self.data
    def getDataLength(self):
        return len(self.data)
    def getDataChunk(self, startIndex, endIndex=-1):
        """This method provides a way to get chunks of data at a time,
        rather than snarfing the entire ball."""
        
        if endIndex == -1:
            endIndex = len(self.data)
        return self.data[startindex, endIndex]
    def getError(self):
        """If there was an error message, return it."""
        try:
            return self.error
        except:
            return None
    def setError(self, errorstr):
        """Modify the error message within this response."""
        self.error = errorstr
    def setData(self, data):
        """Modify the data within the response.  You probably don't want to
        do this."""
        self.data = data
        return None
    def looksLikeDir(self, data):
        """Takes a chunk of data and returns true if it looks like directory
        data, and false otherwise.  This is tricky, and is of course not 100%.
        Basically, we look at each line, see if the first bit in the line is a
        legal type, check to see that the line has at least 3 tabs in it.  If
        all of the first 20 lines of the data follow that rule, then it's good
        enough to be used as directory data.  If not, it gets chucked.  Notice
        that if this really is a directory but it's using file types we've
        never heard of (see gopher.py) then it will still get thrown out.
        Bummer.  This should only be called anyway if the type indictator is
        incorrect, so cope.  :)"""
                
        lines = split(data, "\n", 20)

        for line in lines:
            if count(line, "\t") < 3:
                return None               # Not enough tabs.  Bummer.
            byte = line[0]
            try:
                resp = responses[byte]
            except:
                continue
            try:
                resp = errors[byte]
            except:
                return None

        return 1     # Matched all tests for 20 lines.  Looks OK.  Maybe.
    
    def parseResponse(self, data):
        """Takes a lump of data, and tries to parse it as if it was a directory
        response.  It will set the responses array to the proper thing if the
        result was good, so that you can use self.getRepsonses() to access
        them.  Otherwise it raises GopherResponseException"""
        
        self.responses = []

        if self.type != RESPONSE_DIR:
            if self.looksLikeDir(data):       # Looks good?
                self.type = RESPONSE_DIR      # Treat it like a directory.
            else:
                raise GopherResponseException, "This isn't a directory."

        self.lines = split(data, "\r\n")

        for line in self.lines:
            if len(line) <= 1:
                continue

            # Type of the resource.  See gopher.py for the possibilities.
            stype = "%s" % line[0]

            # Gopher protocol uses tab delimited fields...
            linedata = split(line[1:], "\t")
            name    = "Unknown"              # Silly defaults
            locator = "Unknown"
            host    = "Unknown"
            port    = 70

            try:
                name = linedata[0]           # Assign the right things in the
            except IndexError: pass          # right places (maybe)
            try:                             # Catch those errors coming from
                locator = linedata[1]        # the line not being split into 
            except IndexError: pass          # enough tokens.  Realistically,
            try:                             # if those errors happen,
                host = linedata[2]           # something is really screwed up
            except IndexError: pass          # and there's not much we can do
            try:                             # anyway.  
                port = linedata[3]
            except IndexError: pass

            try:
                remainder = linedata[4:]     # Extra Gopher+ fields.
            except:
                remainder = []


            # UMN gopherd errors do this sometimes.  It's quite annoying.
            # they list the response type as 'directory' and then put the host
            # as 'error.host' to flag errors
            if host == 'error.host' and stype != RESPONSE_BLURB:
                stype = RESPONSE_ERR
                
            newresource = GopherResource.GopherResource(stype, host,
                                                        port, locator, name,
                                                        remainder)
            
            # Are the options set to allow us to get info?
            # THIS SHOULD BE USED WITH CAUTION since it can slow things down
            # more than you might think.
            if Options.program_options.getOption('grab_resource_info'):
                if len(remainder) >= 1 and remainder[0] == '+':
                    try:
                        conn = GopherConnection.GopherConnection()
                        info = conn.getInfo(newresource)
                        newresource.setInfo(info)
                    except GopherConnectionException, estr:
                        print "***(GCE) can't get info: %s" % estr
                    except Exception, estr:
                        print "***(unknown) Couldn't %s %s" % (Exception,estr)

            self.responses.append(newresource)
        return None
# End GopherResponse
