# GUIDirectory.py
# Written by David Allen <s2mdalle@titan.vcu.edu>
#
# Graphical representation of directory views.
########################################################################

from gopher import *
from Tkinter import *
import Pmw
from string import *

import GopherResource
import GopherResponse
import GopherConnection
import ContentFrame
import Cache
import Options

class GUIDirectory(ContentFrame.ContentFrame, Frame):
    verbose = None
    DEFAULT_COLOR = "#000000"
    FOUND_COLOR   = "#00FF00"
    LINK_COLOR    = "#0000FF"
    ACTIVE_COLOR  = "#FF0000"
    
    def __init__(self, parent_widget, parent_object, resp,
                 resource, filename=None, menuAssocs={}):
        Frame.__init__(self, parent_widget)  # Superclass constructor
        self.searchlist = []    # Searchable terms...
        self.parent = parent_object
        self.cursel = None
        self.menuAssocs = menuAssocs
        self.createPopup()
        
        labeltext = "%s:%d" % (resource.getHost(), int(resource.getPort()))
        
        if resource.getName() != '' and resource.getLocator() != '':
            label2 = "\"%s\" ID %s" % (resource.getName(),
                                       resource.getLocator())
        else:
            label2 = "    "

        if len(label2) > 50:
            label2 = label2[0:47] + "..."

        Label(self, text=labeltext).pack(side='top', expand=0, fill='x')
        Label(self, text=label2).pack(side='top', expand=0, fill='x')
        scrolled_window = Pmw.ScrolledFrame(self,
                                            hscrollmode='dynamic',
                                            vscrollmode='static')
        scrolled_window.pack(fill='both', expand=1)
        sbox = Frame(scrolled_window.interior())
        sbox.pack(side='top', expand=1, fill='both')

        print "Responses..."
        responses = resp.getResponses()
        
        for x in range(0, len(responses)):
            r = responses[x]
            if self.verbose:
                print "Response: %s" % r.getName()
            l = Label(sbox, text=r.getType())
            
            if r.getTypeCode() == RESPONSE_BLURB:
                blurb_label = Label(sbox, foreground=self.DEFAULT_COLOR,
                                    text=r.getName())
                blurb_label.grid(row=x, columnspan=3,
                                 col=1, sticky=W)
                self.searchlist.append([r.getName(), blurb_label,
                                        self.DEFAULT_COLOR])
                
            else:
                # Trick Tk into passing arguments with the function and
                # get around Python's weird namespacing.
                def fn(event, self=parent_object, r=r, *args):
                    self.goElsewhere(r)
                    return None
                        
                def color_widget(event, self=self, *args):
                    wid = event.widget
                    wid.configure(foreground=self.ACTIVE_COLOR)
                    return None

                def dopopup(event, resource=r, self=self):
                    # This binds resource to a parameter that
                    # one of several popup commands will use.
                    return self.popupMenu(event, resource)
                
                # Don't make it clickable if it's an error.  But if it
                # isn't an error, connect these signals.
                if r.getTypeCode() != RESPONSE_ERR:
                    default_color = self.LINK_COLOR
                    b = Label(sbox, foreground=self.LINK_COLOR,
                              text=r.getName())
                    b.bind('<ButtonRelease-1>', fn)
                    b.bind('<Button-1>', color_widget)
                    b.bind('<Button-3>', dopopup)
                else:
                    default_color = self.DEFAULT_COLOR
                    b = Label(sbox, foreground=self.DEFAULT_COLOR,
                              text=r.getName())


                # Each entry in the searchlist is the name of a widget as a
                # string, the widget itself, and the widget's default color
                # The color is needed because 'finding' things requires
                # changing their color.
                self.searchlist.append([r.getName(), b, default_color])
                
                l.grid(row=x, col=0, sticky=W)
                b.grid(row=x, col=1, columnspan=2, sticky=W)

                cacheobj = Cache.Cache()
                
                optname = 'display_info_in_directories'
                if Options.program_options.getOption(optname):
                    if r.getInfo() != None:
                        i = r.getInfo()
                        t = i.getAdmin()
                        
                        if len(t) > 40:
                            t = t[0:40]
                            Label(sbox, text=t).grid(row=x, col=3,
                                                     columnspan=2)
                        else:
                            Label(sbox, text=t).grid(row=x, col=3,
                                                     columnspan=2)
                    else:
                        Label(sbox, text="  ").grid(row=x, col=3,
                                                    columnspan=2)

                # Possibly report to the user whether or not a given file is
                # present in cache.  I like to know this, but some people
                # don't, so make it a settable option.
                if Options.program_options.getOption('show_cached'):
                    if cacheobj.isInCache(r):
                        Label(sbox, text="Cached").grid(row=x, col=5)
                    else:
                        Label(sbox, text=" ").grid(row=x, col=5)
                        
        return None
        # End __init__

    def createPopup(self):
        """Pop-up menu on right click on a message"""
        self.popup = Menu(self)
        self.popup['tearoff'] = FALSE
        self.popup.add_command(label='Info', command=self.infoPopup)
        self.popup.add_command(label='Cache Status', command=self.cacheStatus)

        for key in self.menuAssocs.keys():
            self.popup.add_command(label=key,
                                   command=self.menuAssocs[key])

    def popupMenu(self, event, item):
        """Display pop-up menu on right click on a message"""
        self.cursel = item
        self.popup.tk_popup(event.x_root, event.y_root)
        return None
    
    def cacheStatus(self, resource=None):
        c = Cache.Cache()
        if resource is None:
            resource = self.cursel
            
        resp = c.isInCache(resource)

        if resp is None:
            str = "Resource\n%s\nis not in the cache." % resource.toURL()
            self.parent.genericError(str, "Cache Status")
        else:
            url = resource.toURL()
            [filename, size] = resp
            str = "%s\nis in the cache as\n%s\n and is %s bytes" % (url,
                                                                    filename,
                                                                    size)
            self.parent.genericError(str, "Cache Status")
        return None

    def infoPopup(self, resource=None):
        if resource is None:
            resource = self.cursel
        
        if resource.getInfo() is None:
            try:
                conn = GopherConnection.GopherConnection()
                info = conn.getInfo(resource)
                resource.setInfo(info)
            except Exception, errstr:
                url = resource.toURL()
                str = "Cannot display information about\n%s:\n%s" % (url,
                                                                     errstr)
                self.parent.genericError(str, "Error:")
                return None

        info = resource.getInfo()
        str = "Resource information:\n%s\n%s" % (resource.toURL(),
                                                 info.toString())
        self.parent.genericError(str, "Information:")
        return None
            
    def find(self, term, caseSensitive=None, lastIdentifier=None):
        """Overrides the same definition in ContentFrame.ContentFrame"""

        if lastIdentifier is None:  # Note this is a distinct case from when 
            self.lastIndex = -1     # lastIdentifier is false (possibly 0)
        else:
            # This could be 0 or any positive number.
            self.lastIndex = lastIdentifier

        try:
            self.lastIndex = int(self.lastIndex) # Unstringify
        except:
            raise(Exception,
                  "Illegal last ID passed to %s.find()" % self.__class__)

        print "LAST FOUND INDEX was %s." % self.lastIndex
        
        # Bounds checking on lastIndex...
        if self.lastIndex < -1 or self.lastIndex > len(self.searchlist):
            print "*****Something's messed up.  Bad lastIdentifier."
        elif self.lastIndex >= 0:
            [old_label, old_widget, color] = self.searchlist[self.lastIndex]
            old_widget.configure(foreground=color)

        found      = None
        foundIndex = -1
        
        for x in range(self.lastIndex+1, len(self.searchlist)):
            [label, label_widget, color] = self.searchlist[x]
            print "Looking at index %d through \"%s\"..." % (x, label)
            if not caseSensitive:
                # If we're not doing case-sensitive compares, convert them
                # both to lower case and we'll compare them that way.
                label = lower(label)
                term  = lower(term)
                
            if find(label, term) != -1:
                # Term was found in this label
                foundIndex = x
                found = 1
                # Find only one match per call to this function.  Bail out.
                break

        if found:
            [found_label, found_widget, color] = self.searchlist[foundIndex]

            print "Found \"%s\" in \"%s\" at index %d" % (term,
                                                          found_label,
                                                          foundIndex)
            # Turn the one that was found bright green.
            found_widget.configure(foreground=self.FOUND_COLOR)
            print "RETURNING NEW FOUND INDEX: %s" % foundIndex
            return foundIndex
        else:
            return None  # This signals that it wasn't found, and to reset

        # Everything else this class needs gets inherited from other classes.
