# Cache.py
# Written by David Allen <s2mdalle@titan.vcu.edu>
# Released under the terms of the GNU General Public License
#
# Handles cache-file related operations.
import os

from gopher         import *
import GopherResource
import GopherResponse
import Options

CacheException = "Error"

class Cache:
    def __init__(self, *args):
        self._d = None
        return None

    def getCacheDirectory(self):
        try:
            dir = Options.program_options.getOption('cache_directory')
            return os.path.abspath(dir)
        except:
            return os.path.abspath("cache")

    def getCachePrefix(self):
        try:
            dir = Options.program_options.getOption('cache_prefix')
            return os.path.abspath(dir)
        except:
            return os.path.abspath("cache%scache" % os.sep)

    def getCacheStats(self):
        cdir = self.getCacheDirectory()
        
        def stat_it(filename, cdir=cdir):
            fullpath = "%s%s%s" % (cdir, os.sep, filename)
            info = os.stat(fullpath)
            return info[6]

        # Get a list of all file sizes of all files in the directory.
        # This does NOT recurse into subdirectories.
        flist = map(stat_it, os.listdir(cdir))

        sum = float(0)
        
        for num in flist:
            sum = sum + float(num)

        kbtotal = sum / float(1024)
        mbtotal = sum / (float(1024) * float(1024))
        
        return "There are\n%s Kb\n(%s MB)\nin %s files in cache." % (
            kbtotal,
            mbtotal,
            len(flist))

    def isInCache(self, resource):
        """Takes a resource, and returns true if the resource seems to have
        an available cache file associated with it, and None otherwise."""
        filename = self.getCachePrefix() + resource.toFilename()
        
        try:
            info = os.stat(os.path.abspath(filename))
            return [os.path.abspath(filename), info[6]]
        except OSError:
            return None
        return None
        
    def uncache(self, resource):
        """Takes a resource, and returns either None if the given resource
        is not cached on disk, or it returns a GopherResponse corresponding
        to what would have been gotten if it had been fetched from the
        server."""
        
        filename = self.getCachePrefix() + resource.toFilename()

        try:
            # See if the file exists...
            tuple = os.stat(filename)
            print "File %s of size %d exists." % (filename, tuple[6])
        except OSError:
            # The file doesn't exist, we can't uncache it.
            return None

        resp = GopherResponse.GopherResponse()
        resp.setType(resource.getTypeCode())
        resp.setHost(resource.getHost())
        resp.setPort(resource.getPort())
        resp.setLocator(resource.getLocator())
        resp.setName(resource.getName())
        
        try:
            fp = open(filename, "r")

            # Consider reworking this somehow.  Slurp the entire file into
            # buffer.
            buffer = fp.read()
            fp.close()

            try:
                resp.parseResponse(buffer)
                resp.setData(None)
                # print "Loaded cache is a directory entry."
            except:
                # print "Loaded cache is not a directory."
                resp.setData(buffer)
                
            # Got it!  Loaded from cache anyway...
            # print "UNCACHE found data for use."
            return resp       
        except IOError, errstr:
            raise CacheException, "Couldn't read data on\n%s:\n%s" % (filename,
                                                                      errstr)

        # We failed.  Oh well...
        return None
    
    def cache(self, resp, resource):
        """Takes a GopherResponse and a GopherResource.  Saves the content of
        the response to disk, and returns the filename saved to."""
        filename = self.getCachePrefix() + resource.toFilename()
        try:
            fp = open(filename, "w")

            if resp.getData() == None:    # This is a directory entry.
                response_lines = resp.getResponses()
                # Each response line is a GopherResource
                # write them as if it was a file served by the gopher server.
                # that way it can be easily reparsed when loaded from the
                # cache.
                for response_line in response_lines:
                    fp.write(response_line.toProtocolString())

                # write the string terminator.  This isn't really needed
                # since it isn't data, but it helps fool our other objects
                # into thinking that it's dealing with data off of a socket
                # instead of data from a file.  So do it.
                fp.write("\r\n.\r\n") 
            else:
                fp.write(resp.getData())

            fp.flush()
            fp.close()
        except IOError, errstr:
            # Some error writing data to the file.  Bummer.
            raise CacheException, "Couldn't write to\n%s:\n%s" % (filename,
                                                                  errstr)
        # Successfully wrote the data - return the filename that was used
        # to save the data into.  (Absolute path)
        return os.path.abspath(filename)
        

