// protoCommon.js: shared JS functions for Protozilla
// Uses:
//   chrome://global/content/globalOverlay.js
//   chrome://global/content/nsDragAndDrop.js
//   chrome://global/content/nsJSSupportsUtils.js
//   chrome://global/content/nsJSComponentManager.js
//   chrome://global/content/nsTransferable.js


const nsILocalFile = Components.interfaces.nsILocalFile;

const NS_PROTOZILLA_CONTRACTID =
    "@mozilla.org/protozilla/protozilla;1";

const NS_IPCSERVICE_CONTRACTID =
    "@mozilla.org/process/ipc-service;1";

const NS_PROCESSINFO_CONTRACTID =
    "@mozilla.org/xpcom/process-info;1";

var pzilla = Components.classes[NS_PROTOZILLA_CONTRACTID].createInstance();
window.pzilla = pzilla.QueryInterface(Components.interfaces.nsIProtozilla);

///////////////////////////////////////////////////////////////////////////////
// File read/write operations

const NS_LOCAL_FILE_CONTRACTID = "@mozilla.org/file/local;1";
const NS_LOCALFILEOUTPUTSTREAM_CONTRACTID =
                              "@mozilla.org/network/file-output-stream;1";

const NS_RDONLY      = 0x01;
const NS_WRONLY      = 0x02;
const NS_CREATE_FILE = 0x08;
const NS_TRUNCATE    = 0x20;
const DEFAULT_FILE_PERMS = 0600;

function CreateFileStream(filePath, permissions) {

  var localFile = Components.classes[NS_LOCAL_FILE_CONTRACTID].createInstance(Components.interfaces.nsILocalFile);
  localFile.initWithPath(filePath);

  var fileStream = Components.classes[NS_LOCALFILEOUTPUTSTREAM_CONTRACTID].createInstance(Components.interfaces.nsIFileOutputStream);

  if (!permissions)
    permissions = DEFAULT_FILE_PERMS;
  var flags = NS_WRONLY | NS_CREATE_FILE | NS_TRUNCATE;

  fileStream.init(localFile, flags, permissions);

  return fileStream;
}

///////////////////////////////////////////////////////////////////////////////

var gLogLevel = 3;     // Output only errors/warnings by default
var gLogFileStream = null;

function WRITE_LOG(str) {
  dump(str);

  if (gLogFileStream) {
    gLogFileStream.write(str, str.length);
    gLogFileStream.flush();
  }
}

function DEBUG_LOG(str) {
  if (gLogLevel >= 4)
    WRITE_LOG(str);
}

function WARNING_LOG(str) {
  if (gLogLevel >= 3)
    WRITE_LOG(str);
}

function ERROR_LOG(str) {
  if (gLogLevel >= 2)
    WRITE_LOG(str);
}

function CONSOLE_LOG(str) {
  if (gLogLevel >= 3)
    WRITE_LOG(str);

  pzilla.console.write(str);
}

var ipcService;
var gProcessInfo;

try {
  ipcService = Components.classes[NS_IPCSERVICE_CONTRACTID].getService();
  ipcService = ipcService.QueryInterface(Components.interfaces.nsIIPCService);

  gProcessInfo = Components.classes[NS_PROCESSINFO_CONTRACTID].getService();
  gProcessInfo = gProcessInfo.QueryInterface(Components.interfaces.nsIProcessInfo);

  var nspr_log_modules = gProcessInfo.getEnv("NSPR_LOG_MODULES");

  var matches = nspr_log_modules.match(/protoCommon:(\d+)/);

  if (matches && (matches.length > 1)) {
    gLogLevel = matches[1];
    WARNING_LOG("protoCommon: gLogLevel="+gLogLevel+"\n");

    if (gLogLevel >= 4)
       gLogFileStream = CreateFileStream("protdbg2.txt");
  }

} catch (ex) {
}

var gPromptService = Components.classes["@mozilla.org/embedcomp/prompt-service;1"].getService(Components.interfaces.nsIPromptService);

function ProtoAlert(mesg) {
  return gPromptService.alert(window, "Protozilla Alert", mesg);
}

function ProtoConfirm(mesg) {
  return gPromptService.confirm(window, "Protozilla Confirm", mesg);
}

function ProtoError(mesg) {
  return gPromptService.alert(window, "Protozilla Error", mesg);
}

function RemoveOverlay(urlSpec) {
   var overlayFile = window.pzilla.getFileOfProperty("AChrom");
   overlayFile.append("overlayinfo");
   overlayFile.append("communicator");
   overlayFile.append("content");
   overlayFile.append("overlays.rdf");

   DEBUG_LOG("protoCommon.js: RemoveOverlay: url = "+urlSpec+"\n");
   var overlayRemoved = false;
   try {
      var fileContents = window.pzilla.readFileContents(overlayFile, -1);

      var overlayPat = new RegExp("\\s*<RDF:li>\\s*"+urlSpec+"\\s*</RDF:li>");

      var overlayMatch = fileContents.search(overlayPat);

      DEBUG_LOG("protoCommon.js: RemoveOverlay: overlayMatch = "+overlayMatch+"\n");

      if (overlayMatch != -1) {

         var newFileContents = fileContents.replace(overlayPat, "");

         window.pzilla.writeFileContents(overlayFile, newFileContents, 0);

         overlayRemoved = true;
      }
   } catch (ex) {
   }

   return overlayRemoved;
}

function LoadRestrictedURL(urlSpec) {

  var urlComps = urlSpec.split(":");
  if (urlComps[0].search(/\+$/) >= 0) {
    urlSpec = window.pzilla.addPrivilegeToSpec(urlSpec);
  }

  // Load URL in new window
  window.open(urlSpec);

  return;
}

function LoadRestrictedURLFromBar() {
  var resUrlBar = document.getElementById('urlText');
  var urlSpec = resUrlBar.value;

  if (urlSpec.search(/^exec:/) == 0) {
    // ***TESTING*** pzilla.exec
    var cmd = urlSpec.substr(5);
    window.pzilla.console.write("protozilla> "+cmd+"\n");
    window.pzilla.console.write(window.pzilla.execSh(cmd));
    return false;
  }

  LoadRestrictedURL(urlSpec);

  return false;
}

function CopyURL(urlSpec) {
  if (!urlSpec)
    return;

  var resUrlBar = document.getElementById('urlText');
  resUrlBar.value = urlSpec;
}

function GetSelectedTreeCell(treeId) {
  var tree = document.getElementById(treeId);
  var items = tree.selectedItems;

  if (items.length == 0) {
      ProtoAlert("Please select item from list");
      return null;
  }

  return items[0].firstChild.firstChild;
}

// Return first line of a file (after skipping the shell path, if any)
// Returns null string for large files
function GetFirstLine(localFile) {
  if (!localFile.exists || (localFile.fileSize > 20000))
    return "";

  if (localFile.isDirectory())
    return "directory";

  var fileContents = window.pzilla.readFileContents(localFile, 200);

  if (fileContents.search(/^\s*#!/) != -1) {
    // Skip shell path line
    var lineLen = fileContents.indexOf("\n")+1;
    fileContents = fileContents.substr(lineLen);
  }

  // Extract first line
  var offset = fileContents.search(/\r?\n/);
  var firstLine = (offset==-1) ? fileContents
                             : fileContents.substr(0,offset);

  return firstLine.replace(/^\W*/,"");
}

function ProtoFilePicker(title, displayDir, save, filter) {
  const nsIFilePicker = Components.interfaces.nsIFilePicker;
  var filePicker = Components.classes["@mozilla.org/filepicker;1"].createInstance();
  filePicker = filePicker.QueryInterface(nsIFilePicker);

  var mode = save ? nsIFilePicker.modeSave : nsIFilePicker.modeOpen;

  filePicker.init(window, title, mode);

  if (displayDir) {
    var localFile = Components.classes[NS_LOCAL_FILE_CONTRACTID].createInstance(nsILocalFile);

    localFile.initWithPath(displayDir);
    filePicker.displayDirectory = localFile;
  }

  if (filter) {
    filePicker.appendFilter(filter[0], filter[1]);

  } else {
    filePicker.appendFilter("Scripts", "*.pl; *.py; *.js; *.sh; *.csh");
    filePicker.appendFilter("URL redirect", "*.url");
  }

  filePicker.appendFilters(nsIFilePicker.filterAll);

  if (filePicker.show() == nsIFilePicker.returnCancel)
    return null;

  var file = filePicker.file.QueryInterface(Components.interfaces.nsILocalFile);

  return file;
}

// Returns filename on success, null string on failure
function CopyFile(urlSpec, destDir) {

  if (!urlSpec)
    ProtoError("Invalid URL");

  var colonOffset = urlSpec.indexOf(":");
  var schemeName = urlSpec.substr(0,colonOffset).toLowerCase();

  if (schemeName != "file") {
    ProtoError("Only file: URLs can be dragged and dropped\n");
    return "";
  }

  var filePath = urlSpec.substr(colonOffset+1);
  filePath = filePath.replace(/^\/\//,"");

  DEBUG_LOG("protoCommon.js: CopyFile: filePath="+filePath+"\n");

  var sourceFile = Components.classes[NS_LOCAL_FILE_CONTRACTID].createInstance(nsILocalFile);

  sourceFile.initWithPath(filePath);

  if (!sourceFile.exists()) {
    ProtoError("File "+sourceFile.path+" does not exist");
    return "";
  }

  if (sourceFile.isDirectory()) {
    ProtoError("Cannot copy directory "+sourceFile.path);
    return "";
  }

  var targetFile = destDir.clone();
  targetFile.append(sourceFile.leafName);

  if (targetFile.exists()) {
    var confirm = ProtoConfirm("File "+targetFile.path+" already exists."+
                               " Do you wish to overwrite it?");

    if (!confirm)
      return "";
  }

  // Copy file
  var succeeded = false;
  try {
    sourceFile.copyToFollowingLinks(destDir, null);
    succeeded = true;

  } catch (ex) {
    ProtoError("Unable to copy file "+sourceFile.path+
               " to directory "+destDir.path+"\n");
  }

  return succeeded ? sourceFile.leafName : "";
}

function URLDragObserver(dragElement) {
   DEBUG_LOG("protoCommon.js: URLDragObserver()\n");
   this.dragElement = dragElement;
}

URLDragObserver.prototype = {

   onDragStart: function (aEvent)
    {
      var dragElement = this.dragElement;

      var urlString;
      
      var urlAttr = dragElement.getAttribute('urlAttr');
      
      if (urlAttr) {
         urlString = dragElement.getAttribute(urlAttr);
      
      } else {
         urlString = dragElement.getAttribute('dragURL');
      
         if (!urlString)
            urlString = dragElement.getAttribute('href');
      
         if (!urlString)
            urlString = dragElement.getAttribute('src');

         if (!urlString)
            urlString = dragElement.getAttribute('value');
      }
      
      var schemePat = /^[a-z]([a-z0-9+.\-])*:/;
      
      var baseURL = dragElement.getAttribute('baseURL');
      
      if (baseURL && (urlString.search(schemePat) == -1))
         urlString = baseURL + urlString;
      
      if (urlString.search(schemePat) == -1)
         urlString = "";
      
      var textString = dragElement.getAttribute('dragText');
      if (!textString)
         textString = dragElement.getAttribute('value');
      
      var htmlString;
      if (urlString) {
         if (textString)
            htmlString = "<a href='" + urlString + "'>" + textString+ "</a>";
         else
            htmlString = "<a href='" + urlString + "'>" + urlString + "</a>";
      }
      
      if (!textString)
         textString = urlString;
      else if (urlString)
         urlString = urlString + "\n" + textString;
      
      var flavours = { };
      
      if (urlString) 
        flavours["text/x-moz-url"] = { width: 2, data: urlString};
      
      if (htmlString)
        flavours["text/html"]      = { width: 2, data: htmlString };
      
      if (textString)
        flavours["text/unicode"]   = { width: 2, data: textString };
      
      DEBUG_LOG("protoCommon.js: URLDragObserver.onDragStart: urlString='"+
                urlString+"'\n");

      return flavours;
  }
}

function URLDropObserver(URLHandler) {
   this.URLHandler = URLHandler;
}

URLDropObserver.prototype = {
   URLHandler: null,

   onDrop: function (aEvent, aData, aDragSession)
    {
      var urlSpec = RetrieveURLSpecFromDropData(aData);

      DEBUG_LOG("protoCommon.js: URLDropObserver.onDrop: urlSpec='"+urlSpec+"'\n");

      this.URLHandler(urlSpec);
    },
    
  onDragOver: function (aEvent, aFlavour, aDragSession)
    {
      DEBUG_LOG("protoCommon.js: URLDropObserver.onDragOver:\n");
    },
    
  onDragExit: function (aEvent, aDragSession)
    {
      DEBUG_LOG("protoCommon.js: URLDropObserver.onDragExit:\n");
    },
        
  getSupportedFlavours: function ()
    {
      var flavourList = { };
      flavourList["application/x-moz-file"] = { width: 2, iid: "nsIFile" };
      flavourList["text/x-moz-url"] = { width: 2, iid: "nsISupportsWString" };
      flavourList["text/unicode"]   = { width: 2, iid: "nsISupportsWString" };
      return flavourList;
    }  
};

function RetrieveURLSpecFromDropData(aData)
{
   var dropData = (('length' in aData) && aData.length) ? aData[0] : aData;

   if (('length' in aData) && aData.length) {
      for (var k=0; k<aData.length; k++)
        DEBUG_LOG("protoCommon.js: RetrieveURLSpecFromDropData: flavour["+k+"]="+aData[0].flavour+"\n");
   } else {
        DEBUG_LOG("protoCommon.js: RetrieveURLSpecFromDropData: flavour="+dropData.flavour+"\n");
   }

   var dataObj = dropData.data.data;

   DEBUG_LOG("protoCommon.js: RetrieveURLSpecFromDropData: dataObj = '"+dataObj+"'\n");

   switch (dropData.flavour) {
   case "text/unicode":
   case "text/x-moz-url":
      var dataStr = dataObj.toString();

      // Check if valid URI
      var colonOffset = dataStr.indexOf(":");

      if (colonOffset <= 0) {
         // Look for url-encoded colon (pathological case)
         colonOffset = dataStr.toLowerCase().indexOf("%3a");
         if (colonOffset <= 0)
            return ""; // No scheme in URI

         dataStr = dataStr.substr(0,colonOffset)+":"+
                   dataStr.substr(colonOffset+3);
      }

      var schemeName = dataStr.substr(0,colonOffset).toLowerCase();
      if (schemeName.search(/^[a-z]([a-z0-9+.\-])*$/) == -1)
         return ""; // Invalid scheme in URI

      var lineEnd = dataStr.search(/[\r\n]/);
      if (lineEnd != -1) {
         // Trim anything past line end (to remove document title info etc.)
         DEBUG_LOG("protoCommon.js: RetrieveURLSpecFromDropData: lineEnd = '"+lineEnd+"'\n");
         dataStr = dataStr.substr(0,lineEnd);
      }
    
      return dataStr;
      break;

   case "application/x-moz-file":
      dataObj = dataObj.QueryInterface(Components.interfaces.nsIFile);
      if (dataObj) {
        var fileURL = nsJSComponentManager.createInstance("@mozilla.org/network/standard-url;1", "nsIFileURL");
        fileURL.file = dataObj;
        return fileURL.spec;
      }
   }             
   return null;                                                         
}
