/***************************************************************************

  Xmame 3Dfx console-mode driver

  Written based on Phillip Ezolt's svgalib driver by Mike Oliphant -

    oliphant@ling.ed.ac.uk

    http://www.ling.ed.ac.uk/~oliphant/glmame

***************************************************************************/
#define __SVGAFX_C

#include <vga.h>
#include <vgagl.h>
#include <vgakeyboard.h>
#include <vgamouse.h>
#include <signal.h>
#include <linux/kd.h>
#include <sys/ioctl.h>
#include <glide.h>
#include "xmame.h"
#include "devices.h"

int  InitVScreen(void);
void CloseVScreen(void);
int  InitGlide(void);
int  SetResolution(struct rc_option *option, const char *arg, int priority);

extern struct rc_option fx_opts[];

static int console_fd       = -1;
static int leds             =  0;
static int release_signal   =  0;
static int acquire_signal   =  0;
static struct sigaction release_sa;
static struct sigaction oldrelease_sa;
static struct sigaction acquire_sa;
static struct sigaction oldacquire_sa;

struct rc_option display_opts[] = {
   /* name, shortname, type, dest, deflt, min, max, func, help */
   { NULL, 		NULL,			rc_link,	fx_opts,
     NULL,		0,			0,		NULL,
     NULL },
   { NULL,		NULL,			rc_end,		NULL,
     NULL,		0,			0,		NULL,
     NULL }
};

int sysdep_init(void)
{
   if (InitGlide()!=OSD_OK)
      return OSD_NOT_OK;
   if (vga_init())
      return OSD_NOT_OK;
   return OSD_OK;
}

void sysdep_close(void)
{
}

void release_handler(int n)
{
   grSstControl(GR_CONTROL_DEACTIVATE);
   oldrelease_sa.sa_handler(n);
   sigaction(release_signal, &release_sa, NULL);
   sigaction(acquire_signal, &acquire_sa, NULL);
}

void acquire_handler(int n)
{
   oldacquire_sa.sa_handler(n);
   sigaction(release_signal, &release_sa, NULL);
   sigaction(acquire_signal, &acquire_sa, NULL);
   grSstControl(GR_CONTROL_ACTIVATE);
   keyboard_clearstate();
   if (console_fd >= 0)
      ioctl(console_fd, KDSETLED, leds);
}

/* This name doesn't really cover this function, since it also sets up mouse
   and keyboard. This is done over here, since on most display targets the
   mouse and keyboard can't be setup before the display has. */
int sysdep_create_display(void)
{
  if(bitmap->depth == 16)
  {
     fprintf(stderr_file, "%s doesn't support 16bpp video modes\n", title);
     return OSD_NOT_OK;
  }

  fprintf(stderr_file,
     "info: using FXmame v0.5 driver for xmame, written by Mike Oliphant\n");
  
  if (InitVScreen() != OSD_OK)
     return OSD_NOT_OK;
     
  /* with newer svgalib's the console switch signals are only active if a
     graphics mode is set, so we set one which each card should support */
  vga_setmode(G320x200x16);
  
  /* newer svgalib's use different signals */
  if (vga_setmode(-1)<0x1410)
  {
    fprintf(stderr_file, "info: svgalib version older then 1.4.1 detected, using old style signals\n");
    release_signal = SIGUSR1;
    acquire_signal = SIGUSR2;
  }
  else
  {
    fprintf(stderr_file, "info: svgalib version 1.4.1 or newer detected, using new style signals\n");
    release_signal = SIGPROF;
    acquire_signal = SIGUNUSED;
  }
  
  /* catch console switch signals to enable / disable the vga pass through */
  memset(&release_sa, 0, sizeof(struct sigaction));
  memset(&acquire_sa, 0, sizeof(struct sigaction));
  release_sa.sa_handler = release_handler;
  acquire_sa.sa_handler = acquire_handler;
  sigaction(release_signal, &release_sa, &oldrelease_sa);
  sigaction(acquire_signal, &acquire_sa, &oldacquire_sa);

   /* init the keyboard */
   if ((console_fd = keyboard_init_return_fd()) < 0)
   {
      fprintf(stderr_file, "Svgalib: Error: Couldn't open keyboard\n");
      return OSD_NOT_OK;
   }
   ioctl(console_fd, KDSETLED, leds);
   local_key=keyboard_getstate();

   /* 
      not sure if this is the best site but mouse init routine must be 
      called after video initialization...
   */   
   if(use_mouse)
   {
	if (mouse_init("/dev/mouse", vga_getmousetype(), MOUSE_DEFAULTSAMPLERATE))
	{
		perror("mouse_init");
		fprintf(stderr_file,"SVGALib: failed to open mouse device\n");
		use_mouse=0;
	}
	else
	{
		/* fix ranges and initial position of mouse */
		mouse_setrange_6d(-500,500, -500,500, -500,500, -500,500,
                   -500,500, -500,500, MOUSE_6DIM);
		mouse_setposition_6d(0, 0, 0, 0, 0, 0, MOUSE_6DIM);
	}
   }
   
  return OSD_OK;
}


/* shut up the display */
void osd_close_display(void)
{
   /* close glide */
   CloseVScreen();
   
   /* close svgalib, first video then input */
   sigaction(release_signal, &oldrelease_sa, NULL);
   sigaction(acquire_signal, &oldacquire_sa, NULL);
   vga_setmode(TEXT);

   if (console_fd >= 0)
   {
      ioctl(console_fd, KDSETLED, 8);
      keyboard_close();
   }

   if (use_mouse)
      mouse_close();
   
   /* and don't forget to free the bitmap */
   osd_free_bitmap(bitmap);
}

void sysdep_mouse_poll (void)
{
	int i, mouse_buttons;
	
	mouse_update();
	
	mouse_getposition_6d(&mouse_data[0].deltas[0],
           &mouse_data[0].deltas[1],
           &mouse_data[0].deltas[2],
           &mouse_data[0].deltas[3],
           &mouse_data[0].deltas[4],
           &mouse_data[0].deltas[5]);
	
	/* scale down the delta's to some more sane values */
	for(i=0; i<6; i++)
	   mouse_data[0].deltas[i] /= 20;

	mouse_buttons = mouse_getbutton();

        for(i=0; i<MOUSE_BUTTONS; i++)
        {
           mouse_data[0].buttons[i] = mouse_buttons & (0x01 << i);
        }

	mouse_setposition_6d(0, 0, 0, 0, 0, 0, MOUSE_6DIM);
}

static const int led_flags[3] = {
  LED_NUM,
  LED_CAP,
  LED_SCR
};

void osd_led_w(int led,int on)
{
	if (led>=3) return;
	if (on)
		leds |=  led_flags[led];
	else
		leds &= ~led_flags[led];
	if (console_fd >= 0)
		ioctl(console_fd, KDSETLED, leds);
}
