/*
 * OpenStep input functions - placed here to separate them out from the
 * display functions. This code handles keyboard and mouse input using the
 * OpenStep event queue mechanism. Unknown events are passed onward to allow
 * minaturisation of the window.
 *
 * -bat. 14/3/2000
 */

#import <AppKit/AppKit.h>
#import <libc.h>
#import "xmame.h"
#import "osdepend.h"
#import "driver.h"
#import "keyboard.h"
#import "devices.h"

/*
 * Keyboard variables
 */

static unsigned char ibm_keymap[256];	/* map ascii to ibm keycodes */
static NSDate *the_past = nil;

/*
 * External window variable
 */

extern NSWindow *thisWin;

/*
 * Keyboard init - all I really do here is set up the keymapping array
 * between ASCII values and MAME keycodes. Yes, it was all done by
 * hand, though there aren't actually that many of them. We also initialise
 * the "distant past" variable here as it belongs to the keyboard code.
 */

void
keyboard_init(void)
{
	int i;

	/* create the past */
	the_past = [NSDate distantPast];
	[the_past retain];

	/* create local array */
	local_key = malloc(256*sizeof(unsigned char));	/* only use first 128 */

	/* zero certain arrays */
	for(i=0;i<256;i++) {
		ibm_keymap[i]=0;
		local_key[i]=0;
	}

	/* and now we set up this big tedious array	*/
	ibm_keymap['0']=KEY_0;
	ibm_keymap['1']=KEY_1;
	ibm_keymap['2']=KEY_2;
	ibm_keymap['3']=KEY_3;
	ibm_keymap['4']=KEY_4;
	ibm_keymap['5']=KEY_5;
	ibm_keymap['6']=KEY_6;
	ibm_keymap['7']=KEY_7;
	ibm_keymap['8']=KEY_8;
	ibm_keymap['9']=KEY_9;

	ibm_keymap['-']=KEY_MINUS;
	ibm_keymap['_']=KEY_MINUS;
	ibm_keymap['+']=KEY_EQUALS;
	ibm_keymap['=']=KEY_EQUALS;
	ibm_keymap['\t']=KEY_TAB;

	ibm_keymap['=']=KEY_EQUALS;
	ibm_keymap['\t']=KEY_TAB;
	ibm_keymap['\r']=KEY_ENTER;
	ibm_keymap['\n']=KEY_ENTER;

	ibm_keymap['q']=KEY_Q;
	ibm_keymap['w']=KEY_W;
	ibm_keymap['e']=KEY_E;
	ibm_keymap['r']=KEY_R;
	ibm_keymap['t']=KEY_T;
	ibm_keymap['y']=KEY_Y;
	ibm_keymap['u']=KEY_U;
	ibm_keymap['i']=KEY_I;
	ibm_keymap['o']=KEY_O;
	ibm_keymap['p']=KEY_P;
	ibm_keymap['[']=KEY_OPENBRACE;
	ibm_keymap[']']=KEY_CLOSEBRACE;

	ibm_keymap['a']=KEY_A;
	ibm_keymap['s']=KEY_S;
	ibm_keymap['d']=KEY_D;
	ibm_keymap['f']=KEY_F;
	ibm_keymap['g']=KEY_G;
	ibm_keymap['h']=KEY_H;
	ibm_keymap['j']=KEY_J;
	ibm_keymap['k']=KEY_K;
	ibm_keymap['l']=KEY_L;
	ibm_keymap[';']=KEY_COLON;
	ibm_keymap[':']=KEY_COLON;
	ibm_keymap['\'']=KEY_QUOTE;
	ibm_keymap['@']=KEY_QUOTE;
	ibm_keymap['~']=KEY_TILDE;
	ibm_keymap['#']=KEY_TILDE;

	ibm_keymap['z']=KEY_Z;
	ibm_keymap['x']=KEY_X;
	ibm_keymap['c']=KEY_C;
	ibm_keymap['v']=KEY_V;
	ibm_keymap['b']=KEY_B;
	ibm_keymap['n']=KEY_N;
	ibm_keymap['m']=KEY_M;
	ibm_keymap[',']=KEY_COMMA;
	ibm_keymap['<']=KEY_COMMA;
	ibm_keymap['.']=KEY_STOP;
	ibm_keymap['>']=KEY_STOP;
	ibm_keymap['/']=KEY_SLASH;
	ibm_keymap['?']=KEY_SLASH;

	ibm_keymap['*']=KEY_ASTERISK;
	ibm_keymap[' ']=KEY_SPACE;

	ibm_keymap[8]=KEY_BACKSPACE;
	ibm_keymap[27]=KEY_ESC;
	ibm_keymap[96]=KEY_NUMLOCK;
	ibm_keymap[127]=KEY_DEL;
}


/*
 * Nothing needs doing to close the keybaord, but we release the variable
 * used to hold the diatant past here as it is a keyboard variable.
 */

void
sysdep_keyboard_close(void)
{
	[the_past release];
	the_past = nil;
}

/*
 * Get the mouse location and use this to set the deltas relative to
 * the last time. We only use this to get the position of the mouse, it's
 * buttons come in as events in the normal way.
 */

void
sysdep_mouse_poll(void)
{
	static NSPoint last = {0.0, 0.0};
	NSPoint current;

	if(!use_mouse)		/* to save time */
		return;

	current = [thisWin mouseLocationOutsideOfEventStream];

	mouse_data[0].deltas[0] = current.x - last.x;
	mouse_data[0].deltas[1] = last.y - current.y; /* inverted */

	last = current;
}

/*
 * Here we are passed an NSEvent from the keyboard and expected to pass
 * back the MAME keycode associated with it. For ASCII characters there
 * is a simple lookup table, for Unicode characters we use a switch statement
 * with the constants defined in NSEvent.h. This function also deals with
 * using the command key to emulate the function keys.
 */

static inline int
event_to_keycode(NSEvent *keyevent)
{
	int keycode;
	unichar buf[2];	/* just in case theres more than 1 */
	NSString *string = [keyevent charactersIgnoringModifiers];
	[string getCharacters:buf range:NSMakeRange(0,1)];
	keycode = buf[0];

	/* check to see if string is ASCII */
	if([string canBeConvertedToEncoding:NSASCIIStringEncoding]) {
		keycode=ibm_keymap[keycode];
	} else {
		 switch(keycode) {
			 case NSUpArrowFunctionKey:
				keycode=KEY_UP;
				break;
			 case NSDownArrowFunctionKey:
				keycode=KEY_DOWN;
				break;
			 case NSLeftArrowFunctionKey:
				keycode=KEY_LEFT;
				break;
			 case NSRightArrowFunctionKey:
				keycode=KEY_RIGHT;
				break;
			 case NSF1FunctionKey:
				keycode=KEY_F1;
				break;
			 case NSF2FunctionKey:
				keycode=KEY_F2;
				break;
			 case NSF3FunctionKey:
				keycode=KEY_F3;
				break;
			 case NSF4FunctionKey:
				keycode=KEY_F4;
				break;
			 case NSF5FunctionKey:
				keycode=KEY_F5;
				break;
			 case NSF6FunctionKey:
				keycode=KEY_F6;
				break;
			 case NSF7FunctionKey:
				keycode=KEY_F7;
				break;
			 case NSF8FunctionKey:
				keycode=KEY_F8;
				break;
			 case NSF9FunctionKey:
				keycode=KEY_F9;
				break;
			 case NSF10FunctionKey:
				keycode=KEY_F10;
				break;
			 case NSF11FunctionKey:
				keycode=KEY_F11;
				break;
			 case NSF12FunctionKey:
				keycode=KEY_F12;
				break;
			 case NSInsertFunctionKey:
				keycode=KEY_INSERT;
				break;
			 case NSDeleteFunctionKey:
				keycode=KEY_DEL;
				break;
			 case NSHomeFunctionKey:
				keycode=KEY_HOME;
				break;
			 case NSEndFunctionKey:
				keycode=KEY_END;
				break;
			 case NSPageUpFunctionKey:
				keycode=KEY_PGUP;
				break;
			 case NSPageDownFunctionKey:
				keycode=KEY_PGDN;
				break;
			 case NSPrintScreenFunctionKey:
				keycode=KEY_PRTSCR;
				break;
			 case NSScrollLockFunctionKey:
				keycode=KEY_SCRLOCK;
				break;
			 case NSPauseFunctionKey:
				keycode=KEY_PAUSE;
				break;
			 default:
				keycode=0;
		 }
	}

	/* deal with command key */
	if([keyevent modifierFlags] & NSCommandKeyMask)
		switch(keycode) {
			case KEY_1:
				keycode=KEY_F1;
				break;
			case KEY_2:
				keycode=KEY_F2;
				break;
			case KEY_3:
				keycode=KEY_F3;
				break;
			case KEY_4:
				keycode=KEY_F4;
				break;
			case KEY_5:
				keycode=KEY_F5;
				break;
			case KEY_6:
				keycode=KEY_F6;
				break;
			case KEY_7:
				keycode=KEY_F7;
				break;
			case KEY_8:
				keycode=KEY_F8;
				break;
			case KEY_9:
				keycode=KEY_F9;
				break;
			case KEY_0:
				keycode=KEY_F10;
				break;
			case KEY_MINUS:
				keycode=KEY_F11;
				break;
			case KEY_EQUALS:
				keycode=KEY_F12;
				break;
		}

	return keycode;
}

/*
 * This loop collect all events from the queue and deals with them by
 * updating the local_key array for key up and down events. Anything we
 * dont use gets passed on to whoever might want it. We handle the mouse
 * button events in this loop too, passing them downwards to allow other
 * things such as minaturisation to happen.
 */

void
sysdep_update_keyboard(void)
{
	NSEvent *event=nil;

	for(;;) {
		/* get the next event */
		event= [NSApp nextEventMatchingMask:NSAnyEventMask
				untilDate:the_past inMode:NSDefaultRunLoopMode
				dequeue:YES];
		/* break out of the loop if there are no more events */
		if(event==nil)
			break;

		/* deal with the event */
		switch([event type]) {
			case NSKeyDown:
				local_key[event_to_keycode(event)] = 1;
				break;
			case NSKeyUp:
				local_key[event_to_keycode(event)] = 0;
				break;
			case NSLeftMouseDown:
				mouse_data[0].buttons[0] = 1;
				[NSApp sendEvent:event];
				break;
			case NSLeftMouseUp:
				mouse_data[0].buttons[0] = 0;
				[NSApp sendEvent:event]; 
				break;
			case NSRightMouseDown:
				mouse_data[0].buttons[1] = 1;
				[NSApp sendEvent:event];
				break;
			case NSRightMouseUp:
				mouse_data[0].buttons[1] = 0;
				[NSApp sendEvent:event];
				break;
			default:
				[NSApp sendEvent:event];
				break;
		}
	}
}

/*
 * We do not have access to the keyboard LED's
 */

void osd_led_w(int led,int on) 
{
}
