/***************************************************************************

  machine.c

  Functions to emulate general aspects of the machine (RAM, ROM, interrupts,
  I/O ports)

  TODO:
		- Implement unimplemented SAM registers
		- Implement unimplemented interrupts (serial, keyboard, timer)
		- Implement 63.5usec interrupt
		- Choose and implement more appropriate rations for the speed up poke
		- Support .PAK files generated by PC-Dragon
***************************************************************************/

#include "driver.h"
#include "cpu/m6809/m6809.h"
#include "machine/6821pia.h"

UINT8 *coco_rom;
static int coco3_enable_64k;
static int coco3_mmu[16];
static int coco3_gimereg[8];

/* from vidhrdw/dragon.c */
extern void coco3_ram_b1_w (int offset, int data);
extern void coco3_ram_b2_w (int offset, int data);
extern void coco3_ram_b3_w (int offset, int data);
extern void coco3_ram_b4_w (int offset, int data);
extern void coco3_ram_b5_w (int offset, int data);
extern void coco3_ram_b6_w (int offset, int data);
extern void coco3_ram_b7_w (int offset, int data);
extern void coco3_ram_b8_w (int offset, int data);
extern void coco3_vh_sethires(int hires);

extern void d_pia1_pb_w(int offset, int data);
extern void coco3_pia1_pb_w(int offset, int data);

static void coco3_set_irq_line(int mask, int state);
static void d_pia1_pa_w(int offset, int data);
static int  d_pia1_cb1_r(int offset);
static int  d_pia0_ca1_r(int offset);
static int  d_pia0_pa_r(int offset);
static int  d_pia1_pa_r(int offset);
static void d_pia0_pb_w(int offset, int data);
static void d_pia1_cb2_w(int offset, int data);
static void d_pia0_cb2_w(int offset, int data);
static void d_pia1_ca2_w(int offset, int data);
static void d_pia0_ca2_w(int offset, int data);
static void d_pia0_irq_a(int state);
static void d_pia0_irq_b(int state);
static void coco3_pia0_irq_a(int state);
static void coco3_pia0_irq_b(int state);


static void coco3_timer_recalculate(int newcounterval);


static struct pia6821_interface dragon_pia_intf[] =
{
	/* PIA 0 */
	{
		/*inputs : A/B,CA/B1,CA/B2 */ d_pia0_pa_r, 0, d_pia0_ca1_r, 0, 0, 0,
		/*outputs: A/B,CA/B2	   */ 0, d_pia0_pb_w, d_pia0_ca2_w, d_pia0_cb2_w,
		/*irqs	 : A/B			   */ d_pia0_irq_a, d_pia0_irq_b
	},

	/* PIA 1 */
	{
		/*inputs : A/B,CA/B1,CA/B2 */ d_pia1_pa_r, 0, 0, d_pia1_cb1_r, 0, 0,
		/*outputs: A/B,CA/B2	   */ d_pia1_pa_w, d_pia1_pb_w, d_pia1_ca2_w, d_pia1_cb2_w,
		/*irqs	 : A/B			   */ 0, 0
	}
};

static struct pia6821_interface coco3_pia_intf[] =
{
	/* PIA 0 */
	{
		/*inputs : A/B,CA/B1,CA/B2 */ d_pia0_pa_r, 0, d_pia0_ca1_r, 0, 0, 0,
		/*outputs: A/B,CA/B2	   */ 0, d_pia0_pb_w, d_pia0_ca2_w, d_pia0_cb2_w,
		/*irqs	 : A/B			   */ coco3_pia0_irq_a, coco3_pia0_irq_b
	},

	/* PIA 1 */
	{
		/*inputs : A/B,CA/B1,CA/B2 */ d_pia1_pa_r, 0, 0, d_pia1_cb1_r, 0, 0,
		/*outputs: A/B,CA/B2	   */ d_pia1_pa_w, coco3_pia1_pb_w, d_pia1_ca2_w, d_pia1_cb2_w,
		/*irqs	 : A/B			   */ 0, 0
	}
};

static int cart_inserted;
static UINT8 pia0_pb, sound_mux, tape_motor;
static UINT8 joystick_axis, joystick;
static int d_dac;

/***************************************************************************
  dev init
***************************************************************************/
static int load_pak_into_region(void *fp, int *pakbase, int *paklen, UINT8 *mem, int segaddr, int seglen)
{
	if (*paklen) {
		if (*pakbase < segaddr) {
			/* We have to skip part of the PAK file */
			int skiplen;

			skiplen = segaddr - *pakbase;
			if (osd_fseek(fp, skiplen, SEEK_CUR)) {
				if (errorlog) fprintf(errorlog,"Could not fully read PAK.\n");
				return 1;
			}

			*pakbase += skiplen;
			*paklen -= skiplen;
		}

		if (*pakbase < segaddr + seglen) {
			mem += *pakbase - segaddr;
			seglen -= *pakbase - segaddr;

			if (seglen > *paklen)
				seglen = *paklen;

			if (osd_fread(fp, mem, seglen) < seglen) {
				if (errorlog) fprintf(errorlog,"Could not fully read PAK.\n");
				return 1;
			}

			*pakbase += seglen;
			*paklen -= seglen;
		}
	}
	return 0;
}

/* PAK file loader */

/* PAK files have the following format:
 *
 * length		(two bytes, little endian)
 * base address (two bytes, little endian, typically 0xc000)
 * ...data... (size is length)
 * extra info
 *
 * The format for PAK files just plain bites - the extra info is snapshot info
 * and it is loaded with internal state specific to Jeff's emulator. What ever
 * happened to desiging clean file formats that don't need to be changed with
 * every version of ones program?
 *
 * For alignment purposes, some 16 bit values are divided into two UINT8s
 */

#ifdef LSB_FIRST
#define ENDIANIZE(x) (x)
#else
#define ENDIANIZE(x) ((((x) >> 8) + ((x) << 8)) & 0xffff)
#endif

/* All versions */
typedef struct {
	UINT16 length;
	UINT16 start;
} pak_header;

/* All versions */
typedef struct {
	char name[33];
} pak_trailer1;

/* Only version 1.2 */
typedef struct {
	UINT8 debug_dumpflag;
	UINT8 debug_disassembleflag;
	UINT16 debug_disassembleaddr;
} pak_trailer2;

/* All versions */
typedef struct {
	UINT16 reg_pc;
	UINT16 reg_x;
	UINT16 reg_y;
	UINT16 reg_u;
	UINT16 reg_s;
	UINT8 dummy_zero1;
	UINT8 reg_dp;
	UINT8 reg_b;
	UINT8 reg_a;
} pak_trailer3;

/* Only version 1.2 */
typedef struct {
	UINT16 debug_unknown;
} pak_trailer4;

/* All versions */
typedef struct {
	UINT8 flags_8086_lsb; /* ?!? */
	UINT8 flags_8086_msb; /* ?!? */
	UINT8 reg_cc;
} pak_trailer5;

/* All versions except 1.4 */
typedef struct {
	UINT16 lowmem_readseg;
	UINT16 lowmem_writeseg;
	UINT16 himem_readseg;
	UINT16 himem_writeseg;
} pak_trailer6;

/* Only version 1.4 */
typedef struct {
	UINT8 page_status;
	UINT8 rom_status;
	UINT8 io_ff02;
	UINT8 io_ff03;
	UINT8 io_ff22;
} pak_trailer7;

/* All versions */
typedef struct {
	UINT16 video_base;
	UINT16 video_end;
} pak_trailer8;

/* All versions; come before pak_trailer5 in 1.4 */
typedef struct {
	UINT8 dummy_zero2[6];
} pak_trailer9;

/* Used by PC-Dragon; UINT16's are little endian */
typedef struct {
	/* Info */
	UINT8 data1[2];
	UINT8 magic1[2];
	UINT8 checksum;
	UINT8 pak_version;
	UINT8 ext_version;
	UINT8 emu_version;
	UINT8 state_only;
	UINT8 architecture;
	UINT8 rom_start;

	/* Machine state */
	UINT8 irq_cycles_lsb;
	UINT8 irq_cycles_msb;
	UINT8 screen_size;
	UINT8 pia[9];
	UINT8 last_shift;
	UINT8 filemem[8];

	/* Debugger */
	UINT16 last_text_base;
	UINT16 breakpoints[6];
	UINT16 break_operation;
	UINT16 dasm_pcr;
	UINT16 fill_start;
	UINT16 fill_end;
	UINT8 fill_value;
	UINT8 left_window;
	UINT16 temp_break;
	UINT16 break_value;
	UINT8 break_control;
	UINT8 break_type;
	UINT16 internal;
	UINT16 dumpaddr;
} pcd_info1;

/* Used by PC-Dragon; UINT16's are little endian */
typedef struct {
	UINT8 pakdata[65];
	UINT8 dummy1;
	UINT8 background;
	UINT8 foreground;
	UINT8 vmode[2][2][6];
	UINT8 border[2][2];
	UINT8 dummy2;
	UINT16 irq_rate;
	UINT8 servicemem[16];
	UINT8 speed;
	UINT8 lpt_and_swapping;
	UINT8 hardjoy_resol;
} pcd_info2;

/* All versions */
typedef struct {
	UINT8 writeprotect;
	char disk_directory[66];
	char disk_name[4][32];
	char pak_directory[66];
	UINT8 crlf;
	UINT8 keyboard_mode;
	UINT8 speed_lsb;
	UINT8 speed_msb;
	UINT8 left_joystick;
	UINT8 right_joystick;
	UINT8 lowercase_lsb;
	UINT8 lowercase_msb;
	UINT8 sound;
	UINT8 artifact;
	UINT8 dragon_rom;
} pak_trailer10;

/* Only version 1.4 */
typedef struct {
	UINT16 joystick_limits[8];
	UINT16 clock;
	UINT8 drive_mode;
	UINT8 volume;
	UINT8 cassette_mode;
	char cassette_directory[66];
	char cassette_name[33];
} pak_trailer11;

/* All versions except 1.4 */
typedef struct {
	UINT8 dummy_zero3[4];
	UINT16 video_base2;
	UINT16 video_end2;
	UINT16 dummy_zero4;
	UINT8 io_ff22;
	UINT8 io_ff02;
	UINT8 io_ff03;
} pak_trailer12;

#define PAK_V12_SIZE	(sizeof(pak_trailer1) + sizeof(pak_trailer2) + \
	sizeof(pak_trailer3) + sizeof(pak_trailer4) + sizeof(pak_trailer5) + \
	sizeof(pak_trailer6) + sizeof(pak_trailer8) + \
	sizeof(pak_trailer9) + sizeof(pak_trailer10) + sizeof(pak_trailer12))

#define PAK_V14_SIZE	(sizeof(pak_trailer1) + sizeof(pak_trailer3) + \
	sizeof(pak_trailer9) + sizeof(pak_trailer5) + \
	sizeof(pak_trailer7) + sizeof(pak_trailer8) + sizeof(pak_trailer10) + \
	sizeof(pak_trailer11))

#define PAK_VOTHER_SIZE	(sizeof(pak_trailer1) + sizeof(pak_trailer3) + \
	sizeof(pak_trailer5) + sizeof(pak_trailer6) + \
	sizeof(pak_trailer8) + sizeof(pak_trailer9) + sizeof(pak_trailer10) + \
	sizeof(pak_trailer12))

typedef struct {
	UINT16 video_base;
	UINT16 video_end;
	UINT16 reg_pc;
	UINT16 reg_x;
	UINT16 reg_y;
	UINT16 reg_u;
	UINT16 reg_s;
	UINT8 reg_dp;
	UINT8 reg_b;
	UINT8 reg_a;
	UINT8 reg_cc;
	UINT8 io_ff02;
	UINT8 io_ff03;
	UINT8 io_ff22;
	UINT8 enable_hiram;
} pak_trailer;

/* This function takes a set of bytes, interprets a PAK trailer and
 * extracts the interesting information
 */
static int pak_decode_trailer(UINT8 *rawtrailer, int rawtrailerlen, pak_trailer *trailer)
{
	pak_trailer3 p3;
	pak_trailer5 p5;
	pak_trailer8 p8;

	union {
		pak_trailer7 p7;
		struct {
			pak_trailer6 p6;
			pak_trailer12 p12;
		} s;
	} u1;

	switch(rawtrailerlen) {
	case PAK_V12_SIZE:
	case PAK_V14_SIZE:
	case PAK_VOTHER_SIZE:
		break;
	default:
		return 1;
	}

	rawtrailer += sizeof(pak_trailer1);

	if (rawtrailerlen == PAK_V12_SIZE) {
		rawtrailer += sizeof(pak_trailer2);
	}

	memcpy(&p3, rawtrailer, sizeof(pak_trailer3));
	rawtrailer += sizeof(pak_trailer3);

	if (rawtrailerlen == PAK_V14_SIZE) {
		rawtrailer += sizeof(pak_trailer9);
	}
	else if (rawtrailerlen == PAK_V12_SIZE) {
		rawtrailer += sizeof(pak_trailer4);
	}

	memcpy(&p5, rawtrailer, sizeof(pak_trailer5));
	rawtrailer += sizeof(pak_trailer5);

	if (rawtrailerlen != PAK_V14_SIZE) {
		memcpy(&u1.s.p6, rawtrailer, sizeof(pak_trailer6));
		rawtrailer += sizeof(pak_trailer6);
	}
	else {
		memcpy(&u1.p7, rawtrailer, sizeof(pak_trailer7));
		rawtrailer += sizeof(pak_trailer7);
	}

	memcpy(&p8, rawtrailer, sizeof(pak_trailer8));
	rawtrailer += sizeof(pak_trailer8);

	if (rawtrailerlen != PAK_V14_SIZE) {
		rawtrailer += sizeof(pak_trailer9);
	}

	rawtrailer += sizeof(pak_trailer10);

	if (rawtrailerlen != PAK_V14_SIZE) {
		memcpy(&u1.s.p12, rawtrailer, sizeof(pak_trailer12));
		rawtrailer += sizeof(pak_trailer12);
	}
	else {
		rawtrailer += sizeof(pak_trailer11);
	}

	trailer->reg_pc = ENDIANIZE(p3.reg_pc);
	trailer->reg_x = ENDIANIZE(p3.reg_x);
	trailer->reg_y = ENDIANIZE(p3.reg_y);
	trailer->reg_u = ENDIANIZE(p3.reg_u);
	trailer->reg_s = ENDIANIZE(p3.reg_s);
	trailer->reg_dp = p3.reg_dp;
	trailer->reg_b = p3.reg_b;
	trailer->reg_a = p3.reg_a;
	trailer->reg_cc = p5.reg_cc;

	if (rawtrailerlen == PAK_V14_SIZE) {
		trailer->io_ff02 = u1.p7.io_ff02;
		trailer->io_ff03 = u1.p7.io_ff03;
		trailer->io_ff22 = u1.p7.io_ff22;
		trailer->enable_hiram = (u1.p7.rom_status == 0xdf);
	}
	else {
		trailer->io_ff02 = u1.s.p12.io_ff02;
		trailer->io_ff03 = u1.s.p12.io_ff03;
		trailer->io_ff22 = u1.s.p12.io_ff22;
		trailer->enable_hiram = (u1.s.p6.himem_readseg == 0);
	}

	trailer->video_base = ENDIANIZE(p8.video_base);
	trailer->video_end = ENDIANIZE(p8.video_end);

	return 0;
}

static void pak_load_trailer(const pak_trailer *trailer)
{
	extern void dragon_sam_display_offset(int offset, int data);
	extern void dragon_sam_vdg_mode(int offset, int data);
	int i, value;

	cpu_set_reg(M6809_PC, trailer->reg_pc);
	cpu_set_reg(M6809_X, trailer->reg_x);
	cpu_set_reg(M6809_Y, trailer->reg_y);
	cpu_set_reg(M6809_U, trailer->reg_u);
	cpu_set_reg(M6809_S, trailer->reg_s);
	cpu_set_reg(M6809_DP, trailer->reg_dp);
	cpu_set_reg(M6809_B, trailer->reg_b);
	cpu_set_reg(M6809_A, trailer->reg_a);
	cpu_set_reg(M6809_CC, trailer->reg_cc);

	/* I seem to only be able to get a small amount of the PIA state from the
	 * snapshot trailers. Thus I am going to configure the PIA myself. The
	 * following PIA writes are the same thing that the CoCo ROM does on
	 * startup. I wish I had a better solution
	 */
	cpu_writemem16(0xff1d, 0x00);
	cpu_writemem16(0xff1f, 0x00);
	cpu_writemem16(0xff1c, 0x00);
	cpu_writemem16(0xff1e, 0xff);
	cpu_writemem16(0xff1d, 0x34);
	cpu_writemem16(0xff1f, 0x34);
	cpu_writemem16(0xff21, 0x00);
	cpu_writemem16(0xff23, 0x00);
	cpu_writemem16(0xff20, 0xfe);
	cpu_writemem16(0xff22, 0xf8);
	cpu_writemem16(0xff21, 0x34);
	cpu_writemem16(0xff23, 0x34);
	cpu_writemem16(0xff22, 0x00);
	cpu_writemem16(0xff20, 0x02);

	cpu_writemem16(0xff03, trailer->io_ff03);	/* d_pia0_cb2_w */
	cpu_writemem16(0xff02, trailer->io_ff02);	/* d_pia0_pb_w */
	cpu_writemem16(0xff22, trailer->io_ff22);	/* d_pia1_pb_w */

	/* For some reason, this seems to screw things up; I'm not sure whether it
	 * is because I'm using the wrong method to get access
	 * trailer->enable_hiram or whether it is osmething else
	 */
	/* cpu_writemem16(0xffde + trailer->enable_hiram, 0); */

	value = trailer->video_base >> 9;
	for (i = 0; i < 6; i++) {
		dragon_sam_display_offset(i * 2 + (value & 1), 0);
		value >>= 1;
	}

	switch(trailer->video_end - trailer->video_base) {
	case 512:
		value = 0;
		break;
	case 1024:
		value = 1;
		break;
	case 1536:
	case 2048:
		value = 2;
		break;
	case 3072:
		value = 4;
		break;
	case 6144:
	default:
		value = 6;
		break;
	}

	for (i = 0; i < 3; i++) {
		dragon_sam_vdg_mode(i * 2 + (value & 1), 0);
		value >>= 1;
	}
}

static int trailer_load = 0;
static pak_trailer trailer;

static void pak_load_trailer_callback(int param)
{
	pak_load_trailer(&trailer);
}

static int generic_rom_load(int id, UINT8 *rambase, UINT8 *rombase, UINT8 *pakbase)
{
	void *fp;

	fp = image_fopen (IO_SNAPSHOT, id, OSD_FILETYPE_IMAGE_R, 0);
	if (fp)
	{
		int paklength;
		int pakstart;

		pak_header header;
		int trailerlen;
		UINT8 trailerraw[500];

		if (osd_fread(fp, &header, sizeof(header)) < sizeof(header)) {
			if (errorlog) fprintf(errorlog,"Could not fully read PAK.\n");
			osd_fclose(fp);
			return 1;
		}

		paklength = ENDIANIZE(header.length);
		pakstart = ENDIANIZE(header.start);
		if (pakstart == 0xc000)
			cart_inserted = 1;

		if ((paklength == 0) || (paklength > 0xff00))
			paklength = 0xff00;

		if (osd_fseek(fp, paklength, SEEK_CUR)) {
			if (errorlog) fprintf(errorlog,"Could not fully read PAK.\n");
			osd_fclose(fp);
			return 1;
		}

		trailerlen = osd_fread(fp, trailerraw, sizeof(trailerraw));
		if (trailerlen) {
			if (pak_decode_trailer(trailerraw, trailerlen, &trailer)) {
				if (errorlog) fprintf(errorlog,"Invalid or unknown PAK trailer.\n");
				osd_fclose(fp);
				return 1;
			}

			trailer_load = 1;
		}

		if (osd_fseek(fp, sizeof(pak_header), SEEK_SET)) {
			if (errorlog) fprintf(errorlog,"Unexpected error while reading PAK.\n");
			osd_fclose(fp);
			return 1;
		}

		/* Since PAK files allow the flexibility of loading anywhere in
		 * the base RAM or ROM, we have to do tricks because in MESS's
		 * memory, RAM and ROM may be separated, hense this function's
		 * two parameters.
		 *
		 * Similarly, some PAKs appear to be loading into high RAM.  I
		 * am not completely sure how to distinguish this, but I can
		 * guess
		 */

		/* Get the RAM portion */
		if (load_pak_into_region(fp, &pakstart, &paklength, rambase, 0x0000, 0x8000)) {
			osd_fclose(fp);
			return 1;
		}

		if (pakstart == 0x8000) {
			/* We are probably loading into high RAM */
			if ((rombase - rambase) < 0x10000) {
				if (load_pak_into_region(fp, &pakstart, &paklength, rambase, 0x8000, 0x7F00)) {
					osd_fclose(fp);
					return 1;
				}
			}
		}
		else {
			/* Get the ROM portion */
			if (load_pak_into_region(fp, &pakstart, &paklength, rombase, 0x8000, 0x4000)) {
				osd_fclose(fp);
				return 1;
			}
			/* Get the PAK portion */
			if (load_pak_into_region(fp, &pakstart, &paklength, pakbase, 0xC000, 0x3F00)) {
				osd_fclose(fp);
				return 1;
			}
		}
		osd_fclose(fp);
	}
	return INIT_OK;
}

int dragon32_rom_load(int id)
{
	UINT8 *ROM = memory_region(REGION_CPU1);
	return generic_rom_load(id, &ROM[0], &ROM[0x8000], &ROM[0xc000]);
}

int dragon64_rom_load(int id)
{
	UINT8 *ROM = memory_region(REGION_CPU1);
	return generic_rom_load(id, &ROM[0], &ROM[0x10000], &ROM[0x14000]);
}

int coco3_rom_load(int id)
{
	UINT8 *ROM = memory_region(REGION_CPU1);
	return generic_rom_load(id, &ROM[0x70000], &ROM[0x80000], &ROM[0x8c000]);
}

/***************************************************************************
  Misc
***************************************************************************/

int dragon_mapped_irq_r(int offset)
{
	return coco_rom[0x3ff0 + offset];
}

void dragon_sam_speedctrl(int offset, int data)
{
	/* The infamous speed up poke.
	 *
	 * This was a SAM switch that occupied 4 addresses:
	 *
	 *		$FFD9	(set)	R1
	 *		$FFD8	(clear)	R1
	 *		$FFD7	(set)	R0
	 *		$FFD6	(clear)	R0
	 *
	 * R1:R0 formed the following states:
	 *		00	- slow
	 *		01	- dual speed
	 *		1x	- fast
	 *
	 * R1 controlled whether the video addressing was speeded up and R0
	 * did the same for the CPU.  On pre-CoCo 3 machines, setting R1 caused
	 * the screen to display garbage because the M6847 could not display
	 * fast enough.
	 *
	 * TODO:  Make the overclock more accurate.  I am not sure of the exact
	 * speedup effects on overall performance.
	 */
    timer_set_overclock(0, 1+(offset&1));
}

void dragon_sam_page_mode(int offset, int data)
{
	/* Page mode - allowed switching between the low 32k and the high 32k,
	 * assuming that 64k wasn't enabled
	 *
	 * TODO:  Actually implement this.  Also find out what the CoCo 3 did with
	 * this (it probably ignored it)
	 */
}

void dragon_sam_memory_size(int offset, int data)
{
	/* Memory size - allowed restricting memory accesses to something less than
	 * 32k
	 *
	 * This was a SAM switch that occupied 4 addresses:
	 *
	 *		$FFDD	(set)	R1
	 *		$FFDC	(clear)	R1
	 *		$FFDB	(set)	R0
	 *		$FFDA	(clear)	R0
	 *
	 * R1:R0 formed the following states:
	 *		00	- 4k
	 *		01	- 16k
	 *		10	- 64k
	 *		11	- static RAM (??)
	 *
	 * If something less than 64k was set, the low RAM would be smaller and
	 * mirror the other parts of the RAM
	 *
	 * TODO:  Actually implement this.  Also find out what the CoCo 3 did with
	 * this (it probably ignored it)
	 */
}

/***************************************************************************
   CoCo 3 Interrupts
***************************************************************************/

enum {
	COCO3_INT_TMR	= 0x20,		/* Timer */
	COCO3_INT_HBORD	= 0x10,		/* Horizontal border sync */
	COCO3_INT_VBORD	= 0x08,		/* Vertical border sync */
	COCO3_INT_EI2	= 0x04,		/* Serial data */
	COCO3_INT_EI1	= 0x02,		/* Keyboard */
	COCO3_INT_EI0	= 0x01		/* Cartridge */
};

static void coco3_set_irq_line(int mask, int state)
{
	if ((coco3_gimereg[2] & mask) && (coco3_gimereg[0] & 0x20))
		cpu_set_irq_line(0, M6809_IRQ_LINE, state);
	if ((coco3_gimereg[3] & mask) && (coco3_gimereg[0] & 0x10))
		cpu_set_irq_line(0, M6809_FIRQ_LINE, state);
}

/* TODO: Make this interrupt called, and also - make the CoCo 2 HBORD work */
static void coco3_hbord(void)
{
	pia_0_ca1_w(0, 0);
	coco3_set_irq_line(COCO3_INT_HBORD, 0);
}

void coco3_vbord(void)
{
	pia_0_cb1_w(0, 0);
	coco3_set_irq_line(COCO3_INT_VBORD, 0);
}

/***************************************************************************
  CoCo 3 Timer

  The CoCo 3 had a timer that had would activate when first written to, and
  would decrement over and over again until zero was reached, and at that
  point, would flag an interrupt.  At this point, the timer starts back up
  again.
***************************************************************************/

static void *coco3_timer;
static int coco3_timer_interval;	/* interval: 1=70 nsec, 0=63.5 usec */
static int coco3_timer_base;
static int coco3_timer_counter;
extern void coco3_vh_blink(void);

static void coco3_timer_init(void)
{
	coco3_timer = 0;
	coco3_timer_interval = 0;
}

static void coco3_timer_callback(int dummy)
{

	/*static void coco3_timer_recalculate(int newcounterval); */

	coco3_timer = 0;
	coco3_set_irq_line(COCO3_INT_TMR, 0);
	coco3_timer_recalculate(coco3_timer_base);
	coco3_vh_blink();

}

static double coco3_timer_interval_time(void)
{
	return coco3_timer_interval ? TIME_IN_NSEC(70) : TIME_IN_USEC(63.5);
}

/* This function takes the value in coco3_timer_counter, and sets up the timer
 * and the coco3_time_counter_time for it
 */
static void coco3_timer_recalculate(int newcounterval)
{
	if (coco3_timer)
		timer_remove(coco3_timer);

	if (newcounterval) {
		coco3_timer = timer_set(newcounterval * coco3_timer_interval_time(),
			0, coco3_timer_callback);
	}
	else {
		coco3_timer = 0;
	}

	coco3_timer_counter = newcounterval;
}

static int coco3_timer_r(void)
{
	int result = 0;

	if (coco3_timer) {
		result = coco3_timer_counter -
			(timer_timeleft(coco3_timer) / coco3_timer_interval_time());

		/* This shouldn't happen, but I'm prepared anyways */
		if (result < 0)
			result = 0;
		else if (result > 4095)
			result = 4095;
	}
	return result;	/* result = 0..4095 */
}

static void coco3_timer_w(int data)	/* data = 0..4095 */
{
	coco3_timer_base = (data & 4095);
	coco3_timer_recalculate(coco3_timer_base);
}

static void coco3_timer_msb_w(int data)
{
	coco3_timer_w(((data & 0x0f) << 8) | (coco3_timer_base & 0xff));
}

static void coco3_timer_lsb_w(int data)
{
	coco3_timer_w((coco3_timer_base & 0xf00) | (data & 0xff));
}

static void coco3_timer_set_interval(int interval)
{
	int oldtimerval;

	if (interval != coco3_timer_interval) {
		if (coco3_timer) {
			oldtimerval = coco3_timer_r();
			coco3_timer_interval = interval;
			coco3_timer_recalculate(oldtimerval);
		}
		else {
			coco3_timer_interval = interval;
		}
	}
}

/***************************************************************************
  MMU
***************************************************************************/

/* from vidhrdw/dragon.c */
extern void coco_ram_w(int offset, int data);

void dragon64_ram_w(int offset, int data)
{
	coco_ram_w(offset + 0x8000, data);
}

void dragon64_sam_himemmap(int offset, int data)
{
	UINT8 *RAM = memory_region(REGION_CPU1);
	if (offset) {
		cpu_setbank(1, &RAM[0x8000]);
		cpu_setbankhandler_w(1, dragon64_ram_w);
	}
	else {
		cpu_setbank(1, coco_rom);
		cpu_setbankhandler_w(1, MWA_ROM);
	}
}

/* Coco 3 */

int coco3_mmu_lookup(int block)
{
	int result;

	if (coco3_gimereg[0] & 0x40) {
		if (coco3_gimereg[1] & 1)
			block += 8;
		result = coco3_mmu[block];
	}
	else {
		result = block + 56;
	}
	return result;
}

int coco3_mmu_translate(int block, int offset)
{
	if ((block == 7) && (coco3_gimereg[0] & 8))
		return 0x7e000 + offset;
	else
		return (coco3_mmu_lookup(block) * 0x2000) + offset;
}

static void coco3_mmu_update(int lowblock, int hiblock)
{
	UINT8 *RAM = memory_region(REGION_CPU1);
	typedef void (*writehandler)(int wh_offset, int data);
	static writehandler handlers[] = {
		coco3_ram_b1_w, coco3_ram_b2_w,
		coco3_ram_b3_w, coco3_ram_b4_w,
		coco3_ram_b5_w, coco3_ram_b6_w,
		coco3_ram_b7_w, coco3_ram_b8_w
	};

	int hirom_base, lorom_base;
	int i;

	hirom_base = ((coco3_gimereg[0] & 3) == 2) ? 0x0000 : 0x8000;
	lorom_base = ((coco3_gimereg[0] & 3) != 3) ? 0x0000 : 0x8000;

	for (i = lowblock; i <= hiblock; i++) {
		if ((i >= 4) && !coco3_enable_64k) {
			cpu_setbank(i + 1, &coco_rom[(i >= 6 ? hirom_base : lorom_base) + ((i-4) * 0x2000)]);
			cpu_setbankhandler_w(i + 1, MWA_ROM);
		}
		else {
			cpu_setbank(i + 1, &RAM[coco3_mmu_lookup(i) * 0x2000]);
			cpu_setbankhandler_w(i + 1, handlers[i]);
		}
	}
}

int coco3_mmu_r(int offset)
{
	return coco3_mmu[offset];
}

void coco3_mmu_w(int offset, int data)
{
	data &= 0x3f;
	coco3_mmu[offset] = data;

	/* Did we modify the live MMU bank? */
	if ((offset >> 3) == (coco3_gimereg[1] & 1))
		coco3_mmu_update(offset & 7, offset & 7);
}

int coco3_gime_r(int offset)
{
	int result = 0;

	switch(offset) {
	case 4:	/* Timer MSB */
		result = coco3_timer_r() >> 8;
		break;

	case 5:	/* Timer LSB */
		result = coco3_timer_r() & 0xff;
		break;

	default:
		result = coco3_gimereg[offset];
		break;
	}
	return result;
}

void coco3_gime_w(int offset, int data)
{
	coco3_gimereg[offset] = data;

	/* Features marked with '!' are not yet implemented */
	switch(offset) {
	case 0:
		/*	$FF90 Initialization register 0
		 *		  Bit 7 COCO 1=CoCo compatible mode
		 *		  Bit 6 MMUEN 1=MMU enabled
		 *		  Bit 5 IEN 1 = GIME chip IRQ enabled
		 *		  Bit 4 FEN 1 = GIME chip FIRQ enabled
		 *		  Bit 3 MC3 1 = RAM at FEXX is constant
		 *		  Bit 2 MC2 1 = standard SCS (Spare Chip Select)
		 *		  Bit 1 MC1 ROM map control
		 *		  Bit 0 MC0 ROM map control
		 */
		coco3_vh_sethires(data & 0x80 ? 0 : 1);
		coco3_mmu_update(0, 7);
		break;

	case 1:
		/*	$FF91 Initialization register 1Bit 7 Unused
		 *		  Bit 6 Unused
		 *		  Bit 5 TINS Timer input select; 1 = 70 nsec, 0 = 63.5 usec
		 *		  Bit 4 Unused
		 *		  Bit 3 Unused
		 *		  Bit 2 Unused
		 *		  Bit 1 Unused
		 *		  Bit 0 TR Task register select
		 */
		coco3_mmu_update(0, 7);
		coco3_timer_set_interval(data & 0x20 ? 1 : 0);
		break;

	case 2:
		/*	$FF92 Interrupt request enable register
		 *		  Bit 7 Unused
		 *		  Bit 6 Unused
		 *		  Bit 5 TMR Timer interrupt
		 *		  Bit 4 HBORD Horizontal border interrupt
		 *		  Bit 3 VBORD Vertical border interrupt
		 *		! Bit 2 EI2 Serial data interrupt
		 *		! Bit 1 EI1 Keyboard interrupt
		 *		  Bit 0 EI0 Cartridge interrupt
		 */
		break;

	case 3:
		/*	$FF93 Fast interrupt request enable register
		 *		  Bit 7 Unused
		 *		  Bit 6 Unused
		 *		  Bit 5 TMR Timer interrupt
		 *		  Bit 4 HBORD Horizontal border interrupt
		 *		  Bit 3 VBORD Vertical border interrupt
		 *		! Bit 2 EI2 Serial border interrupt
		 *		! Bit 1 EI1 Keyboard interrupt
		 *		  Bit 0 EI0 Cartridge interrupt
		 */
		break;

	case 4:
		/*	$FF94 Timer register MSB
		 *		  Bits 4-7 Unused
		 *		  Bits 0-3 High order four bits of the timer
		 */
		coco3_timer_msb_w(data);
		break;

	case 5:
		/*	$FF95 Timer register LSB
		 *		  Bits 0-7 Low order eight bits of the timer
		 */
		coco3_timer_lsb_w(data);
		break;
	}
}

void coco3_sam_himemmap(int offset, int data)
{
	coco3_enable_64k = offset;
	coco3_mmu_update(4, 7);
}

/***************************************************************************
  PIA
***************************************************************************/

int dragon_interrupt(void)
{
	pia_0_cb1_w (0, 1);
	return ignore_interrupt();
}

static void d_pia1_pa_w(int offset, int data)
{
	d_dac = data & 0xfa;
	if (sound_mux)
		DAC_data_w(0,d_dac);
}

static int d_pia0_ca1_r(int offset)
{
	return 0;
}

static int d_pia1_cb1_r(int offset)
{
	return cart_inserted;
}

static void d_pia1_cb2_w(int offset, int data)
{
	sound_mux = data;
}

static void d_pia0_cb2_w(int offset, int data)
{
	joystick = data;
}

static void d_pia1_ca2_w(int offset, int data)
{
	if (tape_motor ^ data)
	{
		device_status(IO_CASSETTE, 0, data ? 1 : 0);
		tape_motor = data;
	}
}

static void d_pia0_ca2_w(int offset, int data)
{
	joystick_axis = data;
}

static int d_pia0_pa_r(int offset)
{
	int porta=0x7f;

	if ((input_port_0_r(0) | pia0_pb) != 0xff) porta &= ~0x01;
	if ((input_port_1_r(0) | pia0_pb) != 0xff) porta &= ~0x02;
	if ((input_port_2_r(0) | pia0_pb) != 0xff) porta &= ~0x04;
	if ((input_port_3_r(0) | pia0_pb) != 0xff) porta &= ~0x08;
	if ((input_port_4_r(0) | pia0_pb) != 0xff) porta &= ~0x10;
	if ((input_port_5_r(0) | pia0_pb) != 0xff) porta &= ~0x20;
	if ((input_port_6_r(0) | pia0_pb) != 0xff) porta &= ~0x40;
	if (d_dac <= (joystick_axis? input_port_8_r(0): input_port_7_r(0)))
		porta |= 0x80;
	porta &= ~input_port_9_r(0);

	return porta;
}

static int d_pia1_pa_r(int offset)
{
	return (device_input(IO_CASSETTE, 0) >= 0) ? 1 : 0;
}

static void d_pia0_pb_w(int offset, int data)
{
	pia0_pb = data;
}

static void d_pia0_irq_a(int state)
{
	cpu_set_irq_line(0, M6809_IRQ_LINE, state);
}

static void d_pia0_irq_b(int state)
{
	cpu_set_irq_line(0, M6809_IRQ_LINE, state);
}

static void coco3_pia0_irq_a(int state)
{
	if (!(coco3_gimereg[0] & 0x20))
		d_pia0_irq_a(state);
}

static void coco3_pia0_irq_b(int state)
{
	if (!(coco3_gimereg[0] & 0x20))
		d_pia0_irq_b(state);
}

/***************************************************************************
  Joystick autocenter
***************************************************************************/

static int autocenter_val;

static void autocenter_timer_proc(int data)
{
	struct InputPort *in;
	int dipport, dipmask, portval;
	
	dipport = (data & 0xff00) >> 8;
	dipmask = data & 0x00ff;
	portval = readinputport(dipport) & dipmask;
	
	if (autocenter_val != portval) {
		/* Now go through all inputs, and set or reset IPF_CENTER on all
		 * joysticks
		 */
		for (in = Machine->input_ports; in->type != IPT_END; in++) {
			if (((in->type & ~IPF_MASK) > IPT_ANALOG_START)
					&& ((in->type & ~IPF_MASK) < IPT_ANALOG_END)) {
				/* We found a joystick */
				if (portval)
					in->type |= IPF_CENTER;
				else
					in->type &= ~IPF_CENTER;
			}
		}
	}
}

static void autocenter_init(int dipport, int dipmask)
{
	autocenter_val = -1;
	timer_pulse(TIME_IN_HZ(10), (dipport << 8) | dipmask, autocenter_timer_proc);
}

/***************************************************************************
  Machine Initialization
***************************************************************************/


static void generic_init_machine(struct pia6821_interface *piaintf)
{
	pia_config(0, PIA_STANDARD_ORDERING | PIA_8BIT, &piaintf[0]);
	pia_config(1, PIA_STANDARD_ORDERING | PIA_8BIT, &piaintf[1]);
	pia_reset();

	if (trailer_load) {
		trailer_load = 0;
		timer_set(0, 0, pak_load_trailer_callback);
	}

	autocenter_init(10, 0x04);
}

void dragon32_init_machine(void)
{
	generic_init_machine(dragon_pia_intf);

	coco_rom = memory_region(REGION_CPU1) + 0x8000;

	if (cart_inserted)
		cpu_set_irq_line(0, M6809_FIRQ_LINE, ASSERT_LINE);
}

void coco_init_machine(void)
{
	generic_init_machine(dragon_pia_intf);

	coco_rom = memory_region(REGION_CPU1) + 0x10000;

	if (cart_inserted)
		cpu_set_irq_line(0, M6809_FIRQ_LINE, ASSERT_LINE);
	dragon64_sam_himemmap(0, 0);
}


void dragon64_init_machine(void)
{
	dragon32_init_machine();
	coco_rom = memory_region(REGION_CPU1) + 0x10000;
	dragon64_sam_himemmap(0, 0);
}

void coco3_init_machine(void)
{
	int i;

	generic_init_machine(coco3_pia_intf);

	coco_rom = memory_region(REGION_CPU1) + 0x80000;

	if (cart_inserted)
		coco3_set_irq_line(COCO3_INT_EI0, ASSERT_LINE);

	coco3_enable_64k = 0;
	for (i = 0; i < 7; i++) {
		coco3_mmu[i] = coco3_mmu[i + 8] = 56 + i;
		coco3_gimereg[i] = 0;
	}
	coco3_mmu_update(0, 7);
	coco3_timer_init();
}

void dragon_stop_machine(void)
{
}

#define WAVEENTRY_HIGH  32767
#define WAVEENTRY_LOW   -32768
#define WAVEENTRY_NULL  0
#define WAVESAMPLES_BYTE    8*4
#define WAVESAMPLES_HEADER  3000
#define WAVESAMPLES_TRAILER 1000

static INT16* fill_wave_byte(INT16 *p, UINT8 b)
{
    int i;
    /* Each byte in a .CAS file is read bit by bit, starting at bit 0, and
     * ending with bit 7.  High bits are decoded into {l,h} (a 2400hz pulse)
     * and low bits are decoded into {l,l,h,h} (a 1200hz pulse)
     */
    for (i = 0; i < 8; i++) {
        *(p++) = WAVEENTRY_LOW;
        if (((b >> i) & 0x01) == 0) {
            *(p++) = WAVEENTRY_LOW;
            *(p++) = WAVEENTRY_HIGH;
        }
        *(p++) = WAVEENTRY_HIGH;
    }
    return p;
}

static int coco_cassette_fill_wave(INT16 *buffer, int length, UINT8 *bytes)
{
    static UINT8 block_type = 0;
    static UINT8 block_length = 0;
    static UINT8 block_chksum = 0;
    static int state = 0;
    int i;
    UINT8 b;
    INT16 *p;

    p = buffer;

    if (bytes == CODE_HEADER) {
        for (i = 0; i < WAVESAMPLES_HEADER; i++)
            *(p++) = WAVEENTRY_NULL;
        block_type = 0x00;  /* reset the block type */
    }
    else if (bytes == CODE_TRAILER) {
        /* fill in one magic byte */
        p = fill_wave_byte(p, 0x55);
        for (i = 0; i < WAVESAMPLES_TRAILER; i++)
            *(p++) = WAVEENTRY_NULL;
    }
    else {
        b = bytes[0];

        switch (state)
        {
        case 0: /* sync bytes */
            if (b == 0x3c)
            {
                if (errorlog) fprintf(errorlog,"COCO wave block start $%02x\n", b);
                state = 1;  /* block type following */
                return 0;
            }
            else
            {
                if (errorlog) fprintf(errorlog,"COCO wave skip sync $%02x\n", b);
                return 0;
            }
            break;
        case 1: /* block type */
            if (errorlog) fprintf(errorlog,"COCO wave block type $%02x\n", b);
            block_chksum = b;
            state = 2;  /* block length following */

            /* was the last block a filename block? */
			if (block_type == 0x00 || block_type == 0xff) {
				if (errorlog) fprintf(errorlog,"COCO filling silence %d\n", WAVESAMPLES_HEADER);
				/* silence */
				for (i = 0; i < WAVESAMPLES_HEADER; i++)
					*(p++) = WAVEENTRY_NULL;
                /* sync data */
                for (i = 0; i < 128; i++)
                    p = fill_wave_byte(p, 0x55);
            }
            /* now fill in the magic bytes */
            p = fill_wave_byte(p, 0x55);
            p = fill_wave_byte(p, 0x3c);
            block_type = b;
            break;
        case 2: /* block length */
            if (errorlog) fprintf(errorlog,"COCO wave block length $%02x (%d)\n", b, b);
            block_length = b;
            block_chksum += b;
            state = 3;  /* data */
            break;
        case 3: /* data bytes */
            if (block_length)
            {
                block_length--;
                block_chksum += b;
            }
            else
            {
                if (errorlog) fprintf(errorlog,"COCO wave block checksum read $%02x, calculated $%02x\n", b, block_chksum);
                state = 0;
                p = fill_wave_byte(p, b);
                /* one trailing magic byte 0x55 */
                b = 0x55;
            }
            break;
        }
        p = fill_wave_byte(p, b);
    }
    return p - buffer;
}

int coco_cassette_init(int id)
{
	void *file;

	file = image_fopen(IO_CASSETTE, id, OSD_FILETYPE_IMAGE_RW, OSD_FOPEN_READ);
	if( file )
	{
		struct wave_args wa;
		memset(&wa, 0, sizeof(&wa));
		wa.file = file;
		wa.chunk_size = 1;
		wa.chunk_samples = 8*4;	/* 8 bits * 4 samples */
		wa.smpfreq = 4800; /* cassette samples go at 4800 baud */
		wa.fill_wave = coco_cassette_fill_wave;
		wa.header_samples = WAVESAMPLES_HEADER;
		wa.trailer_samples = WAVESAMPLES_TRAILER;
		if( device_open(IO_CASSETTE,id,0,&wa) )
			return INIT_FAILED;
		/* immediately pause the output */
        device_status(IO_CASSETTE,id,0);
	}
	return INIT_OK;
}

void coco_cassette_exit(int id)
{
	device_close(IO_CASSETTE,id);
}

int coco3_floppy_r(int offset)
{
	extern int coco_floppy_r(int offset_c);
	return ((coco3_gimereg[0] & 0x04) || (offset >= 0x10)) ? coco_floppy_r(offset) : 0;
}

void coco3_floppy_w(int offset, int data)
{
	extern void coco_floppy_w(int offset_c, int data_c);
	if ((coco3_gimereg[0] & 0x04) || (offset >= 0x10))
		coco_floppy_w(offset, data);
}
