#ifndef __TWKMODES_H
#define __TWKMODES_H

/* Tweaked modes definitions, taken from mame-dos */

/* indices into our register array */
#define	CLOCK_INDEX		0
#define	H_TOTAL_INDEX		1
#define	H_DISPLAY_INDEX		2
#define	H_BLANKING_START_INDEX	3
#define	H_BLANKING_END_INDEX	4
#define	H_RETRACE_START_INDEX	5
#define	H_RETRACE_END_INDEX	6
#define	V_TOTAL_INDEX		7
#define	OVERFLOW_INDEX		8
#define	MAXIMUM_SCANLINE_INDEX	10
#define	V_RETRACE_START_INDEX	11
#define	V_RETRACE_END_INDEX	12
#define	V_END_INDEX		13
#define	MEM_OFFSET_INDEX	14
#define	UNDERLINE_LOC_INDEX	15
#define	V_BLANKING_START_INDEX	16
#define	V_BLANKING_END_INDEX	17
#define MODE_CONTROL_INDEX	18
#define	MEMORY_MODE_INDEX	20

/* -----------------------------------------------------------------------------------------*/
/* NOTES:                                                                                   */
/* horizontal/vertical retrace and blanking timings are genereted by - center_mode()        */
/* most 'scanline' versions of tweaked modes are generated by        - make_scanline_mode() */
/* but some mode's need a custom scanline register set                                      */
/* -----------------------------------------------------------------------------------------*/

/* PC Monitor Tweaked Modes */
/* ------------------------ */


/* 224x288 - vertical mode */
/* Dot Clock - 28MHz */
/* Horizontal Scanrate ~ 35.0KHz*/
/* Vertical Refresh Rate ~60Hz */
/* Mode Type - Linear */
/* Notes: uses a very high horizontal scanrate to fit 288 display lines into 60Hz refresh cycle */
Register scr224x288[] =
{
	{ 0x3c2, 0x00, 0xe7},{ 0x3d4, 0x00, 0x5f},{ 0x3d4, 0x01, 0x37},
	{ 0x3d4, 0x02, 0x38},{ 0x3d4, 0x03, 0x82},{ 0x3d4, 0x04, 0x49},
	{ 0x3d4, 0x05, 0x9a},{ 0x3d4, 0x06, 0x54},{ 0x3d4, 0x07, 0xf0},
	{ 0x3d4, 0x08, 0x00},{ 0x3d4, 0x09, 0x61},{ 0x3d4, 0x10, 0x40},
	{ 0x3d4, 0x11, 0xac},{ 0x3d4, 0x12, 0x3f},{ 0x3d4, 0x13, 0x1c},
	{ 0x3d4, 0x14, 0x40},{ 0x3d4, 0x15, 0x40},{ 0x3d4, 0x16, 0x4a},
	{ 0x3d4, 0x17, 0xa3},{ 0x3c4, 0x01, 0x01},{ 0x3c4, 0x04, 0x0e}
};

Register scr224x288scanlines[] =
{
	{ 0x3c2, 0x00, 0xe3},{ 0x3d4, 0x00, 0x5f},{ 0x3d4, 0x01, 0x37},
	{ 0x3d4, 0x02, 0x38},{ 0x3d4, 0x03, 0x82},{ 0x3d4, 0x04, 0x4a},
	{ 0x3d4, 0x05, 0x9a},{ 0x3d4, 0x06, 0x43},{ 0x3d4, 0x07, 0x1f},
	{ 0x3d4, 0x08, 0x00},{ 0x3d4, 0x09, 0x60},{ 0x3d4, 0x10, 0x2a},
	{ 0x3d4, 0x11, 0xac},{ 0x3d4, 0x12, 0x1f},{ 0x3d4, 0x13, 0x1c},
	{ 0x3d4, 0x14, 0x40},{ 0x3d4, 0x15, 0x27},{ 0x3d4, 0x16, 0x3a},
	{ 0x3d4, 0x17, 0xa3},{ 0x3c4, 0x01, 0x01},{ 0x3c4, 0x04, 0x0e}
};

/* Dot Clock - 28MHz */
/* Horizontal Scanrate ~ 32.4KHz*/
/* Vertical Refresh Rate ~60Hz */
/* Mode Type - Linear */
/* Notes: uses a high horizontal scanrate to fit 256 display lines into 60Hz refresh cycle */
Register scr240x256[] =
{
	{ 0x3c2, 0x00, 0xe7},{ 0x3D4, 0x00, 0x67},{ 0x3d4, 0x01, 0x3b},
	{ 0x3d4, 0x02, 0x40},{ 0x3D4, 0x03, 0x42},{ 0x3d4, 0x04, 0x4e},
	{ 0x3d4, 0x05, 0x9a},{ 0x3D4, 0x06, 0x23},{ 0x3d4, 0x07, 0xb2},
	{ 0x3d4, 0x08, 0x00},{ 0x3D4, 0x09, 0x61},{ 0x3d4, 0x10, 0x0a},
	{ 0x3d4, 0x11, 0xac},{ 0x3D4, 0x12, 0xff},{ 0x3d4, 0x13, 0x1e},
	{ 0x3d4, 0x14, 0x40},{ 0x3d4, 0x15, 0x07},{ 0x3D4, 0x16, 0x11},
	{ 0x3d4, 0x17, 0xa3},{ 0x3c4, 0x01, 0x01},{ 0x3c4, 0x04, 0x0e}
};

/* 256x240 - horizontal mode */
/* Dot Clock - 25MHz */
/* Horizontal Scanrate ~ 34.7KHz*/
/* Vertical Refresh Rate ~60Hz */
/* Mode Type - Linear */
Register scr256x240[] =
{
	{ 0x3c2, 0x00, 0xe3},{ 0x3d4, 0x00, 0x55},{ 0x3d4, 0x01, 0x3f},
	{ 0x3d4, 0x02, 0x80},{ 0x3d4, 0x03, 0x90},{ 0x3d4, 0x04, 0x49},
	{ 0x3d4, 0x05, 0x80},{ 0x3D4, 0x06, 0x43},{ 0x3d4, 0x07, 0xb2},
	{ 0x3d4, 0x08, 0x00},{ 0x3D4, 0x09, 0x61},{ 0x3d4, 0x10, 0x04},
	{ 0x3d4, 0x11, 0xac},{ 0x3D4, 0x12, 0xdf},{ 0x3d4, 0x13, 0x20},
	{ 0x3d4, 0x14, 0x40},{ 0x3d4, 0x15, 0x07},{ 0x3D4, 0x16, 0x11},
	{ 0x3d4, 0x17, 0xa3},{ 0x3c4, 0x01, 0x01},{ 0x3c4, 0x04, 0x0e}
};

/* 256x256 - horizontal mode */
/* Dot Clock - 25MHz */
/* Horizontal Scanrate ~34.7KHz*/
/* Vertical Refresh Rate ~57Hz */
/* Mode Type - Linear */
Register scr256x256[] =
{
	{ 0x3c2, 0x00, 0xe3},{ 0x3d4, 0x00, 0x55},{ 0x3d4, 0x01, 0x3f},
	{ 0x3d4, 0x02, 0x40},{ 0x3d4, 0x03, 0x90},{ 0x3d4, 0x04, 0x49},
	{ 0x3d4, 0x05, 0x80},{ 0x3D4, 0x06, 0x60},{ 0x3d4, 0x07, 0xb2},
	{ 0x3d4, 0x08, 0x00},{ 0x3D4, 0x09, 0x61},{ 0x3d4, 0x10, 0x28},
	{ 0x3d4, 0x11, 0xac},{ 0x3D4, 0x12, 0xff},{ 0x3d4, 0x13, 0x20},
	{ 0x3d4, 0x14, 0x40},{ 0x3d4, 0x15, 0x07},{ 0x3D4, 0x16, 0x11},
	{ 0x3d4, 0x17, 0xa3},{ 0x3c4, 0x01, 0x01},{ 0x3c4, 0x04, 0x0e}
};

/* 256x256 - vertical mode */
/* Dot Clock - 28MHz */
/* Horizontal Scanrate ~ 31KHz*/
/* Vertical Refresh Rate ~57Hz */
/* Mode Type - Linear */
/* Notes: uses a faster dot clock than horizontal the 256x256 mode to make a thinner display */
Register scr256x256ver[] =
{
	{ 0x3c2, 0x00, 0xe7},{ 0x3D4, 0x00, 0x6c},{ 0x3d4, 0x01, 0x3f},
	{ 0x3d4, 0x02, 0x40},{ 0x3D4, 0x03, 0x42},{ 0x3d4, 0x04, 0x51},
	{ 0x3d4, 0x05, 0x9a},{ 0x3D4, 0x06, 0x23},{ 0x3d4, 0x07, 0xb2},
	{ 0x3d4, 0x08, 0x00},{ 0x3D4, 0x09, 0x61},{ 0x3d4, 0x10, 0x0a},
	{ 0x3d4, 0x11, 0xac},{ 0x3D4, 0x12, 0xff},{ 0x3d4, 0x13, 0x20},
	{ 0x3d4, 0x14, 0x40},{ 0x3d4, 0x15, 0x07},{ 0x3D4, 0x16, 0x11},
	{ 0x3d4, 0x17, 0xa3},{ 0x3c4, 0x01, 0x01},{ 0x3c4, 0x04, 0x0e}
};

/* 288x224 - horizontal mode */
/* Dot Clock - 25MHz */
/* Horizontal Scanrate ~31.2KHz*/
/* Vertical Refresh Rate ~60Hz */
/* Mode Type - Linear */
Register scr288x224[] =
{
	{ 0x3c2, 0x00, 0xe3},{ 0x3d4, 0x00, 0x5f},{ 0x3d4, 0x01, 0x47},
	{ 0x3d4, 0x02, 0x49},{ 0x3d4, 0x03, 0x90},{ 0x3d4, 0x04, 0x50},
	{ 0x3d4, 0x05, 0x80},{ 0x3d4, 0x06, 0x0c},{ 0x3d4, 0x07, 0x3e},
	{ 0x3d4, 0x08, 0x00},{ 0x3d4, 0x09, 0x41},{ 0x3d4, 0x10, 0xd8},
	{ 0x3d4, 0x11, 0xac},{ 0x3d4, 0x12, 0xbf},{ 0x3d4, 0x13, 0x24},
	{ 0x3d4, 0x14, 0x40},{ 0x3d4, 0x15, 0xc2},{ 0x3d4, 0x16, 0xf4},
	{ 0x3d4, 0x17, 0xa3},{ 0x3c4, 0x01, 0x01},{ 0x3c4, 0x04, 0x0e}
};

/* 240x320 - vertical mode */
/* Dot Clock - 28MHz */
/* Horizontal Scanrate ~36.8KHz*/
/* Vertical Refresh Rate ~56Hz */
/* Mode Type - Planar */
/* Notes: uses a very high horizontal scanrate to fit 320 display lines into 56Hz refresh cycle */
Register scr240x320[] =
{
	{ 0x3c2, 0x00, 0xe7},{ 0x3d4, 0x00, 0x5a},{ 0x3d4, 0x01, 0x3b},
	{ 0x3d4, 0x02, 0x3c},{ 0x3d4, 0x03, 0x96},{ 0x3d4, 0x04, 0x47},
	{ 0x3d4, 0x05, 0x14},{ 0x3d4, 0x06, 0x8c},{ 0x3d4, 0x07, 0xf0},
	{ 0x3d4, 0x08, 0x00},{ 0x3d4, 0x09, 0x61},{ 0x3d4, 0x10, 0x80},
	{ 0x3d4, 0x11, 0x92},{ 0x3d4, 0x12, 0x7f},{ 0x3d4, 0x13, 0x1e},
	{ 0x3d4, 0x14, 0x00},{ 0x3d4, 0x15, 0x80},{ 0x3d4, 0x16, 0x84},
	{ 0x3d4, 0x17, 0xe3},{ 0x3c4, 0x01, 0x01},{ 0x3c4, 0x04, 0x06}
};

/* 320x240 - horizontal mode */
/* Dot Clock - 25MHz */
/* Horizontal Scanrate ~ 31.2KHz*/
/* Vertical Refresh Rate ~60Hz */
/* Mode Type - Planar */
Register scr320x240[] =
{
	{ 0x3c2, 0x00, 0xe3},{ 0x3d4, 0x00, 0x5f},{ 0x3d4, 0x01, 0x4f},
	{ 0x3d4, 0x02, 0x50},{ 0x3d4, 0x03, 0x82},{ 0x3d4, 0x04, 0x54},
	{ 0x3d4, 0x05, 0x80},{ 0x3d4, 0x06, 0x0c},{ 0x3d4, 0x07, 0x3e},
	{ 0x3d4, 0x08, 0x00},{ 0x3d4, 0x09, 0x41},{ 0x3d4, 0x10, 0xe9},
	{ 0x3d4, 0x11, 0xac},{ 0x3d4, 0x12, 0xdf},{ 0x3d4, 0x13, 0x28},
	{ 0x3d4, 0x14, 0x00},{ 0x3d4, 0x15, 0xe7},{ 0x3d4, 0x16, 0x06},
	{ 0x3d4, 0x17, 0xe3},{ 0x3c4, 0x01, 0x01},{ 0x3c4, 0x04, 0x06}
};

/* 336x240 - horizontal mode */
/* Dot Clock - 25MHz */
/* Horizontal Scanrate ~ 31.2KHz*/
/* Vertical Refresh Rate ~60Hz */
/* Mode Type - Planar */
Register scr336x240[] =
{
	{ 0x3c2, 0x00, 0xe3},{ 0x3d4, 0x00, 0x5f},{ 0x3d4, 0x01, 0x53},
	{ 0x3d4, 0x02, 0x56},{ 0x3d4, 0x03, 0x80},{ 0x3d4, 0x04, 0x57},
	{ 0x3d4, 0x05, 0x9b},{ 0x3d4, 0x06, 0x0c},{ 0x3d4, 0x07, 0x3e},
	{ 0x3d4, 0x08, 0x00},{ 0x3d4, 0x09, 0x41},{ 0x3d4, 0x10, 0xe5},
	{ 0x3d4, 0x11, 0xac},{ 0x3d4, 0x12, 0xdf},{ 0x3d4, 0x13, 0x2a},
	{ 0x3d4, 0x14, 0x00},{ 0x3d4, 0x15, 0xe8},{ 0x3d4, 0x16, 0x02},
	{ 0x3d4, 0x17, 0xe3},{ 0x3c4, 0x01, 0x01},{ 0x3c4, 0x04, 0x06}
};

/* 384x224 - horizontal mode */
/* Dot Clock - 28MHz */
/* Horizontal Scanrate ~ 31KHz*/
/* Vertical Refresh Rate ~60Hz */
/* Mode Type - Planar */
/* Notes: this mode is basically identical to 384x240, but allows VGA triple buffering */
Register scr384x224[] =
{
	{ 0x3c2, 0x00, 0xe7},{ 0x3d4, 0x00, 0x6c},{ 0x3d4, 0x01, 0x5f},
	{ 0x3d4, 0x02, 0x61},{ 0x3d4, 0x03, 0x8b},{ 0x3d4, 0x04, 0x62},
	{ 0x3d4, 0x05, 0x86},{ 0x3d4, 0x06, 0x0c},{ 0x3d4, 0x07, 0x3e},
	{ 0x3d4, 0x08, 0x00},{ 0x3d4, 0x09, 0x41},{ 0x3d4, 0x10, 0xd8},
	{ 0x3d4, 0x11, 0xac},{ 0x3d4, 0x12, 0xbf},{ 0x3d4, 0x13, 0x30},
	{ 0x3d4, 0x14, 0x00},{ 0x3d4, 0x15, 0xc2},{ 0x3d4, 0x16, 0xf4},
	{ 0x3d4, 0x17, 0xe3},{ 0x3c4, 0x01, 0x01},{ 0x3c4, 0x04, 0x06}
};

/* 384x240 - horizontal mode */
/* Dot Clock - 28MHz */
/* Horizontal Scanrate ~ 31KHz*/
/* Vertical Refresh Rate ~60Hz */
/* Mode Type - Planar */
Register scr384x240[] =
{
	{ 0x3c2, 0x00, 0xe7},{ 0x3d4, 0x00, 0x6c},{ 0x3d4, 0x01, 0x5f},
	{ 0x3d4, 0x02, 0x61},{ 0x3d4, 0x03, 0x8b},{ 0x3d4, 0x04, 0x62},
	{ 0x3d4, 0x05, 0x86},{ 0x3d4, 0x06, 0x0c},{ 0x3d4, 0x07, 0x3e},
	{ 0x3d4, 0x08, 0x00},{ 0x3d4, 0x09, 0x41},{ 0x3d4, 0x10, 0xe9},
	{ 0x3d4, 0x11, 0xac},{ 0x3d4, 0x12, 0xdf},{ 0x3d4, 0x13, 0x30},
	{ 0x3d4, 0x14, 0x00},{ 0x3d4, 0x15, 0xe8},{ 0x3d4, 0x16, 0x04},
	{ 0x3d4, 0x17, 0xe3},{ 0x3c4, 0x01, 0x01},{ 0x3c4, 0x04, 0x06}
};

/* 384x256 - horizontal mode */
/* Dot Clock - 28MHz */
/* Horizontal Scanrate ~ 31KHz*/
/* Vertical Refresh Rate ~57Hz */
/* Mode Type - Planar */
Register scr384x256[] =
{
	{ 0x3c2, 0x00, 0xe7},{ 0x3d4, 0x00, 0x6c},{ 0x3d4, 0x01, 0x5f},
	{ 0x3d4, 0x02, 0x61},{ 0x3d4, 0x03, 0x8b},{ 0x3d4, 0x04, 0x62},
	{ 0x3d4, 0x05, 0x86},{ 0x3D4, 0x06, 0x23},{ 0x3d4, 0x07, 0x36},
	{ 0x3d4, 0x08, 0x00},{ 0x3D4, 0x09, 0x61},{ 0x3d4, 0x10, 0xff},
	{ 0x3d4, 0x11, 0xac},{ 0x3D4, 0x12, 0xff},{ 0x3d4, 0x13, 0x30},
	{ 0x3d4, 0x14, 0x00},{ 0x3d4, 0x15, 0x07},{ 0x3D4, 0x16, 0x11},
	{ 0x3d4, 0x17, 0xe3},{ 0x3c4, 0x01, 0x01},{ 0x3c4, 0x04, 0x06}
};

/* old svgalib tweak modes */

/* 320x204 - horizontal mode */
/* Mode Type - Linear */
Register scr320x204[] =
{
	{ 0x3c2, 0x00, 0xe3},{ 0x3d4, 0x00, 0x5f},{ 0x3d4, 0x01, 0x4f},
	{ 0x3d4, 0x02, 0x50},{ 0x3d4, 0x03, 0x82},{ 0x3d4, 0x04, 0x54},
	{ 0x3d4, 0x05, 0x80},{ 0x3d4, 0x06, 0xbf},{ 0x3d4, 0x07, 0x1f},
	{ 0x3d4, 0x08, 0x00},{ 0x3d4, 0x09, 0x41},{ 0x3d4, 0x10, 0x9c},
	{ 0x3d4, 0x11, 0x8e},{ 0x3d4, 0x12, 0x97},{ 0x3d4, 0x13, 0x28},
	{ 0x3d4, 0x14, 0x40},{ 0x3d4, 0x15, 0x96},{ 0x3d4, 0x16, 0xb9},
	{ 0x3d4, 0x17, 0xa3},{ 0x3c4, 0x01, 0x01},{ 0x3c4, 0x04, 0x0e},
};

/* 200x320 - vertical mode */
/* Mode Type - Linear */
Register scr200x320[] =
{
	{ 0x3c2, 0x00, 0xe3},{ 0x3d4, 0x00, 0x5f},{ 0x3d4, 0x01, 0x31},
	{ 0x3d4, 0x02, 0x38},{ 0x3d4, 0x03, 0x82},{ 0x3d4, 0x04, 0x4a},
	{ 0x3d4, 0x05, 0x9a},{ 0x3d4, 0x06, 0x4e},{ 0x3d4, 0x07, 0x1f},
	{ 0x3d4, 0x08, 0x00},{ 0x3d4, 0x09, 0x40},{ 0x3d4, 0x10, 0x40},
	{ 0x3d4, 0x11, 0x90},{ 0x3d4, 0x12, 0x3f},{ 0x3d4, 0x13, 0x19},
	{ 0x3d4, 0x14, 0x40},{ 0x3d4, 0x15, 0x80},{ 0x3d4, 0x16, 0x40},
	{ 0x3d4, 0x17, 0xa3},{ 0x3c4, 0x01, 0x01},{ 0x3c4, 0x04, 0x0e}
};

Register *make_scanline_mode(Register *inreg,int entries)
{
	static Register outreg[32];
	int maxscan,maxscanout;
	int overflow,overflowout;
	int ytotalin,ytotalout;
	int ydispin,ydispout;
	int vrsin,vrsout,vreout,vblksout,vblkeout;
/* first - check's it not already a 'non doubled' line mode */
	maxscan = inreg[MAXIMUM_SCANLINE_INDEX].value;
	if ((maxscan & 1) == 0)
	/* it is, so just return the array as is */
  		return inreg;
/* copy across our standard display array */
	memcpy (&outreg, inreg, entries * sizeof(Register));
/* keep hold of the overflow register - as we'll need to refer to it a lot */
	overflow = inreg[OVERFLOW_INDEX].value;
/* set a large line compare value  - as we won't be doing any split window scrolling etc.*/
	maxscanout = 0x40;
/* half all the y values */
/* total */
	ytotalin = inreg[V_TOTAL_INDEX].value;
	ytotalin |= ((overflow & 1)<<0x08) | ((overflow & 0x20)<<0x04);
    ytotalout = ytotalin >> 1;
/* display enable end */
	ydispin = inreg[13].value | ((overflow & 0x02)<< 0x07) | ((overflow & 0x040) << 0x03);
	ydispin ++;
	ydispout = ydispin >> 1;
	ydispout --;
	overflowout = ((ydispout & 0x100) >> 0x07) | ((ydispout && 0x200) >> 0x03);
	outreg[V_END_INDEX].value = (ydispout & 0xff);
/* avoid top over scan */
	if ((ytotalin - ydispin) < 40 && !center_y)
	{
  		vrsout = ydispout;
		/* give ourselves a scanline cushion */
		ytotalout += 2;
	}
  	else
	{
/* vertical retrace start */
		vrsin = inreg[V_RETRACE_START_INDEX].value | ((overflow & 0x04)<<0x06) | ((overflow & 0x80)<<0x02);
		vrsout = vrsin >> 1;
	}
/* check it's legal */
	if (vrsout < ydispout)
		vrsout = ydispout;
/*update our output overflow */
	overflowout |= (((vrsout & 0x100) >> 0x06) | ((vrsout & 0x200) >> 0x02));
	outreg[V_RETRACE_START_INDEX].value = (vrsout & 0xff);
/* vertical retrace end */
	vreout = vrsout + 2;
/* make sure the retrace fits into our adjusted display size */
	if (vreout > (ytotalout - 9))
		ytotalout = vreout + 9;
/* write out the vertical retrace end */
	outreg[V_RETRACE_END_INDEX].value &= ~0x0f;
	outreg[V_RETRACE_END_INDEX].value |= (vreout & 0x0f);
/* vertical blanking start */
	vblksout = ydispout + 1;
/* check it's legal */
	if(vblksout > vreout)
		vblksout = vreout;
/* save the overflow value */
	overflowout |= ((vblksout & 0x100) >> 0x05);
	maxscanout |= ((vblksout & 0x200) >> 0x04);
/* write the v blank value out */
	outreg[V_BLANKING_START_INDEX].value = (vblksout & 0xff);
/* vertical blanking end */
	vblkeout = vreout + 1;
/* make sure the blanking fits into our adjusted display size */
	if (vblkeout > (ytotalout - 9))
		ytotalout = vblkeout + 9;
/* write out the vertical blanking total */
	outreg[V_BLANKING_END_INDEX].value = (vblkeout & 0xff);
/* update our output overflow */
	overflowout |= ((ytotalout & 0x100) >> 0x08) | ((ytotalout & 0x200) >> 0x04);
/* write out the new vertical total */
	outreg[V_TOTAL_INDEX].value = (ytotalout & 0xff);

/* write out our over flows */
	outreg[OVERFLOW_INDEX].value = overflowout;
/* finally the max scan line */
	outreg[MAXIMUM_SCANLINE_INDEX].value = maxscanout;
/* and we're done */
	return outreg;

}

void center_mode(Register *pReg)
{
	int center;
	int hrt_start, hrt_end, hrt, hblnk_start, hblnk_end;
	int vrt_start, vrt_end, vert_total, vert_display, vblnk_start, vrt, vblnk_end;
/* check for empty array */
	if (!pReg)
		return;
/* vertical retrace width */
	vrt = 2;
/* check the clock speed, to work out the retrace width */
	if (pReg[CLOCK_INDEX].value == 0xe7)
		hrt = 11;
	else
		hrt = 10;
/* our center x tweak value */
	center = center_x;
/* check for double width scanline rather than half clock (15.75kHz modes) */
	if( pReg[H_TOTAL_INDEX].value > 0x96)
	{
		center<<=1;
		hrt<<=1;
	}
/* set the hz retrace */
	hrt_start = pReg[H_RETRACE_START_INDEX].value;
	hrt_start += center;
/* make sure it's legal */
	if (hrt_start <= pReg[H_DISPLAY_INDEX].value)
		hrt_start = pReg[H_DISPLAY_INDEX].value + 1;
	pReg[H_RETRACE_START_INDEX].value = hrt_start;
/* set hz retrace end */
	hrt_end = hrt_start + hrt;
/* make sure it's legal */
	if( hrt_end > pReg[H_TOTAL_INDEX].value)
		hrt_end = pReg[H_TOTAL_INDEX].value;

/* set the hz blanking */
	hblnk_start = pReg[H_DISPLAY_INDEX].value + 1;
/* make sure it's legal */
	if (hblnk_start > hrt_start)
		hblnk_start = pReg[H_RETRACE_START_INDEX].value;

	pReg[H_BLANKING_START_INDEX].value = hblnk_start;
/* the horizontal blanking end */
	hblnk_end = hrt_end + 2;
/* make sure it's legal */
	if( hblnk_end > pReg[H_TOTAL_INDEX].value)
		hblnk_end = pReg[H_TOTAL_INDEX].value;
/* write horizontal blanking - include 7th test bit (always 1) */
	pReg[H_BLANKING_END_INDEX].value = (hblnk_end & 0x1f) | 0x80;
/* include the 5th bit of the horizontal blanking in the horizontal retrace reg. */
	hrt_end = ((hrt_end & 0x1f) | ((hblnk_end & 0x20) << 2));
	pReg[H_RETRACE_END_INDEX].value = hrt_end;


/* get the vt retrace */
	vrt_start = pReg[V_RETRACE_START_INDEX].value | ((pReg[OVERFLOW_INDEX].value & 0x04) << 6) |
				((pReg[OVERFLOW_INDEX].value & 0x80) << 2);

/* set the new retrace start */
	vrt_start += center_y;
/* check it's legal, get the display line count */
	vert_display = (pReg[V_END_INDEX].value | ((pReg[OVERFLOW_INDEX].value & 0x02) << 7) |
				((pReg[OVERFLOW_INDEX].value & 0x40) << 3)) + 1;

	if (vrt_start < vert_display)
		vrt_start = vert_display;

/* and get the vertical line count */
	vert_total = pReg[V_TOTAL_INDEX].value | ((pReg[OVERFLOW_INDEX].value & 0x01) << 8) |
				((pReg[OVERFLOW_INDEX].value & 0x20) << 4);



	pReg[V_RETRACE_START_INDEX].value = (vrt_start & 0xff);
	pReg[OVERFLOW_INDEX].value &= ~0x84;
	pReg[OVERFLOW_INDEX].value |= ((vrt_start & 0x100) >> 6);
	pReg[OVERFLOW_INDEX].value |= ((vrt_start & 0x200) >> 2);
	vrt_end = vrt_start + vrt;


	if (vrt_end > vert_total)
		vrt_end = vert_total;

/* write retrace end, include CRT protection and IRQ2 bits */
	pReg[V_RETRACE_END_INDEX].value = (vrt_end  & 0x0f) | 0x80 | 0x20;

/* get the start of vt blanking */
	vblnk_start = vert_display + 1;
/* check it's legal */
	if (vblnk_start > vrt_start)
		vblnk_start = vrt_start;
/* and the end */
	vblnk_end = vrt_end + 2;
/* check it's legal */
	if (vblnk_end > vert_total)
		vblnk_end = vert_total;
/* set vblank start */
	pReg[V_BLANKING_START_INDEX].value = (vblnk_start & 0xff);
/* write out any overflows */
	pReg[OVERFLOW_INDEX].value &= ~0x08;
	pReg[OVERFLOW_INDEX].value |= ((vblnk_start & 0x100) >> 5);
	pReg[MAXIMUM_SCANLINE_INDEX].value &= ~0x20;
	pReg[MAXIMUM_SCANLINE_INDEX].value |= ((vblnk_start &0x200) >> 4);
/* set the vblank end */
	pReg[V_BLANKING_END_INDEX].value = (vblnk_end & 0xff);
}

#endif /* ifndef __TWKMODES_H */
