/* m6847.c -- Implementation of Motorola 6847 video hardware chip
 *
 * Nate Woods
 *
 * Originally based on src/mess/vihrdw/dragon.c by Mathis Rosenhauer
 */

#include "m6847.h"
#include "vidhrdw/generic.h"

static int vram_mask;
static int video_offset;
static int video_gmode;
static int video_vmode;
static int video_rowheight;
static m6847_vblank_proc vblankproc;
static int artifact_dipswitch;

/* The CoCo 3 Video Hardware shares these variables, which is why they are not static */
int m6847_full_refresh;

typedef void (*artifactproc)(int *artifactcolors);

#define MAX_VRAM 6144

static unsigned char palette[] = {
	0x00,0x00,0x00, /* BLACK */
	0x00,0xff,0x00, /* GREEN */
	0xff,0xff,0x00, /* YELLOW */
	0x00,0x00,0xff, /* BLUE */
	0xff,0x00,0x00, /* RED */
	0xff,0xff,0xff, /* BUFF */
	0x00,0xff,0xff, /* CYAN */
	0xff,0x00,0xff, /* MAGENTA */
	0xff,0x80,0x00, /* ORANGE */
	0x00,0x80,0x00, /* ARTIFACT GREEN/RED */
	0x00,0x80,0x00, /* ARTIFACT GREEN/BLUE */
	0xff,0x80,0x00, /* ARTIFACT BUFF/RED */
	0x00,0x80,0xff, /* ARTIFACT BUFF/BLUE */
	0x00,0x40,0x00,	/* ALPHANUMERIC DARK GREEN */
	0x00,0xff,0x00,	/* ALPHANUMERIC BRIGHT GREEN */
	0x40,0x10,0x00,	/* ALPHANUMERIC DARK ORANGE */
	0xff,0xc4,0x18,	/* ALPHANUMERIC BRIGHT ORANGE */
};

static unsigned char fontdata8x12[] =
{
	0x00, 0x00, 0x00, 0x1c, 0x22, 0x02, 0x1a, 0x2a, 0x2a, 0x1c, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x14, 0x22, 0x22, 0x3e, 0x22, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3c, 0x12, 0x12, 0x1c, 0x12, 0x12, 0x3c, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1c, 0x22, 0x20, 0x20, 0x20, 0x22, 0x1c, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3c, 0x12, 0x12, 0x12, 0x12, 0x12, 0x3c, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3e, 0x20, 0x20, 0x3c, 0x20, 0x20, 0x3e, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3e, 0x20, 0x20, 0x3c, 0x20, 0x20, 0x20, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1e, 0x20, 0x20, 0x26, 0x22, 0x22, 0x1e, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x22, 0x22, 0x3e, 0x22, 0x22, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1c, 0x08, 0x08, 0x08, 0x08, 0x08, 0x1c, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x02, 0x02, 0x02, 0x02, 0x22, 0x22, 0x1c, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x24, 0x28, 0x30, 0x28, 0x24, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x3e, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x36, 0x2a, 0x2a, 0x22, 0x22, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x32, 0x2a, 0x26, 0x22, 0x22, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3e, 0x22, 0x22, 0x22, 0x22, 0x22, 0x3e, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3c, 0x22, 0x22, 0x3c, 0x20, 0x20, 0x20, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1c, 0x22, 0x22, 0x22, 0x2a, 0x24, 0x1a, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3c, 0x22, 0x22, 0x3c, 0x28, 0x24, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1c, 0x22, 0x10, 0x08, 0x04, 0x22, 0x1c, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3e, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x1c, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x22, 0x22, 0x14, 0x14, 0x08, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x22, 0x22, 0x2a, 0x2a, 0x36, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x22, 0x14, 0x08, 0x14, 0x22, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x22, 0x14, 0x08, 0x08, 0x08, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3e, 0x02, 0x04, 0x08, 0x10, 0x20, 0x3e, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x38, 0x20, 0x20, 0x20, 0x20, 0x20, 0x38, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x20, 0x20, 0x10, 0x08, 0x04, 0x02, 0x02, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x0e, 0x02, 0x02, 0x02, 0x02, 0x02, 0x0e, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x1c, 0x2a, 0x08, 0x08, 0x08, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x08, 0x10, 0x3e, 0x10, 0x08, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x14, 0x14, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x14, 0x14, 0x36, 0x00, 0x36, 0x14, 0x14, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x1e, 0x20, 0x1c, 0x02, 0x3c, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x32, 0x32, 0x04, 0x08, 0x10, 0x26, 0x26, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x28, 0x28, 0x10, 0x2a, 0x24, 0x1a, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x10, 0x20, 0x20, 0x20, 0x10, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x04, 0x02, 0x02, 0x02, 0x04, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x08, 0x1c, 0x3e, 0x1c, 0x08, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x08, 0x08, 0x3e, 0x08, 0x08, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x10, 0x20, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x02, 0x02, 0x04, 0x08, 0x10, 0x20, 0x20, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x24, 0x24, 0x24, 0x24, 0x24, 0x18, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x18, 0x08, 0x08, 0x08, 0x08, 0x1c, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1c, 0x22, 0x02, 0x1c, 0x20, 0x20, 0x3e, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1c, 0x22, 0x02, 0x0c, 0x02, 0x22, 0x1c, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x04, 0x0c, 0x14, 0x3e, 0x04, 0x04, 0x04, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3e, 0x20, 0x3c, 0x02, 0x02, 0x22, 0x1c, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1c, 0x20, 0x20, 0x3c, 0x22, 0x22, 0x1c, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3e, 0x02, 0x04, 0x08, 0x10, 0x20, 0x20, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1c, 0x22, 0x22, 0x1c, 0x22, 0x22, 0x1c, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1c, 0x22, 0x22, 0x1e, 0x02, 0x02, 0x1c, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x18, 0x18, 0x08, 0x10, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x04, 0x08, 0x10, 0x20, 0x10, 0x08, 0x04, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x00, 0x3e, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x08, 0x04, 0x02, 0x04, 0x08, 0x10, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x24, 0x04, 0x08, 0x08, 0x00, 0x08, 0x00, 0x00,

	/* Semigraphics 6 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x0f, 0x0f,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf0, 0xf0, 0xf0, 0xf0,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
	0x00, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,
	0x00, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x0f, 0x0f, 0xf0, 0xf0, 0xf0, 0xf0,
	0x00, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff,
	0x00, 0x00, 0x00, 0x00, 0xf0, 0xf0, 0xf0, 0xf0, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0xf0, 0xf0, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0x0f,
	0x00, 0x00, 0x00, 0x00, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
	0x00, 0x00, 0x00, 0x00, 0xf0, 0xf0, 0xf0, 0xf0, 0xff, 0xff, 0xff, 0xff,
	0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x0f, 0x0f, 0x0f, 0x0f,
	0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0, 0xf0, 0xf0,
	0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0x0f, 0x0f, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x0f, 0x0f, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x0f, 0x0f,
	0x0f, 0x0f, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0xf0, 0xf0, 0xf0, 0xf0,
	0x0f, 0x0f, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
	0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00,
	0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,
	0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0xf0, 0xf0, 0xf0, 0xf0,
	0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff,
	0x0f, 0x0f, 0x0f, 0x0f, 0xf0, 0xf0, 0xf0, 0xf0, 0x00, 0x00, 0x00, 0x00,
	0x0f, 0x0f, 0x0f, 0x0f, 0xf0, 0xf0, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0x0f,
	0x0f, 0x0f, 0x0f, 0x0f, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
	0x0f, 0x0f, 0x0f, 0x0f, 0xf0, 0xf0, 0xf0, 0xf0, 0xff, 0xff, 0xff, 0xff,
	0x0f, 0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,
	0x0f, 0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff, 0x0f, 0x0f, 0x0f, 0x0f,
	0x0f, 0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0, 0xf0, 0xf0,
	0x0f, 0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xf0, 0xf0, 0xf0, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xf0, 0xf0, 0xf0, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x0f, 0x0f,
	0xf0, 0xf0, 0xf0, 0xf0, 0x00, 0x00, 0x00, 0x00, 0xf0, 0xf0, 0xf0, 0xf0,
	0xf0, 0xf0, 0xf0, 0xf0, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
	0xf0, 0xf0, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00,
	0xf0, 0xf0, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,
	0xf0, 0xf0, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0x0f, 0xf0, 0xf0, 0xf0, 0xf0,
	0xf0, 0xf0, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff,
	0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0x00, 0x00, 0x00, 0x00,
	0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0x0f,
	0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
	0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xff, 0xff, 0xff, 0xff,
	0xf0, 0xf0, 0xf0, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,
	0xf0, 0xf0, 0xf0, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x0f, 0x0f, 0x0f, 0x0f,
	0xf0, 0xf0, 0xf0, 0xf0, 0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0, 0xf0, 0xf0,
	0xf0, 0xf0, 0xf0, 0xf0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x0f, 0x0f,
	0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0xf0, 0xf0, 0xf0, 0xf0,
	0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0x0f, 0x0f, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00,
	0xff, 0xff, 0xff, 0xff, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,
	0xff, 0xff, 0xff, 0xff, 0x0f, 0x0f, 0x0f, 0x0f, 0xf0, 0xf0, 0xf0, 0xf0,
	0xff, 0xff, 0xff, 0xff, 0x0f, 0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0, 0xf0, 0xf0, 0x00, 0x00, 0x00, 0x00,
	0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0, 0xf0, 0xf0, 0x0f, 0x0f, 0x0f, 0x0f,
	0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
	0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0, 0xf0, 0xf0, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0f, 0x0f, 0x0f, 0x0f,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0, 0xf0, 0xf0,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,

	/* Block Graphics (Semigraphics 4 Graphics ) */
	0x00,0x00,0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00, 0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x00,0x00,0x00,0x00,0x00,0x00, 0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,
	0x00,0x00,0x00,0x00,0x00,0x00, 0xff,0xff,0xff,0xff,0xff,0xff,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f, 0x00,0x00,0x00,0x00,0x00,0x00,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f, 0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f, 0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f, 0xff,0xff,0xff,0xff,0xff,0xff,
	0xf0,0xf0,0xf0,0xf0,0xf0,0xf0, 0x00,0x00,0x00,0x00,0x00,0x00,
	0xf0,0xf0,0xf0,0xf0,0xf0,0xf0, 0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0xf0,0xf0,0xf0,0xf0,0xf0,0xf0, 0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,
	0xf0,0xf0,0xf0,0xf0,0xf0,0xf0, 0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff, 0x00,0x00,0x00,0x00,0x00,0x00,
	0xff,0xff,0xff,0xff,0xff,0xff, 0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0xff,0xff,0xff,0xff,0xff,0xff, 0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,
	0xff,0xff,0xff,0xff,0xff,0xff, 0xff,0xff,0xff,0xff,0xff,0xff,

	/* Lower case */
	0x00, 0x00, 0x00, 0x0C, 0x12, 0x10, 0x38, 0x10, 0x12, 0x3C, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x1C, 0x02, 0x1E, 0x22, 0x1E, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x20, 0x20, 0x3C, 0x22, 0x22, 0x22, 0x3C, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x1C, 0x20, 0x20, 0x20, 0x1C, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x02, 0x02, 0x1E, 0x22, 0x22, 0x22, 0x1E, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x1C, 0x22, 0x3E, 0x20, 0x1C, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x0C, 0x12, 0x10, 0x38, 0x10, 0x10, 0x10, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x1E, 0x22, 0x22, 0x22, 0x1E, 0x02, 0x1C, 
	0x00, 0x00, 0x00, 0x20, 0x20, 0x3C, 0x22, 0x22, 0x22, 0x22, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x08, 0x00, 0x18, 0x08, 0x08, 0x08, 0x1C, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x04, 0x00, 0x0C, 0x04, 0x04, 0x04, 0x04, 0x24, 0x18, 
	0x00, 0x00, 0x00, 0x20, 0x20, 0x24, 0x28, 0x38, 0x24, 0x22, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x18, 0x08, 0x08, 0x08, 0x08, 0x08, 0x1C, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x3C, 0x2A, 0x2A, 0x2A, 0x2A, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x2C, 0x32, 0x22, 0x22, 0x22, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x1C, 0x22, 0x22, 0x22, 0x1C, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x3C, 0x22, 0x22, 0x22, 0x3C, 0x20, 0x20, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x1E, 0x22, 0x22, 0x22, 0x1E, 0x02, 0x03, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x2C, 0x32, 0x20, 0x20, 0x20, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x1E, 0x20, 0x1C, 0x02, 0x3C, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x10, 0x3C, 0x10, 0x10, 0x10, 0x12, 0x0C, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x22, 0x22, 0x22, 0x26, 0x1A, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x22, 0x22, 0x14, 0x14, 0x08, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x22, 0x2A, 0x2A, 0x1C, 0x14, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x22, 0x14, 0x08, 0x14, 0x22, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x22, 0x22, 0x22, 0x22, 0x1E, 0x02, 0x1C, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x3E, 0x04, 0x08, 0x10, 0x3E, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x08, 0x10, 0x10, 0x20, 0x10, 0x10, 0x08, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x08, 0x08, 0x08, 0x00, 0x08, 0x08, 0x08, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x08, 0x04, 0x04, 0x02, 0x04, 0x04, 0x08, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x08, 0x08, 0x08, 0x08, 0x2A, 0x1C, 0x08, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x08, 0x04, 0x3E, 0x04, 0x08, 0x00, 0x00, 0x00, 0x00
};

static void calc_videoram_size(void)
{
	static int videoramsizes[] = {
		512,	512,	512,	512,	512,	512,	512,	512,
		1024,	1024,	2048,	1536,	3072,	3072,	6144,	6144
	};
	static int rowheights[] = {
		12,		12,		12,		12,		12,		12,		12,		12,
		3,		3,		3,		2,		2,		1,		1,		1
	};
	videoram_size = videoramsizes[video_vmode >> 1];
	video_rowheight = rowheights[video_vmode >> 1];
}

/* --------------------------------------------------
 * Initialization and termination
 * -------------------------------------------------- */

void m6847_vh_init_palette(unsigned char *sys_palette, unsigned short *sys_colortable,const unsigned char *color_prom)
{
	memcpy(sys_palette,palette,sizeof(palette));
}

int internal_m6847_vh_start(int maxvram)
{
	m6847_full_refresh = 1;
	video_offset = 0;
	video_gmode = 0;
	video_vmode = 0;
	vblankproc = NULL;
	artifact_dipswitch = -1;

	calc_videoram_size();

	videoram_size = maxvram;
	if (generic_vh_start())
		return 1;

	return 0;
}

int m6847_vh_start(void)
{
	return internal_m6847_vh_start(MAX_VRAM);
}

void m6847_set_vram(void *ram, int rammask)
{
	videoram = ram;
	vram_mask = rammask;
}

void m6847_set_vram_mask(int rammask)
{
	vram_mask = rammask;
}

void m6847_set_vblank_proc(m6847_vblank_proc proc)
{
	vblankproc = proc;
}

void m6847_set_artifact_dipswitch(int sw)
{
	artifact_dipswitch = sw;
}

/* --------------------------------------------------
 * The big one - updating the display
 * -------------------------------------------------- */

#define PLOT_PIXEL(x, y, c) \
	{ \
		int xx, yy, cc; \
		xx = (x); yy = (y); cc = (c); \
		if (use_plotpixel) \
			plot_pixel(bitmap, xx, yy, cc); \
		else \
			bitmap->line[yy][xx] = cc; \
	}

#define MARK_DIRTY(x1, y1, x2, y2) \
	{ if (!use_plotpixel) \
		osd_mark_dirty((x1), (y1), (x2), (y2), 0); }


#define MARK_DIRTY_PIX(x, y, w, h) \
	{ if (!use_plotpixel) { \
		int xx, yy; \
		xx = (x); yy = (y); \
		osd_mark_dirty(xx, yy, xx+(w)-1, yy+(h)-1, 0); \
	} }

/*
 * Note that 'sizex' is in bytes, and 'sizey' is in pixels
 */
void blitgraphics2(struct osd_bitmap *bitmap, UINT8 *vrambase, int vrampos,
	int vramsize, UINT8 *db, const int *metapalette, int sizex, int sizey,
	int basex, int basey, int scalex, int scaley, int additionalrowbytes)
{
	int x, y;
	int px, py;
	int fg, bg;
	int p, b;
	int use_plotpixel;
	UINT8 *vidram;

	if (metapalette) {
		bg = Machine->pens[metapalette[0]];
		fg = Machine->pens[metapalette[1]];
	}
	else {
		bg = Machine->pens[0];
		fg = Machine->pens[1];
	}

	vidram = vrambase + vrampos;
	use_plotpixel = Machine->orientation || (Machine->color_depth != 8);

	if (!db)
		MARK_DIRTY(basex, basey, basex + scalex*sizex*8, basey + scaley*sizey);

	for (y = 0; y < sizey; y++) {
		for (x = 0; x < sizex; x++) {
			if (!db || *db) {
				for (b = 0; b < 8; b++) {
					p = ((*vidram) & (1<<(7-b))) ? fg : bg;

					for (py = 0; py < scaley; py++) {
						for (px = 0; px < scalex; px++) {
							PLOT_PIXEL((x * 8 + b) * scalex + px + basex, y * scaley + py + basey, p);
						}
					}
				}
				if (db) {
					MARK_DIRTY_PIX(x*8*scalex+basex, y*scaley+basey, scalex*8, scaley);
					*(db++) = 0;
				}
			}
			else {
				db++;
			}
			vidram++;
		}
		vidram += additionalrowbytes;
		if (db)
			db += additionalrowbytes;

		/* Check to see if the video RAM has wrapped around */
		if (vidram > vrambase + vramsize)
			vidram -= vramsize;
	}
}

void blitgraphics4(struct osd_bitmap *bitmap, UINT8 *vrambase, int vrampos,
	int vramsize, UINT8 *db, const int *metapalette, int sizex, int sizey,
	int basex, int basey, int scalex, int scaley, int additionalrowbytes)
{
	int x, y;
	int px, py;
	int c[4];
	int p, b;
	int use_plotpixel;
	UINT8 *vidram;

	if (metapalette) {
		c[0] = Machine->pens[metapalette[0]];
		c[1] = Machine->pens[metapalette[1]];
		c[2] = Machine->pens[metapalette[2]];
		c[3] = Machine->pens[metapalette[3]];
	}
	else {
		c[0] = Machine->pens[0];
		c[1] = Machine->pens[1];
		c[2] = Machine->pens[2];
		c[3] = Machine->pens[3];
	}

	vidram = vrambase + vrampos;
	use_plotpixel = Machine->orientation || (Machine->color_depth != 8);

	if (!db)
		MARK_DIRTY(basex, basey, basex + scalex*sizex*4, basey + scaley*sizey);

	for (y = 0; y < sizey; y++) {
		for (x = 0; x < sizex; x++) {
			if (!db || *db) {
				for (b = 0; b < 4; b++) {
					p = c[(((*vidram) >> (6-(2*b)))) & 3];

					for (py = 0; py < scaley; py++) {
						for (px = 0; px < scalex; px++) {
							PLOT_PIXEL((x * 4 + b) * scalex + px + basex, y * scaley + py + basey, p);
						}
					}
				}
				if (db) {
					MARK_DIRTY_PIX(x*4*scalex+basex, y*scaley+basey, scalex*4, scaley);
					*(db++) = 0;
				}
			}
			else {
				db++;
			}
			vidram++;
		}
		vidram += additionalrowbytes;
		if (db)
			db += additionalrowbytes;

		/* Check to see if the video RAM has wrapped around */
		if (vidram > vrambase + vramsize)
			vidram -= vramsize;
	}
}

static void blitgraphics4artifact(struct osd_bitmap *bitmap, UINT8 *vrambase,
	int vrampos, int vramsize, UINT8 *db, const int *metapalette, int sizex,
	int sizey, int basex, int basey, int scalex, int scaley)
{
	/* Arifacting isn't truely the same resolution as PMODE 3
	 * it has a bias to the higher resolution.  We need to
	 * emulate this because some things are unreadable if we
	 * say that its just like PMODE 3 with different colors
	 */

	/* This is the blur correction table. For any given pixel,
	 * you can take color of that pixel, the color to the left
	 * and right, and compute what the two resulting pixels
	 * should look like.  The table is accessed like this:
	 *
	 *		blurcorrection[pix(x,y)][pix(x-1,y)][pix(x+1,y)]
	 *
	 * This gives you a pair of pixels to display pix(x,y) as
	 *
	 * Special thanks to Derek Snider for coming up with the
	 * basis for the values in this table
	 */
	static int blurcorrection[2][4][4][2] = {
		/* pixel color 1 */
		{
			{{0, 1}, {0, 1}, {0, 3}, {0, 3}},
			{{1, 1}, {1, 1}, {1, 3}, {1, 3}},
			{{0, 1}, {0, 1}, {0, 3}, {0, 3}},
			{{1, 1}, {1, 1}, {1, 3}, {1, 3}}
		},

		/* pixel color 2 */
		{
			{{2, 0}, {2, 0}, {2, 2}, {2, 2}},
			{{3, 0}, {3, 0}, {3, 2}, {3, 2}},
			{{2, 0}, {2, 0}, {2, 2}, {2, 2}},
			{{3, 0}, {3, 0}, {3, 2}, {3, 2}}
		}
	};

	int x, y;
	int px, py;
	int c[4];
	int p, b;
	int lastp, nextp;
	int c1, c2;
	int nextdirty;
	int use_plotpixel;
	UINT8 *vidram;

	c[0] = Machine->pens[metapalette[0]];
	c[1] = Machine->pens[metapalette[1]];
	c[2] = Machine->pens[metapalette[2]];
	c[3] = Machine->pens[metapalette[3]];

	vidram = vrambase + vrampos;
	use_plotpixel = Machine->orientation || (Machine->color_depth != 8);

	for (y = 0; y < sizey; y++) {
		nextdirty = 0;
		for (x = 0; x < sizex; x++) {
			if (db[0] || ((x < (sizex-1)) && db[1]) || nextdirty) {
				nextp = (vidram[0] >> 6) & 3;
				lastp = (x == 0) ? nextp : (vidram[-1] & 3);

				for (b = 0; b < 4; b++) {
					p = nextp;

					if (b < 3)
						nextp = (((*vidram) >> (6-(2*(b+1))))) & 3;
					else if (x < (sizex-1))
						nextp = (vidram[1] >> 6) & 3;

					switch(p) {
					case 0:
						c1 = c2 = c[0];
						break;
					case 3:
						c1 = c2 = c[3];
						break;
					default:
						c1 = c[blurcorrection[p-1][lastp][nextp][0]];
						c2 = c[blurcorrection[p-1][lastp][nextp][1]];
						break;
					}

					for (py = 0; py < scaley; py++) {
						for (px = 0; px < scalex; px++) {
							PLOT_PIXEL((x * 8 + b * 2 + 0) * scalex + px + basex, y * scaley + py + basey, c1);
							PLOT_PIXEL((x * 8 + b * 2 + 1) * scalex + px + basex, y * scaley + py + basey, c2);
						}
					}

					lastp = p;
				}
				MARK_DIRTY_PIX(x*8*scalex+basex, y*scaley+basey, scalex*8, scaley);
				
				if (*db) {
					nextdirty = 1;
					*db = 0;
				}
			}
			db++;
			vidram++;
		}

		/* Check to see if the video RAM has wrapped around */
		if (vidram > vrambase + vramsize)
			vidram -= vramsize;
	}
}

void blitgraphics16(struct osd_bitmap *bitmap, UINT8 *vrambase,
	int vrampos, int vramsize, UINT8 *db, int sizex, int sizey, int basex,
	int basey, int scalex, int scaley, int additionalrowbytes)
{
	int x, y;
	int px, py;
	int p1, p2;
	UINT8 *vidram;
	int use_plotpixel;

	vidram = vrambase + vrampos;
	use_plotpixel = Machine->orientation || (Machine->color_depth != 8);

	if (!db)
		MARK_DIRTY(basex, basey, basex + scalex*sizex*2, basey + scaley*sizey);

	for (y = 0; y < sizey; y++) {
		for (x = 0; x < sizex; x++) {
			if (!db || *db) {
				p1 = Machine->pens[(*vidram >> 4) & 0x0f];
				p2 = Machine->pens[(*vidram >> 0) & 0x0f];

				for (py = 0; py < scaley; py++) {
					for (px = 0; px < scalex; px++) {
						PLOT_PIXEL((x * 2 + 0) * scalex + px + basex, y * scaley + py + basey, p1);
						PLOT_PIXEL((x * 2 + 1) * scalex + px + basex, y * scaley + py + basey, p2);
					}
				}
				if (db) {
					MARK_DIRTY_PIX(x*2*scalex+basex, y*scaley+basey, scalex*2, scaley);
					*(db++) = 0;
				}
			}
			else {
				db++;
			}
			vidram++;
		}
		vidram += additionalrowbytes;
		if (db)
			db += additionalrowbytes;

		/* Check to see if the video RAM has wrapped around */
		if (vidram > vrambase + vramsize)
			vidram -= vramsize;
	}
}

/* This is a refresh function used by the Dragon/CoCo as well as the CoCo 3 when in lo-res
 * mode.  Internally, it treats the colors like a CoCo 3 and uses the 'metapalette' to
 * translate those colors to the proper palette.
 *
 * video_vmode
 *     bit 4	1=graphics, 0=text
 *     bit 3    resolution high bit
 *     bit 2    resolution low bit
 *     bit 1    1=b/w graphics, 0=color graphics
 *     bit 0	color set
 */
void internal_m6847_vh_screenrefresh(struct osd_bitmap *bitmap, const int *metapalette,
	UINT8 *vrambase, int vrampos, int vramsize,
	int has_lowercase, int basex, int basey, int wf, artifactproc artifact)
{
	int x, y, fg, bg, x2, y2;
	int rowbytes;
	int artifacting;
	UINT8 *db;
	UINT8 b;
	int artifactpalette[4];
	UINT8 *cptr;

	db = dirtybuffer;
	if (m6847_full_refresh) {
		memset(dirtybuffer, 1, videoram_size);
		m6847_full_refresh = 0;
	}

	if (video_gmode & 0x10)
	{
		if ((video_gmode & 0x02) && !(artifact && ((video_gmode & 0x1e) == M6847_MODE_G4R)))
		{
			/* Resolution modes */

			rowbytes = ((video_gmode & 0x1e) == M6847_MODE_G4R) ? 32 : 16;
			blitgraphics2(bitmap, vrambase, vrampos, vramsize, db, &metapalette[video_gmode & 0x1 ? 10 : 8],
				rowbytes, 192 / video_rowheight, basex, basey, (32 / rowbytes) * wf, video_rowheight, 0);
		}
		else
		{
			/* Color modes */
			rowbytes = ((video_gmode & 0x1e) != M6847_MODE_G1C) ? 32 : 16;

			/* Are we doing PMODE 4 artifact colors? */
			artifacting = ((video_gmode & 0x0c) == 0x0c) && (video_gmode & 0x02);
			if (artifacting) {
				/* I am here because we are doing PMODE 4 artifact colors */
				artifactpalette[0] = metapalette[video_gmode & 0x1 ? 10: 8];
				artifactpalette[3] = metapalette[video_gmode & 0x1 ? 11: 9];
				artifact(artifactpalette);

				blitgraphics4artifact(bitmap, vrambase, vrampos, vramsize, db, artifactpalette,
					rowbytes, 192 / video_rowheight, basex, basey, 32 / rowbytes * wf, video_rowheight);
			}
			else {
				/* If not, calculate offset normally */
				blitgraphics4(bitmap, vrambase, vrampos, vramsize, db, &metapalette[video_gmode & 0x1 ? 4: 0],
					rowbytes, 192 / video_rowheight, basex, basey, 64 / rowbytes * wf, video_rowheight, 0);
			}
		}
	}
	else
	{
		UINT8 *vidram;

		vidram = vrambase + vrampos;

		for (y = 0; y < (192 / video_rowheight); y++) {
			for (x = 0; x < 32; x++) {
				if (*db) {
					b = *vidram;

					if (!has_lowercase && (video_gmode & 0x02)) {
						/* Semigraphics 6 */
						bg = 8;
						fg = ((b >> 6) & 0x3) + (video_gmode & 0x1 ? 4: 0);
						b = 64 + (b & 0x3f);
					}
					else if (*vidram & 0x80) {
						/* Semigraphics 4 */
						bg = 8;
						fg = (b >> 4) & 0x7;
						b = 128 + (b & 0x0f);
					}
					else {
						/* Text */
						bg = (video_gmode & 0x01) ? 14 : 12;

						/* On the M6847T1 and the CoCo 3 GIME chip, bit 2 of video_gmode
						 * reversed the colors
						 *
						 * TODO: Find out what the normal M6847 did with bit 2
						 */
						if (video_gmode & 0x04)
							bg ^= 1;

						/* Is this character lowercase or inverse? */
						if ((video_gmode & 0x02) && (b < 0x20)) {
							/* This character is lowercase */
							b += 144;
						}
						else if (b < 0x40) {
							/* This character is inverse */
							bg ^= 1;
						}

						fg = bg;
						bg ^= 1;
						b &= 0xbf;
					}

					cptr = &fontdata8x12[((int) b) * 12];
					bg = Machine->pens[metapalette[bg]];
					fg = Machine->pens[metapalette[fg]];

					for (y2 = (y * video_rowheight); y2 < ((y+1) * video_rowheight); y2++) {
						b = cptr[y2 % 12];
						for (x2 = 0; x2 < (8*wf); x2++) {
							plot_pixel(bitmap, x*8*wf+x2+basex, y2+basey, b & (1<<(7-(x2/wf))) ? fg : bg);
						}
					}

					*db = 0;
				}
				vidram++;
				db++;
			}
		}

		/* Check to see if the video RAM has wrapped around */
		if (vidram > vrambase + vramsize)
			vidram -= vramsize;
	}
}


static void m6847_artifact_red(int *artifactcolors)
{
	switch(artifactcolors[3]) {
	case 1:
		artifactcolors[2] = 10;
		artifactcolors[1] = 9;
		break;
	case 5:
		artifactcolors[2] = 12;
		artifactcolors[1] = 11;
		break;
	}
}

static void m6847_artifact_blue(int *artifactcolors)
{
	switch(artifactcolors[3]) {
	case 1:
		artifactcolors[1] = 10;
		artifactcolors[2] = 9;
		break;
	case 5:
		artifactcolors[1] = 12;
		artifactcolors[2] = 11;
		break;
	}
}

void internal_m6847_drawborder(struct osd_bitmap *bitmap, int screenx, int screeny, int pen)
{
	int left, right, top, bottom;
	int borderpen;
	struct rectangle r;

	borderpen = Machine->pens[pen];

	left = (bitmap->width - screenx) / 2;
	right = left + screenx;
	top = (bitmap->height - screeny) / 2;
	bottom = top + screeny;

	r.min_x = 0;
	r.min_y = 0;
	r.max_x = bitmap->width - 1;
	r.max_y = top-1;
	fillbitmap(bitmap, borderpen, &r);
	r.min_y = bottom;
	r.max_y = bitmap->height - 1;
	fillbitmap(bitmap, borderpen, &r);
	r.min_y = top;
	r.max_x = left-1;
	r.max_y = bottom-1;
	fillbitmap(bitmap, borderpen, &r);
	r.min_x = right;
	r.max_x = bitmap->width - 1;
	fillbitmap(bitmap, borderpen, &r);
}

static void m6847_drawborder(struct osd_bitmap *bitmap, int screenx, int screeny)
{
	int pen = 0;

	switch(m6847_get_bordercolor()) {
	case M6847_BORDERCOLOR_BLACK:
		pen = 0;
		break;
	case M6847_BORDERCOLOR_GREEN:
		pen = 1;
		break;
	case M6847_BORDERCOLOR_WHITE:
		pen = 5;
		break;
	case M6847_BORDERCOLOR_ORANGE:
		pen = 8;
		break;
	}
	internal_m6847_drawborder(bitmap, screenx, screeny, pen);
}

void m6847_vh_update(struct osd_bitmap *bitmap,int full_refresh)
{
	static int m6847_metapalette[] = {
		1, 2, 3, 4, 5, 6, 7, 8, 0, 1, 0, 5, 13, 14, 15, 16
	};
	static artifactproc artifacts[] = {
		NULL,
		m6847_artifact_red,
		m6847_artifact_blue
	};
	int artifact_value;

	if (full_refresh)
		m6847_full_refresh = 1;

	if (vblankproc)
		vblankproc();

	artifact_value = (artifact_dipswitch == -1) ? 0 : readinputport(artifact_dipswitch);

	if (full_refresh || m6847_full_refresh)
		m6847_drawborder(bitmap, 256, 192);

	internal_m6847_vh_screenrefresh(bitmap, m6847_metapalette, videoram,
		video_offset, vram_mask + 1, FALSE,
		(bitmap->width - 256) / 2, (bitmap->height - 192) / 2,
		1, artifacts[artifact_value & 3]);
}

/* --------------------------------------------------
 * Petty accessors
 * -------------------------------------------------- */

void m6847_set_gmode(int mode)
{
	mode &= 0x1f;

	if (mode != video_gmode) {
		video_gmode = mode;
		m6847_full_refresh = 1;
	}
}

void m6847_set_vmode(int mode)
{
	mode &= 0x1f;

	if (mode != video_vmode) {
		video_vmode = mode;
		calc_videoram_size();
		m6847_full_refresh = 1;
	}
}

int m6847_get_gmode(void)
{
	return video_gmode;
}

int m6847_get_vmode(void)
{
	return video_vmode;
}

void m6847_set_video_offset(int offset)
{
	offset &= vram_mask;
	if (offset != video_offset) {
		video_offset = offset;
		m6847_full_refresh = 1;
	}
}

int m6847_get_video_offset(void)
{
	return video_offset;
}

void m6847_touch_vram(int offset)
{
	if (!m6847_full_refresh) {
		offset -= video_offset;
		offset &= vram_mask;

		if (offset < videoram_size)
			dirtybuffer[offset] = 1;
	}
}

int m6847_get_bordercolor(void)
{
	/* TODO: Verify this table.  I am pretty sure that it is true
	 * for the CoCo 3 and M6847T1, but I'm not sure if it is true
	 * for plain M6847
	 */
	static int bordercolortable[] = {
		/* Text modes */
		M6847_BORDERCOLOR_BLACK,	M6847_BORDERCOLOR_BLACK,
		M6847_BORDERCOLOR_BLACK,	M6847_BORDERCOLOR_BLACK,
		M6847_BORDERCOLOR_BLACK,	M6847_BORDERCOLOR_BLACK,
		M6847_BORDERCOLOR_BLACK,	M6847_BORDERCOLOR_BLACK,
		M6847_BORDERCOLOR_BLACK,	M6847_BORDERCOLOR_BLACK,
		M6847_BORDERCOLOR_GREEN,	M6847_BORDERCOLOR_ORANGE,
		M6847_BORDERCOLOR_BLACK,	M6847_BORDERCOLOR_BLACK,
		M6847_BORDERCOLOR_BLACK,	M6847_BORDERCOLOR_BLACK,

		M6847_BORDERCOLOR_GREEN,	M6847_BORDERCOLOR_WHITE,
		M6847_BORDERCOLOR_GREEN,	M6847_BORDERCOLOR_WHITE,
		M6847_BORDERCOLOR_GREEN,	M6847_BORDERCOLOR_WHITE,
		M6847_BORDERCOLOR_GREEN,	M6847_BORDERCOLOR_WHITE,
		M6847_BORDERCOLOR_GREEN,	M6847_BORDERCOLOR_WHITE,
		M6847_BORDERCOLOR_GREEN,	M6847_BORDERCOLOR_WHITE,
		M6847_BORDERCOLOR_GREEN,	M6847_BORDERCOLOR_WHITE,
		M6847_BORDERCOLOR_GREEN,	M6847_BORDERCOLOR_WHITE		
	};
	return bordercolortable[video_gmode];
}

void m6847_get_bordercolor_rgb(int *red, int *green, int *blue)
{
	switch(m6847_get_bordercolor()) {
	case M6847_BORDERCOLOR_BLACK:
		*red = 0;
		*green = 0;
		*blue = 0;
		break;

	case M6847_BORDERCOLOR_GREEN:
		*red = 0;
		*green = 255;
		*blue = 0;
		break;

	case M6847_BORDERCOLOR_WHITE:
		*red = 255;
		*green = 255;
		*blue = 255;
		break;

	case M6847_BORDERCOLOR_ORANGE:
		*red = 255;
		*green = 128;
		*blue = 0;
		break;
	}
}

