/***************************************************************************

  machine.c

  Functions to emulate general aspects of the machine (RAM, ROM, interrupts,
  I/O ports)

  References:
		There are two main references for the info for this driver
		- Tandy Color Computer Unravelled Series
					(http://www.giftmarket.org/unravelled/unravelled.shtml)
		- Assembly Language Programming For the CoCo 3 by Laurence A. Tepolt
		- Kevin K. Darlings GIME reference
					(http://www.cris.com/~Alxevans/gime.txt)
		- Sock Masters's GIME register reference
					(http://www.axess.com/twilight/sock/gime.html)
		- Robert Gault's FAQ
					(http://home.att.net/~robert.gault/Coco/FAQ/FAQ_main.htm)
		- Discussions with L. Curtis Boyle (LCB) and John Kowalski (JK)

  TODO:
		- Implement serial/printer
		- Implement unimplemented SAM registers
		- Implement unimplemented interrupts (serial)
		- Choose and implement more appropriate ratios for the speed up poke
		- Support .PAK files generated by PC-Dragon
		- Find out what happens if an interrupt is triggered for both the IRQ
		  and FIRQ (which one actually runs?  or does both?)
		- Find out when most interrupts "turn off" (Take note of all of the
		  places where I raise the interrupt and then turn it off)
		- Make the wd179x DRQ and IRQ work the way they did in reality.  The
		  problem is that the wd179x code doesn't implement any timing, and as
		  such behaves differently then the real thing.
		- Handle resets correctly

  In the CoCo, all timings should be exactly relative to each other.  This
  table shows how all clocks are relative to each other (info: JK):
		- Main CPU Clock				0.89 MHz
		- Horizontal Sync Interrupt		15.7 kHz/63.5us	(57 clock cycles)
		- Vertical Sync Interrupt		60 Hz			(14934 clock cycles)
		- Composite Video Color Carrier	3.58 MHz/279ns	(1/4 clock cycles)

  It is also noting that the CoCo 3 had two sets of VSync interrupts.  To quote
  John Kowalski:

	One other thing to mention is that the old vertical interrupt and the new
	vertical interrupt are not the same..  The old one is triggered by the
	video's vertical sync pulse, but the new one is triggered on the next scan
	line *after* the last scan line of the active video display.  That is : new
	vertical interrupt triggers somewheres around scan line 230 of the 262 line
	screen (if a 200 line graphics mode is used, a bit earlier if a 192 line
	mode is used and a bit later if a 225 line mode is used).  The old vsync
	interrupt triggers on scanline zero.

	230 is just an estimate [(262-200)/2+200].  I don't think the active part
	of the screen is exactly centered within the 262 line total.  I can
	research that for you if you want an exact number for scanlines before the
	screen starts and the scanline that the v-interrupt triggers..etc.
***************************************************************************/

#include "driver.h"
#include "cpu/m6809/m6809.h"
#include "machine/6821pia.h"
#include "mess/machine/wd179x.h"
#include "mess/vidhrdw/m6847.h"

static UINT8 *coco_rom;
static int coco3_enable_64k;
static int coco3_mmu[16];
static int coco3_gimereg[8];
static int pia0_irq_a, pia0_irq_b;
static int pia1_firq_a, pia1_firq_b;
static int gime_firq, gime_irq;
static int cart_inserted;
static UINT8 pia0_pb, sound_mux, tape_motor;
static UINT8 joystick_axis, joystick;
static int d_dac;
static int d_sam_memory_size;

/* from vidhrdw/dragon.c */
extern WRITE_HANDLER ( coco3_ram_b1_w );
extern WRITE_HANDLER ( coco3_ram_b2_w );
extern WRITE_HANDLER ( coco3_ram_b3_w );
extern WRITE_HANDLER ( coco3_ram_b4_w );
extern WRITE_HANDLER ( coco3_ram_b5_w );
extern WRITE_HANDLER ( coco3_ram_b6_w );
extern WRITE_HANDLER ( coco3_ram_b7_w );
extern WRITE_HANDLER ( coco3_ram_b8_w );
extern WRITE_HANDLER ( coco3_ram_b9_w );
extern void coco3_vh_sethires(int hires);

static WRITE_HANDLER ( d_pia1_pb_w );
static WRITE_HANDLER ( coco3_pia1_pb_w );
static WRITE_HANDLER ( d_pia1_pa_w );
static READ_HANDLER (  d_pia1_cb1_r );
static READ_HANDLER (  d_pia0_ca1_r );
static READ_HANDLER (  d_pia0_pa_r );
static READ_HANDLER (  d_pia1_pa_r );
static WRITE_HANDLER ( d_pia0_pb_w );
static WRITE_HANDLER ( d_pia1_cb2_w);
static WRITE_HANDLER ( d_pia0_cb2_w);
static WRITE_HANDLER ( d_pia1_ca2_w);
static WRITE_HANDLER ( d_pia0_ca2_w);
static void d_pia0_irq_a(int state);
static void d_pia0_irq_b(int state);
static void d_pia1_firq_a(int state);
static void d_pia1_firq_b(int state);
static void coco3_pia0_irq_a(int state);
static void coco3_pia0_irq_b(int state);
static void coco3_pia1_firq_a(int state);
static void coco3_pia1_firq_b(int state);

#define LOG_PAK			0
#define LOG_WAVE		0
#define LOG_INT_MASKING	0
#define LOG_INT_TMR		0
#define LOG_INT_COCO3	0
#define LOG_GIME		0
#define LOG_MMU			0

#define COCO_CPU_SPEED	(TIME_IN_HZ(894886))

#define COCO_TIMER_HSYNC		(COCO_CPU_SPEED * 57.0)
#define COCO_TIMER_VSYNC		(COCO_CPU_SPEED * 14934.0)
#define COCO_TIMER_CMPCARRIER	(COCO_CPU_SPEED * 0.25)

static void coco3_timer_recalculate(int newcounterval, int allowreset);

static struct pia6821_interface dragon_pia_intf[] =
{
	/* PIA 0 */
	{
		/*inputs : A/B,CA/B1,CA/B2 */ d_pia0_pa_r, 0, d_pia0_ca1_r, 0, 0, 0,
		/*outputs: A/B,CA/B2	   */ 0, d_pia0_pb_w, d_pia0_ca2_w, d_pia0_cb2_w,
		/*irqs	 : A/B			   */ d_pia0_irq_a, d_pia0_irq_b
	},

	/* PIA 1 */
	{
		/*inputs : A/B,CA/B1,CA/B2 */ d_pia1_pa_r, 0, 0, d_pia1_cb1_r, 0, 0,
		/*outputs: A/B,CA/B2	   */ d_pia1_pa_w, d_pia1_pb_w, d_pia1_ca2_w, d_pia1_cb2_w,
		/*irqs	 : A/B			   */ d_pia1_firq_a, d_pia1_firq_b
	}
};

static struct pia6821_interface coco3_pia_intf[] =
{
	/* PIA 0 */
	{
		/*inputs : A/B,CA/B1,CA/B2 */ d_pia0_pa_r, 0, d_pia0_ca1_r, 0, 0, 0,
		/*outputs: A/B,CA/B2	   */ 0, d_pia0_pb_w, d_pia0_ca2_w, d_pia0_cb2_w,
		/*irqs	 : A/B			   */ coco3_pia0_irq_a, coco3_pia0_irq_b
	},

	/* PIA 1 */
	{
		/*inputs : A/B,CA/B1,CA/B2 */ d_pia1_pa_r, 0, 0, d_pia1_cb1_r, 0, 0,
		/*outputs: A/B,CA/B2	   */ d_pia1_pa_w, coco3_pia1_pb_w, d_pia1_ca2_w, d_pia1_cb2_w,
		/*irqs	 : A/B			   */ coco3_pia1_firq_a, coco3_pia1_firq_b
	}
};

/***************************************************************************
  PAK files

  PAK files were originally for storing Program PAKs, but the file format has
  evolved into a snapshot file format, with a file format so convoluted and
  changing to make it worthy of Microsoft.
***************************************************************************/
static int load_pak_into_region(void *fp, int *pakbase, int *paklen, UINT8 *mem, int segaddr, int seglen)
{
	if (*paklen) {
		if (*pakbase < segaddr) {
			/* We have to skip part of the PAK file */
			int skiplen;

			skiplen = segaddr - *pakbase;
			if (osd_fseek(fp, skiplen, SEEK_CUR)) {
#if LOG_PAK
				logerror("Could not fully read PAK.\n");
#endif
				return 1;
			}

			*pakbase += skiplen;
			*paklen -= skiplen;
		}

		if (*pakbase < segaddr + seglen) {
			mem += *pakbase - segaddr;
			seglen -= *pakbase - segaddr;

			if (seglen > *paklen)
				seglen = *paklen;

			if (osd_fread(fp, mem, seglen) < seglen) {
#if LOG_PAK
				logerror("Could not fully read PAK.\n");
#endif
				return 1;
			}

			*pakbase += seglen;
			*paklen -= seglen;
		}
	}
	return 0;
}

/* PAK file loader */

/*	PAK files have the following format:
 *
 *	length		(two bytes, little endian)
 *	base address (two bytes, little endian, typically 0xc000)
 *	...data... (size is length)
 *	optional trailer
 *
 *	The format for PAK files just plain bites - the extra info is snapshot info
 *	and it is loaded with internal state specific to Jeff's emulator. What ever
 *	happened to desiging clean file formats that don't need to be changed with
 *	every version of ones program?
 *
 *	For alignment purposes, some 16 bit values are divided into two UINT8s
 *
 *	The PAK file format's trailer defines the state of the machine other then
 *	the core memory.  This trailer comes in many variations.  I have labeled
 *	them in the following way (trailer sizes are in parentheses):
 *
 *		V12		- The version in JeffV's emulator version 1.2
 *		V14		- The version in JeffV's emulator version 1.4	(457)
 *		VOTHER	- From some "other" version of JeffV's emulator
 *		VLITE	- From yet another version of JeffV's emulator	(35)
 *
 *	The following table shows what segments appear in each trailer variant:
 *
 *	Trailer Segment		V12		V14		VOTHER	VLITE
 *	---------------		---		---		------	-----
 *	pak_trailer1		1		1		1		1
 *	pak_trailer2		2
 *	pak_trailer3		3		2		2		2
 *	pak_trailer4		4		3		3
 *	pak_trailer5		5
 *	pak_trailer6		6		5		4
 *	pak_trailer7		7				5
 *	pak_trailer8				6
 *	pak_trailer9		8		7		6
 *	pak_trailer10		9		4		7
 *	pak_trailer11		10		8		8
 *	pak_trailer12				9
 *	pak_trailer13		11				9
 */

#ifdef LSB_FIRST
#define ENDIANIZE(x) (x)
#else
#define ENDIANIZE(x) ((((x) >> 8) + ((x) << 8)) & 0xffff)
#endif

/* All versions */
typedef struct {
	UINT16 length;
	UINT16 start;
} pak_header;

/* All versions */
typedef struct {
	char name[33];
} pak_trailer1;

/* Only version 1.2 */
typedef struct {
	UINT8 debug_dumpflag;
	UINT8 debug_disassembleflag;
	UINT16 debug_disassembleaddr;
} pak_trailer2;

/* All versions */
typedef struct {
	UINT16 reg_pc;
} pak_trailer3;

/* All versions except lite */
typedef struct {
	UINT16 reg_x;
	UINT16 reg_y;
	UINT16 reg_u;
	UINT16 reg_s;
	UINT8 dummy_zero1;
	UINT8 reg_dp;
	UINT8 reg_b;
	UINT8 reg_a;
} pak_trailer4;

/* Only version 1.2 */
typedef struct {
	UINT16 debug_unknown;
} pak_trailer5;

/* All versions except lite */
typedef struct {
	UINT8 flags_8086_lsb; /* ?!? */
	UINT8 flags_8086_msb; /* ?!? */
	UINT8 reg_cc;
} pak_trailer6;

/* All versions except 1.4 and lite*/
typedef struct {
	UINT16 lowmem_readseg;
	UINT16 lowmem_writeseg;
	UINT16 himem_readseg;
	UINT16 himem_writeseg;
} pak_trailer7;

/* Only version 1.4 */
typedef struct {
	UINT8 page_status;
	UINT8 rom_status;
	UINT8 io_ff02;
	UINT8 io_ff03;
	UINT8 io_ff22;
} pak_trailer8;

/* All versions except lite */
typedef struct {
	UINT16 video_base;
	UINT16 video_end;
} pak_trailer9;

/* All versions except lite; come before pak_trailer6 in 1.4 */
typedef struct {
	UINT8 dummy_zero2[6];
} pak_trailer10;

/* Used by PC-Dragon; UINT16's are little endian */
typedef struct {
	/* Info */
	UINT8 data1[2];
	UINT8 magic1[2];
	UINT8 checksum;
	UINT8 pak_version;
	UINT8 ext_version;
	UINT8 emu_version;
	UINT8 state_only;
	UINT8 architecture;
	UINT8 rom_start;

	/* Machine state */
	UINT8 irq_cycles_lsb;
	UINT8 irq_cycles_msb;
	UINT8 screen_size;
	UINT8 pia[9];
	UINT8 last_shift;
	UINT8 filemem[8];

	/* Debugger */
	UINT16 last_text_base;
	UINT16 breakpoints[6];
	UINT16 break_operation;
	UINT16 dasm_pcr;
	UINT16 fill_start;
	UINT16 fill_end;
	UINT8 fill_value;
	UINT8 left_window;
	UINT16 temp_break;
	UINT16 break_value;
	UINT8 break_control;
	UINT8 break_type;
	UINT16 internal;
	UINT16 dumpaddr;
} pcd_info1;

/* Used by PC-Dragon; UINT16's are little endian */
typedef struct {
	UINT8 pakdata[65];
	UINT8 dummy1;
	UINT8 background;
	UINT8 foreground;
	UINT8 vmode[2][2][6];
	UINT8 border[2][2];
	UINT8 dummy2;
	UINT16 irq_rate;
	UINT8 servicemem[16];
	UINT8 speed;
	UINT8 lpt_and_swapping;
	UINT8 hardjoy_resol;
} pcd_info2;

/* All versions except lite */
typedef struct {
	UINT8 writeprotect;
	char disk_directory[66];
	char disk_name[4][32];
	char pak_directory[66];
	UINT8 crlf;
	UINT8 keyboard_mode;
	UINT8 speed_lsb;
	UINT8 speed_msb;
	UINT8 left_joystick;
	UINT8 right_joystick;
	UINT8 lowercase_lsb;
	UINT8 lowercase_msb;
	UINT8 sound;
	UINT8 artifact;
	UINT8 dragon_rom;
} pak_trailer11;

/* Only version 1.4 */
typedef struct {
	UINT16 joystick_limits[8];
	UINT16 clock;
	UINT8 drive_mode;
	UINT8 volume;
	UINT8 cassette_mode;
	char cassette_directory[66];
	char cassette_name[33];
} pak_trailer12;

/* All versions except 1.4 and lite */
typedef struct {
	UINT8 dummy_zero3[4];
	UINT16 video_base2;
	UINT16 video_end2;
	UINT16 dummy_zero4;
	UINT8 io_ff22;
	UINT8 io_ff02;
	UINT8 io_ff03;
} pak_trailer13;

#define PAK_V12_SIZE	(sizeof(pak_trailer1) + sizeof(pak_trailer2) + \
	sizeof(pak_trailer3) + sizeof(pak_trailer4) + sizeof(pak_trailer5) + \
	sizeof(pak_trailer6) + sizeof(pak_trailer7) + sizeof(pak_trailer9) + \
	sizeof(pak_trailer10) + sizeof(pak_trailer11) + sizeof(pak_trailer13))

#define PAK_V14_SIZE	(sizeof(pak_trailer1) + sizeof(pak_trailer3) + \
	sizeof(pak_trailer4) + sizeof(pak_trailer10) + sizeof(pak_trailer6) + \
	sizeof(pak_trailer8) + sizeof(pak_trailer9) + sizeof(pak_trailer11) + \
	sizeof(pak_trailer12))

#define PAK_VOTHER_SIZE	(sizeof(pak_trailer1) + sizeof(pak_trailer3) + \
	sizeof(pak_trailer4) + sizeof(pak_trailer6) + sizeof(pak_trailer7) + \
	sizeof(pak_trailer9) + sizeof(pak_trailer10) + sizeof(pak_trailer11) + \
	sizeof(pak_trailer13))

#define PAK_VLITE_SIZE (sizeof(pak_trailer1) + sizeof(pak_trailer3))

typedef struct {
	UINT16 video_base;
	UINT16 video_end;
	UINT16 reg_pc;
	UINT16 reg_x;
	UINT16 reg_y;
	UINT16 reg_u;
	UINT16 reg_s;
	UINT8 reg_dp;
	UINT8 reg_b;
	UINT8 reg_a;
	UINT8 reg_cc;
	UINT8 io_ff02;
	UINT8 io_ff03;
	UINT8 io_ff22;
	UINT8 enable_hiram;
} pak_trailer;

/* This function takes a set of bytes, interprets a PAK trailer and
 * extracts the interesting information
 */
static int pak_decode_trailer(UINT8 *rawtrailer, int rawtrailerlen, pak_trailer *trailer)
{
	pak_trailer3 p3;
	pak_trailer4 p4;
	pak_trailer6 p6;
	pak_trailer9 p9;

	union {
		pak_trailer8 p8;
		struct {
			pak_trailer7 p7;
			pak_trailer13 p13;
		} s;
	} u1;

	switch(rawtrailerlen) {
	case PAK_V12_SIZE:
	case PAK_V14_SIZE:
	case PAK_VOTHER_SIZE:
	case PAK_VLITE_SIZE:
		break;
	default:
		return 1;
	}

	memset(&p3, 0, sizeof(p3));
	memset(&p4, 0, sizeof(p4));
	memset(&p6, 0, sizeof(p6));
	memset(&p9, 0, sizeof(p9));
	memset(&u1, 0, sizeof(u1));

	rawtrailer += sizeof(pak_trailer1);

	if (rawtrailerlen == PAK_V12_SIZE) {
		rawtrailer += sizeof(pak_trailer2);
	}

	memcpy(&p3, rawtrailer, sizeof(pak_trailer3));
	rawtrailer += sizeof(pak_trailer3);

	if (rawtrailerlen != PAK_VLITE_SIZE) {
		memcpy(&p4, rawtrailer, sizeof(pak_trailer4));
		rawtrailer += sizeof(pak_trailer4);
	}

	if (rawtrailerlen == PAK_V14_SIZE) {
		rawtrailer += sizeof(pak_trailer10);
	}
	else if (rawtrailerlen == PAK_V12_SIZE) {
		rawtrailer += sizeof(pak_trailer5);
	}

	if (rawtrailerlen != PAK_VLITE_SIZE) {
		memcpy(&p6, rawtrailer, sizeof(pak_trailer6));
		rawtrailer += sizeof(pak_trailer6);
	}

	if (rawtrailerlen == PAK_V14_SIZE) {
		memcpy(&u1.p8, rawtrailer, sizeof(pak_trailer8));
		rawtrailer += sizeof(pak_trailer8);
	}
	else if (rawtrailerlen != PAK_VLITE_SIZE) {
		memcpy(&u1.s.p7, rawtrailer, sizeof(pak_trailer7));
		rawtrailer += sizeof(pak_trailer7);
	}

	if (rawtrailerlen != PAK_VLITE_SIZE) {
		memcpy(&p9, rawtrailer, sizeof(pak_trailer9));
		rawtrailer += sizeof(pak_trailer9);
	}

	if ((rawtrailerlen != PAK_V14_SIZE) && (rawtrailerlen != PAK_VLITE_SIZE)) {
		rawtrailer += sizeof(pak_trailer10);
	}

	if (rawtrailerlen != PAK_VLITE_SIZE) {
		rawtrailer += sizeof(pak_trailer11);
	}

	if (rawtrailerlen == PAK_V14_SIZE) {
		rawtrailer += sizeof(pak_trailer12);
	}
	else if (rawtrailerlen != PAK_VLITE_SIZE) {
		memcpy(&u1.s.p13, rawtrailer, sizeof(pak_trailer13));
		rawtrailer += sizeof(pak_trailer13);
	}

	trailer->reg_pc = ENDIANIZE(p3.reg_pc);
	trailer->reg_x = ENDIANIZE(p4.reg_x);
	trailer->reg_y = ENDIANIZE(p4.reg_y);
	trailer->reg_u = ENDIANIZE(p4.reg_u);
	trailer->reg_s = ENDIANIZE(p4.reg_s);
	trailer->reg_dp = p4.reg_dp;
	trailer->reg_b = p4.reg_b;
	trailer->reg_a = p4.reg_a;
	trailer->reg_cc = p6.reg_cc;

	switch(rawtrailerlen) {
	case PAK_V14_SIZE:
		trailer->io_ff02 = u1.p8.io_ff02;
		trailer->io_ff03 = u1.p8.io_ff03;
		trailer->io_ff22 = u1.p8.io_ff22;
		trailer->enable_hiram = (u1.p8.rom_status == 0xdf);
		trailer->video_base = ENDIANIZE(p9.video_base);
		trailer->video_end = ENDIANIZE(p9.video_end);
		break;

	case PAK_VLITE_SIZE:
		/* This is a "lite" format that gives no trailer information except
		 * for the program counter... I have to set everything up as default
		 * including the stack pointer... arg this sucks...
		 */
		trailer->io_ff02 = 0xff;
		trailer->io_ff03 = 0x34;
		trailer->io_ff22 = 0x00;
		trailer->enable_hiram = 0;
		trailer->video_base = 0x400;
		trailer->video_end = 0x600;
		trailer->reg_s = 0x3d7;
		break;

	default:
		trailer->io_ff02 = u1.s.p13.io_ff02;
		trailer->io_ff03 = u1.s.p13.io_ff03;
		trailer->io_ff22 = u1.s.p13.io_ff22;
		trailer->enable_hiram = (u1.s.p7.himem_readseg == 0);
		trailer->video_base = ENDIANIZE(p9.video_base);
		trailer->video_end = ENDIANIZE(p9.video_end);
		break;
	}
	return 0;
}

static void pak_load_trailer(const pak_trailer *trailer)
{
	extern void dragon_sam_display_offset(int offset, int data);
	extern void dragon_sam_vdg_mode(int offset, int data);
	int i, value;

	cpu_set_reg(M6809_PC, trailer->reg_pc);
	cpu_set_reg(M6809_X, trailer->reg_x);
	cpu_set_reg(M6809_Y, trailer->reg_y);
	cpu_set_reg(M6809_U, trailer->reg_u);
	cpu_set_reg(M6809_S, trailer->reg_s);
	cpu_set_reg(M6809_DP, trailer->reg_dp);
	cpu_set_reg(M6809_B, trailer->reg_b);
	cpu_set_reg(M6809_A, trailer->reg_a);
	cpu_set_reg(M6809_CC, trailer->reg_cc);

	/* I seem to only be able to get a small amount of the PIA state from the
	 * snapshot trailers. Thus I am going to configure the PIA myself. The
	 * following PIA writes are the same thing that the CoCo ROM does on
	 * startup. I wish I had a better solution
	 */
	cpu_writemem16(0xff1d, 0x00);
	cpu_writemem16(0xff1f, 0x00);
	cpu_writemem16(0xff1c, 0x00);
	cpu_writemem16(0xff1e, 0xff);
	cpu_writemem16(0xff1d, 0x34);
	cpu_writemem16(0xff1f, 0x34);
	cpu_writemem16(0xff21, 0x00);
	cpu_writemem16(0xff23, 0x00);
	cpu_writemem16(0xff20, 0xfe);
	cpu_writemem16(0xff22, 0xf8);
	cpu_writemem16(0xff21, 0x34);
	cpu_writemem16(0xff23, 0x34);
	cpu_writemem16(0xff22, 0x00);
	cpu_writemem16(0xff20, 0x02);

	cpu_writemem16(0xff03, trailer->io_ff03);	/* d_pia0_cb2_w */
	cpu_writemem16(0xff02, trailer->io_ff02);	/* d_pia0_pb_w */
	cpu_writemem16(0xff22, trailer->io_ff22);	/* d_pia1_pb_w */

	/* For some reason, this seems to screw things up; I'm not sure whether it
	 * is because I'm using the wrong method to get access
	 * trailer->enable_hiram or whether it is osmething else
	 */
	/* cpu_writemem16(0xffde + trailer->enable_hiram, 0); */

	value = trailer->video_base >> 9;
	for (i = 0; i < 6; i++) {
		dragon_sam_display_offset(i * 2 + (value & 1), 0);
		value >>= 1;
	}

	switch(trailer->video_end - trailer->video_base) {
	case 512:
		value = 0;
		break;
	case 1024:
		value = 1;
		break;
	case 1536:
	case 2048:
		value = 2;
		break;
	case 3072:
		value = 4;
		break;
	case 6144:
	default:
		value = 6;
		break;
	}

	for (i = 0; i < 3; i++) {
		dragon_sam_vdg_mode(i * 2 + (value & 1), 0);
		value >>= 1;
	}
}

static int trailer_load = 0;
static pak_trailer trailer;

static void pak_load_trailer_callback(int param)
{
	pak_load_trailer(&trailer);
}

static int generic_rom_load(int id, UINT8 *rambase, UINT8 *rombase, UINT8 *pakbase)
{
	void *fp;

	fp = image_fopen (IO_SNAPSHOT, id, OSD_FILETYPE_IMAGE_R, 0);
	if (fp)
	{
		int paklength;
		int pakstart;

		pak_header header;
		int trailerlen;
		UINT8 trailerraw[500];

		if (osd_fread(fp, &header, sizeof(header)) < sizeof(header)) {
#if LOG_PAK
			logerror("Could not fully read PAK.\n");
#endif
			osd_fclose(fp);
			return 1;
		}

		paklength = header.length ? ENDIANIZE(header.length) : 0x10000;
		pakstart = ENDIANIZE(header.start);
		if (pakstart == 0xc000)
			cart_inserted = 1;

		if (osd_fseek(fp, paklength, SEEK_CUR)) {
#if LOG_PAK
			logerror("Could not fully read PAK.\n");
#endif
			osd_fclose(fp);
			return 1;
		}

		trailerlen = osd_fread(fp, trailerraw, sizeof(trailerraw));
		if (trailerlen) {
			if (pak_decode_trailer(trailerraw, trailerlen, &trailer)) {
#if LOG_PAK
				logerror("Invalid or unknown PAK trailer.\n");
#endif
				osd_fclose(fp);
				return 1;
			}

			trailer_load = 1;
		}

		if (osd_fseek(fp, sizeof(pak_header), SEEK_SET)) {
#if LOG_PAK
			logerror("Unexpected error while reading PAK.\n");
#endif
			osd_fclose(fp);
			return 1;
		}

		/* Now that we are done reading the trailer; we can cap the length */
		if (paklength > 0xff00)
			paklength = 0xff00;

		/* Since PAK files allow the flexibility of loading anywhere in
		 * the base RAM or ROM, we have to do tricks because in MESS's
		 * memory, RAM and ROM may be separated, hense this function's
		 * two parameters.
		 *
		 * Similarly, some PAKs appear to be loading into high RAM.  I
		 * am not completely sure how to distinguish this, but I can
		 * guess
		 */

		/* Get the RAM portion */
		if (load_pak_into_region(fp, &pakstart, &paklength, rambase, 0x0000, 0x8000)) {
			osd_fclose(fp);
			return 1;
		}

		if (pakstart == 0x8000) {
			/* We are probably loading into high RAM */
			if ((rombase - rambase) < 0x10000) {
				if (load_pak_into_region(fp, &pakstart, &paklength, rambase, 0x8000, 0x7F00)) {
					osd_fclose(fp);
					return 1;
				}
			}
		}
		else {
			/* Get the ROM portion */
			if (load_pak_into_region(fp, &pakstart, &paklength, rombase, 0x8000, 0x4000)) {
				osd_fclose(fp);
				return 1;
			}
			/* Get the PAK portion */
			if (load_pak_into_region(fp, &pakstart, &paklength, pakbase, 0xC000, 0x3F00)) {
				osd_fclose(fp);
				return 1;
			}
		}
		osd_fclose(fp);
	}
	return INIT_OK;
}

int dragon32_rom_load(int id)
{
	UINT8 *ROM = memory_region(REGION_CPU1);
	return generic_rom_load(id, &ROM[0], &ROM[0x8000], &ROM[0xc000]);
}

int dragon64_rom_load(int id)
{
	UINT8 *ROM = memory_region(REGION_CPU1);
	return generic_rom_load(id, &ROM[0], &ROM[0x10000], &ROM[0x14000]);
}

int coco3_rom_load(int id)
{
	UINT8 *ROM = memory_region(REGION_CPU1);
	return generic_rom_load(id, &ROM[0x70000], &ROM[0x80000], &ROM[0x8c000]);
}

/***************************************************************************
  Interrupts

  The Dragon/CoCo2 have two PIAs.  These PIAs can trigger interrupts.  PIA0
  is set up to trigger IRQ on the CPU, and PIA1 can trigger FIRQ.  Each PIA
  has two output lines, and an interrupt will be triggered if either of these
  lines are asserted.

  -----  IRQ
  6809 |-<----------- PIA0
       |
       |
       |
       |
       |
       |-<----------- PIA1
  -----

  The CoCo 3 still supports these interrupts, but the GIME can chose whether
  "old school" interrupts are generated, or the new ones generated by the GIME

  -----  IRQ
  6809 |-<----------- PIA0
       |       |                ------
       |       -<-------<-------|    |
       |                        |GIME|
       |       -<-------<-------|    |
       | FIRQ  |                ------
       |-<----------- PIA1
  -----

  In an email discussion with JK, he informs me that when GIME interrupts are
  enabled, this actually does not prevent PIA interrupts.  Apparently JeffV's
  CoCo 3 emulator did not handle this properly.
***************************************************************************/

enum {
	COCO3_INT_TMR	= 0x20,		/* Timer */
	COCO3_INT_HBORD	= 0x10,		/* Horizontal border sync */
	COCO3_INT_VBORD	= 0x08,		/* Vertical border sync */
	COCO3_INT_EI2	= 0x04,		/* Serial data */
	COCO3_INT_EI1	= 0x02,		/* Keyboard */
	COCO3_INT_EI0	= 0x01,		/* Cartridge */

	COCO3_INT_ALL	= 0x3f
};

static void d_recalc_irq(void)
{
	if ((pia0_irq_a == ASSERT_LINE) || (pia0_irq_b == ASSERT_LINE))
		cpu_set_irq_line(0, M6809_IRQ_LINE, ASSERT_LINE);
	else
		cpu_set_irq_line(0, M6809_IRQ_LINE, CLEAR_LINE);
}

static void d_recalc_firq(void)
{
	if ((pia1_firq_a == ASSERT_LINE) || (pia1_firq_b == ASSERT_LINE))
		cpu_set_irq_line(0, M6809_FIRQ_LINE, ASSERT_LINE);
	else
		cpu_set_irq_line(0, M6809_FIRQ_LINE, CLEAR_LINE);
}

static void coco3_recalc_irq(void)
{
	if ((coco3_gimereg[0] & 0x20) && gime_irq)
		cpu_set_irq_line(0, M6809_IRQ_LINE, ASSERT_LINE);
	else
		d_recalc_irq();
}

static void coco3_recalc_firq(void)
{
	if ((coco3_gimereg[0] & 0x10) && gime_firq)
		cpu_set_irq_line(0, M6809_FIRQ_LINE, ASSERT_LINE);
	else
		d_recalc_firq();
}

static void d_pia0_irq_a(int state)
{
	pia0_irq_a = state;
	d_recalc_irq();
}

static void d_pia0_irq_b(int state)
{
	pia0_irq_b = state;
	d_recalc_irq();
}

static void d_pia1_firq_a(int state)
{
	pia1_firq_a = state;
	d_recalc_firq();
}

static void d_pia1_firq_b(int state)
{
	pia1_firq_b = state;
	d_recalc_firq();
}

static void coco3_pia0_irq_a(int state)
{
	pia0_irq_a = state;
	coco3_recalc_irq();
}

static void coco3_pia0_irq_b(int state)
{
	pia0_irq_a = state;
	coco3_recalc_irq();
}

static void coco3_pia1_firq_a(int state)
{
	pia1_firq_a = state;
	coco3_recalc_firq();
}

static void coco3_pia1_firq_b(int state)
{
	pia1_firq_a = state;
	coco3_recalc_firq();
}

static void coco3_raise_interrupt(int mask, int state)
{
	if (state) {
		if ((coco3_gimereg[0] & 0x20) && (coco3_gimereg[2] & mask)) {
			gime_irq |= (coco3_gimereg[2] & mask);
			coco3_recalc_irq();

#if LOG_INT_COCO3
			logerror("CoCo3 Interrupt: Raising IRQ\n");
#endif
		}
		if ((coco3_gimereg[0] & 0x10) && (coco3_gimereg[3] & mask)) {
			gime_firq |= (coco3_gimereg[3] & mask);
			coco3_recalc_firq();

#if LOG_INT_COCO3
			logerror("CoCo3 Interrupt: Raising FIRQ\n");
#endif
		}
	}
}

/***************************************************************************
  PIA
***************************************************************************/

int dragon_interrupt(void)
{
	pia_0_cb1_w (0, 1);
	return ignore_interrupt();
}

static READ_HANDLER ( d_pia0_ca1_r )
{
	return 0;
}

static READ_HANDLER ( d_pia1_cb1_r )
{
	return cart_inserted;
}

static WRITE_HANDLER ( d_pia1_cb2_w )
{
	int status;

#if LOG_WAVE
	logerror("CoCo: Turning cassette speaker %s\n", data ? "on" : "off");
#endif

	status = device_status(IO_CASSETTE, 0, -1);
	status &= ~2;
	if (!data)
		status |= 2;
	device_status(IO_CASSETTE, 0, status);

	sound_mux = data;
}

static WRITE_HANDLER ( d_pia1_pa_w )
{
	/*
	 *	This port appears at $FF20
	 *
	 *	Bits
	 *  7-2:	DAC or cassette
	 *    1:	Serial out
	 */
	d_dac = data & 0xfc;
	if (sound_mux)
		DAC_data_w(0,d_dac);
	else
		device_output(IO_CASSETTE, 0, ((int) d_dac - 0x80) * 0x102);

}

/*
 * This port appears at $FF23
 *
 * The CoCo 1/2 and Dragon kept the gmode and vmode separate, and the CoCo
 * 3 tied them together.  In the process, the CoCo 3 dropped support for the
 * semigraphics modes
 */

static WRITE_HANDLER( d_pia1_pb_w )
{
	m6847_set_gmode(data >> 3);
}

static WRITE_HANDLER( coco3_pia1_pb_w )
{
	m6847_set_mode(data >> 3);
}

static WRITE_HANDLER ( d_pia0_cb2_w )
{
	joystick = data;
}

static WRITE_HANDLER ( d_pia1_ca2_w )
{
	int status;

	if (tape_motor ^ data)
	{
		status = device_status(IO_CASSETTE, 0, -1);
		status &= ~1;
		if (data)
			status |= 1;
		device_status(IO_CASSETTE, 0, status);
		tape_motor = data;
	}
}

static WRITE_HANDLER ( d_pia0_ca2_w )
{
	joystick_axis = data;
}

static int keyboard_r(void)
{
	int porta=0x7f;

	if ((input_port_0_r(0) | pia0_pb) != 0xff) porta &= ~0x01;
	if ((input_port_1_r(0) | pia0_pb) != 0xff) porta &= ~0x02;
	if ((input_port_2_r(0) | pia0_pb) != 0xff) porta &= ~0x04;
	if ((input_port_3_r(0) | pia0_pb) != 0xff) porta &= ~0x08;
	if ((input_port_4_r(0) | pia0_pb) != 0xff) porta &= ~0x10;
	if ((input_port_5_r(0) | pia0_pb) != 0xff) porta &= ~0x20;
	if ((input_port_6_r(0) | pia0_pb) != 0xff) porta &= ~0x40;
	if (d_dac <= readinputport(joystick ? (joystick_axis ? 10 : 9) : (joystick_axis ? 8 : 7)))
		porta |= 0x80;
	porta &= ~readinputport(11);

	return porta;
}

static READ_HANDLER ( d_pia0_pa_r )
{
	return keyboard_r();
}

static void coco3_poll_keyboard(int dummy)
{
	static int old_porta = 0x7f;
	int porta;
	int new_porta;

	porta = keyboard_r();
	new_porta = (~old_porta | porta) & 0x7f;
	old_porta = porta;

	/* If there is a new hit on the keyboard, generate EI1 */
	if (new_porta != 0x7f) {
		coco3_raise_interrupt(COCO3_INT_EI1, 1);
		coco3_raise_interrupt(COCO3_INT_EI1, 0);
	}
}

static READ_HANDLER ( d_pia1_pa_r )
{
	return (device_input(IO_CASSETTE, 0) >= 0) ? 1 : 0;
}

static WRITE_HANDLER ( d_pia0_pb_w )
{
	pia0_pb = data;
}

/***************************************************************************
  Misc
***************************************************************************/

int dragon_mapped_irq_r(int offset)
{
	return coco_rom[0x3ff0 + offset];
}

void dragon_sam_speedctrl(int offset, int data)
{
	/* The infamous speed up poke.
	 *
	 * This was a SAM switch that occupied 4 addresses:
	 *
	 *		$FFD9	(set)	R1
	 *		$FFD8	(clear)	R1
	 *		$FFD7	(set)	R0
	 *		$FFD6	(clear)	R0
	 *
	 * R1:R0 formed the following states:
	 *		00	- slow          0.89 MHz
	 *		01	- dual speed    ???
	 *		1x	- fast          1.78 MHz
	 *
	 * R1 controlled whether the video addressing was speeded up and R0
	 * did the same for the CPU.  On pre-CoCo 3 machines, setting R1 caused
	 * the screen to display garbage because the M6847 could not display
	 * fast enough.
	 *
	 * TODO:  Make the overclock more accurate.  In dual speed, ROM was a fast
	 * access but RAM was not.  I don't know how to simulate this.
	 */
    timer_set_overclock(0, 1+(offset&1));
}

void coco3_sam_speedctrl(int offset, int data)
{
	/* The CoCo 3 only had $FFD8-$FFD9 */
	dragon_sam_speedctrl(offset + 2, data);
}

void dragon_sam_page_mode(int offset, int data)
{
	/* Page mode - allowed switching between the low 32k and the high 32k,
	 * assuming that 64k wasn't enabled
	 *
	 * TODO:  Actually implement this.  Also find out what the CoCo 3 did with
	 * this (it probably ignored it)
	 */
}

static void recalc_vram_size(void)
{
	static int vram_masks[] = { 0xfff, 0x3fff, 0xffff, 0xffff };

	m6847_set_vram_mask(vram_masks[d_sam_memory_size % 4]);
}

void dragon_sam_memory_size(int offset, int data)
{
	/* Memory size - allowed restricting memory accesses to something less than
	 * 32k
	 *
	 * This was a SAM switch that occupied 4 addresses:
	 *
	 *		$FFDD	(set)	R1
	 *		$FFDC	(clear)	R1
	 *		$FFDB	(set)	R0
	 *		$FFDA	(clear)	R0
	 *
	 * R1:R0 formed the following states:
	 *		00	- 4k
	 *		01	- 16k
	 *		10	- 64k
	 *		11	- static RAM (??)
	 *
	 * If something less than 64k was set, the low RAM would be smaller and
	 * mirror the other parts of the RAM
	 *
	 * TODO:  Find out what "static RAM" is
	 * TODO:  This should affect _all_ memory accesses, not just video ram
	 * TODO:  Verify that the CoCo 3 ignored this
	 */
	if (offset & 1)
		d_sam_memory_size &= ~(1 << (offset / 2));
	else
		d_sam_memory_size |= 1 << (offset / 2);

	recalc_vram_size();
}

/***************************************************************************
  CoCo 3 Timer

  The CoCo 3 had a timer that had would activate when first written to, and
  would decrement over and over again until zero was reached, and at that
  point, would flag an interrupt.  At this point, the timer starts back up
  again.

  I am deducing that the timer interrupt line was asserted if the timer was
  zero and unasserted if the timer was non-zero.  Since we never truly track
  the timer, we just use timer callback (coco3_timer_callback() asserts the
  lin)
***************************************************************************/

static void *coco3_timer;
static int coco3_timer_interval;	/* interval: 1=280 nsec, 0=63.5 usec */
static int coco3_timer_base;
static int coco3_timer_counter;
extern void coco3_vh_blink(void);

static void coco3_timer_init(void)
{
	coco3_timer = 0;
	coco3_timer_interval = 0;
}

static void coco3_timer_uncallback(int dummy)
{
	/* This "uncallback" is used to unassert the line */
}

static void coco3_timer_reset(void)
{
	/* This resets the timer back to the original value
	 *
	 * JK tells me that when the timer resets, it gets reset to a value that
	 * is 1 (with the 1986 GIME) above or 2 (with the 1987 GIME) above the
	 * value written into the timer.  coco3_timer_base keeps track of the value
	 * placed into the variable, so we increment that here
	 *
	 * For now, we are emulating the 1986 GIME
	 */
	coco3_timer_recalculate(coco3_timer_base ? coco3_timer_base + 1 : coco3_timer_base, 1);
}

static void coco3_timer_callback(int dummy)
{
#if LOG_INT_TMR
	logerror("CoCo3 GIME: Triggering TMR interrupt\n");
#endif
	coco3_timer = 0;
	coco3_raise_interrupt(COCO3_INT_TMR, 1);

	/* HACKHACK - This should not happen until the next timer tick */
	coco3_raise_interrupt(COCO3_INT_TMR, 0);

	coco3_timer_reset();
	coco3_vh_blink();

}

static double coco3_timer_interval_time(void)
{
	/* Most CoCo 3 docs, including the specs that Tandy released, say that the
	 * high speed timer is 70ns (half of the speed of the main clock crystal).
	 * However, it seems that this is in error, and the GIME timer is really a
	 * 280ns timer (one eighth the speed of the main clock crystal.  Gault's
	 * FAQ agrees with this
	 */
	return coco3_timer_interval ? COCO_TIMER_CMPCARRIER : COCO_TIMER_HSYNC;
}

/* This function takes the value in coco3_timer_counter, and sets up the timer
 * and the coco3_time_counter_time for it
 */
static void coco3_timer_recalculate(int newcounterval, int allowreset)
{
	if (coco3_timer)
		timer_remove(coco3_timer);

	if (newcounterval || !allowreset) {
		coco3_timer = timer_set(newcounterval * coco3_timer_interval_time(),
			0, coco3_timer_callback);
#if LOG_INT_TMR
		logerror("CoCo3 GIME: Timer will elapse in %f\n", newcounterval * coco3_timer_interval_time());
		if (!coco3_timer)
			logerror("CoCo3 GIME: Timer allocation failed!\n");
#endif
	}
	else {
		coco3_timer = 0;
#if LOG_INT_TMR
		logerror("CoCo3 GIME: Turning timer off\n");
#endif
	}

	coco3_timer_counter = newcounterval;
}

static int coco3_timer_r(void)
{
	int result = 0;

	if (coco3_timer) {
		result = coco3_timer_counter -
			(timer_timeleft(coco3_timer) / coco3_timer_interval_time());

		/* This shouldn't happen, but I'm prepared anyways */
		if (result < 0)
			result = 0;
		else if (result > 4095)
			result = 4095;
	}
	return result;	/* result = 0..4095 */
}

static void coco3_timer_w(int data)	/* data = 0..4095 */
{
	coco3_timer_base = (data & 4095);
	coco3_timer_reset();
}

static void coco3_timer_msb_w(int data)
{
	coco3_timer_w(((data & 0x0f) << 8) | (coco3_timer_base & 0xff));
}

static void coco3_timer_lsb_w(int data)
{
	coco3_timer_w((coco3_timer_base & 0xf00) | (data & 0xff));
}

static void coco3_timer_set_interval(int interval)
{
	int oldtimerval;

	if (interval != coco3_timer_interval) {
		if (coco3_timer) {
			oldtimerval = coco3_timer_r();

			coco3_timer_interval = interval;
			coco3_timer_recalculate(oldtimerval, 0);
		}
		else {
			coco3_timer_interval = interval;
		}
	}
}

/***************************************************************************
  MMU
***************************************************************************/

/* from vidhrdw/dragon.c */
extern void coco_ram_w(int offset, int data);

WRITE_HANDLER ( dragon64_ram_w )
{
	coco_ram_w(offset + 0x8000, data);
}

void dragon64_sam_himemmap(int offset, int data)
{
	UINT8 *RAM = memory_region(REGION_CPU1);
	if (offset) {
		cpu_setbank(1, &RAM[0x8000]);
		cpu_setbankhandler_w(1, dragon64_ram_w);
	}
	else {
		cpu_setbank(1, coco_rom);
		cpu_setbankhandler_w(1, MWA_ROM);
	}
}

/* Coco 3 */

static int coco3_mmu_lookup(int block, int forceram)
{
	int result;

	if (coco3_gimereg[0] & 0x40) {
		if (coco3_gimereg[1] & 1)
			block += 8;
		result = coco3_mmu[block];
	}
	else {
		result = block + 56;
	}

	/* Are we actually in ROM?
	 *
	 * In our world, ROM is represented by memory blocks 0x40-0x47
	 *
	 * 0x40			Extended Color Basic
	 * 0x41			Color Basic
	 * 0x42			Reset Initialization
	 * 0x43			Super Extended Color Basic
	 * 0x44-0x47	Cartridge ROM
	 *
	 * This is the level where ROM is mapped, according to Tepolt (p21)
	 */
	if ((result >= 0x3c) && !coco3_enable_64k && !forceram) {
		static const int rommap[4][4] = {
			{ 0x40, 0x41, 0x46, 0x47 },
			{ 0x40, 0x41, 0x46, 0x47 },
			{ 0x40, 0x41, 0x42, 0x43 },
			{ 0x44, 0x45, 0x46, 0x47 }
		};
		result = rommap[coco3_gimereg[0] & 3][result - 0x3c];
	}
	
	return result;
}

int coco3_mmu_translate(int block, int offset)
{
	int forceram;

	/* Block 8 is the 0xfe00 block; and it is treated differently */
	if (block == 8) {
		if (coco3_gimereg[0] & 8)
			return 0x7fe00 + offset;
		block = 7;
		offset += 0x1e00;
		forceram = 1;
	}
	else {
		forceram = 0;
	}
	return (coco3_mmu_lookup(block, forceram) * 0x2000) + offset;
}

static void coco3_mmu_update(int lowblock, int hiblock)
{
	UINT8 *RAM = memory_region(REGION_CPU1);
	typedef void (*writehandler)(UINT32 wh_offset, UINT32 data);
	static writehandler handlers[] = {
		coco3_ram_b1_w, coco3_ram_b2_w,
		coco3_ram_b3_w, coco3_ram_b4_w,
		coco3_ram_b5_w, coco3_ram_b6_w,
		coco3_ram_b7_w, coco3_ram_b8_w,
		coco3_ram_b9_w
	};

	int i;
	UINT8 *p;

	for (i = lowblock; i <= hiblock; i++) {
		p = &RAM[coco3_mmu_translate(i, 0)];
		cpu_setbank(i + 1, p);
		cpu_setbankhandler_w(i + 1, ((p - RAM) >= 0x80000) ? MWA_ROM : handlers[i]);
#if LOG_MMU
		logerror("CoCo3 GIME MMU: Logical $%04x ==> Physical $%05x\n",
			(i == 8) ? 0xfe00 : i * 0x2000,
			p - RAM);
#endif
	}
}

int coco3_mmu_r(int offset)
{
	return coco3_mmu[offset] & 0x3f;
}

void coco3_mmu_w(int offset, int data)
{
	data &= 0x3f;
	coco3_mmu[offset] = data;

	/* Did we modify the live MMU bank? */
	if ((offset >> 3) == (coco3_gimereg[1] & 1)) {
		offset &= 7;
		coco3_mmu_update(offset, (offset == 7) ? 8 : offset);
	}
}

/***************************************************************************
  GIME Registers (Reference: Super Extended Basic Unravelled)
***************************************************************************/

int coco3_gime_r(int offset)
{
	int result = 0;

	switch(offset) {
	case 2:	/* Read pending IRQs */
		result = gime_irq;
		if (result) {
			gime_irq = 0;
			coco3_recalc_irq();
		}
		break;

	case 3:	/* Read pending FIRQs */
		result = gime_firq;
		if (result) {
			gime_firq = 0;
			coco3_recalc_firq();
		}
		break;

	case 4:	/* Timer MSB/LSB; these arn't readable */
	case 5:
		/* JK tells me that these values are indeterminate; and $7E appears
		 * to be the value most commonly returned
		 */
		result = 0x7e;
		break;

	default:
		result = coco3_gimereg[offset];
		break;
	}
	return result;
}

void coco3_gime_w(int offset, int data)
{
	coco3_gimereg[offset] = data;

#if LOG_GIME
	logerror("CoCo3 GIME: $%04x <== $%02x pc=$%04x\n", offset + 0xff90, data, m6809_get_pc());
#endif

	/* Features marked with '!' are not yet implemented */
	switch(offset) {
	case 0:
		/*	$FF90 Initialization register 0
		 *		  Bit 7 COCO 1=CoCo compatible mode
		 *		  Bit 6 MMUEN 1=MMU enabled
		 *		  Bit 5 IEN 1 = GIME chip IRQ enabled
		 *		  Bit 4 FEN 1 = GIME chip FIRQ enabled
		 *		  Bit 3 MC3 1 = RAM at FEXX is constant
		 *		  Bit 2 MC2 1 = standard SCS (Spare Chip Select)
		 *		  Bit 1 MC1 ROM map control
		 *		  Bit 0 MC0 ROM map control
		 */
		coco3_vh_sethires(data & 0x80 ? 0 : 1);
		coco3_mmu_update(0, 8);
		break;

	case 1:
		/*	$FF91 Initialization register 1
		 *		  Bit 7 Unused
		 *		  Bit 6 Unused
		 *		  Bit 5 TINS Timer input select; 1 = 70 nsec, 0 = 63.5 usec
		 *		  Bit 4 Unused
		 *		  Bit 3 Unused
		 *		  Bit 2 Unused
		 *		  Bit 1 Unused
		 *		  Bit 0 TR Task register select
		 */
		coco3_mmu_update(0, 8);
		coco3_timer_set_interval(data & 0x20 ? 1 : 0);
		break;

	case 2:
		/*	$FF92 Interrupt request enable register
		 *		  Bit 7 Unused
		 *		  Bit 6 Unused
		 *		  Bit 5 TMR Timer interrupt
		 *		  Bit 4 HBORD Horizontal border interrupt
		 *		  Bit 3 VBORD Vertical border interrupt
		 *		! Bit 2 EI2 Serial data interrupt
		 *		  Bit 1 EI1 Keyboard interrupt
		 *		  Bit 0 EI0 Cartridge interrupt
		 */
#if LOG_INT_MASKING
		logerror("CoCo3 IRQ: Interrupts { %s%s%s%s%s%s} enabled\n",
			(data & 0x20) ? "TMR " : "",
			(data & 0x10) ? "HBORD " : "",
			(data & 0x08) ? "VBORD " : "",
			(data & 0x04) ? "EI2 " : "",
			(data & 0x02) ? "EI1 " : "",
			(data & 0x01) ? "EI0 " : "");
#endif
		break;

	case 3:
		/*	$FF93 Fast interrupt request enable register
		 *		  Bit 7 Unused
		 *		  Bit 6 Unused
		 *		  Bit 5 TMR Timer interrupt
		 *		  Bit 4 HBORD Horizontal border interrupt
		 *		  Bit 3 VBORD Vertical border interrupt
		 *		! Bit 2 EI2 Serial border interrupt
		 *		  Bit 1 EI1 Keyboard interrupt
		 *		  Bit 0 EI0 Cartridge interrupt
		 */
#if LOG_INT_MASKING
		logerror("CoCo3 FIRQ: Interrupts { %s%s%s%s%s%s} enabled\n",
			(data & 0x20) ? "TMR " : "",
			(data & 0x10) ? "HBORD " : "",
			(data & 0x08) ? "VBORD " : "",
			(data & 0x04) ? "EI2 " : "",
			(data & 0x02) ? "EI1 " : "",
			(data & 0x01) ? "EI0 " : "");
#endif
		break;

	case 4:
		/*	$FF94 Timer register MSB
		 *		  Bits 4-7 Unused
		 *		  Bits 0-3 High order four bits of the timer
		 */
		coco3_timer_msb_w(data);
		break;

	case 5:
		/*	$FF95 Timer register LSB
		 *		  Bits 0-7 Low order eight bits of the timer
		 */
		coco3_timer_lsb_w(data);
		break;
	}
}

void coco3_sam_himemmap(int offset, int data)
{
	coco3_enable_64k = offset;
	coco3_mmu_update(4, 8);
}

/***************************************************************************
  Joystick autocenter
***************************************************************************/

static int autocenter_val;

static void autocenter_timer_proc(int data)
{
	struct InputPort *in;
	int dipport, dipmask, portval;

	dipport = (data & 0xff00) >> 8;
	dipmask = data & 0x00ff;
	portval = readinputport(dipport) & dipmask;

	if (autocenter_val != portval) {
		/* Now go through all inputs, and set or reset IPF_CENTER on all
		 * joysticks
		 */
		for (in = Machine->input_ports; in->type != IPT_END; in++) {
			if (((in->type & ~IPF_MASK) > IPT_ANALOG_START)
					&& ((in->type & ~IPF_MASK) < IPT_ANALOG_END)) {
				/* We found a joystick */
				if (portval)
					in->type |= IPF_CENTER;
				else
					in->type &= ~IPF_CENTER;
			}
		}
	}
}

static void autocenter_init(int dipport, int dipmask)
{
	autocenter_val = -1;
	timer_pulse(TIME_IN_HZ(10), (dipport << 8) | dipmask, autocenter_timer_proc);
}

/***************************************************************************
  Cassette support
***************************************************************************/

#define WAVEENTRY_HIGH  32767
#define WAVEENTRY_LOW   -32768
#define WAVEENTRY_NULL  0
#define WAVESAMPLES_BYTE    8*4
#define WAVESAMPLES_HEADER  3000
#define WAVESAMPLES_TRAILER 1000

static INT16* fill_wave_byte(INT16 *p, UINT8 b)
{
    int i;
    /* Each byte in a .CAS file is read bit by bit, starting at bit 0, and
     * ending with bit 7.  High bits are decoded into {l,h} (a 2400hz pulse)
     * and low bits are decoded into {l,l,h,h} (a 1200hz pulse)
     */
    for (i = 0; i < 8; i++) {
        *(p++) = WAVEENTRY_LOW;
        if (((b >> i) & 0x01) == 0) {
            *(p++) = WAVEENTRY_LOW;
            *(p++) = WAVEENTRY_HIGH;
        }
        *(p++) = WAVEENTRY_HIGH;
    }
    return p;
}

static int get_cas_block(const UINT8 *bytes, int length, UINT8 *block, int *blocklen)
{
	int i, j, state, phase=0;
	UINT8 newb, b;
	UINT8 block_length=0, block_checksum=0;

	b = 0x00;
	state = 0;

	for (i = 0; i < length; i++) {
		newb = bytes[i];
		for (j = 0; j < 8; j++) {
			b >>= 1;
			if (newb & 1)
				b |= 0x80;
			newb >>= 1;

			if (state == 0) {
				/* Searching for a block */
				if (b == 0x3c) {
					/* Found one! */
					phase = j;
					state = 1;
				}
			}
			else if (j == phase) {
				switch(state) {
				case 1:
					/* Found file type */
					block_checksum = b;
					state++;
					break;

				case 2:
					/* Found file size */
					block_length = b;
					block_checksum += b;
					state++;
					*blocklen = ((int) block_length) + 3;
					break;

				case 3:
					/* Data byte */
					if (block_length) {
						block_length--;
						block_checksum += b;
					}
					else {
						/* End of block! check the checksum */
						if (b != block_checksum) {
							/* Checksum error */
							return 0;
						}
						/* We got a block! Return new position */
						*block = b;
						return i + 1;
					}
					break;

				}
				*(block++) = b;
			}
		}
	}
	/* Couldn't find a block */
	return 0;
}

static int wave_size;

static int coco_cassette_fill_wave(INT16 *buffer, int length, UINT8 *bytes)
{
	int i, usedbytes, blocklen, position;
	UINT8 last_blocktype;
    INT16 *p;
	UINT8 block[258];	/* 255 bytes per block + 3 (type, length, checksum) */

    p = buffer;

    if (bytes == CODE_HEADER) {
        for (i = 0; i < WAVESAMPLES_HEADER; i++)
            *(p++) = WAVEENTRY_NULL;
    }
    else if (bytes == CODE_TRAILER) {
        /* Fill in one magic byte; then the empty trailer */
        for (i = fill_wave_byte(p, 0x55) - p; i < WAVESAMPLES_TRAILER; i++)
            *(p++) = WAVEENTRY_NULL;
    }
    else {
		/* This is the actual code that processes the CAS data.  CAS files are
		 * a problem because they are a legacy of previous CoCo emulators, and
		 * most of these emulators patch the system as a shortcut.  In doing
		 * so, they make the CoCo more tolerant of short headers and lack of
		 * delays between blocks.  This legacy is reflected in most CAS files
		 * in use, and thus presents a problem for a pure hardware emulation
		 * like MESS.
		 *
		 * One alternative is to preprocess the data on the CAS, file by file,
		 * but this proves to be problematic because in the process, legitimate
		 * data that is unrecognized by the preprocessor may get dropped.
		 *
		 * The approach taken here is a hybrid approach - it retrieves the data
		 * block by block until an error occurs; be it the end of the CAS or a
		 * corrupt (?!) block.  When "legitimate" blocks are done processing,
		 * the remainder of the data is added to the waveform in a traditional
		 * manner.  The result has proven to work quite well.
		 *
		 * One slight issue; strict bounds checking is not done and as such,
		 * this code could theoretically overflow.  However, I made sure that
		 * double the amount of required memory was set aside so such overflows
		 * may be very rare in practice
		 */

		position = 0;
		last_blocktype = 0;

		while((usedbytes = get_cas_block(&bytes[position], wave_size, block, &blocklen)) > 0) {
			#if LOG_WAVE
				logerror("COCO found block; position=0x%06x type=%i len=%i\n", position, (int) block[0], blocklen);
			#endif

			/* was the last block a filename block? */
			if ((last_blocktype == 0) || (last_blocktype == 0xff)) {
				#if LOG_WAVE
					logerror("COCO filling silence %d\n", WAVESAMPLES_HEADER);
				#endif

				/* silence */
				for (i = 0; i < WAVESAMPLES_HEADER; i++)
					*(p++) = WAVEENTRY_NULL;
				/* sync data */
				for (i = 0; i < 128; i++)
					p = fill_wave_byte(p, 0x55);
			}
			/* Now fill in the magic bytes */
			p = fill_wave_byte(p, 0x55);
			p = fill_wave_byte(p, 0x3c);

			/* Write the block */
			for (i = 0; i < blocklen; i++)
				p = fill_wave_byte(p, block[i]);

			/* Now fill in the last magic byte */
			p = fill_wave_byte(p, 0x55);

			/* Move the pointers ahead etc */
			position += usedbytes;
			wave_size -= usedbytes;
			last_blocktype = block[0];
		}

		/* We havn't been able to decipher any further blocks; so we are going
		 * to output the rest of the CAS verbatim
		 */
		#if LOG_WAVE
			logerror("COCO leaving %i extraneous bytes; position=%i\n", wave_size, position);
		#endif

		for (i = 0; i < wave_size ; i++)
			p = fill_wave_byte(p, bytes[position + i]);
    }
    return p - buffer;
}

int coco_cassette_init(int id)
{
	void *file;
	struct wave_args wa;

	file = image_fopen(IO_CASSETTE, id, OSD_FILETYPE_IMAGE_RW, OSD_FOPEN_READ);
	if( file )
	{
		wave_size = osd_fsize(file);

		memset(&wa, 0, sizeof(&wa));
		wa.file = file;
		wa.chunk_size = wave_size;
		wa.chunk_samples = 8*8 * wave_size;	/* 8 bits * 4 samples */
		wa.smpfreq = 4800; /* cassette samples go at 4800 baud */
		wa.fill_wave = coco_cassette_fill_wave;
		wa.header_samples = WAVESAMPLES_HEADER;
		wa.trailer_samples = WAVESAMPLES_TRAILER;
		wa.display = 1;
		if( device_open(IO_CASSETTE,id,0,&wa) )
			return INIT_FAILED;

		/* immediately pause/mute the output */
        device_status(IO_CASSETTE,id,2);
		return INIT_OK;
	}

	file = image_fopen(IO_CASSETTE, id, OSD_FILETYPE_IMAGE_RW, OSD_FOPEN_RW_CREATE);
	if( file )
    {
		memset(&wa, 0, sizeof(&wa));
		wa.file = file;
		wa.display = 1;
		wa.smpfreq = 11025;
		if( device_open(IO_CASSETTE,id,1,&wa) )
            return INIT_FAILED;

		/* immediately pause the output */
        device_status(IO_CASSETTE,id,2);
		return INIT_OK;
    }

	return INIT_FAILED;
}

void coco_cassette_exit(int id)
{
	device_close(IO_CASSETTE,id);
}

/***************************************************************************
  Floppy disk controller
 ***************************************************************************
 * The CoCo and Dragon both use the Western Digital 1793 floppy disk
 * controller.  The wd1793's variables are mapped to $ff48-$ff4b on the CoCo
 * and on $ff40-$ff43 on the Dragon.  In addition, there is another register
 * called DSKREG that controls the interface with the wd1793.  DSKREG is
 * detailed below:.  But they appear to be
 *
 * References:
 *		CoCo:	Disk Basic Unravelled
 *		Dragon:	Inferences from the PC-Dragon source code
 * ---------------------------------------------------------------------------
 * DSKREG - the control register
 * CoCo ($ff40)                            Dragon ($ff48)
 *
 * Bit                                     Bit
 *	7 halt enable flag                      7 ???
 *	6 drive select #3                       6 ???
 *	5 density flag (0=single, 1=double)     5 halt enable flag
 *	4 write precompensation                 4 ???
 *	3 drive motor activation                3 ???
 *	2 drive select #2                       2 drive motor activation
 *	1 drive select #1                       1 drive select high bit
 *	0 drive select #0                       0 drive select low bit
 * ---------------------------------------------------------------------------
 */

static const char *floppy_name[4];
static int haltenable;
static int dskreg;
static int raise_nmi;

enum {
	HW_COCO,
	HW_DRAGON
};

static void coco_fdc_init(void)
{
	wd179x_init(1);
	dskreg = -1;
	raise_nmi = 0;
}

int coco_floppy_init(int id)
{
	floppy_name[id] = device_filename(IO_FLOPPY,id);
	return INIT_OK;
}

void coco_floppy_exit(int id)
{
	wd179x_select_drive(id, 0, NULL, NULL);
	floppy_name[id] = NULL;
}

static void coco_fdc_callback(int event)
{
	/* In all honesty, I believe that I should be able to tie the WD179X IRQ
	 * directly to the 6809 NMI input.  But it seems that if I do that, the NMI
	 * occurs when the last byte of a read is made without any delay.  This
	 * means that we drop the last byte of every sector read or written.  Thus,
	 * we will delay the NMI
	 */
	switch(event) {
	case WD179X_IRQ_CLR:
		m6809_set_nmi_line(CLEAR_LINE);
		break;
	case WD179X_IRQ_SET:
		raise_nmi = 1;
		break;
	case WD179X_DRQ_CLR:
		cpu_set_halt_line(0, CLEAR_LINE);
		break;
	case WD179X_DRQ_SET:
		/* I should be able to specify haltenable instead of zero, but several
		 * programs don't appear to work
		 */
		cpu_set_halt_line(0, 0 /*haltenable*/ ? ASSERT_LINE : CLEAR_LINE);
		break;
	}
}

static void set_dskreg(int data, int hardware)
{
	UINT8 drive = 0;
	UINT8 head = 0;
	int motor_mask = 0;
	int haltenable_mask = 0;
	int tracks;
	void *fd;

	switch(hardware) {
	case HW_COCO:
		if ((dskreg & 0x1cf) == (data & 0xcf))
			return;

		/* An email from John Kowalski informed me that if the last drive is
		 * selected, and one of the other drive bits is selected, then the
		 * second side is selected.  If multiple bits are selected in other
		 * situations, then both drives are selected, and any read signals get
		 * yucky.
		 */
		motor_mask = 0x08;
		haltenable_mask = 0x80;

		if (data & 0x04)
			drive = 2;
		else if (data & 0x02)
			drive = 1;
		else if (data & 0x01)
			drive = 0;
		else if (data & 0x40)
			drive = 3;
		else
			motor_mask = 0;

		head = ((data & 0x40) && (drive != 3)) ? 1 : 0;
		break;

	case HW_DRAGON:
		if ((dskreg & 0x127) == (data & 0x27))
			return;
		drive = data & 0x03;
		motor_mask = 0x04;
		haltenable_mask = 0x20;
		break;
	}

	haltenable = data & haltenable_mask;
	dskreg = data;

	if (data & motor_mask) {
		fd = wd179x_select_drive(drive, head, coco_fdc_callback, floppy_name[drive]);
		if (fd) {
			/* For now, assume that real floppies are always 35 tracks */
			tracks = (fd == REAL_FDD) ? 35 : (osd_fsize(fd) / (18*256));
			wd179x_set_geometry(drive, tracks, 1, 18, 256, 0, 0, 1);
		}
	}
	else {
		wd179x_stop_drive();
	}

}

static int dc_floppy_r(int offset)
{
	int result = 0;

	switch(offset & 0x0f) {
	case 8:
		result = wd179x_status_r(0);
		break;
	case 9:
		result = wd179x_track_r(0);
		break;
	case 10:
		result = wd179x_sector_r(0);
		break;
	case 11:
		result = wd179x_data_r(0);
		break;
	}
	return result;
}

static void dc_floppy_w(int offset, int data, int hardware)
{
	switch(offset & 0x0f) {
	case 0:
	case 1:
	case 2:
	case 3:
	case 4:
	case 5:
	case 6:
	case 7:
		if (raise_nmi) {
			m6809_set_nmi_line(ASSERT_LINE);
			raise_nmi = 0;
		}
		set_dskreg(data, hardware);
		break;
	case 8:
		wd179x_command_w(0, data);
		break;
	case 9:
		wd179x_track_w(0, data);
		break;
	case 10:
		wd179x_sector_w(0, data);
		break;
	case 11:
		wd179x_data_w(0, data);
		break;
	};
}

/* ---------------------------------------------------- */

int coco_floppy_r(int offset)
{
	return dc_floppy_r(offset);
}

void coco_floppy_w(int offset, int data)
{
	dc_floppy_w(offset, data, HW_COCO);
}

int coco3_floppy_r(int offset)
{
	return ((coco3_gimereg[0] & 0x04) || (offset >= 0x10)) ? coco_floppy_r(offset) : 0;
}

void coco3_floppy_w(int offset, int data)
{
	if ((coco3_gimereg[0] & 0x04) || (offset >= 0x10))
		coco_floppy_w(offset, data);
}

int dragon_floppy_r(int offset)
{
	return dc_floppy_r(offset ^ 8);
}

void dragon_floppy_w(int offset, int data)
{
	dc_floppy_w(offset ^ 8, data, HW_DRAGON);
}

/***************************************************************************
  Machine Initialization
***************************************************************************/

static void dragon_hblank(int dummy)
{
	pia_0_ca1_w(0, 0);
}

static void coco3_hblank(int dummy)
{
	pia_0_ca1_w(0, 0);
	coco3_raise_interrupt(COCO3_INT_HBORD, 1);
	coco3_raise_interrupt(COCO3_INT_HBORD, 0);
}

void coco3_vblank(void)
{
	pia_0_cb1_w(0, 0);
	coco3_raise_interrupt(COCO3_INT_VBORD, 1);
	coco3_raise_interrupt(COCO3_INT_VBORD, 0);
}

static void generic_init_machine(struct pia6821_interface *piaintf)
{
	pia0_irq_a = CLEAR_LINE;
	pia0_irq_b = CLEAR_LINE;
	pia1_firq_a = CLEAR_LINE;
	pia1_firq_b = CLEAR_LINE;

	pia_config(0, PIA_STANDARD_ORDERING | PIA_8BIT, &piaintf[0]);
	pia_config(1, PIA_STANDARD_ORDERING | PIA_8BIT, &piaintf[1]);
	pia_reset();

	if (trailer_load) {
		trailer_load = 0;
		timer_set(0, 0, pak_load_trailer_callback);
	}

	coco_fdc_init();
	autocenter_init(12, 0x04);
}

void dragon32_init_machine(void)
{
	d_sam_memory_size = 0;
	generic_init_machine(dragon_pia_intf);

	coco_rom = memory_region(REGION_CPU1) + 0x8000;

	if (cart_inserted)
		cpu_set_irq_line(0, M6809_FIRQ_LINE, ASSERT_LINE);
	timer_pulse(COCO_TIMER_HSYNC, 0, dragon_hblank);
}

void coco_init_machine(void)
{
	d_sam_memory_size = 0;
	generic_init_machine(dragon_pia_intf);

	coco_rom = memory_region(REGION_CPU1) + 0x10000;

	if (cart_inserted)
		cpu_set_irq_line(0, M6809_FIRQ_LINE, ASSERT_LINE);
	dragon64_sam_himemmap(0, 0);
	timer_pulse(COCO_TIMER_HSYNC, 0, dragon_hblank);
}


void dragon64_init_machine(void)
{
	dragon32_init_machine();
	coco_rom = memory_region(REGION_CPU1) + 0x10000;
	dragon64_sam_himemmap(0, 0);
}

void coco3_init_machine(void)
{
	int i;

	/* Tepolt verifies that the GIME registers are all cleared on initialization */

	coco3_enable_64k = 0;
	gime_irq = 0;
	gime_firq = 0;
	for (i = 0; i < 8; i++) {
		coco3_mmu[i] = coco3_mmu[i + 8] = 56 + i;
		coco3_gimereg[i] = 0;
	}

	generic_init_machine(coco3_pia_intf);

	coco_rom = memory_region(REGION_CPU1) + 0x80000;

	if (cart_inserted)
		coco3_raise_interrupt(COCO3_INT_EI0, 1);

	coco3_mmu_update(0, 8);
	coco3_timer_init();

	/* The choise of 50hz is arbitrary */
	timer_pulse(TIME_IN_HZ(50), 0, coco3_poll_keyboard);

	timer_pulse(COCO_TIMER_HSYNC, 0, coco3_hblank);
}

void dragon_stop_machine(void)
{
}

/***************************************************************************
  Other hardware
****************************************************************************

  This section discusses hardware/accessories/enhancements to the CoCo (mainly
  CoCo 3) that exist but are not emulated yet.

  HI-RES JOYSTICK - A joystick that shared the joystick and serial ports that
  provided higher accuracy.  Programs like CoCo Max used it.

  IDE - There is an IDE ehancement for the CoCo 3.  Here are some docs on the
  interface (info courtesy: LCB):

		The IDE interface (which is jumpered to be at either at $ff5x or $ff7x
		is mapped thusly:

		$FFx0 - 1st 8 bits of 16 bit IDE DATA register
		$FFx1 - Error (read) / Features (write) register
		$FFx2 - Sector count register
		$FFx3 - Sector # register
		$FFx4 - Low byte of cylinder #
		$FFx5 - Hi byte of cylinder #
		$FFx6 - Device/head register
		$FFx7 - Status (read) / Command (write) register
		$FFx8 - Latch containing the 2nd 8 bits of the 16 bit IDE DATA reg.

		All of these directly map to the IDE standard equivalents. No drivers
		use	the IRQ's... they all use Polled I/O.

  ORCH-90 STEREO SOUND PACK - This is simply 2 8 bit DACs, with the left and
  right channels at $ff7a & $ff7b respectively (info courtesy: LCB).

  6309 PROCESSOR UPGRADE - The Motorola 6809 is ripped out and a Hitachi 6309
  is put in its place.  Note that there are many undocumented 6309 opcodes, and
  software that uses the 6309 uses these undocumented features.

  TWO MEGABYTE UPGRADE - CoCo's could be upgraded to have two megabytes of RAM.
  This worked by doing the following things (info courtesy: LCB):

		1,  All MMU registers are extended to 8 bits.  Note that even if they
			store eight bits of data, on reading they are only 6 bits.
		2.	The low two bits of $FF9B now extend the video start register.  It
			is worth noting that these two bits are write-only, and if the
			video is at the end of a 512k bank, it wraps around inside the
			current 512k bank.

  EIGHT MEGABYTE UPGRADE - This upgrade extends $FF9B so that video and the MMU
  support 8 MB of memory.  I am not sure about the details.

***************************************************************************/


