                        
/*
**  Handling Of IO Addresses $FF00 To $FFFF.
**
**  Copyright 1993-2000 by Paul D. Burgin. All rights reserved.
*/

#include "dos.h"
#include "bios.h"
#include "time.h"
#include "stdio.h"
#include "conio.h"

#include "build.h"
#include "types.h"
#include "extern.h"
#include "6809cpu.h"
#include "6809regs.h"
#include "macros.h"
#include "doscart.h" /* SWO - September 1998 */ /* DOSEMU */

#include "lfn.h"

#define PRINTER_SCAN	10	/* Max status poll rate in seconds. */
#define PRINTER_WAIT	20	/* Max time for printer to be ready. */
#define PRINTER_INIT	5	/* Min wait after initialisation. */
#define SERIAL_TOUT		10	/* Max time between serial bits of tandy. */

enum port		lpt;
boolean			crlf				= FALSE;
unsigned int	comspeed			= _COM_2400;
boolean			printfile			= FALSE;
FILE			*pf					= NULL;

boolean 		printer_init		= FALSE;
time_t			printer_poll		= 0;
int				printer_status		= 0;
boolean 		printsimple			= TRUE;

unsigned char	tandybits, tandycount;
time_t			tandycall		= 0;

unsigned char key_scan_counter 		= KEY_SCAN_RESOLUTION;
unsigned char hardjoy_counter  		= HARDJOY_RESOLUTION;
unsigned char hardjoy_resolution  	= HARDJOY_RESOLUTION;

	/*********************************************************************/
	/*                                                                   */
	/* Most of the addresses in the range $FF00-$FFFF are not accessible */
	/* by the Dragon and therefore some are used internally by the       */
	/* emulator.                                                         */
	/*                                                                   */
	/* This gives PCD an internal memory map which looks like this:      */
	/*                                                                   */
	/*	$FF00       PIA0A Peripheral Data Register                       */
	/*	$FF01       PIA0A Control Register                               */
	/*	$FF02       PIA0B Peripheral Data Register						 */
	/*	$FF03       PIA0B Control Register                               */
	/*	$FF08       PIA0A Data Direction Register                        */
	/*	$FF0A       PIA0B Data Direction Register                        */
	/*	$FF1F       Mask For Joystick Buttons                            */
	/*	$FF20       PIA1A Peripheral Data Register                       */
	/*	$FF21       PIA1A Control Register                               */
	/*	$FF22       PIA1B Peripheral Data Register						 */
	/*	$FF23       PIA1B Control Register                               */
	/*	$FF28       PIA1A Data Direction Register                        */
	/*	$FF2A       PIA1B Data Direction Register                        */
	/*	$FF2B-$FF2E Current Joystick Positions                           */
	/*	$FF30-$FF38 Keyboard Rollover Table (Real and Pseudo-Real Modes) */
	/*	$FFBE-$FFBF Used For Immediate Addressing                        */
	/*	$FFC0-$FFDF SAM Registers (but the data is not stored here)      */
	/*	$FFE0-$FFEF ROM Vectors, or used as WAM/RAM by Extended Services */
	/*	$FFF0-$FFFF ROM Vectors                                          */
	/*                                                                   */
	/* Some of these addresses are defined as macros, but are best left  */
	/* unchanged. All addresses $FF40-$FFDF may be written to by the     */
	/* emulated Dragon, but return a constant value when read from (with */
	/* the exception of $FFBE-$FFBF, which acts as rapidly changing RAM).*/
	/* The PIAs repeat 4 times each in the Dragon 64, and 8 times in the */
	/* D32 or Tandy CoCo; because of this the internal values in the     */
	/* range $FF00-$FF3F are unreachable by the Dragon. The ROM vectors  */
	/* do not exist in the positions shown; they are repeated from the   */
	/* ROM addresses $BFE0-$BFFF.                                        */
	/*                                                                   */
	/*********************************************************************/

/* Change printfile status (assumed to be paired with change_lpt call). */
void change_printfile(boolean now_to_file)
{
	printfile		= now_to_file;
	printer_init	= FALSE;
}

/* Attempt to change printer device. */
boolean change_lpt(enum port new_port)
{
	if (new_port == lpt)
		return TRUE;

	switch (new_port)
	{
		case COM1:	if (numserial >= 1)
						lpt = COM1;
					else
						return FALSE;
					break;

		case COM2:	if (numserial >= 2)
						lpt = COM2;
					else
						return FALSE;
					break;

		case LPT1:	if (numparallel >= 1)
						lpt = LPT1;
					else
						return FALSE;
					break;

		case LPT2:	if (numparallel >= 2)
						lpt = LPT2;
					else
						return FALSE;
					break;

		default:	lpt = NO_PORTS;
					break;
	}

	printer_init = FALSE;
	return TRUE;
}

/* Change serial baud rate. */
void set_comspeed(unsigned int new_comspeed)
{
	if (comspeed != new_comspeed)
	{
		comspeed = new_comspeed;
		if ((!printfile) && ((lpt == COM1) || (lpt == COM2)))
			printer_init = FALSE;
	}
}

/* Send a character to configured printer device. */
void print_char(unsigned char ch)
{
	if (printfile)
	{
		if (pf != NULL)
			fputc(ch,pf);
	}
	else
	{
		if (lpt < COM1)
			biosprint(0,ch,lpt);
		else if (lpt != NO_PORTS)
			bioscom(1,ch,lpt-2);
	}
}

/* Initialise printing device. */
void print_init(void)
{
	if (lpt < COM1)
		biosprint(1,1,lpt);
	else if (lpt != NO_PORTS)
		bioscom(0,_COM_CHR8 | _COM_STOP1 | _COM_NOPARITY | comspeed, lpt-2);
}

/* Check availability of printing device. */
boolean printer_okay(void)
{
	time_t	t				= time(NULL);

	if ((t - printer_poll) > PRINTER_SCAN)
	{
		printer_poll = t;
		if ((lpt < COM1) && !printfile)
			printer_status = biosprint(2,1,lpt);
		else
			printer_status = 0x90; /* always okay */
	}
	return (((printer_status & 0x10) > 0) && ((printer_status & 0x80) > 0));
}

/* Read from IO address. */
unsigned char get_sampias(unsigned int address)
{
	/* PIA addresses. */
	if (address < 0xff40)
	{
		/* Adjust address for PIA duplication. */
		if (arch == dragon64)
			address &= 0xff27;
		else
			address &= 0xff23;

		/* Select DDR or PDR. */
		switch (address)
		{
		case 0xff00:	if ((memory[0xff01] & 0x04) == 0)
							address = 0xff08;
						break;

		case 0xff02:	if ((memory[0xff03] & 0x04) == 0)
							address = 0xff0a;
						break;

		case 0xff20:	if ((memory[0xff21] & 0x04) == 0)
							address = 0xff28;
						break;

		case 0xff22:	if ((memory[0xff23] & 0x04) == 0)
							address = 0xff2a;
						break;
		}

		/* Action PIA address. */
		switch (address)
		{

		case 0xff00:	/* Keyboard/joystick button read. */
						memory[0xff00] = 0xff;

						/* Need to poll keyboard at all? */
						if (memory[0xff02] != 0xff)
						{
							if (new_int9_set)
							{
								/* Scan keyboard - build return value. */
								create_matrix();
								if ((memory[0xff02] & 0x01) == 0)
									 memory[0xff00] &= memory[MATRIX+1];
								if ((memory[0xff02] & 0x02) == 0)
									 memory[0xff00] &= memory[MATRIX+2];
								if ((memory[0xff02] & 0x04) == 0)
									 memory[0xff00] &= memory[MATRIX+3];
								if ((memory[0xff02] & 0x08) == 0)
									 memory[0xff00] &= memory[MATRIX+4];
								if ((memory[0xff02] & 0x10) == 0)
									 memory[0xff00] &= memory[MATRIX+5];
								if ((memory[0xff02] & 0x20) == 0)
									 memory[0xff00] &= memory[MATRIX+6];
								if ((memory[0xff02] & 0x40) == 0)
									 memory[0xff00] &= memory[MATRIX+7];
								if ((memory[0xff02] & 0x80) == 0)
									 memory[0xff00] &= memory[MATRIX+8];
							}
							else
							{
								/* Pseudo-hardware keyboard. */
								key_scan_reset = 0;

								/* Time to deliver new value yet? */
								if (++key_scan_counter >
									(KEY_SCAN_RESOLUTION + KEY_SCAN_DEADBAND))
								{
									key_scan_counter = 0;
									old_scan_keys();
								}

								/* Also scan if previously no */
								/* keys pressed.              */
								else if (memory[MATRIX] == 0xff)
								{
									/* Any key waiting? */
									if (iskey())
									{
										key_scan_counter = 0;
										old_scan_keys();
									}
									else
									{
										/* Shift keys changed? */
										if (((bioskey(2) & 0x43) != 0)
											!= last_shift)
										{
											key_scan_counter = 0;
											old_scan_keys();
										}
									}
								}

								/* Build return value if keys scanned. */
								if (key_scan_counter < KEY_SCAN_RESOLUTION)
								{
									if ((memory[0xff02] & 0x01) == 0)
										 memory[0xff00] &= memory[MATRIX+1];
									if ((memory[0xff02] & 0x02) == 0)
										 memory[0xff00] &= memory[MATRIX+2];
									if ((memory[0xff02] & 0x04) == 0)
										 memory[0xff00] &= memory[MATRIX+3];
									if ((memory[0xff02] & 0x08) == 0)
										 memory[0xff00] &= memory[MATRIX+4];
									if ((memory[0xff02] & 0x10) == 0)
										 memory[0xff00] &= memory[MATRIX+5];
									if ((memory[0xff02] & 0x20) == 0)
										 memory[0xff00] &= memory[MATRIX+6];
									if ((memory[0xff02] & 0x40) == 0)
										 memory[0xff00] &= memory[MATRIX+7];
									if ((memory[0xff02] & 0x80) == 0)
										 memory[0xff00] &= memory[MATRIX+8];
								}
							}
						}

						/* Time to update button yet? */
						if (button_counter >= BUTTON_RESOLUTION)
						{
							button_counter = 0;
							update_fire_buttons();
						}

						/* Combine keyboard and joystick button data. */
						memory[0xff00] &= memory[JOY_BUTTON];

						/* Time to update joystick positions yet? */
						if (hardjoy_counter >= hardjoy_resolution)
						{
							hardjoy_counter = 0;
							update_joysticks(TEMPJOY);
						}
						else
							hardjoy_counter++;

						/* Update joystick comparator. */
						if ((memory[0xff01] & 0x08) == 0)
						{
							if ((memory[0xff20] >> 2) > joy_x)
								memory[0xff00] &= 0x7f;
						}
						else
						{
							if ((memory[0xff20] >> 2) > joy_y)
								memory[0xff00] &= 0x7f;
						}
						break;

		case 0xff22:    if ((memory[0xff2a] & 0x01) == 0)
							memory[0xff22] = (memory[0xff22] & 0xfe) | (!printer_okay());
						break;

		}
		return(memory[address]);
	}

	/* intercept DOS hardware reads - SWO September 1998
		($ff40-$ff43 & $ff48) */ /* DOSEMU */
	else if ((address <= 0xff48) && (arch != tandy))
	{
		switch(address)
		{
			case 0xff40 :	return (dos_cart_read_status());
			case 0xff43 : 	return (dos_cart_read_ff43());

			default 	:	return (memory[address]);
		}
	}

	/* ROM vectors and emulator services address space. */
	else if (address >= 0xffe0)
	{
		/* Re-map interrupt vectors into top of ROM. */
		if ((address >= 0xfff0)
				|| (memory[0xffee] != 'P') || (memory[0xffef] != 'C'))
			return(memory[address - 0x4000]);

		/* Execute PC or DOS services. */
		else if (address == 0xffee)
			return(emu_service(TRUE));
		else if (address == 0xffef)
			return(emu_service(FALSE));

		/* PC service RAM 0xffe0-0xffef */
		else
			return(memory[address]);
	}

	/* Return zero at 0xff42 to allow SuperDos to boot. */
	/* Removed - SWO */ /* DOSEMU */
#if 0
	else if (address == 0xff42)
		return(0x00);
#endif

	/* Unused IO space. */
	return(UNMAPPED);
}

/* Write to IO address. */
void set_sampias(unsigned char value, unsigned int address)
{
	boolean pdr = FALSE;

	/* Adjust address for PIA duplication. */
	if (address < 0xff40)
	{
		if (arch == dragon64)
			address &= 0xff27;
		else
			address &= 0xff23;

		/* Select DDR or PDR. */
		switch (address)
		{
		case 0xff00:	if ((memory[0xff01] & 0x04) == 0)
							address = 0xff08;
						else
							pdr = TRUE;
						break;
		case 0xff02:	if ((memory[0xff03] & 0x04) == 0)
							address = 0xff0a;
						else
							pdr = TRUE;
						break;
		case 0xff20:	if ((memory[0xff21] & 0x04) == 0)
							address = 0xff28;
						else
							pdr = TRUE;
						break;
		case 0xff22:	if ((memory[0xff23] & 0x04) == 0)
							address = 0xff2a;
						else
							pdr = TRUE;
						break;
		}
	}

	/* Action IO addresses. */
	switch (address)
	{
			/* Strobe printer. */
			case 0xff20:	if (((memory[0xff28] & 0x02) > 0))
							{
								if (arch == tandy)
								{
									time_t t = time(NULL);

									if ((t - tandycall) > SERIAL_TOUT)
										tandycount = 0;
									tandycall = t;
									if ((tandycount == 0) || (tandycount == 10))
									{
										if ((value & 0x02) > 0)
											++tandycount;
									}
									else
									{
										tandycount++;
										tandybits = (tandybits >> 1) | ((value & 0x02) << 6);
									}
								}
								else
									tandybits = memory[0xff02];
								if (((arch != tandy)
									&& (((memory[0xff20] & 0x02) == 0)
									&& ((value & 0x02) != 0)))
									|| ((arch == tandy) && (tandycount == 11)))
								{
									time_t begin_t = time(NULL);
									printer_poll = tandycount = 0;
									if (!printer_init)
									{
										printer_init = TRUE;
										if (printfile)
										{
											if (pf == NULL)
												pf = lfn_fopen("printer.out","wb");
										}
										else if (!printsimple)
										{
											print_init();
											while ((time(NULL) - begin_t) < PRINTER_INIT);
										}
									}
									if (!printsimple)
									{
										while (!printer_okay()
										&& ((time(NULL) - begin_t) < PRINTER_WAIT));
									}
									if (printer_okay())
									{
										print_char(tandybits);
										if ((crlf) && (tandybits == 13))
											print_char(10);
									}
								}
							}
							break;

			/* Change video mode. */
			case 0xff22:	set_vmode(value);
							break;

			/* DOS hardware writes - SWO September 1998 */ /* DOSEMU++ */
			case 0xff40:    if (arch != tandy)
								dos_cart_command(value);
							break;

			case 0xff41:
			case 0xff42:    break;

			case 0xff43:    if (arch != tandy)
								dos_cart_write_ff43(value);
							break;

			case 0xff48:    if (arch != tandy)
								dos_cart_write_ff48(value);
							break;
			/* DOS hardware writes - SWO September 1998 */ /* DOSEMU-- */

			/* Change video memory size. */
			case 0xffc0:    new_screen_size &= 0x06;
							refresh_count = 0;
							break;

			case 0xffc1:	new_screen_size |= 0x01;
							refresh_count = 0;
							break;

			case 0xffc2:	new_screen_size &= 0x05;
							refresh_count = 0;
							break;

			case 0xffc3:	new_screen_size |= 0x02;
							refresh_count = 0;
							break;

			case 0xffc4:	new_screen_size &= 0x03;
							refresh_count = 0;
							break;

			case 0xffc5:	new_screen_size |= 0x04;
							refresh_count = 0;
							break;

			/* Change video base address. */
			case 0xffc6:    new_screen_base &= 0xfdff;
							refresh_count = 0;
							break;

			case 0xffc7:	new_screen_base |= 0x0200;
							refresh_count = 0;
							break;

			case 0xffc8:	new_screen_base &= 0xfbff;
							refresh_count = 0;
							break;

			case 0xffc9:	new_screen_base |= 0x0400;
							refresh_count = 0;
							break;

			case 0xffca:	new_screen_base &= 0xf7ff;
							refresh_count = 0;
							break;

			case 0xffcb:	new_screen_base |= 0x0800;
							refresh_count = 0;
							break;

			case 0xffcc:	new_screen_base &= 0xefff;
							refresh_count = 0;
							break;

			case 0xffcd:    new_screen_base |= 0x1000;
							refresh_count = 0;
							break;

			case 0xffce:	new_screen_base &= 0xdfff;
							refresh_count = 0;
							break;

			case 0xffcf:	new_screen_base |= 0x2000;
							refresh_count = 0;
							break;

			case 0xffd0:	new_screen_base &= 0xbfff;
							refresh_count = 0;
							break;

			case 0xffd1:	new_screen_base |= 0x4000;
							refresh_count = 0;
							break;

			case 0xffd2:	new_screen_base &= 0x7fff;
							refresh_count = 0;
							break;

			case 0xffd3:	new_screen_base |= 0x8000;
							refresh_count = 0;
							break;

			/* Change between map modes. */
			case 0xffde:	mapmode1 = FALSE;
							break;

			case 0xffdf:	if (arch != dragon32)
								mapmode1 = TRUE;
							break;
	}

	/* Store value (or part of it) at specified address. */
	if (address <= MEMORY_SIZE)
	{
		if (pdr)
		{
			memory[address] &= ~memory[address+8];
			memory[address] |= (value & memory[address+8]);
		}
		else
			memory[address] = value;
	}
}
