                                                  
/*
**  Partial Emulation Of Dragon DOS Cartridge Hardware.
**
**  Copyright 1998-2000 by SWO & PDB. All rights reserved.
**  Initially created September 1998 by Stewart Orchard.
**
**  Maintains virtual disk files for drives 1 to 4.
**
**  Note that the real DOS hardware cannot manipulate the CART flag
**  in the way implemented here. It does, however, simplify programming
**  by taking FIRQ SYNCs out of the equation. This may cause incompatibility
**  with other operating systems.
**
**  Assumes that logical and physical sector & track numbers are identical.
**
**  To improve response of DOS, invalid sector addresses cause an immediate
**  ?RF ERROR. Normally the system would timeout & eventually return a ?NR
**  ERROR or similar.
**
**  Tested OK with DOS V1.0:
**  	Nasty bugs manifest themselves when copying from disk to disk. I'm
**  	fairly confident that these bugs are in the DOS rather than in the
**  	disk emulation code.
**
**  Tested OK with DOS E6:
**  	Much more robust!
**
**  Superficially tested with OS9:
**  	Seems to be OK.
*/
                                     
#include "dos.h"
#include "dir.h"
#include "stdio.h"
#include "conio.h"
#include "string.h"
#include "io.h"
#include "graphics.h"

#include "build.h"
#include "types.h"
#include "extern.h"
#include "6809cpu.h"
#include "6809regs.h"
#include "macros.h"

#include "lfn.h"

/* Macros for hardware registers. */
#define COMREG memory[0xff40]
#define TRKREG memory[0xff41]
#define SECREG memory[0xff42]
#define DATREG memory[0xff43]
#define CTLREG memory[0xff48]

/* Macros for CART flag manipulation. */
#define SETCART memory[0xff23] |= 0x80
#define CLRCART memory[0xff23] &= 0x7f

/* Bit meanings in 'dos_cart_status'. */
typedef enum
{
	DCS_OK		= 0x00,
	DCS_BUSY	= 0x01, 	/* operation in progress */
	DCS_TRK0 	= 0x04, 	/* head is over track zero */
	DCS_RF		= 0x10, 	/* sector address record not found */
	DCS_WP		= 0x40  	/* disk is write protected */
} STAT_TYPE;

/* Possible values of 'operation'. */
typedef enum
{
	DC_IDLE,
	DC_READ,	/* sector read in progress */
	DC_WRITE,   /* sector write in progress */
	DC_FORMAT,  /* format track in progress */
	DC_SEEK     /* seek in progress */
} OP_TYPE;

/* Used to tell CPU engine in 'DRAGON.C' to cause NMI. */
extern boolean nmi_flag;

/* Status for each drive. */
typedef enum
{
	dsk_closed,
	dsk_ready,
	dsk_open,
	dsk_failed
} VDRV_STATUS;

/* Structure which holds the virtual disk information for each drive. */
typedef struct
{
	FILE *fp;					/* virtual disk file */
	char fname[LFN_MAXPATH];    /* filename entered by user */
	char realpath[MAXPATH];		/* short real full path name */
	VDRV_STATUS status;			/* closed/ready/open/failed status */
	int tracks;     			/* number of tracks (40 or 80) */
	int sides;      			/* number of sides (1 or 2) */
	unsigned int track;			/* current track */
	unsigned char wp;			/* 0=off, 1=flagged on, 2=attrib on */
	unsigned long offset_h;		/* offset to start of header */
	unsigned long offset_d;		/* offset to start of data */
} VDRV_TYPE;

/* Structure for virtual disk header. */
typedef struct
{
	/* v1.0 */
	unsigned char  id1;			/* signature byte 1 */
	unsigned char  id2;			/* signature byte 2 */
	unsigned short header_len;	/* total header length (offset to data) */
	unsigned char  ver_actual;	/* version of VDK format */
	unsigned char  ver_compat;	/* backwards compatibility version */
	unsigned char  source_id;	/* identity of file source */
	unsigned char  source_ver;	/* version of file source */
	unsigned char  tracks;		/* number of tracks (40 or 80) */
	unsigned char  sides;		/* number of sides (1 or 2) */
	unsigned char  flags;		/* various flags */
	unsigned char  compression;	/* compression flags and name length */
} VDK_HEADER;

/* Virtual disk header defines. */
#define VDK_ID1				'd'
#define VDK_ID2				'k'
#define VDK_VEROUT			0x10
#define VDK_VERIN			0x10
#define VDK_SRCIDOUT		('P')
#define VDK_SRCVEROUT		((VER_MAJOR << 4) | (VER_MINOR))
#define VDKFLAGS_WP			0x01
#define VDKFLAGS_ALOCK		0x02
#define VDKFLAGS_FLOCK		0x04
#define VDKFLAGS_DISKSET	0x08
#define VDK_COMPBITS		3
#define VDK_COMPMASK		0x07
#define VDK_COMPOUT			0x00
#define VDK_MAXNAME			31

/* Drive light colours. */
#define DL_CLOSED	((BLACK<<4)	+ DARKGRAY)
#define DL_ALERT	((DARKGRAY<<4) + GREEN)
#define DL_RO		((BLACK<<4) + RED)
#define DL_RW		((BLACK<<4) + GREEN)
#define DL_READY	DL_CLOSED /* ((BLACK<<4) + LIGHTGRAY) */
#define DL_FAILED	DL_CLOSED /* ((DARKGRAY<<4) + DARKGRAY) */

/* State of the drive lights. */
int drive_light[4] = {DL_CLOSED, DL_CLOSED, DL_CLOSED, DL_CLOSED};

/* Initialise drive information. */
VDRV_TYPE vdrive[4] =
	{NULL, "blank1", "", dsk_closed, 0, 0, 0, FALSE, 0, 0,
	 NULL, "blank2", "", dsk_closed, 0, 0, 0, FALSE, 0, 0,
	 NULL, "blank3", "", dsk_closed, 0, 0, 0, FALSE, 0, 0,
	 NULL, "blank4", "", dsk_closed, 0, 0, 0, FALSE, 0, 0};

/* Pointer to current drive details. */
VDRV_TYPE *vdrv = vdrive;

/* Disk head obtained from command byte. */
int side;

/* Keeps track of number of bytes transferred during sector operations. */
int byte_count;

/* Holds hardware status flags. */
STAT_TYPE dos_cart_status = DCS_OK;

/* Current internal operation. */
OP_TYPE operation = DC_IDLE;

/* Function prototypes. */
void dos_cart_nmi(STAT_TYPE ret_stat);
void dos_cart_reset(void);
void init_sector_xfer(OP_TYPE xfer_type);
void seek(unsigned int track);
void open_disk_file(int drive);

/* Handle write to command register $ff40. */
void dos_cart_command(char byte)
{
	static char errbuf[16];

	side = (byte & 0x02) >> 1;  /* Disk head is encoded in command byte. */

	/* Action the command byte. */
	switch (byte & 0xfc)
	{
					/* Seek track zero. */
		case 0x00 : seek(TRKREG = 0);
					break;

					/* Seek track number in $ff43. */
		case 0x10 :	seek(TRKREG = DATREG);
					break;

					/* Step track in +ve direction (no update of $ff41). */
		case 0x40 : seek(vdrv->track + 1);
					break;

					/* Dragon DOS sector read. */
		case 0x88 : init_sector_xfer(DC_READ);
					break;

					/* Dragon DOS sector write. */
		case 0xa8 : if (vdrv->wp)
						dos_cart_nmi(DCS_WP);   /* Write-protected. */
					else
						init_sector_xfer(DC_WRITE);
					break;

		/* In case of 'format track' command, do nothing other than to
		   signal operation complete on next write to $ff43. */
		case 0xf0 :
		case 0xf4 : if (vdrv->wp)
						dos_cart_nmi(DCS_WP);   /* Write-protected. */
					else
					{
						SETCART;
						dos_cart_status = DCS_OK;
						operation = DC_FORMAT;
					}
					break;

					/* Reset hardware status. */
		case 0xd0 :	dos_cart_reset();
					break;

/*
		Not all valid functions are supported:

		$2x - step +ve if not track zero, no update of track register.
		$3x - step +ve if not track zero.
		$5x - step +ve
		$6x - step -ve, no update of track register.
		$7x - step -ve

		seek subvariants:
			$x4 - check track number after seek.
			$x8 - flagged interrupt.

		$80 - read long sector.

		$90 - read multiple long sectors.
		$98 - read multiple sectors.

		$a0 - write long sector?

		$b0 - write multiple long sectors?
		$b8 - write multiple sectors?

		$cx - read next sector address record.

		$dx subvariants:
			$d4 - index pulse interrupt enable.
			$d8 - interrupt on $ff40 read?

		$ex - raw read track? (complement of format track?)
*/

		default   :	sprintf(errbuf, "$FF40 = $%2X", byte);
					selection_box(3,22,0,FALSE,FALSE,FALSE,99,
					"DOS hardware command",
					"not supported:",
					errbuf);
	}
}

/* Seek track. */
void seek(unsigned int track)
{
	operation = DC_SEEK;
	dos_cart_status = DCS_OK;

	if (vdrv->status == dsk_open)
	{
		vdrv->track = track;

		if (vdrv->wp)
			dos_cart_status = DCS_WP;

		if (!track)
			dos_cart_status |= DCS_TRK0;
	}

	dos_cart_nmi(dos_cart_status);
}

/* Set up for sector transfer. */
void init_sector_xfer(OP_TYPE xfer_type)
{
	unsigned long	filepos;
	unsigned int	sector;

	operation = xfer_type;
	dos_cart_status = DCS_OK;

	byte_count = 256;
	sector = SECREG - 1;

	/* Decide whether we are going to find the requested sector. */
	if ((side < vdrv->sides)
		 && (vdrv->track < vdrv->tracks)
		 && (vdrv->track == TRKREG)
		 && (sector < 18)
		 && (vdrv->status == dsk_open))
	{
		filepos = (vdrv->track * vdrv->sides + side) * 18 + sector; /* LSN */
		fseek(vdrv->fp, (filepos<<8)+vdrv->offset_d, SEEK_SET);
		SETCART;
	}
	else
	{
		/* Cause immediate error rather than allow time out. */
		dos_cart_nmi(DCS_RF);
	}
}

/*
	Produce a byte for data register.
	Triggers NMI when sector operation complete.
*/
char dos_cart_read_ff43(void)
{
	if (operation == DC_READ)
	{
		if (byte_count--)
		{
			SETCART;
			return (fgetc(vdrv->fp));
		}
		else
		{
			dos_cart_nmi(DCS_OK);
			return (0xff);     /* Dummy return value. */
		}
	}
	/* If no operation in progress then register behaves like memory. */
	else return (DATREG);
}

/*
	Deal with byte written to data register.
	Triggers NMI when sector operation complete.
*/
void dos_cart_write_ff43(char byte)
{
	switch (operation)
	{
		case DC_WRITE:	SETCART;
						fputc(byte, vdrv->fp);
						if (!--byte_count)
							dos_cart_nmi(DCS_OK);
						break;

		case DC_FORMAT:	dos_cart_nmi(DCS_OK);
						break;
	}
}

/* Trigger NMI if enabled. */
void dos_cart_nmi(STAT_TYPE ret_stat)
{
	if (CTLREG & 0x20)
	{
		nmi_flag = TRUE;
		operation = DC_IDLE;
	}
	/* Keep SEEK condition if NMI disabled (for OS9 - see below). */
	else if (operation != DC_SEEK)
		operation = DC_IDLE;

	dos_cart_status = ret_stat;

	CLRCART;
}

/* Handle read from status register at $ff40. */
char dos_cart_read_status(void)
{
	/*
		OS9 compatibility - if NMI was disabled for a seek operation
		then pretend the hardware is busy for one read.
	*/
	if (operation == DC_SEEK)
	{
		operation = DC_IDLE;
		return (dos_cart_status | DCS_BUSY);
	}

	return (dos_cart_status);
}

/*
	Reset disk controller registers (power up & hard reset).

	What actually happens is that the controller does a
	'seek track zero' from track 255 which normally takes
	a few seconds. The emulator approach reflects the
	register state when DOS E6 soft-resets the cartridge.
*/
void initialise_doscart_regs(void)
{
	unsigned int hw_address;

	/* Set all DOS related registers to zero. */
	for (hw_address = 0xff40; hw_address < 0xff4f; hw_address++)
		memory[hw_address] = 0;

	nmi_flag = FALSE;
	CLRCART;
	operation = DC_SEEK;

	/* Status is actually 'busy'; the OS9 mod above will look after this. */
	dos_cart_status = DCS_OK;

	TRKREG = 0xfe;
	SECREG = 0x01;
	DATREG = 0x00;
}

/* Reset disk controller status. */
void dos_cart_reset(void)
{
	nmi_flag = FALSE;
	CLRCART;
	operation = DC_IDLE;
	dos_cart_status = DCS_OK;

	/* If disk is active, update track zero & write protect bits. */
	if (vdrv->status == dsk_open)
	{
		if (vdrv->wp)
			dos_cart_status = DCS_WP;

		if (!vdrv->track)
			dos_cart_status |= DCS_TRK0;
	}
}

/* Display disk drive activation lights. */
void update_drive_lights(void)
{
	struct viewporttype oldview;
	int i, x, color;

	if (vmode < 0)
	{
		/* Draw drive lights on text display. */
		window(1,1,8,2);
		gotoxy(1,1);
		textattr(drive_light[0]); cprintf("1");
		textattr(drive_light[1]); cprintf("2");
		textattr(drive_light[2]); cprintf("3");
		textattr(drive_light[3]); cprintf("4");
		new_window(WIN_DRAGON);
	}
	else
	{
		/* Draw drive lights on graphics display. */
		getviewsettings(&oldview);
		setviewport(4,2,63,7, 1);

		/* Black for shadow effect. */
		setpalette(9, 0);

		for (i=0, x=0; i<4; i++, x+=13)
		{
			/* Draw shadow effect. */
			setfillstyle(SOLID_FILL, 9);
			bar(x+1, 1, x+11, 3);

			/*
				Obtain colour the lazy way.
				Assumes that palette colours 8+ are untouched
				(except for 9 which is now black for shadow).
			*/
			color = (drive_light[i] & 0x0f) | 8;

			/* Draw light. */
			setfillstyle(SOLID_FILL, color);
			bar(x, 0, x+10, 2);
		}

		/* Restore original viewport. */
		setviewport(oldview.left, oldview.top,
					oldview.right, oldview.bottom, oldview.clip);
	}
}

/*
	Use writes to $ff48 to control virtual disk files.
	File is opened when a drive is accessed & all files
	are closed when the disk motors are turned off.
*/
void dos_cart_write_ff48(char byte)
{
	int drive, i;

	/* Return if nothing of interest happening. */
	if ((byte & 0x07) == (CTLREG & 0x07))
		return;

	drive = byte & 0x03;
	vdrv = &vdrive[drive];

	/* Disk motors activated? */
	if ((byte & 0x04))
	{
		/* Switch off existing drive light. */
		for (i=0; i<4; i++)
		{
			if ((drive_light[i] == DL_RO) || (drive_light[i] == DL_RW))
				drive_light[i] = DL_READY;
		}

		/* Open a file if necessary. */
		if ((vdrv->status != dsk_open) && (vdrv->status != dsk_failed))
		{
			drive_light[drive] = DL_ALERT;
			update_drive_lights();
			open_disk_file(drive);
		}

		/* Change write protected disks to red and writeable disks to green. */
		if (vdrv->status == dsk_open)
		{
			if (vdrv->wp)
				drive_light[drive] = DL_RO;
			else
				drive_light[drive] = DL_RW;
		}
		else
			drive_light[drive] = DL_FAILED;
	}
	else
	{
		/* Close all files when disk motors turned off. */
		for (i=0; i<4; i++)
		{
			if (vdrive[i].status == dsk_open)
			{
				fclose(vdrive[i].fp);
				vdrive[i].status = dsk_ready;
				drive_light[i] = DL_READY;
			}
		}
	}
	update_drive_lights();
}

/* Read virtual disk header and check for validity. */
boolean examine_header(VDK_HEADER *hdr, FILE *hfp)
{
	if (fread(hdr, sizeof(VDK_HEADER), 1, hfp) < 1)
	{
		selection_box(4,23,0,FALSE,FALSE,FALSE,99,
			"      FATAL ERROR",
			"      ===========","",
			"Failure reading header!");
		return(FALSE);
	}

	/* Check for valid id bytes. */
	if ((hdr->id1 != VDK_ID1) || (hdr->id2 != VDK_ID2))
	{
		selection_box(4,21,0,FALSE,FALSE,FALSE,99,
			"     FATAL ERROR",
			"     ===========","",
			"Invalid virtual disk!");
		return(FALSE);
	}

	/* Check version of format. */
	if (hdr->ver_compat != VDK_VERIN)
	{
		selection_box(5,20,0,FALSE,FALSE,FALSE,99,
			"     FATAL ERROR",
			"     ===========","",
			"Virtual disk version",
			"   not supported!");
		return(FALSE);
	}

	/* Check geometry of disk - sides. */
	if ((hdr->sides < 1) || (hdr->sides > 2))
	{
		selection_box(5,14,0,FALSE,FALSE,FALSE,99,
			" FATAL  ERROR",
			" ============","",
			"Invalid number",
			"of disk sides.");
		return(FALSE);
	}

	/* Check geometry of disk - tracks. */
	if ((hdr->tracks != 40) && (hdr->tracks != 80))
	{
		selection_box(5,14,0,FALSE,FALSE,FALSE,99,
			" FATAL  ERROR",
			" ============","",
			"Invalid number",
			"of VDK tracks.");
		return(FALSE);
	}

	/* Check for compressed data - not currently supported. */
	if (hdr->compression & VDK_COMPMASK)
	{
		selection_box(5,20,0,FALSE,FALSE,FALSE,99,
			"     FATAL ERROR",
			"     ===========","",
			" Compressed virtual",
			"disks not supported!");
		return(FALSE);
	}

	/* Don't use a disk which has the forced lock turned on. */
	if (hdr->flags & VDKFLAGS_FLOCK)
	{
		selection_box(4,20,0,FALSE,FALSE,FALSE,99,
			"     FATAL ERROR",
			"     ===========","",
			"File is unavailable!");
		return(FALSE);
	}

	/* Check with user before using a disk    */
	/* which has the advisary lock turned on. */
	if (hdr->flags & VDKFLAGS_ALOCK)
	{
		if (selection_box(3,20,0,TRUE,FALSE,FALSE,99,
				"File is unavailable!","",
				" Open anyway (y/n)?") != 'Y')
		{
			return(FALSE);
		}
	}

	return(TRUE);
}

/*
   Open virtual disk file.

	If necessary, get filename from user.
	First open in read only mode check flag for write-protection.
	If open fails, flag as dsk_failed (equivalent to not ready).
	Otherwise if disk appears to be writable try to reopen in update mode.
*/
void open_disk_file(int drive)
{
	FILE		*fp;
	VDK_HEADER	headr;
	boolean		happy = FALSE;
	int			i;

	/* If previously had a good filename try to reuse it. */
	if (vdrv->status == dsk_ready)
	{
		fp = lfn_fopen(vdrv->realpath, "rb");
		if (fp != NULL)
		{
			/* Seek disk header. */
			if (fseek(fp, vdrv->offset_h, SEEK_SET) == 0)
			{
				/* Read and check disk header. */
				happy = examine_header(&headr, fp);
			}

			if (!happy)
				fclose(fp);
		}

		/* Tell user that previous file is no longer available. */
		if (!happy)
			selection_box(2,20,0,FALSE,FALSE,FALSE,99,
				"Current virtual disk",
				"cannot be re-opened!");
	}

	/* Get new filename if necessary. */
	while (!happy)
	{
		happy = generic_open(
				DSKDRIVE1 + drive,".vdk",
				"  VIRTUAL DISK FILE ",&fp,
				vdrv->realpath,
				vdrv->fname,FALSE,
				vdrv->fname,FALSE);

		if (happy)
		{
			/* Convert any forward slashes in realpath to backslashes. */
			while (strchr(vdrv->realpath,'/') != NULL)
				*strchr(vdrv->realpath,'/') = '\\';

			/* Check to see if filename is already in use. */
			for (i=0; i<4; i++)
			{
				if ((i != drive) &&
					((vdrive[i].status == dsk_ready) ||
					 (vdrive[i].status == dsk_open)))
				{
					if (strcmpi(vdrv->realpath, vdrive[i].realpath) == 0)
					{
						fclose(fp);
						selection_box(1,20,0,FALSE,FALSE,FALSE,99,
							"Disk already in use!");
						happy = FALSE;
					}
				}
			}

			if (happy)
			{
				unsigned char disk_num = 1;
				unsigned char disk_nm[22];
				unsigned long filepos;

				/* Read and check disk header. */
next_disk:		vdrv->offset_h = ftell(fp);
				happy = examine_header(&headr, fp);

				/* Check for multi-disk file. */
				if (happy && (headr.flags & VDKFLAGS_DISKSET))
				{
					sprintf(disk_nm," Open disk %d (y/n)?",disk_num);
					if (selection_box(4,21,0,TRUE,FALSE,FALSE,99,
							"   VIRTUAL DISKSET",
							"   ===============","",
							disk_nm) != 'Y')
					{
						disk_num++;
						filepos = ((unsigned long)headr.tracks*(unsigned long)headr.sides*18)<<8;
						filepos += (vdrv->offset_h + (unsigned long)(headr.header_len));
						fseek(fp, filepos, SEEK_SET);
						goto next_disk;
					}
				}

				if (!happy)
					fclose(fp);
			}
		}
		else
		{
			vdrv->status = dsk_failed;
			return;
		}
	}

	/* Store virtual disk header information. */
	vdrv->offset_d = vdrv->offset_h + (unsigned long)headr.header_len;
	vdrv->tracks   = headr.tracks;
	vdrv->sides    = headr.sides;
	vdrv->wp       = headr.flags & VDKFLAGS_WP;
	vdrv->fp       = fp;
	vdrv->status   = dsk_open;

	/* Attempt to reopen file in update mode. */
	fp = lfn_fopen(vdrv->realpath, "rb+");
	if (fp == NULL)
	{
		/* Cannot write to file - flag as read only by operating system. */
		vdrv->wp = 2;
	}
	else
	{
		/* Substitute writeable file and close read-only file. */
		fclose(vdrv->fp);
		vdrv->fp = fp;
	}
}

/* Virtual disk menu (F2 key). */
void disk_menu(void)
{
	unsigned char	optbox[8][26];
	unsigned char	menukey;
	unsigned char	cur_art;
	int 			vdrv_num;
	VDK_HEADER		headr;

	if (arch == tandy)
	{
		selection_box(2,14,0,FALSE,FALSE,FALSE,99,
						"Disk emulation",
						"not available!");
	}
	else do
	{
		for (vdrv_num = 0; vdrv_num < 4; vdrv_num++)
		{
			sprintf(optbox[vdrv_num],"%d - Drive %d........",vdrv_num+1,vdrv_num+1);
			switch(vdrive[vdrv_num].status)
			{
				case dsk_closed:	strcat(optbox[vdrv_num],"Closed"); break;
				case dsk_ready:		strcat(optbox[vdrv_num],".Ready"); break;
				case dsk_open:		strcat(optbox[vdrv_num],"..Open"); break;
				case dsk_failed:	strcat(optbox[vdrv_num],"Failed"); break;
			}

			/* Display write protect status if open or ready. */
			sprintf(optbox[vdrv_num+4],"%d - Virtual Disk %d.....%s",vdrv_num+5,vdrv_num+1,
				((vdrive[vdrv_num].status == dsk_open) ||
				  (vdrive[vdrv_num].status == dsk_ready)) ?
				   (vdrive[vdrv_num].wp ? "RO" : "RW") : "??");
		}

		switch (menukey = selection_box(11,25,8,FALSE,FALSE,FALSE,99,
				"    VIRTUAL DISK MENU","",
				optbox[0],optbox[1],
				optbox[2],optbox[3],"",
				optbox[4],optbox[5],
				optbox[6],optbox[7]))
		{
			case 1:
			case 2:
			case 3:
			case 4:
					vdrv_num = menukey - 1;

					switch (vdrive[vdrv_num].status)
					{
						case dsk_closed:	beep();
											break;
						case dsk_open:
						case dsk_ready:
											if (CTLREG & 0x04)  /* Disk motors on? */
											{
												selection_box(2,22,0,FALSE,FALSE,FALSE,99,
													"Cannot change settings",
													"whilst disk(s) active!");
												break;
											}
						case dsk_failed:	vdrive[vdrv_num].status = dsk_closed;
											drive_light[vdrv_num] = DL_CLOSED;
											update_drive_lights();
											break;

					/* Shouldn't occur after motor check above. */
#if 0
						case dsk_open:		fclose(vdrive[vdrv_num].fp);
											vdrive[vdrv_num].status = dsk_ready;
											drive_light[vdrv_num] = DL_READY;
											update_drive_lights();
											break;
#endif
					}
					break;

			case 5:
			case 6:
			case 7:

#if 0
/* PDB original code++ */

			case 8:	vdrv_num = menukey - 5;
					if (vdrive[vdrv_num].status != dsk_open)
						beep();
					else
					{
						switch(vdrive[vdrv_num].wp)
						{
							case FALSE:
							case TRUE:	if (fseek(vdrive[vdrv_num].fp, vdrive[vdrv_num].offset_h, SEEK_SET) == 0)
										{
											if (fread(&headr, sizeof(VDK_HEADER), 1, vdrive[vdrv_num].fp) > 0)
											{
												headr.flags ^= VDKFLAGS_WP;
												if (fseek(vdrive[vdrv_num].fp, vdrive[vdrv_num].offset_h, SEEK_SET) == 0)
												{
													if (fwrite(&headr, sizeof(VDK_HEADER), 1, vdrive[vdrv_num].fp) > 0)
													{
														fflush(vdrive[vdrv_num].fp);
														vdrive[vdrv_num].wp = !vdrive[vdrv_num].wp;
														if (vdrive[vdrv_num].wp)
															drive_light[vdrv_num] = DL_RO;
														else
															drive_light[vdrv_num] = DL_RW;
														update_drive_lights();
													}
												}
											}
										}
										break;

							default:    selection_box(2,17,0,FALSE,FALSE,FALSE,99,
											"Disk is RO by the",
											"operating system!");
										break;
						}
					}
					break;

/* PDB original code-- */
#else
/* SWO new code++ */

			case 8:
					vdrv_num = menukey - 5;

					/* Do we have a filename? */
					if ((vdrive[vdrv_num].status != dsk_open)
						   && (vdrive[vdrv_num].status != dsk_ready))
						beep();
					else
					{
						VDRV_TYPE *saved_vdrv = vdrv;

						/* Open_disk_file() loads into *vdrv. */
						if (CTLREG & 0x04)  /* Disk motors on? */
						{
							selection_box(2,22,0,FALSE,FALSE,FALSE,99,
								"Cannot change settings",
								"whilst disk(s) active!");
							break;
						}

						vdrv = &vdrive[vdrv_num];
						/*
							This will check if file is still available.
							Will prompt for a new filename if old file
							has been nicked (and maybe load in a disk set).

							Bad plan? Perhaps a 'no_prompt' flag is required.
						*/
						open_disk_file(vdrv_num);

						if (vdrv->status == dsk_open)
						{
							switch(vdrv->wp)
							{
								case FALSE:
								case TRUE:	if (fseek(vdrv->fp, vdrv->offset_h, SEEK_SET) == 0)
											{
												if (fread(&headr, sizeof(VDK_HEADER), 1, vdrv->fp) > 0)
												{
													headr.flags ^= VDKFLAGS_WP;
													if (fseek(vdrv->fp, vdrv->offset_h, SEEK_SET) == 0)
													{
														if (fwrite(&headr, sizeof(VDK_HEADER), 1, vdrv->fp) > 0)
														{
															vdrv->wp = !vdrv->wp;
														}
													}
												}
											}
											break;

								default:    selection_box(2,17,0,FALSE,FALSE,FALSE,99,
												"Disk is RO by the",
												"operating system!");
											break;
							}
							fclose(vdrv->fp);
							vdrv->status = dsk_ready;
							vdrv = saved_vdrv;
						}
						else
						{
							/* Any special action for re-open failure. */

							/* Already set by open_disk_file(). */
							/* vdrive[vdrv_num].status = dsk_failed; */
						}
					}
					break;

/* SWO new code-- */
#endif

		}
	} while ((menukey != 0) && (menukey != 99));
}

/* Create a new virtual disk file. */
void create_disk(FILE *fptr)
{
	/* New virtual disk geometry and name. */
	unsigned char	new_tracks, new_sides = 0;
	unsigned char	new_vdk_name[VDK_MAXNAME + 1];

	/* Default header for new virtual disk. */
	VDK_HEADER		headr =
					{
						VDK_ID1,
						VDK_ID2,
						sizeof(VDK_HEADER) + (sizeof(new_vdk_name) - 1),
						VDK_VEROUT,
						VDK_VERIN,
						VDK_SRCIDOUT,
						VDK_SRCVEROUT,
						0, 0, 0,
						VDK_COMPOUT
					};

	/* Ask user what size of disk to create. */
	while (new_sides < 1)
	{
		switch(selection_box(6,25,4,FALSE,FALSE,FALSE,99,
					"    VIRTUAL DISK SIZE","",
					"1 - 40 track/single sided",
					"2 - 40 track/double sided",
					"3 - 80 track/single sided",
					"4 - 80 track/double sided"))
		{
			case 1:		new_tracks = 40; new_sides = 1; break;
			case 2:		new_tracks = 40; new_sides = 2; break;
			case 3:		new_tracks = 80; new_sides = 1; break;
			case 4:		new_tracks = 80; new_sides = 2; break;
			default:	beep(); break;
		}
	}

	/* Update disk header data. */
	headr.tracks = new_tracks;
	headr.sides  = new_sides;
	memset(new_vdk_name, '\0', sizeof(new_vdk_name));
	sprintf(new_vdk_name, "(%d track/%s sided)", new_tracks,
		(new_sides == 1) ? "single" : "double");
	headr.compression |= (strlen(new_vdk_name) << VDK_COMPBITS);

	/* Write virtual disk header and name. */
	if (fwrite(&headr, sizeof(VDK_HEADER), 1, fptr) > 0)
	{
		if (fwrite(new_vdk_name, (sizeof(new_vdk_name) - 1), 1, fptr) > 0)
		{
#if 0
			if (selection_box(1,16,0,TRUE,FALSE,FALSE,99,
					"Wipe disk (y/n)?") != 'N')
			{
#endif
				unsigned char	sectorbytes[256];
				int				sbx, sby;
				int				new_sectors, nsd, sec1, sec2;

				/* Write a full disk of zeros. */
				memset(sectorbytes,'\0',256);
				create_box(3, 23);
				cprintf("\n\r Wiping Disk ");
				textbackground(BLACK);
				sbx = wherex(); sby = wherey();
				cprintf("         ");
				gotoxy(sbx, sby);
				textattr((BLACK << 4) + WHITE);
				new_sectors = new_tracks * new_sides * 18;
				nsd = new_sectors / 9;
				for (sec1 = 0; sec1 < 9; sec1++)
				{
					for (sec2 = 0; sec2 < nsd; sec2++)
					{
						fwrite(sectorbytes, 256, 1, fptr);
					}
					putch(177);
				}
				rewind(fptr);
				remove_box();
#if 0
			}
			else
			{
				unsigned long filepos;

				/* Seek to end of disk and write a byte. */
				filepos = (unsigned long)new_tracks * (unsigned long)new_sides;
				filepos = (filepos * 18) << 8;
				filepos += (unsigned long)headr.header_len - 1L;
				if (fseek(fptr,filepos,SEEK_SET) == 0)
				{
					fputc('Z',fptr); /* Safe on DriveSpace disks? */
					rewind(fptr);
				}
			}
#endif
		}
	}
}

/* Determine whether any disks might have unwritten data. */
boolean safe_to_exit(void)
{
	int	vdrive_num;

	for (vdrive_num = 0; vdrive_num < 4; vdrive_num++)
	{
		if ((vdrive[vdrive_num].status == dsk_open)
			&& (vdrive[vdrive_num].wp == FALSE)
			&& (CTLREG & 0x04))
		{
				return FALSE;
		}
	}
	return TRUE;
}

