            
/*
**  Built-In Disassembler And Debugger.
**
**  Copyright 1993-2000 by Paul D. Burgin. All rights reserved.
*/

#include "stdio.h"
#include "conio.h"
#include "string.h"
#include "dos.h"
#include "time.h"
#include "ctype.h"
#include "key.h"

#include "build.h"
#include "types.h"
#include "extern.h"
#include "6809cpu.h"
#include "6809regs.h"
#include "macros.h"

extern clock_t this_time;

/* Variables used by the debugger. */
enum winuse			left_window = MINI_DRAGON;
unsigned int		temp_breakpoint = 0x4e21, intern_break,
					breakpoint[6] =
						{ NOBREAKPOINT, NOBREAKPOINT, NOBREAKPOINT,
							NOBREAKPOINT, NOBREAKPOINT, NOBREAKPOINT };
unsigned int		break_op = 0x12;
unsigned int		diss_pcr = 0x8000;
unsigned int		breakvalue = 0x0000;
unsigned int		dump_address = 0x400, dump_start = 0, dump_end;
boolean				edit_ascii = FALSE;
unsigned int		fill_start = 0x0600, fill_end = 0x1dff;
unsigned char		fill_value = 0x00;
boolean				await_breakpoint = FALSE;
enum brktyp			breaktype;
boolean				exit_debugger;
boolean				debug_disp = FALSE;
unsigned int		last_pcr;
unsigned char		cursor_line;
unsigned char		drag_x = 5;
unsigned char		*reg_name;
unsigned char		deb_saved_ff22;
signed char			deb_saved_new_vmode;
unsigned char		deb_saved_new_screen_size;
unsigned int		deb_saved_new_screen_base;
boolean				debug_init;
unsigned int		eff_addr, eff_next;
boolean				eff_size, eff_jsub, eff_cond;
instr_mode			eff_mode;
unsigned int		pcr_backlog[9], so_backlog[5];
unsigned char		so_backlog_cnt, backlog_cnt = 0;
unsigned char		backlog_full;

/* Definitions for relative positions and sizes of some windows. */
#define REGS_X		67
#define REGS_Y  	15

#define STACK_X		41
#define STACK_Y		15
#define STACK_DEPTH	4

#define BREAK_X		STACK_X
#define BREAK_Y		20

#define DIS_DEPTH	9

/* Text strings for disassembly of instructions. */
unsigned char d_texts[ILLEGAL+1][6] =
{
	"ABX",	"ADC",	"ADD",	"ADD",	"AND",	"ASR",	"BIT",	"CLR",	"CMP",
	"CMP",	"COM",	"CWAI",	"DAA",	"DEC",	"EOR",	"EXG",	"INC",	"",
	"LD",	"LD",	"LEA",	"LSL",	"LSR",	"MUL",	"NEG",	"NOP",	"OR",
	"PSH",	"PUL",	"ROL",	"ROR",	"RTI",	"RTS",	"SBC",	"SEX",	"ST",
	"ST",	"SUB",	"SUB",	"SWI",	"SYNC",	"TFR",	"TST",	"RES",	"OR",
	"AND",	"DOIO",	"???"
};

/* Text strings for branch instructions. */
unsigned char branch_texts[16][4] =
{
	"BRA",	"BRN",	"BHI",	"BLS",	"BHS",	"BLO",	"BNE",	"BEQ",
	"BVC",	"BVS",	"BPL",	"BMI",	"BGE",	"BLT",	"BGT",	"BLE"
};

/* Output register names. */
void print_reg(__register preg)
{
	switch (preg)
	{
		case A_REG:		reg_name = "A";		break;
		case B_REG:		reg_name = "B";		break;
		case D_REG:		reg_name = "D";		break;
		case X_REG:		reg_name = "X";		break;
		case Y_REG:		reg_name = "Y";		break;
		case U_REG:		reg_name = "U";		break;
		case S_REG:		reg_name = "S";		break;
		case CC_REG:	reg_name = "CC";	break;
		case DP_REG:	reg_name = "DP";	break;
		case PC_REG:	reg_name = "PC";	break;
		default:		return;
	}
	cprintf(reg_name);
}

/* Display name of register in TFR and EXG instructions. */
void tfr_exg_reg(unsigned char rbyte)
{
	switch (rbyte)
	{
		case 0x00:	putch('D');			break;
		case 0x01:	putch('X');			break;
		case 0x02:	putch('Y');			break;
		case 0x03:	putch('U');			break;
		case 0x04:	putch('S');			break;
		case 0x05:	cprintf("PC");		break;
		case 0x08:	putch('A');			break;
		case 0x09:	putch('B');			break;
		case 0x0a:	cprintf("CC");		break;
		case 0x0b:	cprintf("DP");		break;
		default:	cprintf("#$FFFF");	break;
	}
}

/* Disassemble one instruction. */
unsigned int disassemble(register unsigned int d_address)
{
	ins_t				*ip;
	unsigned char		indirect;			/* Indirection flag.            */
	unsigned char		opcode;				/* Current opcode.              */
	unsigned char		secireg;
	boolean				ext1, ext2;			/* Page 1 and page 2 flags.     */
	instr_type			i_type;				/* Type (group) of instruction. */
	instr_mode			a_mode;				/* Current addressing mode.     */
	unsigned int		i_size;
	unsigned char		boff;
	__register			__r, idx_reg;
	int					oldx;
	unsigned char		minus_r;
	unsigned char		r_plus;
	reg16				offset;
	unsigned char		psh_pul_regs[22] = ""; /* Includes extra comma. */
	unsigned int		effective;
	boolean				e_size;
	boolean				store_effective;

	store_effective = (d_address == pcr.u_value);

	/* Read next instruction opcode. */
	opcode = far_get_mem8(d_address);
	cprintf("%04X %02X",d_address++,opcode);

	/* Check page of opcode, read another byte if necessary. */
	ext1 = (opcode == EXT1);
	ext2 = (opcode == EXT2);

	if (ext1 || ext2)
	{
		while ((opcode == EXT1) || (opcode == EXT2))
			opcode = far_get_mem8(d_address++);

		cprintf("%02X",opcode);
	}

	/* Lookup details of instruction in table. */
	ip = &i_table[opcode * sizeof(instruction)];

	/* Get type, mode, register and size of instruction. */
	i_type		= *ip++;
	a_mode		= *ip++;
	__r			= *ip;
	ip			+= 2;
	i_size		= *ip++;

	/* Transform branches into long branches if required. */
	if (ext1 && (a_mode == RELATIVE))
		i_size++;

	/* Fetch rest of instruction. */
	boff = 0;
	while (--i_size > 0)
		cprintf("%02X",i_buffer[boff++] = far_get_mem8(d_address++));

	/* Process certain page 1 and page 2 instructions. */
	if ((i_type == SUB16) && (__r == D_REG))
	{
		if (ext1)
			i_type = CMP16;
		else if (ext2)
		{
			i_type = CMP16;
			__r = U_REG;
		}
	}
	else if (ext1 && (__r == U_REG))
		__r = S_REG;
	else if (__r == X_REG)
	{
		if (ext1)
			__r = Y_REG;
		else if (ext2)
			__r = S_REG;
	}

	/* Preserve cursor location and advance to column 16. */
	oldx = wherex();
	while (wherex() < 16)
		putch(' ');

	/* Output name of opcode. */
	if (i_type == JMP)
	{
		if (opcode == BSR)
			cprintf("BSR");
		else if (opcode == LBRA)
			cprintf("LBRA");
		else if (opcode == LBSR)
			cprintf("LBSR");
		else if (a_mode == RELATIVE)
		{
			if (ext1)
				putch('L');
			cprintf(branch_texts[opcode-0x20]);
		}
		else if (opcode > JSR_MIN)
			cprintf("JSR");
		else
			cprintf("JMP");
	}
	else
		cprintf(d_texts[i_type]);

	/* Output name of the implicit register (except for CWAI or DAA). */
	if ((i_type != CWAI) && (i_type != DAA))
		print_reg(__r);
	if (i_type != SWI)
	{
		/* Advance to column 22 (except for SWI instructions). */
		while (wherex() < 22)
			putch(' ');
	}

	/* Disassemble addressing mode. */
	e_size = 16;
	switch (a_mode)
	{
		case INDEXED:	minus_r = r_plus = offset.u_value = 0;
						secireg = i_buffer[0] & 0x0f;
						switch(idx_reg = ((i_buffer[0] >> 5) & 0x03))
						{
							case OFF_X:		effective = x_reg.u_value;
											break;
							case OFF_Y:		effective = y_reg.u_value;
											break;
							case OFF_U:		effective = u_reg.u_value;
											break;
							case OFF_S:		effective = s_reg.u_value;
											break;
						}

						if (i_buffer[0] < 0x80) /* 4 bit plus sign. */
						{
							indirect = FALSE;
							if ((i_buffer[0] & 0x10) == 0)
								offset.u_value = (i_buffer[0] & 0x000f);
							else
								offset.u_value = (i_buffer[0] | 0xfff0);
						}
						else
						{
							indirect = i_buffer[0] & 0x10;
							switch (secireg)
							{
								case 0: /* Allows illegal (,r+). */
									r_plus = 1;
									break;

								case 1:
									r_plus = 2;
									break;

								case 2: /* Allows illegal (,-r). */
									minus_r = 1;
									break;

								case 3:
									minus_r = 2;
									break;

								case 4:
									break;

								case 5:
									offset.s_value = b_reg.s_value;
									break;

								case 6:
									offset.s_value = a_reg.s_value;
									break;

								case 7: /* 7 is illegal. */
									break;

								case 8:
									offset.s_value = (signed char)far_get_mem8(d_address);
									i_size = 1;
									break;

								case 9:
									offset.u_value = far_get_mem16(d_address);
									i_size = 2;
									break;

								case 10: /* 10 is illegal. */
									break;

								case 11:
									offset.u_value = d_reg_u_value;
									break;

								case 12:
									offset.u_value = d_address + (signed char)far_get_mem8(d_address) + 1;
									i_size = 1;
									idx_reg = OFF_PCR;
									break;

								case 13:
									offset.u_value = d_address + far_get_mem16(d_address) + 2;
									i_size = 2;
									idx_reg = OFF_PCR;
									break;

								case 14: /* 14 is illegal. */
									break;

								case 15:
									offset.u_value = far_get_mem16(d_address);
									indirect = i_size = 2;
									idx_reg = OFF_NONE;
									break;
							}
						}
						if (indirect != 0x00)
							putch('(');
						if (i_buffer[0] >= 0x80)
						{
							switch(secireg)
							{
								case 5:		putch('B');
											break;
								case 6:		putch('A');
											break;
								case 11:	putch('D');
											break;
								default:	goto numeric_offset;
							}
						}
						else
						{
							numeric_offset:

								if (offset.u_value != 0)
								{
									if ((idx_reg != OFF_PCR) && (idx_reg != OFF_NONE))
										cprintf("%d",offset.s_value);
									else
										cprintf("$%04X",offset.u_value);
								}
						}
						if (idx_reg != OFF_NONE)
							putch(',');
						switch(minus_r)
						{
							case 2:			putch('-');
							case 1:			putch('-');
											effective -= minus_r;
						}
						switch (idx_reg)
						{
							case OFF_X:		putch('X');
											break;
							case OFF_Y:		putch('Y');
											break;
							case OFF_U:		putch('U');
											break;
							case OFF_S:		putch('S');
											break;
							case OFF_PCR:	cprintf("PC");
											effective = pcr.u_value;
											break;
						}
						switch (r_plus)
						{
							case 2:			putch('+');
							case 1:			putch('+');
						}
						effective += offset.s_value;
						if (indirect != 0x00)
						{
							effective = far_get_mem16(offset.u_value);
							putch(')');
						}
						gotoxy(oldx,wherey());
						while (i_size > 0)
						{
							i_size--;
							cprintf("%02X", far_get_mem8(d_address++));
						}
						break;

		case IMMEDIATE:	if ((__r == A_REG) || (__r == B_REG) || (__r == CC_REG))
						{
							e_size = 8;
							cprintf("#$%02X", effective = i_buffer[0]);
						}
						else
							cprintf("#$%04X", effective = (i_buffer[0] << 8) + i_buffer[1]);
						break;

		case DIRECT:    effective = (dp_reg.u_value << 8) + i_buffer[0];
						cprintf("<$%02X", i_buffer[0]);
						break;

		case EXTENDED:	cprintf("$%04X", effective = (i_buffer[0] << 8) + i_buffer[1]);
						break;

		case RELATIVE:	if (!ext1 && (opcode != LBRA) && (opcode != LBSR))
							effective = d_address + (signed char)i_buffer[0];
						else
							effective = d_address + (i_buffer[0] << 8) + i_buffer[1];
						cprintf("$%04X", effective);
						break;
	}

	/* Add additional information for some instructions. */
	switch (i_type)
	{
		case SWI:	if (ext1)
						putch('2');
					else if (ext2)
						putch('3');
					break;

		case _DOIO:	cprintf("$%02X",i_buffer[0]);
					break;

		case PSH:
		case PUL:	if (i_buffer[0] & 0x02)
						strcpy(psh_pul_regs,"A,");
					if (i_buffer[0] & 0x04)
					{
						if (psh_pul_regs[0] == 'A')
							strcpy(psh_pul_regs,"D,");
						else
							strcpy(psh_pul_regs,"B,");
					}
					if (i_buffer[0] & 0x10)
						strcat(psh_pul_regs,"X,");
					if (i_buffer[0] & 0x20)
						strcat(psh_pul_regs,"Y,");
					if (i_buffer[0] & 0x40)
					{
						if (__r == S_REG)
							strcat(psh_pul_regs,"U,");
						else
							strcat(psh_pul_regs,"S,");
					}
					if (i_buffer[0] & 0x08)
						strcat(psh_pul_regs,"DP,");
					if (i_buffer[0] & 0x01)
						strcat(psh_pul_regs,"CC,");
					if (i_buffer[0] & 0x80)
						strcat(psh_pul_regs,"PC,");
					if ((i_size = strlen(psh_pul_regs)) > 0)
						psh_pul_regs[i_size-1] = '\0';
					cprintf(psh_pul_regs);
					break;

		case TFR:
		case EXG:	tfr_exg_reg(i_buffer[0] >> 4);
					putch(',');
					tfr_exg_reg(i_buffer[0] & 0x0f);
					break;
	}

	/* If this is the current instruction then remember the effective */
	/* address, conditional path, and cond and jsub flags etc.        */
	if (store_effective)
	{
		eff_addr = effective;
		eff_next = d_address;
		eff_size = e_size;
		eff_mode = a_mode;
		eff_jsub = ((i_type == JMP)
			&& ((opcode == BSR) || (opcode == LBSR) || (opcode > JSR_MIN)));
		eff_cond = ((a_mode == RELATIVE)
			&& (opcode != BRA) && (opcode != BSR)
			&& (opcode != LBRA) && (opcode != LBSR));
	}

	/* Return the address of the next instruction. */
	return(d_address);
}

/* Change to another window and set colours. */
void new_window(win_type win_where)
{
	switch (win_where)
	{
		case WIN_DRAGON:	window(drag_x,5,drag_x+31,20);
							update_cursor();
							break;

		case WIN_DIS:		window(41,5,76,13);
							textattr((CYAN << 4) + LIGHTCYAN);
							break;

		case WIN_DIS2:		window(41,4+backlog_cnt,76,13);
							textattr((CYAN << 4) + BLACK);
							break;

		case WIN_DIALOGUE:	window(3,23,36,23);
							textattr((BLUE << 4) + YELLOW);
							clrscr();
							break;

		case WIN_S_STACK:	window(STACK_X+5,STACK_Y,STACK_X+8,STACK_Y+STACK_DEPTH);
							textattr((CYAN << 4) + BROWN);
							break;

		case WIN_U_STACK:	window(STACK_X+16,STACK_Y,STACK_X+19,STACK_Y+STACK_DEPTH);
							textattr((CYAN << 4) + BROWN);
							break;

		case WIN_REGS:		window(REGS_X,REGS_Y,REGS_X+3,REGS_Y+7);
							textattr((CYAN << 4) + BROWN);
							break;

		case WIN_FLAGS:		window(REGS_X+9,REGS_Y,REGS_X+9,REGS_Y+7);
							textattr((CYAN << 4) + BROWN);
							break;
	}
}

/* Re-draw the '>' and '<' cursor in small disassembly window. */
void move_cursor(void)
{
	textattr((CYAN << 4) + BLACK);
	window(40,5,40,13);
	clrscr();
	gotoxy(1, cursor_line);
	putch(16);
	window(77,5,77,13);
	clrscr();
	gotoxy(1, cursor_line);
	putch(17);
}

/* Update the memory dump window. */
void update_dump(void)
{
	unsigned int	chars, lines, dump_addr;
	unsigned char	dump_line[8];

	if ((dump_start + 0x80) < dump_start)
	{
		if (((dump_address+0x8000) < (dump_start+0x8000)) || ((dump_address+0x8000) > (dump_start + 0x807f)))
			dump_start = dump_address & 0xfff8;
	}
	else if ((dump_address < dump_start) || (dump_address > (dump_start + 0x7f)))
		dump_start = dump_address & 0xfff8;
	dump_addr = dump_start;
	window(drag_x,5,drag_x+31,21);
	for (lines = 0; lines < 16; lines++)
	{
		textattr((CYAN << 4) | BLUE);
		cprintf(" %04X ", dump_addr);
		textattr((CYAN << 4) | YELLOW);
		for (chars = 0; chars < 8; chars++)
		{
#ifdef STRIPY
			if ((chars & 1) == 0)
#endif
				textattr((CYAN << 4) | YELLOW);
#ifdef STRIPY
			else
				textattr((CYAN << 4) | WHITE);
#endif
			cprintf("%02X",dump_line[chars] = far_get_mem8(dump_addr++));
		}
		putch(' ');
		textattr((CYAN << 4) | MAGENTA);
		for (chars = 0; chars < 8; chars++)
			cprintf("%c",isprint(dump_line[chars]) ? dump_line[chars] : '.');
		cprintf(" \n");
	}
	dump_end = dump_addr;
}

/* Draw border around left hand window. */
void left_border(void)
{
	window(drag_x-1,4,drag_x+33,21);
	switch (left_window)
	{
		case MINI_DRAGON:	textattr((back_ary[palette] << 4) + LIGHTGREEN);
							break;

		case HELP1:
		case HELP2:
		case MEM_DUMP:		textattr((CYAN << 4) + DARKGRAY);
							break;
	}
	border(34,18);
}

/* Initialise and refresh the left hand window. */
void init_left(void)
{
	left_border();
	new_window(WIN_DRAGON);

	switch (left_window)
	{
		case MINI_DRAGON:	video_enabled = TRUE;
							refresh_video();
							break;

		case HELP1:			video_enabled = FALSE;
							textcolor(YELLOW);
							clrscr();
							cprintf("\n   --= Debugger Help Page =--\r\n\r\n");
							cprintf("      B   Quick Breakpoint\r\n");
							cprintf("      D   Disassemble\r\n");
							cprintf("      F   Change CPU Flags\r\n");
							cprintf("      G   Go To Breakpoint\r\n");
							cprintf("      H   Next Help Page\r\n");
							cprintf("      I   Fill Memory\r\n");
							cprintf("      M   Edit Memory\r\n");
							cprintf("      R   Change Register\r\n");
							cprintf("      Q   Break On RTS\r\n");
							cprintf("      S   Single Step-Into\r\n");
							cprintf("      T   View Text Screen\r\n");
							cprintf("      X   Execute From\r\n");
							break;

		case HELP2:			video_enabled = FALSE;
							textcolor(YELLOW);
							clrscr();
							cprintf("\n   --= Debugger Help Page =--\r\n\r\n");
							cprintf("      A   Effective Address\r\n");
							cprintf("      E   Quick Edit Memory\r\n");
							cprintf("      H   Previous Help Page\r\n");
							cprintf("      N   Path Not Taken\r\n");
							cprintf("      O   Break On Opcode\r\n");
							cprintf("      V   Break On Value\r\n");
							cprintf("     1-6  Set Breakpoint\r\n");
							cprintf("     SPC  Single Step-Over\r\n");
							cprintf("     ESC  Quit Debugger\r\n");
							cprintf("     RET  Go To Cursor\r\n");
							cprintf("     TAB  Toggle Breakpoint\r\n");
							cprintf("        Move Cursor");
							break;

		case MEM_DUMP:		video_enabled = FALSE;
							break;
	}
}

/* Update the small disassembler window (from cursor position onwards). */
void update_dis(void)
{
	unsigned char future;

	if (backlog_cnt < 1)
	{
		pcr_backlog[0] = pcr.u_value;
		cursor_line = backlog_cnt = 1;
		move_cursor();
	}
	future = backlog_cnt;

	new_window(WIN_DIS2);
	clrscr();
	textcolor(BLUE);
	last_pcr = disassemble(pcr.u_value);
	textcolor(LIGHTCYAN);
	while (wherey() <= ((DIS_DEPTH) - backlog_cnt))
	{
		cprintf("\r\n");
		last_pcr = disassemble(pcr_backlog[future++] = last_pcr);
	}
}

/* Update the current register values window. */
void update_regs(void)
{
	new_window(WIN_REGS);
	cprintf("%04X\n%04X\n%04X\n%04X\n%04X\r\n",
		pcr.u_value,x_reg.u_value,y_reg.u_value,s_reg.u_value,u_reg.u_value);
	cprintf("%02X\r\n%02X\r\n%02X",
		dp_reg.u_value,a_reg.u_value,b_reg.u_value);
}

/* Update the flags section of right hand side window. */
void update_flags(void)
{
	new_window(WIN_FLAGS);
	cprintf("E:%X\nF:%X\nH:%X\nI:%X\n",e_flag,f_flag,h_flag,i_flag);
	cprintf("N:%X\nZ:%X\nV:%X\nC:%X",n_flag,z_flag,v_flag,c_flag);
}

/* Update the system stack section of right hand side window. */
void update_s_stack(void)
{
	new_window(WIN_S_STACK);
	cprintf("%04X\n%04X\n%04X\n%04X",
		far_get_mem16(s_reg.u_value),	far_get_mem16(s_reg.u_value+2),
		far_get_mem16(s_reg.u_value+4),	far_get_mem16(s_reg.u_value+6));
}

/* Update the user stack section of right hand side window. */
void update_u_stack(void)
{
	new_window(WIN_U_STACK);
	cprintf("%04X\n%04X\n%04X\n%04X",
		far_get_mem16(u_reg.u_value),	far_get_mem16(u_reg.u_value+2),
		far_get_mem16(u_reg.u_value+4),	far_get_mem16(u_reg.u_value+6));
}

/* Update the breakpoints section of right hand side window. */
void update_break(void)
{
	window(BREAK_X+5,BREAK_Y,BREAK_X+8,BREAK_Y+2);
	textattr((CYAN << 4) + BROWN);
	cprintf("%04X\n%04X\n%04X",breakpoint[0],breakpoint[1],breakpoint[2]);

	window(BREAK_X+16,BREAK_Y,BREAK_X+19,BREAK_Y+2);
	cprintf("%04X\n%04X\n%04X",breakpoint[3],breakpoint[4],breakpoint[5]);
}

/* Update all the sections of the right hand side window */
/* (use only after initialisation of the window).        */
void update_main(void)
{
	/* Call the above functions to update some window sections. */
	update_regs();
	update_flags();
	update_s_stack();
	update_u_stack();
	move_cursor();

	/* Change colour of previous disassembly. */
	if (backlog_cnt > 1)
	{
		new_window(WIN_DIS);
		if (backlog_full)
		{
			gotoxy(1,DIS_DEPTH);
			putch('\n');
		}
		gotoxy(1,backlog_cnt-1);
		disassemble(pcr_backlog[backlog_cnt-2]);
	}

	/* Disassemble future instructions. */
	update_dis();
}

/* Initialise the main (right hand side) window. */
void init_main(void)
{
	unsigned int i_cnt;

	/* Draw the disassembler window. */
	window(39,4,78,23);
	textattr((CYAN << 4) + DARKGRAY);
	border(40,21);

	/* Draw the divisions in the window. */
	gotoxy(1,DIS_DEPTH+2);
	putch(199);
	for(i_cnt=0; i_cnt<38; i_cnt++)
		putch(196);
	putch(182);
	gotoxy(24,DIS_DEPTH+2);
	putch(194);
	gotoxy(24,20);
	putch(207);
	gotoxy(34,DIS_DEPTH+2);
	putch(194);
	gotoxy(34,20);
	putch(207);
	gotoxy(1,BREAK_Y-4);
	putch(199);

	/* Stack window. */
	window(STACK_X-1,STACK_Y,STACK_X+20,BREAK_Y+2);
	cprintf(" S+00:      U+00:     \n  +02:       +02:     \n");
	cprintf("  +04:       +04:     \n  +06:       +06:     \n");

	/* Breakpoint window. */
	for(i_cnt=0; i_cnt<22; i_cnt++)
		putch(196);
	cprintf("\n Brk1:      Brk4:     \n Brk2:      Brk5:     \n Brk3:      Brk6:     ");

	/* Registers and flags sections. */
	window(REGS_X-5,REGS_Y,REGS_X+10,REGS_Y+7);
	cprintf("%c PC:     %c E:  \n",179,179);
	cprintf("%c  X:     %c F:  \n",179,179);
	cprintf("%c  Y:     %c H:  \n",179,179);
	cprintf("%c  S:     %c I:  \n",179,179);
	cprintf("%c  U:     %c N:  \n",180,179);
	cprintf("%c DP:     %c Z:  \n",179,179);
	cprintf("%c  A:     %c V:  \n",179,179);
	cprintf("%c  B:     %c C:  ",179,179);

	/* Add breakpoint values. */
	update_break();

	/* Add disassembly of any backlogged opcodes. */
	new_window(WIN_DIS);
	clrscr();
	if (backlog_cnt > 2)
	{
		disassemble(pcr_backlog[0]);
		cprintf("\n\r");
		if (backlog_cnt > 3)
		{
			disassemble(pcr_backlog[1]);
			cprintf("\n\r");
			if (backlog_cnt > 4)
			{
				disassemble(pcr_backlog[2]);
				backlog_full = FALSE;
			}
		}
	}
}

/* Initialise the debugger screen. */
void init_debug_screen(void)
{
	unsigned int cnt = (80 * 24);

	/* Temporarilly force Dragon into text mode. */
	memory[0xff22] &= 0x0f;
	new_screen_size	= 0;

	/* Check how much work needs to be done - if we are only    */
	/* stepping then we don't need to re-draw the whole screen. */
	if (!debug_disp)
	{
		/* Update address cursor. */
		cursor_line = backlog_cnt;

		/* Switch back to text mode if necessary. */
		if (vmode >= 0)
		{
			new_screen_base = last_text_base;
			execute_vmode(TRUE);
		}

		textmode(C80);
		if (text_mode == C80)
			clrscr();

		/* Redefine UDG graphics for low resolution graphics blocks. */
		if (install_udg)
			define_graphics();

		/* Move text screen one column to the left whilst in debugger. */
		drag_x = 4;

		/* Check whether we already have the base debugger screen */
		/* cached - if not then re-draw it from scratch.          */
		if (!debug_init)
		{
			/* Background. */
			textattr((LIGHTGRAY << 4) + DARKGRAY);
			while (cnt-- > 0)
			{
				putch(177);
				if ((cnt % 80) == 0)
					putch('\n');
			}

			/* Help bar. */
			textattr((LIGHTGRAY << 4) + RED);
			cprintf("  R = Reg  F = Flags  S = Step  G = Go Break  1-6 = Set Break  ESC = Exit  ");

			/* Title. */
			window(3,2,78,2);
			textattr((LIGHTGRAY << 4) + BLUE);
			clrscr();
			cprintf("              --= PC-DRAGON V%d.%02d DEBUGGER & DISASSEMBLER =--",VER_MAJOR,VER_MINOR);
			gettext(1,1,80,25,debug_screen);
			debug_init = TRUE;
		}
		else
		{
			/* Restore debugger screen from the cache. */
			puttext(1,1,80,25,debug_screen);
		}

		/* Draw the windows. */
		init_main();
		init_left();
	}
	else
	{
		/* All we need to do is refresh the border of the LHS window. */
		left_border();
	}

	/* Update the memory dump window (if active). */
	if (left_window == MEM_DUMP)
		update_dump();
}

/* Generic numeric input dialogue box. */
boolean	value_box(unsigned char *vtit, unsigned int *vadd,
	unsigned char *vfm, boolean from_debugger)
{
	unsigned char stroff = 0;
	unsigned char *strfm = "%x";
	unsigned char local_string[40];

	sprintf(local_string,vfm,*vadd);
	if (!input_box(FALSE,local_string,2,16,vtit,""))
	{
		if (from_debugger)
		{
			new_window(WIN_DIALOGUE);
			cprintf("Command aborted.");
		}
		return(FALSE);
	}
	strupr(local_string);
	if (strcmp(local_string,"PC") == 0)
		*vadd = pcr.u_value;
	else if (strcmp(local_string,"X") == 0)
		*vadd = x_reg.u_value;
	else if (strcmp(local_string,"Y") == 0)
		*vadd = y_reg.u_value;
	else if (strcmp(local_string,"S") == 0)
		*vadd = s_reg.u_value;
	else if (strcmp(local_string,"U") == 0)
		*vadd = u_reg.u_value;
	else
	{
		switch (local_string[0])
		{
			case 'o':
			case 'O':	strfm = "%o";
			case '$':	stroff = 1;
						break;

			case '#':	strfm = "%d";
						stroff = 1;
						break;
		}
		if (sscanf(&local_string[stroff],strfm,vadd) == 0)
		{
			if (from_debugger)
			{
				new_window(WIN_DIALOGUE);
				cprintf("Format of new value incorrect.");
				beep();
			}
			return(FALSE);
		}
	}
	return(TRUE);
}

/* Change global breakpoint. */
void get_breakpoint(unsigned char b_num)
{
	unsigned char b_title[20];

	new_window(WIN_DIALOGUE);
	cprintf("Enter new breakpoint value.");
	sprintf(b_title, "  BREAKPOINT %d", b_num + 1);
	if (value_box(b_title, &breakpoint[b_num], "$%04X", TRUE))
	{
		new_window(WIN_DIALOGUE);
		cprintf("Breakpoint %d set to $%04X.", b_num + 1, breakpoint[b_num]);
		update_break();
	}
}

/* Set/delete global breakpoint at current cursor address. */
void tab(void)
{
	unsigned int	sval = pcr_backlog[cursor_line-1];
	unsigned char	brknum;
	boolean			brkfound = FALSE;

	new_window(WIN_DIALOGUE);

	/* Try to cancel breakpoint for current address. */
	for (brknum = 0; brknum < 6; brknum++)
	{
		if (breakpoint[brknum] == sval)
		{
			breakpoint[brknum] = NOBREAKPOINT;
			if (!brkfound)
			{
				cprintf("Breakpoint at $%04X cancelled.",sval);
				brkfound = TRUE;
			}
		}
	}

	/* Try to add breakpoint for current address. */
	if (!brkfound)
	{
		/* Look for free slot. */
		for (brknum = 0; (brknum < 6) && (!brkfound); brknum++)
		{
			if (breakpoint[brknum] == NOBREAKPOINT)
			{
				breakpoint[brknum] = sval;
				cprintf("Added breakpoint at $%04X.",sval);
				brkfound = TRUE;
			}
		}
	}

	if (brkfound)
		update_break();
	else
	{
		beep();
		cprintf("No free breakpoints.");
	}
}

/* Large disassembly window. */
void big_diss(void)
{
	unsigned char	d_cnt;
	boolean			d_another = TRUE, reinit = FALSE;

	while (d_another)
	{
		new_window(WIN_DIALOGUE);
		cprintf("Enter disassembly start address.");
		if (value_box("DISASSEMBLE FROM", &diss_pcr, "$%04X", TRUE))
		{
			new_window(WIN_DIALOGUE);
			cprintf("Press SPACE, ENTER, D or ESC.");

			window(39,4,78,23);
			textattr((CYAN << 4) + DARKGRAY);
			border(40,21);

			window(40,5,77,22);
			clrscr();
			textcolor(BLUE);
			reinit = TRUE;

			putch(' ');
			diss_pcr = disassemble(diss_pcr);
			d_cnt = 17;

			while (d_cnt > 0)
			{
				while(d_cnt > 0)
				{
					cprintf("\r\n ");
					diss_pcr = disassemble(diss_pcr);
					d_cnt--;
				}

				switch(toupper(getch2(TRUE)))
				{
					case 'D':	break;

					case 10:
					case 13:	d_cnt = 1;
								break;

					case 32:	d_cnt = 17;
								break;

					default:	beep();
					case 3:		new_window(WIN_DIALOGUE);
								cprintf("Disassembly aborted.");
								d_another = FALSE;
								break;
				}
			}
		}
		else
			d_another = FALSE;
	}
	if (reinit)
	{
		init_main();
		update_main();
	}
}

/* Fill memory with constant value. */
void fill(void)
{
	unsigned int fill16 = fill_value;

	new_window(WIN_DIALOGUE);
	cprintf("Enter fill start address.");
	if (!value_box("   FILL START", &fill_start, "$%04X", TRUE))
		return;

	new_window(WIN_DIALOGUE);
	cprintf("Enter fill end address.");
	if (!value_box("    FILL END", &fill_end, "$%04X", TRUE))
		return;

	new_window(WIN_DIALOGUE);
	cprintf("Enter fill value.");
	if (!value_box("   FILL VALUE", &fill16, "$%02X", TRUE))
		return;

	fill_value = fill16 & 0x00ff;

	/* Note: use of '!=' deliberately allows wrap-around. */
	fill16 = fill_start;
	do
	{
		far_set_mem8(fill16, fill_value);
	} while (fill16++ != fill_end);

	/* Update RHS window. */
	update_s_stack();
	update_u_stack();
	update_dis();

	/* Update the memory dump window (if active). */
	if (left_window == MEM_DUMP)
		update_dump();

	new_window(WIN_DIALOGUE);
	cprintf("Filled $%04X to $%04X with $%02X.",
		fill_start,fill_end,fill_value);
}

/* Move cursor to appropriate position in the memory editor window. */
void edit_goto(void)
{
	int x_h, y_h;

	x_h = edit_ascii ? (24 + (dump_address & 0x0007))
			: (7 + ((dump_address & 0x0007) << 1));
	y_h = 1 + (((dump_address & 0xfff8) - dump_start) / 8);

	window(drag_x,5,drag_x+31,21);
	gotoxy(x_h, y_h);
}

/* Memory editor. */
void edit_memory(boolean ask_edit_address)
{
	boolean			first_hex, exit_edit = FALSE;
	unsigned char	herechar, herekey;

	if (ask_edit_address)
	{
		new_window(WIN_DIALOGUE);
		cprintf("Enter memory address to edit.");
		if (!value_box("  EDIT ADDRESS", &dump_address, "$%04X", TRUE))
			return;
	}

	left_window = MEM_DUMP;
	init_left();
	update_dump();
	new_window(WIN_DIALOGUE);
	cprintf("Type new value, TAB or ESC.");

	first_hex = TRUE;
	while (!exit_edit)
	{
		herechar = far_get_mem8(dump_address);

		edit_ascii = !edit_ascii;
		edit_goto();
		textattr((CYAN << 4) + WHITE);
		if (edit_ascii)
			putch(isprint(herechar) ? herechar : '.');
		else
			cprintf("%02X",herechar);

		edit_ascii = !edit_ascii;
		edit_goto();
		textattr((WHITE << 4) + BROWN);
		if (edit_ascii)
			putch(isprint(herechar) ? herechar : '.');
		else
		{
			if (!first_hex)
				textattr((LIGHTGRAY << 4) + BROWN);
			cprintf("%1X",herechar >> 4);
			textattr((WHITE << 4) + BROWN);
			cprintf("%1X",herechar & 0x0f);
		}

		endkey = rightkey = pgdnkey = upkey = FALSE;
		herekey = getch2(TRUE);
		if (isxdigit(herekey) && (!edit_ascii))
		{
			if (herekey <= '9')
				herekey -= '0';
			else
				herekey = (tolower(herekey) - 'a') + 10;
			if (first_hex)
				herechar = (herechar & 0x0f) + (herekey << 4);
			else
				herechar = (herechar & 0xf0) + herekey;
			first_hex = !first_hex;
			herekey = 'a';
		}
		switch(herekey)
		{
			case 3:		if (endkey)
							goto defcase;
						exit_edit = TRUE;
						break;

			case 8:		if (!edit_ascii)
							first_hex = !first_hex;
						else
							first_hex = FALSE;
						if (!first_hex)
							dump_address--;
						goto chkup;

			case 9:		if (rightkey)
						{
							if (!edit_ascii)
								first_hex = !first_hex;
							else
								first_hex = TRUE;
							goto incaddr;
						}
						else
						{
							edit_ascii	= !edit_ascii;
							first_hex	= TRUE;
						}
						break;

			case 10:	first_hex = TRUE;
						dump_address += 8;
						goto chkdown;

			case 13:    if (!pgdnkey)
							goto defcase;
						first_hex = TRUE;
						dump_address += 128;
						dump_start += 128;
						break;

			case 19:	first_hex = TRUE;
						dump_address -= 128;
						dump_start -= 128;
						break;

			case 94:    if (!upkey)
							goto defcase;
						first_hex = TRUE;
						dump_address -= 8;
			chkup:		if ((dump_address & 0xfff8) == (dump_start - 8))
							dump_start -= 8;
						break;

			case 'a':	if (!edit_ascii)
							goto setchar;
			default:
			defcase:	if (!edit_ascii)
							beep();
						else
						{
							first_hex = TRUE;
							herechar = herekey;
			setchar:		far_set_mem8(dump_address, herechar);
			incaddr:		if (first_hex)
								dump_address++;
			chkdown:		if ((dump_address & 0xfff8) == dump_end)
								dump_start += 8;
						}
						break;
		}
		update_dump();
	}
	new_window(WIN_DIALOGUE);
	cprintf("Press E to resume editing.");

	/* May need to update disassembly. */
	update_dis();
}

/* Execute from a new PC address. */
boolean get_new_pc(void)
{
	new_window(WIN_DIALOGUE);
	cprintf("Enter execution address.");
	return(value_box("  EXECUTE FROM", &pcr.u_value, "$%04X", TRUE));
}

/* Set break on an opcode. */
boolean get_breakop(void)
{
	unsigned int local_op = break_op;

	new_window(WIN_DIALOGUE);
	cprintf("Enter break opcode.");
	if (!value_box("  BREAK OPCODE", &local_op, ((local_op > 0x00ff) ? "$%04X" : "$%02X"), TRUE))
		return(FALSE);
	if (local_op > 0x00ff)
	{
		switch (local_op >> 8)
		{
			case 0x10:
			case 0x11:	break;

			default:	beep();
						new_window(WIN_DIALOGUE);
						cprintf("Invalid opcode.");
						return(FALSE);
		}
	}
	break_op = local_op;
	return(TRUE);
}

/* Execute with a temporary breakpoint. */
boolean goto_address(void)
{
	new_window(WIN_DIALOGUE);
	cprintf("Enter temporary breakpoint.");
	return(value_box("   EXECUTE TO", &temp_breakpoint, "$%04X", TRUE));
}

/* Modify a register. */
void change_regs(void)
{
	reg8			*reg8_;
	reg16			*reg16_;
	__register		__r;
	instr_bits		__b;
	unsigned int	rlocal;
	unsigned char	r_title[20];

	new_window(WIN_DIALOGUE);
	cprintf("Select A,B,X,Y,S,U,DP,CC or PC.");
	switch(toupper(getch2(TRUE)))
	{
		case 'A':	__r = A_REG;
					break;
		case 'B':	__r = B_REG;
					break;
		case 'X':	__r = X_REG;
					break;
		case 'Y':	__r = Y_REG;
					break;
		case 'S':	__r = S_REG;
					break;
		case 'U':	__r = U_REG;
					break;
		case 'D':	__r = DP_REG;
					break;
		case 'C':	__r = CC_REG;
					break;
		case 'P':	__r = PC_REG;
					break;
		default:	beep();
		case 3:		clrscr();
					cprintf("No register selected.");
					return;
	}

	set_reg();
	if (__b == B8)
	{
		if (__r == CC_REG)
			rlocal = get_cc_reg();
		else
			rlocal = (unsigned int)(reg8_->u_value);
	}
	else
		rlocal = reg16_->u_value;

	clrscr();
	cprintf("Enter new value for ");
	print_reg(__r);
	putch('.');
	if ((__r == CC_REG) || (__r == PC_REG) || (__r == DP_REG))
		sprintf(r_title,"  %s REGISTER",reg_name);
	else
		sprintf(r_title,"   %s REGISTER",reg_name);
	if (value_box(r_title, &rlocal, ((__b == B8) ? "$%02X" : "$%04X"), TRUE))
	{
		new_window(WIN_DIALOGUE);
		if (__b == B8)
		{
			if (__r == CC_REG)
				far_set_cc_reg(rlocal & 0x00ff);
			else if ((rlocal > 0xff) && ((__r == A_REG) || (__r == B_REG)))
			{
				__b = B16;
				__r = D_REG;
				d_reg_u_value = rlocal;
			}
			else
				reg8_->u_value = (rlocal & 0x00ff);
		}
		else
			reg16_->u_value = rlocal;
		print_reg(__r);
		cprintf(" register set to $");
		if (__b == B8)
			cprintf("%02X.",rlocal & 0x00ff);
		else
			cprintf("%04X.",rlocal);

		/* Update debugger displays. */
		update_regs();
		switch(__r)
		{
			case S_REG:		update_s_stack();
							break;

			case U_REG:		update_u_stack();
							break;

			case PC_REG:	if (backlog_cnt > 0)
								pcr_backlog[backlog_cnt-1] = pcr.u_value;
							update_dis();
							break;

			case CC_REG:	update_flags();
							break;
		}
	}
}

/* Toggle CPU flags. */
void change_flags(void)
{
	boolean iterate = TRUE;

	new_window(WIN_DIALOGUE);
	cprintf("Press E,F,H,I,N,Z,V,C or ESC.");
	while(iterate)
	{
		switch(toupper(getch2(TRUE)))
		{
			case 'E':	e_flag = !e_flag; break;
			case 'F':	f_flag = !f_flag; break;
			case 'H':	h_flag = !h_flag; break;
			case 'I':	i_flag = !i_flag; break;
			case 'N':	n_flag = !n_flag; break;
			case 'Z':	z_flag = !z_flag; break;
			case 'V':	v_flag = !v_flag; break;
			case 'C':	c_flag = !c_flag; break;

			case 3:
			case 13:	iterate = FALSE;
						new_window(WIN_DIALOGUE);
						cprintf("CC register set to $%02X.",get_cc_reg());
						break;

			default:	beep();
						iterate = FALSE;
						new_window(WIN_DIALOGUE);
						cprintf("Flag not changed.");
						break;
		}
		update_flags();
	}
}

/* Execute until a register equals a specifed value (or changes). */
boolean get_brk_reg(void)
{
	reg8			*reg8_;
	reg16			*reg16_;
	__register		__r;
	instr_bits		__b;
	unsigned int	rlocal, roldlocal;
	unsigned char	r_title[20];

	new_window(WIN_DIALOGUE);
	cprintf("Select A,B,X,Y,S,U,DP,CC or PC.");
	switch(toupper(getch2(TRUE)))
	{
		case 'A':	__r = A_REG;
					breaktype = AREG_BREAK;
					break;
		case 'B':	__r = B_REG;
					breaktype = BREG_BREAK;
					break;
		case 'X':	__r = X_REG;
					breaktype = XREG_BREAK;
					break;
		case 'Y':	__r = Y_REG;
					breaktype = YREG_BREAK;
					break;
		case 'S':	__r = S_REG;
					breaktype = SREG_BREAK;
					break;
		case 'U':	__r = U_REG;
					breaktype = UREG_BREAK;
					break;
		case 'D':	__r = DP_REG;
					breaktype = DPREG_BREAK;
					break;
		case 'C':	__r = CC_REG;
					breaktype = CCREG_BREAK;
					break;
		case 'P':	__r = PC_REG;
					breaktype = PCREG_BREAK;
					break;
		default:	beep();
		case 3:		clrscr();
					cprintf("Register break aborted.");
					return FALSE;
	}

	set_reg();
	if (__b == B8)
	{
		if (__r == CC_REG)
			rlocal = get_cc_reg();
		else
			rlocal = (unsigned int)(reg8_->u_value);
	}
	else
		rlocal = reg16_->u_value;

	roldlocal = rlocal;
	clrscr();
	cprintf("Enter ");
	print_reg(__r);
	cprintf(" register break value.");
	if ((__r == CC_REG) || (__r == PC_REG) || (__r == DP_REG))
		sprintf(r_title," %s BREAK VALUE",reg_name);
	else
		sprintf(r_title,"  %s BREAKVALUE",reg_name);
	if (value_box(r_title, &rlocal, ((__b == B8) ? "$%02X" : "$%04X"), TRUE))
	{
		if (__b == B8)
		{
			if ((rlocal > 0xff) && ((__r == A_REG) || (__r == B_REG)))
			{
				breaktype = DREG_BREAK;
				reg_name = "D";
			}
			else
				rlocal &= 0x00ff;
		}
		if ((rlocal == roldlocal) ||
			((breaktype == DREG_BREAK) && (rlocal == d_reg_u_value)))
		{
			new_window(WIN_DIALOGUE);
			cprintf("Break on %s register change?", reg_name);
			switch(toupper(getch2(TRUE)))
			{
				case 'Y':	breaktype += (NOTX_BREAK - XREG_BREAK);
				case 'N':	break;

				default:	beep();
				case 3:		clrscr();
							cprintf("Register break aborted.");
							return FALSE;
			}
		}
		breakvalue = rlocal;
		return(TRUE);
	}
	return(FALSE);
}

/* Main function for debugger. */
void debugger(void)
{
	unsigned char	inkey;
	unsigned char	next_op;
	unsigned int	next_op_add;

	/* Update tracking of PC values. */
	if (backlog_cnt < 5)
	{
		backlog_cnt++;
		backlog_full = FALSE;
	}
	else
	{
		pcr_backlog[0] = pcr_backlog[1];
		pcr_backlog[1] = pcr_backlog[2];
		pcr_backlog[2] = pcr_backlog[3];
		pcr_backlog[3] = pcr_backlog[4];
		backlog_full = TRUE;
	}
	pcr_backlog[backlog_cnt-1] = pcr.u_value;

	/* Determine why we have entered the debugger? */
	if (await_breakpoint)
	{
		/* Looking for a stop condition - check */
		/* whether the condition has occurred.  */
		switch(breaktype)
		{
			case TEMP_BREAK:	if (pcr.u_value != temp_breakpoint)
									goto std_brk;
								break;

			case OPCODE0_BREAK:	if ((unsigned int)far_get_mem8(pcr.u_value) != break_op)
									goto std_brk;
								break;

			case OPCODE1_BREAK:	if ((unsigned int)far_get_mem8(pcr.u_value) != (break_op >> 8))
									goto std_brk;
								next_op_add = pcr.u_value;
								do
								{
									next_op_add++;
									next_op = far_get_mem8(next_op_add);
								} while ((next_op == EXT1) || (next_op == EXT2));
								if ((unsigned int)next_op != (break_op & 0x00ff))
									goto std_brk;
								break;

			case XREG_BREAK:	if (x_reg.u_value != breakvalue)
									goto std_brk;
								break;

			case YREG_BREAK:	if (y_reg.u_value != breakvalue)
									goto std_brk;
								break;

			case SREG_BREAK:	if (s_reg.u_value != breakvalue)
									goto std_brk;
								break;

			case UREG_BREAK:	if (u_reg.u_value != breakvalue)
									goto std_brk;
								break;

			case DREG_BREAK:	if (d_reg_u_value != breakvalue)
									goto std_brk;
								break;

			case AREG_BREAK:	if (a_reg.u_value != (unsigned char)breakvalue)
									goto std_brk;
								break;

			case BREG_BREAK:	if (b_reg.u_value != (unsigned char)breakvalue)
									goto std_brk;
								break;

			case PCREG_BREAK:	if (pcr.u_value != breakvalue)
									goto std_brk;
								break;

			case CCREG_BREAK:	if (get_cc_reg() != (unsigned char)breakvalue)
									goto std_brk;
								break;

			case DPREG_BREAK:	if (dp_reg.u_value != (unsigned char)breakvalue)
									goto std_brk;
								break;

			case NOTX_BREAK:	if (x_reg.u_value == breakvalue)
									goto std_brk;
								breakvalue = x_reg.u_value;
								break;

			case NOTY_BREAK:	if (y_reg.u_value == breakvalue)
									goto std_brk;
								breakvalue = y_reg.u_value;
								break;

			case NOTS_BREAK:	if (s_reg.u_value == breakvalue)
									goto std_brk;
								breakvalue = s_reg.u_value;
								break;

			case NOTU_BREAK:	if (u_reg.u_value == breakvalue)
									goto std_brk;
								breakvalue = u_reg.u_value;
								break;

			case NOTD_BREAK:	if (d_reg_u_value == breakvalue)
									goto std_brk;
								breakvalue = d_reg_u_value;
								break;

			case NOTA_BREAK:	if (a_reg.u_value == (unsigned char)breakvalue)
									goto std_brk;
								breakvalue = a_reg.u_value;
								break;

			case NOTB_BREAK:	if (b_reg.u_value == (unsigned char)breakvalue)
									goto std_brk;
								breakvalue = b_reg.u_value;
								break;

			case NOTPC_BREAK:	if (pcr.u_value == breakvalue)
									goto std_brk;
								breakvalue = pcr.u_value;
								break;

			case NOTCC_BREAK:	if (get_cc_reg() == (unsigned char)breakvalue)
									goto std_brk;
								breakvalue = get_cc_reg();
								break;

			case NOTDP_BREAK:	if (dp_reg.u_value == (unsigned char)breakvalue)
									goto std_brk;
								breakvalue = dp_reg.u_value;
								break;

			case STEP_OVER:		if (pcr.u_value != intern_break)
									goto std_brk;
								pcr_backlog[0] = so_backlog[0];
								pcr_backlog[1] = so_backlog[1];
								pcr_backlog[2] = so_backlog[2];
								pcr_backlog[3] = so_backlog[3];
								pcr_backlog[4] = so_backlog[4];
								backlog_cnt = so_backlog_cnt;
								if (backlog_cnt < 5)
								{
									backlog_cnt++;
									backlog_full = FALSE;
								}
								else
								{
									pcr_backlog[0] = pcr_backlog[1];
									pcr_backlog[1] = pcr_backlog[2];
									pcr_backlog[2] = pcr_backlog[3];
									pcr_backlog[3] = pcr_backlog[4];
									backlog_full = TRUE;
								}
								pcr_backlog[backlog_cnt-1] = pcr.u_value;
								break;

			case NOT_SETUP:		if (pcr.u_value == eff_next)
									intern_break = eff_addr;
								else
									intern_break = eff_next;
								breaktype = NOT_BRANCH;
								goto std_brk;

			case NOT_BRANCH:	if (pcr.u_value != intern_break)
									goto std_brk;
								break;

			case STD_BREAK:
			std_brk:			if ((pcr.u_value != breakpoint[0])
										&& (pcr.u_value != breakpoint[1])
										&& (pcr.u_value != breakpoint[2])
										&& (pcr.u_value != breakpoint[3])
										&& (pcr.u_value != breakpoint[4])
										&& (pcr.u_value != breakpoint[5]))
									return;

								breaktype = STD_BREAK;
								break;
		}
	}

	/* Preserve the current Dragon graphics mode before we change it */
	/* (also inhibits any changes being made by the memory editor).  */
	deb_saved_ff22				= memory[0xff22];
	deb_saved_new_vmode			= new_vmode;
	deb_saved_new_screen_size	= new_screen_size;
	deb_saved_new_screen_base	= new_screen_base;

	/* Draw screen (if necessary). */
	init_debug_screen();

	/* Output an appropriate message on the status line. */
	new_window(WIN_DIALOGUE);
	if (await_breakpoint)
	{
		switch(breaktype)
		{
			case TEMP_BREAK:
			case STD_BREAK:		cprintf("Halted at breakpoint $%04X.",
									pcr.u_value);
								break;

			case OPCODE0_BREAK:	cprintf("Halted at opcode $%02X.",break_op);
								break;

			case OPCODE1_BREAK:	cprintf("Halted at opcode $%04X.",break_op);
								break;

			case XREG_BREAK:
			case YREG_BREAK:
			case SREG_BREAK:
			case UREG_BREAK:
			case DREG_BREAK:
			case AREG_BREAK:
			case BREG_BREAK:
			case PCREG_BREAK:
			case CCREG_BREAK:
			case DPREG_BREAK:
			case NOTX_BREAK:
			case NOTY_BREAK:
			case NOTS_BREAK:
			case NOTU_BREAK:
			case NOTD_BREAK:
			case NOTA_BREAK:
			case NOTB_BREAK:
			case NOTPC_BREAK:
			case NOTCC_BREAK:
			case NOTDP_BREAK:	cprintf("Halted with ");
								switch (breaktype)
								{
									case XREG_BREAK:
									case NOTX_BREAK:	putch('X'); break;

									case YREG_BREAK:
									case NOTY_BREAK:	putch('Y'); break;

									case SREG_BREAK:
									case NOTS_BREAK:	putch('S'); break;

									case UREG_BREAK:
									case NOTU_BREAK:	putch('U'); break;

									case DREG_BREAK:
									case NOTD_BREAK:	putch('D'); break;

									case AREG_BREAK:
									case NOTA_BREAK:	putch('A'); break;

									case BREG_BREAK:
									case NOTB_BREAK:	putch('B'); break;

									case PCREG_BREAK:
									case NOTPC_BREAK:	cprintf("PC"); break;

									case CCREG_BREAK:
									case NOTCC_BREAK:	cprintf("CC"); break;

									case DPREG_BREAK:
									case NOTDP_BREAK:	cprintf("DP"); break;
								}
								switch (breaktype)
								{
									case AREG_BREAK:
									case BREG_BREAK:
									case CCREG_BREAK:
									case DPREG_BREAK:
									case NOTA_BREAK:
									case NOTB_BREAK:
									case NOTCC_BREAK:
									case NOTDP_BREAK:	cprintf(" equal to $%02X.",breakvalue);
														break;

									default:			cprintf(" equal to $%04X.",breakvalue);
														break;

								}
								break;

			case STEP_OVER:		cprintf("Stepped over to $%04X.",
									pcr.u_value);
								break;

			case NOT_BRANCH:	cprintf("Loop broken at $%04X.",
									pcr.u_value);
								break;
		}
	}
	else if (debug_disp)
		cprintf("Stepped to $%04X.",pcr.u_value);
	else
		cprintf("Break at $%04X.",pcr.u_value);

	/* Update all the sections of the right hand side window. */
	update_main();

	debug_disp			= TRUE;
	await_breakpoint	= FALSE;
	exit_debugger		= FALSE;

	/* Perform required functions. */
	while (!exit_debugger)
	{
		setcursortype(_NOCURSOR);
		switch (inkey = toupper(getch2(TRUE)))
		{
			/* Note: Without the key.asm fix the following functions cause */
			/* the real mode keyboard to hang: B, O, X, V(sometimes).      */

			case 3:		in_debug = FALSE;
						exit_debugger = TRUE;
						backlog_cnt = 0;
						break;

			case 94:	if (cursor_line > 1)
						{
							cursor_line--;
							move_cursor();
						}
						else
						{
							beep();
							new_window(WIN_DIALOGUE);
							if (backlog_cnt == 0)
								cprintf("Press HOME to align with PC.");
							else
								cprintf("Cannot scroll upwards.");
						}
						break;

			case 8:
			case 9:		tab();
						break;

			case 10:	if (cursor_line < 9)
							cursor_line++;
						else
						{
							if (backlog_cnt > 0)
								backlog_cnt--;

							pcr_backlog[0] = pcr_backlog[1];
							pcr_backlog[1] = pcr_backlog[2];
							pcr_backlog[2] = pcr_backlog[3];
							pcr_backlog[3] = pcr_backlog[4];
							pcr_backlog[4] = pcr_backlog[5];
							pcr_backlog[5] = pcr_backlog[6];
							pcr_backlog[6] = pcr_backlog[7];
							pcr_backlog[7] = pcr_backlog[8];

							new_window(WIN_DIS);
							gotoxy(1,DIS_DEPTH);
							putch('\n');
							last_pcr = disassemble(pcr_backlog[8] = last_pcr);
						}
						move_cursor();
						break;

			case 12:	if (backlog_cnt == 0)
							update_dis();
						else
						{
							cursor_line = backlog_cnt;
							move_cursor();
						}
						break;

			case 13:	temp_breakpoint = pcr_backlog[cursor_line-1];
						breaktype = TEMP_BREAK;
						await_breakpoint = TRUE;
						exit_debugger = TRUE;
						break;

			case 32:	if ((cursor_line == backlog_cnt) && (cursor_line < 5))
							cursor_line++;
						else if (backlog_cnt == 0)
							cursor_line = 1;
						if (eff_jsub)
						{
							so_backlog[0] = pcr_backlog[0];
							so_backlog[1] = pcr_backlog[1];
							so_backlog[2] = pcr_backlog[2];
							so_backlog[3] = pcr_backlog[3];
							so_backlog[4] = pcr_backlog[4];
							so_backlog_cnt = backlog_cnt;
							intern_break = eff_next;
							breaktype = STEP_OVER;
							await_breakpoint = TRUE;
						}
						exit_debugger = TRUE;
						break;

			case '1':
			case '2':
			case '3':
			case '4':
			case '5':
			case '6':	get_breakpoint(inkey - '1');
						break;

			case 'A':	new_window(WIN_DIALOGUE);
						switch (eff_mode)
						{
							case NO_MODE:   cprintf("No effective address (illegal).");
											beep();
											break;

							case INHERENT:	cprintf("No effective address (inherent).");
											beep();
											break;

							default:		if (eff_size < 16)
												cprintf("Effective address $%02X.", eff_addr);
											else
												cprintf("Effective address $%04X.", eff_addr);
											break;
						}
						break;

			case 'B':	if (goto_address())
						{
							breaktype = TEMP_BREAK;
							await_breakpoint = TRUE;
							exit_debugger = TRUE;
						}
						break;

			case 'D':	big_diss();
						break;

			case 'E':	edit_memory(FALSE);
						break;

			case 'F':	change_flags();
						break;

			case 'G':	breaktype = STD_BREAK;
						await_breakpoint = TRUE;
						exit_debugger = TRUE;
						break;

			case 'H':	left_window = (left_window == HELP1) ? HELP2 : HELP1;
						init_left();
						new_window(WIN_DIALOGUE);
						cprintf("Press H for other help page.");
						break;

			case 'I':	fill();
						break;

			case 'M':	edit_memory(TRUE);
						break;

			case 'N':   if (eff_cond)
						{
							breaktype = NOT_SETUP;
							await_breakpoint = TRUE;
							exit_debugger = TRUE;
						}
						else
						{
							beep();
							new_window(WIN_DIALOGUE);
							cprintf("Not a conditional branch.");
						}
						break;

			case 'O':	if (get_breakop())
						{
							breaktype = (break_op > 0x00ff) ? OPCODE1_BREAK : OPCODE0_BREAK;
							await_breakpoint = TRUE;
							exit_debugger = TRUE;
						}
						break;

			case 'Q':	break_op = 0x39;
						breaktype = OPCODE0_BREAK;
						await_breakpoint = TRUE;
						exit_debugger = TRUE;
						break;

			case 'R':	change_regs();
						break;

			case 'S':	if ((cursor_line == backlog_cnt) && (cursor_line < 5))
							cursor_line++;
						else if (backlog_cnt == 0)
							cursor_line = 1;
						exit_debugger = TRUE;
						break;

			case 'T':	left_window = MINI_DRAGON;
						init_left();
						new_window(WIN_DIALOGUE);
						cprintf("Text screen $%04X-$%04X.",screen_base,screen_end);
						break;

			case 'V':	if (get_brk_reg())
						{
							await_breakpoint = TRUE;
							exit_debugger = TRUE;
						}
						break;

			case 'X':	if (get_new_pc())
						{
							if (backlog_cnt > 0)
								pcr_backlog[backlog_cnt-1] = pcr.u_value;
							breaktype = STD_BREAK;
							await_breakpoint = TRUE;
							exit_debugger = TRUE;
						}
						break;

			case 0x86:	break; /* Quietly ignore F10 key. */

			default:	beep();
						new_window(WIN_DIALOGUE);
						if ((left_window == HELP1) || (left_window == HELP2))
							cprintf("No such command.");
						else
							cprintf("Press H for help.");
						break;
		}
	}

	/* Restore the Dragon video mode. */
	memory[0xff22]	= deb_saved_ff22;
	new_vmode		= deb_saved_new_vmode;
	new_screen_size	= deb_saved_new_screen_size;
	new_screen_base	= deb_saved_new_screen_base;

	/* Draw and refresh the Dragon screen if we are leaving the debugger. */
	if (!in_debug || await_breakpoint)
	{
		drag_x++;
		debug_disp		= FALSE;
		video_enabled	= TRUE;
		restore_vmode();
	}
	else
		new_window(WIN_DRAGON);

	if (new_int9_set)
		Set_New_Int9();

	this_time = 0;
}
