/* dfa.c - determinisitic extended regexp routines for GNU
   Copyright (C) 1988 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* Written June, 1988 by Mike Haertel
   Modified July, 1988 by Arthur David Olson to assist BMG speedups  */

#include <stdio.h>
#include <assert.h>

#if defined(USG) || defined(STDC_HEADERS) || defined(VMS)
#include <string.h>
#ifndef index
#define index strchr
#endif
#else
#include <strings.h>
#endif

#include "dfa.h"

#if __STDC__
typedef void *ptr_t;
#else
typedef char *ptr_t;
#endif

static void	regmust();

static ptr_t
xcalloc(n, s)
     int n;
     size_t s;
{
  ptr_t r = calloc(n, s);

  if (!r)
    regerror("Memory exhausted");
  return r;
}

ptr_t				/* Not static, so alloca.o can use it.  */
xmalloc(n)
     size_t n;
{
  ptr_t r = malloc(n);

  assert(n != 0);
  if (!r)
    regerror("Memory exhausted");
  return r;
}

#ifdef VMS
ptr_t				/* Not static, so shell_mung can use it. */
#else
static ptr_t
#endif
xrealloc(p, n)
     ptr_t p;
     size_t n;
{
  ptr_t r = realloc(p, n);

  assert(n != 0);
  if (!r)
    regerror("Memory exhausted");
  return r;
}

#define CALLOC(p, t, n) ((p) = (t *) xcalloc((n), sizeof (t)))
#define MALLOC(p, t, n) ((p) = (t *) xmalloc((n) * sizeof (t)))
#define REALLOC(p, t, n) ((p) = (t *) xrealloc((ptr_t) (p), (n) * sizeof (t)))

/* Reallocate an array of type t if nalloc is too small for index. */
#define REALLOC_IF_NECESSARY(p, t, nalloc, index) \
  if ((index) >= (nalloc))			  \
    {						  \
      while ((index) >= (nalloc))		  \
	(nalloc) *= 2;				  \
      REALLOC(p, t, nalloc);			  \
    }

#ifdef DEBUG
#include <stdio.h>

static void
prtok(t)
     _token t;
{
  char *s;

  if (t < 0)
    fprintf(stderr, "END");
  else if (t < _NOTCHAR)
    fprintf(stderr, "%c", t);
  else
    {
      switch (t)
	{
	case _EMPTY: s = "EMPTY"; break;
	case _BACKREF: s = "BACKREF"; break;
	case _BEGLINE: s = "BEGLINE"; break;
	case _ALLBEGLINE: s = "ALLBEGLINE"; break;
	case _ENDLINE: s = "ENDLINE"; break;
	case _ALLENDLINE: s = "ALLENDLINE"; break;
	case _BEGWORD: s = "BEGWORD"; break;
	case _ENDWORD: s = "ENDWORD"; break;
	case _LIMWORD: s = "LIMWORD"; break;
	case _NOTLIMWORD: s = "NOTLIMWORD"; break;
	case _QMARK: s = "QMARK"; break;
	case _STAR: s = "STAR"; break;
	case _PLUS: s = "PLUS"; break;
	case _CAT: s = "CAT"; break;
	case _OR: s = "OR"; break;
	case _LPAREN: s = "LPAREN"; break;
	case _RPAREN: s = "RPAREN"; break;
	default: s = "SET"; break;
	}
      fprintf(stderr, "%s", s);
    }
}
#endif /* DEBUG */

/* Stuff pertaining to charsets. */

static int
tstbit(b, c)
     int b;
     _charset c;
{
  return c[b / INTBITS] & 1 << b % INTBITS;
}

static void
setbit(b, c)
     int b;
     _charset c;
{
  c[b / INTBITS] |= 1 << b % INTBITS;
}

static void
clrbit(b, c)
     int b;
     _charset c;
{
  c[b / INTBITS] &= ~(1 << b % INTBITS);
}

static void
copyset(src, dst)
     const _charset src;
     _charset dst;
{
  int i;

  for (i = 0; i < _CHARSET_INTS; ++i)
    dst[i] = src[i];
}

static void
zeroset(s)
     _charset s;
{
  int i;

  for (i = 0; i < _CHARSET_INTS; ++i)
    s[i] = 0;
}

static void
notset(s)
     _charset s;
{
  int i;

  for (i = 0; i < _CHARSET_INTS; ++i)
    s[i] = ~s[i];
}

static int
equal(s1, s2)
     const _charset s1;
     const _charset s2;
{
  int i;

  for (i = 0; i < _CHARSET_INTS; ++i)
    if (s1[i] != s2[i])
      return 0;
  return 1;
}

/* A pointer to the current regexp is kept here during parsing. */
static struct regexp *reg;

/* Find the index of charset s in reg->charsets, or allocate a new charset. */
static int
charset_index(s)
     const _charset s;
{
  int i;

  for (i = 0; i < reg->cindex; ++i)
    if (equal(s, reg->charsets[i]))
      return i;
  REALLOC_IF_NECESSARY(reg->charsets, _charset, reg->calloc, reg->cindex);
  ++reg->cindex;
  copyset(s, reg->charsets[i]);
  return i;
}

/* Syntax bits controlling the behavior of the lexical analyzer. */
static syntax_bits, syntax_bits_set;

/* Flag for case-folding letters into sets. */
static case_fold;

/* Entry point to set syntax options. */
void
regsyntax(bits, fold)
     int bits;
     int fold;
{
  syntax_bits_set = 1;
  syntax_bits = bits;
  case_fold = fold;
}

/* Lexical analyzer. */
static const char *lexstart;	/* Pointer to beginning of input string. */
static const char *lexptr;	/* Pointer to next input character. */
static lexleft;			/* Number of characters remaining. */
static caret_allowed;		/* True if backward context allows ^
				   (meaningful only if RE_CONTEXT_INDEP_OPS
				   is turned off). */
static closure_allowed;		/* True if backward context allows closures
				   (meaningful only if RE_CONTEXT_INDEP_OPS
				   is turned off). */

/* Note that characters become unsigned here. */
#define FETCH(c, eoferr)   	      \
  {			   	      \
    if (! lexleft)	   	      \
      if (eoferr)	   	      \
	regerror(eoferr);  	      \
      else		   	      \
	return _END;	   	      \
    (c) = (unsigned char) *lexptr++;  \
    --lexleft;		   	      \
  }

static _token
lex()
{
  _token c, c2;
  int invert;
  _charset cset;

  FETCH(c, (char *) 0);
  switch (c)
    {
    case '^':
      if (! (syntax_bits & RE_CONTEXT_INDEP_OPS)
	  && (!caret_allowed ||
	      (syntax_bits & RE_TIGHT_VBAR) && lexptr - 1 != lexstart))
	goto normal_char;
      caret_allowed = 0;
      return syntax_bits & RE_TIGHT_VBAR ? _ALLBEGLINE : _BEGLINE;

    case '$':
      if (syntax_bits & RE_CONTEXT_INDEP_OPS || !lexleft
	  || (! (syntax_bits & RE_TIGHT_VBAR)
	      && ((syntax_bits & RE_NO_BK_PARENS
		   ? lexleft > 0 && *lexptr == ')'
		   : lexleft > 1 && *lexptr == '\\' && lexptr[1] == ')')
		  || (syntax_bits & RE_NO_BK_VBAR
		      ? lexleft > 0 && *lexptr == '|'
		      : lexleft > 1 && *lexptr == '\\' && lexptr[1] == '|'))))
	return syntax_bits & RE_TIGHT_VBAR ? _ALLENDLINE : _ENDLINE;
      goto normal_char;

    case '\\':
      FETCH(c, "Unfinished \\ quote");
      switch (c)
	{
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	case '8':
	case '9':
	  caret_allowed = 0;
	  closure_allowed = 1;
	  return _BACKREF;

	case '<':
	  caret_allowed = 0;
	  return _BEGWORD;

	case '>':
	  caret_allowed = 0;
	  return _ENDWORD;

	case 'b':
	  caret_allowed = 0;
	  return _LIMWORD;

	case 'B':
	  caret_allowed = 0;
	  return _NOTLIMWORD;

	case 'w':
	case 'W':
	  zeroset(cset);
	  for (c2 = 0; c2 < _NOTCHAR; ++c2)
	    if (ISALNUM(c2))
	      setbit(c2, cset);
	  if (c == 'W')
	    notset(cset);
	  caret_allowed = 0;
	  closure_allowed = 1;
	  return _SET + charset_index(cset);

	case '?':
	  if (syntax_bits & RE_BK_PLUS_QM)
	    goto qmark;
	  goto normal_char;

	case '+':
	  if (syntax_bits & RE_BK_PLUS_QM)
	    goto plus;
	  goto normal_char;

	case '|':
	  if (! (syntax_bits & RE_NO_BK_VBAR))
	    goto or;
	  goto normal_char;

	case '(':
	  if (! (syntax_bits & RE_NO_BK_PARENS))
	    goto lparen;
	  goto normal_char;

	case ')':
	  if (! (syntax_bits & RE_NO_BK_PARENS))
	    goto rparen;
	  goto normal_char;

	default:
	  goto normal_char;
	}

    case '?':
      if (syntax_bits & RE_BK_PLUS_QM)
	goto normal_char;
    qmark:
      if (! (syntax_bits & RE_CONTEXT_INDEP_OPS) && !closure_allowed)
	goto normal_char;
      return _QMARK;

    case '*':
      if (! (syntax_bits & RE_CONTEXT_INDEP_OPS) && !closure_allowed)
	goto normal_char;
      return _STAR;

    case '+':
      if (syntax_bits & RE_BK_PLUS_QM)
	goto normal_char;
    plus:
      if (! (syntax_bits & RE_CONTEXT_INDEP_OPS) && !closure_allowed)
	goto normal_char;
      return _PLUS;

    case '|':
      if (! (syntax_bits & RE_NO_BK_VBAR))
	goto normal_char;
    or:
      caret_allowed = 1;
      closure_allowed = 0;
      return _OR;

    case '\n':
      if (! (syntax_bits & RE_NEWLINE_OR))
	goto normal_char;
      goto or;

    case '(':
      if (! (syntax_bits & RE_NO_BK_PARENS))
	goto normal_char;
    lparen:
      caret_allowed = 1;
      closure_allowed = 0;
      return _LPAREN;

    case ')':
      if (! (syntax_bits & RE_NO_BK_PARENS))
	goto normal_char;
    rparen:
      caret_allowed = 0;
      closure_allowed = 1;
      return _RPAREN;

    case '.':
      zeroset(cset);
      notset(cset);
      clrbit('\n', cset);
      caret_allowed = 0;
      closure_allowed = 1;
      return _SET + charset_index(cset);

    case '[':
      zeroset(cset);
      FETCH(c, "Unbalanced [");
      if (c == '^')
	{
	  FETCH(c, "Unbalanced [");
	  invert = 1;
	}
      else
	invert = 0;
      do
	{
	  FETCH(c2, "Unbalanced [");
	  if (c2 == '-')
	    {
	      FETCH(c2, "Unbalanced [");
	      while (c <= c2)
		{
		  setbit(c, cset);
		  if (case_fold)
		    if (ISUPPER(c))
		      setbit(tolower(c), cset);
		    else if (ISLOWER(c))
		      setbit(toupper(c), cset);
		  ++c;
		}
	      FETCH(c, "Unbalanced [");
	    }
	  else
	    {
	      setbit(c, cset);
	      if (case_fold)
		if (ISUPPER(c))
		  setbit(tolower(c), cset);
		else if (ISLOWER(c))
		  setbit(toupper(c), cset);
	      c = c2;
	    }
	}
      while (c != ']');
      if (invert)
	notset(cset);
      caret_allowed = 0;
      closure_allowed = 1;
      return _SET + charset_index(cset);

    default:
    normal_char:
      caret_allowed = 0;
      closure_allowed = 1;
      if (case_fold && ISALPHA(c))
	{
	  zeroset(cset);
	  if (isupper(c))
	    c = tolower(c);
	  setbit(c, cset);
	  setbit(toupper(c), cset);
	  return _SET + charset_index(cset);
	}
      return c;
    }
}

/* Recursive descent parser for regular expressions. */

static _token tok;		/* Lookahead token. */
static depth;			/* Current depth of a hypothetical stack
				   holding deferred productions.  This is
				   used to determine the depth that will be
				   required of the real stack later on in
				   reganalyze(). */

/* Add the given token to the parse tree, maintaining the depth count and
   updating the maximum depth if necessary. */
static void
addtok(t)
     _token t;
{
  REALLOC_IF_NECESSARY(reg->tokens, _token, reg->talloc, reg->tindex);
  reg->tokens[reg->tindex++] = t;

  switch (t)
    {
    case _QMARK:
    case _STAR:
    case _PLUS:
      break;

    case _CAT:
    case _OR:
      --depth;
      break;

    default:
      ++reg->nleaves;
    case _EMPTY:
      ++depth;
      break;
    }
  if (depth > reg->depth)
    reg->depth = depth;
}

/* The grammar understood by the parser is as follows.

   start:
     regexp
     _ALLBEGLINE regexp
     regexp _ALLENDLINE
     _ALLBEGLINE regexp _ALLENDLINE

   regexp:
     regexp _OR branch
     branch

   branch:
     branch closure
     closure

   closure:
     closure _QMARK
     closure _STAR
     closure _PLUS
     atom

   atom:
     <normal character>
     _SET
     _BACKREF
     _BEGLINE
     _ENDLINE
     _BEGWORD
     _ENDWORD
     _LIMWORD
     _NOTLIMWORD
     <empty>

   The parser builds a parse tree in postfix form in an array of tokens. */

#if __STDC__
static void regexp(void);
#else
static void regexp();
#endif

static void
atom()
{
  if (tok >= 0 && tok < _NOTCHAR || tok >= _SET || tok == _BACKREF
      || tok == _BEGLINE || tok == _ENDLINE || tok == _BEGWORD
      || tok == _ENDWORD || tok == _LIMWORD || tok == _NOTLIMWORD)
    {
      addtok(tok);
      tok = lex();
    }
  else if (tok == _LPAREN)
    {
      tok = lex();
      regexp();
      if (tok != _RPAREN)
	regerror("Unbalanced (");
      tok = lex();
    }
  else
    addtok(_EMPTY);
}

static void
closure()
{
  atom();
  while (tok == _QMARK || tok == _STAR || tok == _PLUS)
    {
      addtok(tok);
      tok = lex();
    }
}

static void
branch()
{
  closure();
  while (tok != _RPAREN && tok != _OR && tok != _ALLENDLINE && tok >= 0)
    {
      closure();
      addtok(_CAT);
    }
}

static void
regexp()
{
  branch();
  while (tok == _OR)
    {
      tok = lex();
      branch();
      addtok(_OR);
    }
}

/* Main entry point for the parser.  S is a string to be parsed, len is the
   length of the string, so s can include NUL characters.  R is a pointer to
   the struct regexp to parse into. */
void
regparse(s, len, r)
     const char *s;
     size_t len;
     struct regexp *r;
{
  reg = r;
  lexstart = lexptr = s;
  lexleft = len;
  caret_allowed = 1;
  closure_allowed = 0;

  if (! syntax_bits_set)
    regerror("No syntax specified");

  tok = lex();
  depth = r->depth;

  if (tok == _ALLBEGLINE)
    {
      addtok(_BEGLINE);
      tok = lex();
      regexp();
      addtok(_CAT);
    }
  else
    regexp();

  if (tok == _ALLENDLINE)
    {
      addtok(_ENDLINE);
      addtok(_CAT);
      tok = lex();
    }

  if (tok != _END)
    regerror("Unbalanced )");

  addtok(_END - r->nregexps);
  addtok(_CAT);

  if (r->nregexps)
    addtok(_OR);

  ++r->nregexps;
}

/* Some primitives for operating on sets of positions. */

/* Copy one set to another; the destination must be large enough. */
static void
copy(src, dst)
     const _position_set *src;
     _position_set *dst;
{
  int i;

  for (i = 0; i < src->nelem; ++i)
    dst->elems[i] = src->elems[i];
  dst->nelem = src->nelem;
}

/* Insert a position in a set.  Position sets are maintained in sorted
   order according to index.  If position already exists in the set with
   the same index then their constraints are logically or'd together.
   S->elems must point to an array large enough to hold the resulting set. */
static void
insert(p, s)
     _position p;
     _position_set *s;
{
  int i;
  _position t1, t2;

  for (i = 0; i < s->nelem && p.index < s->elems[i].index; ++i)
    ;
  if (i < s->nelem && p.index == s->elems[i].index)
    s->elems[i].constraint |= p.constraint;
  else
    {
      t1 = p;
      ++s->nelem;
      while (i < s->nelem)
	{
	  t2 = s->elems[i];
	  s->elems[i++] = t1;
	  t1 = t2;
	}
    }
}

/* Merge two sets of positions into a third.  The result is exactly as if
   the positions of both sets were inserted into an initially empty set. */
static void
merge(s1, s2, m)
     _position_set *s1;
     _position_set *s2;
     _position_set *m;
{
  int i = 0, j = 0;

  m->nelem = 0;
  while (i < s1->nelem && j < s2->nelem)
    if (s1->elems[i].index > s2->elems[j].index)
      m->elems[m->nelem++] = s1->elems[i++];
    else if (s1->elems[i].index < s2->elems[j].index)
      m->elems[m->nelem++] = s2->elems[j++];
    else
      {
	m->elems[m->nelem] = s1->elems[i++];
	m->elems[m->nelem++].constraint |= s2->elems[j++].constraint;
      }
  while (i < s1->nelem)
    m->elems[m->nelem++] = s1->elems[i++];
  while (j < s2->nelem)
    m->elems[m->nelem++] = s2->elems[j++];
}

/* Delete a position from a set. */
static void
delete(p, s)
     _position p;
     _position_set *s;
{
  int i;

  for (i = 0; i < s->nelem; ++i)
    if (p.index == s->elems[i].index)
      break;
  if (i < s->nelem)
    for (--s->nelem; i < s->nelem; ++i)
      s->elems[i] = s->elems[i + 1];
}

/* Find the index of the state corresponding to the given position set with
   the given preceding context, or create a new state if there is no such
   state.  Newline and letter tell whether we got here on a newline or
   letter, respectively. */
static int
state_index(r, s, newline, letter)
     struct regexp *r;
     _position_set *s;
     int newline;
     int letter;
{
  int hash = 0;
  int constraint;
  int i, j;

  newline = newline ? 1 : 0;
  letter = letter ? 1 : 0;

  for (i = 0; i < s->nelem; ++i)
    hash ^= s->elems[i].index + s->elems[i].constraint;

  /* Try to find a state that exactly matches the proposed one. */
  for (i = 0; i < r->sindex; ++i)
    {
      if (hash != r->states[i].hash || s->nelem != r->states[i].elems.nelem
	  || newline != r->states[i].newline || letter != r->states[i].letter)
	continue;
      for (j = 0; j < s->nelem; ++j)
	if (s->elems[j].constraint
	    != r->states[i].elems.elems[j].constraint
	    || s->elems[j].index != r->states[i].elems.elems[j].index)
	  break;
      if (j == s->nelem)
	return i;
    }

  /* We'll have to create a new state. */
  REALLOC_IF_NECESSARY(r->states, _dfa_state, r->salloc, r->sindex);
  r->states[i].hash = hash;
  MALLOC(r->states[i].elems.elems, _position, s->nelem);
  copy(s, &r->states[i].elems);
  r->states[i].newline = newline;
  r->states[i].letter = letter;
  r->states[i].backref = 0;
  r->states[i].constraint = 0;
  r->states[i].first_end = 0;
  for (j = 0; j < s->nelem; ++j)
    if (r->tokens[s->elems[j].index] < 0)
      {
	constraint = s->elems[j].constraint;
	if (_SUCCEEDS_IN_CONTEXT(constraint, newline, 0, letter, 0)
	    || _SUCCEEDS_IN_CONTEXT(constraint, newline, 0, letter, 1)
	    || _SUCCEEDS_IN_CONTEXT(constraint, newline, 1, letter, 0)
	    || _SUCCEEDS_IN_CONTEXT(constraint, newline, 1, letter, 1))
	  r->states[i].constraint |= constraint;
	if (! r->states[i].first_end)
	  r->states[i].first_end = r->tokens[s->elems[j].index];
      }
    else if (r->tokens[s->elems[j].index] == _BACKREF)
      {
	r->states[i].constraint = _NO_CONSTRAINT;
	r->states[i].backref = 1;
      }

  ++r->sindex;

  return i;
}

/* Find the epsilon closure of a set of positions.  If any position of the set
   contains a symbol that matches the empty string in some context, replace
   that position with the elements of its follow labeled with an appropriate
   constraint.  Repeat exhaustively until no funny positions are left.
   S->elems must be large enough to hold the result. */
void
epsclosure(s, r)
     _position_set *s;
     struct regexp *r;
{
  int i, j;
  int *visited;
  _position p, old;

  MALLOC(visited, int, r->tindex);
  for (i = 0; i < r->tindex; ++i)
    visited[i] = 0;

  for (i = 0; i < s->nelem; ++i)
    if (r->tokens[s->elems[i].index] >= _NOTCHAR
	&& r->tokens[s->elems[i].index] != _BACKREF
	&& r->tokens[s->elems[i].index] < _SET)
      {
	old = s->elems[i];
	p.constraint = old.constraint;
	delete(s->elems[i], s);
	if (visited[old.index])
	  {
	    --i;
	    continue;
	  }
	visited[old.index] = 1;
	switch (r->tokens[old.index])
	  {
	  case _BEGLINE:
	    p.constraint &= _BEGLINE_CONSTRAINT;
	    break;
	  case _ENDLINE:
	    p.constraint &= _ENDLINE_CONSTRAINT;
	    break;
	  case _BEGWORD:
	    p.constraint &= _BEGWORD_CONSTRAINT;
	    break;
	  case _ENDWORD:
	    p.constraint &= _ENDWORD_CONSTRAINT;
	    break;
	  case _LIMWORD:
	    p.constraint &= _LIMWORD_CONSTRAINT;
	    break;
	  case _NOTLIMWORD:
	    p.constraint &= _NOTLIMWORD_CONSTRAINT;
	    break;
	  default:
	    break;
	  }
	for (j = 0; j < r->follows[old.index].nelem; ++j)
	  {
	    p.index = r->follows[old.index].elems[j].index;
	    insert(p, s);
	  }
	/* Force rescan to start at the beginning. */
	i = -1;
      }

  free(visited);
}

/* Perform bottom-up analysis on the parse tree, computing various functions.
   Note that at this point, we're pretending constructs like \< are real
   characters rather than constraints on what can follow them.

   Nullable:  A node is nullable if it is at the root of a regexp that can
   match the empty string.
   *  _EMPTY leaves are nullable.
   * No other leaf is nullable.
   * A _QMARK or _STAR node is nullable.
   * A _PLUS node is nullable if its argument is nullable.
   * A _CAT node is nullable if both its arguments are nullable.
   * An _OR node is nullable if either argument is nullable.

   Firstpos:  The firstpos of a node is the set of positions (nonempty leaves)
   that could correspond to the first character of a string matching the
   regexp rooted at the given node.
   * _EMPTY leaves have empty firstpos.
   * The firstpos of a nonempty leaf is that leaf itself.
   * The firstpos of a _QMARK, _STAR, or _PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos of its_PLUS node is the firstpos                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            