
/*
** Written by Tim Endres, time@ice.com
**
** This source may be used in any fashion that pleases you.
** I would be flattered if you left my notice here.
**
** I love hacking, I love the Internet, I am sorry.
**
** Daddy misses you Sky Marie...
*/

/*
** upsl0
** Bring up the SLIP link (for A/UX). This will dial and connect
** via a simple expect/send script, perform slattach, ifconfig,
** and route to complete your network instantiation needs.
*/

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <setjmp.h>
#include <termio.h>
#include <fcntl.h>
#include <signal.h>
#include <syslog.h>
#include <stdio.h>
#include <errno.h>

char	*progname;

char	*device_name = "/dev/tty0";
char	*script_name = "/etc/slip.dial";
char	*baud_rate = "19200";
int		do_connect = 1;
int		do_attach = 1;
int		do_config = 1;
int		do_route = 1;
int		debug_level = 0;

char	*attach_path = "/etc/slattach";
char	*config_path = "/etc/ifconfig";
char	*route_path = "/usr/etc/route";
char	*route_metric = "2";

char	*host_name = "client";
char	*gateway_name = "server";

char	*interface = "sl0";
char	*netmask = "0xFFFFFF00";

int		io_min_chars=1;	/* Min characters to satisfy read. */
int		io_min_time=1;	/* Min 1/10 secs interchar to sate read. */
int		max_connect_time = 60;
int		sleep_time = 1;

char	linebuf[512];

int		send_len;
char	send[512];
int		expect_len;
char	expect[512];
int		resend_len;
char	resend[512];

char	inbuf[512];

char	*search_ptr;
char	*search_limit;
char	*search_append;
char	search_buf[2048];

char	uu_resend_str[1024];
int		uu_resend_delay = 4;
int		uu_break_duration = 0;		/* NO BREAK, unless requested!!! */


#define OK       0
#define TIMEOUT  -1     /* -1 is returned by readbyte() upon timeout */
#define ERROR    -2
#define RETRYMAX 10

#define TRUE	1
#define FALSE	0

/* globals */

/* locals */

#define DEBUG(a, b) ((debug_level >= a) ? fprintf b, a : 0)

extern int		optind, opterr;
extern char		*optarg;
char			*filename;

char			*printable_data();

usage()
	{
	fprintf(stderr, "usage: %s [options...] client_hostname server_hostname\n", progname);
	fprintf(stderr, "Options:\n");
	fprintf(stderr, "       [-d device] [-b baud] [-t timeout] [-n only dial]\n");
	fprintf(stderr, "       [-D dbglvl] [-m route_metric] [-M MIN_CHARS] [-T MIN_TIME]\n");
	fprintf(stderr, "       [-C ifconfig_path] [-A slattach_path] [-S script_path]\n");
	fprintf(stderr, "       [-N no dial] [-a no attach] [-c no ifconfig] [-r no route] [-s sleep_time]\n");
	}

main(argc, argv)
int		argc;
char	*argv[];
	{
	int		c, i, bytes, devfd, result;
	time_t	now;
	char	buf[4], *nameptr;
	int		opt_char;
	int		opt_err_flag = 0;

	progname = argv[0];

	while ((opt_char=getopt(argc, argv, "aA:b:cC:d:D:m:M:Nnrs:S:T:t:")) != EOF)
		switch (opt_char) {
			case 'r':
				do_route = 0;
				break;
			case 'c':
				do_config = 0;
				break;
			case 'a':
				do_attach = 0;
				break;
			case 'b':
				baud_rate = optarg;
				break;
			case 'd':
				device_name = optarg;
				break;
			case 'N':
				do_connect = 0;
				break;
			case 'n':
				do_attach = 0;
				do_config = 0;
				do_route = 0;
				break;
			case 'A':
				attach_path = optarg;
				break;
			case 'C':
				config_path = optarg;
				break;
			case 's':
				sleep_time = atoi(optarg);
				break;
			case 't':
				max_connect_time = atoi(optarg);
				break;
			case 'm':
				route_metric = optarg;
				break;
			case 'M':
				io_min_chars = atoi(optarg);
				break;
			case 'T':
				io_min_time = atoi(optarg);
				break;
			case 'S':
				script_name = optarg;
				break;
			case 'D':
				debug_level = atoi(optarg);
				break;
			case 'u':
			case '?':
			default:
				opt_err_flag++;
				break;
			}

	if (opt_err_flag > 0)
		{
		fprintf(stderr, "%s: bad option\n", progname);
		usage();
		exit(1);
		}

	if (argc != optind + 2)
		{
		fprintf(stderr, "%s: not enough arguments\n", progname);
		usage();
		exit(1);
		}

	host_name = argv[optind++];
	gateway_name = argv[optind];

	if (do_connect)
		{
		result = run_connect_script(max_connect_time);
		}
	else
		{
		result = 0;
		}

	if (result != -1)
		{
		sleep(sleep_time);

		if (do_attach)
			{
			result = do_slattach();
			if (debug_level >= 1)
				fprintf(stderr, "ATTACH returns %d.\n", result);
			}

		if (do_config)
			{
			result = do_ifconfig();
			if (debug_level >= 1)
				fprintf(stderr, "CONFIG returns %d.\n", result);
			}

		if (do_route)
			{
			result = add_route();
			if (debug_level >= 1)
				fprintf(stderr, "ADD ROUTE returns %d.\n", result);
			}
		}
	
	exit(0);
	}

run_connect_script(timeout)
int		timeout;
	{
	int		devfd, ch, i, length;
	int		found, result = 0;
	time_t	end_time;
	FILE	*sfp;

	sfp = fopen(script_name, "r");
	if (sfp == NULL)
		{
		fprintf(stderr, "error #%d opening script file '%s'\n",
					errno, script_name);
		return -1;
		}

	/*
	** The O_NDELAY prevents us from blocking until the
	** CARRIER DETECT line is high, which won't happen
	** until we are connected, which won't happen until
	** *after* this open().
	*/
	devfd = open(device_name, O_RDWR | O_NDELAY, 0666);
	if (devfd == -1)
		{
		fprintf(stderr, "error #%d opening '%s'\n", errno, device_name);
		fclose(sfp);
		return -1;
		}

	/* do the equivalent of "stty -n device -modem"... */
	clocal_mode(devfd);

	close(devfd);

	/* NOW without O_NDELAY so read() blocks more appropriately... */
	devfd = open(device_name, O_RDWR, 0666);
	if (devfd == -1)
		{
		fprintf(stderr, "error #%d opening '%s'\n", errno, device_name);
		fclose(sfp);
		return -1;
		}

	script_mode(devfd);

	setbaud(devfd, baud_rate);

	end_time = time(0) + timeout;

	for ( ; time(0) < end_time ; )
		{
		if (fgets(inbuf, sizeof(inbuf)-2, sfp) == NULL)
			{
			if (debug_level >= 1)
				fprintf(stderr, "end of script on expect, SUCCESS\n");
			break;
			}
		
		if (debug_level >= 5)
			fprintf(stderr, "INLINE [%d]<%s>\n", strlen(inbuf), inbuf);
		
		length = strlen(inbuf);
		if (inbuf[length-1] == '\n')
			inbuf[length-1] = '\0';
		
		if (inbuf[0] != '\0')
			{
			filter_expect(expect, inbuf);
			expect_len = strlen(expect);
			
			if (debug_level >= 3)
				fprintf(stderr, "EXPECT <%s>\n", printable_data(expect));

			search_ptr = search_buf;
			search_append = search_buf;
			search_limit = &search_buf[sizeof(search_buf) - 1];

			for (found = 0 ; ! found && time(0) < end_time ; )
				{
				ch = readbyte(devfd, 2);
				
				if (debug_level >= 5)
					fprintf(stderr, "readbyte(%d) -> x%02X <%c>\n",
							devfd, ch, ch);
		
				if (ch != TIMEOUT)
					{
					if (search_append >= search_limit - 1)
						{
						char	*ptr;
						
						for ( ptr = search_buf ; ; )
							{
							*ptr++ = *search_ptr++;
							if (search_ptr >= search_append)
								break;
							}
						
						*ptr = '\0';
						search_append = ptr;
						search_ptr = search_buf;
						}
					
					*search_append++ = ch;
					*search_append = '\0';
					
					/* now check the expect... */
					for ( ; (search_ptr + expect_len) <= search_append ; )
						{
						if (debug_level >= 5)
							{
							fprintf(stderr, "CHECK: [%d]<%s> ",
									expect_len, printable_data(expect));
							fprintf(stderr, "CHECK: <%s>\n",
									printable_data(search_ptr));
							}
						
						if (strncmp(expect, search_ptr, expect_len) == 0)
							{
							if (debug_level >= 1)
								fprintf(stderr, "FOUND: <%s>\n",
										printable_data(expect));
							found = 1;
							break;
							}
						else
							{
							++search_ptr;
							}
						}
					}
				}
			
			if (! found)
				{
				fprintf(stderr, "TIMEOUT expecting <%s>\n", expect);
				close(devfd);
				fclose(sfp);
				reset_mode(devfd);
				return -1;
				}
			}
		else
			{
			if (debug_level >= 3)
				fprintf(stderr, "EXPECT <empty>\n");
			}

		if (fgets(inbuf, sizeof(inbuf)-2, sfp) == NULL)
			{
			if (debug_level >= 1)
				fprintf(stderr, "SUCCESS end of script on send.\n");
			break;
			}
		else
			{
			if (debug_level >= 5)
				fprintf(stderr, "INLINE [%d]<%s>\n", strlen(inbuf), inbuf);

			length = strlen(inbuf);
			if (inbuf[length-1] == '\n')
				inbuf[length-1] = '\0';

			if (inbuf[0] != '\0')
				{
				filter_send(send, inbuf);
				send_len = strlen(send);
				
				if (debug_level >= 3)
					fprintf(stderr, "SEND <%s>\n", printable_data(send));
				
				/*bytes =*/ write(devfd, send, send_len);
				}
			else
				{
				if (debug_level >= 3)
					fprintf(stderr, "SEND <empty>\n");
				}
			}
		}

	reset_mode(devfd);

	close(devfd);
	fclose(sfp);

	return result;
	}

do_slattach()
	{
	int		child, result, wstatus, slow, shigh;
	char	*argv[8];
	char	cmd_buf[1024];

	sprintf(cmd_buf, "%s %s %s %s %s",
			attach_path, device_name, host_name, gateway_name, baud_rate);

	if (debug_level >= 3)
		fprintf(stderr, "exec: %s\n", cmd_buf);
	
	child = fork();
	
	switch (child)
		{
		case 0:		/* CHILD */

			/*
			** A/UX slattach has a bug in that it does not
			** detach itself from the controling tty, and
			** thus signals (such as Ctl-C to ping) cause
			** slattach to quit. We will compensate for this.
			*/
			disassociate();

			argv[0] = attach_path;
			argv[1] = device_name;
			argv[2] = host_name;
			argv[3] = gateway_name;
			argv[4] = baud_rate;
			argv[5] = (char *)0;
			
			execv(attach_path, argv);
			break;
		
		case -1:	/* DEATH */
			fprintf(stderr, "fatal - fork() fails with %d\n", errno);
			abort();
			break;
		
		default:	/* PARENT */
			result = wait(&wstatus);
			slow = (wstatus & 0x0008);
			shigh = ((wstatus >> 8) & 0x0008);
			if (debug_level >= 3)
				fprintf(stderr, "ATTACH '%s'[pid=%d] returns %03o %03o).\n",
						attach_path, result,
						(wstatus & 0x08), ((wstatus >> 8) & 0x0008) );
			else if (debug_level >= 1)
				fprintf(stderr, "ATTACH %s.\n",
						((slow==0 && shigh==0) ? "SUCCEEDED" : "FAILED") );
			
			result = (slow == 0) ? shigh : ( (slow == 0177) ? shigh : slow );
			break;
		}
	
	return result;
	}

do_ifconfig()
	{
	int		child, result, wstatus, slow, shigh;
	char	*argv[8];
	char	cmd_buf[1024];

	sprintf(cmd_buf, "%s %s netmask %s",
				config_path, interface, netmask);
	if (debug_level >= 3)
		fprintf(stderr, "exec: %s\n", cmd_buf);
	
	child = fork();
	
	switch (child)
		{
		case 0:		/* CHILD */
			argv[0] = config_path;
			argv[1] = interface;
			argv[2] = "netmask";
			argv[3] = netmask;
			argv[4] = (char *)0;
			
			execv(config_path, argv);
			break;
		
		case -1:	/* DEATH */
			fprintf(stderr, "fatal - fork() fails with %d\n", errno);
			abort();
			break;
		
		default:	/* PARENT */
			result = wait(&wstatus);
			if (debug_level >= 3)
				fprintf(stderr, "CONFIG '%s'[pid=%d] returns %03o %03o.\n",
						attach_path, result,
						(wstatus & 0x08), ((wstatus >> 8) & 0x0008) );
			else if (debug_level >= 1)
				fprintf(stderr, "CONFIG %s.\n",
						((slow==0 && shigh==0) ? "SUCCEEDED" : "FAILED") );

			result = (slow == 0) ? shigh : ( (slow == 0177) ? shigh : slow );
			break;
		}
	
	return result;
	}

add_route()
	{
	int		child, result, wstatus, slow, shigh;
	char	*argv[8];
	char	cmd_buf[1024];

	sprintf(cmd_buf, "%s add default %s %s",
			route_path, gateway_name, route_metric);
	fprintf(stderr, "exec: %s\n", cmd_buf);
	
	child = fork();
	
	switch (child)
		{
		case 0:		/* CHILD */
			argv[0] = route_path;
			argv[1] = "add";
			argv[2] = "default";
			argv[3] = gateway_name;
			argv[4] = route_metric;
			argv[5] = (char *)0;
			
			execv(route_path, argv);
			break;
		
		case -1:	/* DEATH */
			fprintf(stderr, "fatal - fork() fails with %d\n", errno);
			abort();
			break;
		
		default:	/* PARENT */
			result = wait(&wstatus);
			slow = (wstatus & 0x0008);
			shigh = ((wstatus >> 8) & 0x0008);
			if (debug_level >= 3)
				fprintf(stderr, "ROUTE ADD '%s'[pid=%d] returns $03o %03o.\n",
						route_path, result,
						(wstatus & 0x08), ((wstatus >> 8) & 0x0008) );
			else if (debug_level >= 1)
				fprintf(stderr, "ROUTE ADD %s.\n",
						((slow==0 && shigh==0) ? "SUCCEEDED" : "FAILED") );
			
			result = (slow == 0) ? shigh : ( (slow == 0177) ? shigh : slow );
			break;
		}
	
	return result;
	}

/*
**  getdevline() inputs a line from the device.
**
**  Maxtime is the timeout for the first character, set to 6 seconds for
**  retries.
*/

int
getdevline(fd, buffer, maxsize, maxtime)
int		fd;
char	*buffer;
int		maxsize;
int		maxtime;
	{
	int		count = 0;
	int		c;
	char	*pbuf;

	pbuf = buffer;
	for (count=0 ; count < maxsize ; ++count)
		{
		c = readbyte(fd, maxtime);
		if (debug_level >= 7)
			fprintf(stderr, "readbyte(%d) returns %d\n", fd, c);

		if (c == TIMEOUT)
			{
			*pbuf++ = '\0';
			return TIMEOUT;
			}

		*pbuf++ = c;
		
		if (c == '\r' || c == '\n')
			{
			break;
			}
		}

	*pbuf++ = '\0';
	return count;
	}

static struct termio save_term;

int
clocal_mode(fildes)
int     fildes;
	{
	int     myerr;
	struct termio term;

	myerr = ioctl(fildes, TCGETA, &term);

	/* We do not want to block waiting for CD to be high... */
	term.c_cflag |= CLOCAL;
	
	myerr = ioctl(fildes, TCSETA, &term);
	
	return myerr;
	}

int
script_mode(fildes)
int     fildes;
	{
	int     myerr;
	struct termio term;

	myerr = ioctl(fildes, TCGETA, &term);

	memcpy(&save_term, &term, sizeof(save_term));

	term.c_iflag  = IGNBRK | IGNPAR;
	term.c_oflag  = 0;
	term.c_lflag  = 0;
	
	/* We want 8 bit mode, and we want to read... */
	term.c_cflag |= CS8 | CREAD;
	
	/* We want no parity one stop bit, and do *not* drop DTR on close()... */
	term.c_cflag &= ~(PARENB | CSTOPB | HUPCL);
	
	/* Make sure the saved termio knows about HUPCL... */
	save_term.c_cflag &= ~(HUPCL);
	
	term.c_cc[VEOF] = io_min_chars;
	term.c_cc[VEOL] = io_min_time;
	
	myerr = ioctl(fildes, TCSETA, &term);
	
	return myerr;
	}

reset_mode(fildes)
int     fildes;
	{
	int     myerr = 0;

	myerr = ioctl(fildes, TCSETA, &save_term);

	return myerr;
	}

struct
	{
	char *rate;
	int  bits;
	}
rate_table[] =
	{
	"1200",		B1200,
	"2400",		B2400,
	"9600",		B9600,
	"19200",	B19200,
	"38400",	B38400,
	NULL,		-1
	};

setbaud(fildes, rate)
int     fildes;
char	*rate;
	{
	int     i, myerr;
	struct termio term;

	for (i = 0 ; rate_table[i].rate != NULL ; ++i)
		{
		if (strcmp(rate, rate_table[i].rate) == 0)
			break;
		}
	if (rate_table[i].rate == NULL)
		{
		return;
		}

	myerr = ioctl(fildes, TCGETA, &term);

	term.c_cflag &= ~(CBAUD);
	term.c_cflag |= rate_table[i].bits;

	myerr = ioctl(fildes, TCSETA, &term);
	}


/*
    readbyte() is used as a single-character terminal input routine
*/

static
alrm()
	{
	if (debug_level >= 7)
		fprintf(stderr, "ALARM\n");
	}

int readbyte(fd, seconds)
int		fd;
int		seconds;
	{
	/* 128 is the standard XMODEM blk size */
	static char rxbuf[128], *p;
	static int count = 0;

	if (count > 0)
		{
		count--;
		return(*p++ & 0xff);
		}

	if (seconds > 0)
		{
		signal(SIGALRM, alrm);
		alarm((unsigned)seconds);
		}

	count = read(fd, p=rxbuf, 128);
	
	if (debug_level >= 10)
		fprintf(stderr, "read(%d) = %d [%d]\n",
					fd, count, errno);

	if (seconds > 0)
		alarm(0);

	if (count < 1)
		{
		return(TIMEOUT);
		}

	count--;
	return(*p++ & 0xff);
	}

sendbyte(c)
int		c;
	{
	unsigned char	och;

	och = c & 0xFF;
	write(1, &och, 1);
	}

purge(fd)
int		fd;
	{
	while(readbyte(fd, 1) != -1)
		;
	}

char *
printable_data(data)
char	*data;
	{
	char			*p, *b;
	static char		buffer[1024];

	for (p = data,b = buffer ; *p != '\0' ; ++p)
		{
		if (*p < ' ')
			{
			sprintf(b, "\\%03o", *p);
			b += strlen(b);
			}
		else if (*p > '~')
			{
			sprintf(b, "\\%03o", *p);
			b += strlen(b);
			}
		else
			{
			*b++ = *p;
			}
		}

	*b = '\0';
	return buffer;
	}

filter_expect(into, from)
char	*into;
char	*from;
	{
	filter_send_expect(into, from);
	}

filter_send(into, from)
char	*into;
char	*from;
	{
	filter_send_expect(into, from);
	}

filter_send_expect(into, from)
char	*into;
char	*from;
	{
	char	*save;

	/* uu_break_duration = 0;*/		/* NO BREAK, unless requested!!! */
	save = into;
	for ( ; *from ; ) {
		if (*from == '\\') {
			switch (*(from + 1)) {
				case '\\':
					*into++ = '\\'; from += 2;
					break;
				case 'B': case 'b':
					uu_break_duration += 4;
					from += 2;
					break;
				case 'r':
					*into++ = '\015'; from += 2;
					break;
				case 'n':
					*into++ = '\012'; from += 2;
					break;
				case 't':
					*into++ = '\011'; from += 2;
					break;
				default:
					if (isdigit(*(from+1)) &&
						isdigit(*(from+2)) &&
						isdigit(*(from+3)))
						{
						*into = ((*(from+1) - '0') * 64) +
								((*(from+2) - '0') * 8) +
								(*(from+3) - '0');
						into++; from += 4;
						}
					else {
						*into++ = *from++;
						}
					break;
				}
			}
		else {
			*into++ = *from++;
			}
		}
	
	*into = '\0';
	}

filter_resend(from)
char	*from;
	{
	char	*into;
	int		delay_set = 0;

	uu_resend_delay = 4;
	uu_break_duration = 0;		/* NO BREAK, unless requested!!! */
	uu_resend_str[0] = '\0';
	if (*from == '\0')
		return;
	
	for (into=uu_resend_str; *from ; ) {
		if (*from == '\\') {
			switch (*(from + 1)) {
				case '\\':
					*into++ = '\\'; from += 2;
					break;
				case 'B': case 'b':
					uu_break_duration += 4;
					from += 2;
					break;
				case 'r':
					*into++ = '\015'; from += 2;
					break;
				case 'd':
					if (! delay_set) {
						delay_set = 1;
						uu_resend_delay = 0;
						}
					if (isdigit(*(from + 2))) {
						uu_resend_delay += ( (*(from + 2) - '0') * 60 );
						from += 3;
						}
					else {
						uu_resend_delay += (3 * 60);	/* default */
						from += 2;
						}
					break;
				case 'n':
					*into++ = '\012'; from += 2;
					break;
				case 't':
					*into++ = '\011'; from += 2;
					break;
				default:
					if (isdigit(*(from+1)) &&
						isdigit(*(from+2)) &&
						isdigit(*(from+3)))
						{
						*into = ((*(from+1) - '0') * 64) +
								((*(from+2) - '0') * 8) +
								(*(from+3) - '0');
						into++; from += 4;
						}
					else {
						*into++ = *from++;
						}
					break;
				}
			}
		else {
			*into++ = *from++;
			}
		}
	
	*into = '\0';
	}

disassociate()
	{
	int fd;
	
	fd = open("/dev/tty", O_RDONLY);
	if (fd != -1)
		{
		ioctl(fd, TIOCNOTTY, 0);
		close(fd);
		}

	return;
	}

