/*
 * play.c -- This program plays sounds through the Macintosh built-in
 *           sound port under A/UX.
 *
 *           This software has been placed in the public domain by its
 *           author, Paul Sander (paul@sander.cupertino.ca.us).
 */

#include <stdio.h>
#include <sys/ssioctl.h>
#include <sys/sm_aux.h>

extern	int	errno;
extern	char	*sys_errlist[];
extern	char	*optarg;
extern	int	optind;
extern	int	opterr;

/* Name of sound output device files */
char	*sndReset="/dev/snd/reset";
char	*sndRaw="/dev/snd/raw";

/* Size of buffer used for transferring snd file to output */
#define BLKSIZE	1024

/* Reset sound device to its default state */
void resetSnd()
{
	FILE	*f;

	f = fopen(sndReset,"r");
	if ( f != (FILE*)NULL )
	{
		fputs("\n",f);
		fclose(f);
	}
}

/* Copy sound from f to s */
void play(f,s)
FILE	*f;
FILE	*s;
{
	char	buf[BLKSIZE];
	int	len;

	len = fread(buf,1,BLKSIZE,f);
	while ( len > 0 )
	{
		fwrite(buf,1,len,s);
		len = fread(buf,1,BLKSIZE,f);
	}
}

/* Main program */
main(argc,argv)
int	argc;
char	*argv[];
{
	FILE			*f;		/* snd file */
	FILE			*s;		/* Sound device */
	int			arg;		/* arg letter */
	unsigned int		speed;		/* sample rate */
	char			*cp;		/* scans parameter */
	struct rawSndCtl	rawSndCtl;	/* Sound device ctrl struct */

	speed = 0;	/* Assume default speed */

	/* Scan for command line arguments */
	for ( optind = 1; (arg = getopt(argc,argv,"s:")) != EOF; )
	{
		switch ( arg )
		{
	case 's':
			/* Scan speed, with optional "k" multiplier */
			speed = atoi(optarg);
			for ( cp = optarg; *cp >= '0' && *cp <= '9'; cp++ )
			{ }
			if ( strcmp(cp,"k") == 0 || strcmp(cp,"K") == 0 )
			{
				speed *= 1000;
			}
			if ( speed > 22000 )
			{
				fprintf(stderr,"Invalid speed %s\n",optarg);
				exit(1);
			}
			break;
	default:
			exit(1);
		}
	}

	resetSnd();	/* Reset sound device */

	/* Set speed if given */
	if ( speed != 0 )
	{
		/*
		 * Speed changes take affect at the time the device is
		 * closed, so we must open it once to set the speed, then
		 * close it, then play the sound.
		 */
		s = fopen(sndRaw,"w");
		if ( s == (FILE*) NULL )
		{
			fprintf(stderr,"Unable to open %s for writing:\n%s\n",
			        sndRaw, sys_errlist[errno]);
			return 1;
		}
		rawSndCtl.sampleRate = ( speed * 65536 / 22000 );
		rawSndCtl.flags = 0;
		if ( ioctl(fileno(s),SND_RAW_CTL,&rawSndCtl) < 0 )
		{
			fprintf(stderr,"Can't set speed to %d:  %s\n",
			        speed,sys_errlist[errno]);
		}
		fclose(s);
	}

	/* Open the sound device for playing the sound */
	s = fopen(sndRaw,"w");
	if ( s == (FILE*) NULL )
	{
		fprintf(stderr,"Unable to open %s for writing:\n%s\n",sndRaw,
		        sys_errlist[errno]);
		return 1;
	}

	/* Assume stdin if no files given on command line, else cat the files */
	if ( optind < argc )
	{
		while ( optind < argc )
		{
			f = fopen(argv[optind],"r");
			if ( f == (FILE*) NULL )
			{
				fprintf(stderr,
				        "Unable to read %s for reading:\n%s\n",
				        argv[optind],
				        sys_errlist[errno]);
			}
			else
			{
				play(f,s);
				fclose(f);
			}
			optind++;
		}
	}
	else
	{
		play(stdin,s);
	}
	fclose(s);
	return 0;
}

/*** End of File ***/

