/*---------------------------------------------------------------------------
 * SCSI Probe for A/UX
 *
 * Copyright (c) 1994 by Herb Weiner (herbw@wiskit.com). All rights reserved.
 * This software is provided on an AS IS basis without warranty or support of
 * any kind.  Permission is granted to distribute this software provided that
 * this entire copyright notice is retained, and provided that the package is
 * provided (including source code) without charge.  Please email bug reports
 * and enhancements to the author.  Enjoy this software, and please share it.
 *---------------------------------------------------------------------------
 */

#include <stdio.h>
#include <fcntl.h>
#include <sys/scsiccs.h>

#include "scsilib.h"

extern int			errno;
extern char			*sys_errlist[];

static char			*my_name;
static int			opt_help = 0;
static int			opt_all = 0;
static int			opt_noheader = 0;
static int			opt_product = 0;
static int			opt_revision = 0;
static int			opt_vendor = 0;
static int			opt_vp = 0;
static int			opt_vpr = 0;
static int			opt_disktab = 0;
static int			opt_diskinfo = 0;
static int			opt_version = 0;
static int			error_count = 0;

static const int	SCSI_TIMEOUT = 5;

/*---------------------------------------------------------------------------
 *	NAME
 *		extract_string ()			- Extract string from character array
 *
 *	SYNOPSIS
 *		void long extract_string (
 *		char		*result,		- Result string
 *		const char	*cp,			- Pointer to string to extract
 *		const int	length)			- Length of string
 *
 *	RETURN VALUE
 *		None
 *
 *	DESCRIPTION
 *		extract_string extracts a string from a character array.
 *---------------------------------------------------------------------------
 */

static void extract_string (char *result, const char *cp, const int length)
{
	int				i;

	for (i = 0; i < length; i++)
		result[i] = cp[i];
	result[length] = '\0';

	for (i = length - 1; i >= 0; i--)
	{
		if (result[i] == ' ')
			result[i] = '\0';
		else
			break;
	}
}

/*---------------------------------------------------------------------------
 *	NAME
 *		extract_int ()				- Extract integer from character array
 *
 *	SYNOPSIS
 *		unsigned long extract_int (
 *		const char	*cp,			- Pointer to value to extract
 *		const int	length)			- Length of value
 *
 *	RETURN VALUE
 *		Extracted value
 *
 *	DESCRIPTION
 *		extract_int extracts an integer from a character array.
 *---------------------------------------------------------------------------
 */

static unsigned long extract_int (const char *cp, const int length)
{
	unsigned long	result = 0;
	int				i;

	if ((cp == (const char *) NULL) || (length == 0))
		return (0);

	for (i = 0; i < length; i++)
		result = (result << 8) + (*cp++ & 255);

	return (result);
}

/*---------------------------------------------------------------------------
 *	NAME
 *		make_disktab ()				- Make disktab entry for SCSI Device
 *
 *	SYNOPSIS
 *		void make_disktab (
 *		int			fd,				- File Descriptor of SCSI Device
 *		const char	*name,			- Name of SCSI Device
 *		const char	*vp)			- Name of Vendor and Product
 *
 *	RETURN VALUE
 *		None
 *
 *	DESCRIPTION
 *		make_disktab generates a disktab entry for a single SCSI device.
 *---------------------------------------------------------------------------
 */

static void make_disktab (int fd, const char *name, const char *vp)
{
	typedef struct
	{
		int		page;
		char	*buffer;
		int		size;
	}
	page_struct;

	typedef struct
	{
		long	*value;
		char	*buffer;
		int		size;
		char	*disktab_keyword;
		char	*diskinfo_description;
	}
	disk_struct;

	const int	COMMAND_LENGTH = 6;
	const int	HEADER_LENGTH = 4;
	const int	BLOCK_DESCRIPTOR_LENGTH = 8;
	const int	PAGE_OFFSET = HEADER_LENGTH + BLOCK_DESCRIPTOR_LENGTH;
	char		scsi_command [COMMAND_LENGTH];
	char		page3 [PAGE_OFFSET + 24];
	char		page4 [PAGE_OFFSET + 24];
	page_struct	pages [] =
	{
				{3, page3, sizeof (page3)},
				{4, page4, sizeof (page4)},
	};
	int			number_of_pages = (sizeof (pages)) / (sizeof (page_struct));
	long		block_count;
	long		block_length;
	long		sectors;
	long		phys_length;
	long		interleave;
	long		track_skew;
	long		cyl_skew;
	long		cylinders;
	long		tracks;
	long		rpm;
	long		capacity;
	long		calculated_ns;
	disk_struct	disk_params [] =
	{
				{&capacity,		NULL,						0,	NULL,	"actual formatted capacity (KB)"},
				{&block_count,	page3 + HEADER_LENGTH + 1,	3,	NULL,	"number of blocks"},
				{&block_length,	page3 + HEADER_LENGTH + 5,	3,	NULL,	"block size"},
				{&phys_length,	page3 + PAGE_OFFSET + 12,	2,	NULL,	"sector size"},
				{&sectors,		page3 + PAGE_OFFSET + 10,	2,	NULL,	"reported sectors per track"},
				{&calculated_ns,NULL,						0,	"ns",	"calculated sectors per track"},
				{&tracks,		page4 + PAGE_OFFSET + 5,	1,	"nt",	"tracks per cylinder"},
				{&cylinders,	page4 + PAGE_OFFSET + 2,	3,	"nc",	"number of cylinders"},
				{&interleave,	page3 + PAGE_OFFSET + 14,	2,	NULL,	"interleave factor"},
				{&track_skew,	page3 + PAGE_OFFSET + 16,	2,	NULL,	"track skew"},
				{&cyl_skew,		page3 + PAGE_OFFSET + 18,	2,	NULL,	"cylinder skew"},
				{&rpm,			page4 + PAGE_OFFSET + 20,	2,	"rm",	"rotational speed (rpm)"},
	};
	int			number_of_disk_params = (sizeof (disk_params)) / (sizeof (disk_struct));
	int			i;
	int			j;
	int			bytes_read;
	scsiresult	result;
	int			error_flag = 0;

	scsi_command [0] = SOP_GETMODE;
	scsi_command [1] = 0;				/* LUN = 0; DBD = 0 */
	scsi_command [2] = 0;				/* PC = 0; Page Code = 0 */
	scsi_command [3] = 0;
	scsi_command [4] = 0;				/* Length = 0 */
	scsi_command [5] = 0;				/* Flag = 0; Link = 0 */

	for (i = 0; i < number_of_pages; i++)
	{
		for (j = 0; j < pages[i].size; j++)
			pages[i].buffer[j] = 0;

		scsi_command [2] = pages[i].page;
		scsi_command [4] = pages[i].size;

		bytes_read = scsi_read (fd, scsi_command, COMMAND_LENGTH, pages[i].buffer,
								pages[i].size, SCSI_TIMEOUT, &result);

		if (bytes_read == -1)
		{
			fprintf (stderr, "%-13.13s <%s>\n", name, sys_errlist[errno]);
			error_flag = 1;
			break;
		}
		else if (result.errno > 0)
		{
			fprintf (stderr, "%-13.13s <%s>\n",
							name, scsi_strings[result.errno]);
			error_flag = 1;
			break;
		}
	}

	if (error_flag)
		error_count++;
	else
	{
		for (i = 0; i < number_of_disk_params; i++)
			*disk_params[i].value = extract_int (disk_params[i].buffer, disk_params[i].size);
		capacity = (block_count * block_length) >> 10;
		calculated_ns = (block_count / cylinders) / tracks;

		if (opt_disktab)
		{
			fprintf (stdout, "%s:ty=winchester", vp);
			for (i = 0; i < number_of_disk_params; i++)
				if ((disk_params[i].disktab_keyword != (char *) NULL) && (*disk_params[i].value != 0))
					fprintf (stdout, ":%s#%d", disk_params[i].disktab_keyword, *disk_params[i].value);
			fprintf (stdout, "\n");
		}

		if (opt_diskinfo)
		{
			for (i = 0; i < number_of_disk_params; i++)
				if (disk_params[i].diskinfo_description != (char *) NULL)
					fprintf (stdout, "    %-30.30s %d\n", disk_params[i].diskinfo_description, *disk_params[i].value);
		}
	}
}

/*---------------------------------------------------------------------------
 *	NAME
 *		probe ()					- Probe a single open SCSI Device
 *
 *	SYNOPSIS
 *		void probe (
 *		int			fd,				- File Descriptor of SCSI Device
 *		const char	*name)			- Name of SCSI Device
 *
 *	RETURN VALUE
 *		None
 *
 *	DESCRIPTION
 *		probe obtains and reports results for a single SCSI device.
 *---------------------------------------------------------------------------
 */

static void probe (int fd, const char *name)
{
	const int	COMMAND_LENGTH = 6;
	const int	INQUIRY_LENGTH = 36;
	char		scsi_command [COMMAND_LENGTH];
	char		scsi_data [INQUIRY_LENGTH];
	int			i;
	int			bytes_read;
	scsiresult	result;
	char		vendor [9];
	char		product [17];
	char		revision [5];
	char		vp [26];

	scsi_command [0] = SOP_INQ;
	scsi_command [1] = 0;
	scsi_command [2] = 0;
	scsi_command [3] = 0;
	scsi_command [4] = INQUIRY_LENGTH;
	scsi_command [5] = 0;

	for (i = 0; i < INQUIRY_LENGTH; i++)
		scsi_data [i] = 0;

	bytes_read = scsi_read (fd, scsi_command, COMMAND_LENGTH, scsi_data,
							INQUIRY_LENGTH, SCSI_TIMEOUT, &result);

	if (bytes_read == -1)
	{
		if (opt_all)
			fprintf (stdout, "%-13.13s <%s>\n", name, sys_errlist[errno]);
		error_count++;
		return;
	}
	else if (result.errno > 0)
	{
		if (opt_all)
			fprintf (stdout, "%-13.13s <%s>\n",
						name, scsi_strings[result.errno]);
		error_count++;
		return;
	}

	extract_string (vendor, scsi_data+8, 8);
	extract_string (product, scsi_data+16, 16);
	extract_string (revision, scsi_data+32, 4);
	strcpy (vp, vendor);
	strcat (vp, "_");
	strcat (vp, product);
	for (i = 0; vp[i] != '\0'; i++)
		if (vp[i] == ' ')
			vp[i] = '_';

	if (opt_disktab)
	{
		if (*scsi_data == 0)
			make_disktab (fd, name, vp);
		else
		{
			if (opt_all)
				fprintf (stdout, "%-13.13s <not a disk>\n", name);
			error_count++;
		}
	}
	else if (opt_vp)
		fprintf (stdout, "%s\n", vp);
	else if (opt_vendor)
		fprintf (stdout, "%s\n", vendor);
	else if (opt_product)
		fprintf (stdout, "%s\n", product);
	else if (opt_revision)
		fprintf (stdout, "%s\n", revision);
	else
		fprintf (stdout, "%-13.13s %-8.8s %-16.16s %s\n",
					name, vendor, product, revision);
	if (opt_diskinfo && (! opt_disktab) && (*scsi_data == 0))
		make_disktab (fd, name, vp);
}

/*---------------------------------------------------------------------------
 *	NAME
 *		main ()						- scsiprobe main function
 *
 *	SYNOPSIS
 *		main (
 *		int			argc,			- Number of Arguments
 *		char		*argv[])		- Argument Vector
 *
 *	RETURN VALUE
 *		exits with a count of the number of errors detected.
 *
 *	DESCRIPTION
 *		This function parses the options and probes the specified devices1.
 *---------------------------------------------------------------------------
 */

main (int argc, char *argv[])
{
	int			fd;
	char		name [256];
	int			i;
	FILE		*my_ls;

	typedef struct
	{
		char	*name;
		int		*value;
		char	*help;
	}
	option_structure;

	static option_structure option_list [] =
		{
			"-?",	&opt_help,		"Display these help messages",
			"-a",	&opt_all,		"Display all devices",
			"-d",	&opt_disktab,	"Generate disktab entries",
			"-D",	&opt_diskinfo,	"Display disk information",
			"-h",	&opt_help,		"Display these help messages",
			"-n",	&opt_noheader,	"Omit header line",
			"-p",	&opt_product,	"Display Product field only",
			"-r",	&opt_revision,	"Display Revision field only",
			"-v",	&opt_vendor,	"Display Vendor field only",
			"-V",	&opt_version,	"Display Program version and copyright",
		};
	int	number_of_options = sizeof (option_list) / sizeof (option_structure);

	my_name = *argv++;
	argc--;

	while ((argc > 0) && (**argv == '-'))
	{
		for (i = 0; i < number_of_options; i++)
			if (strcmp (option_list [i].name, *argv) == 0)
			{
				*(option_list [i].value) ^= 1;
				argv++;
				argc--;
				break;
			}
		if (i == number_of_options)
		{
			fprintf (stderr, "ERROR: unrecognized option %s\n", *argv);
			opt_help = 1;
			break;
		}
	}

	if (opt_help)
	{
		fprintf (stderr, "usage: %s -options [device...]\n", my_name);
		fprintf (stderr, "options:\n");
		for (i = 0; i < number_of_options; i++)
			fprintf (stderr, "  %s: %s\n",
							option_list [i].name, option_list [i].help);
		exit (1);
	}

	if (opt_version)
	{
		fprintf (stderr, "scsiprobe version 1.11, copyright (c) 1994 Herb Weiner <herbw@wiskit.com>\n");
		exit (1);
	}

	opt_vp = opt_vendor & opt_product;
	opt_vpr = opt_vendor | opt_product | opt_revision;

	if ((! opt_vpr) && (! opt_noheader) && (! opt_disktab))
		fprintf (stdout, "device        vendor   product          rev\n");

	if (argc > 0)
	{
		for (i = 0; i < argc; i++)
		{
			fd = open (argv [i], O_RDONLY);
			if (fd == -1)
			{
				if (opt_all)
					fprintf (stdout, "%-13.13s <%s>\n",
									argv [i], sys_errlist[errno]);
				error_count++;
			}
			else
			{
				probe (fd, argv [i]);
				close (fd);
			}
		}
	}
	else
	{
		my_ls = popen ("/bin/ls /dev/scsi/*", "r");

		while (fgets (name, sizeof(name), my_ls) != (char *) NULL)
		{
			i = strlen (name);
			if ((i > 0) && (name[i-1] == '\n')) 
				name[i-1] = '\0';

			fd = open (name, O_RDONLY);
			if (fd == -1)
			{
				if (opt_all)
					fprintf (stdout, "%-13.13s <%s>\n",
									name, sys_errlist[errno]);
				error_count++;
			}
			else
			{
				probe (fd, name);
				close (fd);
			}
		}

		pclose (my_ls);
	}

	exit (error_count);
}
