/*---------------------------------------------------------------------------
 * SCSI Library for A/UX
 *
 * Copyright (c) 1994 by Herb Weiner (herbw@wiskit.com). All rights reserved.
 * This software is provided on an AS IS basis without warranty or support of
 * any kind.  Permission is granted to distribute this software provided that
 * this entire copyright notice is retained, and provided that the package is
 * provided (including source code) without charge.  Please email bug reports
 * and enhancements to the author.  Enjoy this software, and please share it.
 *---------------------------------------------------------------------------
 */

#include <stdio.h>
#include <sys/ioctl.h>
#define SCSIREQ
#include <sys/scsireq.h>

#include "scsilib.h"

#define SCSISTART	_IOWR('S',1,struct userscsireq)

/*---------------------------------------------------------------------------
 *	NAME
 *		scsi_common ()				- Common SCSI Read and Write Function
 *
 *	SYNOPSIS
 *		long scsi_common (
 *		int			fd,				- File Descriptor of SCSI Device
 *		int			mode,			- read = 1; write = 0
 *		char		*commandPtr,	- SCSI Command to execute
 *		long		commandLen,		- Length of SCSI Command
 *		char		*dataPtr,		- Input or Output Buffer
 *		long		dataLength,		- Amount of Data to Transfer
 *		long		timeout,		- Timeout in Seconds
 *		scsiresult	*result)		- Result Structure
 *
 *	RETURN VALUE
 *		-1                          - Failed
 *		number of bytes transferred	- Success
 *
 *	DESCRIPTION
 *		scsi_common reads or writes from an open SCSI device (/dev/scsi/*)
 *		using the A/UX kernel scsi driver.
 *---------------------------------------------------------------------------
 */

static long scsi_common (int fd, int mode, char *commandPtr, long commandLen,
			char *dataPtr, long dataLength, long timeout, scsiresult *result)
{
	struct	userscsireq
	{
		char			*cmdbuf;	/* buffer containing command block */
		unsigned char	cmdlen;		/* length of command buffer */
		unsigned char	padding1;	/* padding to word boundary */
		char			*databuf;	/* buffer containing data to transfer */
		unsigned long	datalen;	/* length of the data buffer, MAXIMUM 8k */
		unsigned long	datasent;	/* length of data actually transferred */
		char			*sensebuf;	/* optional error result from sense cmd */
		unsigned char	senselen;	/* length of sense buffer, MAX 100 bytes */
		unsigned char	sensesent;	/* length of sense data received */
		unsigned short	flags;		/* 1 = read; 0 = write */
		unsigned char	msg;		/* mode byte returned on cmd complete */
		unsigned char	stat;		/* completion status byte */
		unsigned char	ret;		/* return code from SCSI Manager */
		unsigned char	padding2;	/* padding to word boundary */
		short			timeout;	/* maximum timeout in seconds */
	};

	struct userscsireq	usr;
	int					ioctl_result;
	int					dataNumber;

	usr.cmdbuf			= commandPtr;
	usr.cmdlen			= commandLen;
	usr.padding1		= 0;
	usr.databuf			= dataPtr;
	usr.datalen			= dataLength;	/* Maximum 8192 bytes */
	usr.datasent		= 0;

	if (result != (scsiresult *) NULL)
	{
		usr.sensebuf	= result->sense;
		usr.senselen	= sizeof (result->sense);
	}
	else
	{
		usr.sensebuf	= (char *) NULL;
		usr.senselen	= 0;
	}

	usr.sensesent		= 0;
	usr.flags			= mode;			/* Read = 1; Write = 2 */
	usr.msg				= 0;
	usr.stat			= 0;
	usr.ret				= 0;
	usr.padding2		= 0;
	usr.timeout			= timeout;		/* in seconds */

	ioctl_result = ioctl (fd, SCSISTART, (unsigned char *)&usr);

	if (result != (scsiresult *) NULL)
	{
		result->sense_length = usr.sensesent;
		result->status = usr.stat;
		result->message = usr.msg;
		result->errno = usr.ret;
	}

	if (ioctl_result == -1)
		return (-1);
	else
		return (usr.datasent);
}

/*---------------------------------------------------------------------------
 *	NAME
 *		scsi_read ()				- SCSI Read Function
 *
 *	SYNOPSIS
 *		long scsi_read (
 *		int			fd,				- File Descriptor of SCSI Device
 *		char		*commandPtr,	- SCSI Command to execute
 *		long		commandLen,		- Length of SCSI Command
 *		char		*dataPtr,		- Input Buffer
 *		long		dataLength,		- Amount of Data to Read
 *		long		timeout,		- Timeout in Seconds
 *		scsiresult	*result)		- Result Structure
 *
 *	RETURN VALUE
 *		-1                          - Failed
 *		number of bytes transferred	- Success
 *
 *	DESCRIPTION
 *		scsi_read reads from an open SCSI device (/dev/scsi/*) using the
 *		scsi_common function.
 *---------------------------------------------------------------------------
 */

long scsi_read (int fd, char *commandPtr, long commandLen, char *dataPtr,
				long dataLength, long timeout, scsiresult *result)
{
	return (scsi_common (fd, 1, commandPtr, commandLen, dataPtr, dataLength,
					timeout, result));
}

/*---------------------------------------------------------------------------
 *	NAME
 *		scsi_write ()				- SCSI Write Function
 *
 *	SYNOPSIS
 *		long scsi_write (
 *		int			fd,				- File Descriptor of SCSI Device
 *		char		*commandPtr,	- SCSI Command to execute
 *		long		commandLen,		- Length of SCSI Command
 *		char		*dataPtr,		- Output Buffer
 *		long		dataLength,		- Amount of Data to Write
 *		long		timeout,		- Timeout in Seconds
 *		scsiresult	*result)		- Result Structure
 *
 *	RETURN VALUE
 *		-1                          - Failed
 *		number of bytes transferred	- Success
 *
 *	DESCRIPTION
 *		scsi_write writes to an open SCSI device (/dev/scsi/*) using the
 *		scsi_common function.
 *---------------------------------------------------------------------------
 */

long scsi_write (int fd, char *commandPtr, long commandLen, char *dataPtr,
				long dataLength, long timeout, scsiresult *result)
{
	return (scsi_common (fd, 0, commandPtr, commandLen, dataPtr, dataLength,
				timeout, result));
}
