/*
 *     Copyright CERN, Geneva 1989 - Copyright and any other
 *     appropriate legal protection of these computer programs
 *     and associated documentation reserved in all countries
 *     of the world.
 */

/*
 *
 */
#include "3270.h"
#include "telnet.h"
#include "globals.h"

/* Some tcp/ip socket externals */

#include <sys/socket.h>
#include <netdb.h>

#ifdef sys_3b2

#include <sys/in.h>
#include <sys/stream.h>
#include <sys/ip.h>
#include <sys/tcp.h>
#include <sys/inet.h>

#else /* sys_3b2 */

#include <netinet/in.h>
#include <netinet/tcp.h>

/* NB: May also need inet.h for inet_addr or inet_ntoa */

/* #include <arpa/inet.h> */

#endif /* sys_3b2 */

/* Library routines definition (where not in a standard header!) */

#ifndef htons
extern  u_short         htons();
#endif /* htons */

#ifndef inet_addr
extern  unsigned long   inet_addr();
#endif /* inet_addr */

#ifndef inet_ntoa
extern  char            *inet_ntoa();
#endif /* inet_ntoa */

char *usernsave;

#define MAXDEF   2048 /* max length of a definition (in chars) */

/*
 * Allocate a buffer of a given size (in terms of characters).
 * In order to ensure proper alignment we will make it return
 * (unsigned long *). All buffering therefore will be in terms of
 * such buffers.
 * Remember, of course, to round up the buffer size.
 */

#ifdef STANDARD_C

unsigned long   *alloc_buffer(int buffer_size, char fill_char)

#else /* STANDARD_C */

unsigned long   *alloc_buffer(buffer_size, fill_char)

int     buffer_size;
char    fill_char;

#endif /* STANDARD_C */

{
	unsigned long   *result;
	int             fail_count = 0;

	buffer_size = (buffer_size + 3) & ~3;

	while ((result = (unsigned long *)malloc((unsigned)buffer_size)) == (unsigned long *)NULL)
	{
		fail_count++;
		sleep(2);
	}

#ifdef LOGFILE
	if (fail_count)
		logit("Fail to get buffer of size %d", buffer_size);
#endif /* LOGFILE */

	(void) memset((void *)result, (int)fill_char, (size_t)buffer_size);

#ifdef DEBUG
	(void)fprintf(outf, "alloc_buffer: (%d) returns 0x%lx\n",
			    buffer_size, result);
#endif /* DEBUG */

	return (result);
}

/* Reallocate a character buffer to become bigger */

#ifdef STANDARD_C

unsigned long   *incr_buffer(unsigned long *old_buffer, int oldb_size, int increment, char fill_char)

#else /* STANDARD_C */

unsigned long   *incr_buffer(old_buffer, oldb_size, increment, fill_char)

unsigned long   *old_buffer;
int             oldb_size;
int             increment;
char            fill_char;

#endif /* STANDARD_C */

{
	unsigned long   *result;

#ifdef DEBUG
	(void)fprintf(outf, "incr_buffer : (0x%lx , %d , %d, 0x%x)\n",
			    old_buffer, oldb_size, increment, fill_char);
#endif /* DEBUG */

	result = alloc_buffer(oldb_size + increment, fill_char);
	(void) memcpy((void *) result, (void *) old_buffer, (size_t) oldb_size);
	free((void *)old_buffer);
	return (result);
}

#ifdef STANDARD_C

void    usage(char *prog_name)

#else /* STANDARD_C */

void    usage(prog_name)

char    *prog_name;

#endif /* STANDARD_C */

{
	int     opt_ci;

	reset_tty(0, FALSE);


#ifdef LDEBUG
	(void)fprintf(stderr,"usage: %s [-d[system to DIAL]] [-p port] [-l]", prog_name);
#else /* LDEBUG */
	(void)fprintf(stderr,"usage: %s [-d[system to DIAL]] [-p port_number]", prog_name);
#endif /* LDEBUG */

	for (opt_ci = 0 ; opt_ci < OPT_CSIZE ; opt_ci++)
	{
		if (opt_cindex[opt_ci] >= 0)
		{
			if (!opt_smaxlen[opt_cindex[opt_ci]])
				(void)fprintf(stderr, " [-%c]", opt_cchar[opt_ci]);
			else
				(void)fprintf(stderr, " [-%c[name]]", opt_cchar[opt_ci]);
		}
	}
	(void)fprintf(stderr," host [user]\n");
	sleep(1);
	exit(1);
}

#ifdef STANDARD_C

int     main(int argc, char **argv)

#else /* STANDARD_C */

int     main(argc, argv)

int     argc;
char    **argv;

#endif /* STANDARD_C */

{
	int     port = DEFAULTPORT;
	int     arg;
	char    *keymap_def;
	char    *keymap_file;
	char    *argp;
	int     sizer_OK;
	struct  in_addr ip_astruct;

#ifdef OPTDIR

#ifdef TTY
	int     fscan_res;
#endif /* TTY */

#endif /* OPTDIR */

#ifdef LDEBUG
	int logflag = 0;
#endif /* LDEBUG */

	if (!(ttyn = ttyname(0)))
		exit(999);
	ttypid = getpid();
	save_tty();             /* save initial tty setup               */

	/* process options */

	for (arg = 1; arg<argc && argv[arg][0] == '-'; arg++)
	{
		for (argp = argv[arg]+1; *argp; argp++)
		{
			int     opt_ci;
			int     opt_pi;

			switch(*argp)
			{

#ifdef LDEBUG
				case 'l': /* undocumented "log debug" flag */
					logflag++;
					break;
#endif /* LDEBUG */

				case 'd': /* remainder of string is system to setup DIAL to */

					dialname = argp + 1;
					argp += strlen(dialname);

					break;

				case 'p': /* need a port number now */

					if (!(*(++argp)))
					{
						if (++arg == argc)
							usage(argv[0]);
						argp = argv[arg];
					}
					for (port = 0; *argp; argp++)
					{
						if (isdigit(*argp))
							port = 10*port + (int)(*argp - '0');
						else
							usage(argv[0]);
					}

					if (!port)
						usage(argv[0]);

					argp--;
					break;

				default:
					opt_pi = -1;
					for (opt_ci = 0 ; opt_ci < OPT_CSIZE ; opt_ci++)
					{
						if (opt_cchar[opt_ci] ==  *argp)
						{
							if ((opt_pi = opt_cindex[opt_ci]) < 0)
								break;
							if (!opt_smaxlen[opt_pi])
							{
								/* A simple setting */


								*((int *)opt_pointer[opt_pi]) = opt_cival[opt_ci];

							}
							else
							{
								/* remainder of the string is the printer */

								if (!*(argp + 1))
									usage(argv[0]);
								strncpy((char *)opt_pointer[opt_pi], argp + 1, 10);
								argp += strlen((char *)opt_pointer[opt_pi]);
								break;
							}
						}
					}
					if (opt_pi < 0)
						usage(argv[0]);
				break;
			}
		}
	}

	if ((argc > arg+2) || (argc < arg+1))
		usage(argv[0]);

	progname = argv[0];           /* BS */
	remotename = argv[arg];
	if (argc != arg+1)
		username = argv[arg+1];

	/* Make sure that files can be created with world write */

	(void)umask(0);

#ifdef DEBUG
	(void)fprintf(stderr,"Connecting to port %d on host %s\r\n", port, remotename);
	(void)fprintf(stderr,"Trying...\r\n");
	(void)fflush(stderr);
#endif /* DEBUG */

	net = netopen(remotename,port);

#ifdef DEBUG
	(void)fprintf(stderr,"Opened on net = %d\r\n",net);
	(void)fflush(stderr);
#endif /* DEBUG */

#ifdef OPTDIR
	/*
	 * Now we want to establish a user name so that we can create a
	 * file containing the user's preferred options and knowledge
	 * of the message of the day.
	 * The call can include a user name (and from the TAGIBM usage,
	 * whereby the program is called from a shell script it will
	 * always have a name, as well as normally being called from
	 * the same guest account).
	 * We can also get the user name by getlogin().
	 * Which should we use?
	 * Well, it rather depends on whether we are running in a
	 * workstation (single or multi-user environment) or in a TAGIBM
	 * environment (calls from the shell script).
	 *
	 * We will assume that the symbol TTY normally implies a TAGIBM-type
	 * environment, so that we will prefer the user name as given in
	 * the program call. However, since one might well run this version
	 * from a workstation anyway, if there is no such user name then we
	 * will pick up the name of the calling user.
	 *
	 * If TTY is not set then we will assume a workstation and will
	 * prefer to take the name of the calling user. This allows for the
	 * user to put different user names in different calls (to the same
	 * IBM system) but keep one single options file (for that particular
	 * IBM system).
	 *
	 * If we are using the user name included in the call (which is
	 * only the case if the TTY symbol is set!) then we rather
	 * want a name which includes the remote host indicator.
	 * However, if it is the default host then we want the plain
	 * user name. For a single-user workstation (TTY not defined)
	 * the plain system user name is taken.
	 *
	 * For a multiuser system, including the TAGIBM general gateway,
	 * the complication is that we may be given an IP number or a name.
	 * In addition, there may be several equivalent IP names or numbers.
	 * We will therefore check in the OPTDIR directory for a file
	 * called "default_hosts" which can contain all of the equivalent
	 * names and IP addresses for the default host.
	 * We will also use the IP address, rather than the name, which
	 * avoids complications such as upper/lower case, pseudonyms etc.
	 */

	ip_astruct.s_addr = (u_long) ip_address;
	strcpy(ip_chaddr, inet_ntoa(ip_astruct));

#ifdef TTY
	if (username)
	{
		strncpy(print_user, username, 9);

		strncpy(opt_file, set_fname(OPTDIR, username), 99);
		fscan_res = -1;

		if ((opt_fstr = fopen(set_fname(OPTDIR, "default_hosts"), "r")) != NULL)
		{
			char    read_string[100];

			while ((fscan_res = fscanf(opt_fstr, "%99s\n", read_string)) >= 0)
				if (!strcmp(read_string, inet_ntoa(ip_astruct)))
					break;
		}
		if (fscan_res < 0)
		{
			strncat(opt_file, "_", 99 - strlen(opt_file));
			strncat(opt_file, ip_chaddr, 99 - strlen(opt_file));
		}
	}
	else
	{
		strncpy(print_user, (getlogin() ? getlogin() : "anonymous"), 9);
		strncpy(opt_file, set_fname(OPTDIR, print_user), 99);
	}

#else /* TTY */
	strncpy(print_user, (getlogin() ? getlogin() : getpwuid(getuid())->pw_name), 9);
	strncpy(opt_file, set_fname(OPTDIR, print_user), 99);
#endif /* TTY */

#endif /* OPTDIR */

	usernsave = (username ? username : "");

#ifdef LOGFILE
	logit("program entry", 0);
#endif /* LOGFILE */

#ifdef DEBUG
	if ((creat("/tmp/outf", 0666) < 0) ||
	    ((outf = fopen("/tmp/outf","w")) == NULL))
	{
		perror("outf");
		outf=stderr;
		end_program(-1001, "cannot open outf", 0);
	}
	setbuf(outf,(char *)NULL);
#endif /* DEBUG */

#ifdef LDEBUG
	if (logflag)
	{

#ifdef DEBUG
		log = outf;
#else /* DEBUG */
		if ((creat("/tmp/logf", 0666) < 0) ||
		    ((log = fopen("/tmp/logf","w")) == NULL))
		{
			perror("log");
			log=stderr;
			end_program(-1002, "cannot open logfile", 0);
		}
		setbuf(log,(char *)NULL);
#endif /* DEBUG */

	}                           
#endif /* LDEBUG */

	/*
	 * Now fill in any options which have not been set.
	 * The return value may indicate if the user has already seen
	 * an ANNOUNCE file.
	 */

	announce_id = get_options();

#ifdef DEBUG
	(void)fprintf(outf, "main        : announce_id value is %d\n",
			    announce_id);
#endif /* DEBUG */


	/* find out what kind of terminal (i.e.  number of lines)       */
	/*                                (and:  number of columns (BS) */
	if ((term_type = getenv("TERM")) == (char *)0) {
		(void)fprintf(stderr, "Terminal type not set!\n");
		(void)sleep(2);
		end_program(-1003, "No terminal type", 0);
	}

#ifdef DEBUG
	(void)fprintf(outf, "main        : look for termcap information for terminal type %s\n",
			    term_type);
#endif /* DEBUG */

	if (tgetent(termcap, term_type) != 1) {
		(void)fprintf(stderr,"Cannot find specs for terminal type %s\n",
			term_type);
		(void)sleep(2);
		end_program(-1004, "terminal type not in termcap", 0);
	}

	init_tty();             /* general tty setup                    */

	zap();			/* get terminal description		*/

	/*
	 * Many workstations have an xterm entry which says that a
	 * host-writeable status line is possible, but they lie.
	 * This is true of DEC Ultrix, Sun, RS6000 and maybe others.
	 * (although on the RS6000 there IS an extended xterm, called
	 * aixterm, which HAS this possibility)
	 * I shall therefore ignore this information for any terminal
	 * of type xterm unless the symbol XTERM_SL is defined!
	 */

#ifndef XTERM_SL

	if (!strncmp(term_type, "xterm", 5))
		HS = FALSE;

#endif /* XTERM_SL */

	/*
	 * Some termcap/terminfo entries (particularly for xterm) do
	 * not have the standard ANSI sequences for KS or KE (to set the
	 * keypad into or out of application mode: this is a nuisance
	 * for a lot of people.
	 * Almost all termcap/terminfo entries will not have the special
	 * entries that I create to describe how to switch between
	 * 80 and 132 column (SN and SW) and how to move right a given
	 * number of positions. These are so standard that I will allow
	 * them to be compiled into the program anyway. This compile
	 * option will be defined by symbols as follows :-
	 *
	 * ANSI_ALL             force in these ANSI sequences for all
	 *                      types of terminal.
	 *
	 * ANSI_VT              force in these ANSI sequences for terminals
	 *                      which appear to be DEC VTx00 (i.e. the
	 *                      terminal type starts with "vt")
	 *
	 * ANSI_XTERM           force in these ANSI sequences for terminals
	 *                      which appear to be xterm (i.e. the terminal
	 *                      type is "xterm").
	 */

#ifdef ANSI_ALL
#define ANSI_FORCE
#endif /* ANSI_ALL */

#ifdef ANSI_VT
#define ANSI_FORCE
#endif /* ANSI_VT */

#ifdef ANSI_XTERM
#define ANSI_FORCE
#endif /* ANSI_XTERM */

#ifdef ANSI_FORCE

	{

#ifdef ANSI_ALL

		bool    ansi_force = TRUE;

#else /* ANSI_ALL */

		bool    ansi_force = FALSE;

#ifdef ANSI_VT

		if (!strncmp(term_type, "vt", 2))
			ansi_force = TRUE;

#endif /* ANSI_VT */

#ifdef ANSI_XTERM

		if (!strncmp(term_type, "xterm", 5))
			ansi_force = TRUE;

#endif /* ANSI_XTERM */

#endif /* ANSI_ALL */

		if (ansi_force)
		{
			KE = "\033[?1l\033>";
			KS = "\033[?1h\033=";
			SN = "\033[?3l";
			SW = "\033[?3h";

#ifdef EXTENDED

			/*
			 * It is unfortunately true that the colour version of xterm
			 * (colxterm is its normal name) requires different sequences to
			 * DEC VT340 and DECterm/dxterm. Pity!
			 *
			 * The differences :-
			 * 1. Sending the reverse video sequence (ESC[7m) to (col)xterm
			 *    causes swap of foreground and background colours also.
			 *    This is not the case on DEC VT340 and DECterm/dxterm: the
			 *    reversed colours must be exactly specified!
			 * 2. DEC VT340 and DECterm/dxterm have a "neutral" colour, which
			 *    is white on black or black on white according as the user
			 *    switches between "Dark Text, Light Background" and
			 *    "Light Text, Dark Background". (col)xterm has nothing like this,
			 *    so we specifically have to choose. For the moment I specifically
			 *    choose black as foreground on xterm and nothing as background
			 *    (because if I did choose a background then after clearing the
			 *    screen it goes to the xterm-defined background colour!).
			 *
			 * This, of course, means that we ought to offer the xterm user an option
			 * to have white instead of black (assuming he has a dark background)
			 * and/or to have the whole thing inverted (like the flip possibility
			 * of DECterm/dxterm): for future study!
			 *
			 * (I choose black because I happen to run it on an X terminal which
			 * automatically flips the background to white when I choose any foreground
			 * colour: complaints not to me!).
			 */

			if (strncmp(term_type, "vt", 2))
			{
				/* ANSI normal mode colour sequences: all with foreground only */

				C0 = R0 = "\033[32m";   /* default   */
				C1 = R1 = "\033[34m";   /* blue      */
				C2 = R2 = "\033[31m";   /* red       */
				C3 = R3 = "\033[35m";   /* pink      */
				C4 = R4 = "\033[32m";   /* green     */
				C5 = R5 = "\033[36m";   /* turquoise */
				C6 = R6 = "\033[33m";   /* yellow    */
				C7 = R7 = "\033[30m";   /* black     */
			}
			else
			{
				/* DECterm/dxterm normal mode colour sequences: all with foreground and background */

				C0 = "\033[32m\033[49m";        /* Foreground default   Background neutral */
				C1 = "\033[34m\033[49m";        /* Foreground blue      Background neutral */
				C2 = "\033[31m\033[49m";        /* Foreground red       Background neutral */
				C3 = "\033[35m\033[49m";        /* Foreground pink      Background neutral */
				C4 = "\033[32m\033[49m";        /* Foreground green     Background neutral */
				C5 = "\033[36m\033[49m";        /* Foreground turquoise Background neutral */
				C6 = "\033[33m\033[49m";        /* Foreground yellow    Background neutral */
				C7 = "\033[39m\033[49m";        /* Foreground neutral   Background neutral */

				R0 = "\033[42m\033[39m";        /* Background default   Foreground neutral */
				R1 = "\033[44m\033[39m";        /* Background blue      Foreground neutral */
				R2 = "\033[41m\033[39m";        /* Background red       Foreground neutral */
				R3 = "\033[45m\033[39m";        /* Background pink      Foreground neutral */
				R4 = "\033[42m\033[39m";        /* Background green     Foreground neutral */
				R5 = "\033[46m\033[39m";        /* Background turquoise Foreground neutral */
				R6 = "\033[43m\033[39m";        /* Background yellow    Foreground neutral */
				R7 = "\033[49m\033[39m";        /* Background neutral   Foreground neutral */
			}

#endif /* EXTENDED */

		}
	}

#endif /* ANSI_FORCE */

	cchange_setup();        /* set up the style change strings      */

	set_tcstr();            /* set up standard termcap strings      */

	if (!CM)
		end_program(-1 , "Sorry, cannot do 3270 emulation on your terminal type!", 0);

	f_puttcs(tcsIS, TRUE);
	f_puttcs(tcsCL, TRUE);

	/* find what are the current real lines and columns */

	sizer_OK = sizer();
	f_flush();

	if (scroll_line > 0)
	{
		f_puts( "Warning: I will ignore your scroll line setting!\n", FALSE);
		(void)sleep(1);
	}

	if ((LINES < 24) || (COLS < 80))
	{
		if (COLS < 80)
		{
			f_putf( "Warning: Your screen seems only to have %d columns!\n",
				 FALSE, COLS);
			if (COLS <= 2)
			{

#ifdef sys_apollo

	/* An Apollo seems to indicate one less than the real size! */

				COLS = tgetnum("co") + 1;

#else /* sys_apollo */

				COLS = tgetnum("co");

#endif /* sys_apollo */

				f_putf( "I will assume the default value (%d) for your terminal type\n",
					 FALSE, COLS);
			}
		}
		else
		{

#ifdef sys_apollo

	/* An Apollo seems to indicate one less than the real size! */

			COLS = tgetnum("co") + 1;

#else /* sys_apollo */

			COLS = tgetnum("co");

#endif /* sys_apollo */

			f_putf( "Warning: I cannot evaluate your screen columns: assume %d\n",
				 FALSE, COLS);
		}

		if (LINES < 24)
		{
			f_putf( "Warning: Your screen seems only to have %d lines!\n",
				 FALSE, LINES);
			if (LINES <= 2)
			{

#ifdef sys_apollo

	/* An Apollo seems to indicate one less than the real size! */

				LINES = tgetnum("li") + 1;

#else /* sys_apollo */

				LINES = tgetnum("li");

#endif /* sys_apollo */

				f_putf( "I will assume the default value (%d) for your terminal type\n",
					 FALSE, LINES);
			}
		}
		else
		{

#ifdef sys_apollo

	/* An Apollo seems to indicate one less than the real size! */

			LINES = tgetnum("li") + 1;

#else /* sys_apollo */

			LINES = tgetnum("li");

#endif /* sys_apollo */

			f_putf( "Warning: I cannot evaluate your screen lines: assume %d\n",
				 FALSE, LINES);
		}

		f_flush();
		(void)sleep(1);
		SN = NULL;
		SW = NULL;
		AM = FALSE;
		VW = FALSE;
		sizer_OK = -1;

#ifdef DEBUG
	(void)fprintf(outf, "main        : termcap says LINES = %d, COLS = %d\n", LINES, COLS);
#endif /* DEBUG */

	}

#ifdef DEBUG
	(void)fprintf(outf, "main        : term = %s\n", term_type);
	(void)fprintf(outf, "main        : LINES = %d, COLS = %d\n", LINES, COLS);
	(void)fprintf(outf, "main        : Automatic margins is %s , VT-wrap (VW) is %s\n",
		   (AM ? "ON" : "OFF") , (VW ? "ON" : "OFF"));
#endif /* DEBUG */

	if (LINES < 24)
		end_program(-2 , "Your screen seems to have only %d lines!",
					LINES);
	else if (COLS < 80)
		end_program(-3 , "Your screen seems to have only %d columns!",
					COLS);

	/* Now pre-suppose that there is not to be any screen flipping */
	/* and all screens will have a normal 24*80 possibility        */

	linelen = 80;
	normal_linelen = 80;
	rows = 24;
	normal_rows = 24;
	altscreen = 1;

	init_COLS = COLS;
	init_LINES = LINES;

#ifdef DEBUG
	(void)fprintf(outf, "main        : original terminal %d lines %d cols\n",
		      init_LINES, init_COLS);
#endif /* DEBUG */

	/*
	 * If we cannot do a proper check on screen size let us not allow
	 * any changes of size!
	 */

	if (sizer_OK <= 0)
	{
		tcslen[tcsSN] = 0;
		tcslen[tcsSW] = 0;
		tcsptr[tcsSN] = "";
		tcsptr[tcsSW] = "";
	}

	/* The user might have an alternate screen possible */

	/*
	 * At CERN we allow for a 24*132 screen possibility.
	 * This changes things if the user starts with a screen
	 * which already has at least 132 columns and between 24 and 26
	 * lines (if it has more lines then it can be a real model-5).
	 * In this case the alternate screen will be set up as 24*132,
	 * and the first READ BUFFER request from the IBM (occurring
	 * after an ERASE WRITE ALTERNATE) will return 24*132 = 3168 nulls.
	 * If, however, the user starts with a smaller screen, so that
	 * the normal screen would be 24*80, but the terminal can actually
	 * offer a 24*132 screen then the initial response to the same
	 * READ BUFFER for the alternate screen will return 24*80 = 1920
	 * nulls but the program will assume that any subsequent switch
	 * to the alternate screen (which can be done at CERN by a special
	 * modification allowing a so-called model-6) will be for a 24*132
	 * screen.
	 * For IBM systems which do not have this CERN IBM modification
	 * there should be no switching to the alternate screen anyway.
	 * However, having found one case where this does happen I make
	 * this special code only get compiled in if the DENSE symbol
	 * is defined (should be -DDENSE in the Makefile)
	 *
	 * Let us provisionally assume no screen switching.
	 */

	normal_LINES = LINES;
	normal_COLS = COLS;
	altern_LINES = LINES;
	altern_COLS = COLS;

	if (altflip = (SW && SN))
	{
		/*
		 * In principle the terminal can switch between 80 and 132 column modes.
		 * However, this may not really be true, in particular for xterm terminal
		 * usage where the user has forgotten to start up with the appropriate
		 * -132 switch.
		 * We can check by trying to switch and check the screen sizes, but this
		 * takes a bit of time, which is particularly annoying for slow
		 * asynchronous terminals on RS232 lines and which have a pretty simple
		 * default setup (80 or 132 columns and at least 24 lines).
		 * We will therefore skip this checking in some cases.
		 * We will, however, check if it looks like xterm.
		 */

#ifdef DENSE

		if (((COLS == 80) && (LINES < 27)) ||
		    (COLS == 132))
		{
			normal_COLS = 80;
			altern_COLS = 132;
		}
		else
		{
			/*
			 * If we want to offer the possibility of being an IBM model 5
			 * (normal screen 24*80, alternate screen 27*132) we need to
			 * check the screen sizes.
			 * Since, however, the later logic currently prefers being a model
			 * 3 (32*80) or 4 (43*80) UNLESS the screen already has at least
			 * 132 columns there is not much point in checking unless the
			 * current number of lines is 24-31 or the current number of
			 * columns is >131.
			 */

			if ((COLS > 131) || (LINES < 32))
			{
				if (COLS != 80)
				{
					f_puttcs(tcsSN, FALSE);
					f_puttcs(tcsCL, TRUE);
					if (!(sizer_OK = sizer()))
					{
						f_puts( "Warning: After switch to 80-column mode I cannot find the screen size!\n", TRUE);
						sleep(1);

#ifdef DEBUG
	(void)fprintf(outf, "main        : sizer failure after switch to normal screen!original terminal %d lines %d cols\n",
		      init_LINES, init_COLS);
#endif /* DEBUG */

						LINES = init_LINES;
						COLS = init_COLS;
					}
					if (COLS != 80)
					{
						f_puts( "Warning: Cannot switch between 80 and 132 column modes!\n", TRUE);
						sleep(1);

#ifdef DEBUG
	(void)fprintf(outf, "main        : Switch to 80-column mode fails\n");
#endif /* DEBUG */

						sizer_OK = 0;
					}
				}
				if (sizer_OK)
				{
					normal_COLS = COLS;
					normal_LINES = LINES;
					f_puttcs(tcsSW, FALSE);
					f_puttcs(tcsCL, TRUE);
					if (!(sizer_OK = sizer()))
					{
						f_puts( "Warning: After switch to 132-column mode I cannot find the screen size!\n", TRUE);
						sleep(1);

#ifdef DEBUG
	(void)fprintf(outf, "main        : sizer failure after switch to normal screen!original terminal %d lines %d cols\n",
		      init_LINES, init_COLS);
#endif /* DEBUG */

						LINES = normal_LINES;
						COLS = normal_COLS;
					}
					if (COLS == 80)
					{
						f_puts( "Warning: Cannot switch between 80 and 132 column modes!\n", TRUE);
						sleep(1);

#ifdef DEBUG
	(void)fprintf(outf, "main        : Switch to 132-column mode fails\n");
#endif /* DEBUG */

						sizer_OK = 0;
					}
					if (sizer_OK)
					{
						/*
						 * Cater for some systems (like an Atari ST with Uniterm)
						 * which switch to wide mode but cannot quite handle 132 columns.
						 */

						COLS = 132;
						altern_COLS = COLS;
						altern_LINES = LINES;
					}
					else
					{
						/*
						 * Something is wrong in our switching between screens.
						 * Just assume the original standard screen and allow no switching.
						 */

						tcslen[tcsSN] = 0;
						tcslen[tcsSW] = 0;
						tcsptr[tcsSN] = "";
						tcsptr[tcsSW] = "";
						altflip = FALSE;
					}
				}
			}
		}

#else /* DENSE */

		if ((COLS == 132) && (LINES > 26))
		{
			normal_COLS = 80;
			altern_COLS = 132;
		}
		else
		{
			/*
			 * If we want to offer the possibility of being an IBM model 5
			 * (normal screen 24*80, alternate screen 27*132) we need to
			 * check the screen sizes.
			 * Since, however, the later logic currently prefers being a model
			 * 3 (32*80) or 4 (43*80) UNLESS the screen already has at least
			 * 132 columns there is not much point in checking unless the
			 * current number of lines is 27-31 or the current number of
			 * columns is >131.
			 */

			if ((COLS > 131) ||
			    ((LINES > 26) && (LINES < 32)))
			{
				if (COLS != 80)
				{
					f_puttcs(tcsSN, FALSE);
					f_puttcs(tcsCL, TRUE);
					if (!(sizer_OK = sizer()))
					{
						f_puts( "Warning: After switch to 80-column mode I cannot find the screen size!\n", TRUE);
						sleep(1);

#ifdef DEBUG
	(void)fprintf(outf, "main        : sizer failure after switch to normal screen!original terminal %d lines %d cols\n",
		      init_LINES, init_COLS);
#endif /* DEBUG */

						LINES = init_LINES;
						COLS = init_COLS;
					}
					if (COLS != 80)
					{
						f_puts( "Warning: Cannot switch between 80 and 132 column modes!\n", TRUE);
						sleep(1);

#ifdef DEBUG
	(void)fprintf(outf, "main        : Switch to 80-column mode fails\n");
#endif /* DEBUG */

						sizer_OK = 0;
					}
				}
				if (sizer_OK)
				{
					normal_COLS = COLS;
					normal_LINES = LINES;
					f_puttcs(tcsSW, FALSE);
					f_puttcs(tcsCL, TRUE);
					if (!(sizer_OK = sizer()))
					{
						f_puts( "Warning: After switch to 132-column mode I cannot find the screen size!\n", TRUE);
						sleep(1);

#ifdef DEBUG
	(void)fprintf(outf, "main        : sizer failure after switch to normal screen!original terminal %d lines %d cols\n",
		      init_LINES, init_COLS);
#endif /* DEBUG */

						LINES = normal_LINES;
						COLS = normal_COLS;
					}
					if (COLS == 80)
					{
						f_puts( "Warning: Cannot switch between 80 and 132 column modes!\n", TRUE);
						sleep(1);

#ifdef DEBUG
	(void)fprintf(outf, "main        : Switch to 132-column mode fails\n");
#endif /* DEBUG */

						sizer_OK = 0;
					}
					if (sizer_OK)
					{
						/*
						 * Cater for some systems (like an Atari ST with Uniterm)
						 * which switch to wide mode but cannot quite handle 132 columns.
						 */

						COLS = 132;
						altern_COLS = COLS;
						altern_LINES = LINES;
					}
					else
					{
						/*
						 * Something is wrong in our switching between screens.
						 * Just assume the original standard screen and allow no switching.
						 */

						tcslen[tcsSN] = 0;
						tcslen[tcsSW] = 0;
						tcsptr[tcsSN] = "";
						tcsptr[tcsSW] = "";
						altflip = FALSE;
					}
				}
			}
		}

#endif /* DENSE */

	}

	if (normal_COLS == altern_COLS)
		altflip = FALSE;

	/* Now set up the two screens and terminal type */

	if ((altern_COLS >= 132) &&
	    ((altern_LINES < 32) || (init_COLS >= 132)))
	{
		/* Set up a wide display and a normal display */

		altern_linelen = linelen = 132;
		if (LINES < 27)
		{
			altern_rows = 24;

#ifdef EXTENDED
			TERMNAME = "IBM-3279-2-E";
#else /* EXTENDED */
			TERMNAME = "IBM-3278-2";
#endif /* EXTENDED */

			/* user may want to start in 80-col mode */

			if (init_COLS < 132)
				linelen = 80;
		}
		else
		{
			altern_rows = rows = 27;

#ifdef EXTENDED
			TERMNAME = "IBM-3279-5-E";
			wsf_a_rows = altern_rows;
			wsf_a_cols = altern_linelen;
#else /* EXTENDED */
			TERMNAME = "IBM-3278-5";
#endif /* EXTENDED */

		}
	}
	else
	{
		altern_linelen = 80;
		if ((altern_COLS >= 81) &&
		    (altern_LINES == normal_LINES))
		{
			f_puttcs(tcsSN, FALSE);
			f_puttcs(tcsCL, TRUE);
			COLS = altern_COLS = normal_COLS;
			altflip = FALSE;
		}
		if (LINES >= 32)
		{
			if (LINES >= 43)
			{
				altern_rows = rows = 43;

#ifdef EXTENDED
				TERMNAME = "IBM-3279-4-E";
				wsf_a_rows = altern_rows;
#else /* EXTENDED */
				TERMNAME = "IBM-3278-4";
#endif /* EXTENDED */

			}
			else
			{
				altern_rows = rows = 32;

#ifdef EXTENDED
				TERMNAME = "IBM-3279-3-E";
				wsf_a_rows = altern_rows;
#else /* EXTENDED */
				TERMNAME = "IBM-3278-3";
#endif /* EXTENDED */

			}
			altflip = FALSE;
		}
		else
		{
			altern_rows = rows = 24;

#ifdef EXTENDED
			TERMNAME = "IBM-3279-2-E";
#else /* EXTENDED */
			TERMNAME = "IBM-3278-2";
#endif /* EXTENDED */

		}
	}

	normal_slinenum = (normal_LINES > normal_rows ? normal_rows+1 : 0);
	altern_slinenum = (altern_LINES > altern_rows ? altern_rows+1 : 0);

#ifdef DEBUG
	(void)fprintf(outf, "main        : normal terminal %d x %d in %d x %d\n",
		      normal_rows, normal_linelen, normal_LINES, normal_COLS);
	(void)fprintf(outf, "main        : alternate terminal %d x %d in %d x %d\n",
		      altern_rows, altern_linelen, altern_LINES, altern_COLS);
	(void)fprintf(outf, "main        : altflip is %d, altscreen is %d, slinenum is %d\n",
		      altflip, altscreen, slinenum);
	(void)fprintf(outf, "main        : normal slinenum is %d, alternate slinenum is %d\n",
		      normal_slinenum, altern_slinenum);
#endif /* DEBUG */

	slinenum = (LINES > rows ? rows+1 : 0);
	screensize = linelen*rows;

	/* allocate space for the display memory */
	screen = (short *) alloc_buffer((int)(screensize * sizeof(short)), '\0');

#ifdef EXTENDED
	screen_ex = (unsigned char *) alloc_buffer((int)(screensize * sizeof(unsigned char)), '\0');
#endif /* EXTENDED */

	s_point =(short *) alloc_buffer((int)(screensize * sizeof(short)), '\0');

	/* Also the savebuf for screen info to and from the IBM */
	saveb_size = screensize + SAVEB_INCR;
	savep = savebuf = (unsigned char *) alloc_buffer(saveb_size, '\0');

	firstfield = -1;

	/* parse the profile.
	  * First, we look for a keyboard mapping entry.
	  * If it exists and doesn't start with '/', use it as the profile;
	  * we are done.
	  * (If MAP3270 exists and starts with '/', take it as the name of
	  * the profile file, otherwise use the default PROFILE).
	  * Look for an entry matching KEYMAP (if defined).
	  * If not then try for one matching the teminal type.
	  * If we still fail then try "unknown".
	  * If all else fails, use a wired-in definition.
	  */

	parse_init();

	/* find out whether KEYMAP has been defined */

	if (!(keymap_type = getenv("KEYMAP")))
		keymap_type = term_type;
	else
		(void)fprintf(stderr,"Environment variable KEYMAP specifies key mapping %s:\r\n",
					keymap_type);

	if (!(keymap_file = getenv("MAP3270")))
		keymap_file = PROFILE;
	else
		(void)fprintf(stderr,"Environment variable MAP3270 specifies mapping file %s:\r\n",
					keymap_file);

	if (*keymap_file != '/')
		keymap_def = keymap_file;
	else
	{

#ifdef DEBUG
	(void)fprintf(outf, "main        : look for keymap_type <%s> in file %s \n",
			      keymap_type, keymap_file);
#endif /* DEBUG */

		keymap_def = (char *) alloc_buffer(MAXDEF, '\0');
		if (!find_def(keymap_def, keymap_file, keymap_type) ||
		    *keymap_def==0)
		{
			int response;

			(void)fprintf(stderr,"Keyboard mapping type '%s' unknown\r\nType 'y' to use default setup (else will quit): ",keymap_type);
			empty_input();
			if (((response = f_getc(60)) != 'y') && (response != 'Y'))
				end_program(-1005, "Default setup not OK : quit", 0);
			(void)fprintf(stderr,"%c\r\nOK, I will use the default setup\r\n", response);
			while (((response = f_getc(5)) != '\012') && (response != EOF))
			{

#ifdef DEBUG
	(void)fprintf(outf, "main        : drop character <0x%x>\n", response);
#endif /* DEBUG */

			}

#ifdef DEBUG
	(void)fprintf(outf, "main        : look for keymap_type <%s> in file %s \n",
			      "unknown", keymap_file);
#endif /* DEBUG */

			if (!find_def(keymap_def, keymap_file, "unknown") ||
			     *keymap_def==0)
			{

#ifdef DEBUG
	(void)fprintf(outf, "main        : cannot find map type <unknown>\n");
#endif /* DEBUG */

				strcpy(keymap_def, default_def);
			}
		}
	}

	(void) parse_profile(keymap_def);
	if (keymap_def != keymap_file)
		free((void *) keymap_def);
	buildtrie();

	/* set up curscr data structures and modes */
	if (initscr() == ERR)
	{
		curscr = NULL;
		end_program(-1 , "screen initialisation failed", 0);
	}
	char_image = curscr->char_y;
	ccset_image = curscr->ccset_y;

#ifdef EXTENDED
	ccext_image = curscr->ccext_y;
#endif /* EXTENDED */

#ifdef DEBUG
	(void)fprintf(outf, "main        : set net = %d\n", net);
#endif /* DEBUG */

#ifdef FROMNET
	fromnet = creat("/tmp/fromnet",0666);

#ifdef DEBUG
	(void)fprintf(outf, "main        : set fromnet = %d\n", fromnet);
#endif /* DEBUG */

	if (fromnet < 0)
		end_program(-1081 , "cannot create fromnet", 0);
#endif /* FROMNET */

#ifdef TONET
	tonet = creat("/tmp/tonet",0666);

#ifdef DEBUG
	(void)fprintf(outf, "main        : set tonet = %d\n", tonet);
#endif /* DEBUG */

	if (tonet<0)
		end_program(-1082 , "cannot create tonet", 0);
#endif /* TONET */

#ifdef DEBUG
	(void)fprintf(stderr, "Connected to %s.\r\n", remotename);       /* BS */
	(void)fprintf(stderr, "Escape character is '%s'.\r\n", eschar);  /* BS */
	(void)sleep(2);
#endif /* DEBUG */

#ifdef LOGFILE
	logit("Begin 3270 emulation: Model %d",
	       (int)(TERMNAME[9] - '0'));
#endif /* LOGFILE */

	if ((announce_id <= 0) ||
	    (announce_id != fileout(set_fname(LIBDIR, ANNOUNCE), -1)))
	{
		announce_id = fileout(set_fname(LIBDIR, ANNOUNCE), 1);

#ifdef OPTDIR
		(void) set_options(0);
#endif /* OPTDIR */

	}

	/*
	 * Most output goes directly to /dev/tty (actually to fd_output).
	 * Buffering may depend on what type of output device is in use
	 * but we try to setup block buffering : there is no point in having
	 * line buffering because a 3270 is not line-oriented.
	 */

#ifndef sys_apollo

	setvbuf(stdout, NULL, _IOFBF, BUFSIZ);                               /*  (JMG)   */

#endif /* sys_apollo */

	setup_tty(TRUE);            /* setup the tty for 3270               */

	/* Set up any signal handling now */
	set_signals();

	reader();
	end_program(0, "Normal IBM logoff", 0);

/*
 * Seems that for some reason we can loop after an exit()
 * may be due to an interrupt after we call exit.
 * If this happens then the only way may be to kill the task
 */

/*      kill(getpid(),SIGKILL); */

	exit(0);
	return(0);
}

#ifdef LOGFILE
/*
 * Write a message to the log file
 */

#ifdef STANDARD_C

void    logit(char *log_mess, int arg1)

#else /* STANDARD_C */

void    logit(log_mess, arg1)

char    *log_mess;

#endif /* STANDARD_C */

{
	FILE    *logfile;
	char    logit_mess[250]; /* to store logit message    */

	if (ttyn)
	{
		sprintf(&logit_mess[0], "%s %d %s %s %s %s ",
					 (strlen(ttyn) > 8 ? ttyn+8 : ttyn),
					 ttypid, getstime(),
					 progname, remotename, usernsave);
		sprintf(&logit_mess[strlen(&logit_mess[0])],
			log_mess, arg1);

		if ((logfile = fopen(LOGFILE, "a")) != (FILE *) NULL)
		{
			(void)fprintf(logfile, "%s\n", &logit_mess[0]);
			fclose(logfile);
		}
	}
}

/*
 * Try to write out the current user screen to the logfile.
 */

#ifdef STANDARD_C

void    logscreen()

#else /* STANDARD_C */

void    logscreen()

#endif /* STANDARD_C */

{
	int     i;
	char    linebuf[133];

	if (curscr)
	{
		for (i = 0; i < rows ; i++)
		{
			(void) memcpy((void *) linebuf, (void *) curscr->char_y[i], (size_t) linelen);
			linebuf[linelen] = '\0';
			logit(linebuf, 0);
		}
	}
}

#endif /* LOGFILE */

/*
 * General program termination routine.
 *
 * First parameter is the error number:
 *   zero: no real error: show the user the message.
 *   nonzero: error indicator and message are the parameters.
 *            Error indicator interpretation:-
 *            if negative the user may not be on his 3270 screen
 *                      (but the term_flag mode really tells us!)
 *            if  = -1    then put just the message, not the error indicator, to the user.
 *            if <= -900  then do not put the message to the user.
 *            if <= -1000 (catastrophic!) be extra careful.
 *
 * Second parameter is the error message.
 *
 * In the case of zero error number the routine will return, otherwise
 * it will call the exit routine.
 */

#ifdef STANDARD_C

void    end_program(int code , char *message , int int1)

#else /* STANDARD_C */

void    end_program(code , message , int1)

int     code;
char    *message;
int     int1;

#endif /* STANDARD_C */

{
	int     ferror_res;
	char    form_mess[132]; /* to store formatted message    */

	extern int errno;

	if (ending_program)
	{
		/* Clearly we have got into a loop calling end_program */

#ifdef LOGFILE
	logit("Emergency: recursive call of end_program", 0);
#endif /* LOGFILE */

		reset_tty(0, FALSE);
		exit(999);
	}

	/* Flag if we are now in the end_program routine on a genuine error */

	ending_program = code;

	/* unset signal traps and kill any outstanding timers first */

	unset_signals();

#ifdef TTY
	set_cpu_timer(FALSE);
#endif /* TTY */

	sprintf(&form_mess[0], message, int1);

#ifdef LOGFILE
	logit(form_mess, 0);
#endif /* LOGFILE */

#ifdef DEBUG
	if (outf)
	{
		(void)fprintf(outf,"end_program : (%d) : %s errno = %d : \n" ,
				code, getstime(),  errno);
		(void)fprintf(outf, "end_program : %s", form_mess);
		(void)fprintf(outf,"\n");
	}
#endif /* DEBUG */

	if (ferror_res = ferror(stdin))
	{

#ifdef DEBUG
	if (outf)
	{
		(void)fprintf(outf, "end_program : %s ferror(stdin) = %d\n",
				getstime(), ferror_res);
	}
#endif /* DEBUG */

#ifdef LOGFILE
	logit("ferror(stdin) = %d", ferror_res);
#endif /* LOGFILE */

		(void) clearerr(stdin);
	}

	if (ferror_res = ferror(fd_stream))
	{

#ifdef DEBUG
	if (outf)
	{
		(void)fprintf(outf, "end_program : %s ferror(fd_stream) = %d\n",
				getstime(), ferror_res);
	}
#endif /* DEBUG */

#ifdef LOGFILE
	logit("ferror(fd_stream) = %d", ferror_res);
#endif /* LOGFILE */

		(void) clearerr(fd_stream);
	}

	/*
	 * If no (longer) buffered output to the user then make
	 * a quick and simple exit.
	 * This can happen either before the initial call of init_tty
	 * or after the final call of reset_tty(0, FALSE)
	 */

	if (!term_mode)
	{
		(void)fprintf(stderr, "%s\n", form_mess);
		reset_tty(0, FALSE);
		sleep(1);
		exit(code);
	}

	/*
	 * if in asynchronous mode clear any hanging output
	 * and change to synchronous mode.
	 */

	if (!synch_op)
	{
		f_clearop(1);
		reset_tty(SYNCH_NORMAL, TRUE);
	}

	if (curscr)
	{
		/* Maybe try to switch back to the initial screen size */

		if (normal_COLS != altern_COLS)
		{
			/*
			 * User had two different screens.
			 * Switch back to whichever looks nearer the original.
			 */

			if (altscreen)
				clearsn = (init_COLS < 132 ? 1 : 0);
			else
				clearsn = (init_COLS < 132 ? 0 : 2);

			clscreen();
		}

		clearnormal();
	}

	if (term_mode)
	{
		slclear();
		if (sline)
			slterm();
	}

	if (code)
	{
		if (code > -900)
		{
			if (code != -1)
				(void)f_putf("error(%d)", FALSE, code);
			(void)f_puts(form_mess, FALSE);
			(void)f_puts("\r\n", TRUE);
			(void)sleep(1);
			f_puttcs(tcsCL, TRUE);
		}
	}
	else
	{
		(void)f_puts(form_mess, FALSE);
		(void)f_puts("\r\n", TRUE);
	}

	if (code >= -900)
	{
		(void)sleep(1);

		/* reset the terminal state and clear the screen */

		f_puttcs(tcsME, FALSE);
		f_puttcs(tcsIS, FALSE);
		f_puttcs(tcsCL, TRUE);
	}

	if (code >= -1000)
	{
		f_puttcs(tcsTE, TRUE);

#ifdef DEBUG
	(void)fprintf(outf, "end_program : %s cleaned up screen now\n",
			getstime());
#endif /* DEBUG */

	}

	/* now free the interpreted termcap strings */

	free((void *) tcsptr[0]);

	/* Now the screen buffers */

	if (screen)
	{
		free((void *)screen);

#ifdef EXTENDED
		free((void *)screen_ex);
#endif /* EXTENDED */

		free((void *)s_point);
		screen = NULL;
	}

	if (savebuf)
		free((void *) savebuf);

#ifdef LOGFILE
	logit((code ? "Error (%d) exit" : "Normal exit") , code);
#endif /* LOGFILE */

	/* try to close the socket down */

	if (net)
		close_socket();

	reset_tty(0, FALSE);

#ifdef DEBUG
	(void)fprintf(outf, "end_program : %s called reset_tty\n",
			getstime());
#endif /* DEBUG */


	if (code)
	{

#ifdef DEBUG
	(void)fprintf(outf, "end_program : %s now call exit(%d)\n",
			getstime(), code);
#endif /* DEBUG */
		exit(code);
	}
}

/*
 * Get the options for this particular user.
 * Include the specs for the last ANNOUNCE file written to this user
 * and return this as a positive integer.
 * In the case that no options file exists then return -1.
 * Much of the above is only possible if there is a defined
 * directory for the options files (OPTDIR).
 * If there is then remember that it may contain a file called default_hosts
 * which lists the hosts considered as the local default (and there may
 * be several if the IBM has several entry ports).
 */

#ifdef STANDARD_C

int     get_options(void)

#else /* STANDARD_C */

int     get_options()

#endif /* STANDARD_C */

{
	int     opt_nread = 0;

#ifdef OPTDIR

	int     inp_integer;
	char    inp_string[41] = "                                        ";

#ifdef DEBUG
	(void)fprintf(outf, "get_options : opt_file is %s\n",
			opt_file);
#endif /* DEBUG */

	if ((opt_fstr = fopen(opt_file, "r")) != NULL)
	{
		for (opt_nread = 0 ; opt_nread < OPT_COUNT ; opt_nread++)
		{
			/* If the call was specifically with options then do not read from the file */

			if (!opt_smaxlen[opt_nread])
			{
				/* Numeric parameter */

				if (fscanf(opt_fstr, opt_format[opt_nread], &inp_integer) != 1)
					break;

				if (*((int *)opt_pointer[opt_nread]) < 0)
				{
					/* OK, call had no input specification */

#ifdef DEBUG
	(void)fprintf(outf, "get_options : replace option %d (%d) by %d\n",
			    opt_nread, *((int *)opt_default[opt_nread]), inp_integer);
#endif /* DEBUG */

					*((int *)opt_pointer[opt_nread]) = inp_integer;

				}
			}
			else
			{
				/* string parameter */

				if (fscanf(opt_fstr, opt_format[opt_nread], inp_string) != 1)
					break;

				if (!strcmp((char *)opt_pointer[opt_nread], (char *)opt_default[opt_nread]))
				{
					/* OK, call had no input specification */

#ifdef DEBUG
		(void)fprintf(outf, "get_options : replace option %d <%s> by <%s>\n",
				    opt_nread, (char *)opt_default[opt_nread], &inp_string);
#endif /* DEBUG */

					strcpy((char *)opt_pointer[opt_nread], inp_string);
				}
			}
		}
		(void)fclose(opt_fstr);

#ifdef DEBUG
	(void)fprintf(outf, "get_options : managed to read in %d options\n",
			opt_nread);
#endif /* DEBUG */

	}
#endif /* OPTDIR */

	for (opt_nread = 0 ; opt_nread < OPT_COUNT ; opt_nread++)
	{
		if (!opt_smaxlen[opt_nread])
		{

#ifdef DEBUG
	(void)fprintf(outf, "get_options : default option %d (%d)\n",
			    opt_nread, *((int *)opt_default[opt_nread]));
#endif /* DEBUG */

			if (*((int *)opt_pointer[opt_nread]) < 0)
			{
				*((int *)opt_pointer[opt_nread]) = *((int *)opt_default[opt_nread]);

#ifdef DEBUG
	(void)fprintf(outf, "get_options : option %d set to %d\n",
			opt_nread, *((int *)opt_pointer[opt_nread]));
#endif /* DEBUG */

			}
		}
		else
		{

#ifdef DEBUG
	(void)fprintf(outf, "get_options : default option %d (%s)\n",
			    opt_nread, (char *)opt_default[opt_nread]);
#endif /* DEBUG */

			if (!strcmp((char *)opt_pointer[opt_nread], (char *)opt_default[opt_nread]))
			{
				strcpy((char *)opt_pointer[opt_nread], (char *)opt_default[opt_nread]);

#ifdef DEBUG
	(void)fprintf(outf, "get_options : option %d set to %s\n",
			opt_nread, (char *)opt_pointer[opt_nread]);
#endif /* DEBUG */

			}
		}
	}

	return ((int)*((int *)opt_pointer[0]));
}

#ifdef OPTDIR

/*
 * Write a new options file for this particular user.
 * If called with a parameter of zero then simply rewrite the existing
 * default options. This call is normally to rewrite the indicator of
 * which announce file has just been read.
 * A call with non-zero means update with the current settings.
 * A negative return value indicates failure.
 */

#ifdef STANDARD_C

int     set_options(int all_options)

#else /* STANDARD_C */

int     set_options(all_options)

int     all_options;

#endif /* STANDARD_C */

{
	int     opt_nwrite = 0;

#ifdef DEBUG
	(void)fprintf(outf, "set_options : (%d) : opt_file is %s\n",
			all_options, opt_file);
#endif /* DEBUG */

	if ((creat(opt_file, 0666) < 0) ||
	    ((opt_fstr = fopen(opt_file, "w")) == NULL))
	{

#ifdef DEBUG
	(void)fprintf(outf, "cannot write options file : create returned %d\n",
			creat(opt_file, 0666));
#endif /* DEBUG */

		return (-1);
	}

	for (opt_nwrite = 0 ; opt_nwrite < (all_options ? OPT_COUNT : 1) ; opt_nwrite++)
	{
		if (!opt_smaxlen[opt_nwrite])
			*((int *)opt_default[opt_nwrite]) = *((int *)opt_pointer[opt_nwrite]);
		else
			strcpy((char *)opt_default[opt_nwrite], (char *)opt_pointer[opt_nwrite]);
	}
	for (opt_nwrite = 0 ; opt_nwrite < OPT_COUNT ; opt_nwrite++)
	{
		if (!opt_smaxlen[opt_nwrite])
			(void)fprintf(opt_fstr, opt_format[opt_nwrite], *((int *)opt_pointer[opt_nwrite]));
		else
			(void)fprintf(opt_fstr, opt_format[opt_nwrite], (char *)opt_pointer[opt_nwrite]);
	}
	(void)fclose(opt_fstr);

	return (opt_nwrite);
}
#endif /* OPTDIR */

/*
 * Write out a given file if possible
 * Return a positive indicator specifying the file (add up the characters).
 * If unable to open the file then return -1.
 * If outflag is negative then do not actually output it.
 * If outflag is positive then wait until the user acks with a Return
 */

#ifdef STANDARD_C

int     fileout(char *filename, int outflag)

#else /* STANDARD_C */

int     fileout(filename, outflag)

char    *filename;
int     outflag;

#endif /* STANDARD_C */

{
	FILE    *filestr;
	int     ch;
	int     i, n;
	int     counter = -1;
	char    buf[BUFSIZ];

#ifdef DEBUG
	(void)fprintf(outf, "fileout     : Output file <%s>, outflag = %d\n",
			filename, outflag);
#endif /* DEBUG */

	if ((filestr = fopen(filename , "r")) != NULL)
	{
		if (outflag >= 0)
			f_puttcs(tcsCL, FALSE);
		while((n = fread((void *) &buf[0], 1, BUFSIZ, filestr)) > 0)
		{
			if (outflag >= 0)
				f_addchars(&buf[0], n, FALSE);
			for (i = 0 ; i < n ; i++)
				counter += buf[i];
		}
		(void) fclose(filestr);
		if (counter < 0)
			counter = -counter;
		if (outflag > 0)
		{
			f_puts("\nWhen you have read this type simply the <Return> key ", FALSE);
			empty_input();
			while (((ch = f_getc(60)) != EOF) && (ch != '\n') && (ch != '\r'))
			       ;
			f_puttcs(tcsCL, FALSE);
		}
		f_flush();
	}

#ifdef DEBUG
	(void)fprintf(outf, "fileout     : Return file value %d\n",
			counter);
#endif /* DEBUG */

	return (counter);
}

/*
 * netopen
 * opens the user end of a socket and connects it to the specified port on
 * the specified host.  Returns a file-descriptor attached to the socket.
 * The socket will be opened O_RDWR.  Exit(1) with an error message if the
 * socket couldn't be opened.  If the host argument is a host name (rather
 *  than a dotted address), also prints the host address (in dotted form)
 *  on stderr.
 *
 * Parameters:
 * host   name of host to connect to
 * port   port on the host to connect to
 */

#ifdef STANDARD_C

int     netopen(char *host, int port)

#else /* STANDARD_C */

int     netopen(host, port)

char    *host;
int     port;

#endif /* STANDARD_C */

{
		struct  hostent         *gethost();

	register struct hostent         *hp;
	register                        s;
		struct  sockaddr_in     rmt_address;
		int     setres;

	(void) memset((void *)&rmt_address, 0, (size_t) sizeof (struct sockaddr_in));
	hp = gethost(host);
	if (hp == (struct hostent *)0)
		end_program(-4 , "unknown host", 0);

	rmt_address.sin_family = hp->h_addrtype;
	if((s = socket(hp->h_addrtype, SOCK_STREAM, 0)) < 0)
		end_program(-5 , "cannot get socket on host", 0);
	if (bind(s, (struct sockaddr *)&rmt_address, sizeof (rmt_address)) < 0)
		end_program(-6 , "cannot bind to host", 0);

	(void) memcpy((void *)&rmt_address.sin_addr, (void *) hp->h_addr, (size_t) hp->h_length);
	rmt_address.sin_port = htons((unsigned short)port);

/* Try to set maximum segment size option */

#ifdef TCP_MAXSEG
	{
		int     optval = 1460;
		int     optlen = sizeof (int);

		setres = setsockopt(s, IPPROTO_TCP, TCP_MAXSEG, (char *)(&optval), optlen);

#ifdef DEBUG
	if (setres < 0)
	{
		(void)fprintf(stderr, "netopen     : setsockopt for TCP_MAXSEG returns %d\n",
				setres);
		(void) sleep(2);
	}
#endif /* DEBUG */

	}
#endif /* TCP_MAXSEG */

/* Ensure data goes out immediately */

#ifdef TCP_NODELAY
	{
		int     optval = 1;
		int     optlen = sizeof (int);

		setres = setsockopt(s, IPPROTO_TCP, TCP_NODELAY, &optval, optlen);

#ifdef DEBUG
	if (setres < 0)
	{
		(void)fprintf(stderr, "netopen     : setsockopt for TCP_NODELAY returns %d\n",
				setres);
		(void) sleep(2);
	}
#endif /* DEBUG */

	}
#endif /* TCP_NODELAY */

	if (connect(s, (struct sockaddr *)&rmt_address, sizeof (rmt_address)) < 0)
		end_program(-7 , "unable to connect to host", 0);

	return s;
}

/*
 * Look up a host name and return the internet address.
 * Return 0 on failure.
 * If the host name starts with a digit, assume it is a "dotted address".
 * Otherwise (if successful), print the dotted address on stderr.
 */

#ifdef STANDARD_C

struct  hostent *gethost(char *name)

#else /* STANDARD_C */

struct  hostent *gethost(name)

char    *name;

#endif /* STANDARD_C */

{

#ifndef inet_addr
	extern  unsigned long   inet_addr();
#endif /* inet_addr */

	static struct hostent hostentry;

#ifndef IPAD_ONLY
	register struct hostent *hp;
#endif /* IPAD_ONLY */

	if  (*name==0) return((struct hostent *)0);
	if ('0' <= *name && *name <= '9')
	{
		/* dotted address */
		hostentry.h_addrtype = AF_INET;
		ip_address = inet_addr(name);
		if (ip_address == -1) return((struct hostent *)0);
#ifdef h_addr
		/* In version 4.3 BSD UNIX, h_addr is a macro for h_addr_list[0] */
		{ 
			static char *addr_list[2];
			addr_list[0] = (char *) & ip_address;
			addr_list[1] = (char *) 0;
			hostentry.h_addr_list = addr_list;
		}
#else /* h_addr */
		hostentry.h_addr = (char *)&ip_address;
#endif /* h_addr */
		hostentry.h_length = sizeof (long);
		return(&hostentry);
	}

#ifdef IPAD_ONLY
	end_program(-21, "Sorry, I can only handle IP addresses, not names", 0);
	return((struct hostent *) NULL);
#else /* IPAD_ONLY */
	if ((hp = gethostbyname(name)) != (struct hostent *)NULL)
		ip_address = *((unsigned long *)hp->h_addr);
	(void)fflush(stderr);
	return(hp);
#endif /* IPAD_ONLY */

}
