/*
 *     Copyright CERN, Geneva 1989 - Copyright and any other
 *     appropriate legal protection of these computer programs
 *     and associated documentation reserved in all countries
 *     of the world.
 */

/*
 * routines to handle status line I/O
 */

#include "telnet.h"
#include "3270.h"
#include "globals.h"

/* a copy of the status line */
char    slimage_array[133];

/* a saved copy of the status line */
char    slisave_array[133];

/* a line of blanks (enough for at least a full line */
char    blanks_array[133];

#define SLFMSZ  22              /* maximum size of slfset message       */
#define SLFOFF  11              /* offset of slfset message             */

bool    slflags = FALSE;        /* True if there is an slfset message   */
bool    slfsave = FALSE;        /* Saved value of slflags after slsave  */
int     slchars;                /* Chars to put into status line        */

/* The required extended characteristics for the status line */

#ifdef EXTENDED

#define SL_SE   0x30            /* Colour 3 (purple), no highlighting   */
#define SL_SEB  0x41            /* Colour 3 (green), blinking           */

#endif /* EXTENDED */


/* Now all of the status line routines */

/*
 * Put a string onto the status line at an offset position.
 * A negative value implies from the start but without changing the
 * style: this is used to clear the line.
 * The bells parameter indicates how many times to ring the bell to warn
 * the user: 1 means an information message, >1 is a warning.
 * For an information message the bell will NOT be rung if the user
 * has a status line.
 * Note, however, that no bells will be rung if the user has switched
 * to quiet mode (bells_mode = 0).
 */

#ifdef STANDARD_C

void    slputstr(char *slstr, int offset, int bells)

#else /* STANDARD_C */

void    slputstr(slstr, offset, bells)

char    *slstr;
int     offset;
int     bells;

#endif /* STANDARD_C */

{
	int             slstr_len;      /* length of string to be inserted */
	int             offreal;        /* the real offset. */
	int             i;
	unsigned char   slstyle;        /* style of stuff to be put in */
	unsigned char   style_save;     /* save current screen style */

#ifdef EXTENDED
	unsigned char   slextend;       /* extend mode of stuff to be put in */
	unsigned char   extend_save;    /* save current extend mode */
#endif /* EXTENDED */

	int             n_bells;        /* the real number of bell rings */

#ifdef DEBUG
	(void)fprintf(outf, "slputstr    : %d bells: at offset %d put string\n<%s>\n",
		  bells, offset, slstr);
#endif /* DEBUG */

	n_bells = bells_mode * bells;

	offreal = (offset < 0 ? 0 : offset);
	slstr_len = strlen(slstr);
	if (sline)
	{
		if (bells > 1)
		{
			slstyle = STATL_BLINK;

#ifdef EXTENDED
			slextend = SL_SEB;
#endif /* EXTENDED */

		}
		else
		{
			slstyle = STATL;

#ifdef EXTENDED
			slextend = SL_SE;
#endif /* EXTENDED */

			n_bells = 0;
		}

		style_save = curscr->_style;

#ifdef EXTENDED
		extend_save = curscr->_extend;
#endif /* EXTENDED */

		if (slinenum)
		{

#ifdef DEBUG
	(void)fprintf(outf, "slputstr    : move from ly = %d, lx = %d, style = %d\n",
		  ly, lx, curscr->_style);
#endif /* DEBUG */

			domvcur(ly, lx, slinenum - 1, offreal);
			if (offset >= 0)
			{

#ifdef EXTENDED
				set_cse(slstyle, slextend, TRUE);
#else /* EXTENDED */
				set_cse(slstyle, TRUE);
#endif /* EXTENDED */

			}
			f_puts(slstr, FALSE);
			domvcur(slinenum - 1, offreal+slstr_len, ly, lx);
			if (offset >= 0)
			{

#ifdef EXTENDED
				set_cse(style_save, extend_save, TRUE);
#else /* EXTENDED */
				set_cse(style_save, TRUE);
#endif /* EXTENDED */

			}
		}
		else
		{
			/*
			 * For a VT300 the sequence to select the host-writeable status line
			 * needs no parameters. However, some other sequences such as
			 * those on aixterm in an rs6000 expect a count of how far down
			 * the line to go.
			 * For simplicity we always go to the start then rewrite the start
			 * of the status line up to where the new stuff will go. We then
			 * add in the new stuff.
			 * However, some selection sequences actually blank out the status
			 * line, meaning that we then have to rewrite the rest of it. This
			 * can be a bit annoying, especially on a LAT-connected ASCII terminal
			 * when all that we are doing is inserting then deleting the
			 * character which says the IBM is unable to accept input.
			 * For safety we assume that the status line WILL be blanked out
			 * when selected by the TS sequence UNLESS the symbol NO_SL_CLEAR
			 * is defined.
			 * On systems running ASCII terminals, including the TAGIBM type
			 * of application, it is highly recommended to ensure that no
			 * termcap sequence includes the blanking out and to include the
			 * -DNO_SL_CLEAR option in the Makefile.
			 */

			/* f_puttcs(tcsTS, FALSE); */
			tputs(tgoto(TS, 0, 0), 1, f_putcs);

			if (offset >= 0)
			{

#ifdef EXTENDED
				set_cse(STATL, SL_SE, TRUE);
#else /* EXTENDED */
				set_cse(STATL, TRUE);
#endif /* EXTENDED */

			}
			slimage_array[offreal] = '\0';
			f_puts(slimage_array, FALSE);
			if (offset >= 0)

#ifdef EXTENDED
				set_cse(slstyle, slextend, TRUE);
#else /* EXTENDED */
				set_cse(slstyle, TRUE);
#endif /* EXTENDED */

			f_puts(slstr, FALSE);
			if (offset >= 0)

#ifdef EXTENDED
				set_cse(STATL, SL_SE, TRUE);
#else /* EXTENDED */
				set_cse(STATL, TRUE);
#endif /* EXTENDED */


			/*
			 * The next line is only needed if the ts string in the
			 * termcap entry includes clearing the status line.
			 * It is much faster if it does not.
			 */

#ifndef NO_SL_CLEAR

			f_puts(slimage_array+offreal+slstr_len, FALSE);

#endif /* NO_SL_CLEAR */

			if (offset >= 0)
			{

#ifdef EXTENDED
				set_cse(style_save, extend_save, TRUE);
#else /* EXTENDED */
				set_cse(style_save, TRUE);
#endif /* EXTENDED */

			}
			f_puttcs(tcsFS, FALSE);
		}
		f_flush();
	}

	(void)strncpy(slimage_array + offreal , slstr , (size_t) slstr_len);

#ifdef DEBUG
	(void)fprintf(outf, "slputstr    : status line is now %d chars\n<%s>\n",
		  strlen(slimage_array), slimage_array);
#endif /* DEBUG */

	for (i = n_bells ; i ; i--)
		DING
	if (bells > 1)
		clear_kbin();
}

/*
 * put a number of blanks onto the status line at an offset position
 */

#ifdef STANDARD_C

void    slerase(int count, int offset, int bells)

#else /* STANDARD_C */

void    slerase(count, offset, bells)

int     count;
int     offset;
int     bells;

#endif /* STANDARD_C */

{
	blanks_array[count] = '\0';
	slputstr(blanks_array, offset, bells);
	blanks_array[count] = ' ';
}

/*
 * save status line image (for later restore)
 */

#ifdef STANDARD_C

void    slsave(void)

#else /* STANDARD_C */

void    slsave()

#endif /* STANDARD_C */

{
	int i;

	for (i = 0 ; i <= 132 ; i++)
		slisave_array[i] = slimage_array[i];
	slfsave = slflags;

#ifdef DEBUG
	(void)fprintf(outf, "slsave      : status line save\n");
	(void)fprintf(outf, "slsave      : saved line is now %d chars\n<%s>\n",
		  strlen(slisave_array), slisave_array);
#endif /* DEBUG */

	slisave_array[slchars] = ' ';
	slclear();
	return;
}

/*
 * restore saved status line and redraw it
 */

#ifdef STANDARD_C

void    slrestore(void)

#else /* STANDARD_C */

void    slrestore()

#endif /* STANDARD_C */

{
	slchars = (VW ? linelen : linelen-1);
	slimage_array[slchars] = '\0';
	slisave_array[slchars] = '\0';
	slputstr(slisave_array, 0, 0);
	slflags = slfsave;

#ifdef DEBUG
	(void)fprintf(outf, "slrestore   : status line restore\n");
	(void)fprintf(outf, "slrestore   : restored line is now %d chars\n<%s>\n",
		  strlen(slimage_array), slimage_array);
#endif /* DEBUG */

	return;
}

/*
 * clear status line
 */

#ifdef STANDARD_C

void    slclear(void)

#else /* STANDARD_C */

void    slclear()

#endif /* STANDARD_C */

{
	slerase(slchars, -1, 0);
}

/*
 * refresh status line
 */

#ifdef STANDARD_C

void    slrefresh(void)

#else /* STANDARD_C */

void    slrefresh()

#endif /* STANDARD_C */

{
	int i;
	char    slicopy_array[133];

#ifdef DEBUG
      (void)fprintf(outf, "slrefresh   : slchars %d, linelen %d VW %d\n",
			      slchars,linelen,VW);
#endif /* DEBUG */

	slimage_array[slchars] = ' ';
	slchars = (VW ? linelen : linelen-1);
	slimage_array[slchars] = '\0';

	for (i = 0 ; i <= 132 ; i++)
		slicopy_array[i] = slimage_array[i];
	slputstr(slicopy_array, 0, 0);
}

/*
 * initialise status line
 */

#ifdef STANDARD_C

void    slinit(void)

#else /* STANDARD_C */

void    slinit()

#endif /* STANDARD_C */

{
	int i;

#ifdef DEBUG
	(void)fprintf(outf, "slinit      : status line initialise\n");
#endif /* DEBUG */

	for (i = 0 ; i <= 132 ; i++)
	{
		slimage_array[i] = ' ';
		blanks_array[i] = ' ';
	}
	slchars = (VW ? linelen : linelen-1);
	slimage_array[slchars] = '\0';
	if (sline)
	{
		if (!slinenum)
		{
			f_puttcs(tcsDS, FALSE);
		}
	}
	slputstr("ONLINE   ", 0 , 1);
	return;
}

/*
 * terminate status line usage
 */

#ifdef STANDARD_C

void    slterm(void)

#else /* STANDARD_C */

void    slterm()

#endif /* STANDARD_C */

{
	slclear();
	slinenum = 0;
	sline = FALSE;
}

/*
 *      Put a temporary information message onto the status line.
 *      This will always go to a fixed place, and will be removed
 *      by a call of sfclear pretty soon afterwards.
 *      As usual, more than one bell implies a warning!
 */

#ifdef STANDARD_C

void    slfset(char *slstr, int bells)

#else /* STANDARD_C */

void    slfset(slstr, bells)

char    *slstr;
int     bells;

#endif /* STANDARD_C */

{
	char    slfstr[SLFMSZ+1];

	if (slflags)
		slfclear();
	slfstr[SLFMSZ] = '\0';
	strncpy(slfstr, slstr, SLFMSZ);
	slputstr(slfstr, SLFOFF, bells);
	slflags = TRUE;
}

/*
 *      Clear any flags in status line
 */

#ifdef STANDARD_C

void    slfclear(void)

#else /* STANDARD_C */

void    slfclear()

#endif /* STANDARD_C */

{
	if (slflags)
	{

#ifdef DEBUG
	(void)fprintf(outf, "slfclear    : slflags entry %x\n",
		  slflags);
#endif /* DEBUG */

		slerase(SLFMSZ, SLFOFF, 0);
		slflags = FALSE;
	}
}
