/*
 *     Copyright CERN, Geneva 1989 - Copyright and any other
 *     appropriate legal protection of these computer programs
 *     and associated documentation reserved in all countries
 *     of the world.
 */

/*
 * Work out the size of the screen.
 * Also check up on other items which affect termcap sequences,
 * such as how the display wraps at the right hand side, and
 * especially at the bottom right hand corner.
 */

#include "globals.h"

#ifdef NOSIZER

/*
 * Don't try to check the screen size and termcap parameters.
 * Return -1 to indicate immutable screen size values.
 */

#ifdef STANDARD_C

int     sizer(void)

#else /* STANDARD_C */

int     sizer()

#endif /* STANDARD_C */

{                                                     
	LINES = tgetnum("li");
	COLS = tgetnum("co");

#ifdef DEBUG
	(void)fprintf(outf, "sizer       : tgetnum reports %d lines, %d columns\n",
				LINES, COLS);
#endif /* DEBUG */

#ifdef sys_apollo
	/* An Apollo seems to indicate one less than the real size! */
	/* I wonder if there are any others that do likewise?       */

	LINES += 1;
	COLS ++1;
#endif /* DEBUG */

	return(-1);
}

#else /* NOSIZER */

int     cline, ccol;

int     maxlines = 70;
int     maxcols = 134;

/* for the terminal mode                        */

struct  TTY_STRUCT      oldtty, newtty;

#ifdef sys_sun

#define GETSIZE         "\033[18t"
#define SIZE            "8;%d;%dt"
#define TERMNAME        "sun"

#endif /* sys_sun */

#ifdef sys_apollo

#define GETSIZE         "\033[50n"
#define SIZE            "%d;%dS"
#define TERMNAME        "apollo"

#endif /* sys_apollo */

char    *getsize = NULL;
char    *size = NULL;

/*
 * Get the next input character (but not waiting too long!).
 * Return -1 on no available character.
 * Return the character if there is one.
 */

#define rdbuf_max       32

static  char    rdbuf[rdbuf_max];
static  char    *rdbuf_p        = rdbuf;        /* pointer to next char */
static  int     rdbuf_count     = 0;            /* count of remainder   */

#ifdef STANDARD_C

int     next_char(int prods)

#else /* STANDARD_C */

int     next_char(prods)

int     prods;

#endif /* STANDARD_C */

{
	int     i, selres;

#define TICKS   50000

	static  int     secs = 0;
	static  int     ticks = TICKS;

	/* next try for characters already read from the keyboard */
	while (!rdbuf_count)
	{
		/* We may have to prod for input! */

		for (i = 0 ; i <= prods; i++)
		{
			if (i)
			{
				/* No response: let's increase the timer */

				if ((ticks += TICKS) >= 1000000)
				{
					ticks -= 1000000;
					secs += 1;
				}

#ifdef DEBUG
		(void)fprintf(outf, "sizer       : increase wait to %d:%d\n",
					secs, ticks);
#endif /* DEBUG */

				/* if waiting seconds then prod */

				if (secs)
				{
					f_putc(PC ? PC : '\0');
					f_flush();
				}
			}

			selres = select_check(1, 1, 0, 1, secs, ticks, -1951, "sizer select");
			if (selres)
				break;
		}
		if (!selres)
			return (-1);

		rdbuf_count = read(0,rdbuf,rdbuf_max);
		if (rdbuf_count==0)
		{
			/* "Can't happen", but it doesn't hurt to be safe */
			break;
		}
		if (rdbuf_count < 0)
			end_program(1915, "keyboard read failure %d", rdbuf_count);
		rdbuf_p = rdbuf;

#ifdef DEBUG
	if (rdbuf_count)
		(void)fprintf(outf, "next_char   : read in %d characters",rdbuf_count);
	for (i = 0 ; i < rdbuf_count ; i++)
	{
		if (!(i % 20))
		(void)fprintf(outf, "\nnext_char   : ");
		(void)fprintf(outf, "%s", print_ascii((unsigned short)rdbuf[i]));
	}
		(void)fprintf(outf, "\n");
#endif /* DEBUG */

	}

	if (rdbuf_count)
	{
		rdbuf_count--;
		return ((*rdbuf_p++) & 0x00ff);
	}
	return (-1);
}

/*
 * Get the cursor position by sending the standard enquiry sequence.
 * The column and line positions go into ccol and cline.
 * Normal function return value is > 0.
 * In case of failure return -1 for timeout, 0 of could not scan for two numbers.
 */

#ifdef STANDARD_C

extern  wait_scanf(void)

#else /* STANDARD_C */

int     wait_scanf()

#endif /* STANDARD_C */

{
#define READMAX 100

	int     char_in;
	int     scann;
	char    readbuf[READMAX];
	int     readind = 0;
	char    size_lch;

	cline = -1;

	/* Now clear any hanging input before getting the size */

	empty_input();

	f_puts(getsize, TRUE);

	/*
	 * first wait until we get CSI or ESC [
	 * but allow for the case where the sending system sends CSI
	 * but the data path is only 7-bit, so it comes as ESC.
	 */

	while ((char_in = next_char(20)) >= 0)
	{
		if (char_in == 0x9b)
			break;
		if (char_in == 0x1b)
		{
			char_in = next_char(5);
			if ((char_in >= 0) && (char_in != '['))
			{
				readbuf[readind++] = char_in;
			}
			break;
		}

#ifdef DEBUG
	(void)fprintf(outf, "wait_scanf  : ignore initial input character %s\n",
				print_ascii((unsigned short)char_in));
#endif /* DEBUG */

	}

	if (char_in >= 0)
	{
		/* Now let us read in until at least the last character of size */

		size_lch = size[strlen(size) - 1];
		while (readind < (READMAX - 1))
		{
			if ((char_in = next_char(5)) < 0)
				break;
			readbuf[readind++] = char_in;

			if (char_in == size_lch)
				break;
		}
	}

	if (char_in >= 0)
	{
		/* Now keep reading until timeout or buffer full */

		while (readind < (READMAX - 1))
		{
			if ((char_in = next_char(0)) < 0)
			{
				char_in = 256;
				break;
			}
			readbuf[readind++] = char_in;
		}

		readbuf[readind] = '\0';
		scann = sscanf(readbuf, size, &cline, &ccol);

#ifdef DEBUG
	(void)fprintf(outf, "wait_scanf  : string read in (%d characters) \"",
				readind);
	for (readind = 0 ; readbuf[readind] ; readind++)
		(void)fprintf(outf, "%s", print_ascii((unsigned short)readbuf[readind]));
	(void)fprintf(outf, "\"\n");
	(void)fprintf(outf, "wait_scanf  : sscanf returns %d, finds cline = %d, ccol = %d, returns %d\n",
				scann, cline, ccol, ((scann == 2) ? scann : 0));
#endif /* DEBUG */

		if (scann == 2)
			return (scann);
		else
			return (0);
	}
	else
		return (-1);

}

/*
 * Routine to size the screen and set associated variables.
 * Return 1 if the screen size has been found correctly.
 * In case of a problem return 0 (with COLS = LINES = 0)
 */

#ifdef STANDARD_C

int     sizer(void)

#else /* STANDARD_C */

int     sizer()

#endif /* STANDARD_C */

{                                                     
	int     i;
	int     getset_res;

	LINES = 0;
	COLS = 0;

	/*
	 * Really I need RAW mode set for this. However, some
	 * terminal servers seem to then switch into passall mode.
	 * Some terminals then cannot handle the rate of input and
	 * try to signal with X-OFF, which slides through the terminal
	 * server but does not get handled properly.
	 * To handle it properly I would have to check for input after
	 * each character output, which would be a pain.
	 */

#ifdef USE_SYSV_TERMIO

#ifdef USE_TERMIOS

	while ((getset_res = tcgetattr(fd_output, &oldtty)) < 0)
		sys_fail(-1960, getset_res);

#else /* USE_TERMIOS */

	while ((getset_res = ioctl(fd_output, TCGETA, &oldtty)) < 0)
		sys_fail(-1960, getset_res);

#endif /* USE_TERMIOS */

	newtty = oldtty;
	newtty.c_lflag = (newtty.c_lflag & ~ICANON & ~ECHO & ~ECHOE & ~ECHOK & ~ECHONL);
	newtty.c_cc[VMIN] = 1;
	newtty.c_cc[VTIME] = 1;

#ifdef USE_TERMIOS

	while ((getset_res = tcsetattr(fd_output, TCSADRAIN, &newtty)) < 0)
		sys_fail(-1961, getset_res);

#else /* USE_TERMIOS */

	while ((getset_res = ioctl(fd_output, TCSETAW, &newtty)) < 0)
		sys_fail(-1961, getset_res);

#endif /* USE_TERMIOS */

#else /* USE_SYSV_TERMIO */

	while ((getset_res = ioctl(fd_output, TIOCGETP, (char *)&oldtty)) < 0)
		sys_fail(-1960, getset_res);
	newtty = oldtty;
	newtty.sg_flags = (newtty.sg_flags & ~ECHO) | RAW;
	while ((getset_res = ioctl(fd_output, TIOCSETP, (char *)&newtty)) < 0)
		sys_fail(-1961, getset_res);

#endif /* USE_SYSV_TERMIO */

#ifdef GETSIZE

	/* On a workstation one can sometimes get the size directly */

	if (!strncmp(term_type, TERMNAME, sizeof(TERMNAME)))
	{
		getsize = GETSIZE;
		size = SIZE;

		if (wait_scanf() > 0)
		{

#ifdef DEBUG
	(void)fprintf(outf, "sizer       : Special workstation sequence gives %d lines, %d columns\n",
		   cline, ccol);
#endif /* DEBUG */

			LINES = cline;
			COLS = ccol;
			AM = FALSE;
			VW = FALSE;
			return(1);
		}
	}

#endif /* GETSIZE */

	getsize = "\033[6n";
	size = "%d;%dR";

	scroll_line = -1;
	for (COLS = 0 ; !COLS ; )
	{
		if (maxcols)
		{
			int mincols=0;

			tputs(tgoto(CM, 0, 0) , 1 , f_putcs);

			/* Read in cursor position */
			/* If non-positive return (timeout) the terminal does not respond */

			if (wait_scanf() <= 0)
				break;

#ifdef DEBUG
	(void)fprintf(outf, "sizer       : After initial home cline=%d, ccol=%d\n",
		   cline, ccol);
#endif /* DEBUG */

			if ((cline != 1) || (ccol != 1))
			{

#ifdef DEBUG
	(void)fprintf(outf, "sizer       : This stupid emulator won't go home\n");
#endif /* DEBUG */

				break;
			}

			/* first move far right to guess the approx number of cols */
			tputs(tgoto(CM, 78, 0) , 1 , f_putcs);
			if (wait_scanf() <= 0)
				break;

#ifdef DEBUG
	(void)fprintf(outf, "sizer       : After tgoto(CM, 78, 0) cline=%d, ccol=%d\n",
			cline, ccol);
#endif /* DEBUG */

			/* some stupid emulators now say they are still at (1,1) */
			if (cline > 1)
			{
				mincols = ((78 - ccol)/(cline-1));
			}
			else
			{
				if (ccol < 10)
				{

#ifdef DEBUG
	(void)fprintf(outf, "sizer       : This is a failure to move the cursor: very ODD\n");
#endif /* DEBUG */

					break;
				}

				if ((mincols = ccol) == 79)
				{
					for (i = 79 ; i < 82 ; i++)
					{
						f_putc(' ');
					}
					if (wait_scanf() <= 0)
						break;

#ifdef DEBUG
	(void)fprintf(outf, "sizer       : After 82 right moves cline=%d, ccol=%d, mincols=%d\n",
		   cline, ccol, mincols);
#endif /* DEBUG */

					if (cline == 2)
						COLS = 82 - ccol;
					if ((cline == 1) && (ccol == mincols+3))
					{
						tputs(tgoto(CM, 130, 0) , 1 , f_putcs);
						if (wait_scanf() <= 0)
						{
							COLS = -1;
							break;
						}

#ifdef DEBUG
	(void)fprintf(outf, "sizer       : After tgoto(CM, 130, 0) cline=%d, ccol=%d\n",
			cline, ccol);
#endif /* DEBUG */

						if (cline > 1)
						{
							mincols = ((78 - ccol)/(cline-1));
						}
						else
						{
							if ((mincols = ccol) == 131)
							{
								for (i = 131 ; i < maxcols ; i++)
								{
									f_putc(' ');
								}
								if (wait_scanf() <= 0)
								{
									COLS = -1;
									break;
								}

#ifdef DEBUG
	(void)fprintf(outf, "sizer       : After %d right moves cline=%d, ccol=%d, mincols=%d\n",
		   maxcols, cline, ccol, mincols);
#endif /* DEBUG */

							}
						}
						if (cline == 2)
							COLS = maxcols - ccol;
					}
				}
				if (COLS)
					mincols = COLS;
				else
					if (ccol == maxcols)
						COLS = ccol;
					else
						mincols -= 2;
			}

#ifdef DEBUG
	(void)fprintf(outf, "sizer       : After initial right moves cline=%d, ccol=%d, mincols=%d\n",
		   cline, ccol, mincols);
#endif /* DEBUG */

			tputs(tgoto(CM, 0, 0) , 1 , f_putcs);
			f_flush();

			/*
			 * Now go to the bottom line of the [scroll] region.
			 * with the standard down sequence (if it exists).
			 */

			if (tcslen[tcsDO])
			{
				for (i = 0 ; i < maxlines ; i++)
					f_puttcs(tcsDO, FALSE);
				if (wait_scanf() <= 0)
					break;

#ifdef DEBUG
	(void)fprintf(outf, "sizer       : After initial moves down cline=%d, ccol=%d\n",
		   cline, ccol);
#endif /* DEBUG */

				while (scroll_line < cline)
				{
					scroll_line = cline;
					f_puttcs(tcsDO, FALSE);
					if (wait_scanf() <= 0)
					{
						scroll_line = 0;
						break;
					}
				}

				if (scroll_line == -1)
					break;
				if (scroll_line > cline)
				{
					/*
					 * Most likely the downward move has wrapped around to the
					 * top line (has been seen on aixterm on IBM RS6000).
					 * Leave the scroll line as the furthest down seen.
					 */

#ifdef DEBUG
	(void)fprintf(outf, "sizer       : down from line %d goes to line %d\n",
				scroll_line, cline);
#endif /* DEBUG */

				}

#ifdef DEBUG
	(void)fprintf(outf, "sizer       : evaluated scroll_line as %d\n", scroll_line);
#endif /* DEBUG */

			}

			/*
			 * Now use direct cursor move sequences to see if there is in fact
			 * any space below the scroll line (if found!).
			 */

			tputs(tgoto(CM, 0, maxlines - 1) , 1 , f_putcs);
			f_flush();
			if (wait_scanf() <= 0)
			{
				break;
			}

#ifdef DEBUG
	(void)fprintf(outf, "sizer       : direct move to 0,%d gives cline = %d\n",
				maxlines - 1, cline);
#endif /* DEBUG */

			if (cline == maxlines)
			{
				/*
				 * Careful: the windowing might be giving us a windowing
				 * beyond the real size. Let us stick to the scroll line.
				 */
#ifdef DEBUG
	(void)fprintf(outf, "sizer       : refuse to believe the window goes to line %d\n",
				cline);
#endif /* DEBUG */

				LINES = 24;
			}
			else
			{
				LINES = cline;
			}

			if (scroll_line > LINES)
				LINES = scroll_line;

#ifdef DEBUG
	(void)fprintf(outf, "sizer       : evaluated LINES as %d\n", LINES);
#endif /* DEBUG */

			if (COLS == maxcols)
			{
				AM = FALSE;
				VW = FALSE;
			}
			else
			{
				/* now move right to find real EOL and wrap mode */
				tputs(tgoto(CM, mincols-1, 0) , 1 , f_putcs);
				for (i = mincols ; i < maxcols ; i++)
				{
					mincols = i;
					f_putc(' ');
					if (wait_scanf() <= 0)
					{
						COLS = -1;
						break;
					}

#ifdef DEBUG
	(void)fprintf(outf, "sizer       : after space right have mincols=%d, ccol=%d, cline=%d\n",
		mincols, ccol, cline);
#endif /* DEBUG */

					if (ccol <= mincols)
					    break;
				}
				if (COLS < 0)
					break;

#ifdef DEBUG
	(void)fprintf(outf, "sizer       : after moving right have mincols=%d, ccol=%d, cline=%d\n",
		mincols, ccol, cline);
#endif /* DEBUG */

				if (ccol == mincols)
				{
					COLS = ccol;
					AM = FALSE;

					/* Now check if in VT-style wrap */
					f_putc(' ');
					if (wait_scanf() <= 0)
						break;

#ifdef DEBUG
	(void)fprintf(outf, "sizer       : another cursor right gets ccol=%d\n",
		ccol);
#endif /* DEBUG */

					if (ccol < mincols)
						AM &= TRUE;
				}
				else
				{
					COLS = mincols + 1 - ccol;
					AM &= TRUE;
				}

				VW = (ccol == 2) & AM;
			}
		}
		else
		{
			if (wait_scanf() <= 0)
				break;
			LINES = cline;
			COLS = ccol;

#ifdef DEBUG
	(void)fprintf(outf, "sizer       : non-default screen size request gives %d x %d\n",
		LINES, COLS);
#endif /* DEBUG */

		}
	}

	if (COLS < 0)
	{

#ifdef DEBUG
	(void)fprintf(outf, "sizer       : Timeout while checking columns\n");
#endif /* DEBUG */

		COLS = 0;
	}

	if (LINES < 0)
	{

#ifdef DEBUG
	(void)fprintf(outf, "sizer       : Timeout while checking lines\n");
#endif /* DEBUG */

		COLS = 0;
	}

#ifdef USE_SYSV_TERMIO

#ifdef USE_TERMIOS

	while ((getset_res = tcsetattr(fd_output, TCSADRAIN, &oldtty)) < 0)
		sys_fail(-1962, getset_res);

#else /* USE_TERMIOS */

	while ((getset_res = ioctl(fd_output, TCSETA, &oldtty)) < 0)
		sys_fail(-1962, getset_res);

#endif /* USE_TERMIOS */

#else /* USE_SYSV_TERMIO */

	while ((getset_res = ioctl(fd_output, TIOCSETP, (char *)&oldtty)) < 0)
		sys_fail(-1962, getset_res);

#endif /* USE_SYSV_TERMIO */

#ifdef DEBUG
		(void)fprintf(outf, "sizer       : set LINES %d COLS %d\n", LINES, COLS);
#endif /* DEBUG */

	/*
	 * If the terminal has [perhaps] a scrolling region which
	 * is shorter than the screen size then we cannot make any
	 * use of the automatic margins or wrap mode or CR/LF or
	 * even cursor down: pity.
	 */

	if (scroll_line == LINES)
		scroll_line = 0;
	else
	{
		AM = FALSE;
		VW = FALSE;
	}
	return (COLS ? 1 : 0);
}                                                     

#endif /* NOSIZER */
