/*
 *     Copyright CERN, Geneva 1989 - Copyright and any other
 *     appropriate legal protection of these computer programs
 *     and associated documentation reserved in all countries
 *     of the world.
 */

/*
 * Routines for parsing a terminal definition that defines escape sequences
 * from the keyboard, and for dealing with those escape sequences.
 * See also finddef.c.
 *
 *
 * The format of "profile" file is the same as "map" files used by
 * the Berkeley tn3270 program:
 *
 * Everything inside a "string" (see below) is left "as is"
 * (but newlines inside strings are not allowed).  Otherwise, whitespace
 * is ignored and everything from a "#" up to end-of-line is ignored.
 * After this space-and-comment stripping, the format of a file is
 *
 *     (name ("|" name)* definition)+
 *
 * where "definition" is
 *     "{" (action "=" string ( "|" string )* ";")* "}"
 *
 * where "name" is a terminal name (sequence of alphanumerics), "action" is
 * one of the action names listed in action.h, and "string" is similar
 * in format to a C string:
 *
 *     "'" ( "^" char
 *         | "\" ("E" | "n" | "r" | "b" | "t" | "f" | "v" | "'")
 *         | "\" octdigit [ octdigit [octdigit]]
 *         | otherchar
 *         )* "'"
 *
 * The meanings of the escapes are
 *     "^" char = (char=='?' ? '\177' : (c & 037))
 *         (NB: even if 'char' is "'")
 *     "\E" = escape (^[)
 *     "\n" = newline (^J)
 *     "\t" = horizontal tab (^I)
 *     "\r" = return (^M)
 *     "\b" = backspace (^H)
 *     "\f" = formfeed (^L)
 *     "\v" = vertical tab (^K)
 *     (the last three are not recognized by tn3270)
 *     "\" num = character with value 'num', where num is a one-, two-, or
 *         three-digit octal number (this is also an extension over tn3270).
 *         If, however, num is zero we convert it to 0x80 to avoid it
 *         looking like an end of string character.
 *     In all other cases
 *     "\" char = char
 *         (including the cases char="\" and char="'")
 *
 * It is illegal for any string to be prefix (or equal to) another in
 * any one definition.
 *
 */

# include "3270.h"
# include "globals.h"

/* List of the escape characters and their translations */

char    esc_ch[] = {   'E' ,  'n' ,  't' ,  'r' ,  'b' ,  'f' ,    'v' , '\0'};
char    esc_tr[] = {'\033' , '\n' , '\t' , '\r' , '\b' , '\f' , '\013' , '\0'};

/* debugging output (used if errors found) */

#ifdef STANDARD_C

char    *pr(char *s, int n)

#else /* STANDARD_C */

char    *pr(s,n)

char    *s;
int     n;

#endif /* STANDARD_C */

{
	static char prbuf[100];
	char *p;

	if (n<0) n = strlen(s);
	p = prbuf;
	while (n--) {
		if (*s & 0200) {
			(void) sprintf(p,"\\%03o",*s);
			p += 4;
		}
		else if (*s == 0177) {
			*p++ = '^';
			*p++ = '?';
		}
		else if (*s < 040) switch(*s) {
		case '\n': 
			*p++ = '\\'; 
			*p++ = 'n'; 
			break;
		case '\r': 
			*p++ = '\\'; 
			*p++ = 'r'; 
			break;
		case '\t': 
			*p++ = '\\'; 
			*p++ = 't'; 
			break;
		default: 
			*p++ = '^'; 
			*p++ = *s + '@'; 
			break;
		}
		else *p++ = *s;
		s++;
	}
	*p = 0;
	return prbuf;
}


/* The following data structures are allocated in parse_init() and freed by
 * buildtrie().  They are used to hold escape sequences (sorted by
 * escape[].string) until they can be built into a trie.
 */

char *string_space,  /* array of size MAXESCAPECHARS */
     *string_free;   /* next free slot in string_space */

struct escape {
	char *string;
	int action;
} 

*escapes;    /* array of size MAXESCAPES */

int n_strings;   /* number of elements of escapes[] in use */

#ifdef STANDARD_C

void    parse_init(void)

#else /* STANDARD_C */

void    parse_init()

#endif /* STANDARD_C */

{
	string_space = (char *) alloc_buffer(MAXESCAPECHARS, '\0');
	string_free = string_space;
	escapes = (struct escape *) alloc_buffer(MAXESCAPES * sizeof *escapes , '\0');
	n_strings = 0;
}

/* Parse the profile string in 's' and stuff the results into the
 * data structures string_space, escapes, and n_strings.
 * 's' is assumed to be in the form returned by find_def()--i.e., all
 * newlines, comments, and excess white-space have been removed.
 * Return 1 on success, 0 on failure (syntax error).
 */

#define PError(msg) { parse_error(msg,actionname); return 0; }

#ifdef STANDARD_C

void    parse_error(char *msg, char *action)

#else /* STANDARD_C */

void    parse_error(msg, action)

char    *msg;
char    *action;

#endif /* STANDARD_C */

{
	if (profline)
		(void)fprintf(stderr,"file %s, profile starting line %d:\r\n",
		profile, profline);
	else
		(void)fprintf(stderr,"environment variable MAP3270:\r\n");
	if (*action)
		(void)fprintf(stderr,"action \"%s\": ",action);
	(void)fprintf(stderr,"%s\r\n",msg);
	(void) fflush(stderr);
	(void)sleep(5);
}

#ifdef STANDARD_C

int     parse_profile(char *p)

#else /* STANDARD_C */

int     parse_profile(p)

char    *p;

#endif /* STANDARD_C */

{
	char *q;
	int action;
	char actionname[30];
	struct escape *e, *f;
	char *getstring(), *input;
	struct action_entry *ae;

	char *esc;      /* (BS) */

	*actionname = 0;
	while (*p && *p != '{') p++;
	if (!*p++) PError("missing {");
	while (*p) {
			/* get the next action name */
			q = actionname;
			while (*p && q-actionname < sizeof actionname && *p != '=')
				*q++ = *p++;
			if (*p++ != '=') PError("missing =");
			*q = '\0';

			/* look up the action name */
			action = 0;
			for (ae=actions; ae->name; ae++) {
				if (strcmp(ae->name,actionname)==0) {
					action = ae - actions + 1;
					break;
				}
			}

#ifdef DEBUG
			if (action == 0) PError("unknown action");
#endif /* DEBUG */

			for(;;) { /* for each definition of this action */
				/* get the next string definition */
				if (*p != '\'') PError("missing open quote");
				input = string_free;
				q = p;                                  /* BS */
				p = getstring(p, &string_free);
				if ((string_free - string_space) >= MAXESCAPECHARS)
				{
					end_program(601, "Emergency: MAXESCAPECHARS (%d) too small to handle all the mapping strings",
								MAXESCAPECHARS);
				}

				if (p==0) PError("missing close quote");

				/* record first instance of telnet escape char to show user (BS) */
				if (eschar[0] == 0 && strcmp(actionname, "escape") == 0) {
					esc = eschar;
					while (*q != '\'') q++;
					q++;
					while (*q != '\'') *esc++ = *q++;
					*esc = 0;
				}

         		/* insert <string,action> pair into escapes[] */
				for (e = escapes; e < &escapes[n_strings] && strcmp(e->string,input)<0; e++);
			    /* a binary search would be more efficient, but I'm too lazy */

				if (e < &escapes[n_strings] && strcmp(e->string,input)==0) {
					/* already there */
					(void)fprintf(stderr,"Warning: duplicate escape string '%s' ",
					pr(input,-1));
					(void)fprintf(stderr,"for actions %s and %s\r\n",
					actionname, actions[e->action - 1].name);
					(void) fflush(stderr);
					(void)sleep(15);
				}

				else {
					/* move down to get more space */
					for (f = &escapes[n_strings-1]; f>=e; --f) {
						(f+1)->string = f->string;
						(f+1)->action = f->action;
					}
					e->string = input;
					e->action = action;
					if (n_strings >= MAXESCAPES)
					{
						end_program(602, "Emergency: MAXESCAPES (%d) too small to handle all the mapping strings",
									MAXESCAPES);
					}
					n_strings++;
				}

				/* any more definitions for this action? */
				if (*p == ';') {
					p++;
					break;
				}
				if (*p != '|') PError("missing | or ;");
				p++;
			} /* for each definition of this action */

			/* any more? */
			if (*p=='}') {
				if (eschar[0] == 0) {                    /* BS */
					*actionname = 0;
					parse_error("No telnet escape action specified", actionname);
					end_program(603, "Emergency: No action specified for escape sequence", 0);
				}
			else return 1;
			}
		}
    	PError("missing }");
}

/* Parse a c-form string from p and put the result into the string_space
 * pointed to by qq.  Update *qq to point to the position following the result.
 * Return the first position past the end of the string p.
 */

#ifdef STANDARD_C

char    *getstring(char *p, char **qq)

#else /* STANDARD_C */

char    *getstring(p,qq)

char    *p;
char    **qq;

#endif /* STANDARD_C */

{
	char    *q = *qq;
	char    e_char;
	int     i;

	p++;
	for (;;) switch(*p) {

	default: 
		*q++ = *p++; 
		continue;

	case '\'': /* end of string */
		*q++ = 0;
		*qq = q;
		return p+1;

	case '\0': 
		return 0;

	case '\\':
		switch(*++p) {
		default: 
			for (i = 0 ; (e_char = esc_ch[i]) ; i++)
			{
				if (*p == e_char)
				{
					*q++ = esc_tr[i];
					break;
				}
			}
			if (!e_char)
				*q++ = *p;
			p++;
			break;

		case '\0': 
			return 0;

		case '0': 
		case '1': 
		case '2': 
		case '3': 
		case '4': 
		case '5':
		case '6': 
		case '7':
			*q = *p++ - '0';
			if (*p >= '0' && *p <='7')
			{
				*q = (*q << 3) | *p++ - '0';
				if (*p >= '0' && *p <='7')
				{
					*q = (*q << 3) | *p++ - '0';
				}
			}
			if (!(*q))
				*q = 0x80;
			q++;
		}
		break;

	case '^':
		if (*++p == 0) return 0;
		if (*p != '?') {
			*q++ = *p++ & 037;
			break;
		}
		*q++ = '\177';
		p++;
		break;
	}
}

/* The arrays escapes[] and string_space[] have the raw data.
 * escapes[] is sorted by the string field.
 * Turn it into a trie, and release the space.
 */

#ifdef STANDARD_C

void    buildtrie(void)

#else /* STANDARD_C */

void    buildtrie()

#endif /* STANDARD_C */

{
	int i,j;
	char c;

	escapes[n_strings].string = "";
	i = 0;
	next_escape = 1; /* can't use slot 0 in the escape_char/escape_ptr array */
	while (i<n_strings) {
		c = escapes[i].string[0];
		for (j = i+1; j<n_strings && escapes[j].string[0] == c; j++) ;
			escape_start[c] = update_trie(i,j,1);
		i = j;
	}
	free((void *)escapes);
	free((void *)string_space);
}

/*
 * Update the trie to include the node coresponding to a prefix w of length p
 * and all of its descendents.  All of the relevent strings will be
 * found in the escapes[] array in positions i..j-1, lexicographically sorted.
 * Return a pointer to the new node, and update next_escape.
 * Note that if w is actually equal to a string, rather than being a proper
 * prefix of one (in which case there must be exactly one such string, since
 * no string should be a proper prefix of another), there is no node associated
 * with w.  In this case, return the negative of the action corresponding to w.
 */

#ifdef STANDARD_C

int     update_trie(int i, int j, int p)

#else /* STANDARD_C */

int     update_trie(i,j,p)

int     i;
int     j;
int     p;

#endif /* STANDARD_C */

{
	int k, c, d, n, result, size;

	c = escapes[i].string[p];
	if (c==0) {
		/* escapes[i].string == w */
		if (j > i+1) {
			(void)fprintf(stderr,"Warning: escape string \"%s\" ",
			pr(escapes[i].string,-1));
			(void)fprintf(stderr,"is a prefix of \"%s\"\r\n",
			pr(escapes[i+1].string,-1));
			for (k = i+2; k<j; k++) {
				(void)fprintf(stderr,"\tand\"%s\"\r\n",
				pr(escapes[k].string,-1));
			}
			(void) fflush(stderr);
			(void)sleep(15);
		}
		return -escapes[i].action;
	}

	/* save a pointer to the current node */
	result = next_escape;

	/* calculate the size of the current node */
	size = 0;
	for (k=i; k<j; k++) {
		if (c != (d = escapes[k].string[p])) {
			size++;
			c = d;
		}
	}
	size = size + 2;
	/* allocate space for it */
	n = next_escape;
	next_escape = next_escape + size;

	c = escapes[i].string[p];

	/* for each child of the current node: */
	while (i<j) {
		/* find the next segment of strings with equal values in position p */
		for (k = i+1;
   k<j && c == (d = escapes[k].string[p]);
   k++) ;

		/* make it point to the appropriate subtree */
		escape_char[n] = c;
		escape_ptr[n++] = update_trie(i,k,p+1);

		/* move on to the next child */
		i = k;
		c = d;
	}
	escape_char[n] = FLAG;
	return result;
}

