/*
 *     Copyright CERN, Geneva 1989 - Copyright and any other
 *     appropriate legal protection of these computer programs
 *     and associated documentation reserved in all countries
 *     of the world.
 */

/*
 * Small program to look for the definition of keyboard escape sequences for
 * a given terminal type and generate a 'setenv' command to put the
 * definition into the environment.
 * See also finddef.c.
 *
 *
 */
#include <stdio.h>

#ifndef PROFILE
#define PROFILE "/etc/map3270"
#endif /* PROFILE */

#undef  stderr                          /* (BS) */
#define stderr stdout                   /* (BS) */

/* Name and line of the profile currently being parsed (for error messages) */

extern  char    *profile;
extern  int     profline;

extern  void *malloc();
extern  char *getenv(), *find_def();

extern  char *default_def;
int debug;

#define MAXDEF   2048 /* max length of a definition (in chars) */

/* ARGSUSED */

#ifdef STANDARD_C

int main(int argc, char **argv)

#else /* STANDARD_C */

main(argc,argv)
char **argv;

#endif /* STANDARD_C */

{
	char *keymap_type;
	char *keymap_def;
	char *keymap_file;

	/* undocumented -d flag */
	if (argc>1 && strcmp(argv[1],"-d")==0)
	{
		debug++;
		argc--;
		argv++;
	}

	/*
	 * Set up the required map name.
	 * This may be a parameter in the call.
	 * If no such parameter then it may be the environment
	 * variable KEYMAP (which would by convention be
	 *   <terminal type>-<mapping type>
	 * If KEYMAP is not defined then just take the terminal
	 * type from environment variable TERM: if this also is
	 * undefined then use "unknown".
	 * Store the map name in keymap_type
	 */

	if (argc>1)
		keymap_type = argv[1];
	else
	{
		/* find out whether KEYMAP has been defined */

		if (!(keymap_type = getenv("KEYMAP")))
			keymap_type = getenv("TERM");
	}
	if (!keymap_type)
		keymap_type = "unknown";
	if (debug)
		(void)fprintf(stderr,"keymap_type=\"%s\"\n",keymap_type);

	/*
	 * parse the profile.
	 * First, we look for a MAP3270 entry.
	 * If it exists and doesn't start with '/', use it as the profile;
	 * we are done.
	 * (If MAP3270 exists and starts with '/', take it as the name of
	 * the profile file, otherwise use the default file PROFILE).
	 * Leave the file name in keymap_file.
	 */

	if (keymap_file = getenv("MAP3270"))
	{
		if (debug)
			(void)fprintf(stderr,"MAP3270 exists\n");
		if (*keymap_file != '/')
		{
			keymap_def = keymap_file;
			if (debug)
				(void)fprintf(stderr,"keymap_def set from MAP3270\n");
			goto done;
		}
	}
	else
		keymap_file = PROFILE;
	if (debug) (void)fprintf(stderr,"term file %s\n",keymap_file);

	/*
	 * Now look for the keymap_type entry in keymap_file.
	 * If not found then look for "unknown" in keymap_file.
	 * If this also fails then use default_def
	 */

	while (!(keymap_def = (char *) malloc(MAXDEF)))
		(void)sleep(5);

	if (!find_def(keymap_def, keymap_file, keymap_type) ||
	    *keymap_def==0)
	{
		if (debug) (void)fprintf(stderr,"%s not found\n",keymap_type);
		if (!find_def(keymap_def, keymap_file, "unknown") ||
		    *keymap_def==0)
		{
			if (debug) (void)fprintf(stderr,"\"unknown\" not found; use default\n");
			strcpy(keymap_def, default_def);
		}
	}
done:
	printf("setenv MAP3270 \"");
	while (*keymap_def) putq(*keymap_def++);
	printf("\";\n");
	printf("unset noglob;\n");
	if (keymap_def != keymap_file)
		free((void *) keymap_def);
	exit(0);
}

#ifdef STANDARD_C

int putq(int c)

#else /* STANDARD_C */

putq(c)

#endif /* STANDARD_C */

{
	switch(c)
	{
		case '"':
		case '$':
		case '`':
		case '*':
			printf("\"'%c'\"",c);
			break;
		case '!':
			printf("\\%c",c);
			break;
		default:
			putchar(c);
	}
}
