/*
 *     Copyright CERN, Geneva 1989 - Copyright and any other
 *     appropriate legal protection of these computer programs
 *     and associated documentation reserved in all countries
 *     of the world.
 */

/*
 * iomodule.c
 *
 * All of the I/O should be centralised in here
 * (except that system dependent stuff may go elsewhere!).
 */

#include "globals.h"
#include "3270.h"
#include "telnet.h"
#ifdef _AUX_SOURCE
#include <termios.h>
#endif
#include <signal.h>

#ifdef TTY

#include <setjmp.h>

#endif /* TTY */

#ifndef USE_STDIO

#include <fcntl.h>

#endif /* USE_STDIO */

#ifdef sys_3b2

#include <sys/inet.h>

#endif /* sys_3b2 */

#ifdef sys_rs6000

#include <sys/select.h>

#endif /* sys_rs6000 */


#ifdef DEBUG
char *telcmds[] = {
	"EOR", "SE", "NOP", "DMARK", "BRK", "IP", "AO", "AYT", "EC",
	"EL", "GA", "SB", "WILL", "WONT", "DO", "DONT", "IAC",
};
#define telcmd(x) (((x) >= TELCMD_EOR) ? (telcmds[(x)-TELCMD_EOR]) : "Unknown command")

char *telopts[] = {
	"BINARY", "ECHO", "RCP", "SUPPRESS GO AHEAD", "NAME",
	"STATUS", "TIMING MARK", "RCTE", "NAOL", "NAOP",
	"NAOCRD", "NAOHTS", "NAOHTD", "NAOFFD", "NAOVTS",
	"NAOVTD", "NAOLFD", "EXTEND ASCII", "LOGOUT", "BYTE MACRO",
	"DATA ENTRY TERMINAL", "SUPDUP", "??", "??", "TERMINAL TYPE",
	"END-OF-RECORD"
};
#define telopt(x) ((((x) <= TELOPT_EOR) || ((x) == TELOPT_EXOPL)) ? (telopts[(x)]) : "Unknown option")

#endif /* DEBUG */

/*
 * The array of the signals which we will notice.
 * For the moment we just have a standard routine which may note
 * them in the LOGFILE and then ignore them.
 * Finish the list with -1.
 */

int     signal_list[] = {  SIGHUP,  SIGINT, SIGQUIT,  SIGILL,
			  SIGTRAP,  SIGIOT,  SIGEMT,  SIGFPE,
		       /* SIGBUS,  SIGSEGV,  SIGSYS, SIGPIPE, */
		       /*          SIGSEGV,  SIGSYS, SIGPIPE, */
					     SIGSYS, SIGPIPE,

#ifdef SIGXCPU
			  SIGXCPU,
#endif /* SIGXCPU */

			  -1 };

/*
 * Conversion of a two character escape sequence of the form
 * ESC <7-bit character>
 * to an 8-bit character.
 * or of an 8-bit character (> 127)
 * back to the 7-bit character which should follow the ESC.
 * This array is indexed by the (7-bit or 8-bit) character.
 * It gives either zero or the corresponding (8-bit or 7-bit) character.
 */

char    eight_cc[256] =
		{ 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 ,
		  0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 ,
		  0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 ,
		  0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 ,
		  0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 ,
		  0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x9c ,
		  0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 ,
		  0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 ,
		  0x00 , 0x00 , 0x00 , 0x00 , 0x84 , 0x85 , 0x00 , 0x00 ,
		  0x88 , 0x00 , 0x00 , 0x00 , 0x00 , 0x8d , 0x8e , 0x8f ,
		  0x90 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 ,
		  0x00 , 0x00 , 0x00 , 0x9b , 0x00 , 0x00 , 0x00 , 0x00 ,
		  0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 ,
		  0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 ,
		  0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 ,
		  0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 ,
		  0x00 , 0x00 , 0x00 , 0x00 , 0x44 , 0x45 , 0x00 , 0x00 ,
		  0x48 , 0x00 , 0x00 , 0x00 , 0x00 , 0x4d , 0x4e , 0x4f ,
		  0x50 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 ,
		  0x00 , 0x00 , 0x00 , 0x5b , 0x2f , 0x00 , 0x00 , 0x00 ,
		  0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 ,
		  0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 ,
		  0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 ,
		  0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 ,
		  0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 ,
		  0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 ,
		  0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 ,
		  0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 ,
		  0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 ,
		  0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 ,
		  0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 ,
		  0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00
		};

/*
 * Output to the user is rather special.
 * Once the save_tty routine has been called then output should be done
 * by any of the special routines f_addchars, f_putc, f_putcs, f_puttcs,
 * f_putf or f_puts. This lasts until the final call of reset_tty(0,*), which
 * is done in the end_program routine.
 * This save_tty is called at the start of the main program.
 *
 * Once save_tty has been called then these output routines write to
 * the /dev/tty file, i.e. they are not redirected if stdout is redirected.
 * They may write in an asynchronous or a synchronous fashion according
 * as to whether the user is or is not in 3270 mode (i.e. does or does
 * not see a 3270 screen). The initial setting after the init_tty call
 * is synchronous, i.e. a call will not return until the write has been
 * accepted. This mode is also used when the user has switched to the
 * on-line help screens. In this mode there is a check whether the user
 * is actually receiving output: he might be on a terminal server and have
 * switched to a different session. If the user is not receiving output
 * then after a relatively short time the program will abort.
 *
 * The switch to asynchronous mode is made with the call of set_tty.
 * In this mode the program attempts to write to the user's screen, but
 * if the write fails then the output will be put in a linked structure
 * of output buffers and the program will keep retrying.
 * The switch back to synchronous mode is made with a call of
 * reset_tty(>0,*): this is done when the user selects the on-line help.
 * reset_tty(<0,*) also switches back to synchronous mode, but keeping
 * the same tty setup as for 3270 mode (input with CBREAK).
 *
 * The reason for asynchronous mode is twofold. Firstly, the user may be
 * on a terminal server and have switched sessions: in this case we
 * effectively keep talking to the IBM (which might asynchronously send
 * him something) and wait until he returns. Secondly, the user may have
 * a rather slow line and may wish to send some information to the IBM
 * which then clears the user screen: in this case the program can throw
 * away all outstanding queued output.
 *
 * Flushing output to the user's screen is sometimes necessary, particularly
 * since the f_putc character output routine does not do this. Most of
 * the output routines have a flush option included, but there is also
 * a separate f_flush routine in case of need.
 */

/*
 * The basic output buffer structure follows.
 * It includes the linkage to the next buffer, the actual character
 * count and the number of characters already output.
 * It also includes space for a final string terminator.
 */

#define OPBUFSIZE       490

struct  op_buff {
			struct  op_buff *next_opbf;
			int             ch_size;
			int             ch_count;
			int             ch_sent;
			char            buffer[OPBUFSIZE+1];
		};

struct  op_buff *first_opbf = NULL;
struct  op_buff *last_opbf = NULL;

int     ch_left = 0;

#ifndef USE_STDIO

/* initial state of fd_output file descriptor status */

int     fds_init;

#endif /* USE_STDIO */

/* String buffer for printf-form output */

#define OPSTRSIZE       128

char    op_string[OPSTRSIZE];

/* select call masks and timer field */

fd_set  readmask, writemask, exceptmask;
struct  timeval waitst;

/* main select call timer (seconds and microseconds) */

	int     timer_sec;
	int     timer_usec;

/*
 * Wait values for the various select calls
 *
 * The most important are the select calls in reader
 * of which the main one is used for interval timing
 */

#ifdef TTY

#ifdef DEBUG

#define reader_sec      5       /* every ? seconds */
#define reader1_sec          0  /* short wait after stuff from IBM (efficiency) */
#define reader1_usec     50000

#define idle_warn      60       /* Intervals before user is warned */
#define idle_wnum       3       /* How many warnings to give the user */

#else /* DEBUG */

#define reader_sec     30       /* every ? seconds */
#define reader1_sec          0  /* short wait after stuff from IBM (efficiency) */
#define reader1_usec     50000

#define idle_warn     120       /* Intervals before user is warned */
#define idle_wnum       3       /* How many warnings to give the user */

#endif /* DEBUG */

#else /* TTY */

#ifdef DEBUG

#define reader_sec      2       /* every ? seconds */
#define reader1_sec          0  /* short wait after stuff from IBM (efficiency) */
#define reader1_usec     50000

#else /* DEBUG */

#define reader_sec      1       /* every ? seconds */
#define reader1_sec          0  /* short wait after stuff from IBM (efficiency) */
#define reader1_usec     50000

#endif /* DEBUG */

#endif /* TTY */

/*
 * After receiving data from the IBM which unlocks the keyboard it is
 * an unfortunate fact that some IBM software gets screwed if it does
 * not get a chance to put out more stuff.
 * This is sometimes the case with applications which first send a type
 * of acknowledgement (including screen unlock!) then either request to
 * read back the screen or continue with a new screen. In either case there
 * can be a delay because the IBM refuses to send any continuation data
 * until it receives the ACK, whereas this system may delay an ack for
 * efficiency reasons.
 * For this reason we may need an extra wait after such IBM data.
 * However, if we are also wanting to retry hung asynchronous output
 * we may prefer to have a minimum number (op_wcmin) of short waits.
 */

#define reader2_sec          0  /* longer wait after stuff from IBM has unlocked */
/* #define reader2_usec    250000 */
#define reader2_usec    1000

int     op_wcount = 0;
int     op_wcntst = 0;

#ifdef TTY

#define reader3_sec          0  /* short wait before retrying hung output */
#define reader3_usec     75000  /* just above the normal 70ms LAT timer */

#define op_wcmin             1  /* # of short waits before allowing input again */
#define op_wcmax           100  /* # of short waits before moving to long wait */

#else /* TTY */

#define reader3_sec          0  /* short wait before retrying hung output */
#define reader3_usec     50000  /* assume on a workstation */

#define op_wcmin             1  /* # of short waits before allowing input again */
#define op_wcmax            20  /* # of short waits before moving to long wait */

#endif /* TTY */

#define writnet_sec        300  /* wait to send stuff to IBM */
#define writnet_usec         0

#define clkbin_sec           0  /* timeout in clearing out the keyboard input */
#define clkbin_usec     200000

#define disccr_sec           0  /* timeout when discarding spare CRs in graphic input */
#define disccr_usec       1000

#ifdef TTY

/*
 * Values for using with the use of the timer of used CPU time.
 */

#ifdef DEBUG

#define cpu_ratio       2       /* minimum acceptable ratio */
#define cpu_interval   60       /* every 60 seconds of CPU */

#else /* DEBUG */

#define cpu_ratio       5       /* minimum acceptable ratio */
#define cpu_interval   60       /* every 60 seconds of CPU */

#endif /* DEBUG */

#define close_sec       2       /* CPU time we may spend in the socket shutdown */

static  jmp_buf environment;    /* Save an environment for a longjmp */

#endif /* TTY */

/*
 * The various input buffers are now defined.
 * User input is first read into the kbbuf array, of size kbbuf_max.
 * This is scanned immediately to see what can be dealt with and what
 * may have to wait until the IBM is receiving input.
 * In scanning it we may have to temporarily store what looks like
 * partial escape sequences in the pushback stack.
 * When the input has been scanned then that part which cannot be dealt
 * with immediately is put into the circular buffer kbread, of size kbread_max.
 * This array is of type short, rather than char, so that where we have
 * recognised an escape sequence we can just store the index of the
 * sequence (flagged by being negated).
 * Clearly, we have to avoid allowing new user input when the circular
 * buffer gets full!
 */

/* First the initial kbbuf specifications */

#define kbbuf_max       128

static  char    kbbuf[kbbuf_max];               /* The buffer itself    */
static  char    *kbbuf_p        = kbbuf;        /* pointer to next char */
static  int     kbbuf_count     = 0;            /* count of remainder   */

/* Next the pushback stack */

#define pb_max          50

static  short   pushback_stack[pb_max];         /* The buffer itself    */
static  int     pb_top          = 0;            /* last item pushed     */

/* Finally, the scanned input circular buffer kbread */

#define kbread_max      256

static  short   kbread[kbread_max];             /* The buffer itself    */
static  short   *kbread_p       = kbread;       /* pointer to read pos  */
static  short   *kbrput_p       = kbread;       /* pointer to write pos */
static  int     kbread_count    = 0;            /* count of remainder   */


/*
 * We also need buffers for stuff read from the IBM.
 * This is read into the netbuf array, of size NETBSIZE.
 * netbuf_count is the remaining unprocessed input count.
 * netbuf_p is the pointer to the next character.
 */

static  char    netbuf[NETBSIZE];
static  char    *netbuf_p       = netbuf;
static  int     netbuf_count    = 0;


short	ospeed = -1;

/*
 * Now for the terminal I/O stuff.
 * This depends on which interface we have: old or new tty interface of BSD,
 * termio interface of System V Interface Definition (SVID) or the termios
 * interface of POSIX.
 * Default is the BSD tty interface.
 */

struct  TTY_STRUCT      otty, ntty;

#ifndef USE_STDIO

struct  TTY_STRUCT      otty1;

#endif /* USE_STDIO */

#ifdef USE_SYSV_TERMIO

/*
 * Define the special character "ignore" value.
 * In the POSIX world this is the symbol POSIX_VDISABLE.
 * However, we might not know this value, either because we don't
 * have POSIX or because the stupid system did not have unistd.h.
 * In this case it could need a value dependent on the workstation type.
 */

#ifndef POSIX_VDISABLE

#define POSIX_VDISABLE  0

#endif /* POSIX_VDISABLE */

#else /* USE_SYSV_TERMIO */

/* for the terminal special chars..        (BS) */

#ifdef TTY
struct  tchars  otc;
struct  tchars  notc = { -1, -1, -1, -1, -1, -1 };
#endif /* TTY */

struct  ltchars oltc;
struct  ltchars noltc = { -1, -1, -1, -1, -1, -1 };

#endif /* USE_SYSV_TERMIO */

/*
 * General system call failure routine.
 * The given code is passed on to end_program, so should obey the same rules:
 *   zero: no real error: show the user the message.
 *   nonzero: error indicator and message are the parameters.
 *            Error indicator interpretation:-
 *            if negative the user may not be on his 3270 screen
 *                      (but the term_flag mode really tells us!)
 *            if <= -900 then do not put the message to the user.
 *            if <= -1000 (catastrophic!) be extra careful.
 */


#ifdef STANDARD_C

void    sys_fail(int code_num, int ret_val)

#else /* STANDARD_C */

void    sys_fail(code_num, ret_val)

int     code_num;
int     ret_val;

#endif /* STANDARD_C */

{
	extern int errno;
	extern int sys_nerr;
	extern char *sys_errlist[];

	static int error_count = 0;
	       int errno_save = errno;

	if (errno_save)
	{

#ifdef LOGFILE
		logit(((errno_save < sys_nerr) ? sys_errlist[errno_save] : "Unknown error %d"), errno_save);
#endif /* LOGFILE */

#ifdef DEBUG
	(void)fprintf(outf,"sys_fail    : (%d) : %s errno_save = %d, return value = %d : \n" ,
			code_num, getstime(),  errno_save, ret_val);
	(void)fprintf(outf,"sys_fail    : %s\n", ((errno_save < sys_nerr) ? sys_errlist[errno_save] :  "Unknown error"));
#endif /* DEBUG */

		if (code_num >= -900)
		{
			(void)f_puts("system error: ", FALSE);
			(void)f_puts(sys_errlist[errno_save], FALSE);
			(void)f_puts("/r/n", TRUE);

			(void)sleep(2);
		}
	}

	if ((ret_val == EINTR) && (++error_count < 10))
		return;

	if (ending_program)
		exit(code_num);
	end_program(code_num, "Unexpected system call failure: errno = %d",
				errno_save);
}

/*
 * Routines concerned with terminal output buffers
 */

/* Get a new output buffer and add to the end of the list */

#ifdef STANDARD_C

void    get_opbf(void)

#else /* STANDARD_C */

void    get_opbf()

#endif /* STANDARD_C */

{
	struct  op_buff *opbf_addr;

	opbf_addr = (struct op_buff *) alloc_buffer(sizeof(struct op_buff), '\0');
	opbf_addr->ch_size = OPBUFSIZE;
	opbf_addr->ch_count = 0;
	opbf_addr->ch_sent = 0;
	opbf_addr->next_opbf = NULL;
	if (first_opbf)
	{
		last_opbf->next_opbf = opbf_addr;

#ifdef DEBUG
	(void)fprintf(outf, "get_opbf    : add new output buffer structure 0x%lx to the chain\n",
				opbf_addr);
#endif /* DEBUG */

	}
	else
	{
		first_opbf = opbf_addr;

#ifdef DEBUG
	(void)fprintf(outf, "get_opbf    : begin the new output buffer chain\n");
#endif /* DEBUG */

	}
	last_opbf = opbf_addr;
	return;
}

/*
 * Release the first buffer in the output buffer list.
 * However, if the parameter is TRUE and this buffer is the only one
 * in the list then keep it.
 * Note that we should not release the only one unless we are prepared to
 * get a new one (with get_opbf) before calling the output routines.
 */

#ifdef STANDARD_C

void    rel_opbf(bool keep_one)

#else /* STANDARD_C */

void    rel_opbf(keep_one)

bool    keep_one;

#endif /* STANDARD_C */

{
	struct  op_buff *opbf_addr;

	if (first_opbf->next_opbf)
	{
		opbf_addr = first_opbf;
		first_opbf = first_opbf->next_opbf;
		free((void *)opbf_addr);

#ifdef DEBUG
	(void)fprintf(outf, "rel_opbf    : remove old output buffer structure 0x%lx from the chain\n",
				opbf_addr);
#endif /* DEBUG */

	}
	else
	{
		if (keep_one)
		{
			first_opbf->ch_count = 0;
			first_opbf->ch_sent = 0;
		}
		else
		{
			free((void *)first_opbf);
			first_opbf = NULL;
		}
	}
	return;
}



/* unset the catching of any signals that we have been catching */

#ifdef STANDARD_C

void    unset_signals(void)

#else /* STANDARD_C */

void    unset_signals()

#endif /* STANDARD_C */

{
	int signum, sigval;

	for (signum = 0 ; (sigval = signal_list[signum++]) >= 0 ; )
		(void)signal(sigval, SIG_IGN);
}

#ifdef STANDARD_C

SIGNAL_T catch_signal(int sig_num)

#else /* STANDARD_C */

SIGNAL_T catch_signal(sig_num)

int     sig_num;

#endif /* STANDARD_C */

{
	/* reset default signal handling to avoid infinite loop possibility */

	unset_signals();

#ifdef LOGFILE
	logit("Caught termination signal %d",
		sig_num);
#endif /* LOGFILE */

	end_program(-999, "termination signal %d received", sig_num);
	SIGNAL_RETURN;
}

/* set up for any signals we want to catch */

#ifdef STANDARD_C

void    set_signals(void)

#else /* STANDARD_C */

void    set_signals()

#endif /* STANDARD_C */

{
	int signum, sigval;

	for (signum = 0 ; (sigval = signal_list[signum++]) >= 0 ; )
	{
		(void)signal(sigval, catch_signal);

#ifdef DEBUG
	(void)fprintf(outf, "set_signals : Catch signal %d\n", sigval);
#endif /* DEBUG */

	}
}

#ifdef TTY

/* set up timers so as to have various checks that things are OK */


/*
 * Timer to interrupt at intervals of CPU time (user plus system)
 * At each interrupt we will check the ratio of the CPU time to the
 * real time.
 * If this ratio is too high then we assume that we are in a loop
 * and we force an exit.
 */

time_t  cpu_prev, cpu_now;

#ifdef STANDARD_C

SIGNAL_T cpu_int(int sig_num)

#else /* STANDARD_C */

SIGNAL_T cpu_int(sig_num)

int     sig_num;

#endif /* STANDARD_C */

{
	static  entry_count = 0;

	cpu_prev = cpu_now;
	if ((int)(time( (time_t *) &cpu_now) - cpu_prev) < (cpu_interval * cpu_ratio))
	{
		/* Looks like we have a tight loop, folks */
		/* Afraid that we have to kill this task  */
		/* without even any messages!             */

#ifdef LOGFILE
		logit("Emergency: CP looping: %d CPU seconds",
			cpu_interval);
		logscreen();
#endif /* LOGFILE */

		/* maybe try to kill with a dump */
		if (entry_count++ > 10)
			kill(getpid(), SIGBUS);
	}
	SIGNAL_RETURN;
}

#ifdef STANDARD_C

void    set_cpu_timer(bool setflag)

#else /* STANDARD_C */

void    set_cpu_timer(setflag)

bool    setflag;

#endif /* STANDARD_C */

{
	struct  itimerval       cpu_val;
	int                     int_secs;
	int                     result;

	int_secs = (setflag ? cpu_interval : 0);

	cpu_val.it_value.tv_sec = int_secs;
	cpu_val.it_value.tv_usec = 0;
	cpu_val.it_interval.tv_sec = int_secs;
	cpu_val.it_interval.tv_usec = 0;
	while ((result = setitimer(ITIMER_VIRTUAL, &cpu_val, (struct itimerval *) NULL)) < 0)
		sys_fail(381, result);
	(void) time( (time_t *) &cpu_now);
}

#ifdef STANDARD_C

void    start_cpu_timer(void)

#else /* STANDARD_C */

void    start_cpu_timer()

#endif /* STANDARD_C */

{
	(void)signal(SIGVTALRM, cpu_int);
	set_cpu_timer(TRUE);
}


/*
 * Timer to interrupt at intervals of real time.
 * These are used to check on hung synchronous output.
 *
 */

#ifdef STANDARD_C

SIGNAL_T real_int(int sig_num)

#else /* STANDARD_C */

SIGNAL_T real_int(sig_num)

int     sig_num;

#endif /* STANDARD_C */

{

#ifdef LOGFILE
	if (!ending_program)
		logit("Error: SIGALRM when synch_op = %d", synch_op);
		logscreen();
#endif /* LOGFILE */

	if (synch_op)
		(void) longjmp(environment, 1);

	/* Must be user in 3270 mode: program error if we get here! */

#ifdef LOGFILE
		logit("Emergency: SIGALRM while in asynchronous mode", 0);
		logscreen();
#endif /* LOGFILE */

	/* try to kill with a dump */
	kill(getpid(), SIGBUS);
	SIGNAL_RETURN;
}

#endif /* TTY */

/*
 * get time into a string (no terminating \n)
 */

#define S_TIMEMAX 30
static  char    s_time[S_TIMEMAX];

#ifdef STANDARD_C

char    *getstime(void)

#else /* STANDARD_C */

char    *getstime()

#endif /* STANDARD_C */

{
	int     stn;
	struct  timeval tv;
	struct  timezone tz;

	(void)gettimeofday(&tv, &tz);
	(void)strncpy (&s_time[0], ctime((time_t *)&tv.tv_sec) , S_TIMEMAX);
	s_time[S_TIMEMAX - 1] = '\0';

	for (stn = S_TIMEMAX - 2 ; stn ; stn--)
	{
		if (s_time[stn] == '\n')
		{
			/* get rid of the year */
			if (stn > 5)
				stn -= 5;
			/* add in the milliseconds */
			(void)sprintf(s_time + stn , ".%3d" , (tv.tv_usec / 1000));
			break;
		}
	}
	return (&s_time[0]);
}

/*
 * Compress a 7-bit controls string for 8-bit controls.
 * Return the length of the compressed string.
 */

#ifdef STANDARD_C

int     f_compress(char *string)

#else /* STANDARD_C */

int     f_compress(string)

char    *string;

#endif /* STANDARD_C */

{
	unsigned char   *in;
	unsigned char   *out;

	if (!(in = (unsigned char *)string))
		return (0);

	out = in;
	do
	{
		if ((*in == ESC) && (*in < 128) && eight_cc[(int)*(in+1)])
		{
			in++;
			*in = eight_cc[(int)*in];
		}
	} while (*out++ = *in++);

#ifdef DEBUG
	(void)fprintf(outf, "f_compress  : compressed string is \"");
	for (in = (unsigned char *)string; *in ; in++)
		(void)fprintf(outf, "%s", print_ascii((unsigned short)*in));
	(void)fprintf(outf, "\"\n");
#endif /* DEBUG */

	return (strlen(string));
}

/*
 * Uncompress an 8-bit controls string for 7-bit controls.
 * Return the length of the uncompressed string.
 */

#ifdef STANDARD_C

int     f_uncompress(char *string)

#else /* STANDARD_C */

int     f_uncompress(string)

char    *string;

#endif /* STANDARD_C */

{
	unsigned char   *in;
	unsigned char   *out;
	unsigned char   *last;

	if (!(in = (unsigned char *)string))
		return (0);

	last = in + strlen(string);
	do
	{
		if ((*(in) > 127) && eight_cc[(int)*(in)])
		{
			for (out = last+1 ; out != in ; out--)
				*(out) = *(out-1);
			*(in++) = ESC;
			*(in) = eight_cc[(int)*in];
		}
	} while (*(++in));

#ifdef DEBUG
	(void)fprintf(outf, "f_uncompress: uncompressed string is \"");
	for (in = (unsigned char *)string; *in ; in++)
		(void)fprintf(outf, "%s", print_ascii((unsigned short)*in));
	(void)fprintf(outf, "\"\n");
#endif /* DEBUG */

	return (strlen(string));
}

/*
 * General selector routine.
 * Will call select and check the result.
 * On a negative (error) return we ignore error EINTR
 * (up to MAX_EINTR times!)
 * then treat as an error and exit via end_program() routine.
 * An exception is also treated as an error unless the given
 * error number is zero, in which case we return.
 * return value is that of the select call.
 */

#define MAX_EINTR 100

#ifdef STANDARD_C

int     select_check(int nfds, int readmval, int writemval, int exceptmval, int sec, int usec, int errnum, char *errmess)

#else /* STANDARD_C */

int     select_check(nfds, readmval, writemval, exceptmval, sec, usec, errnum, errmess)

int     nfds;
int     readmval, writemval, exceptmval;
int     sec, usec;
int     errnum;
char    *errmess;

#endif /* STANDARD_C */

{

	int     selres;


	int     count = MAX_EINTR;

	while (count--)
	{
		readmask.fds_bits[0] = readmval;
		writemask.fds_bits[0] = writemval;

#ifdef sys_3b2

		/* Seems the 3b2 does not support exception handling */
		exceptmask.fds_bits[0] = 0;

#else /* sys_3b2 */

		exceptmask.fds_bits[0] = exceptmval;

#endif /* sys_3b2 */

		waitst.tv_sec = sec;
		waitst.tv_usec = usec;
		selres = select(nfds, (readmval ? &readmask : (fd_set *)0),
				      (writemval ? &writemask : (fd_set *)0),
				      (exceptmval ? &exceptmask : (fd_set *)0),
				      (sec < 0 ? (struct timeval *) NULL : &waitst));
		if (selres < 0)
		{
			if (errno == EINTR)
				continue;
			end_program(errnum, errmess, selres);
		}
		if (selres && exceptmask.fds_bits[0])
		{
			/* Oh, dear, an exception: */
			if (errnum)
				end_program(errnum, errmess, selres);
		}
		return (selres);
	}
	end_program(-1399, "Too many (%d) select EINTR errors", MAX_EINTR);
	return (selres);
}

#ifdef DEBUG

/*
 * Print out the information in the structure defining the terminal interface.
 * This might be tty, termio or termios.
 * Print normally onto outf unless it is not defined, in which case
 * use stdout.
 */

#ifdef STANDARD_C

void    show_tty(char *message, struct  TTY_STRUCT *tty_structure)

#else /* STANDARD_C */

void    show_tty(message, tty_structure)

char    *message;
struct  TTY_STRUCT      *tty_structure;

#endif /* STANDARD_C */

{
	FILE    *debug_file;

	debug_file = (outf ? outf : stdout);

	(void)fprintf(debug_file, "%s\n", message);

#ifdef USE_SYSV_TERMIO

	(void)fprintf(debug_file, "show_tty    : iflag  is 0%lo\n", (unsigned long)tty_structure->c_iflag);
	(void)fprintf(debug_file, "show_tty    : oflag  is 0%lo\n", (unsigned long)tty_structure->c_oflag);
	(void)fprintf(debug_file, "show_tty    : cflag  is 0%lo\n", (unsigned long)tty_structure->c_cflag);
	(void)fprintf(debug_file, "show_tty    : lflag  is 0%lo\n", (unsigned long)tty_structure->c_lflag);

#ifdef USE_TERMIOS

	(void)fprintf(debug_file, "show_tty    : ispeed is %d\n",   cfgetispeed(tty_structure));
	(void)fprintf(debug_file, "show_tty    : ospeed is %d\n",   cfgetospeed(tty_structure));

#endif /* USE_TERMIOS */

	(void)fprintf(debug_file, "show_tty    : erase  is 0x%x\n", tty_structure->c_cc[VERASE]);
	(void)fprintf(debug_file, "show_tty    : kill   is 0x%x\n", tty_structure->c_cc[VKILL]);
	(void)fprintf(debug_file, "show_tty    : VMIN   is %d\n",   tty_structure->c_cc[VMIN]);
	(void)fprintf(debug_file, "show_tty    : VTIME  is %d\n",   tty_structure->c_cc[VTIME]);

#else /* USE_SYSV_TERMIO */

	(void)fprintf(debug_file, "show_tty    : ispeed is %d\n",   tty_structure->sg_ispeed);
	(void)fprintf(debug_file, "show_tty    : ospeed is %d\n",   tty_structure->sg_ospeed);
	(void)fprintf(debug_file, "show_tty    : erase  is 0x%x\n", tty_structure->sg_erase);
	(void)fprintf(debug_file, "show_tty    : kill   is 0x%x\n", tty_structure->sg_kill);
	(void)fprintf(debug_file, "show_tty    : flags  is 0x%x\n", tty_structure->sg_flags);

#endif /* USE_SYSV_TERMIO */

}

#endif /* DEBUG */

/* save initial tty mode */

#ifdef STANDARD_C

void    save_tty(void)

#else /* STANDARD_C */

void    save_tty()

#endif /* STANDARD_C */

{
	int     getset_res;

#ifdef USE_STDIO

	fd_output = 1;
	fd_stream = stdout;

#else  /* USE_STDIO */

#ifdef USE_SYSV_TERMIO

#ifdef USE_TERMIOS

	while ((getset_res = tcgetattr(1, &otty1)) < 0)
		sys_fail(-1310, getset_res);

#else  /* USE_TERMIOS */

	while ((getset_res = ioctl(1, TCGETA, &otty1)) < 0)
		sys_fail(-1310, getset_res);

#endif /* USE_TERMIOS */

#else  /* USE_SYSV_TERMIO */

	while ((getset_res = ioctl(1, TIOCGETP, (char *)&otty1)) < 0)
		sys_fail(-1310, getset_res);

#endif /* USE_SYSV_TERMIO */

#ifdef DEBUG
	show_tty("save_tty    : Flags settings for stdout", &otty1);
#endif /* DEBUG */

	while ((fd_output = open("/dev/tty", O_RDWR + O_NDELAY)) < 0)
		sys_fail(-1311, fd_output);
	while ((fd_stream = fdopen(fd_output, "a")) == NULL)
		sys_fail(-1312, 0);

#endif /* USE_STDIO */

#ifdef USE_SYSV_TERMIO

#ifdef USE_TERMIOS

	while ((getset_res = tcgetattr(fd_output, &otty)) < 0)
		sys_fail(-1313, getset_res);

	ospeed = cfgetospeed(&otty);

#else  /* USE_TERMIOS */

	while ((getset_res = ioctl(fd_output, TCGETA, &otty)) < 0)
		sys_fail(-1313, getset_res);

	ospeed = otty.c_cflag & 0x1f;

#endif /* USE_TERMIOS */

#else  /* USE_SYSV_TERMIO */

	while ((getset_res = ioctl(fd_output, TIOCGETP, (char *)&otty)) < 0)
		sys_fail(-1313, getset_res);

#ifdef TTY
	while ((getset_res = ioctl(0, TIOCGETC, (char *)&otc)) < 0)
		sys_fail(-1314, getset_res);
#endif /* TTY */

	while ((getset_res = ioctl(0, TIOCGLTC, (char *)&oltc)) < 0)
		sys_fail(-1315, getset_res);
	ospeed = otty.sg_ospeed;

#endif /* USE_SYSV_TERMIO */

#ifdef DEBUG
	show_tty("save_tty    : Flags settings for fd_output", &otty);
#endif /* DEBUG */

	/* Now set it up like we want for dialogue */

	reset_tty(SYNCH_NORMAL, FALSE);
}

/* initialise the tty and find its raw descriptor number */

#ifdef STANDARD_C

void    init_tty(void)

#else /* STANDARD_C */

void    init_tty()

#endif /* STANDARD_C */

{

#ifndef USE_STDIO

	/* Save the current (synchronous?) output mode */

	while ((fds_init = fcntl(fd_output ,F_GETFL, 0)) < 0)
		sys_fail(-1320, fds_init);

#ifdef DEBUG
	(void)fprintf(outf, "init_tty    : initial fd_output (%d) file descriptor status = 0x%x\n",
				fd_output, fds_init);
#endif /* DEBUG */

#endif /* USE_STDIO */

	get_opbf();
	term_mode = -1;
}

/*
 * Set the tty mode to what we want for 3270 emulation.
 * Maybe also setup the changed tty parameters.
 * Note that the setting call will hang up until output is quiescent, so
 * make sure that we are not in a state where user output is stuck!
 * For safety we also reset to 7-bit controls if we were in 8-bit controls.
 */

#ifdef STANDARD_C

void    setup_tty(bool set_ntty)

#else /* STANDARD_C */

void    setup_tty(set_ntty)

bool    set_ntty;

#endif /* STANDARD_C */

{
	int     getset_res;

#ifndef USE_STDIO
	int     fcntl_res;
#endif /* USE_STDIO */

	if (eight_bc)
	{
		/*
		 * Now we know we were using eight-bit controls.
		 * For safety change the outgoing tcs control sequences
		 * and the style change sequences back to 7-bit ones.
		 */

		int     tcsind;
		int     from, to;

		for (tcsind = 0 ; tcsind < tcsMAX ; tcsind++)
		{
			if (tcslen[tcsind])
				tcslen[tcsind] = f_uncompress((char *)tcsptr[tcsind]);
		}

#ifndef EXTENDED
		for (from = 0 ; from < MAXSTYLES ; from++)
		{
			for (to = 0 ; to < MAXSTYLES ; to++)
			{
				if (change_sl[from][to])
					change_sl[from][to] = f_uncompress(change_cp[from][to]);
			}
		}
#endif /* EXTENDED */

		eight_bc = FALSE;
	}

	if (set_ntty)
	{
		f_clearop(1);
		ntty = otty;

#ifdef USE_SYSV_TERMIO

		ntty.c_iflag = (ntty.c_iflag & ~BRKINT & ~ISTRIP & ~INLCR & ~IGNCR & ~ICRNL & ~IXOFF) | IGNBRK | IXON;
		ntty.c_oflag = (ntty.c_oflag & ~OLCUC & ~ONLCR & ~OCRNL & ~ONOCR & ~ONLRET & ~TAB3);
		ntty.c_lflag = (ntty.c_lflag & ~ISIG & ~ICANON & ~XCASE & ~ECHO & ~ECHOE & ~ECHOK & ~ECHONL);

#ifdef USE_TERMIOS

		while ((getset_res = tcsetattr(fd_output, TCSADRAIN, &ntty)) < 0)
			sys_fail(-1330, getset_res);
		ospeed = cfgetospeed(&ntty);

#else /* USE_TERMIOS */

		while ((getset_res = ioctl(fd_output, TCSETAW, &ntty)) < 0)
			sys_fail(-1330, getset_res);
		ospeed = ntty.c_cflag & 0x1f;

#endif /* USE_TERMIOS */

#else /* USE_SYSV_TERMIO */

		ntty.sg_flags = (ntty.sg_flags & ~ECHO & ~CRMOD & ~RAW & ~XTABS) | CBREAK;

		while ((getset_res = ioctl(fd_output, TIOCSETP, (char *) &ntty)) < 0)
			sys_fail(-1330, getset_res);
		ospeed = ntty.sg_ospeed;

#endif /* USE_SYSV_TERMIO */

#ifdef DEBUG
	show_tty("setup_tty   : Flags settings for 3270 output", &ntty);
#endif /* DEBUG */

	}

#ifdef TRANSP_IO
	if (!trflag)
#endif /* TRANSP_IO */

	{

#ifdef DEBUG
	(void)fprintf(outf, "setup_tty   : Now send the TI and KS strings\n");
#endif /* DEBUG */

		f_puttcs(tcsTI, FALSE);
		f_puttcs(tcsKS, FALSE);

#ifdef DEBUG
	if (strlen(tcsptr[tcsKS]))
		(void)fprintf(outf, "setup_tty   : Keyboard initialised\n");
	else
		(void)fprintf(outf, "setup_tty   : No ks string found\n");
#endif /* DEBUG */

		f_flush();
	}

	/* kill action of some special characters on input */

#ifdef USE_SYSV_TERMIO

#ifdef TTY

	ntty.c_cc[VINTR] = POSIX_VDISABLE;
	ntty.c_cc[VQUIT] = POSIX_VDISABLE;
	ntty.c_cc[VERASE] = POSIX_VDISABLE;
	ntty.c_cc[VKILL] = POSIX_VDISABLE;
	ntty.c_cc[VEOF] = POSIX_VDISABLE;
	ntty.c_cc[VEOL] = POSIX_VDISABLE;
	ntty.c_cc[VSTART] = otty.c_cc[VSTART];
	ntty.c_cc[VSTOP] = otty.c_cc[VSTOP];

#endif /* TTY */

	ntty.c_cc[VMIN] = POSIX_VDISABLE;
	ntty.c_cc[VTIME] = POSIX_VDISABLE;

#ifdef USE_TERMIOS

	ntty.c_cc[VSUSP] = POSIX_VDISABLE;

/*
 * Following settings are in ifdefs because not every termios header
 * file agrees on the names. In particular, my Ultrix version has
 * VRPRNT instead of VREPRINT and VFLUSH instead of VDISCARD.
 */

#ifdef VDSUSP
	ntty.c_cc[VDSUSP] = POSIX_VDISABLE;
#endif /* VDSUSP */

#ifdef VREPRINT
	ntty.c_cc[VREPRINT] = POSIX_VDISABLE;
#endif /* VREPRINT */

#ifdef VRPRNT
	ntty.c_cc[VRPRNT] = POSIX_VDISABLE;
#endif /* VRPRNT */

#ifdef VDISCARD
	ntty.c_cc[VDISCARD] = POSIX_VDISABLE;
#endif /* VDISCARD */

#ifdef VFLUSH
	ntty.c_cc[VFLUSH] = POSIX_VDISABLE;
#endif /* VFLUSH */

#ifdef VWERASE
	ntty.c_cc[VWERASE] = POSIX_VDISABLE;
#endif /* VWERASE */

#ifdef VLNEXT
	ntty.c_cc[VLNEXT] = POSIX_VDISABLE;
#endif /* VLNEXT */

#ifdef VQUOTE
	ntty.c_cc[VQUOTE] = POSIX_VDISABLE;
#endif /* VQUOTE */

	while ((getset_res = tcsetattr(0, TCSADRAIN, &ntty)) < 0)
		sys_fail(-1331, getset_res);

#else /* USE_TERMIOS */

	while ((getset_res = ioctl(0, TCSETAW, &ntty)) < 0)
		sys_fail(-1331, getset_res);

#endif /* USE_TERMIOS */

#else /* USE_SYSV_TERMIO */

#ifdef TTY
	notc.t_startc = otc.t_startc;
	notc.t_stopc = otc.t_stopc;

	while ((getset_res = ioctl(0, TIOCSETC, (char *)&notc)) < 0)
		sys_fail(-1331, getset_res);
#endif /* TTY */

	while ((getset_res = ioctl(0, TIOCSLTC, (char *)&noltc)) < 0)
		sys_fail(-1332, getset_res);

#endif /* USE_SYSV_TERMIO */

#ifndef USE_STDIO

	/* Now switch to asynchronous no-delay output and set 3270-mode flag */

	while ((fcntl_res = fcntl(fd_output, F_SETFL, FNDELAY + FASYNC)) < 0)
		sys_fail(-1333, fcntl_res);

#ifdef DEBUG
	(void)fprintf(outf, "setup_tty   : set fd_output (%d) synch mode file descriptor status to 0x%x\n",
				fd_output, fcntl(fd_output ,F_GETFL, 0));
#endif /* DEBUG */

#endif /* USE_STDIO */

	term_mode = 1;
	synch_op = 0;
}

/*
 * Reset initial tty mode.
 * However, the call may be for menu dialogue or at program termination.
 * The parameter req_mode is zero if it is program termination.
 * It is otherwise positive for dialogue mode (in which we want to go
 * back to a slightly modified version of the user's setup).
 * It can also be negative if all that we want to do is to switch to
 * synchronous mode (but still with the 3270 asynchronous setup).
 * In either case the absolute value is the delay in seconds which
 * is allowed for the synchronous output to succeed.
 * The req_flag parameter is TRUE if this is a "real" call of reset_tty,
 * so that the term_mode and synch_op values should be set and any hanging
 * output flushed with f_flush. The synch_op value will be then the
 * absolute value of req_mode.
 * A FALSE value for req_flag is used when reset_tty is called from the
 * initial save_tty routine: at this time the special f_* output routines
 * have not yet been initialised.
 */

#ifdef STANDARD_C

void    reset_tty(int req_mode, bool req_flag)

#else /* STANDARD_C */

void    reset_tty(req_mode, req_flag)

int     req_mode;
bool    req_flag;

#endif /* STANDARD_C */

{
	int     getset_res;

#ifndef USE_STDIO
	int     fcntl_res;
#endif /* USE_STDIO */

#ifdef DEBUG
	(void)fprintf((outf ? outf : stdout), "reset_tty   : (%d, %d) reset tty setup according to parameter\n",
				req_mode, req_flag);
#endif /* DEBUG */

	if (req_mode < 0)
	{
		setup_tty(FALSE);
	}
	else
	{
		if (req_mode)
		{

#ifdef USE_SYSV_TERMIO

#ifdef USE_TERMIOS

			tcflag_t        save_iflag = otty.c_iflag;
			tcflag_t        save_oflag = otty.c_oflag;
			tcflag_t        save_lflag = otty.c_lflag;

#else /* USE_TERMIOS */

			unsigned short  save_iflag = otty.c_iflag;
			unsigned short  save_oflag = otty.c_oflag;
			unsigned short  save_lflag = otty.c_lflag;

#endif /* USE_TERMIOS */

			unsigned char   save_MIN   = otty.c_cc[VMIN];
			unsigned char   save_TIME  = otty.c_cc[VTIME];


			otty.c_iflag = (otty.c_iflag & ~BRKINT & ~ISTRIP & ~INLCR & ~IGNCR & ~IXOFF) | IGNBRK | IXON | ICRNL;
			otty.c_oflag = (otty.c_oflag & ~OLCUC & ~ONLCR & ~OCRNL & ~ONLRET & ~TAB3) | ONLCR;
			otty.c_lflag = (otty.c_lflag & ~ISIG & ~ICANON & ~XCASE & ~ECHO & ~ECHOE & ~ECHOK & ~ECHONL);
			otty.c_iflag = otty.c_iflag | ICRNL;
			otty.c_oflag = otty.c_oflag | ONLCR;
			otty.c_lflag = otty.c_lflag & ~ICANON;
			otty.c_cc[VMIN] = 1;
			otty.c_cc[VTIME] = 0;

#ifdef USE_TERMIOS

			while ((getset_res = tcsetattr(fd_output, TCSANOW, &otty)) < 0)
				sys_fail(-1340, getset_res);

#else /* USE_TERMIOS */

			while ((getset_res = ioctl(fd_output, TCSETAW, &otty)) < 0)
				sys_fail(-1340, getset_res);

#endif /* USE_TERMIOS */

#ifdef DEBUG
	show_tty("reset_tty   : Flags settings as reset for output", &otty);
#endif /* DEBUG */

			otty.c_iflag = save_iflag;
			otty.c_oflag = save_oflag;
			otty.c_lflag = save_lflag;
			otty.c_cc[VMIN] = save_MIN;
			otty.c_cc[VTIME] = save_TIME;

#else /* USE_SYSV_TERMIO */

			short   save_flags = otty.sg_flags;

			otty.sg_flags = (otty.sg_flags & (~RAW & ~ECHO)) | (CBREAK | CRMOD);
			while ((getset_res = ioctl(fd_output, TIOCSETP, (char *)&otty)) < 0)
				sys_fail(-1340, getset_res);

#ifdef DEBUG
	show_tty("reset_tty   : Flags settings as reset for output", &otty);
#endif /* DEBUG */

			otty.sg_flags = save_flags;

#endif /* USE_SYSV_TERMIO */

		}

		if (KE)
			f_puttcs(tcsKE, FALSE);
#ifdef DEBUG
		else
			(void)fprintf((outf ? outf : stdout), "reset_tty   : No ke string found\n");
#endif /* DEBUG */

	}

	/*
	 * We must reset the file descriptor state.
	 * If simply terminating then reset whatever the original setting was,
	 * otherwise switch to synchronous.
	 */

	if (req_mode)
	{

#ifndef USE_STDIO

		while ((fcntl_res = fcntl(fd_output, F_SETFL, O_RDWR)) < 0)
			sys_fail(-1343, fcntl_res);

#ifdef DEBUG
	(void)fprintf((outf ? outf : stdout), "reset_tty   : set r/w fd_output (%d) file descriptor status = 0x%x\n",
					fd_output, fcntl(fd_output ,F_GETFL, 0));
#endif /* DEBUG */

#endif /* USE_STDIO */

#ifdef DEBUG
	if (!outf)
		sleep(5);
#endif /* DEBUG */


		if (req_flag)
		{
			term_mode = -1;
			synch_op = ((req_mode < 0) ? -req_mode : req_mode);
			if (op_wait)
				f_flush();
		}
	}
	else
	{
		f_clearop(0);
		term_mode = 0;

#ifndef USE_STDIO

		while ((fcntl_res = fcntl(fd_output, F_SETFL, fds_init)) < 0)
			sys_fail(-1344, fcntl_res);

#ifdef DEBUG
	(void)fprintf((outf ? outf : stdout), "reset_tty   : reset (0x%x) fd_output (%d) file descriptor status = 0x%x\n",
				fds_init, fd_output, fcntl(fd_output ,F_GETFL, 0));
#endif /* DEBUG */

#endif /* USE_STDIO */

#ifdef DEBUG
	(void)fprintf((outf ? outf : stdout), "reset_tty   : now reset fd_output flags\n");
#endif /* DEBUG */

#ifdef USE_SYSV_TERMIO

#ifdef USE_TERMIOS

		while ((getset_res = tcsetattr(fd_output, TCSADRAIN, &otty)) < 0)
			sys_fail(-1345, getset_res);

#else /* USE_TERMIOS */

		while ((getset_res = ioctl(fd_output, TCSETAW, &otty)) < 0)
			sys_fail(-1345, getset_res);

#endif /* USE_TERMIOS */

#else /* USE_SYSV_TERMIO */

		/* restore action of special characters */

#ifdef TTY
		while ((getset_res = ioctl(0, TIOCSETC, (char *)&otc)) < 0)
			sys_fail(-1341, getset_res);
#endif /* TTY */

		while ((getset_res = ioctl(0, TIOCSLTC, (char *)&oltc)) < 0)
			sys_fail(-1342, getset_res);

		while ((getset_res = ioctl(fd_output, TIOCSETP, (char *)&otty)) < 0)
			sys_fail(-1345, getset_res);

#endif /* USE_SYSV_TERMIO */

#ifdef DEBUG

	/* Read back the fd_output terminal state for debug checks */

#ifdef USE_SYSV_TERMIO

#ifdef USE_TERMIOS

	while ((getset_res = tcgetattr(fd_output, &otty)) < 0)
		sys_fail(-1346, getset_res);

#else /* USE_TERMIOS */

	while ((getset_res = ioctl(fd_output, TCGETA, &otty)) < 0)
		sys_fail(-1346, getset_res);

#endif /* USE_TERMIOS */

#else /* USE_SYSV_TERMIO */

	while ((getset_res = ioctl(fd_output, TIOCGETP, (char *)&otty)) < 0)
		sys_fail(-1346, getset_res);

#endif /* USE_SYSV_TERMIO */

	show_tty("reset_tty   : Flags settings after reset for fd_output", &otty);

#endif /* DEBUG */

#ifndef USE_STDIO

	/* We also have to deal with stdout */

#ifdef DEBUG
	(void)fprintf((outf ? outf : stdout), "reset_tty   : now reset stdout flags\n");
#endif /* DEBUG */

#ifdef USE_SYSV_TERMIO

#ifdef USE_TERMIOS

		while ((getset_res = tcsetattr(1, TCSADRAIN, &otty1)) < 0)
			sys_fail(-1347, getset_res);

#else /* USE_TERMIOS */

		while ((getset_res = ioctl(1, TCSETAW, &otty1)) < 0)
			sys_fail(-1347, getset_res);

#endif /* USE_TERMIOS */

#else /* USE_SYSV_TERMIO */

		while ((getset_res = ioctl(1, TIOCSETP, (char *)&otty1)) < 0)
			sys_fail(-1347, getset_res);

#endif /* USE_SYSV_TERMIO */

#ifdef DEBUG

	/* Read back the stdout terminal state for debug checks */

#ifdef USE_SYSV_TERMIO

#ifdef USE_TERMIOS

	while ((getset_res = tcgetattr(1, &otty1)) < 0)
		sys_fail(-1347, getset_res);

#else /* USE_SYSV_TERMIO */

	while ((getset_res = ioctl(fd_output, TCGETA, &otty1)) < 0)
		sys_fail(-1347, getset_res);

#endif /* USE_SYSV_TERMIO */

#else /* USE_SYSV_TERMIO */

	while ((getset_res = ioctl(fd_output, TIOCGETP, (char *)&otty1)) < 0)
		sys_fail(-1347, getset_res);

#endif /* USE_SYSV_TERMIO */

	show_tty("reset_tty   : Flags settings after reset for stdout", &otty1);
	if (outf)
		sleep(5);

#endif /* DEBUG */

#endif /* USE_STDIO */

	}
}


/*
 * Centralise any character input/output to the 3270 screen here.
 * This allows us not to output if we are in transparent I/O mode, but
 * still to update the screen array as if we did so.
 * Also allows us to check on the result of any output.
 */


/* Get a single character in if it comes within a given time */

#ifdef STANDARD_C

int     f_getc(int secs_timer)

#else /* STANDARD_C */

int     f_getc(secs_timer)

int     secs_timer;

#endif /* STANDARD_C */

{
	/* Wait for getchar input, but not too long */
	if (!select_check(1, 1, 0, 1, secs_timer, 0,
			    -1350, "dialogue(s) select"))
		return(EOF);
	return (fgetc(stdin));
}


/*
 * Clear out any unsent output buffers.
 * The save_buf parameter specifies whether to keep at least
 * one buffer in the list (which will then be empty)
 * or to release the complete list.
 * This means that after f_clearop(0) is called there will no longer
 * be any buffers. Thus, if it is ever required to restart output there
 * must be a call of get_opbf first).
 * If save_buf is negative do NOT clear out any hanging output.
 * If save_buf is positive clear out any hanging output. Note that
 * this should be used with great caution, since it can clear stuff
 * which we thought was definitely output, including stuff done in
 * synchronous mode.
 *
 * Of course, after this clearout we must normally reset the state of the
 * terminal, since we maybe interrupted output while in the status line,
 * while in bold, blink or whatever.
 * This is avoided, however, by a call with save_buf = 2 (necessary
 * in a first call when the strings are not yet set up).
 */

#ifdef STANDARD_C

void    f_clearop(int save_buf)

#else /* STANDARD_C */

void    f_clearop(save_buf)

int     save_buf;

#endif /* STANDARD_C */

{
	ch_left = op_wait = 0;
	if (save_buf >= 0)
	{

#ifdef DEBUG
	if (outf)
		(void)fprintf(outf, "f_clearop   : call for fd_output (%d)\n",
					fd_output);
#endif /* DEBUG */

		(void)fflush(fd_stream);
	}

	while (first_opbf)
	{
		if (save_buf && !(first_opbf->ch_count))
			break;
		rel_opbf(save_buf);
	}

	if (first_opbf && (save_buf != 2))
	{
		f_puttcs(tcsFS, FALSE);
		f_puttcs(tcsME, FALSE);
		f_puttcs(tcsCL, TRUE);
	}
}


/*
 * Write out the current output buffer(s).
 * If all of the output can be written then return -1
 * otherwise return the number of characters that were written
 * (and so zero implies output not possible)
 *
 * If, however, we are in synchronous mode (either the initial user
 * terminal setup or on-line help mode or transparent I/O) then a
 * write could hang up on us.
 * To beat this we set up a timer before the write is attempted.
 * If this timer expires then we can either go back to the 3270 screen
 * (if the user was in on-line help) or exit the program.
 */

#ifdef STANDARD_C

int     f_output(void)

#else /* STANDARD_C */

int     f_output()

#endif /* STANDARD_C */

{
	int     iores;
	int     ch_written = 0;
	char    *op_pointer;
	char    *op_buffer;
	int     chcount;

#ifdef TTY
	int     setjmp_res = 0;
#endif /* TTY */

#ifdef DEBUG
	int     chi;
#endif /* DEBUG */


	op_wait = 0;
	while (chcount = first_opbf->ch_count - first_opbf->ch_sent)
	{
		op_buffer = first_opbf->buffer;
		if (!first_opbf->ch_sent)
			op_buffer[chcount] = '\0';
		op_pointer = op_buffer + first_opbf->ch_sent;

#ifdef DEBUG
/* */
	(void)fprintf(outf, "f_output    : op_buffer 0x%lx of %d characters (preceded by %d sent)\n<",
				op_pointer, chcount, first_opbf->ch_sent);
	for (chi = 0 ; chi < chcount ; chi++)
	{
		(void)fprintf(outf, "%s", print_ascii((unsigned short)*(op_pointer + chi)));
		if (chi && !(chi % 50))
			(void)fprintf(outf, ">\n<");
	}
	(void)fprintf(outf, ">\n");
/* */
#endif /* DEBUG */

#ifdef TTY
		/* In the case of a synchronous write set a timer */

		if (synch_op)
		{
;                       (void) signal(SIGALRM, real_int);
			(void) alarm((unsigned)synch_op);
			setjmp_res = setjmp(environment);
		}

		if (setjmp_res)
		{
			/*
			 * The subsequent write statement has hung.
			 * Kill the program (unless it is already being killed, in
			 * which case just pretend that the write was OK).
			 */

#ifdef DEBUG
	(void)fprintf(outf, "f_output    : synchronous write has hung %s\n",
				(ending_program ? "again" : " "));
#endif /* DEBUG */

			if (!ending_program)
				end_program(-999, "hanging synchronous write", 0);
			iores = chcount;
		}
		else
		{
#endif /* TTY */

			iores = write(fd_output, op_pointer, chcount);

#ifdef DEBUG
	(void)fprintf(outf, "f_output    : synch_op is %d, fd_output is %d, iores = %d\n",
				synch_op, fd_output, iores);
#endif /* DEBUG */


#ifdef TTY
			/* OK, the write has completed: kill the timer */

			(void) alarm(0);
			(void) signal(SIGALRM, SIG_IGN);
		}
#endif /* TTY */
		if (iores != chcount)
		{
			if (iores > 0)
			{

#ifdef DEBUG
	(void)fprintf(outf, "f_output    : partial write iores = %d\n",
				iores);
#endif /* DEBUG */

				first_opbf->ch_sent += iores;
				ch_written += iores;
				ch_left -= iores;
				op_wait = ch_left;
				break;
			}
			if ((errno == EWOULDBLOCK) || (errno == EAGAIN))
			{

#ifdef DEBUG
	(void)fprintf(outf, "f_output    : write iores = %d (write would block), ch_left = %d\n",
				iores, ch_left);
#endif /* DEBUG */

				op_wait = ch_left;
				break;
			}
			else
				end_program(-990, "write error %d on output", iores);
		}

#ifdef DEBUG
	(void)fprintf(outf, "f_output    : completed write: chcount = %d\n",
				chcount);
#endif /* DEBUG */

		ch_written += chcount;
		ch_left -= chcount;
		rel_opbf(TRUE);
	}
	if (!chcount)
	{
		ch_written = -1;
		ch_left = op_wait = 0;
	}

#ifdef DEBUG
	(void)fprintf(outf, "f_output    : exit value is %d (op_wait = %d)\n",
				ch_written, op_wait);
#endif /* DEBUG */

	return (ch_written);
}

/* Flush the output to the user screen (if possible!) */

#ifdef STANDARD_C

void    f_flush(void)

#else /* STANDARD_C */

void    f_flush()

#endif /* STANDARD_C */

{

#ifdef DEBUG
	(void)fprintf(outf, "f_flush     : first_opbf->buffer (0x%lx)->ch_count = %d\n",
				first_opbf->buffer, first_opbf->ch_count);
#endif /* DEBUG */

	if (!op_wait || synch_op)
		(void) f_output();

#ifdef TRANSP_IO
	if (!trflag)
#endif /* TRANSP_IO */

	{
		(void)fflush(fd_stream);
	}
}

/*
 * Add characters into the final output buffer (if possible).
 *
 * One special case is where we are in asynchronous output mode but
 * also in transparent mode.
 * In such a case we just ignore the call.
 * This works because when we finish the transparent mode we will in
 * any case redraw the 3270 screen.
 */

#ifdef STANDARD_C

void    f_addchars(char *chaddr, int count, bool flush)

#else /* STANDARD_C */

void    f_addchars(chaddr, count, flush)

char    *chaddr;
int     count;
bool    flush;

#endif /* STANDARD_C */

{
	char    *chptr = chaddr;
	int     chcopyn = count;
	int     copy_count;


#ifdef TRANSP_IO
	if (!trflag || synch_op)
#endif /* TRANSP_IO */

	{
		copy_count = last_opbf->ch_size - last_opbf->ch_count;
		while (chcopyn > copy_count)
		{
			strncpy(last_opbf->buffer + last_opbf->ch_count, chptr, (size_t) copy_count);
			last_opbf->ch_count = last_opbf->ch_size;
			ch_left += copy_count;
			if (!op_wait)
			{

#ifdef DEBUG
	(void)fprintf(outf, "f_addchars  : call f_output\n");
#endif /* DEBUG */

				(void) f_output();
			}
			if (op_wait)
			{

#ifdef DEBUG
	(void)fprintf(outf, "f_addchars  : get new output buffer\n");
#endif /* DEBUG */

				get_opbf();
			}
			chptr += copy_count;
			chcopyn -= copy_count;
			copy_count = last_opbf->ch_size;
		}
		strncpy(last_opbf->buffer + last_opbf->ch_count, chptr, (size_t) chcopyn);
		last_opbf->ch_count += chcopyn;
		ch_left += chcopyn;
		if (flush)
		{
			if (!op_wait || synch_op)
			{
				if (f_output() > 0)
					f_flush();
			}
		}
	}
}

/* Output a single character to the user screen (if possible!) */

#ifdef STANDARD_C

void    f_putc(char ochar)

#else /* STANDARD_C */

void    f_putc(ochar)

char    ochar;

#endif /* STANDARD_C */

{
	char    character = ochar;

	f_addchars(&character, 1, FALSE);
}

/* Output a string to the user screen (if possible!) */

#ifdef STANDARD_C

void    f_puts(char *string, bool flush)

#else /* STANDARD_C */

void    f_puts(string, flush)

char    *string;
bool    flush;

#endif /* STANDARD_C */

{
	if (string)
		f_addchars(string, (int)strlen(string), flush);
}

/*
 * Output a control string character to the user screen (if possible!)
 * If 8-bit controls output is allowed then escape sequences in the string
 * can be shortened first.
 */

#ifdef STANDARD_C

int     f_putcs(char ochar)

#else /* STANDARD_C */

int     f_putcs(ochar)

char    ochar;

#endif /* STANDARD_C */

{
	char    ochar2 = ochar;

	if (eight_bc)
	{
		if (ESC_oflag)
		{
			if (!(ochar2 = eight_cc[(int)ochar]))
			{
				f_putc((char)ESC);
				ochar2 = ochar;
			}
			ESC_oflag = FALSE;
		}
		else
		{
			if (ochar2 == ESC)
			{
				ESC_oflag = TRUE;
				ochar2 = 0;
			}
		}
	}
	if (ochar2)
		f_putc(ochar2);
	return ((int) ochar);
}

/* Output a particular tcs string to the user screen (if possible!) */

#ifdef STANDARD_C

void    f_puttcs(int tcs_index, bool flush)

#else /* STANDARD_C */

void    f_puttcs(tcs_index, flush)

int     tcs_index;
bool    flush;

#endif /* STANDARD_C */

{
	f_puts(tcsptr[tcs_index], flush);
}

/* Output a formatted string to the user screen (if possible!) */

#ifdef STANDARD_C

void    f_putf(char *string, bool flush, int p1)

#else /* STANDARD_C */

void    f_putf(string, flush, p1)

char    *string;
bool    flush;
int     p1;

#endif /* STANDARD_C */

{
	int     str_length;

	(void)sprintf(&op_string[0], string, p1);
	if ((str_length = strlen(op_string)) >= (OPSTRSIZE - 1))
	{
		/* Looks like we have a possible memory overwrite */
		/* Afraid that we have to kill this task  */
		/* without even any messages!             */

#ifdef LOGFILE
		logit("Emergency: f_putf buffer overflow", 0);
		logscreen();
#endif /* LOGFILE */

		/* try to kill with a dump */
		kill(getpid(), SIGBUS);
	}

	f_addchars(op_string, str_length, flush);
}

#ifdef STANDARD_C

void    empty_input(void)

#else /* STANDARD_C */

void    empty_input()

#endif /* STANDARD_C */

{
	int     selres;

#ifdef DEBUG
	int     dropchar;
#endif /* DEBUG */

	/* make sure there is nothing left to input */
	f_flush();

	if (selres = select_check(1, 1, 0, 1, 0, 100000, -1940, "empty_input select"))
	{
		if (selres < 0)
			end_program(-1950, "empty_input select error %d", selres);

		if (exceptmask.fds_bits[0])
		{
			/* Oh, dear, an exception: au revoir */
			end_program(-1910, "empty_input input exception 0x%x", (int) exceptmask.fds_bits[0]);
		}
		if (readmask.fds_bits[0])
		{

#ifdef DEBUG
	dropchar = fgetc(stdin);
	(void)fprintf(outf, "empty_input : drop spurious character '%s'\n",
					print_ascii((unsigned short)dropchar));
#endif /* DEBUG */

		}
	}
}

/*
 * Return to the most recent action request inside the kbread buffer.
 * Reset the pointers to this and flag the situation by negating the
 * value of kbread_count. This value will be reset positive when more
 * input comes along. It allows us to handle partial escape sequences.
 * It should only be called when there IS an action request!
 */

#ifdef STANDARD_C

void    reset_kbread(void)

#else /* STANDARD_C */

void    reset_kbread()

#endif /* STANDARD_C */

{
	while (*kbread_p >= 0)
	{
		kbread_count++;
		if (kbread_p == kbread)
			kbread_p += kbread_max;
		kbread_p--;
	}
	kbread_count = -kbread_count;
}

/*
 * Add a character to the circular buffer kbread.
 * Return TRUE if OK, FALSE if buffer was full.
 */

#ifdef STANDARD_C

bool    kbr_add(short c)

#else /* STANDARD_C */

bool    kbr_add(c)

short   c;

#endif /* STANDARD_C */

{
	if (kbread_count < 0)
		kbread_count = -kbread_count;
	if (kbread_count >= kbread_max)
		return (FALSE);

#ifdef DEBUG
	(void)fprintf(outf, "kbr_add     : add in ");
	if (c <= 0)
		(void)fprintf(outf, "%d\n",c);
	else
		(void)fprintf(outf, "'%s'\n", print_ascii((unsigned short)c));
#endif /* DEBUG */

	kbread_count++;
	*kbrput_p = c;
	if (++kbrput_p == (kbread + kbread_max))
		kbrput_p = kbread;

	return (TRUE);
}

/*
 * Try to read a character from the circular buffer.
 * Return -999 if nothing is available.
 */

#ifdef STANDARD_C

short   kbread_next(void)

#else /* STANDARD_C */

short   kbread_next()

#endif /* STANDARD_C */

{
	short   result;

	if (kbread_count <= 0)
		return (-999);
	kbread_count--;
	result = *kbread_p;
	if (++kbread_p == (kbread + kbread_max))
		kbread_p = kbread;

#ifdef DEBUG
	(void)fprintf(outf, "kbread_next : return ");
	if (result <= 0)
		(void)fprintf(outf, "%d\n",result);
	else
		(void)fprintf(outf, "'%s'\n",print_ascii((unsigned short)result));
#endif /* DEBUG */

	return (result);
}

/* routine to clear out keyboard input */

#ifdef STANDARD_C

void    clear_kbin(void)

#else /* STANDARD_C */

void    clear_kbin()

#endif /* STANDARD_C */

{
	short           ch;

#ifdef DEBUG
	(void)fprintf(outf, "clear_kbin  : clear keyboard input\n");
	(void)fprintf(outf, "clear_kbin  : pb_top = %d, kbbuf_count = %d\n",
		       pb_top, kbbuf_count);
#endif /* DEBUG */

	while ((ch = kbread_next() != -999))
	{

#ifdef DEBUG
		(void)fprintf(outf, "clear_kbin  : discard kbread entry %d\n", ch);
#endif /* DEBUG */

	}

#ifdef DEBUG
	while (pb_top)
	{
		ch = pushback_stack[--pb_top];
		if (ch < 0)
			(void)fprintf(outf, "clear_kbin  : discard pushback action %d\n", ch);
		else
			(void)fprintf(outf, "clear_kbin  : discard pushback char '%s'\n",
					print_ascii((unsigned short)ch));
	}
#endif /* DEBUG */

	pb_top = 0;
	while (TRUE)
	{

#ifdef DEBUG
		while (kbbuf_count--)
		{
			(void)fprintf(outf, "clear_kbin  : discard kbbuf char ");
			if (*kbbuf_p & 0x80)
				(void)fprintf(outf, "%d\n",*kbbuf_p);
			else
				(void)fprintf(outf, "'%s'\n",print_ascii((unsigned short)*kbbuf_p));
			kbbuf_p++;
		}
#endif /* DEBUG */

		kbbuf_count = 0;
		if (select_check(1, 1, 0, 1, clkbin_sec, clkbin_usec, -1360, "clear_kbin select") <= 0)
			break;
		if (!readmask.fds_bits[0])
			break;
		kbbuf_count = read(0,kbbuf,kbbuf_max);
		if (kbbuf_count==0)
		{
			/* "Can't happen", but it doesn't hurt to be safe */
			break;
		}
		if (kbbuf_count < 0)
			end_program(1315, "keyboard read failure %d", kbbuf_count);
		kbbuf_p = kbbuf;
	}
}

/* routine to simulate ungetc (which doesn't really do the right thing) */

#ifdef STANDARD_C

void    unget(short c)

#else /* STANDARD_C */

void    unget(c)

short   c;

#endif /* STANDARD_C */

{
	if (pb_top < pb_max)
		pushback_stack[pb_top++] = c;
}

/* Little utility to say if kbbuf has anything in */

#ifdef STANDARD_C

int     kbbuf_chars(void)

#else /* STANDARD_C */

int     kbbuf_chars()

#endif /* STANDARD_C */

{
	return (kbbuf_count);
}

/*
 * Try to read a character from the keyboard (or pushback stack).
 * Return EOF if nothing has been typed (so read() would block).
 */

#ifdef STANDARD_C

short   kbbuf_next(void)

#else /* STANDARD_C */

short   kbbuf_next()

#endif /* STANDARD_C */

{
	/* first try for unget() characters */
	if (pb_top)
		return pushback_stack[--pb_top];

	/* next try for characters already read from the keyboard */
	while (!kbbuf_count)
	{
		/* none there; try to get some more from the keyboard */
		if (!(select_check(1, 1, 0, 1, 0, 0, -1362, "kbbuf_next select")))
			return (EOF);

		kbbuf_count = read(0,kbbuf,kbbuf_max);
		kbbuf_p = kbbuf;
		if (kbbuf_count == 0)
		{
			/* "Can't happen", but it doesn't hurt to be safe */

#ifdef DEBUG
	(void)fprintf(outf, "kbbuf_next  : zero return on read\n");
#endif /* DEBUG */


			/* end_program(1000, "keyboard read zero return", 0); */
		}
		if (kbbuf_count < 0)
		{
			end_program(1000, "keyboard read failure %d", kbbuf_count);
		}

#ifdef TTY
		if (idle_count >= idle_warn)
		{

#ifdef DEBUG
	(void)fprintf(outf, "kbbuf_next  : reset screen after user found non-idle\n");
#endif /* DEBUG */

			clear_kbin();
			act_refresh(1);
			slrestore();
			kbbuf_count = 0;
			idle_count = 0;
			return (EOF);
		}
		idle_count = -1;
#endif /* TTY */

#ifdef DEBUG
	if (kbbuf_count)
		(void)fprintf(outf, "kbbuf_next  : read in %d characters\n",
					kbbuf_count);

#endif /* DEBUG */

	}

	if (kbbuf_count)
	{
		/*
		 * Check for special treatment characters.
		 * An 8-bit control character can be changed to two seven-bit
		 * characters (Escape and then a 7-bit printable character).
		 * An 8-bit normal character should imply that we can
		 * output 8-bit characters also.
		 */

		if (*kbbuf_p & 0x80)
		{
			if (iscntrl(*kbbuf_p & 0x7f))
			{
				if (!eight_bc)
				{
					/*
					 * Now we know we can use eight-bit controls.
					 * For efficiency change the outgoing tcs control sequences
					 * and the style change sequences.
					 */

					int     tcsind;
					int     from, to;

					for (tcsind = 0 ; tcsind < tcsMAX ; tcsind++)
					{
						if (tcslen[tcsind])
							tcslen[tcsind] = f_compress((char *)tcsptr[tcsind]);
					}


#ifndef EXTENDED
					for (from = 0 ; from < MAXSTYLES ; from++)
					{
						for (to = 0 ; to < MAXSTYLES ; to++)
						{
							if (change_sl[from][to])
								change_sl[from][to] = f_compress(change_cp[from][to]);
						}
					}
#endif /* EXTENDED */

					eight_bc = TRUE;
				}

				if (ESC_iflag)
				{
					ESC_iflag = FALSE;
					*kbbuf_p -= 0x40;
				}
				else
				{

#ifdef DEBUG
	(void)fprintf(outf, "kbbuf_next  : return Escape character for 8-bit control char \'%s\'\n",
				print_ascii((unsigned short)*kbbuf_p));
#endif /* DEBUG */

					ESC_iflag = TRUE;
					return (ESC);
				}
			}
			else
				eight_bd = TRUE;
		}

		kbbuf_count--;

		if (isprint(*kbbuf_p & 0x7f))
			*kbbuf_p |= SO_imask;

#ifdef DEBUG
	(void)fprintf(outf, "kbbuf_next  : return '%s'\n",
				print_ascii((unsigned short)*kbbuf_p));
#endif /* DEBUG */

		return ((*kbbuf_p++) & 0x00ff);
	}
	return (EOF);
}

/*
 * Process input from the keyboard input buffer kbbuf.
 * Deal with anything which can be dealt with immediately, otherwise
 * put the stuff into the circular buffer kbread.
 * Return the last character/action dealt with, or 0 if nothing done.
 * It may be entered in a "help only" mode to show the user the key action.
 */

#ifdef STANDARD_C

int     do_kbbuf(bool help_only)

#else /* STANDARD_C */

int     do_kbbuf(help_only)

bool    help_only;

#endif /* STANDARD_C */

{
	short processed[10], *p;
	short c;
	short ptr;
	short result = 0;
	int contflag;

	while ((c=kbbuf_next()) != EOF)
	{
		if ((ptr = escape_start[c]) == 0)
		{
			if (help_only)
				help_act((int)-c);
			else
				if (!kbr_add(c))
					return(result);
			result = c;
			continue;
		}

		/* start of an escape sequence? */

#ifdef DEBUG
	(void)fprintf(outf, "do_kbbuf    : Character '%s'> starts an escape sequence\n",
				print_ascii((unsigned short)c));
#endif /* DEBUG */

		p = processed;
		*p++ = c;
		contflag = FALSE;
		while (ptr > 0)
		{
			if ((c = kbbuf_next()) == EOF)
			{
				while (--p >= processed)
					unget(*p);

#ifdef DEBUG
	(void)fprintf(outf, "do_kbbuf    : Incomplete escape sequence: unget %d characters\n",pb_top);
#endif /* DEBUG */

				return(result);
			}

			if (!c)
				c = 0x80;

			for (;;)
			{
				if (escape_char[ptr] == (c & 0xff))
				{
					/* another char matched */
					*p++ = c;
					ptr = escape_ptr[ptr];
					break; /* resume outer loop */
				}
				if (escape_char[ptr] == FLAG)
				{
					/* not found */
					/* skip one character and try the rest to
					 * see if there is an escape sequence there somewhere.
					 */

#ifdef DEBUG
	(void)fprintf(outf, "do_kbbuf    : process_input character '%s': not found\n",
				print_ascii((unsigned short)c));
#endif /* DEBUG */

					unget(c);
					while (--p > processed)
						unget(*p);
					if (help_only)
						help_act(-(*processed));
					else
						if (!kbr_add(*processed))
							return(result);
					result = *processed;
					contflag = TRUE;
					break;
				}
				ptr++;
			}
			if (contflag) break;
		}
		if (contflag) continue;
		ptr = (-ptr) - 1;

#ifdef DEBUG
	(void)fprintf(outf, "do_kbbuf    : Found action number %d (%s)\n",
				ptr, actions[ptr].name);
#endif /* DEBUG */

		if (help_only)
		{
			short *pp;

			f_puts( "\r\nThe key(s) that you typed gave the input sequence\r\n", FALSE);
			for (pp = processed ; pp != p ; pp++)
				help_act(-(*pp));

			help_act(ptr);
			if (!ptr)
				return(-1);
		}
		else
		{
			if (actions[ptr].immediate < 0)
			{
				/*
				 * Ignore if there is more keyboard input waiting.
				 * This is because some stupid terminals sometimes save up
				 * a control character (often SI) and send it before the
				 * next normal character. This happens, for instance, on some
				 * terminals set in 7-bit space parity when the initialise
				 * termcap sequence includes <CSI>62;1"p
				 */

				if (!kbbuf_chars())
					(*actions[ptr].action)(actions[ptr].arg);
			}
			else
				if (!kbr_add(-ptr))
					return(result);
			result = -ptr;

#ifdef TTY
			/* reset the cpu timer regularly */
			set_cpu_timer(TRUE);
#endif /* TTY */

		}
	}
	return (result);
}

/*
 * This a pseudo-handling of keyboard buffer input.
 * It will be simply used to show what would be the action of a key.
 * it will exit either on a simple Enter or a timer expiration.
 * Return value is EOF for timer expiry, 0 for Enter, > 0 otherwise.
 */

#ifdef STANDARD_C

int     read_kbbuf(int sec_timer)

#else /* STANDARD_C */

int     read_kbbuf(sec_timer)

int     sec_timer;

#endif /* STANDARD_C */

{
	int     done_kb;

	while (!kbbuf_chars())
	{
		/* Wait for keyboard input, but not too long */
		if (!select_check(1, 1, 0, 1, sec_timer, 0,
				    -1370, "dialogue(s) select"))
			return (EOF);
		if (done_kb = do_kbbuf(TRUE))
			f_flush();
		if (done_kb == -1)
			return (0);
	}
	return (1);
}


/*
 * send transparently input to the IBM.
 * It may be that we may have to send stuff to the user's screen.
 * If so then we will need to switch to synchronous output mode,
 * for which the timer (in seconds) will be the given parameter.
 *
 * This routine can be called for graphics or for kermit transfers.
 * Kermit is a bit different in that the stuff sent to the IBM should
 * include the 3270 input block header and trailer.
 *
 * We can tell which it is because one of the two possible variables
 * graphics and kermit will be non-zero. Remember, however, that they
 * might not be defined: kermit is defined if we have chosen the TTY
 * compile option, graphics if the GRAPHICS option.
 *
 * It is also possible to call this routine before a 3270 screen has
 * been set up, particularly in cases where the IBM is front-ended by
 * some simple dialogue system. This case can be distinguished by
 * a zero value for binflag.
 * In such cases we should simply send the character(s) typed.
 */

#ifdef STANDARD_C

void    send_transp(int timer)

#else /* STANDARD_C */

void    send_transp(timer)

int     timer;

#endif /* STANDARD_C */

{
	unsigned char *p;
	int     c;
	int     selres;

#ifdef GRAPHICS
	int     synch_save;
#endif /* GRAPHICS */

#ifdef DEBUG
	register int i;
#endif /* DEBUG */

	unsigned char ptrgr[1000], *pi;

#ifdef DEBUG
	(void)fprintf(outf, "send_transp : timer = %d", timer);

#ifdef GRAPHICS
	(void)fprintf(outf, ", graphics = %d", graphics);
#endif /* GRAPHICS */

#ifdef TTY
	(void)fprintf(outf, ", kermit = %d", kermit);
#endif /* TTY */

	(void)fprintf(outf, "\n");
#endif /* DEBUG */

	p  = ptrgr;

	/* Set up synchronous output temporarily */

	if (synch_op)
	{

#ifdef GRAPHICS
		synch_save = synch_op;
#endif /* GRAPHICS */

		synch_op = timer;
	}
	else
		reset_tty(-timer, TRUE);

#ifdef GRAPHICS
	if ((graphics > 0) || (binflag))
#else /* GRAPHICS */
	if (binflag)
#endif /* GRAPHICS */

	{
		/* Build 3270 input block header..      */

		*p++ = ENTERkey;
		*p++ = byte1(cursorpos);
		*p++ = byte2(cursorpos);
	}

	pi = p;

	while (TRUE)
	{
		while ((c=kbbuf_next()) == EOF)
		{
			selres = select_check(1, 1, 0, 1, -1, 0,
					       -1373, "fromkeyboard select");

#ifdef DEBUG
	(void)fprintf(outf, "send_transp : %s after wait fds_bits is %d, select returns %d\n",
		getstime(),readmask.fds_bits[0],selres);
#endif /* DEBUG */

		}

#ifdef DEBUG
	(void)fprintf(outf, "send_transp : %s kbbuf_next gets 0x%2.2x\n",
		getstime(), c);
#endif /* DEBUG */

		c &= 0x7f;

#ifdef TTY
		if (kermit)
		{
			/* Looks like Kermit transfer */

			/*
			 * I don't know why I used to echo the incoming character.
			 * However, it certainly upsets some systems, including Versaterm
			 * on a Mac, so I comment it out and wait for the complaints!
			 */

			/* f_putc((char)c); */

			*p++ = c;
			if (c == '\r')
			{
				break;
			}
		}
#endif /* TTY */

#ifdef GRAPHICS
		if (graphics >= 0)
#endif /* GRAPHICS */

		{
			/* Send input to IBM */

#ifdef GRAPHICS
			if ((pi == p) && (graphics > 0) && binflag)
#else /* GRAPHICS */
			if ((pi == p) && binflag)
#endif /* GRAPHICS */

			{
				if (c == 29)
				{
					/* We allow CTRL-] as Telnet escape char(s) */

#ifdef DEBUG
	(void)fprintf(outf, "send_transp : %s Escape after CTRL-]\n",getstime());
#endif /* DEBUG */

					act_escape(0);
					return;
				}
			}

#ifdef GRAPHICS
			/*
			 * When in graphics input mode all sorts of funnies happen.
			 * Try to deal with the worst of them.
			 * This includes an attempt to handle editing keys.
			 */

			if (c == 4)
				continue; /* Kludge for extra ^D's */

			/* Allow for editing characters */

			if ((c == 8) || (c == 127))
			{
				/* Backspace or Delete : ignore if at start */

				if (p == pi)
					continue;

				/* Delete: backspace, write a blank, backspace */

				f_putc('\010');
				f_putc(' ');
				f_putc('\010');
				f_flush();
				p--;
				continue;
			}
#endif /* GRAPHICS */

			f_putc((char)c);

#ifdef GRAPHICS
			if (graphics >= 0)
#endif /* GRAPHICS */

			{
				if ((c == '\n') && (p != pi) && (*(p-1) == ('\n' | 0x80)))
				{
					*p++ = c | 0x80;
					c = '\r';
					f_putc((char)c);
				}
			}

			f_flush();
			*p = c;

#ifdef GRAPHICS
			if (graphics > 0)
				*p |= 0x80;      /* Add Bit7 */
#endif /* GRAPHICS */

			p++;
			if (c == '\r')
			{

#ifdef GRAPHICS
				if (graphics > 0)
				{

/* */
					if (p != pi+1)
					{
						/* Discard any extra CR (but echo it!) */

#ifdef DEBUG
	(void)fprintf(outf, "send_transp : %s discard rubbish input\n",getstime());
#endif /* DEBUG */

						selres = select_check(1, 1, 0, 1, disccr_sec, disccr_usec,
								       -1374, "fromkeyboard (discard CR) select");

#ifdef DEBUG
	(void)fprintf(outf, "send_transp : %s end of select: result was %d\n",getstime(),selres);
#endif /* DEBUG */

						while ((c = kbbuf_next()) != EOF)
						{

#ifdef DEBUG
	(void)fprintf(outf, "send_transp : %s rubbish kbbuf_next 0x%2.2x\n",getstime(),c);
#endif /* DEBUG */

							if (c == '\r')
							{
								f_putc((char)c);
								f_flush();
							}
						}
					}
/* */
				}
				else
#endif /* GRAPHICS */

				{
					c = '\n';
					f_putc((char)c);
					*p++ = c;
				}

				break;
			}
		}

#ifndef TTY

#ifndef GRAPHICS

		/* May need to break out if neither graphics nor kermit */

		if (binflag)
			break;

#endif /* GRAPHICS */

#endif /* TTY */


	}

#ifdef DEBUG
	(void)fprintf(outf, "send_transp : %s KEYBOARD INPUT %d chars: '",getstime(), (p-pi));
	if ((p-pi) > 1)
		for (i=0; i<(p-pi-1); i++)
			 (void)fprintf(outf, "%c", (*(pi+i)) & 0x7f);
	(void)fprintf(outf, "'\n");

#ifdef GRAPHICS
	if (graphics > 0)
		(void)fprintf(outf, "send_transp :    at [CURSOR(%d,%d)]\n",YX(cursorpos));
#endif /* GRAPHICS */

#ifdef TTY
	if (kermit)
		(void)fprintf(outf, "send_transp :    at [CURSOR(%d,%d)]\n",YX(cursorpos));
#endif /* TTY */

#endif /* DEBUG */

#ifdef GRAPHICS
	if ((graphics > 0) || binflag)
#else /* GRAPHICS */
		if (binflag)
#endif /* GRAPHICS */

	{
		/* Add 3270 input block trailer */

		*p++ = TELCMD_IAC;
		*p++ = TELCMD_EOR;
	}
		/* send off the block */
	writnet(net, ptrgr, p-ptrgr);

#ifdef TONET
	if (write(tonet, (char *)ptrgr, p-ptrgr) < 0)
		;
#endif /* TONET */

#ifdef DEBUG

#ifdef GRAPHICS
	if (binflag)
		(void)fprintf(outf, "send_transp : Input  %d transp. bytes: %2.2x ... %2.2x\n",
			  (p-ptrgr-5), *(ptrgr+3), *(p-3));
#endif /* GRAPHICS */

#endif /* DEBUG */

	/* Lock out keyboard input for now */
	unlock = FALSE;

#ifdef GRAPHICS
	if (graphics)
	{
		/* Reset (a)synchronous mode */

		if (synch_save)
			synch_op = synch_save;
		else
			setup_tty(FALSE);
	}
#endif /* GRAPHICS */

	return;
}

/* Treat stuff from the circular buffer kbread */

#ifdef STANDARD_C

void    do_kbread(void)

#else /* STANDARD_C */

void    do_kbread()

#endif /* STANDARD_C */

{
	short   c;

	while ((c = kbread_next()) != -999)
	{
		if (c > 0)
			act_nil(c);
		else
		{
			(*actions[-c].action)(actions[-c].arg);

#ifdef TTY
		/* reset the cpu timer regularly */
		set_cpu_timer(TRUE);
#endif /* TTY */

			if (!unlock)
				break;
		}
	}
}

/* Put another character into savebuf at position savep */

#ifdef STANDARD_C

void    put_save(unsigned char ch)

#else /* STANDARD_C */

void    put_save(ch)

unsigned char   ch;

#endif /* STANDARD_C */

{
	if (savep == (savebuf + saveb_size))
	{
		savebuf = (unsigned char *) incr_buffer((unsigned long *)savebuf, saveb_size, SAVEB_INCR, '\0');
		savep = savebuf + saveb_size;
		saveb_size += SAVEB_INCR;
	}
	*savep++ = ch;

}

/*
 * We call this routine when the user appears to have had problems.
 * Maybe he switched sessions and has now switched back in.
 * Maybe he used X-OFF/X-ON (naughty fellow!).
 * Maybe network problems caused output to get hung temporarily.
 * Maybe etc. etc.
 * In any case, we will force a master reset and clean up for him.
 * Note that this routine assumes the user is in asynchronous 3270 mode!
 */

#ifdef STANDARD_C

void    reset_user(void)

#else /* STANDARD_C */

void    reset_user()

#endif /* STANDARD_C */

{

#ifdef DEBUG
	(void)fprintf(outf, "reset_user  : refresh the screen etc.\n");
#endif /* DEBUG */

	f_clearop(-1);
	op_wcount = op_wcntst = 0;
	clear_kbin();
	mainreset(cursorpos);
}

/* Retry hung output */

#ifdef STANDARD_C

void    retry_output(bool clock_up)

#else /* STANDARD_C */

void    retry_output(clock_up)

bool    clock_up;

#endif /* STANDARD_C */

{

#ifdef DEBUG
	(void)fprintf(outf, "retry_output: %s: retry hung asynch output by f_output call\n",
			     getstime());
#endif /* DEBUG */

	op_wcntst++;
	if (f_output())
	{
		/* If we were in a long wait then reset the user */

		if (op_wcount < 0)
		{
			reset_user();

#ifdef LOGFILE
		logit("Exit slow wait mode in retry_output(%d)",
			clock_up);
#endif /* LOGFILE */

		}

		/* Some output went out: check if all of it went */

		if (op_wait)
		{
			op_wcount = 1;
		}
		else
		{
			op_wcount = op_wcntst = 0;
		}
	}
	else
	{
		if (clock_up)
		{

			/*
			 * Clock up another failed wait
			 * If we get too many then we stop looking so often!
			 * This we can do by setting op_wcount negative.
			 * We then move to the slower wait.
			 */

			if ((op_wcount >= 0) && ((op_wcount++) > op_wcmax))
			{

#ifdef LOGFILE
		logit("Switch to slow o/p wait: op_wcount = %d",
			op_wcount);
#endif /* LOGFILE */

#ifdef DEBUG
	(void)fprintf(outf, "retry_output: %d retries of hung asynch output: switch to slow timeout\n",
				op_wcount);
#endif /* DEBUG */

			       op_wcount = -1;
			}
		}
	}
}

/*
 * This is where we have the basic select and wait.
 * The select is to return at intervals of real time.
 * At each return we may check for hang-up situations.
 * For a tty this means the user being idle for too long.
 *
 * Since the introduction of asynchronous output to the user's 3270
 * screen it is also possible that there is stacked output for it.
 * The indicator for this is op_wait, which is the count of waiting output
 * which we have actually tried to send out.
 * If op_wait is positive then we should retry at intervals to call the
 * f_output routine, keeping a count of failed tries (in op_wcount).
 * Of course, the difficult thing is to know how long to wait (according
 * to the speed at which we can get the stuff out).
 * If we don't wait long enough then we waste CPU usage: if we wait too
 * long then the user sees delays.
 * On a workstation we don't normally hang up anyway, and even if we do
 * we should not wait too long.
 * On a tty line it may depend on ospeed, but may also depend on the
 * timing interval of a DECserver (normally 70ms intervals).
 * However, in all cases the output to the terminal can be genuinely
 * hung, either because the user has used ^S or because he is on a terminal
 * server and has switched to a different session.
 * For the case where a user is really blocked then we should take special
 * action. If on a workstation it is OK to simply retry at rather longer
 * intervals. If, however, he is on a terminal then it is quite likely
 * that he is on a terminal server and has switched to another session.
 * In this latter case we do NOT always want to continue output when he
 * returns, since his screen will be in a wrong setup. Instead, we can
 * scrap any new output and wait until he appears to return again
 * (maybe he enters a character such as ^g, maybe different, maybe we
 * find that we can output again).
 * When he does return then we force in a master reset (as with ^g)
 * so as to redraw the screen.
 * While he is away we can afford just to simulate screen output (as
 * we do sometimes when a user is in graphics mode), so as to avoid
 * getting lots of new output buffers full.
 */

#ifdef STANDARD_C

void    reader(void)

#else /* STANDARD_C */

void    reader()

#endif /* STANDARD_C */

{
	register int    c;
	unsigned char   temp[50], *p;
	int             selres;
	int             mask;
	extern  int     errno;
	extern  char    *sys_errlist[];

#ifdef TTY
	int             warnings;

	start_cpu_timer();
#endif /* TTY */

	while (TRUE)
	{
		/* Loop until IBM has sent something */

		while (!netbuf_count)
		{
			/*
			 * If we just received stuff from the IBM then in some cases
			 * it is better to wait a short while and see if some more stuff
			 * comes along. However, this only applies if there is something
			 * to be output.
			 * Also, remember that output to the user screen might be hung!
			 * In this case it is probably worth retrying the output.
			 */

			if ((s_pfirst >= 0) || clearsn)
			{
				if (op_wait)
					retry_output(FALSE);

				/*
				 * Have a quick check if more network input.
				 * It is worth waiting a little while for efficiency reasons,
				 * but this wait should be less if running on a workstation.
				 */

				selres = select_check(net+1, 1<<net, 0, 1<<net, reader1_sec, reader1_usec,
						       -1380, "reader efficiency select");
				if (selres && (readmask.fds_bits[0] & (1<<net)))
				{
					if (op_wait)
						retry_output(FALSE);
					break;
				}

#ifdef DEBUG
	(void)fprintf(outf, "reader      : No net input after wait of %d %d\n",
		 waitst.tv_sec, waitst.tv_usec);
#endif /* DEBUG */

				/* OK, we can now update the user screen */

				if (clearsn > 0)
					clscreen();
				update_cs();

				/* Now we may add a further wait before checking keyboard input */
				/* This is because even if keyboard input IS allowed some stupid */
				/* programs expect to be able to send more stuff immediately */

				if (unlock && !op_wait)
				{
					selres = select_check(net+1, 1<<net, 0, 1<<net, reader2_sec, reader2_usec,
							       -1381, "reader (unlock wait) select");
					if (selres && (readmask.fds_bits[0] & (1<<net)))
						break;

#ifdef DEBUG
	(void)fprintf(outf, "reader      : No net input after another wait of %d %d\n",
		 waitst.tv_sec, waitst.tv_usec);
#endif /* DEBUG */

				}
			}

			mask = (1 << net);

			if ((!op_wait) || (op_wcntst > op_wcmin))
			{
				/*
				 * Now deal with any kbbuf input while not in locked state
				 * and also not in the middle of receiving IBM data.
				 */

				while ((kbread_count> 0) && unlock && (savebuf == savep))
					do_kbread();

				/* Now we can wait for any new input */

#ifdef TTY
				if (unlock && (idle_count < 0))
					idle_count = 0;
#endif /* TTY */

				/* Select keyboard input if kbbuf is empty */

#ifdef TRANSP_IO
				/* and if input unlocked or not in transparent mode */
				if ((!kbbuf_count) && (unlock || !trflag))
#else /* TRANSP_IO */
				if (!kbbuf_count)
#endif /* TRANSP_IO */

					mask |= 1;
			}

			/* set the select timer (timer_sec, timer_usec) */

			if (op_wait)
			{
				/* There is hung output: maybe we want to retry immediately */

				if (op_wcount)
				{
					if (op_wcount > 0)
					{
						timer_sec = reader3_sec;
						timer_usec = reader3_usec;
					}
					else
					{
						timer_sec = reader_sec;
						timer_usec = 0;
					}
				}
				else
				{
					timer_sec = 0;
					timer_usec = 0;
				}
			}
			else
			{

#ifdef TTY
				timer_sec = reader_sec;
#else /* TTY */
				timer_sec = -1;
#endif /* TTY */

				timer_usec = 0;
			}

			selres = select_check(net+1, mask , 0 , mask , timer_sec, timer_usec,
					       -1382, "Main reader select");
			if (selres)
			{

#ifdef DEBUG
	(void)fprintf(outf, "reader      : select (unlock = %d, timer %d.%d) returns %d, readmask 0x%x\n",
		      unlock, timer_sec, timer_usec, selres, readmask.fds_bits[0]);
#endif /* DEBUG */

				if (readmask.fds_bits[0] & (1<<net))
					break;
				if (readmask.fds_bits[0] & 1)
				{

#ifdef TRANSP_IO
					if (trflag)
						send_transp(SYNCH_TRANSP);
					else
#endif /* TRANSP_IO */

					if (binflag)
					{
						if (op_wait & (op_wcount < 0))
						{
							reset_user();

#ifdef LOGFILE
		logit("Exit slow wait mode: readmask.fds_bits[0] = 0x%x",
			(int) readmask.fds_bits[0]);
#endif /* LOGFILE */

						}
						else
							(void) do_kbbuf(FALSE);
					}
					else
					{
						send_transp(SYNCH_NORMAL);
						break;
					}
				}
			}
			else
			{
				/* timeout expired */
				/* If hung on asynch output then retry f_output */

				if (op_wait)
					retry_output(TRUE);

#ifdef TTY
				/* if a long (reader_sec) wait then clock up another count */

				if ((!op_wait) || (op_wcount < 0))
				{
					/* check for quiescent user */

					if (idle_count >= 0)
					{
						if (!(++idle_count % idle_warn))
						{
							if ((warnings = (idle_count / idle_warn)) > idle_wnum)
							{
								/* No response to any of the warnings: kill him */

									end_program(301, "You are too quiescent for my liking: au revoir et bientot!", 0);
							}
							if (warnings == 1)
								slsave();
							f_puttcs(tcsCL, FALSE);
							f_putf( "Are you there? Warning %d\r\n",
									  FALSE, warnings);
							if (warnings == idle_wnum)
								f_puts( "This is your last warning!\r\n", FALSE);

							ly = 1;
							lx = 0;
							slrestore();
							slputstr("Are you there?", 55, 3*warnings);
						}
					}
				}
#endif /* TTY */

			}
		}

		/* Now we should have something from the IBM */

		if (op_wcount > 0)
		{
			op_wcount = op_wcntst = 0;
		}

		if ((c=netgetc())==EOF)
		{
			clearscreen();
			return;
		}
		if (c == TELCMD_IAC)
		{
			/* gather up the whole command
			    * note that we are blithely assuming that the input is
			    * syntactically correct (for example, IAC SB and IAC SE match,
			    * EOF doesn't occur in embarrassing places, etc)
			    */
			p = temp;
			*p++ = c;
			switch (*p++ = netgetc())
			{
				case TELCMD_IAC:
					/* escaped IAC */
					c = temp[1];
					goto not_IAC;
				case TELCMD_DO:
				case TELCMD_DONT:
				case TELCMD_WILL:
				case TELCMD_WONT:
					*p++ = netgetc();
					break;
				case TELCMD_SB:
					/* save everything up to and including the next
					      * IAC SE
					      */
					for (;;)
					{
						while ((*p++ = netgetc()) != TELCMD_IAC)
							;
						while ((*p++ = netgetc()) == TELCMD_IAC)
							;
						if (p[-1] == TELCMD_SE) break;
					}
					/* anything else is considered to be a single-character
					     * command
					     */
				default:
					break;
			}
			respondto(temp);
			continue;
		} /* end if (c == TELCMD_IAC) */
not_IAC:
		if (binflag)
			put_save((unsigned char)c);
		else if (c >= 0x20 && c < 0xff)
		{
			fputc(c,stderr);
		}
		else switch(c)
		{
			case '\t':

#ifdef DEBUG
	(void)fprintf(outf, "reader      : Not IAC: control char ox%x with binflag = %d\n",
		      c, binflag);
#endif /* DEBUG */

			case '\n':
			case '\r':
				fputc(c,stderr);
				break;
			default:
				(void)fprintf(stderr,"\\%03o",c);
				break;
		}
	}
}

/* check if the command/option pair was OK */

#ifdef STANDARD_C

void    resp_check(int cmd, int opt, int *flag)

#else /* STANDARD_C */

void    resp_check(cmd, opt, flag)

int     cmd, opt;
int     *flag;

#endif /* STANDARD_C */

{

#ifdef DEBUG
	(void)fprintf(outf, "resp_check  : %s %s>\n",telcmd(cmd), telopt(opt));
#endif /* DEBUG */

	if (!flag)
		end_program(302, "I cannot handle option %d : host may not be a VM host",
					 opt);

	if (cmd == TELCMD_WILL || cmd == TELCMD_DO) {
		respond_opt(cmd,opt,*flag,1);
		*flag = 1;

		/* Check for in binary mode: if so then start 3270 screen */

		if (opt == TELOPT_BINARY) {
			if (!sent_tt)
				end_program(303, "Host doesn't seem to accept a 3270", 0);
			if (will_binary && do_binary && !binflag) {

#ifdef DEBUG
	(void)fprintf(outf, "resp_check  : Now call in the init3270 part\n");
#endif /* DEBUG */

				binflag = 1;
				init3270();
			}
		}
	}
	else {
		/* Must be TELCMD_WONT or TELCMD_DONT */

		respond_opt(cmd,opt,*flag,0);
		*flag = 0;

		/* Check for reverting out of binary mode */

		if (opt == TELOPT_BINARY) {
			if (binflag) {

#ifdef DEBUG
	(void)fprintf(outf, "resp_check  : Now revert to the original simple output\n");
#endif /* DEBUG */

				binflag = 0;
				sent_tt = 0;

				/* switch to normal mode */

				clearnormal();
				reset_tty(SYNCH_NORMAL, TRUE);
			}
		}
	}
}

/* respond to the telnet command pointed to by p. (*p == TELCMD_IAC) */

#ifdef STANDARD_C

void    respondto(unsigned char *p)

#else /* STANDARD_C */

void    respondto(p)
unsigned char *p;

#endif /* STANDARD_C */

{
	char    response[50];
	int     cmd, opt;
	int     *flag = NULL;
	int     write_res;

	cmd = *++p;
	switch(cmd) {

	case TELCMD_DO:
	case TELCMD_DONT:

#ifdef DEBUG
	(void)fprintf(outf, "respondto   : <==<IAC %s %s>\n",
				telcmd(*p), telopt(p[1]));
#endif /* DEBUG */

		switch(opt = *++p) {
		case TELOPT_TERMTYPE:
			flag = &will_tt;
			break;
		case TELOPT_EOR:
			flag = &will_eor;
			break;
		case TELOPT_BINARY:
			flag = &will_binary;
			break;
		case TELOPT_TM:
			flag = &will_tm;
			break;
		}
		resp_check(cmd, opt, flag);
		return;

	case TELCMD_WILL:
	case TELCMD_WONT:

#ifdef DEBUG
	(void)fprintf(outf, "respondto   : <==<IAC %s %s>\n",
				telcmd(*p), telopt(p[1]));
#endif /* DEBUG */

		switch(opt = *++p) {
		case TELOPT_EOR:
			flag = &do_eor;
			break;
		case TELOPT_BINARY:
			flag = &do_binary;
			break;
		case TELOPT_SGA:
			flag = &do_sga;
			break;
		}
		resp_check(cmd, opt, flag);
		return;

	case TELCMD_SB:
		/* currently, only respond to IAC SB TERMTYPE SEND IAC SE */
		/* (void)printf("subnegotiation "); */
		if ((opt = *++p) != TELOPT_TERMTYPE) {
			return;
		}
		if (*++p != SEND) {
			return;
		}

#ifdef DEBUG
	(void)fprintf(outf, "respondto   : <==<IAC SB TERMTYPE SEND IAC SE>\n");
#endif /* DEBUG */

		(void)sprintf(response, "%c%c%c%c%s%c%c",
		TELCMD_IAC, TELCMD_SB, TELOPT_TERMTYPE, ISC,
		TERMNAME, TELCMD_IAC, TELCMD_SE);
		while ((write_res = write(net, response, 6+(int)strlen(TERMNAME))) < 0)
			sys_fail(320, write_res);

#ifdef TONET
		if (write(tonet, response, 6+(int)strlen(TERMNAME)) < 0)
			;
#endif /* TONET */

#ifdef DEBUG
		(void)fprintf(outf, "respondto   : ==><IAC SB TERMTYPE IS \"%s\">\n",TERMNAME);
#endif /* DEBUG */

		sent_tt = 1;
		return;
	case TELCMD_EOR:

#ifdef DEBUG
		(void)fprintf(outf, "respondto   : <==<IAC EOR>\n");
#endif /* DEBUG */

		if (!binflag) {
			(void)printf("protocol error: unexpected EOR\n");
			return;
		}
		interpret();
		savep = savebuf;

		if ((dialname || username) && FORMATTED && cursorpos)
		{
			/*
			 * If there is a request to start up with a DIAL command
			 * then try to force in an Enter plus a request to DIAL.
			 * If dialname is not an empty string then we can even
			 * execute this command.
			 * Otherwise treat the username string as the user ID
			 * Put it into the kbbuf input array
			 * omitting the first char if = '!'.
			 */

			int     i;

			clear_kbin();

			if (dialname)
			{
				static  char    dial[] = "DIAL ";


#ifdef DEBUG
		(void)fprintf(outf, "respondto   : setup dial to <%s>\n",
				dialname);
#endif /* DEBUG */
				(void)kbr_add( (short) '\0');
				for (i = 0 ; i < strlen(dial) ; i++)
					(void)kbr_add( (short) dial[i]);
				for (i = 0 ; i < strlen(dialname) ; i++)
					(void)kbr_add( (short) dialname[i]);
				if (dialname[0])
					(void)kbr_add( (short) '\0');
				dialname = NULL;
			}
			else
			{

#ifdef DEBUG
		(void)fprintf(outf, "respondto   : setup username as <%s>: cursorpos is (%d,%d)\n",
				username, YX(cursorpos));
#endif /* DEBUG */

				for (i = ((username[0] == '!') ? 1 : 0) ; i < strlen(username) ; i++)
					(void)kbr_add( (short) username[i]);

				/* add in backtab + tab to position cursor */

				(void)kbr_add( (short) -2);
				(void)kbr_add( (short) -1);

#ifdef LOGFILE
				if (name_field == -1)
				{
					name_field = cursorpos;
					while (!(screen[name_field] & FIELDSTART))
						DEC(name_field);

#ifdef DEBUG
	(void)fprintf(outf, "respondto   : set name_field to (%d,%d)\n",
				YX(name_field));
#endif /* DEBUG */

				}
#endif /* LOGFILE */
			}
			username = NULL;

#ifdef DEBUG
		(void)fprintf(outf, "respondto   : user or dial name setup\n");
#endif /* DEBUG */

		}
		return;

	default:
		/* ignore everything else */
		(void)printf("Don't understand IAC %d (%03o) ",*p,*p);
		return;
	}
}

static int response[16] = {
	/*             DO         DONT       WILL          WONT  */
	/* 0 0 */ TELCMD_WONT,           0, TELCMD_DONT,         0,
	/* 0 1 */ TELCMD_WILL,          -1, TELCMD_DO,          -1,
	/* 1 0 */          -1, TELCMD_WONT,        -1, TELCMD_DONT,
	/* 1 1 */           0, TELCMD_WILL,         0, TELCMD_DO
	    /* 0: no response; -1: impossible combination */
};

#ifdef STANDARD_C

void    respond_opt(int cmd, int opt, int oldstate, int newstate)

#else /* STANDARD_C */

void    respond_opt(cmd, opt, oldstate, newstate)

int     cmd;
int     opt;
int     oldstate;
int     newstate;

#endif /* STANDARD_C */

{
	int stimulous, r;
	char rbuf[4];
	int     write_res;

	stimulous =
	    cmd == TELCMD_DO ? 0 :
	cmd == TELCMD_DONT ? 1:
	cmd == TELCMD_WILL ? 2: 3;
	r = response[8*oldstate + 4*newstate + stimulous];
	if (r == -1)
		end_program(399, "programming error", 0);
	else if (r) {
		(void) sprintf(rbuf, "%c%c%c",TELCMD_IAC,r,opt);
		while ((write_res = write(net, rbuf, 3)) < 0)
			sys_fail(322, write_res);

#ifdef TONET
		if (write(tonet, rbuf, 3) < 0)
			;
#endif /* TONET */

#ifdef DEBUG
		(void)fprintf(outf, "respondto   : ==><IAC %s %s>\n", telcmd(r), telopt(opt));
#endif /* DEBUG */

	}
}

#ifdef TTY

#ifdef STANDARD_C

SIGNAL_T close_int(int sig_num)

#else /* STANDARD_C */

SIGNAL_T close_int(sig_num)

int     sig_num;

#endif /* STANDARD_C */

{
	/* Looks like shutdown failed on us, folks */
	/* Afraid that we have to kill this task  */
	/* without even any messages!             */

#ifdef LOGFILE
	logit("Emergency: shutdown hanging", 0);
	logscreen();
#endif /* LOGFILE */

	/* try to kill with a dump */
	kill(getpid(), SIGBUS);
	SIGNAL_RETURN;
}
#endif /* TTY */

/*
 * Close down the socket (if it is open).
 */

#ifdef STANDARD_C

void    close_socket(void)

#else /* STANDARD_C */

void    close_socket()

#endif /* STANDARD_C */

{
	/*
	 * This is where we are liable to get hung up.
	 * At least on Ultrix, and maybe elsewhere, there can be
	 * problems if the socket is in a funny state.
	 * On a workstation the user will see (and take action).
	 * However, on a tty (LAT?) connection we need to act!
	 */

	if (net)
	{

#ifdef TTY

		struct  itimerval       close_val;
		int     set_res;

		(void) signal(SIGPROF, close_int);
		close_val.it_value.tv_sec = close_sec;
		close_val.it_value.tv_usec = 0;
		close_val.it_interval.tv_sec = 0;
		close_val.it_interval.tv_usec = 0;
		while ((set_res = setitimer(ITIMER_PROF, &close_val, (struct itimerval *) NULL)) < 0)
			sys_fail(382, set_res);

#endif /* TTY */

		if (shutdown(net,2) < 0)
		{

#ifdef LOGFILE
		logit("shutdown failure in end_program", 0);
#endif /* LOGFILE */

		}
		net = 0;

#ifdef DEBUG
	(void)fprintf(outf, "close_socket: %s called shutdown(net,2)\n",
		getstime());

	}
	else
	{
		(void)fprintf(outf, "close_socket: error call with net = 0\n");
#endif /* DEBUG */

	}
}

/* Routine to write robustly down a socket */

#ifdef STANDARD_C

void    writnet(int s, unsigned char *buf, int n)

#else /* STANDARD_C */

void    writnet(s, buf, n)

int             s;
unsigned char   *buf;
int             n;

#endif /* STANDARD_C */

{
	register int    i, k=0;
	register int    selres;

	while (TRUE)
	{
		selres = select_check(net+1, 0, 1<<net, 1<<net, writnet_sec, writnet_usec,
				       -1390, "writnet (to send to IBM) select");
		if (!selres)
			end_program(304, "timeout waiting to write to IBM", 0);

#ifdef DEBUG
		(void)fprintf(outf,"writnet     : send last %d bytes (of %d bytes) to IBM\n",
				n-k, n);
#endif /* DEBUG */

	     /* i = send(s, buf+k, n-k, 0); */
		while ((i = write(s, (char *)buf+k, n-k)) < 0)
			sys_fail(361, i);

#ifdef DEBUG
		(void)fprintf(outf,"writnet     : sent %d bytes (of %d bytes)\n",
				i, n-k);
#endif /* DEBUG */

		if ((k += i) == n)
			return;
	}
}

#ifdef STANDARD_C

int     netgetc(void)

#else /* STANDARD_C */

int     netgetc()

#endif /* STANDARD_C */

{
/*      unsigned char dummy[4];   */

	int     retry;

	if (netbuf_count-- == 0)
	{

		/* Some systems (sun?) occasionally return a false zero. */

		for (retry = 0 ; retry < 2 ; retry++)
		{
			int     selres;

			while ((netbuf_count = read(net,(char *)netbuf,NETBSIZE)) < 0)
				sys_fail(362, netbuf_count);

#ifdef DEBUG
	(void)fprintf(outf,"netgetc     : read returns %d\n",netbuf_count);
#endif /* DEBUG */

			if (netbuf_count)
				break;

			/*
			 * Wait again to see if there is network input.
			 * This is necessary because some systems would hang up on
			 * this repeat read request.
			 */

			selres = select_check(net+1, 1<<net, 0, 1<<net, 1, 0,
					       -1383, "repeat read select");
			if (selres && (readmask.fds_bits[0] & (1<<net)))
				continue;

#ifdef DEBUG
	(void)fprintf(outf, "netgetc     : No net input after wait of %d %d\n",
		 waitst.tv_sec, waitst.tv_usec);
#endif /* DEBUG */

		}
		if (netbuf_count == 0)
			return (EOF);

#ifdef FROMNET
	if (write(fromnet,(char *)netbuf,netbuf_count) < 0)
		;
#endif /* FROMNET */

/*              writnet(net, dummy, 0);  */

		netbuf_p = netbuf;
		netbuf_count--;
	}
	return((unsigned char)*netbuf_p++);
}
