/*
 *     Copyright CERN, Geneva 1989 - Copyright and any other
 *     appropriate legal protection of these computer programs
 *     and associated documentation reserved in all countries
 *     of the world.
 */

/*
 * Routines for locating the description of a particular terminal in
 * a terminal definition file, and extracting the definition.
 * See also parse.c
 *
 *
 */

#include        <stdio.h>
#include        <ctype.h>
#include        "globals.h"


/* Name and line of the profile currently being parsed (for error messages) */

char    *profile;
int     profline;

char    *default_def= "\
unknown|cern-default\
{\
enter='\^m';\
down='\\E[B'|'\\EOB';\
right='\\E[C'|'\\EOC';\
up='\\E[A'|'\\EOA';\
left='\\E[D'|'\\EOD'|'\^h';\
home='\^a';\
nl='\^j';\
insrt='\^r'|'\\E ';\
delete='\^d'|'\^?';\
einp='\^w';\
tab='\^i';\
btab='\^b';\
clear='\^l';\
pfk1='\\E1';\
pfk2='\\E2';\
pfk3='\\E3';\
pfk4='\\E4';\
pfk5='\\E5';\
pfk6='\\E6';\
pfk7='\\E7';\
pfk8='\\E8';\
pfk9='\\E9';\
pfk10='\\E0';\
pfk11='\\E!';\
pfk12='\\E@';\
pfk13='\\E#';\
pfk14='\\E$';\
pfk15='\\E%';\
pfk16='\\E\\^';\
pfk17='\\E&';\
pfk18='\\E*';\
pfk19='\\E(';\
pfk20='\\E)';\
pfk21='\\E-';\
pfk22='\\E=';\
pfk23='\\E_';\
pfk24='\\E+';\
pa1='\\Ea1';\
pa2='\\Ea2';\
pa3='\\Ea3';\
escape='\^]';\
help='\^e';\
keymap='\^p';\
option='\^o';\
extend='\^x';\
normal='\^n';\
mreset='\^g';\
reshow='\^v';\
reset='\^t';\
}";

/* The following two variables are for communication between find_def()
 * and nextc().
 */

static int line; /* current line number */
static int instring; /* true when inside '...' */

/*
 * Try to find a definition for the indicated terminal in the named file.
 * Return the (unparsed) definition, with newlines and excess white space
 * deleted.
 * If there is no matching terminal definition, return a null string.
 * For other errors (no such file, syntax error, etc), print an
 * appropriate message on stderr and return NULL.
 *
 * As a side effect, set globals profline and profile to point to the line
 * where the definition started.
 *
 * The result is in a static buffer that is overwritten on each call.
 */

#define FDError(msg) { \
	(void)fprintf(stderr, \
	 "Syntax error in definition starting at line %d of file %s: %s\n", \
	 profline,profile,msg); \
	(void) fclose(fi); \
	return NULL; }

/*
 * Get the next character from the file, stripping comments and
 * updating the global 'line' as necessary.  Throw away all newlines.
 * and unquoted white space.  As a service to the user, close all
 * quotes that run past end-of-line or end-of-file (with a warning).
 * As a side-effect, set the global 'instring' non-zero iff we are inside
 * a string.
 *
 * Note: for the purpose of this procedure, the ONLY function of \
 * is to prevent a following ' from terminating a quoted string.
 */

#ifdef STANDARD_C

int     nextc(FILE *fi)

#else /* STANDARD_C */

int nextc(fi)

register FILE *fi;

#endif /* STANDARD_C */

{
	register int    c;
	static  int     quoted = 0; /* set after ^ or \ inside string */

	for (;;)
	{
		c = fgetc(fi);
		if (c==EOF)
		{
			if (instring)
			{
				/* unclosed quote at eof */

				(void)fprintf(stderr,"Unclosed quote in line %d of file %s\n",
					line, profile);
				return('\'');
			}
			return(EOF);
		}
		if (c=='\n')
		{
			line++;
			if (instring)
			{
				/* newline inside string is illegal */

				(void)fprintf(stderr,"Unclosed quote in line %d of file %s\n",
						line, profile);
				return('\'');
			}
			continue; /* throw it away */
		}

		if (!instring)
		{
			if (isspace(c))
				continue; /* delete extra white space */
			if (c=='#')
			{
				/* unquoted comment character */
				/* delete comment */

				while ((c=fgetc(fi))!='\n' && c != EOF)
					;
				line++;
				continue;
			}
			if (c=='\'')
			{
				/* start of quoted string */

				instring = 1;
			}
			return(c);
		}

		/* inside a string */
		if (c=='\'' && !quoted)
		{
			instring = 0;
		}
		else
			if (c=='^' || c=='\\')
				quoted = 1-quoted;
			else
				quoted = 0;

		return(c);
	}
}


#ifdef STANDARD_C

char    *find_def(char *farray, char *fname, char *termname)

#else /* STANDARD_C */

char *find_def(farray,fname,termname)

char *farray, *fname, *termname;

#endif /* STANDARD_C */

{
	register FILE *fi;
	register char *p;
	register int c;
	char *name;
	int found;      /* set if the current definition contains the name 'termname'*/

	fi = fopen(fname,"r");
	if (fi == NULL)
	{
		perror(fname);
		return NULL;
	}
	profile = fname;
	line = 1;

	for (;;)
	{
		/* for each entry */

		c = nextc(fi);
		profline = line;
		p = farray;
		found = 0;

		/* loop over all names for this entry */

		for (;;)
		{
			/* for each name */
			/* invariant: p is pointing at next free location in farray and
			 * c is the first character of the next name.
			 */

			if (c==EOF)
			{
				/* return a null string (name not found in this file) */

				*farray = 0;
				(void) fclose(fi);
				return farray;
			}
			name = p;
			if (!isalnum(c))
				FDError("Missing terminal name")
			*p++ = c;
			while (isalnum(c = nextc(fi)) || c == '-' || c == '_')
			{      /* BS FIX */
				*p++ = c;
			}
			*p = 0;
			if (strcmp(name, termname) == 0)
				found = 1;

			/* any more names? */
			*p++ = c;
			if (c == '{')
				break;
			if (c != '|')
				FDError("Missing |")
			c = nextc(fi);
		} /* for each name */

		/* p[-1] == c == '{' */
		/* look for an unquoted '}' */

		while ((c = nextc(fi)) != '}' || instring)
		{
			if (c == EOF)
			{
				/* this is likely to be a very bad error, so abort */

				FDError("Missing }")
			}
			if (found) /* save the stuff in between */
				*p++ = c;
		}

		if (found)
		{
			*p++ = c;
			*p = '\0';
			(void) fclose(fi);
			return farray;
		}
	} /* for each entry */
} /* find_def */
