/*
 *     Copyright CERN, Geneva 1989 - Copyright and any other
 *     appropriate legal protection of these computer programs
 *     and associated documentation reserved in all countries
 *     of the world.
 */

#include "telnet.h"
#include "3270.h"
#include "globals.h"

/* variables to note new values found */
static  int             field;          /* first position of newly found field */
static  unsigned char   style;          /* style associated with the newfield */

#ifdef EXTENDED
static  unsigned char   f_extend = 0;   /* extended characteristics of current field */
#endif /* EXTENDED */

static          int      yval, xval;     /* position in image line */
static          short    amwrap = FALSE; /* set TRUE in autowrap last col write */

static          int      copy_n;
static          int      mvflag = FALSE;

static          unsigned short          scrchar;

unsigned char   *chsp;          /* pointer to curscr current line character */
unsigned char   *ccssp;         /* pointer to curscr current line style and character set */

#ifdef EXTENDED
static          unsigned char           scrch_ex;
unsigned char   *cextp;         /* pointer to curscr current line extended characteristics */
#endif /* EXTENDED */


int             screeni;        /* index into screen */
int             nlsp;           /* Number of chars up to last non-space char */

#ifdef LDEBUG
	char    *stylename[MAXSTYLES] =  { " NORMAL ",
					   " NORM_BLINK ",
					   " NORM_PROT ",
					   " NORM_PR_BLINK ",
					   " BOLD ",
					   " BOLD_BLINK ",
					   " BOLD_PROT ",
					   " BOLD_PR_BLINK ",
					   " NORM_MOD ",
					   " NORM_M_BLINK ",
					   " BOLD_MOD ",
					   " BOLD_M_BLINK ",
					   " STATL ",
					   " STATL_BLINK ",
					   " INVISIBLE "
					 };
#endif /* LDEBUG */

unsigned char   styletab[128] = { NORMAL, NORMAL, NORM_MOD, NORM_MOD,
				  NORM_PROT, NORM_PROT, NORM_PROT, NORM_PROT,
				  BOLD, BOLD, BOLD_MOD, BOLD_MOD,
				  INVISIBLE, INVISIBLE, INVISIBLE, INVISIBLE,
				  NORMAL, NORMAL, NORM_MOD, NORM_MOD,
				  NORM_PROT, NORM_PROT, NORM_PROT, NORM_PROT,
				  BOLD, BOLD, BOLD_MOD, BOLD_MOD,
				  INVISIBLE, INVISIBLE, INVISIBLE, INVISIBLE,
				  NORM_PROT, NORM_PROT, NORM_PROT, NORM_PROT,
				  NORM_PROT, NORM_PROT, NORM_PROT, NORM_PROT,
				  BOLD_PROT, BOLD_PROT, BOLD_PROT, BOLD_PROT,
				  INVISIBLE, INVISIBLE, INVISIBLE, INVISIBLE,
				  NORM_PROT, NORM_PROT, NORM_PROT, NORM_PROT,
				  NORM_PROT, NORM_PROT, NORM_PROT, NORM_PROT,
				  BOLD_PROT, BOLD_PROT, BOLD_PROT, BOLD_PROT,
				  INVISIBLE, INVISIBLE, INVISIBLE, INVISIBLE,
				  NORM_BLINK, NORM_M_BLINK, NORM_BLINK, NORM_M_BLINK,
				  NORM_BLINK, NORM_BLINK, NORM_BLINK, NORM_BLINK,
				  BOLD_BLINK, BOLD_M_BLINK, BOLD_BLINK, BOLD_M_BLINK,
				  INVISIBLE, INVISIBLE, INVISIBLE, INVISIBLE,
				  NORM_BLINK, NORM_M_BLINK, NORM_BLINK, NORM_M_BLINK,
				  NORM_BLINK, NORM_BLINK, NORM_BLINK, NORM_BLINK,
				  BOLD_BLINK, BOLD_M_BLINK, BOLD_BLINK, BOLD_M_BLINK,
				  INVISIBLE, INVISIBLE, INVISIBLE, INVISIBLE,
				  NORM_PR_BLINK, NORM_PR_BLINK, NORM_PR_BLINK, NORM_PR_BLINK,
				  NORM_PR_BLINK, NORM_PR_BLINK, NORM_PR_BLINK, NORM_PR_BLINK,
				  BOLD_BLINK, BOLD_BLINK, BOLD_BLINK, BOLD_BLINK,
				  INVISIBLE, INVISIBLE, INVISIBLE, INVISIBLE,
				  NORM_PR_BLINK, NORM_PR_BLINK, NORM_PR_BLINK, NORM_PR_BLINK,
				  NORM_PR_BLINK, NORM_PR_BLINK, NORM_PR_BLINK, NORM_PR_BLINK,
				  BOLD_BLINK, BOLD_BLINK, BOLD_BLINK, BOLD_BLINK,
				  INVISIBLE, INVISIBLE, INVISIBLE, INVISIBLE
				};


/*
 * Set up all of the strings to change from one "style" to another.
 * Set up the pointer array change_cp[from][to]
 * and the string lengths in change_sl[from][to]
 */

int     ifrom, ito;
int     chbufsiz;
bool    count_only;

#ifdef STANDARD_C

int     add_char(char addch)

#else /* STANDARD_C */

int     add_char(addch)

char    addch;

#endif /* STANDARD_C */

{
	if (!count_only)
	{
		change_cp[ifrom][ito][change_sl[ifrom][ito]] = (addch ? addch : 0x80);
	}
	chbufsiz++;
	change_sl[ifrom][ito] += 1;
	return(0);
}

#ifdef STANDARD_C

void    add_string(int from, int to, char *tcstring)

#else /* STANDARD_C */

void    add_string(from, to, tcstring)

int     from, to;
char    *tcstring;

#endif /* STANDARD_C */

{
	ifrom = from;
	ito = to;
	tputs(tcstring , 1 , add_char);
}

#ifdef STANDARD_C

void    copy_cp(int from, int to, int oldfrom, int oldto)

#else /* STANDARD_C */

void    copy_cp(from, to, oldfrom, oldto)

int     from, to;
int     oldfrom, oldto;

#endif /* STANDARD_C */

{

	if (!count_only)
	{
		(void)strcpy(change_cp[from][to] + change_sl[from][to],
			     change_cp[oldfrom][oldto]);
	}
	change_sl[from][to] += change_sl[oldfrom][oldto];
	change_rh[from][to] += change_rh[oldfrom][oldto];
	chbufsiz += change_sl[oldfrom][oldto];
}

#ifdef STANDARD_C

void    cchange_setup(void)

#else /* STANDARD_C */

void    cchange_setup()

#endif /* STANDARD_C */

{
	int     from, to, i;
	char    *chbufad = NULL;
	char    *chbufpt = NULL;
	int     pass_number;

#ifdef LDEBUG
	char    *ch;
#endif /* LDEBUG */

	/* This setup differs somewhat according to whether we are running
	 * with extended character attributes. If we are then settings for
	 * highlighting (bold, blinking, reverse video)
	 * attributes are done with these (i.e. the screen_ex array).
	 */



	chbufsiz = 0;
	for (pass_number = 0 ; pass_number < 4 ; pass_number++)
	{
		switch (pass_number)
		{
			case 0:
				count_only = TRUE;
				break;

			default:

#ifdef LDEBUG
	LOG(log, "Before pass %d \n", pass_number);
	for (from = 0 ; from < MAXSTYLES ; from++)
		for (to = 0 ; to < MAXSTYLES ; to++)
		{
			if (change_sl[from][to])
			{
				LOG(log, "from %s to %s string (%d chars) is <",
					stylename[from], stylename[to],
					change_sl[from][to]);

				if (ch = change_cp[from][to])
					for (i = 0 ; i < change_sl[from][to] ; i++)
					{
						if (!(*ch))
							break;
						LOG(log, "%s", print_ascii((unsigned short)*ch));
						ch++;
					}
				LOG(log, ">   change_rh is %d\n", change_rh[from][to]);
			}
		}
#endif /* LDEBUG */

				free((void *) chbufad);

			case 1:
				chbufpt = chbufad = (char *) alloc_buffer(chbufsiz, '\0');
				chbufsiz = 0;
				count_only = FALSE;
				break;
		}

		for (from = 0 ; from < MAXSTYLES ; from++)
		{
			for (to = 0 ; to < MAXSTYLES ; to++)
			{
				if (!count_only)
				{
					change_cp[from][to] = chbufpt;
					chbufpt += (change_sl[from][to] + 1);
				}
				chbufsiz++;
				change_sl[from][to] = 0;
				change_rh[from][to] = FALSE;
			}
		}

		/* temporarily need to know the ME sequence */
		add_string(NORMAL , NORMAL , ME);
		change_rh[NORMAL][NORMAL] = TRUE;

		/* Try to use bright mode */

		if (MD)
		{
			add_string(NORMAL , BOLD , MD) ;
			add_string(BOLD , NORMAL , ME);
			change_rh[BOLD][NORMAL] = TRUE;
			add_string(NORMAL , BOLD_PROT , MD);
			add_string(BOLD_PROT , NORMAL , ME);
			change_rh[BOLD_PROT][NORMAL] = TRUE;

			/*
			 * On a real VT340 a bright blank is different from a normal blank
			 * and so we cannot do any "clear to end of line" for bright
			 * blanks. However, this slows things down a lot, especially
			 * in something like FILELIST on a 132-column line.
			 * I have therefore commented it out!
			 */

		     /* ceol_ok[BOLD] = FALSE;      */
		     /* ceol_ok[BOLD_PROT] = FALSE; */
		}

#ifndef EXTENDED

		/* Try to use reverse video mode */

		if (MR)
		{
			add_string(NORMAL , BOLD_MOD , MR);
			add_string(BOLD_MOD , NORMAL , ME);
			change_rh[BOLD_MOD][NORMAL] = TRUE;
			ceol_ok[BOLD_MOD] = FALSE;
		}

		/* Try to use underscore mode */

/*              if (US)                                                 */
/*              {                                                       */
/*                      move_ok[NORM_MOD] = move_ok[NORM_M_BLINK] = MS; */
/*                      add_string(NORMAL , NORM_MOD , US);             */
/*                      add_string(NORM_MOD , NORMAL , UE);             */
/*                      change_rh[BOLD_MOD][NORMAL] = TRUE;             */
/*              }                                                       */

#endif /* EXTENDED */

		/* Maybe the termcap defines standout mode for status line */

		if (SO)
		{
			move_ok[STATL] = move_ok[STATL_BLINK] = MS;
			add_string(NORMAL , STATL , SO);
			add_string(STATL , NORMAL , SE);
			change_rh[STATL][NORMAL] = TRUE;
			if (MR && !strcmp(MR, SO))
				ceol_ok[STATL] = FALSE;
		}

		/* That is enough for the first pass */

		if (pass_number < 1)
			continue;

		/* On the second pass we need to switch to counting only */

		if (pass_number == 1)
			count_only = TRUE;

#ifndef EXTENDED

		/* Try to use blinking mode */

		if (MB)
		{
			ceol_ok[NORM_BLINK] = FALSE;
			add_string(NORMAL , NORM_BLINK , MB);
			add_string(NORM_BLINK , NORMAL , ME);
			change_rh[NORM_BLINK][NORMAL] = TRUE;
		}

#endif /* EXTENDED */

		for (i = NORMAL+2 ; i < INVISIBLE ; i += 2)
		{
			copy_cp(NORMAL , i+1 , NORMAL, i);

			if (MB)
			{
				ceol_ok[i+1] = FALSE;
				add_string(NORMAL , i+1 , MB);
				add_string(i+1 , NORMAL , ME);
				change_rh[i+1][NORMAL] = TRUE;
			}
			else
				copy_cp(i+1 , NORMAL , i , NORMAL);

			copy_cp(NORMAL+1 , i , NORMAL+1, NORMAL);
			copy_cp(NORMAL+1 , i , NORMAL, i);

			if (MB)
			{
				if (strcmp(change_cp[NORMAL+1][NORMAL] , change_cp[NORMAL][NORMAL]))
				{
					copy_cp(NORMAL+1 , i+1 , NORMAL+1 , NORMAL);
					copy_cp(NORMAL+1 , i+1 , NORMAL , i+1);
					add_string(NORMAL+1 , i+1 , MB);
				}
				else
					copy_cp(NORMAL+1 , i+1 , NORMAL, i);
			}
			else
				copy_cp(NORMAL+1 , i+1 , NORMAL, i);

			copy_cp(i , NORMAL+1 , i, NORMAL);
			if (MB)
			{
				add_string(i , NORMAL+1 , MB);

				if (strcmp(change_cp[i][NORMAL] , change_cp[NORMAL][NORMAL]))
				{
					copy_cp(i+1 , NORMAL+1 , i, NORMAL);
				}
				else
				{
					add_string(i+1 , NORMAL+1 , ME);
					change_rh[i+1][NORMAL+1] = TRUE;
					add_string(i+1 , NORMAL+1 , MB);
				}
			}
			else
				copy_cp(i+1 , NORMAL+1 , i, NORMAL + 1);

		}

#ifndef EXTENDED

		if (MB)
		{
			add_string(INVISIBLE , NORM_BLINK , MB);
			add_string(NORM_BLINK , INVISIBLE , ME);
			change_rh[NORM_BLINK][INVISIBLE] = TRUE;
		}

#endif /* EXTENDED */

		/* That is enough for the second pass */

		if (pass_number < 2)
			continue;

		/* On the third pass we need to switch to counting only */

		if (pass_number == 2)
			count_only = TRUE;

		for (from = NORMAL+2 ; from <= INVISIBLE ; from += 1)
		{
			for (to = NORMAL+2 ; to <= INVISIBLE ; to += 1)
			{
				if (strcmp(change_cp[NORMAL][from] , change_cp[NORMAL][to]))
				{
					if ((from & 0x1) || (to != from+1))
					{
						copy_cp(from , to , from, NORMAL);
						copy_cp(from , to , NORMAL, to);
					}
					else
					{
						if (MB)
							add_string(from , to , MB);
					}
				}
			}
		}
	}

	/* reset NORMAL to NORMAL */
	for (i = 0 ; i < change_sl[NORMAL][NORMAL] ; i++)
		change_cp[NORMAL][NORMAL][i] = '\0';
	change_sl[NORMAL][NORMAL] = 0;
		change_rh[NORMAL][NORMAL] = FALSE;

#ifdef LDEBUG
	LOG(log, "After %d passes\n", pass_number);
	for (from = 0 ; from < MAXSTYLES ; from++)
		for (to = 0 ; to < MAXSTYLES ; to++)
		{
			if (change_sl[from][to])
			{
				LOG(log, "from %s to %s string (%d chars) is <",
					stylename[from], stylename[to],
					change_sl[from][to]);

				if (ch = change_cp[from][to])
					for (i = 0 ; i < change_sl[from][to] ; i++)
					{
						if (!(*ch))
							break;
						LOG(log, "%s", print_ascii((unsigned short)*ch));
						ch++;
					}
				LOG(log, ">   change_rh is %d\n", change_rh[from][to]);
			}
		}
#endif /* LDEBUG */

}

/*
 * Calculate how many output chars would be needed to move from current
 * position, given by ly and lx, to new position.
 * the move might be done by either cursor movement (CM) or the special
 * VT-type cursor right sequence (which is used in fgoto inside mvcur).
 * Remember that after an autowrap at the end of the line we should use
 * straight cursor movement.
 */

#ifdef STANDARD_C

int     movestrcnt(int ny, int nx)

#else /* STANDARD_C */

int     movestrcnt(ny, nx)
int     ny, nx;

#endif /* STANDARD_C */

{
	return (strlen(tgoto(CM, nx, ny)));
}

/*
 * Put out a character onto the user screen.
 * The character is the one at line yv+1, column xv+1.
 * It is already stored in the curscr structure, so we can pick up
 * from there what we are supposed to be displaying.
 */

#ifdef STANDARD_C

void    display_char(int yv, int xv)

#else /* STANDARD_C */

void    display_char(yv, xv)

int     yv;
int     xv;

#endif /* STANDARD_C */

{
	int                     charset;
	unsigned short          prtchar;
	unsigned short          scr_char;
	unsigned char           *chsp1, *ccssp1;

#ifdef EXTENDED
	unsigned short          scr_ch_ex;
	unsigned char           *cextp1;
#endif /* EXTENDED */

		chsp1 = char_image[yv] + xv;
		ccssp1 = ccset_image[yv] + xv;

#ifdef EXTENDED
		cextp1 = ccext_image[yv] + xv;
#endif /* EXTENDED */

		scr_char = (*ccssp1 << 8) + *chsp1;

#ifdef LDEBUG
/*        LOG(log, "display_char    : at (%d,%d) scr_char = 0x%x\n", */
/*                        yv+1, xv+1, scr_char);                 */
#endif /* LDEBUG */

#ifdef EXTENDED
		scr_ch_ex = *cextp1;

#ifdef LDEBUG
/*        LOG(log, "display_char    : and scr_ch_ex = 0x%x\n", */
/*                        scr_ch_ex);                          */
#endif /* LDEBUG */

#endif /* EXTENDED */

	/* Now set the screen output style */

#ifdef EXTENDED
	(void) set_cse((unsigned char)(scr_char >> 12), scr_ch_ex, TRUE);
#else /* EXTENDED */
	(void) set_cse((unsigned char)(scr_char >> 12), TRUE);
#endif /* EXTENDED */

	/*
	 * Now we want to work out what character to print.
	 * This allows us to cater for the special cases where
	 * scr_char is not exactly printable.
	 */

	if (!((prtchar = scr_char) & GE_CHAR))
		prtchar = atop[(int) (prtchar & 0xff)];

	if ((prtchar & GE_UNDERSCORE) && tcslen[tcsUS])
	{

#ifdef EXTENDED

		/* Set this character to be underscore extended attribute */

		(void) set_cse(curscr->_style, (scr_ch_ex & EX_MASKHIGH) | EX_UNDERSCORE, TRUE);

#else /* EXTENDED */
		f_puttcs(tcsUS, FALSE);
#endif /* EXTENDED */

	}

	if (charset = (prtchar & GE_SET_MASK))
	{

#ifdef LDEBUG
	LOG(log, "display_char    : put special character 0x%x\n",
			prtchar);
#endif /* LDEBUG */

		if ((charset > GE_A_8) || !eight_bc)
		{
			int     charsind;
			char    char8;

			if (charset & GE_NBG)
			{
				char8 = ((charset & GE_G_7) ? NBG_notgot : NBG_illegal);
				charsind = 0;
			}
			else
			{
				char8 = (char)prtchar;
				charsind = (charset >> GE_SET_SHIFT) & 0x03;
			}
			f_puts(SO_string[charsind], FALSE);
			f_putc(char8);
			f_puts(SI_string[charsind], FALSE);
		}
		else
			f_putc( (char) prtchar);
	}
	else
		f_putc( (char) prtchar);

	if ((prtchar & GE_UNDERSCORE) && tcslen[tcsUS])
	{

#ifdef EXTENDED

		/* May need to reset the old attribute style */

		(void) set_extend(scr_ch_ex, TRUE);

#else /* EXTENDED */
		f_puttcs(tcsUE, FALSE);
#endif /* EXTENDED */

	}
}

/*
 * perform a mvcur, returning to NORMAL mode if necessary
 */

#ifdef STANDARD_C

void    domvcur(int oy, int ox, int ny, int nx)

#else /* STANDARD_C */

void    domvcur(oy, ox, ny, nx)

int     oy, ox, ny, nx;

#endif /* STANDARD_C */

{

#ifdef LDEBUG
	LOG(log, "domvcur         : (%d,%d) to (%d,%d) amwrap is %d\n",
		  oy+1, ox+1, ny+1, nx+1, amwrap);
#endif /* LDEBUG */

	/*
	 * Now we are at screen position (oy+1, ox+1)
	 * and we want to move to a new screen position (ny+1, nx+1).
	 * The current style (and extend) for characters being output
	 * is in the curscr structure. However, the character at (oy+1, ox+1)
	 * might not necessarily be in this current style (and extend): it
	 * might not even be printed at all (if it is invisible). Its
	 * characteristics can be found in the screen (and screen_ex) arrays.
	 *
	 * It is also possible to change to the new style after the move!
	 *
	 */

	if (!move_ok[curscr->_style])
	{

#ifdef LDEBUG
	LOG(log, "domvcur         : Return to plain mode before move\n");
#endif /* LDEBUG */

#ifdef EXTENDED
		(void) set_cse(NORMAL, 0, TRUE);
#else /* EXTENDED */
		(void) set_cse(NORMAL, TRUE);
#endif /* EXTENDED */

	}

	/* after writing last col on autowrap force a real move */
	if (amwrap)
	{
		amwrap = FALSE;
	}
	else
	{

		/*
		 * In some cases we can find "efficient" cursor move sequences.
		 * However, this can be dangerous for the case where the
		 * termcap sequence for "do" is line feed: some systems seem to
		 * add a carriage return after it when in CBREAK mode.
		 * For workstations (TTY not defined), therefore, we should not look for
		 * particularly short sequences
		 */

#ifdef TTY

		if (ny == oy)
		{
			/* move on same line */

			if (nx > ox)
			{
				if ((nx == (ox+1)) && tcslen[tcsND])
				{
					f_puttcs(tcsND, FALSE);
					return;
				}
				if (RT)
				{
					tputs(tgoto(RT, nx-ox, nx-ox), 1, f_putcs);
					return;
				}
			}
			if ((nx == (ox-1)) && tcslen[tcsLE])
			{
				f_puttcs(tcsLE, FALSE);
				return;
			}
		}
		if ((nx == ox)  && !scroll_line)
		{
			/* move on same column */
			if ((ny == (oy+1)) && tcslen[tcsDO])
			{
				f_puttcs(tcsDO, FALSE);
				return;
			}
			if ((ny == (oy-1)) && tcslen[tcsUP])
			{
				f_puttcs(tcsUP, FALSE);
				return;
			}
		}
		if ((nx == 0) && tcslen[tcsCR] && !scroll_line)
		{
			if ((ny == (oy+1)) && tcslen[tcsDO])
			{
				f_puttcs(tcsDO, FALSE);
				f_puttcs(tcsCR, FALSE);
				return;
			}
			if (ny == oy)
			{
				f_puttcs(tcsCR, FALSE);
				return;
			}
		}

#endif /* TTY */

	}
	tputs(tgoto(CM, nx, ny), 1, f_putcs);
}

/*
 * Check if two characters will be seen as different on the user display.
 *
 * On entry we expect chsp to point to the character already on the user
 * display and ccssp to point to its style and character set.
 * screeni will be an index of the (16-bit) character in the screen array
 * which we wish to have on the user display.
 * Assume also that variables "field"  and "style", local to this module,
 * indicate the field and corresponding style for the character position
 * currently under consideration, i.e. screen[screeni]. These variables
 * are liable to be updated if we are on a field definition position.
 *
 * A TRUE return says that the characters are different, whilst a FALSE
 * return says they are the same (in the sense that they will look the same
 * on the user's display).
 * A by-product of the checking is to set scrchar to be the 16-bit short
 * consisting of (from the most significant end):-
 *
 * 4 bits for the style
 * 4 bits for the character set and Graphics Escape indicators
 * 8 bits for the ASCII 8-bit character
 */

#ifdef EXTENDED

/*
 * In the case that we are implemented extended attributes then we
 * also need to check whether the attribute is different, plus whether
 * the difference will be visible on the screen!
 * Assume on entry that "cextp" will point to the extended attributes
 * currently on the user display.
 * The required attribute for the character is in screen_ex[screeni]
 * whilst that for the field is in f_extend.
 * The former, screen_ex[screeni], will be the character attribute of the
 * character to go to the user display. If it has been specifically set it will take
 * precedence over the latter. However, the actual attributes may be a
 * combination of the two, as evaluated by the get_extend function.
 * The latter, f_extend, may be altered if we are now on a field definition position.
 * Another byproduct of the checking is to set scrch_ex to be the byte
 * representing the latest value for the extended mode.
 */

#endif /* EXTENDED */

#ifdef STANDARD_C

bool    isdiff(void)

#else /* STANDARD_C */

bool    isdiff()

#endif /* STANDARD_C */

{
	register short          c;              /* character in screen array            */
	register short          ch2;            /* character to go onto display         */
	register unsigned char  chstyle;        /* style of display character           */

#ifdef EXTENDED
	register unsigned char  chextend;       /* extended characteristics on display  */
#endif /* EXTENDED */

	/*
	 * Given an index into screen, find what character is to go into
	 * the curses structure and onto the terminal.
	 * The most simple case is a terminal that is black and white, such that
	 * we are only interested in the highBIT (for standout mode).
	 * If the terminal can do blinking then we might also use the blinkingBIT.
	 * A colour display might also be interested in the protectedBIT.
	 */

	chstyle = style;

#ifdef EXTENDED
	chextend = get_extend();
#endif /* EXTENDED */

	if (c = screen[screeni])
	{
		/* check for a field definition character */

		if (c & FIELDSTART)
		{
			field = screeni;
			style = stylebits((int)c);
			chstyle = NORMAL;

#ifdef LDEBUG
	LOG(log, "isdiff          : new field at (%d,%d), style %s\n",
		       YX(screeni), stylename[style]);
#endif /* LDEBUG */

#ifdef EXTENDED
			f_extend = screen_ex[screeni];

#ifdef LDEBUG
	 if (chextend)
		LOG(log, "isdiff          : new extended field attribute 0x%x\n",
			       chextend);
#endif /* LDEBUG */

#endif /* EXTENDED */

			ch2 = ' ';
		}
		/* otherwise, tint c appropriately */
		/* all other chars, including blanks, come with style */

		else
		{
			/* Not a field definition character */

			ch2 = (chstyle == INVISIBLE ? (short)' ' : (c & 0x0fff));

#ifdef EXTENDED
			if (chstyle == INVISIBLE)
				chextend = 0;
#endif /* EXTENDED */

		}
	}
	else
	{
		/* A null character */

		ch2 = ' ';
	}

	scrchar = (((short)chstyle) << 12) | ch2;

#ifdef EXTENDED

	/*
	 * Now, the two nibbles in chextend may each be one of three cases.
	 * 0      Nothing specified for this character.
	 * 8      Request a setting according to the current field.
	 * 9-15   A specific request.
	 */

	 if ((chextend & EX_HSBITS) == EX_HIGHLIGHT)
		 chextend = (chextend & EX_MASKHIGH) | (f_extend & EX_HBITS) | EX_HIGHLIGHT;

	 if ((chextend & EX_CSBITS) == EX_COLOUR)
		 chextend = (chextend & EX_MASKCOLOUR) | (f_extend & EX_CBITS) | EX_COLOUR;

	scrch_ex = chextend;

#endif /* EXTENDED */

	/* Return TRUE if the character is different */

	if ((*chsp != (unsigned char)ch2) || ((*ccssp) & 0x0f) != (ch2 >> 8))
		return (TRUE);

	/*
	 * Even if the character is the same the result on the display may
	 * be different because of a different style which may have to be
	 * shown in a different manner.
	 * We will assume a difference if there is a style difference which
	 * requires escape sequences to change style.
	 */

	if (change_sl[(*ccssp) >> 4][chstyle])
		return (TRUE);

#ifdef EXTENDED

	/*
	 * In extended mode there may be a difference in presentation.
	 * This is true if the extended mode required differs from the
	 * mode currently on the display and if it actually makes a
	 * difference on the screen. For instance, a blank looks no
	 * different in different colours but may look different in
	 * extended highlighting.
	 */

	 if (*cextp != chextend)
	 {
		if ((ch2 != (short)' ') || ((*cextp & EX_HBITS) != (chextend & EX_HBITS)))
			return (TRUE);
	 }

#endif /* EXTENDED */

	/*
	 * The characters are also considered different if they are to
	 * be put as blank but cannot be done by clear to end of line
	 *
	 */

	if ((ch2 == (short)' ')       &&
	    (!ceol_ok[(*ccssp) >> 4]) &&
	    (!ceol_ok[chstyle]))
		return (TRUE);

	/* If we manage to get here then they must look the same on the screen! */

	return (FALSE);
}

/*
 * makeline(scr1,scr2) is called with scr1 and scr2 as pointers to a full or
 * partial line in screen: they are the first and last pointers.
 * It is assumed that the relevant field start to screen[scr1] is field
 * and the corresponding style is style: of course, scr1 can be the
 * field start itself.
 * However, the actual cursor position, current field start and current
 * style are in cursorpos, fieldstart and currentstyle.
 * If output is actually done (because something has changed) then
 * we update these variables.
 * Be careful to avoid filling in the very last screen character if on a
 * real terminal where the screen size matches exactly the 3270
 * screen size and wrap mode is on.
 * In fact, on some emulators this can be done because the cursor appears
 * to stay on the last character of a line, rather than causing a warp.
 * However, this raises extra complications if we then try a cursor movement
 * such as up, or a clear to end of line!
 */

#ifdef EXTENDED

/*
 * In the case of handling extended characteristics we need some extra
 * information about the extended characteristics.
 * It is assumed that the extended characteristics of the relevant field
 * are in f_extend, whilst those of the current characters, which
 * may be different due to some setting of screen attribute, are in
 * screen_ex[screen position].
 * However, the actual current extended characteristics information
 * is in currentextend. This is updated if actual output is done.
 */

#endif /* EXTENDED */

#ifdef STANDARD_C

void    makeline(int scr1, int scr2)

#else /* STANDARD_C */

void    makeline(scr1, scr2)

int     scr1,scr2;

#endif /* STANDARD_C */

{
	unsigned char           *chsp1, *ccssp1;

#ifdef EXTENDED
	unsigned char           *cextp1;
#endif /* EXTENDED */



	register int            lch;
	int                     lchlast;
	int                     i1;
	int                     xval1;
	unsigned char           orig_style;
	int                     puts_n;
	int                     savefield;
	unsigned char           savestyle;
	int                     prevfield;
	unsigned char           prevstyle;

#ifdef EXTENDED
	unsigned char           orig_extend;
	unsigned char           savef_extend;
	unsigned char           prevf_extend;
#endif /* EXTENDED */

	int                     charset;

	yval = scr1/linelen;
	xval = scr1%linelen;
	chsp = char_image[yval] + xval;
	ccssp = ccset_image[yval] + xval;

#ifdef EXTENDED
	cextp = ccext_image[yval] + xval;
#endif /* EXTENDED */

	lch = scr2%linelen;

#ifdef LDEBUG
	LOG(log, "makeline        : entry (%d,%d) to (%d,%d)\n",
	       YX(scr1),YX(scr2));
	LOG(log, "                : field (%d,%d) style %s\n",
	       YX(field),stylename[style]);

#ifdef EXTENDED
	LOG(log, "                : f_extend = 0x%x\n",
	       f_extend);
#endif /* EXTENDED */

#endif /* LDEBUG */

	/* skip to the first screen character which has changed */
	for (screeni = scr1 ; screeni <= scr2 ; screeni++)
	{
		bool    different;

		different = ((screen[screeni] != (short)*chsp) || (style != ((*ccssp) >> 4)));

#ifdef EXTENDED
		if (!different && (get_extend() != (int)(*cextp)))
			different = TRUE;
#endif /* EXTENDED */

		if (different && isdiff())
			break;

		chsp++;
		ccssp++;

#ifdef EXTENDED
		cextp++;
#endif /* EXTENDED */

	}
	if (screeni > scr2)
	{
		/* nothing to put onto the user screen */
		return;
	}

	/* note the current value of field and style */
	/* because isdiff() can reset field and style */

	savefield = field;
	savestyle = style;

#ifdef EXTENDED
	savef_extend = f_extend;
#endif /* EXTENDED */

	xval1 = (i1 = screeni)%linelen;
	chsp1 = chsp;
	ccssp1 = ccssp;

#ifdef LDEBUG
	LOG(log, "makeline        : changed (%d,%d): xval1 = %d\n",
		   YX(i1), xval1);
	LOG(log, "makeline        : '%s' %s",
		   print_ascii(((unsigned short)((*ccssp) & EX_HSBITS) << 8) | (short)*chsp), stylename[(*ccssp) >> 4]);
	LOG(log, " to '%s' %s\n",
		   print_ascii(scrchar), stylename[scrchar >> 12]);
#endif /* LDEBUG */

#ifdef EXTENDED
	cextp1 = cextp;

#ifdef LDEBUG
		LOG(log, "makeline        : changed extended 0x%x to 0x%x\n",
			   *cextp, get_extend());
#endif /* LDEBUG */

#endif /* EXTENDED */

	/*
	 * Now, we normally will have to move the cursor to the first
	 * changed character of this line, i.e. xval1.
	 * However, in cases where we have previously written the last
	 * character of the previous line (which has COLS columns, remember)
	 * and the terminal has autowrap set
	 * and we are updating from the start of the new line (i.e. scr1
	 * is the start of this line, so that xval is zero)
	 * then maybe we don't need to do this.
	 * Remember, though, that we cannot use an escape sequence
	 * without first moving the cursor.
	 * Of course, if xval1 is > zero we will need to copy the first
	 * few characters again, which is only worthwhile if the number
	 * of characters is not larger than a cursor move sequence.
	 * The length of such a move is got from movestrcnt.
	 */

	if ((amwrap) &&
	    (ly == yval) &&
	    (!xval) &&
	    (movestrcnt(yval, xval1) > xval1))
	{
		/* force output from start of line */
		copy_n = xval1;

#ifdef LDEBUG
	LOG(log, "makeline        : autowrap line: yval =%d, copy_n = %d\n",
		       yval, copy_n);
#endif /* LDEBUG */

		/* and force out at least one character */
		if (!copy_n) copy_n = 1;

#ifdef LDEBUG
	LOG(log, "makeline        : autowrap line: force copy_n = %d\n",
		       copy_n);
#endif /* LDEBUG */

		xval1 = 0;
		chsp = chsp1 = char_image[yval];
		ccssp = ccssp1 = ccset_image[yval];

#ifdef EXTENDED
		cextp = cextp1 = ccext_image[yval];
#endif /* EXTENDED */

		i1 = scr1;
		mvflag = FALSE;
	}
	else
	{
		copy_n = 0;
		mvflag = TRUE;
	}

	/*
	 * Now do a first scan of the remaining characters to see
	 * where is the last character which has changed, updating
	 * lch to be the index of this character in screen.
	 * Also set nlsp to be the index of the last character
	 * in the line after which we can use the CE sequence for clear
	 * to end of line: if CE is null then nlsp is so large as never
	 * to be exceeded (i.e. the line length).
	 * In other words, output the first lch+1 characters.
	 * Of course, this number of characters could end before,
	 * in or after the region which is (possibly) modified.
	 * Also, must be careful if the line includes field
	 * definitions (except for a field definition occurring
	 * as the last character of the line.
	 * This is actually an interesting case, because on exit the
	 * values of field and style have to be taken from it.
	 */

	xval = xval1;
	lchlast = lch;
	nlsp = xval1 - 1;

	for (screeni = i1 ; screeni <= scr2 ; screeni++)
	{
		bool    different;

		different = ((screen[screeni] != (short)*chsp) || (style != ((*ccssp) >> 4)));

#ifdef EXTENDED
		if (!different && (get_extend() != (int)(*cextp)))
			different = TRUE;
#endif /* EXTENDED */

		if (different && isdiff())
			lchlast = xval;

		different = (((scrchar & 0xff) != (unsigned short) ' ') ||
			     (!ceol_ok[scrchar >> 8]));

#ifdef EXTENDED
		if (!different && get_extend())
			different = TRUE;
#endif /* EXTENDED */

		if (different)
			nlsp = xval;

		chsp++;
		ccssp++;

#ifdef EXTENDED
		cextp++;
#endif /* EXTENDED */

		xval++;
	}
	lch = lchlast;

	/* maybe do not put out last character */
	if ((lch == (COLS - 1)) && (yval == (LINES - 1)) && AM & !VW)
		lch--;

	if (CE)
	{
		bool    different;

		/* continue scanning of what user is seeing */
		for ( ; xval < COLS ; xval++)
		{
			different =  ((*chsp != ' ') || (!ceol_ok[(*ccssp) >> 4]));

#ifdef EXTENDED
			if (*cextp != (unsigned char)0)
				different = TRUE;
#endif /* EXTENDED */

			if (different)
				nlsp = xval;
			chsp++;
			ccssp++;

#ifdef EXTENDED
			cextp++;
#endif /* EXTENDED */

			xval++;
		}
	}
	else
		nlsp = linelen;

	/* The scanning with isdiff() may have set the field and style
	 * variables to the last field definition in the line.
	 * if so then remember them, because they should be set on exit.
	 */
	prevfield = field;
	prevstyle = style;

#ifdef EXTENDED
	/* Remember also the extended characteristics information */

	prevf_extend = f_extend;
#endif /* EXTENDED */

	/* now reset field and style where we were */

	field = savefield;
	style = savestyle;

#ifdef EXTENDED
	/* Reset also the extended characteristics information */

	f_extend = savef_extend;
#endif /* EXTENDED */

	screeni = i1;
	xval = xval1;
	chsp = chsp1;
	ccssp = ccssp1;

#ifdef LDEBUG
	LOG(log, "makeline        : xval1 = %d, lch = %d, nlsp = %d\n",
		       xval1, lch, nlsp);
#endif /* LDEBUG */

#ifdef EXTENDED
	cextp = cextp1;

#ifdef LDEBUG
		LOG(log, "makeline        : extended characteristics now (0x%x) become 0x%x\n",
			   *cextp, get_extend());
#endif /* LDEBUG */

#endif /* EXTENDED */

	while (xval <= lch)
	{
		/* Each time we come here we have several variables set:-
		 *
		 * copy_n       No. of chars to copy even if unchanged
		 * mvflag       TRUE if we need to move the cursor
		 */

		if ((isdiff()) ||
		    (copy_n)                 ||
		    (xval > nlsp))
		{
			if (mvflag)
			{
				domvcur(ly, lx, yval, xval);
				mvflag = FALSE;
			}

#ifdef LDEBUG
	LOG(log, "makeline        : start copy at (%d,%d): xval = %d, copy_n = %d\n",
		       YX(screeni), xval, copy_n);
#endif /* LDEBUG */

			ly = yval;
			lx = xval;
			while (xval <= lch)
			{
				/* copy across at least copy_n characters */
				/* otherwise stop when the char is already on the display */

				if (xval > nlsp)
				{
					if (((lch - xval) > tcslen[tcsCE]) &&
					    (!copy_n))
					{

#ifdef LDEBUG
	LOG(log, "makeline        : using CE: xval = %d\n",xval );
#endif /* LDEBUG */

						f_puttcs(tcsCE, FALSE);
						lx = xval;
						(void) memset((void *)chsp, ' ' , (size_t)(COLS-xval));
					     /* chsp += (COLS-xval); */
						(void) memset((void *) ccssp, 0, (size_t)(COLS-xval));
					     /* ccssp += (COLS-xval); */

#ifdef EXTENDED
						(void) memset((void *) cextp, 0, (size_t)(COLS-xval));
					     /* cextp += (COLS-xval); */
#endif /* EXTENDED */

					}
					else
					{

#ifdef LDEBUG
	LOG(log, "makeline        : No need to use CE: xval = %d\n",xval );
#endif /* LDEBUG */

						if (copy_n <= (lch - xval))
							copy_n = lch - xval + 1;
					}
				}

				if ((!isdiff()) &&
				    (!copy_n))
				{

#ifdef LDEBUG
	LOG(log, "makeline        : same char '%s' %s at xval = %d, lx = %d\n",
			print_ascii((unsigned short)(((*ccssp) & EX_HSBITS) << 8) | (short)*chsp), stylename[(*ccssp) >> 4], xval, lx);
#endif /* LDEBUG */

					break;
				}

				*chsp++ = scrchar;
				*ccssp++ = (scrchar >> 8);

#ifdef EXTENDED
				*cextp++ = scrch_ex;
#endif /* EXTENDED */

				/* Now put the character onto the user's screen */

				display_char(yval, xval);

				xval++;
				amwrap = FALSE;
				screeni++;
				if (copy_n) --copy_n;
			}

#ifdef LDEBUG
	LOG(log, "makeline        : end of copying at xval = %d, lx = %d\n",
			xval, lx);
#endif /* LDEBUG */

			if (lx == xval) /* if no change */
				break;
			lx = xval;

			/*
			 * must deal with case where we are at the end of
			 * the line, and maybe we have not wrapped around.
			 * In fact, there is a frig if using a terminal
			 * emulator which "stays" on the last column.
			 */
			if (lx == COLS)
			{
				if (AM)
				{
					lx = 0;
					ly++;
					amwrap = TRUE;

#ifdef LDEBUG
	LOG(log, "makeline        : autowrap to ly = %d, lx = %d\n",
			ly , lx);
#endif /* LDEBUG */

				}
				else
				{
					lx--;

#ifdef LDEBUG
	LOG(log, "makeline        : no autowrap: reset  lx = %d\n", lx);
#endif /* LDEBUG */

				}
			}
		}
/*
 * This "else" part rewritten by JMG.
 * The current character in the line, at line index xval and screen position screeni,
 * is what is already on the screen. On this line lch gives the line index
 * of the last character.
 * Somewhere further on there may be a different character, to which we could
 * perhaps move directly with a cursor move sequence. However, it might be
 * less work to simply repeat the characters that are the same!
 * Include code to count how many bytes would be needed to output the characters
 * up to and including the next different character.
 * If this is less than the number of bytes which would be needed by a cursor
 * move then backspace xval over these characters and return to output them.
 * Need to take into account moves into and out of standout mode, as well
 * as underscores.
 * Use of SO/SI also counts if eight-bit character output is not allowed.
 */

		else if (xval < lch)
		{

#ifdef LDEBUG
	LOG(log, "makeline        ; could skip from xval = %d, lx = %d\n",
		       xval, lx);
#endif /* LDEBUG */

			orig_style = curscr->_style;

#ifdef EXTENDED
			orig_extend = curscr->_extend;
#endif /* EXTENDED */

			puts_n = 0;

#ifdef LDEBUG
	LOG(log, "makeline        : set orig_style = 0x%x, puts_n = %d\n",
				    orig_style, puts_n);
#endif /* LDEBUG */

			/* save values in case we decide to backtrack */
			savefield = field;
			savestyle = style;

#ifdef EXTENDED
			savef_extend = f_extend;
#endif /* EXTENDED */

			while (xval <= lch)
			{
				if (!mvflag)
				{
					unsigned short          prntchar;

#ifdef EXTENDED
					puts_n += set_cse(style, f_extend, FALSE);
#else /* EXTENDED */
					puts_n += set_cse(style, FALSE);
#endif /* EXTENDED */

					if (!((prntchar = scrchar) & GE_CHAR))
						prntchar = atop[(int) (prntchar & 0xff)];

					if (prntchar & GE_UNDERSCORE)
						puts_n += (tcslen[tcsUS] + tcslen[tcsUE]);

					if (charset = (prntchar & GE_SET_MASK))
					{
						if ((charset > GE_A_8) || !eight_bc)
						{
							int     charsind;

							if (charset & GE_NBG)
								charsind = 0;
							else
								charsind = (charset >> GE_SET_SHIFT)& 0x03;

							puts_n += (strlen(SO_string[charsind]) + strlen(SI_string[charsind]));
						}
					}

					++puts_n;

					if ((isdiff()) ||
					    (xval > nlsp))
					{
						/*
						 * Now we know that we have to output this character!
						 * We can now work out whether it would have been smarter to output
						 * all characters up to here, rather than jump directly to here.
						 * If so then we will backspace over these characters and reset the flag
						 * to make sure that they are output.
						 * Whether or not we do this we will break out of the loop.
						 */

#ifdef LDEBUG
	LOG(log, "makeline        : now change character at xval = %d, puts_n = %d\n",
			  xval, puts_n);
	LOG(log, "makeline        : '%s' %s",
		   print_ascii(((unsigned short)((*ccssp) & EX_HSBITS) << 8) | (short)*chsp), stylename[(*ccssp) >> 8]);
	LOG(log, " to '%s' %s\n",
		   print_ascii(scrchar), stylename[scrchar >> 12]);

#ifdef EXTENDED
	LOG(log, "makeline        : and change extended style from 0x%x to 0x%x\n",
			  *cextp, scrch_ex);
#endif /* EXTENDED */

#endif /* LDEBUG */

						/* now check against length of cursor move string */
						/* allowing for any necessary screen style change */

#ifdef EXTENDED
						/*
						 * Have to reset curscr to the original setting style
						 * and curscr->_extend to the original extended mode
						 * so that we can then subtract the string necessary to set the
						 * required mode for this character.
						 */

						if (!move_ok[orig_style])
						{
							curscr->_style = orig_style;
							curscr->_extend = orig_extend;
							puts_n -= set_cse(NORMAL, 0, FALSE);
						}
						puts_n -= set_cse(scrchar >> 12, scrch_ex, FALSE);
						curscr->_style = orig_style;
						curscr->_extend = orig_extend;
#else /* EXTENDED */
						/*
						 * Have to reset curscr to the original setting tyle
						 * so that we can then subtract the string necessary to set the
						 * required style for this character.
						 */

						if (!move_ok[orig_style])
						{
							curscr->_style = orig_style;
							puts_n -= set_cse(NORMAL, FALSE);
						}
						puts_n -= set_cse(scrchar >> 12, FALSE);
						curscr->_style = orig_style;
#endif /* EXTENDED */

						if (puts_n < movestrcnt(yval, xval))
						{
							/* better to copy than move cursor */
							xval -= copy_n;
							screeni -= copy_n;
							chsp -= copy_n;
							ccssp -= copy_n;
							field = savefield;
							style = savestyle;

#ifdef EXTENDED
							cextp -= copy_n;
							f_extend = savef_extend;
#endif /* EXTENDED */

#ifdef LDEBUG
	LOG(log, "makeline        : go back to xval = %d, copy_n = %d\n",
		       xval, copy_n);
#endif /* LDEBUG */

						}
						else
						{
							copy_n = 0;
							mvflag = TRUE;
						}

						break;
					}
					++copy_n;

				}
				else
				{
					if (isdiff())
						break;
				}
				screeni++;

				*chsp++ = scrchar;
				*ccssp++ = (scrchar >> 8);

#ifdef EXTENDED
				*cextp++ = scrch_ex;
#endif /* EXTENDED */

				xval++;
			}
		}
		else
			break;
	}
	field = prevfield;
	style = prevstyle;

#ifdef EXTENDED
	f_extend = prevf_extend;
#endif /* EXTENDED */

	return;
}

/*
 * Set the style to that required for characters to be put on the screen.
 * In case we are also using extended characteristics then we also need to
 * set these.
 * The final parameter is to know if we really want to do the setting
 * or if we only want to go through the motions merely to count the
 * number of output characters needed and return this as the result.
 * Note, however, that going through the motions INCLUDES setting the
 * value for curscr->_style or curscr->_extend, so it is recommended to
 * save and restore it!
 */

#ifdef EXTENDED

#ifdef STANDARD_C

int     set_cse(unsigned char new_style, unsigned char new_extend, bool do_set)

#else /* STANDARD_C */

int     set_cse(new_style, new_extend, do_set)

unsigned char   new_style;
unsigned char   new_extend;
	bool    do_set;

#endif /* STANDARD_C */

#else /* EXTENDED */

#ifdef STANDARD_C

int     set_cse(unsigned char new_style, bool do_set)

#else /* STANDARD_C */

int     set_cse(new_style, do_set)

unsigned char   new_style;
	bool    do_set;

#endif /* STANDARD_C */

#endif /* EXTENDED */

{
	int     result = 0;

#ifdef EXTENDED

	if ((curscr->_style == new_style) && (curscr->_extend == new_extend))
		return (0);
	/*
	 * Now we have to see what we want to do for both style and
	 * extended characteristics.
	 * The style that we want is in new_style.
	 * The current style is in curscr->_style.
	 * The extended characteristics that we want are in new_extend.
	 * The current extended characteristics are in curscr->_extend.
	 *
	 * The danger here is that a change of style, such as going from
	 * bold mode to normal mode, can destroy the setting of the
	 * extended highlighting. Equally, a change of extended highlighting
	 * can destroy the style. Thus, if we are doing either then we have
	 * to be rather careful.
	 * The style changes which can do this reset have the change_rh[from][to]
	 * boolean variable as TRUE.
	 * For the extended characteristics we assume that changing the colour
	 * is no problem, whilst changing the highlighting is only a problem if
	 * there is already a highlighting (the low-end three bits of the
	 * current extended characteristics).
	 * In case that either is true then at least one of the changes has to
	 * go in two stages.
	 */

#ifdef LDEBUG
	LOG(log, "set_cse         : at (%d,%d) change style from %s to %s, do_set is %d\n",
				    YX(screeni), stylename[curscr->_style], stylename[new_style], do_set);
	LOG(log, "set_cse         : and change extended mode from 0x%x to 0x%x, do_set is %d\n",
				    curscr->_extend, new_extend, do_set);
#endif /* LDEBUG */

	if ((change_rh[curscr->_style][new_style]) || ((curscr->_extend & EX_HBITS) != (new_extend & EX_HBITS)))
	{
		/* change screen style to NORMAL */

		if (do_set)
		{
			f_puts(change_cp[curscr->_style][NORMAL], FALSE);
		}
		else
		{
			result += change_sl[curscr->_style][NORMAL];
		}
		curscr->_style = NORMAL;

		/* change extended highlighting to none */

		result += set_extend(curscr->_extend & EX_CSBITS, do_set);

		/* then change extended characteristics to what we want */

		result += set_extend(new_extend, do_set);

		/* and finally change screen style to the new style */

		if (do_set)
		{
			f_puts(change_cp[NORMAL][new_style], FALSE);
		}
		else
		{
			result += change_sl[NORMAL][new_style];
		}
		curscr->_style = new_style;
	}
	else
	{
		if (change_sl[curscr->_style][new_style])
		{
			if (do_set)
			{
				f_puts(change_cp[curscr->_style][new_style], FALSE);
			}
			else
			{
				result += change_sl[curscr->_style][new_style];
			}
		}
		curscr->_style = new_style;
		result += set_extend(new_extend, do_set);
	}

#else /* EXTENDED */

	if (curscr->_style != new_style)
	{

#ifdef LDEBUG
	LOG(log, "set_cse         : at (%d,%d) change style from %s to %s, do_set is %d\n",
			YX(screeni), stylename[curscr->_style], stylename[new_style], do_set);
#endif /* LDEBUG */

		/*
		 * change screen mode to what the new_se wants
		 */

		if (do_set)
		{
			f_puts(change_cp[curscr->_style][new_style], FALSE);
		}
		else
		{
			result += change_sl[curscr->_style][new_style];
		}
		curscr->_style = new_style;
	}

#endif /* EXTENDED */

	return (result);
}

#ifdef EXTENDED

/*
 * Evaluate the extended characteristics for screen position screeni.
 * The value of screen_ex[] for this position may indicate a setting
 * done by a specific Set Attributes call.
 * If there is no setting then there may be a setting for that of the
 * valid field, done by a Start Field Extended or Modify Field.
 * There may also be an implicit one according to the field definition,
 * since we like to show different fields in different colours.
 * This is expected to be in f_extend.
 */


#ifdef STANDARD_C

unsigned char   get_extend(void)

#else /* STANDARD_C */

unsigned char   get_extend()

#endif /* STANDARD_C */

{
	unsigned char   result;

#ifdef LDEBUG
/*        LOG(log, "get_extend      : at (%d,%d) screen_ex[] = 0x%x, f_extend = 0x%x\n", */
/*                       YX(screeni), screen_ex[screeni], f_extend);                     */
#endif /* LDEBUG */

	result = screen_ex[screeni];
	if (FORMATTED)
	{
		if (!(result & EX_COLOUR))
		{
			if ((f_extend & EX_COLOUR) || (!(result & EX_CBITS)))
				result = (result & EX_MASKCOLOUR) | (f_extend & EX_CSBITS);
		}
		if (!(result & EX_HIGHLIGHT))
		{
			if ((f_extend & EX_HIGHLIGHT) || (!(result & EX_HBITS)))
				result = (result & EX_MASKHIGH) | (f_extend & EX_HSBITS);
		}
	}

#ifdef LDEBUG
/*        if (result != screen_ex[screeni])                                                        */
/*                LOG(log, "get_extend      : at (%d,%d) extended characteristics will be 0x%x\n", */
/*                               YX(screeni), result);                                             */
#endif /* LDEBUG */

	return (result);
}

/*
 * Set the extended characteristic requested by the IBM.
 * The first parameter is the requested value.
 * The current value is curscr->_extend.
 * The parameter may request a setting of highlight and/or colour.
 * In case either is not requested then we stay with the current value.
 * The second parameter is to know if we really want to do the setting
 * or if we only want to go through the motions merely to count the
 * number of output characters needed and return this as the result.
 * Note, however, that going through the motions INCLUDES setting the
 * value for curscr->_extend, so it is recommended to save and restore it!
 */


#ifdef STANDARD_C

int     set_extend(unsigned char new_extend, bool do_set)

#else /* STANDARD_C */

int     set_extend(new_extend, do_set)

unsigned char   new_extend;
	bool    do_set;

#endif /* STANDARD_C */

{
	/*
	 * Choose the highlight part.
	 * For the blinking we already have the changes.
	 * The reverse video Is not too difficult, although remember that
	 * turning it off uses the ME sequence, which may turn off other
	 * sorts of highlighting. Remember also that switching between
	 * normal and reverse video implies the need for a colour reset.
	 * The underscoring may be tricky, because to turn it off requires
	 * the US sequence, which might or might not turn off the other
	 * sorts of highlighting.
	 * The way to do things therefore seems that if we actually have to
	 * unset a highlighting mode then we assume that we might have
	 * unset other modes, so reset the current attributes.
	 */

	bool    reset_attrib = FALSE;
	bool    reverse_colour = FALSE;
	int     result = 0;

	/* First check on highlighting */

	if ((curscr->_extend & EX_HSBITS) != (new_extend & EX_HSBITS))
	{

		/* Something may change! */

#ifdef LDEBUG
	LOG(log, "set_extend      : current extend is 0x%x, new_extend is to be 0x%x, do_set is %d\n",
			curscr->_extend, new_extend, do_set);
#endif /* LDEBUG */

		if ((curscr->_extend & EX_HBITS) != (new_extend & EX_HBITS))
		{
			if (curscr->_extend & EX_HBITS)
			{
				if (do_set)
				{

#ifdef LDEBUG
	LOG(log, "set_extend      : unset old highlight with extend_unset[%d] (tcs set %d)\n",
			curscr->_extend & EX_HBITS, extend_unset[curscr->_extend & EX_HBITS]);

#endif /* LDEBUG */

					f_puttcs(extend_unset[curscr->_extend & EX_HBITS], FALSE);
				}
				else
				{
					result += tcslen[extend_unset[curscr->_extend & EX_HBITS]];
				}
			}
			reset_attrib = TRUE;
		}

		/* Is there a new highlighting? */

		if (new_extend & EX_HBITS)
		{
			if (do_set)
			{

#ifdef LDEBUG
	LOG(log, "set_extend      : set new highlight with extend_set[%d] (tcs set %d)\n",
			new_extend & EX_HBITS, extend_set[new_extend & EX_HBITS]);

#endif /* LDEBUG */

				f_puttcs(extend_set[new_extend & EX_HBITS], FALSE);
			}
			else
			{
				result += tcslen[extend_set[new_extend & EX_HBITS]];
			}
		}
		if ((new_extend & EX_REVERSE) != (curscr->_extend & EX_REVERSE))
			reverse_colour = TRUE;

		curscr->_extend = ((curscr->_extend & EX_MASKHIGH) | (new_extend & EX_HSBITS));
	}

	/*
	 * Choose the colour now: quite simple!
	 * The basic assumption is that we do not normally have to
	 * deselect an existing colour.
	 * The exception to this rule is if there was a colour but now
	 * there is no colour definition. In this case we assume that
	 * we want the default colour (C0).
	 * The actual setting sequence is different if we are in
	 * reverse video mode. We may have just moved to this mode or
	 * already be in it.
	 */

	if (((curscr->_extend & EX_CSBITS) != (new_extend & EX_CSBITS)) ||
	    (reverse_colour) || reset_attrib)
	{
		/* Colour has to be set! */

		int     new_colour;

#ifdef LDEBUG
	LOG(log, "set_extend      : current extend is 0x%x, new_extend will be 0x%x\n",
			curscr->_extend, new_extend);
#endif /* LDEBUG */


		new_colour = ((new_extend & EX_CBITS) >> 4);
		if (do_set)
		{

#ifdef LDEBUG
	LOG(log, "set_extend      : set new colour with tcs set %d\n",
			new_colour+ ((curscr->_extend & EX_REVERSE) ? tcsR0 : tcsC0));

#endif /* LDEBUG */

			f_puttcs(new_colour + ((curscr->_extend & EX_REVERSE) ? tcsR0 : tcsC0), FALSE);
		}
		else
		{
			result += tcslen[extend_set[new_colour]];
		}
		curscr->_extend = ((curscr->_extend & EX_MASKCOLOUR) |
				   (new_extend & EX_CSBITS));
	}

#ifdef LDEBUG
	LOG(log, "set_extend      : return %d with curscr->_extend as 0x%x\n",
			result, curscr->_extend);

#endif /* LDEBUG */
	return (result);

}

#endif /* EXTENDED */

/*
 * clear the curscr, possibly even going to the alternate screen
 */

#ifdef STANDARD_C

void    clscreen(void)

#else /* STANDARD_C */

void    clscreen()

#endif /* STANDARD_C */

{
	if (clearsn > 0)
	{
		/* have to clear screen first */
		/* may even have to swap to alternate screen */

		/* check if we currently have the wrong screen */
		if (altscreen != clearsn - 1)
		{

#ifdef DEBUG
	(void)fprintf(outf,"clscreen    : Flip to screen %d : COLS = %d, linelen = %d\n",
			clearsn-1 , COLS , linelen);
#endif /* DEBUG */

			/* flip to synchronous mode for important output */

			reset_tty(-SYNCH_NORMAL, TRUE);

#ifdef EXTENDED
			set_cse(NORMAL, 0, TRUE);
			f_extend = 0;
#else /* EXTENDED */
			set_cse(NORMAL, TRUE);
#endif /* EXTENDED */

			style = NORMAL;
			slsave();
			delwin();
			if (altscreen)
			{
				LINES = normal_LINES;
				COLS = normal_COLS;
				linelen = normal_linelen;
				rows = normal_rows;
			}
			else
			{
				LINES = altern_LINES;
				COLS = altern_COLS;
				linelen = altern_linelen;
				rows = altern_rows;
			}
			while ((curscr = newwin()) == ERR)
				(void)sleep(5);
			char_image = curscr->char_y;
			ccset_image = curscr->ccset_y;

			free((void *)screen);

#ifdef EXTENDED
			free((void *)screen_ex);
#endif /* EXTENDED */

			free((void *)s_point);
			screensize = linelen * rows;
			screen = (short *) alloc_buffer((int)(screensize * sizeof(short)), '\0');

#ifdef EXTENDED
			screen_ex = (unsigned char *) alloc_buffer((int)(screensize * sizeof(unsigned char)), '\0');
#endif /* EXTENDED */

			s_point = (short *) alloc_buffer((int)(screensize * sizeof(short)), '\0');
			if (altflip)
			{
				f_puttcs((altscreen ? tcsSN : tcsSW), TRUE);
			}
			clearnormal();

#ifdef DEBUG
	(void)fprintf(outf,"clscreen    : Flip to screen %d\n", clearsn-1);
#endif /* DEBUG */

/*
			reset_tty(SYNCH_NORMAL, TRUE);
			sizer();
			if ((COLS != (altscreen ? normal_COLS : altern_COLS)) ||
			    (LINES != (altscreen ? normal_LINES : altern_LINES)))
				end_program(222, "Somehow I cannot switch back to the other screen");
			}
 */

			/* Flip back to normal asynch mode now */

			setup_tty(FALSE);

			if (altscreen)
			{
				COLS = normal_COLS;
				LINES = normal_LINES;
				linelen = normal_linelen;
				rows = normal_rows;
				slinenum = normal_slinenum;
			}
			else
			{
				COLS = altern_COLS;
				LINES = altern_LINES;
				linelen = altern_linelen;
				rows = altern_rows;
				slinenum = altern_slinenum;
			}

			sline = (slinenum ? TRUE : HS);
			slrestore();

#ifdef DEBUG
	(void)fprintf(outf,"clscreen    : Flipped to screen %d : COLS = %d, linelen = %d\n",
			clearsn-1 , COLS , linelen);
#endif /* DEBUG */

			altscreen = clearsn - 1;
		}
		clearscreen();
		cursorpos = bufferpos = 0;
		clearsn = -clearsn;
		normal_chars = normal_blanks = TRUE;

#ifdef EXTENDED

		/* Do some resetting if in extended attributes */

		reply_highlight = FALSE;
		user_highlight = 0;
		reply_colour = FALSE;
		user_colour = 0;

#endif /* EXTENDED */

	}
}

/*
 * update curscr from screen[startpos] to screen[endpos]
 * note that since this can be called several times in a row, it does
 * not set the current cursor: this must be done in a subsequent call
 * of putcursor() or writecursor().
 */

#ifdef STANDARD_C

void    update_curscr(void)

#else /* STANDARD_C */

void    update_curscr()

#endif /* STANDARD_C */

{
	register int  i, j, y, lastpos;

	lastpos = endpos;
	DEC(lastpos);

#ifdef LDEBUG
	LOG(log,"update_curscr   : (%d,%d) to (%d,%d)\n",
		 YX(startpos), YX(lastpos));
	LOG(log,"                : entry style is %s, _style is %s\n",
		 stylename[style], stylename[curscr->_style]);

#ifdef EXTENDED
	LOG(log,"                : entry extend is 0x%x\n",
		 curscr->_extend);
#endif /* EXTENDED */

#endif /* LDEBUG */

	updateflag++;

	if (FORMATTED)
	{
		field = (startpos < firstfield ? screensize - 1 : startpos);
		while (!(screen[field] & FIELDSTART))
			if (--field < 0)
				field = screensize - 1;
		style = stylebits((int)screen[field]);

#ifdef EXTENDED
		f_extend = screen_ex[field];
#endif /* EXTENDED */

#ifdef LDEBUG
	LOG(log,"update_curscr   : found field start at (%d,%d), %s\n",
		YX(field), stylename[style]);
#ifdef EXTENDED
	LOG(log,"update_curscr   : with f_extend = 0x%x\n",
		f_extend);
#endif /* EXTENDED */

#endif /* LDEBUG */

	}
	else
	{
		field = -1;
		style = NORMAL;

#ifdef EXTENDED
		f_extend = 0;
#endif /* EXTENDED */

#ifdef LDEBUG
	LOG(log,"update_curscr   : unformatted screen \n");
#endif /* LDEBUG */

	}


	i = startpos;
	y = i/linelen;

	/* loop over all of the changed lines, calling makeline */
	do {
		j = (linelen * (y + 1)) - 1;
		if ((lastpos >= i) && (lastpos < j))
			j = lastpos;
		makeline(i,j);
		if (++y >= rows)
			y = 0;
		i = linelen * y;
	} while (j != lastpos);

#ifdef LDEBUG
	LOG(log,"update_curscr   : end with cursor at (%d,%d)\n",
		 ly+1, lx+1);
#endif /* LDEBUG */

}

/*
 * Write the cursor position to the user display.
 * We assume that the necessary variables are already set.
 */

#ifdef STANDARD_C

void    writecursor(void)

#else /* STANDARD_C */

void    writecursor()

#endif /* STANDARD_C */

{

	domvcur(ly, lx, Y, X);
	ly = Y;
	lx = X;
	f_flush();
	updateflag = 0;
}

/*
 * put the cursor where it belongs.
 * and write it out by a call of writecursor().
 */

#ifdef STANDARD_C

void    putcursor(void)

#else /* STANDARD_C */

void    putcursor()

#endif /* STANDARD_C */

{
	int i;

	X = cursorpos%linelen;
	Y = cursorpos/linelen;

#ifdef EXTENDED
	currentextend = screen_ex[cursorpos];
#endif /* EXTENDED */

	if (FORMATTED)
	{
		/* set the global variable fieldstart */

		for (i=cursorpos ; ; i = DECR(i))
		{
			if (screen[i] & FIELDSTART)
			{
				fieldstart = i;
				currentstyle = stylebits((int)screen[i]);

#ifdef EXTENDED
				if (!currentextend)
					currentextend = screen_ex[i];
#endif /* EXTENDED */

				break;
			}
		}
	}
	else
	{
		currentstyle = NORMAL;
	}

#ifdef LDEBUG
	LOG(log, "putcursor       : to (%d,%d), fieldstart to (%d,%d), %s\n",
		      YX(cursorpos), YX(fieldstart), stylename[currentstyle]);

#ifdef EXTENDED
	LOG(log, "putcursor       : extended field value is 0x%x\n",
		      currentextend);
#endif /* EXTENDED */

#endif /* LDEBUG */

	writecursor();
}

/* clear the screen into NORMAL mode */

#ifdef STANDARD_C

void    clearnormal(void)

#else /* STANDARD_C */

void    clearnormal()

#endif /* STANDARD_C */

{

#ifdef LDEBUG
	LOG(log, "clearnormal     : current flags is %s\n",
		      stylename[curscr->_style]);

#ifdef EXTENDED
	LOG(log, "clearnormal     : current extended flags is 0x%x\n",
		      curscr->_extend);
#endif /* EXTENDED */

#endif /* LDEBUG */

	f_clearop(-1);

	curscr->_style = NORMAL;
	X = Y = 0;
	lx = ly = 0;

#ifdef EXTENDED
	/* set the screen extended attributes to neutral */

	(void) set_cse(NORMAL, EX_C_MONO, TRUE);
#endif /* EXTENDED */

	winerase(curscr);         /* erase the contents of curscr */
}

/* clear the current screen (curscr), but not screen */

#ifdef STANDARD_C

void    clearcurscr(void)

#else /* STANDARD_C */

void    clearcurscr()

#endif /* STANDARD_C */

{
	clearnormal();
	if (slinenum)
	{
		slrefresh();
	}
}

/* reset the data structures to represent a cleared screen */

#ifdef STANDARD_C

void    clearscreen(void)

#else /* STANDARD_C */

void    clearscreen()

#endif /* STANDARD_C */

{
	clearcurscr();
	if (screen)
	{
		(void) memset((void *)screen, 0, (size_t)(screensize * sizeof(short)));

#ifdef EXTENDED
		(void) memset((void *)screen_ex, 0, (size_t)(screensize * sizeof(char)));
#endif /* EXTENDED */

	}
	firstfield = -1;
}

#ifdef SHOWFIELDS

#ifdef STANDARD_C

void    showfields(void)

#else /* STANDARD_C */

void    showfields()

#endif /* STANDARD_C */

{
	/* for debugging */
	int i;

#ifdef EXTENDED
	unsigned char   ext_attr = 0;
#endif /* EXTENDED */

#ifdef LDEBUG
	if (!FORMATTED)
	{
	LOG(log,"showfields      : no fields\n");
		return;
	}
	LOG(log,"showfield       : firstfield at %d,%d ...\n",YX(firstfield));
	for (i=0; i<screensize; i++)
		if (screen[i] & FIELDSTART)
		{
			LOG(log,"showfield       : (%2d,%2d] Field byte 0x%4x (%s%s%s%s%s)",
				YX(i), screen[i],
				screen[i]&SFprotectedBITS ? " PROT" : "",
				screen[i]&SFnumericBITS ? " NUM" : "",
				screen[i]&SFspecialBITS ? " bLiNk" : "",
				(screen[i]&SFvisibleBITS)==SFinvisibleBITS ? " INVISIBLE"
				    : (screen[i]&SFvisibleBITS)==SFstandoutBITS ? " BRIGHT"
				    : "",
				screen[i]&SFmodifiedBITS ? " MDT" : "");

#ifdef EXTENDED
			ext_attr = screen_ex[i];
			LOG(log," attribute 0x%x", screen_ex[i]);
#endif /* EXTENDED */

			LOG(log,"\n");
		}

#ifdef EXTENDED

		else
		{
			if (screen_ex[i] && (ext_attr != screen_ex[i]))
			{
				ext_attr = screen_ex[i];
				LOG(log,"showfield       : (%2d,%2d] New extended attribute 0x%x\n",
					YX(i), screen_ex[i]);
			}
		}

#endif /* EXTENDED */

#endif /* LDEBUG */

}
#endif /* SHOWFIELDS */

/* move the cursor "dist" positions (<0 means left).  Make sure
 * the globals "fieldstart" and "currentstyle" are up to date.
 * If doing extended fields also update "currentextend".
 * Note that nothing is yet written: when the cursor is
 * finally positioned then writecursor() is to be called.
 */

#ifdef STANDARD_C

void    movecursor(int dist)

#else /* STANDARD_C */

void    movecursor(dist)

#endif /* STANDARD_C */

{
	register int i;
	register int cp;

#ifdef LDEBUG
	LOG(log, "movecursor      : (%d) at (%d,%d): currentstyle = %d, curscr->_style = %d\n",
		       dist, YX(cursorpos), currentstyle, curscr->_style);
#endif /* LDEBUG */

#ifdef EXTENDED

#ifdef LDEBUG
	LOG(log, "movecursor      : currentextend = %d, curscr->_extend = %d\n",
		       currentextend, curscr->_extend);
#endif /* LDEBUG */

#endif /* EXTENDED */

	if ((cp = cursorpos + dist) < 0)
		cp += screensize;
	cp = cp % screensize;
	for (i = cp ; i != cursorpos ; i = DECR(i))
	{
		if (screen[i] & FIELDSTART)
		{
			fieldstart = i;
			currentstyle = stylebits((int)screen[i]);

#ifdef EXTENDED
			currentextend = screen_ex[i];
#endif /* EXTENDED */

			break;
		}
	}
	X = cp%linelen;
	Y = cp/linelen;
	cursorpos = cp;

#ifdef EXTENDED
	if (screen_ex[cursorpos])
		currentextend = screen_ex[cursorpos];
#endif /* EXTENDED */

}

/*
 *	This routine erases everything on the window.
 *
 * 1/27/81 (Berkeley) @(#)erase.c	1.2
 */

#ifdef STANDARD_C

void    winerase(struct curwin *win)

#else /* STANDARD_C */

void    winerase(win)

struct  curwin  *win;

#endif /* STANDARD_C */

{

	register int            y;

#ifdef DEBUG
	(void)fprintf(outf, "winerase    : WINERASE(%0.2o)\n", win);
#endif /* DEBUG */

	for (y = 0; y < win->_maxy; y++)
	{
		(void) memset((void *)(win->char_y[y]) , ' ' , (size_t)(win->_maxx));
		(void) memset((void *)(win->ccset_y[y]) , 0 , (size_t)(win->_maxx));

#ifdef EXTENDED
		(void) memset((void *)(win->ccext_y[y]) , 0 , (size_t)(win->_maxx));
#endif /* EXTENDED */

	}
	win->_curx = win->_cury = 0;
}
