/*
 *     Copyright CERN, Geneva 1989 - Copyright and any other
 *     appropriate legal protection of these computer programs
 *     and associated documentation reserved in all countries
 *     of the world.
 */

/*
 * 3270.c
 * Simulate an IBM 3278 terminal
 *
 * (With transparent graphics I/O mode by BS: March 1989).
 * (With major fix for tab/backtab--infinite loop if no unprotected fields).
 * (With small fix for VM/XA: minimal functionality for PROGRAM TAB command).
 * (Also small fix to implement EUA command (erase unprotected to address).
 */

/* Includes 7171 method insert mode as an execution-time option (JMG) */

#include "3270.h"
#include "globals.h"
#include "telnet.h"
#include "fcntl.h"

/*
 * The following actions can normally come in any order
 * EXCEPT the first few, for which we might want to know their position
 * so that we can force a call by placing the negated index into
 * the keyboard input.
 */



struct  action_entry    actions[] = {
	"enter",        act_enter,      0x7d,   0,      "enter",
	"tab",          act_tab,        0,      1,      "tab forward",
	"btab",         act_tab,        1,      1,      "tab backward",
	"clear",        act_pa,         0,      0,      "clear screen",
	"home",         act_home,       0,      1,      "move cursor to home",
	"nl",           act_newline,    0,      1,      "new line",
	"left",         act_move,       1,      1,      "cursor left",
	"down",         act_move,       2,      1,      "cursor down",
	"up",           act_move,       3,      1,      "cursor up",
	"right",        act_move,       4,      1,      "cursor right",

	"pfk1",         act_enter,      0xf1,   0,      "pf key 1",
	"pfk2",         act_enter,      0xf2,   0,      "pf key 2",
	"pfk3",         act_enter,      0xf3,   0,      "pf key 3",
	"pfk4",         act_enter,      0xf4,   0,      "pf key 4",
	"pfk5",         act_enter,      0xf5,   0,      "pf key 5",
	"pfk6",         act_enter,      0xf6,   0,      "pf key 6",
	"pfk7",         act_enter,      0xf7,   0,      "pf key 7",
	"pfk8",         act_enter,      0xf8,   0,      "pf key 8",
	"pfk9",         act_enter,      0xf9,   0,      "pf key 9",
	"pfk10",        act_enter,      0x7a,   0,      "pf key 10",
	"pfk11",        act_enter,      0x7b,   0,      "pf key 11",
	"pfk12",        act_enter,      0x7c,   0,      "pf key 12",
	"pfk13",        act_enter,      0xc1,   0,      "pf key 13",
	"pfk14",        act_enter,      0xc2,   0,      "pf key 14",
	"pfk15",        act_enter,      0xc3,   0,      "pf key 15",
	"pfk16",        act_enter,      0xc4,   0,      "pf key 16",
	"pfk17",        act_enter,      0xc5,   0,      "pf key 17",
	"pfk18",        act_enter,      0xc6,   0,      "pf key 18",
	"pfk19",        act_enter,      0xc7,   0,      "pf key 19",
	"pfk20",        act_enter,      0xc8,   0,      "pf key 20",
	"pfk21",        act_enter,      0xc9,   0,      "pf key 21",
	"pfk22",        act_enter,      0x4a,   0,      "pf key 22",
	"pfk23",        act_enter,      0x4b,   0,      "pf key 23",
	"pfk24",        act_enter,      0x4c,   0,      "pf key 24",

	"pa1",          act_pa,         1,      0,      "pa key 1",
	"pa2",          act_pa,         2,      0,      "pa key 2",
	"pa3",          act_pa,         3,      0,      "pa key 3",

	"insrt",        act_insert,     2,      1,      "toggle insert",
	"delete",       act_delete,     0,      1,      "delete character",
	"eeof",         act_erase,      0,      1,      "erase to end of field",

	"reset",        act_reset,      0,     -1,      "reset",
	"mreset",       act_mreset,     0,     -1,      "master reset",
	"reshow",       act_refresh,    0,     -1,      "refresh the screen",

/*      IMPLEMENTED BY BS..               */
	"escape",       act_escape,     0,     -1,      "escape to menu/quit",
	"erase",        act_erbsp,      0,      1,      "erase previous character (destructive backspace)",

/*      and by JMG                       */
	"help",         act_help,       0,     -1,      "get on-line help",
	"option",       act_option,     'o',   -1,      "on-line option choice",
	"keymap",       act_option,     'k',   -1,      "show key mappings",
	"move",         act_move,       0,      1,      "move the cursor to a new (x,y)",
	"extend",       act_ascii,      1,      1,      "switch to extended ASCII character set",
	"normal",       act_ascii,      0,      1,      "switch to normal ASCII character set",
	"lprt",         act_print,      0,     -1,      "save screen dump to file (to print)",
	"fm",           act_fm,         0,      1,      "set field mark",
	"dup",          act_dup,        0,      1,      "duplicate char",
	"eh_v",         act_highlight,  0,     -1,      "use variable (currently valid) highlighting for characters input",
	"eh_f",         act_highlight,  8,     -1,      "set fixed (field) highlighting for characters input",
	"eh_b",         act_highlight,  9,     -1,      "set blink highlighting for characters input",
	"eh_r",         act_highlight,  10,    -1,      "set reverse video highlighting for characters input",
	"eh_u",         act_highlight,  12,    -1,      "set underscore highlighting for characters input",
	"ec_v",         act_colour,     0,     -1,      "use variable (currently valid) colour for characters input",
	"ec_f",         act_colour,     8,     -1,      "set fixed (field) colour for characters input",
	"ec_b",         act_colour,     9,     -1,      "set blue colour for characters input",
	"ec_r",         act_colour,     10,    -1,      "set red colour for characters input",
	"ec_p",         act_colour,     11,    -1,      "set pink colour for characters input",
	"ec_g",         act_colour,     12,    -1,      "set green colour for characters input",
	"ec_t",         act_colour,     13,    -1,      "set turquoise colour for characters input",
	"ec_y",         act_colour,     14,    -1,      "set yellow colour for characters input",
	"ec_n",         act_colour,     15,    -1,      "set neutral (monochrome) for characters input",

/* undocumented debugging options */
	"debugon",      act_debug,      1,     -1,      "turn debug on",
	"debugoff",     act_debug,      0,     -1,      "turn debug off",
	"debug",        act_debug,      2,     -1,      "call debug",

/* misc unimplemented options (listed in tn3270 documentation) */
	"altk",         act_unimpl,     999,   -1,      "altk",
	"aplend",       act_unimpl,     999,   -1,      "aplend",
	"aploff",       act_unimpl,     999,   -1,      "aploff",
	"aplon",        act_unimpl,     999,   -1,      "aplon",
	"clrtab",       act_unimpl,     999,   -1,      "clrtab",
	"colbak",       act_unimpl,     999,   -1,      "colbak",
	"coltab",       act_unimpl,     999,   -1,      "coltab",
	"cursel",       act_unimpl,     999,   -1,      "cursel",
	"deltab",       act_unimpl,     999,   -1,      "deltab",
	"disc",         act_unimpl,     999,   -1,      "disc",
	"dp",           act_unimpl,     999,   -1,      "dp",
	"einp",         act_unimpl,     999,   -1,      "einp",
	"ferase",       act_unimpl,     999,   -1,      "ferase",
	"fieldend",     act_unimpl,     999,   -1,      "fieldend",
	"flinp",        act_unimpl,     999,   -1,      "flinp",
	"fm",           act_unimpl,     999,   -1,      "fm",
	"indent",       act_unimpl,     999,   -1,      "indent",
	"init",         act_unimpl,     999,   -1,      "init",
	"pcoff",        act_unimpl,     999,   -1,      "pcoff",
	"pcon",         act_unimpl,     999,   -1,      "pcon",
	"sethom",       act_unimpl,     999,   -1,      "sethom",
	"setmrg",       act_unimpl,     999,   -1,      "setmrg",
	"settab",       act_unimpl,     999,   -1,      "settab",
	"synch",        act_unimpl,     999,   -1,      "synch",
	"treq",         act_unimpl,     999,   -1,      "treq",
	"undent",       act_unimpl,     999,   -1,      "undent",
	"werase",       act_unimpl,     999,   -1,      "werase",
	"wordbacktab",  act_unimpl,     999,   -1,      "wordbacktab",
	"wordend",      act_unimpl,     999,   -1,      "wordend",
	"wordtab",      act_unimpl,     999,   -1,      "wordtab",
	"xoff",         act_unimpl,     999,   -1,      "xoff",
	"xon",          act_unimpl,     999,   -1,      "xon",

	(char *)0,      act_nil,        0,      0,      (char *)0
};

int     aid = NOattentionKEY; /* last value of the AID (attention indicator) sent */

int     prevSF = -1;        /* previous start field position */
short   prevsf;           /* previous start field value */
bool    infield = -1;      /* when in the middle of multiple SF */
int     screencount = 0;    /* count of consecutive chars to go on screen */

#ifdef EXTENDED

	/* Need to know what extended attributes we have sent to the IBM */

	unsigned char   sent_ex;

#endif /* EXTENDED */


/* Global debugging flag */

int     debug_flag = 0;

#ifdef DEBUG

/* Counters for input and output nulls */

int     nulls_in;
int     nulls_out;

#endif /* DEBUG */


/* translation table to convert from a six bit value to a printable character
 * (used in 12-bit cursor addresses)
 */
unsigned char SIXBITtoEBCDIC[] =
	{
	/* 00 */  0x40, 0xc1, 0xc2, 0xc3,
	/* 04 */  0xc4, 0xc5, 0xc6, 0xc7,
	/* 08 */  0xc8, 0xc9, 0x4a, 0x4b,
	/* 0c */  0x4c, 0x4d, 0x4e, 0x4f,

	/* 10 */  0x50, 0xd1, 0xd2, 0xd3,
	/* 14 */  0xd4, 0xd5, 0xd6, 0xd7,
	/* 18 */  0xd8, 0xd9, 0x5a, 0x5b,
	/* 1c */  0x5c, 0x5d, 0x5e, 0x5f,

	/* 20 */  0x60, 0x61, 0xe2, 0xe3,
	/* 24 */  0xe4, 0xe5, 0xe6, 0xe7,
	/* 28 */  0xe8, 0xe9, 0x6a, 0x6b,
	/* 2c */  0x6c, 0x6d, 0x6e, 0x6f,

	/* 30 */  0xf0, 0xf1, 0xf2, 0xf3,
	/* 34 */  0xf4, 0xf5, 0xf6, 0xf7,
	/* 38 */  0xf8, 0xf9, 0x7a, 0x7b,
	/* 3c */  0x7c, 0x7d, 0x7e, 0x7f
};

char    EBCDICcommand[] = {  0,  0,  0,  0,  0,  0,  0,  0,
			     0,  0,  0,  0,  0,  0, 14, 15,
			     0,  1,  2, 17,  0,  5,  6,  0,
			     0,  0,  0,  0,  0,  0, 13,  0
			  };

/*======== end of global declarations. */

/* translate from "de facto" EBCDIC to ASCII ISO 8859-1 */

unsigned char etoa[] =
{
0x00,0x01,0x02,0x03,0x9c,0x09,0x86,0x7f,0x97,0x8d,0x8e,0x0b,0x0c,0x0d,0x0e,0x0f,
0x10,0x11,0x12,0x13,0x9d,0x85,0x08,0x87,0x18,0x19,0x92,0x8f,0x1c,0x1d,0x1e,0x1f,
0x80,0x81,0x82,0x83,0x84,0x0a,0x17,0x1b,0x88,0x89,0x8a,0x8b,0x8c,0x05,0x06,0x07,
0x90,0x91,0x16,0x93,0x94,0x95,0x96,0x04,0x98,0x99,0x9a,0x9b,0x14,0x15,0x9e,0x1a,
0x20,0xa0,0xe2,0xe4,0xe0,0xe1,0xe3,0xe5,0xe7,0xf1,0xa2,0x2e,0x3c,0x28,0x2b,0x7c,
0x26,0xe9,0xea,0xeb,0xe8,0xed,0xee,0xef,0xec,0xdf,0x21,0x24,0x2a,0x29,0x3b,0x5e,
0x2d,0x2f,0xc2,0xc4,0xc0,0xc1,0xc3,0xc5,0xc7,0xd1,0xa6,0x2c,0x25,0x5f,0x3e,0x3f,
0xf8,0xc9,0xca,0xcb,0xc8,0xcd,0xce,0xcf,0xcc,0x60,0x3a,0x23,0x40,0x27,0x3d,0x22,
0xd8,0x61,0x62,0x63,0x64,0x65,0x66,0x67,0x68,0x69,0xab,0xbb,0xf0,0xfd,0xfe,0xb1,
0xb0,0x6a,0x6b,0x6c,0x6d,0x6e,0x6f,0x70,0x71,0x72,0xaa,0xba,0xe6,0xb8,0xc6,0xa4,
0xb5,0x7e,0x73,0x74,0x75,0x76,0x77,0x78,0x79,0x7a,0xa1,0xbf,0xd0,0x5b,0xde,0xae,
0xac,0xa3,0xa5,0xb7,0xa9,0xa7,0xb6,0xbc,0xbd,0xbe,0xdd,0xa8,0xaf,0x5d,0xb4,0xd7,
0x7b,0x41,0x42,0x43,0x44,0x45,0x46,0x47,0x48,0x49,0xad,0xf4,0xf6,0xf2,0xf3,0xf5,
0x7d,0x4a,0x4b,0x4c,0x4d,0x4e,0x4f,0x50,0x51,0x52,0xb9,0xfb,0xfc,0xf9,0xfa,0xff,
0x5c,0xf7,0x53,0x54,0x55,0x56,0x57,0x58,0x59,0x5a,0xb2,0xd4,0xd6,0xd2,0xd3,0xd5,
0x30,0x31,0x32,0x33,0x34,0x35,0x36,0x37,0x38,0x39,0xb3,0xdb,0xdc,0xd9,0xda,0x00
};

/* translate from ASCII ISO 8859-1 to "de facto" EBCDIC */

unsigned char atoe[] =
{
0x00,0x01,0x02,0x03,0x37,0x2d,0x2e,0x2f,0x16,0x05,0x25,0x0b,0x0c,0x0d,0x0e,0x0f,
0x10,0x11,0x12,0x13,0x3c,0x3d,0x32,0x26,0x18,0x19,0x3f,0x27,0x1c,0x1d,0x1e,0x1f,
0x40,0x5a,0x7f,0x7b,0x5b,0x6c,0x50,0x7d,0x4d,0x5d,0x5c,0x4e,0x6b,0x60,0x4b,0x61,
0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,0xf8,0xf9,0x7a,0x5e,0x4c,0x7e,0x6e,0x6f,
0x7c,0xc1,0xc2,0xc3,0xc4,0xc5,0xc6,0xc7,0xc8,0xc9,0xd1,0xd2,0xd3,0xd4,0xd5,0xd6,
0xd7,0xd8,0xd9,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe8,0xe9,0xad,0xe0,0xbd,0x5f,0x6d,
0x79,0x81,0x82,0x83,0x84,0x85,0x86,0x87,0x88,0x89,0x91,0x92,0x93,0x94,0x95,0x96,
0x97,0x98,0x99,0xa2,0xa3,0xa4,0xa5,0xa6,0xa7,0xa8,0xa9,0xc0,0x4f,0xd0,0xa1,0x07,
0x20,0x21,0x22,0x23,0x24,0x15,0x06,0x17,0x28,0x29,0x2a,0x2b,0x2c,0x09,0x0a,0x1b,
0x30,0x31,0x1a,0x33,0x34,0x35,0x36,0x08,0x38,0x39,0x3a,0x3b,0x04,0x14,0x3e,0xff,
0x41,0xaa,0x4a,0xb1,0x9f,0xb2,0x6a,0xb5,0xbb,0xb4,0x9a,0x8a,0xb0,0xca,0xaf,0xbc,
0x90,0x8f,0xea,0xfa,0xbe,0xa0,0xb6,0xb3,0x9d,0xda,0x9b,0x8b,0xb7,0xb8,0xb9,0xab,
0x64,0x65,0x62,0x66,0x63,0x67,0x9e,0x68,0x74,0x71,0x72,0x73,0x78,0x75,0x76,0x77,
0xac,0x69,0xed,0xee,0xeb,0xef,0xec,0xbf,0x80,0xfd,0xfe,0xfb,0xfc,0xba,0xae,0x59,
0x44,0x45,0x42,0x46,0x43,0x47,0x9c,0x48,0x54,0x51,0x52,0x53,0x58,0x55,0x56,0x57,
0x8c,0x49,0xcd,0xce,0xcb,0xcf,0xcc,0xe1,0x70,0xdd,0xde,0xdb,0xdc,0x8d,0x8e,0xdf
};

/*
 * Special translation for EBCDIC graphics characters, i.e. those which
 * come preceded by graphics escape.
 * The translation may assume that up to 4 character sets are available.
 * It therefore translates a single EBCDIC Graphic Character into a 16-bit
 * value, with the low order 7 bits as the (7-bit) character code.
 * The two bits above that effectively choose the set amongst ASCII 7-bit,
 * ASCII 8-bit, Graphics 7-bit and Graphics 8-bit. There is also a bit to
 * indicate that the character came from the IBM as Graphics Escape, plus
 * another to indicate if the character is to be put out in underscore mode.
 *
 * The values are the symbols, defined in the globals.h header, as follows:-
 *
 * GE_UNDERSCORE         0x400  Character to be shown in underscore mode
 * GE_A_7                0x000  Normal 7-bit ASCII
 * GE_A_8                0x080  Extended 8-bit ASCII
 * GE_G_7                0x100  Normal (DEC Special) graphics
 * GE_G_8                0x180  Extended (DEC Technical) graphics
 * GE_NBG                0x200  Illegal graphic character set
 * GE_NBG + GE_G_7       0x300  Unavailable character set (no bloody good!)
 *
 * The character sets should be in order of availability.
 * Remember that I normally assume that G0 is normal ASCII and G2 is
 * extended ASCII (DEC Latin-1 if on a VT300, otherwise DEC Supplemental
 * Graphics) and that these two sets are loaded into GL and GR respectively,
 * so that extended ASCII characters are also held as full eight-bit characters
 * with the top bit set (needed to make xterm work!).
 * The other two character sets which may be available are DEC Special
 * Graphics (even available on a VT100 but I don't know about xterm)
 * and DEC Technical Graphics (on a VT300).
 * I would normally expect a setup to put DEC Special Graphics into G1
 * and DEC Technical Graphics into G3.
 * The natural order is therefore G0, G1, G2 and G3.
 *
 * For codes which the IBM should NOT send, i.e. undefined codes, the IBM
 * convention is to display them as hyphens (ASCII code 0x2d). However,
 * in order to retain the reversibility of the mapping we put the identity
 * mapping of the 8-bit character plus the GE_NBG bit.
 *
 * More of a problem is the graphics characters for which we cannot find
 * any way of representing them, yet we need to have a unique conversion
 * so that we can reverse it if the IBM asks for the complete screen back.
 * To handle this we use the identity mapping, but with the GE_NBG bit to
 * show that the character as actually printed to the user screen should
 * be a special one (perhaps '?') plus the GE_G_7 bit to indicate that
 * it was at least a legal IBM graphics character.
 *
 * Even given all of the above the perceptive reader will see that the
 * translation tables below really ought to depend upon the capabilities of
 * the terminal in question. I have chosen to make them for a VT300, but
 * with the proviso that when characters can be found in different
 * character sets I always use the simplest ones first.
 */

unsigned short gtoa[] =
{
0x0200, 0x0201, 0x0202, 0x0203, 0x0204, 0x0205, 0x0206, 0x0207, /* 00-07 */
0x0208, 0x0209, 0x020a, 0x020b, 0x020c, 0x020d, 0x020e, 0x020f, /* 08-0f */
0x0210, 0x0211, 0x0212, 0x0213, 0x0214, 0x0215, 0x0216, 0x0217, /* 10-17 */
0x0218, 0x0219, 0x021a, 0x021b, 0x021c, 0x021d, 0x021e, 0x021f, /* 18-1f */
0x0220, 0x0221, 0x0222, 0x0223, 0x0224, 0x0225, 0x0226, 0x0227, /* 20-27 */
0x0228, 0x0229, 0x022a, 0x022b, 0x022c, 0x022d, 0x022e, 0x022f, /* 28-2f */
0x0230, 0x0231, 0x0232, 0x0233, 0x0234, 0x0235, 0x0236, 0x0237, /* 30-37 */
0x0238, 0x0239, 0x023a, 0x023b, 0x023c, 0x023d, 0x023e, 0x023f, /* 38-3f */
0x0240, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445, 0x0446, 0x0447, /* 40-47 */
0x0448, 0x0449, 0x024a, 0x024b, 0x024c, 0x024d, 0x024e, 0x024f, /* 48-4f */
0x0250, 0x044a, 0x044b, 0x044c, 0x044d, 0x044e, 0x044f, 0x0450, /* 50-57 */
0x0451, 0x0452, 0x025a, 0x025b, 0x025c, 0x025d, 0x025e, 0x025f, /* 58-5f */
0x0260, 0x0261, 0x0453, 0x0454, 0x0455, 0x0456, 0x0457, 0x0458, /* 60-67 */
0x0459, 0x045a, 0x026a, 0x026b, 0x026c, 0x026d, 0x026e, 0x026f, /* 68-6f */
0x0270, 0x01de, 0x00a8, 0x0273, 0x0274, 0x0275, 0x0276, 0x0277, /* 70-77 */
0x01df, 0x0279, 0x027a, 0x027b, 0x027c, 0x027d, 0x027e, 0x027f, /* 78-7f */
0x007e, 0x0381, 0x0382, 0x0383, 0x0384, 0x0385, 0x0286, 0x0287, /* 80-87 */
0x0288, 0x0289, 0x01fc, 0x01fe, 0x0179, 0x038d, 0x038e, 0x01fd, /* 88-8f */
0x0390, 0x0391, 0x0392, 0x0393, 0x0394, 0x0395, 0x0296, 0x0297, /* 90-97 */
0x0298, 0x0299, 0x01db, 0x01da, 0x039c, 0x039d, 0x00b1, 0x01fb, /* 98-9f */
0x03a0, 0x00b0, 0x0171, 0x00b7, 0x03a4, 0x02a5, 0x02a6, 0x02a7, /* a0-a7 */
0x02a8, 0x02a9, 0x01dc, 0x01dd, 0x03ac, 0x005b, 0x017a, 0x03af, /* a8-af */
0x01e1, 0x01e5, 0x01e9, 0x01f2, 0x01f7, 0x02b5, 0x01cb, 0x005c, /* b0-b7 */
0x00f7, 0x02b9, 0x01c5, 0x01c4, 0x03bc, 0x005d, 0x017c, 0x0178, /* b8-bf */
0x007b, 0x03c1, 0x03c2, 0x03c3, 0x016d, 0x016c, 0x0174, 0x0176, /* c0-c7 */
0x00a7, 0x02c9, 0x03ca, 0x03cb, 0x02cc, 0x03cd, 0x02ce, 0x03cf, /* c8-cf */
0x007d, 0x03d1, 0x03d2, 0x03d3, 0x016a, 0x016b, 0x0175, 0x0177, /* d0-d7 */
0x03d8, 0x02d9, 0x03da, 0x0021, 0x03dc, 0x03dd, 0x03de, 0x03df, /* d8-df */
0x02e0, 0x03e1, 0x03e2, 0x03e3, 0x02e4, 0x02e5, 0x02e6, 0x02e7, /* e0-e7 */
0x02e8, 0x02e9, 0x03ea, 0x03eb, 0x02ec, 0x03ed, 0x03ee, 0x03ef, /* e8-ef */
0x03f0, 0x03f1, 0x03f2, 0x03f3, 0x03f4, 0x03f5, 0x03f6, 0x03f7, /* f0-f7 */
0x03f8, 0x03f9, 0x02fa, 0x03fb, 0x03fc, 0x03fd, 0x03fe, 0x02ff  /* f8-ff */
};

/*
 * For translating characters back into EBCDIC when we know that they
 * are GE (Graphics Escape) characters we only need an eight-bit table
 * since we will precede the EBCDIC code by a GE byte. However, the
 * table can be indexed by a combination of the character and the
 * particular character set (see above remarks on character sets).
 *
 * Note that this is not be needed for characters which could not be
 * mapped from EBCDIC in the first place (i.e. when received
 * from the IBM), since this will have been the identity mapping
 * plus the GE_NBG bit plus perhaps the GE_G_7 bit.
 */

unsigned char atog[] =
{
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, /* 000-00f */
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, /* 010-01f */
0x00,0xdb,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, /* 020-02f */
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, /* 030-03f */
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x51,0x52,0x53,0x54,0x55,0x56, /* 040-04f */
0x57,0x58,0x59,0x62,0x63,0x64,0x65,0x66,0x67,0x68,0x69,0xad,0xb7,0xbd,0x00,0x00, /* 050-05f */
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, /* 060-06f */
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xc0,0x00,0xd0,0x80,0x00, /* 070-07f */
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, /* 080-08f */
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, /* 090-09f */
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xc8,0x72,0x00,0x00,0x00,0x00,0x00,0x00,0x00, /* 0a0-0af */
0xa1,0x9e,0x00,0x00,0x00,0x00,0x00,0xa3,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, /* 0b0-0bf */
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, /* 0c0-0cf */
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, /* 0d0-0df */
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, /* 0e0-0ef */
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xb8,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, /* 0f0-0ff */
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, /* 100-10f */
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, /* 110-11f */
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, /* 120-12f */
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, /* 130-13f */
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, /* 140-14f */
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, /* 150-15f */
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xd4,0xd5,0xc5,0xc4,0x00,0x00, /* 160-16f */
0x00,0xa2,0x00,0x00,0xc6,0xd6,0xc7,0xd7,0xbf,0x8c,0xae,0x00,0xbe,0x00,0x00,0x00, /* 170-17f */
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, /* 180-18f */
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, /* 190-19f */
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, /* 1a0-1af */
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, /* 1b0-1bf */
0x00,0x00,0x00,0x00,0xbb,0xba,0x00,0x00,0x00,0x00,0x00,0xb6,0x00,0x00,0x00,0x00, /* 1c0-1cf */
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x9b,0x9a,0xaa,0xab,0x71,0x78, /* 1d0-1df */
0x00,0xb0,0x00,0x00,0x00,0xb1,0x00,0x00,0x00,0xb2,0x00,0x00,0x00,0x00,0x00,0x00, /* 1e0-1ef */
0x00,0x00,0xb3,0x00,0x00,0x00,0x00,0xb4,0x00,0x00,0x00,0x9f,0x8a,0x8f,0x8b,0x00  /* 1f0-1ff */
};


/*
 * Even given all of the above it can happen that the EBCDIC to ASCII
 * conversion can give us a non-printable character. This can happen
 * by various methods, including file transfer of a file containing tabs.
 * It can also come with the EBCDIC characters such as Field Mark (0x1F),
 * which we want to put out as an underscored semicolon.
 * To handle this we need an extra table showing what to actually put on
 * the user's screen. This contains 16-bit values to allow us to put
 * exotic characters, including underscored ones, if we so desire.
 *
 * The default is currently an underscored exclamation mark (0x0421).
 * One non-default is intended for the EBCDIC Program Tab character,
 * which has EBCDIC value 0x05 and should then be converted to ASCII 0x09
 * which in turn is printed as an underscored semicolon.
 * However, some programs (including XEDIT) may decide to turn this EBCDIC
 * value into 0x1e before coming out of the IBM: this then seems
 * to be unchanged after going through the atoe[] conversion and so
 * remains as 0x1e, i.e. the ASCII RS control code. I therefore turn this
 * also into an underscored semicolon.
 *
 * It seems that all other EBCDIC values in the range from 0 to 0x3F are
 * converted into the value 0x7f, which in turn converts to the ASCII 0x22.
 * This is the double quote character. This actually makes the above a bit
 * redundant, in the sense that I could put in a specific check for RS.
 * However, the above method is much more safe if things change.
 *
 * Note that for printable values the mapping is the identity.
 *
 * Note that the Graphics to ASCII (gtoa) tables above may give 16-bit
 * values in which the lower byte is a non-printable value: these are
 * distinguishable because this only happens if the upper byte indicates
 * that the original graphics character was illegal or unprintable.
 */

unsigned short atop[] =
{
0x0420, 0x0421, 0x0421, 0x0421, 0x0421, 0x0421, 0x0421, 0x0421, /* 00-07 */
0x0421, 0x043b, 0x0421, 0x0421, 0x0421, 0x0421, 0x0421, 0x0421, /* 08-0f */
0x0421, 0x0421, 0x0421, 0x0421, 0x0421, 0x0421, 0x0421, 0x0421, /* 10-17 */
0x0421, 0x0421, 0x0421, 0x0421, 0x042a, 0x0421, 0x043b, 0x0421, /* 18-1f */
0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027, /* 20-27 */
0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f, /* 28-2f */
0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, /* 30-37 */
0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f, /* 38-3f */
0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, /* 40-47 */
0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, /* 48-4f */
0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, /* 50-57 */
0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f, /* 58-5f */
0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, /* 60-67 */
0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, /* 68-6f */
0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, /* 70-77 */
0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x007e, 0x0421, /* 78-7f */
0x0420, 0x0421, 0x0421, 0x0421, 0x0421, 0x0421, 0x0421, 0x0421, /* 80-87 */
0x0421, 0x0421, 0x0421, 0x0421, 0x0421, 0x0421, 0x0421, 0x0421, /* 88-8f */
0x0421, 0x0421, 0x0421, 0x0421, 0x0421, 0x0421, 0x0421, 0x0421, /* 90-97 */
0x0421, 0x0421, 0x0421, 0x0421, 0x0421, 0x0421, 0x0421, 0x0421, /* 98-9f */
0x00a0, 0x00a1, 0x00a2, 0x00a3, 0x00a4, 0x00a5, 0x00a6, 0x00a7, /* a0-a7 */
0x00a8, 0x00a9, 0x00aa, 0x00ab, 0x00ac, 0x00ad, 0x00ae, 0x00af, /* a8-af */
0x00b0, 0x00b1, 0x00b2, 0x00b3, 0x00b4, 0x00b5, 0x00b6, 0x00b7, /* b0-b7 */
0x00b8, 0x00b9, 0x00ba, 0x00bb, 0x00bc, 0x00bd, 0x00be, 0x00bf, /* b8-bf */
0x00c0, 0x00c1, 0x00c2, 0x00c3, 0x00c4, 0x00c5, 0x00c6, 0x00c7, /* c0-c7 */
0x00c8, 0x00c9, 0x00ca, 0x00cb, 0x00cc, 0x00cd, 0x00ce, 0x00cf, /* c8-cf */
0x00d0, 0x00d1, 0x00d2, 0x00d3, 0x00d4, 0x00d5, 0x00d6, 0x00d7, /* d0-d7 */
0x00d8, 0x00d9, 0x00da, 0x00db, 0x00dc, 0x00dd, 0x00de, 0x00df, /* d8-df */
0x00e0, 0x00e1, 0x00e2, 0x00e3, 0x00e4, 0x00e5, 0x00e6, 0x00e7, /* e0-e7 */
0x00e8, 0x00e9, 0x00ea, 0x00eb, 0x00ec, 0x00ed, 0x00ee, 0x00ef, /* e8-ef */
0x00f0, 0x00f1, 0x00f2, 0x00f3, 0x00f4, 0x00f5, 0x00f6, 0x00f7, /* f0-f7 */
0x00f8, 0x00f9, 0x00fa, 0x00fb, 0x00fc, 0x00fd, 0x00fe, 0x0421  /* f8-ff */
};

#ifdef EXTENDED

/*
 * Now lots of stuff for if we need to do all the extended data stream.
 * For starters we have to be able to deal with structured fields.
 * The first thing is to reply to a basic Read Partition structured
 * field query, to which we have to send back details of what we can do.
 */

/* Character array of initial answer : TEMPORARY only */

unsigned char   wsf_q_response[] = {
				       0x88,                                                   /* AID byte */
					0x00, 0x16, 0x81, 0x86,                                 /* colour */
					 0x00, 0x08,                                             /* 8 pairs */
					  0x00, 0xf4, 0xf1, 0xf1, 0xf2, 0xf2, 0xf3, 0xf3,
					  0xf4, 0xf4, 0xf5, 0xf5, 0xf6, 0xf6, 0xf7, 0xf7,
					0x00, 0x0d, 0x81, 0x87,                                 /* highlighting */
					 0x04,                                                   /* 4 pairs */
					  0x00, 0xf0, 0xf1, 0xf1, 0xf2, 0xf2, 0xf4, 0xf4,
					0x00, 0x1b, 0x81, 0x85,                                 /* char sets */
					 0x82, 0x00,                                             /* general */
					 0x07, 0x10,                                             /* (0x07, 0x10) = 12-point characters */
					 0x00, 0x00, 0x00, 0x00,                                 /* general */
					 0x07,                                                   /* 7-bytes per set */
					  0x00, 0x00, 0x00, 0x00, 0x65, 0x00, 0x25,               /* set 101 page 25 */
					  0x00, 0x00, 0x00, 0x02, 0xb9, 0x00, 0x25,               /* set 697 page 25 */
					0x00, 0x2e, 0x81, 0x81,                                 /* usable area */
					 0x03, 0x00, 0x00, 0x50, 0x00, 0x18, 0x00,               /* cell specs for 80*24 */
					 0x00, 0x01, 0x00, 0x48, 0x00, 0x01, 0x00, 0x48,         /* x,y normal and alt */
					 0x07, 0x10,                                             /* cell size (12-point) */
					 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,                     /* char size, X & Y min & max */
					 0x13, 0x02,                                             /* 2nd screen */
					 0x00, 0x01, 0x00, 0x50, 0x00, 0x18, 0x00,               /* cell specs for 80*24 */
					 0x00, 0x01, 0x00, 0x48, 0x00, 0x01, 0x00, 0x48,         /* x,y normal and alt */
					 0x07, 0x10,                                             /* cell size (12-point) */
					0x00, 0x1c, 0x81, 0xa6,                                 /* implicit partitions */
					 0x00, 0x00,                                             /* flags */
					 0x0b, 0x01, 0x00,                                       /* size */
					 0x00, 0x50, 0x00, 0x18,                                 /* 80*24 normal screen */
					 0x00, 0x50, 0x00, 0x18,                                 /* 80*24 alternate screen */
					 0x0b, 0x02, 0x00,                                       /* size */
					 0x00, 0x07, 0x00, 0x10,                                 /* 12-point normal screen */
					 0x00, 0x07, 0x00, 0x10,                                 /* 12-point alternate screen */
					0x00, 0x07, 0x81, 0x88,                                 /* reply modes */
					 0x00, 0x01, 0x02,                                       /* field, extended, char */
					0x00, 0x16, 0x81, 0x80,                                 /* summary */
					 0x80, 0x81, 0x84, 0x85, 0x86, 0x87, 0x88, 0xa1,         /* query code list */
					 0xa6, 0xa8, 0x96, 0x99, 0xb0, 0xb1, 0xb2, 0xb3,
					 0xb4, 0xb6,
					0x00, 0x08, 0x81, 0x84,                                 /* alphanumeric partitions */
					 0x00, 0x0a, 0x00, 0x04,
					0x00, 0x06, 0x81, 0x99,                                 /* auxiliary devices */
					 0x00, 0x00,
				       0x00, 0x00                                              /* end byte pair */
				   };

#define WSF_Q_LENGTH 1+22+13+27+46+28+7+22+8+6

/*
 * Need to know the number of rows and columns for the alternate screen.
 * (the normal screen is assumed always to be 24*80)
 * Remember that the code allows for a Model 2 to have a 24*132 alternate
 * screen if the symbol DENSE is defined, but this 24*132 must NOT be
 * indicated in the wsf responses.
 */

unsigned char   wsf_a_rows = 24;
unsigned char   wsf_a_cols = 80;

#define WSF_UACOL 1+22+13+27+32         /* Useable area columns index */
#define WSF_UAROW 1+22+13+27+34         /* Useable area rows index */

#define WSF_IPCOL 1+22+13+27+46+14      /* Implicit partitions columns index */
#define WSF_IPROW 1+22+13+27+46+16      /* Implicit partitions rows index */

/* Character strings for the extended highlighting and colour modes */

char            *mode_h[16] = { "varying highlight" , "illegal"   , "illegal" , "illegal" ,
				"illegal"           , "illegal"   , "illegal" , "illegal" ,
				"fixed highlight"   , "blinking"  , "reverse" , "illegal" ,
				"underscore"        , "illegal"   , "illegal" , "illegal"
			     };


char            *mode_c[16] = { "varying colour"    , "illegal"   , "illegal" , "illegal" ,
				"illegal"           , "illegal"   , "illegal" , "illegal" ,
				"fixed colour"      , "blue"      , "red"     , "pink"    ,
				"green"             , "turquoise" , "yellow"  , "neutral"
			     };

/* Character to send back for given highlighting and colours */

unsigned char   send_h[8] = { 0x00, 0xf1, 0xf2, 0x00, 0xf4, 0x00, 0x00, 0x00 };

unsigned char   send_c[8] = { 0x00, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7 };

/*
 * Array giving highlighting and colour required according to attribute.
 * Note, however, that the attribute can require bold style, which is not
 * controlled by any bit here. In other words, if the IBM has a field set
 * to be bold then this array will set it to red but will not set up the
 * switch to high intensity.
 */

unsigned char   ext_attrib[64] = { 0x00, 0x00, 0x01, 0x01, 0x00, 0x00, 0x01, 0x01,
				   0x20, 0x20, 0x21, 0x21, 0x00, 0x00, 0x01, 0x01,
				   0x00, 0x00, 0x01, 0x01, 0x00, 0x00, 0x01, 0x01,
				   0x20, 0x20, 0x21, 0x21, 0x00, 0x00, 0x01, 0x01,
				   0x10, 0x10, 0x11, 0x11, 0x10, 0x10, 0x11, 0x11,
				   0x70, 0x70, 0x71, 0x71, 0x00, 0x00, 0x01, 0x01,
				   0x10, 0x10, 0x11, 0x11, 0x10, 0x10, 0x11, 0x11,
				   0x70, 0x70, 0x71, 0x71, 0x00, 0x00, 0x01, 0x01
				 };

#endif /* EXTENDED */

unsigned char   PAkey[] =
{
	0x6d, /* CLEAR */
	0x6c, /* PA1 */
	0x6e, /* PA2 */
	0x6b  /* PA3 */
};

/*
 * Produce a print string for a 16-bit short which may contain an ASCII character
 * in the bottom half and zero on the top half.
 * If, however, the top half is non-zero then it may contain four bits to be ignored
 * (representing the screen style) and four representing a choice of character set.
 * For more details see the explanation of the screen array in the globals.h header.
 */

char    result_string[10];

char    *cc_low[] =
	{
	  "{NUL}","{SOH}","{STX}","{ETX}","{EOT}","{ENQ}","{ACK}","{BEL}",
	  "{BS}" ,"{HT}" ,"{LF}" ,"{VT}" ,"{FF}" ,"{CR}" ,"{SO}" ,"{SI}" ,
	  "{DLE}","{XON}","{DC2}","{XOF}","{DC4}","{NAK}","{SYN}","{ETB}",
	  "{CAN}","{EM}" ,"{SUB}","{ESC}","{FS}" ,"{GS}" ,"{RS}" ,"{US}"
	};

char    *cc_high[] =
	{
	  "{x80}","{x81}","{x82}","{x83}","{IND}","{NEL}","{SSA}","{ESA}",
	  "{HTS}","{HTJ}","{VTS}","{PLD}","{PLU}","{RI}" ,"{SS2}","{SS3}",
	  "{DCS}","{PU1}","{PU2}","{STS}","{CCH}","{MW}" ,"{SPA}","{EPA}",
	  "{x98}","{x99}","{x9A}","{CSI}","{ST}" ,"{OSC}","{PM}" ,"{APC}"
	};

char    *cc_bslash = "{\\}";

char    *cc_delete = "{DEL}";

char    *cc_FF = "{xFF}";

/* ----------------------- Now let's have some real code! ---------------- */


#ifdef STANDARD_C

char    *print_ascii(unsigned short char_16)

#else /* STANDARD_C */

char    *print_ascii(char_16)

unsigned short  char_16;

#endif /* STANDARD_C */

{
	short   char_12;
	unsigned char   ascii_7bit;

	if ((char_12 = (char_16 & 0xfff)) > 0xff)
	{
		sprintf(result_string, "{x%3.3x}", char_12);
		return (result_string);
	}
	if (isascii((int)char_12))
	{
		if (isprint((int)char_12))
		{
			if (char_12 == (short)'\\')
				return (cc_bslash);
			result_string[0] = (unsigned char)char_12;
			result_string[1] = '\0';
			return (result_string);
		}
		if (char_12 == 0x7f)
			return (cc_delete);
		return (cc_low[char_12]);
	}
	if (isprint(ascii_7bit = (char_12 & 0x7f)))
	{
		sprintf(result_string, "{x%2.2x}", char_12);
		return (result_string);
	}
	if (ascii_7bit == 0x7f)
		return (cc_FF);
	return (cc_high[ascii_7bit]);
}

/* go into full-screen mode */

#ifdef STANDARD_C

void    init3270(void)

#else /* STANDARD_C */

void    init3270()

#endif /* STANDARD_C */

{
	f_puttcs(tcsCL, FALSE);
	ly = 0;
	lx = 0;

#ifdef DEBUG
	(void)fprintf(outf, "init3270    : have cleared screen\n");
#endif /* DEBUG */

	sline = (slinenum ? TRUE : HS);

	slinit();

	/* Miscellaneous initialisation */
	insert = 0;
	if (!sline)
		f_puttcs(tcsIE, TRUE);

#ifdef DEBUG
	(void)fprintf(outf, "init3270    : OK\n");
#endif /* DEBUG */

#ifdef EXTENDED

	/* Set up the alternate screen for the WSF response */

	wsf_q_response[WSF_UACOL] = wsf_q_response[WSF_IPCOL] = wsf_a_cols;
	wsf_q_response[WSF_UAROW] = wsf_q_response[WSF_IPROW] = wsf_a_rows;

#endif /* EXTENDED */

}

/* Little routine to set new firstfield value */

#ifdef STANDARD_C

void    reset_ff(int minpos)

#else /* STANDARD_C */

void    reset_ff(minpos)

int     minpos;

#endif /* STANDARD_C */

{
	register int    new_ff = minpos;

#ifdef DEBUG
	int     old_ff = firstfield;
#endif /* DEBUG */

	for (firstfield = -1 ; new_ff < screensize ; new_ff++)
	{
		if (screen[new_ff] & FIELDSTART)
		{
			firstfield = new_ff;
			break;
		}
	}
	if (firstfield < 0)
		normal_chars = normal_blanks = TRUE;

#ifdef DEBUG
	if (firstfield < 0)
		(void)fprintf(outf, "reset_ff    : reset firstfield from (%d,%d) after (%d,%d) unformatted\n",
			YX(old_ff), YX(minpos));
	else
		(void)fprintf(outf, "reset_ff    : reset firstfield from (%d,%d) after (%d,%d) to (%d,%d)\n",
			YX(old_ff), YX(minpos), YX(firstfield));
#endif /* DEBUG */

}

/* Little routine to update the normal_chars and normal_blanks flags */

#ifdef STANDARD_C

void    update_normal(int i)

#else /* STANDARD_C */

void    update_normal(i)

int     i;

#endif /* STANDARD_C */

{
	unsigned char   field_style;

	field_style = stylebits((int)screen[i]);
	normal_blanks &= ceol_ok[field_style];

#ifdef EXTENDED
	normal_chars &= (!screen_ex[i]);
#else /* EXTENDED */
	normal_chars &= (!change_sl[NORMAL][field_style]);
#endif /* EXTENDED */
}

/* Now update the user screen */

#ifdef STANDARD_C

void    update_cs(void)

#else /* STANDARD_C */

void    update_cs()

#endif /* STANDARD_C */

{
	int             startp_save;
	int             last_start = -1;
	int             last_end = -1;
	int             endp_new;
	int             end_fstart;

#ifdef DEBUG
	unsigned char   end_style;

	(void)fprintf(outf,"update_cs   : (clearsn = %d) range list printout\n",
			    clearsn);
	startpos = s_pfirst;
	while (startpos >= 0)
	{
		if ((endpos = s_point[startpos]) < 0)
			endpos = startpos;
		(void)fprintf(outf,"update_cs   :  (%d,%d) to (%d,%d)\n",
			   YX(startpos), YX(endpos));
		startpos = s_point[endpos];
	}
#endif /* DEBUG */

	startp_save = s_pfirst;

	/* Check for an unformatted screen (should never happen!) */

	if (FORMATTED && !(screen[firstfield] & FIELDSTART))
	{
		/* Should NEVER happen !!!! */

#ifdef LOGFILE
		logit("Screen has become unformatted : firstfield was %d !!!",
			 firstfield);
		logscreen();
#endif /* LOGFILE */

#ifdef DEBUG
	(void)fprintf(outf, "update_cs   : Screen has become unformatted : firstfield was %d\n",
				firstfield);
#endif /* DEBUG */

#ifdef SHOWFIELDS
	showfields();
#endif /* SHOWFIELDS */

		reset_ff(0);
	}

	while (s_pfirst >= 0)
	{
		startpos = s_pfirst;
		if ((endpos = s_point[startpos]) < 0)
			endpos = startpos;
		else
		{
			/*
			 * On a formatted screen we may need to extend
			 * the range to the end of the field.
			 * We must always do this if the screen has not been erased.
			 * However, if the screen has been erased (clearsn != 0)
			 * we need not bother with trailing blanks unless they will
			 * not come out as plain.
			 */

			if (FORMATTED && !(screen[endpos] & FIELDSTART))
			{
				if ((startpos == last_start) &&
				    (endpos == last_end))
				{
					/* Should NEVER happen !!!! */

#ifdef LOGFILE
					logit("Looping in update_cs: %d!!!",
						 startpos);
					logscreen();
#endif /* LOGFILE */

					update_curscr();
					s_pfirst = s_point[endpos];
					continue;
				}

				last_start = startpos;
				last_end = endpos;

#ifdef DEBUG
	(void)fprintf(outf, "update_cs   : do (%d,%d) to (%d,%d)\n",
				       YX(startpos), YX(endpos));
#endif /* DEBUG */


				/* find the field start prior to endpos */

				end_fstart = endpos;
				while (!(screen[end_fstart] & FIELDSTART))
					DEC(end_fstart);

#ifdef DEBUG
	end_style = stylebits((int)screen[end_fstart]);
	(void)fprintf(outf, "update_cs   : end field is at (%d,%d), style 0x%x\n",
				       YX(end_fstart), end_style);
#endif /* DEBUG */

				/*
				 * Extend the end if necessary, i.e. if any field on the current
				 * user screen is in any non-normal mode (bold, reverse video etc.).
				 * Normally we need to extend it up to, but not including, the next
				 * field. However, if the blanks on the screen are all "normal" then
				 * we can omit trailing blanks/nulls.
				 */

				endp_new = endpos;
				if (!normal_chars)
				{
					while (endp_new != startp_save)
					{
						if (screen[endp_new])
						{
							if (screen[endp_new] & FIELDSTART)
								break;
							if (screen[endp_new] != ' ')
								endpos = endp_new;
						}
							INC(endp_new);
					}

#ifdef DEBUG
	(void)fprintf(outf, "update_cs   : last_end = (%d,%d) , endpos = (%d,%d) , endp_new = (%d,%d)\n",
				YX(last_end), YX(endpos), YX(endp_new));
#endif /* DEBUG */

					if (normal_blanks)
						INC(endpos);
					else
						endpos = endp_new;
				}

				if (endpos != last_end)
				{
					bufferpos = endpos;

#ifdef DEBUG
	(void)fprintf(outf, "update_cs   : merge in (%d,%d) to (%d,%d)\n",
				YX(startpos), YX(bufferpos));
#endif /* DEBUG */

					update_inf();
					if (endp_new != startp_save)
						continue;
					startpos = s_pfirst;
					if ((endpos = s_point[startpos]) < 0)
						endpos = startpos;
				}
			}
		}

		/*
		 * If we have to redraw the whole screen and we still
		 * have output in the pipeline then it is normally
		 * more efficient to redraw the whole screen after
		 * scrapping this output.
		 * We may also prefer to redraw the screen if we are getting
		 * stuff from the IBM faster than we can write it!
		 */

		if ((startpos == endpos) && op_wait)
		{

#ifdef DEBUG
	(void)fprintf(outf, "update_cs   : stacked output so redraw whole screen\n");
#endif /* DEBUG */

			redraw(cursorpos);
			s_pfirst = -1;
		}
		else
		{
			update_curscr();
			if (op_wait > screensize)
			{

#ifdef DEBUG
	(void)fprintf(outf, "update_cs   : output lagging so redraw whole screen\n");
#endif /* DEBUG */

				redraw(cursorpos);
				s_pfirst = -1;
			}
			else
				s_pfirst = s_point[endpos];
		}
	}

	/*
	 * Now reset the current screen mode correctly.
	 * curscr->_style says what style we are in.
	 * currentstyle is what we want.
	 * If doing extended characteristics then curscr->_extend
	 * will tell us what is the current extended mode, whilst
	 * currentextend is what we want.
	 */

#ifdef LDEBUG
	LOG(log, "update_cs       : reset screen mode if necessary\n");
	LOG(log, "update_cs       : _style is 0x%x, currentstyle is 0x%x\n",
				curscr->_style, currentstyle);

#ifdef EXTENDED
	LOG(log, "update_cs       : _extend is 0x%x, currentextend is 0x%x\n",
				curscr->_extend, currentextend);
#endif /* EXTENDED */

#endif /* LDEBUG */

#ifdef EXTENDED
	(void) set_cse(currentstyle, currentextend, TRUE);
#else /* EXTENDED */
	(void) set_cse(currentstyle, TRUE);
#endif /* EXTENDED */

	startpos = -1;
	endpos = -1;
	clearsn = 0;

	if (unlock)
	{
		putcursor();
	}
	else
	{

#ifdef TRANSP_IO
		if (trflag)
			f_flush();
		else
#endif /* TRANSP_IO */

		{
			Y = rows-1;
			X = linelen-1;
			writecursor();
		}
	}

	/* end of record: actually do it */

#ifdef DEBUG
	(void)fprintf(outf, "update_cs   : have now updated curscr from screen\n");
#endif /* DEBUG */

#ifdef SHOWFIELDS
	showfields();
#endif /* SHOWFIELDS */

}

/*
 * Interpret a specified section of the global array savebuf, which is
 * assumed to contain a set of orders and data belonging to a write
 * command (WRITE, ERASE/WRITE, ERASE/WRITE ALTERNATE or WRITE STRUCTURED
 * FIELD).
 * Note, however, that the structured field code is only included if the
 * program is compiled with variable EXTENDED defined.
 *
 * The first parameter is the particular command.
 * The other parameters are pointers to the start and to one beyond the end
 * of the buffer region to be treated.
 */

#ifdef STANDARD_C

void    int_orders(int command, unsigned char *startp, unsigned char *endp1p)

#else /* STANDARD_C */

void    int_orders(command, startp, endp1p)

	int     command;
unsigned char   *startp;
unsigned char   *endp1p;

#endif /* STANDARD_C */

{
	register unsigned       char    *p = startp;
	register                int     c, pos, count;
				int     bellflag, protflag;

#ifdef EXTENDED
	short           f_attrib;
#endif /* EXTENDED */

#ifdef DEBUG
	int i;
	int chcount = 0;

	nulls_in = 0;           /* for debugging output */
#endif /* DEBUG */


#ifdef DEBUG
	(void)fprintf(outf, "int_orders  : %d bytes, WCC byte 0x%x\n",
				endp1p - p, *p);
/*
	{
		int             icount;
		unsigned char   *poschar = p;

		for (icount = 0 ; icount < 3000 ; icount++)
		{
			(void)fprintf(outf, "%3x", *poschar);
			if (!(icount % 20))
				(void)fprintf(outf, "\n ");
			if (poschar++ == endp1p) break;
		}
	}
 */
#endif /* DEBUG */


	update_inf();

	/* Safety check against null buffer */

	if (p == endp1p)
		return;


	/* Interpret WCC byte (which contains things like
	 * unlock the keyboard and sound the bell).
	 */

#ifdef DEBUG
	(void)fprintf(outf, "int_orders  :\t[WCC %x: %s%s%s%s]\n",
	*p,
	*p&WCCreset ? " RESET" : "",
	*p&WCCprinter ? " PRINTER" : "",
	*p&WCCbell ? " BELL" : "",
	*p&WCCunlock ? " UNLOCK" : "",
	*p&WCCresetMDT ? " RESETMDT" : "");
#endif /* DEBUG */

#ifdef TTY
	/* Now check for transparent output (e.g. Kermit) */

	if ((*(p+1) == TRANSP_WRITE) && (command == WRITEcommand))
	{
		unlock = *p & WCCunlock;
		p += 2;

#ifdef DEBUG
	(void)fprintf(outf,"int_orders  : Transparent write of %d bytes: %2.2x %2.2x ... %2.2x\n",
			      endp1p-p, *p, *(p+1), *(endp1p-1));
#endif /* DEBUG */

		trflag = kermit = -256;

		reset_tty(-SYNCH_NORMAL, TRUE);
		f_addchars((char *)p, (int)(endp1p-p), TRUE);
		setup_tty(FALSE);

		return;
	}
#endif /* TTY */

	bellflag = *p & WCCbell;
	if (*p & WCCunlock)
	{
		aid = NOattentionKEY;
		if (!unlock)
		{
			unlock = TRUE;
			slputstr( " " , 9, 0);
			putcursor();
		}
	}

	if ((*p & WCCresetMDT) && (command == WRITEcommand))
		for (pos=0; pos<screensize; pos++)
			if (screen[pos] & FIELDSTART)
				screen[pos] &= ~SFmodifiedBITS;

	/*
	 * Now look at all the rest of the stuff from the IBM.
	 * This should consist of a mixture of orders and display data.
	 * In principle this should be only with WRITE, ERASEWRITE or
	 * ERASEWRITE ALTERNATE command data streams.
	 *
	 * A 7171 has also some special orders for extended functions.
	 */

	p++;
	startpos = -1;
	endpos = -1;


	while (p < endp1p)
	{
		if (c = *p++)
		{

#ifdef DEBUG
	if (nulls_in)
		(void)fprintf(outf,"{%d nulls}\n",nulls_in);
	nulls_in = 0;
#endif /* DEBUG */

		}

		switch(c)
		{

			case SFchar: /* start field */

#ifdef DEBUG
	if (chcount)
	{
		(void)fprintf(outf,"}\n");
		chcount = 0;
	}
#endif /* DEBUG */

#ifdef DEBUG
	(void)fprintf(outf, "int_orders  :\t[SF(%d,%d)]\n", YX(bufferpos));
#endif /* DEBUG */

				setup_field((short) *p);

				p++;
				INC(bufferpos);
				continue;

			case MFchar: /* Modify Field */

				/*
				 * This is only valid if already on a field definition.
				 * if so then it can fall through with the SFE setting.
				 */

				if (!FORMATTED || !(screen[bufferpos] & FIELDSTART))
				{

#ifdef DEBUG
	(void)fprintf(outf, "int_orders  :\t[%s(%d,%d) not OK for Modify Field\n",
					  YX(bufferpos));
#endif /* DEBUG */

					p += 2*(*p) + 1;

					continue;
				}

			case SFEchar: /* Start Field Extended */

#ifdef DEBUG
	if (chcount)
	{
		(void)fprintf(outf,"}\n");
		chcount = 0;
	}
#endif /* DEBUG */

				count = *p;

#ifdef DEBUG
	(void)fprintf(outf, "int_orders  :\t[%s(%d,%d) (%d", (c==SFEchar ? "SFE" : "MF"),
					  YX(bufferpos), count);
	for (i = 0 ; i < 2*count ; i++)
	{
		(void)fprintf(outf,",0x%02x",*(p + 1 + i));
	}
	(void)fprintf(outf, ")\n");
#endif /* DEBUG */

#ifdef EXTENDED

				if (c == SFEchar)
					f_attrib = 0;

#ifdef DEBUG
	(void)fprintf(outf, "int_orders  :\t%d pairs to treat!\n",count);
#endif /* DEBUG */

				if (count)
				{
					for ( count = 0 ; count < *p ; count++)
					{
						/* loop through the pairs */

#ifdef DEBUG
	(void)fprintf(outf, "int_orders  :\t[%s(0x%02x,0x%02x)]\n",
		 (c==SFEchar ? "SFE" : "MF"), *(p + 1 + 2*count), *(p + 2 + 2*count));
#endif /* DEBUG */

						switch (*(p + 1 + 2*count))
						{
							case EX_T_F_ATTRIB:
							f_attrib = (*(p + 2 + 2*count));
								break;

							case EX_T_F_VALID:
							case EX_T_F_OUTLINE:
								break;

							case EX_T_HIGHLIGHT:
								screen_ex[bufferpos] = (screen_ex[bufferpos] & EX_MASKHIGH) |
										       (*(p+2+2*count) & EX_HSBITS) | EX_HIGHLIGHT;
								break;

							case EX_T_COLOUR:
								screen_ex[bufferpos] = (screen_ex[bufferpos] & EX_MASKCOLOUR) |
										       ((*(p+2+2*count) & EX_HSBITS) << 4) | EX_COLOUR;
								break;

							default:
								break;
						}
					}
				}
				else
				{
					/* count zero implies reset all attribute types */

					screen_ex[bufferpos] = EX_HIGHLIGHT | EX_COLOUR;
				}

				if (c == SFEchar)
					setup_field(f_attrib);
				else
					current_ex = screen_ex[bufferpos];

#endif /* EXTENDED */

				p += 2*count + 1;
				INC(bufferpos);

				continue;

			case SBAchar: /* set buffer address */

#ifdef DEBUG
	if (chcount)
	{
		(void)fprintf(outf,"}\n");
		chcount = 0;
	}
#endif /* DEBUG */

				update_inf();

				if ((bufferpos = decodepos(p)) >= screensize)
					end_program(101, "IBM trying to send oversize screen: Emergency Abort", 0);

#ifdef DEBUG
	(void)fprintf(outf, "int_orders  :\t[SBA(%d,%d)]\n", YX(bufferpos));
#endif /* DEBUG */

				p += 2;
				continue;

			case EUAchar: /* Erase Unprotected to Address */

#ifdef DEBUG
	if (chcount)
	{
		(void)fprintf(outf,"}\n");
		chcount = 0;
	}
#endif /* DEBUG */

			if ((pos = decodepos(p)) >= screensize)
				end_program(102, "IBM trying to send oversize screen: Emergency Abort", 0);

#ifdef DEBUG
	(void)fprintf(outf, "int_orders  :\t[EUA(%d,%d)]\n", YX(pos));
#endif /* DEBUG */

				/* now bufferpos is the starting location and pos is the
				 * ending location for the erase unprotected.
				 * The idea is to store nulls in all unprotected
				 * character positions from bufferpos to pos-1
				 */

				if (startpos < 0)
					startpos = bufferpos;
				protflag = FALSE;
				if (pos <= bufferpos)
				{
					screencount += (screensize - bufferpos);
					while(bufferpos < screensize)
					{
						if (screen[bufferpos] & FIELDSTART)
						{
							protflag = (screen[bufferpos] & SFprotectedBITS);
						}
						else
						{
							if (!protflag)
							{
								screen[bufferpos] = 0;

#ifdef EXTENDED
								screen_ex[bufferpos] = 0;
#endif /* EXTENDED */

							}
						}
						bufferpos++;
					}
					bufferpos = 0;
				}
				screencount += (pos - bufferpos);
				while (bufferpos < pos)
				{
					if (screen[bufferpos] & FIELDSTART)
					{
						protflag = (screen[bufferpos] & SFprotectedBITS);
					}
					else
					{
						if (!protflag)
						{
							screen[bufferpos] = 0;

#ifdef EXTENDED
							screen_ex[bufferpos] = 0;
#endif /* EXTENDED */

						}
					}
					bufferpos++;;
				}
				p += 2;
				continue;

			case SAchar: /* Set Attribute */

#ifdef DEBUG
	if (chcount)
	{
		(void)fprintf(outf,"}\n");
		chcount = 0;
	}
	(void)fprintf(outf, "int_orders  :\t[SA(0x%02x,0x%02x) at (%d,%d)]\n",
					*p,*(p+1), YX(bufferpos));
#endif /* DEBUG */

#ifdef EXTENDED

				switch (*p)
				{
					case EX_T_ALL_CHAR:
						current_ex = EX_HIGHLIGHT | EX_COLOUR;
						break;

					case EX_T_HIGHLIGHT:
						current_ex = (current_ex & EX_MASKHIGH) |
							     (*(p+1) & EX_HSBITS) | EX_HIGHLIGHT;
						break;
					case EX_T_COLOUR:
						current_ex = (current_ex & EX_MASKCOLOUR) |
							     ((*(p+1) & EX_HSBITS) << 4) | EX_COLOUR;
						break;
					default:
						break;
				}

#ifdef DEBUG
	(void)fprintf(outf, "int_orders  : Reset current extended attribute byte to 0x%x\n",
				current_ex);
#endif /* DEBUG */

#else /* EXTENDED */

#endif /* EXTENDED */

				p += 2;
				continue;

			case ICchar: /* insert cursor */

#ifdef DEBUG
	if (chcount)
	{
		(void)fprintf(outf,"}\n");
		chcount = 0;
	}
#endif /* DEBUG */

				if (startpos < 0)
					startpos = bufferpos;
				cursorpos = bufferpos;
				/* set endpos to note IC */
				endpos = bufferpos;

#ifdef DEBUG
	(void)fprintf(outf, "int_orders  :\t[IC(%d,%d)]\n", YX(cursorpos));
#endif /* DEBUG */

				continue;

			case PTchar: /* Program Tab */

#ifdef DEBUG
	if (chcount)
	{
		(void)fprintf(outf,"}\n");
		chcount = 0;
	}
	(void)fprintf(outf, "int_orders  :\t[PT]\n");
#endif /* DEBUG */

				while ((screen[bufferpos] & (short)(FIELDSTART + SFprotectedBITS)) != (short)FIELDSTART)
				{
					if (bufferpos == (screensize - 1))
						break;
					INC(bufferpos);
					screencount++;
				}
				INC(bufferpos);
				screencount++;
				continue;

			case RAchar: /* repeat to address */

#ifdef DEBUG
	if (chcount)
	{
		(void)fprintf(outf,"}\n");
		chcount = 0;
	}
#endif /* DEBUG */

				if ((pos = decodepos(p)) >= screensize)
					end_program(103, "IBM trying to send oversize screen: Emergency Abort", 0);
				/* now bufferpos is the starting location and pos is the
				    * ending location for the repeating field.
				    */
				if (startpos < 0)
					startpos = bufferpos;
				c = etoa[p[2]];

#ifdef DEBUG
	(void)fprintf(outf, "int_orders  :\t[RA('\\%o':%d,%d", c, YX(bufferpos));
#endif /* DEBUG */

				if (pos <= bufferpos)
				{
					screencount += (screensize - bufferpos);
					while(bufferpos < screensize)
				{
					screen[bufferpos] = c;

#ifdef EXTENDED
					screen_ex[bufferpos] = current_ex;
#endif /* EXTENDED */

					bufferpos++;

				}
					bufferpos = 0;
				}
				screencount += (pos - bufferpos);
				while (bufferpos < pos)
				{
					screen[bufferpos] = c;

#ifdef EXTENDED
					screen_ex[bufferpos] = current_ex;
#endif /* EXTENDED */

					bufferpos++;

				}

#ifdef DEBUG
	(void)fprintf(outf, "->%d,%d)]\n", YX(bufferpos));
#endif /* DEBUG */

				p += 3;

				/* Maybe we have overwritten all the formatted fields */

				if (FORMATTED && (screen[firstfield] == c))
					reset_ff(pos < bufferpos ? bufferpos : pos);
				continue;

			case GEchar: /* Graphics Escape. */

#ifdef DEBUG
	if (chcount)
	{
		(void)fprintf(outf,"}\n");
		chcount = 0;
	}
	(void)fprintf(outf, "int_orders  :\t[GE(0x%02x)]\n", *p);
#endif /* DEBUG */

				screen[bufferpos] =  gtoa[*p++] | GE_CHAR;

				/* Maybe we have overwritten the firstfield position */

				if ((firstfield == bufferpos) && FORMATTED )
					reset_ff(bufferpos);

				if (startpos < 0)
					startpos = bufferpos;
				INC(bufferpos);
				screencount++;
				continue;

			case EXT7171_1: /* first byte of 7171 extended functions */

#ifdef DEBUG
	(void)fprintf(outf, "int_orders  :Perhaps 7171 extended function: Next byte 0x%02x\n", *p);
#endif /* DEBUG */

				/*
				 * Check the next character also.
				 * If not the second part of 7171 extended functions
				 * then fall through to default.
				 */

				if (*p == EXT7171_1)
				{
					/*
					 * OK, it looks like one of the special 7171 functions.
					 * Switch on the ones which we can handle, ignore the rest.
					 * Remember the one case where it is a double byte function!
					 */

					 p += 1;

#ifdef DEBUG
	(void)fprintf(outf, "int_orders  : 7171 extended function: Next byte 0x%02x\n", *p);
#endif /* DEBUG */

					 switch (*p++)
					 {
						case E_ON_NULL:

#ifdef DEBUG
	(void)fprintf(outf, "int_orders  : 7171 Improved Null Processing\n");
#endif /* DEBUG */

							emul7171 = 1;
							break;

						case E_OFF_NULL:

#ifdef DEBUG
	(void)fprintf(outf, "int_orders  : 7171 Back to 3270 Null Processing\n");
#endif /* DEBUG */

							emul7171 = 0;
							break;

						case E_DROP_INS:

#ifdef DEBUG
	(void)fprintf(outf, "int_orders  : 7171 3278 Insert Mode (cancel across attention)\n");
#endif /* DEBUG */

							keep_insert = 0;
							break;

						case E_KEEP_INS:

#ifdef DEBUG
	(void)fprintf(outf, "int_orders  : 7171 3277 Insert Mode (keep across attention)\n");
#endif /* DEBUG */

							keep_insert = 1;
							break;

						case E_SET_MARK:

#ifdef DEBUG
	(void)fprintf(outf, "int_orders  : 7171 set mark 0x%02x at (%d, %d)\n", *p, YX(bufferpos));
#endif /* DEBUG */

							p++;
							break;

						default:

#ifdef DEBUG
	(void)fprintf(outf, "int_orders  : 7171 unimplemented extended function 0x%02x\n", *(p-1));
#endif /* DEBUG */

						break;
					}
					continue;
				}

			default:

#ifdef DEBUG
	if (c)
	{
		if (!chcount)
		{
			if (nulls_in)
			{
				(void)fprintf(outf,"{%d nulls}\n",nulls_in);
				nulls_in = 0;
			}
			(void)fprintf(outf, "{");
		}
		if ((chcount) && ((chcount % 80) == 0))
		{
			(void)fprintf(outf, "\n ");

		}
		(void)fprintf(outf, "%s", print_ascii((unsigned short)etoa[c]));
		chcount++;
	}
	else
	{
		if (chcount)
		{
			(void)fprintf(outf, "}");
			chcount = 0;
		}
		nulls_in++;
	}
#endif /* DEBUG */

				screen[bufferpos] = etoa[c];

#ifdef EXTENDED
				screen_ex[bufferpos] = current_ex;
#endif /* EXTENDED */


				/* Maybe we have overwritten the firstfield position */

				if ((firstfield == bufferpos) && FORMATTED )
					reset_ff(bufferpos);

				if (startpos < 0)
					startpos = bufferpos;
				INC(bufferpos);
				screencount++;
		}
	}

#ifdef DEBUG
	if (nulls_in)
		(void)fprintf(outf,"{%d nulls}\n",nulls_in);
	nulls_in = 0;
	if (chcount)
	{
		(void)fprintf(outf,"}\n");
		chcount = 0;
	}
#endif /* DEBUG */

	update_inf();

#ifdef TRANSP_IO

	/*
	 * If we get here then what came from the IBM was not graphics.
	 * However, we might have been in transparent mode already.
	 * We will assume that transparent mode has ended if we have received
	 * something which has cleared the screen and unlocked input.
	 */

	if (trflag && (clearsn < 0) && unlock)
	{

#ifdef DEBUG
	(void)fprintf(outf, "int_orders  : IBM sent unlock: end graphics with cursorpos at (%d,%d)\n",
				YX(cursorpos));
#endif /* DEBUG */

		trflag = 0;

#ifdef GRAPHICS
		graphics = 0;
#endif /* GRAPHICS */

#ifdef TTY
		kermit = 0;
#endif /* TTY */

		setup_tty(FALSE);
		slrestore();
		act_refresh(1);

		/* We should also clear out any hanging input */

		clear_kbin();

/* Transparent mode end                                     ..(BS) */

	}

#endif /* TRANSP_IO */

	if (bellflag)
	{
		DING
	}


}

/*
 * Interpret write commands in the global array savebuf.
 * Do as a routine because they may also come inside structured field
 * commands (allowed if compiled with the EXTENDED symbol defined).
 *
 * The first parameter is the particular command.
 * The other parameters are pointers to the start and to one beyond the end
 * of the buffer region to be treated.
 */

#ifdef STANDARD_C

void    int_write(int command, unsigned char *startp, unsigned char *endp1p)

#else /* STANDARD_C */

void    int_write(command, startp, endp1p)

	int     command;
unsigned char   *startp;
unsigned char   *endp1p;

#endif /* STANDARD_C */

{
	register unsigned       char    *p = startp;

#ifdef DEBUG
	(void)fprintf(outf, "int_write   : %d bytes, WCC byte 0x%x\n",
				endp1p - p, *p);
/*
	{
		int             icount;
		unsigned char   *poschar = p;

		for (icount = 0 ; icount < 3000 ; icount++)
		{
			(void)fprintf(outf, "%3x", *poschar);
			if (!(icount % 20))
				(void)fprintf(outf, "\n ");
			if (poschar++ == endp1p) break;
		}
	}
 */
#endif /* DEBUG */

#ifdef EXTENDED
			/* Reset extended attributes to zero */

			current_ex = 0;
#endif /* EXTENDED */

	switch (command)
	{
		case ERASEWRITEcommand:

#ifdef DEBUG
	(void)fprintf(outf,"int_write   :\tERASEWRITE: altscreen is %d\n", altscreen);
#endif /* DEBUG */
			/* Clear any stuff not sent to user screen */
			f_clearop(-1);
			s_pfirst = -1;
			startpos = -1;
			endpos = -1;
			clearsn = 1;
			int_orders(command, p, endp1p);

			break;


		case ERASEWRITEALTERNATEcommand:

#ifdef DEBUG
	(void)fprintf(outf,"int_write   :\tERASEWRITE ALTERNATE: altscreen is %d\n", altscreen);
#endif /* DEBUG */

			/* Clear any stuff not sent to user screen */
			f_clearop(-1);
			s_pfirst = -1;
			startpos = -1;
			endpos = -1;
			clearsn = 2;
			int_orders(command, p, endp1p);

			break;


		case WRITEcommand:

#ifdef DEBUG
	(void)fprintf(outf,"int_write   :\tWRITE\n");
#endif /* DEBUG */

			/* This might be transparent (Kermit) : see later */

			bufferpos = cursorpos;

			int_orders(command, p, endp1p);

			break;

		case ERASEALLUNPROTECTEDcommand:

			movecursor(-cursorpos);
			if (FORMATTED)
			{
				int     field_save = -1;
				int     scpos;

				while (next_field(FALSE, SFprotectedBITS))
				{
					if (fieldstart == field_save)
						break;
					if (field_save < 0)
						field_save = fieldstart;
					screen[fieldstart] &= ~SFmodifiedBITS;
					for (scpos = cursorpos; !(screen[scpos] & FIELDSTART); )
					{
						screen[scpos] = '\0';

#ifdef EXTENDED
						screen_ex[scpos] = '\0';
#endif /* EXTENDED */

						INC(scpos);
					}
				}
			}

			unlock = TRUE;
			aid = NOattentionKEY;

			/* Don't think this can have a WCC byte and data! */

			update_inf();

			break;

		default:
			break;

	}

	return;
}

/*
 * Interpret the contents of the global array savebuf as a 3270 command,
 * where the variable savep points to one beyond the end of the buffer.
 * This assumes the whole 3270 command, orders and data is in the buffer.
 * Note that the processing of the orders and data is done in the routine
 * int_orders
 * which not only simplifies and reduces the size of interpret
 * but also allows for cases where structured field commands include
 * internal sets of orders.
 */

#ifdef STANDARD_C

void    interpret(void)

#else /* STANDARD_C */

void    interpret()

#endif /* STANDARD_C */

{
	register unsigned       char    *p = savebuf;
				int     command;

#ifdef DEBUG
				int     pos;
#endif /* DEBUG */

#ifdef EXTENDED
	int             wsf_length;
	int             wsf_l_item;
	int             wsf_counter;
#endif /* EXTENDED */

#ifdef DEBUG
	(void)fprintf(outf, "interpret   : %d bytes, command %d\n",
				savep - savebuf, *p);
/*
	{
		int             icount;
		unsigned char   *poschar = savebuf;

		for (icount = 0 ; icount < 3000 ; icount++)
		{
			(void)fprintf(outf, "%3x", *poschar);
			if (!(icount % 20))
				(void)fprintf(outf, "\n ");
			if (poschar++ == savep) break;
		}
	}
 */
#endif /* DEBUG */

	updateflag = 0;

	command = *p++;

#ifdef GRAPHICS
	/*
	 * At CERN we can get graphics commands from the IBM.
	 * These come as WRITE or ERASEWRITE command, followed by :-
	 *
	 *   c3         Reset, unlock, reset MDT
	 *   11,5d,7f   SBA (24,80)
	 *   11         SBA
	 *   00         Indicates transparent graphics (it cannot
	 *              be a proper SBA, which would be an EBCDIC byte).
	 *   xx         Mode indicator - zero for output, non-zero for input.
	 *
	 * The problem is, however, to recognise when we exit from
	 * transparent graphics: I assume that we will get an
	 * ERASEWRITE accompanied by an unlock of the keyboard.
	 */

	if ((command == WRITEcommand) || (command == ERASEWRITEcommand))
	{
		if ((*p == 0xc3)        &&
		    (*(p+1) == SBAchar) &&
		    (*(p+2) ==  0x5D)   &&
		    (*(p+3) == 0x7F)    &&
		    (*(p+4) == SBAchar) &&
		    (*(p+5) == 0))
		{
			int     count;

			/* log it: looks like graphics! */

#ifdef DEBUG
	(void)fprintf(outf, "interpret   : ASCII:");
	for (pos=0; pos<(savep-savebuf-1); pos++)
	{
		if (!(pos % 20))
			(void)fprintf(outf, "\n");
		(void)fprintf(outf, "-%2.2x", *(p+pos));
	}
	(void)fprintf(outf, "\n");
#endif /* DEBUG */

			/*
			 * If the start of graphics then we want to go to synchronous mode.
			 * First, though, save the status line and clear out any hanging
			 * asynchronous (3270 screen) output.
			 * While the user is in graphics mode we will want to go into
			 * synchronous mode by calling reset_tty(-timer, TRUE).
			 * When it finishes then we will return him to asynchronous mode
			 * and redraw the 3270 screen.
			 */

			if (!trflag)
			{
				/* Provisionally set so that no real output */

				trflag = 1;

				slsave();

				/* Force out the current normal screen if necessary */
				update_cs();

				/* Only call update_inf if necessary!!!! */
				if (s_pfirst >= 0)
					update_inf();

				f_clearop(-1);

				/* For transparent I/O we need to initialise asynchronous */

				reset_tty(-SYNCH_TRANSP, TRUE);
			}

			if (!(trflag = *(p+6)))
				trflag = -1;
			graphics = trflag;

#ifdef DEBUG
	(void)fprintf(outf, "\ninterpret   :\t[TRANSP. INTRODUCER] %s\n",
		    graphics < 0 ? "(OUTPUT)" : "(INPUT)");
#endif /* DEBUG */

			count = savep - savebuf - 1;
			p += 7;
			count -= 7;

			if (graphics < 0)
			{       /* No Input requested, ask for more.. */
				unsigned char ptrgr[100], *pptrgr;

				pptrgr  = ptrgr;   /* Build 3270 input block header.. */
				*pptrgr++ = 0xe8;  /* transparent aid byte */
				*pptrgr++ = byte1(cursorpos);
				*pptrgr++ = byte2(cursorpos);

				/* Just send a <CR> for now, it seems to be enough.....  */

				*pptrgr++ = '\r' | 0x80; /* Added Bit7 */

				    /* Add 3270 input block trailer..       */
				*pptrgr++ = TELCMD_IAC;
				*pptrgr++ = TELCMD_EOR;
				    /* ..and send off the block..           */
				writnet(net, ptrgr, pptrgr-ptrgr);

				/* Lock out keyboard input for now */
				unlock = FALSE;

#ifdef TONET
				if (write(tonet, (char *)ptrgr, pptrgr-ptrgr) < 0)
					;
#endif /* TONET */

#ifdef DEBUG
	(void)fprintf(outf, "interpret   : Input  %d transp. bytes: %2.2x ... %2.2x\n",
	       (pptrgr-ptrgr-5), *(ptrgr+3), *(pptrgr-3));
#endif /* DEBUG */

			}
			else
				unlock = TRUE;

			if (graphics && (count > 0))
			{       /* Output requested..*/

#ifdef DEBUG
	(void)fprintf(outf, "interpret   : Output %d transp. bytes: %2.2x ... %2.2x\n",
	     count, *p, *(p+count-1));
#endif /* DEBUG */

				reset_tty(-SYNCH_TRANSP, TRUE);
				f_addchars((char *)p, count, TRUE);
				p += count;
			}

			/*
			 * Now reset the flag to asynchronous to allow "normal" 3270 screen
			 * output to come in (but it will be thrown away in the output
			 * routine f_output because we are in transparent mode)
			 */

			setup_tty(FALSE);

			return;
		}
	}
#endif /* GRAPHICS */

	if (command >= 0x60)
		command = EBCDICcommand[(command & 0x7f) - 0x60];

	switch (command)
	{

		/*
		 * Write commands get treated in a separate command.
		 * This is because they may be included in structured field commands.
		 */

		case ERASEWRITEcommand:
		case ERASEWRITEALTERNATEcommand:
		case WRITEcommand:
		case ERASEALLUNPROTECTEDcommand:

			int_write(command, p, savep);

			break;

		case READBUFFERcommand:

#ifdef DEBUG
	(void)fprintf(outf,"interpret   :\tREAD BUFFER\n");
#endif /* DEBUG */

			/* Force out the current normal screen if necessary */

			/* There is a special CERN test for the case that the
			 * IBM is trying to read the size of the alternate screen.
			 * If we are currently on the normal screen (24*80)
			 * and the alternate screen is 24*132 (CERN special)
			 * then keep to the 24*80 but set up as if we were on
			 * the alternate screen.
			 */

			if ((clearsn == 2) && (startpos < 0) && (altern_rows == 24) && !altscreen)
			{
				altscreen = 1;
			}

			update_inf();
			update_cs();
			act_enter(-1);
			return;


		case READMODIFIEDcommand:

#ifdef DEBUG
	(void)fprintf(outf,"interpret   :\tREAD MODIFIED BUFFER\n");
#endif /* DEBUG */

			/* Force out the current normal screen if necessary */
			update_inf();
			update_cs();
			act_enter(aid);
			return;


		case WSFcommand:

#ifdef EXTENDED

			/*
			 * Now we get here structured field operations.
			 * In particular, this may be the first thing after setting up
			 * a terminal type of 327x-y-E.
			 * The data will then be a query.
			 * Subsequently there may be several such field operations,
			 * so remember to deal with them all.
			 *
			 * Complicated case may be where one includes in it some
			 * 3270 orders and data. This can happen for embedded
			 * write, erase/write and erase/write alternate operations.
			 */

			wsf_length = savep - p;


#ifdef DEBUG
	(void)fprintf(outf,"interpret   : WSF length is %d\n",
					 wsf_length);
#endif /* DEBUG */

			while (wsf_length > 2)
			{
				wsf_l_item = ((*p << 8) + *(p+1));
				if (!wsf_l_item)
					wsf_l_item = wsf_length;


#ifdef DEBUG
	(void)fprintf(outf,"interpret   : WSF item length is %d, contents are\n",
					 wsf_l_item);
	for (pos = 0; pos < (wsf_l_item - 2) ; pos++)
	{
		if (pos && !(pos % 20))
			(void)fprintf(outf, "\n%2.2x", *(p + 2 + pos));
		else
			(void)fprintf(outf, "-%2.2x", *(p + 2 + pos));
	}
	(void)fprintf(outf, "\n");
#endif /* DEBUG */

				/* Now the first switch on the type of this WSF item */

				switch (*(p+2))
				{
					case WSF_READ_P:

					/*
					 * There is now a second byte to specify the partition, which should
					 * always be 0xff, followed by a third byte which should specify the query.
					 * For the moment assume the second byte is OK.
					 */

					switch (*(p+4))
					{
						case WSF_R_QUERY:

						/* For now send a fixed string back! */

						sendnet(wsf_q_response, wsf_q_response + WSF_Q_LENGTH, 0x88);
						break;

						case WSF_R_QLIST:
						case WSF_R_M_ALL:
						case WSF_R_BUFFER:
						case WSF_R_MOD:

#ifdef DEBUG
	(void)fprintf(outf,"interpret   : WSF query type %d cannot be handled\n",
					 *(p+4));
#endif /* DEBUG */

						break;

						default:

#ifdef DEBUG
	(void)fprintf(outf,"interpret   : WSF query type %d is illegal!\n",
					 *(p+4));
#endif /* DEBUG */

						break;
					}
					break;

					case WSF_SET_R:

					/*
					 * IBM wants to specify the reply mode.
					 * The second byte is supposed to be zero: assume so.
					 * The third byte specifies the mode.
					 */

					switch (*(p+4))
					{
						case WSF_SR_CHAR:

						for (wsf_counter = 5 ; wsf_counter < wsf_l_item ; wsf_counter++)
						{
							switch (*(p+wsf_counter))
							{

								case EX_T_HIGHLIGHT:
									reply_highlight = TRUE;

#ifdef DEBUG
	(void)fprintf(outf,"interpret   : WSF Set Reply Mode: allow Extended Highlighting\n");
#endif /* DEBUG */

									break;

								case EX_T_COLOUR:
									reply_colour = TRUE;

#ifdef DEBUG
	(void)fprintf(outf,"interpret   : WSF Set Reply Mode: allow Extended Colour\n");
#endif /* DEBUG */

									break;

								case EX_T_CHAR_SET:

#ifdef DEBUG
	(void)fprintf(outf,"interpret   : WSF Set Reply Mode: cannot handle Extended Character Set\n");
#endif /* DEBUG */

								break;

								default:

#ifdef DEBUG
	(void)fprintf(outf,"interpret   : WSF Set Reply Mode: Illegal Character Mode %d\n",
							  *(p+4));
#endif /* DEBUG */

								break;
							}
						}
						break;

						case WSF_SR_FIELD:
						case WSF_SR_EXTEND:

#ifdef DEBUG
	(void)fprintf(outf,"interpret   : WSF Set Reply Mode %d cannot be handled\n",
					 *(p+4));
#endif /* DEBUG */

						break;

						default:

#ifdef DEBUG
	(void)fprintf(outf,"interpret   : WSF Set Reply Mode %d is illegal!\n",
					 *(p+4));
#endif /* DEBUG */

						break;
					}

					case WSF_3270DS:

					/*
					 * Should be a command for partition zero.
					 * If so then deal with it, including any embedded orders
					 */

					if (*(p+3) == 0)
					{
						command = *(p+4);

						if (command >= 0x60)
							command = EBCDICcommand[(command & 0x7f) - 0x60];

						int_write(command, p+5, p+wsf_l_item);

						break;
					}

					default:

#ifdef DEBUG
	(void)fprintf(outf,"interpret   : WSF function %d cannot be handled\n",
					 *(p));
#endif /* DEBUG */

					break;
				}

				p += wsf_l_item;
				wsf_length -= wsf_l_item;
			}

#else /* EXTENDED */

			/* Sorry, folks, we cannot yet support WSF */

#ifdef DEBUG
	(void)fprintf(outf,"interpret   :\tWRITE STRUCTURED FIELD\n",command);
	for (pos=0; pos<(savep-savebuf-1); pos++)
	{
		if (!(pos % 20))
			(void)fprintf(outf, "\n%2.2x", *(p+pos));
		else
			(void)fprintf(outf, "-%2.2x", *(p+pos));
	}
	(void)fprintf(outf, "\n");
#endif /* DEBUG */

#endif /* EXTENDED */

			break;

		case READMOD_ALLcommand:

		default:
			/* other commands not supported (treated as WRITE) */

			if (command > 0x1f)
			{
				/* Should NEVER happen !!!! */

#ifdef DEBUG
	(void)fprintf(outf,"interpret   :\tCOMMAND(0x%x) is illegal: ignore!",command);
#endif /* DEBUG */

#ifdef LOGFILE
			logit("Illegal command (0x%x) from IBM!!!",
				 command);
			logscreen();
#endif /* LOGFILE */

				return;
			}

#ifdef DEBUG
	(void)fprintf(outf,"interpret   :\tCOMMAND(0x%x) cannot be handled!\n",command);
#endif /* DEBUG */

			;
	}
}

/*
 * Little routine called from Start Field (SF) and Start Field Extended (SFE)
 * to do the common field setup.
 *
 * The only parameter is the attribute byte.
 * Note that if it is a call from SFE then some of the extended characteristics
 * may have been set. For ones which have not been set we may wish to merge in
 * ones (specifically colour) to indicate visually the type of field.
 */

#ifdef STANDARD_C

void    setup_field(short attrib)

#else /* STANDARD_C */

void    setup_field(attrib)

short   attrib;

#endif /* STANDARD_C */

{
	if (startpos < 0)
		startpos = bufferpos;
	if (infield)
		checkSF();
	infield = TRUE;
	prevsf = screen[(prevSF = bufferpos)];
	screen[bufferpos] = (attrib & ATTRMASK) | FIELDSTART;

#ifdef EXTENDED

	/*
	 * We can use the extended characteristics array to set up the
	 * type of colour and highlighting that we require.
	 * We should only do this if there is not already a specific setting
	 * of either (which there could be if we are called from a SFE).
	 * Of course, this could be overruled by a specific setting
	 * in subsequent Set Attributes.
	 */

	if (!(screen_ex[bufferpos] & EX_HIGHLIGHT))
	{
		/* There is no specific highlighting set */

		screen_ex[bufferpos] = (screen_ex[bufferpos] & EX_MASKHIGH) |
				       (ext_attrib[attrib & 0x3f] & EX_HBITS);
	}

	if (!(screen_ex[bufferpos] & EX_COLOUR))
	{
		/* There is no specific colour set */

		screen_ex[bufferpos] = (screen_ex[bufferpos] & EX_MASKCOLOUR) |
				       (ext_attrib[attrib & 0x3f] & EX_CBITS);
	}

#endif /* EXTENDED */

	/*
	 * Fields which the user will be allowed to change are usually sent
	 * with the Reset Modified Data Tag (MDT) bit already set. This emulator
	 * can put out these fields in such a way as to show them to the user,
	 * in particular if the field is also bright (for which we use reverse
	 * video). However, since the bit can later get cleared we need to
	 * save it in an unused bit position.
	 * Since some people do not like this feature, which is not a 3270 standard,
	 * there is an option to turn it off.
	 * Maybe we should also reconsider this feature if extended fields are
	 * being used, since they can include reverse video directly.
	 */

	if (show_MDT_field)
		screen[bufferpos] |= ((attrib & SFmodifiedBITS) << (SFmodentryBITS - SFmodifiedBITS));

	update_normal(bufferpos);

	if (prevsf != screen[bufferpos])
	{
		screencount++;
/*           if (startpos < 0)                */
/*                   startpos = bufferpos;    */
	}
	else
		screencount++;

#ifdef DEBUG
	(void)fprintf(outf, "setup_field :\t[Field (0x%x) : %s%s%s%s(%d,%d)]\n", attrib,
	attrib&SFprotectedBITS ? " PROT" : "",
	attrib&SFnumericBITS ? " NUM" : "",
	(attrib&SFvisibleBITS)==SFinvisibleBITS ? " INVISIBLE"
	    : (attrib&SFvisibleBITS)==SFstandoutBITS ? " BRIGHT"
	    : "",
	attrib&SFmodifiedBITS ? " MDT" : "",
	YX(bufferpos));
#endif /* DEBUG */

	if (!FORMATTED)
		firstfield = bufferpos;
	else
	{
		if (bufferpos < firstfield)
			firstfield = bufferpos;
	}
}

/*
 * This is a little routine to decide whether to change the field
 * definition so as to make it a blinking field (non-standard).
 * We do this at CERN if it is the prompt
 *                                            MORE...
 *                          or the prompt
 *                                            HOLDING
 * at the bottom right of the screen.
 * This is always a protected field starting 20 characters from the end,
 * i.e. startpos will be screensize-21 and will be a field definition
 * with value FIELDSTART + SFprotectedBITS.
 * We will change this by adding in SFspecialBITS.
 * However, we also need to stop the blinking, which we do by inserting
 * a field definition after the MORE... in place of a blank.
 * This will have value 0xa0, which is field definition and protected.
 *
 * If, however, we have extended attributes then we do it instead by
 * setting the blinking attribute on and off. Of course, this requires
 * that the field either has no existing highlighting attribute.
 *
 */

#ifdef STANDARD_C

int     checkSF(void)

#else /* STANDARD_C */

int     checkSF()

#endif /* STANDARD_C */

{
	static  short   more[9] = { 'M','O','R','E','.','.','.',' ',' '
				  };

	static  short   hold[9] = { 'H','O','L','D','I','N','G',' ',' '
				  };

	if ((prevSF == screensize-21) &&
	    (prevsf == (short)(FIELDSTART + SFprotectedBITS)) &&
	    ((!memcmp((void *)(screen+prevSF+1) , (void *)(more) , 9 * sizeof(short))) |
	     (!memcmp((void *)(screen+prevSF+1) , (void *)(hold) , 9 * sizeof(short)))))
	{

#ifdef EXTENDED

		if (!(screen_ex[prevSF] & EX_HBITS))
		{
			screen_ex[prevSF] |= (EX_HIGHLIGHT | EX_C_RED | EX_BLINK);
			screen[prevSF+8] = FIELDSTART + SFprotectedBITS;
			if (!(screen_ex[prevSF + 8] & EX_HBITS))
				screen_ex[prevSF + 8] |= EX_HIGHLIGHT;

#ifdef DEBUG
	(void)fprintf(outf,"checkSF     : set screen_ex[%d] = 0x%x, screen_ex[%d] = 0x%x\n",
		      prevSF, screen_ex[prevSF], prevSF+8, screen_ex[prevSF + 8]);
#endif /* DEBUG */

		}

#else /* EXTENDED */

		screen[prevSF] |= SFspecialBITS;
		screen[prevSF+8] = FIELDSTART + SFprotectedBITS;

#endif /* EXTENDED */

	}
}

/* update screen from information received */

#ifdef STANDARD_C

void    update_inf(void)

#else /* STANDARD_C */

void    update_inf()

#endif /* STANDARD_C */

{
#define NEAR    2               /* nearness criterion for combining ranges */

	short   *startp;
	int     starti, endi, w_endi;
	int     starti_ext, endi_ext, w_endi_ext;
	int     w_endpos;

	if (clearsn > 0)
		clscreen();
	if (startpos >= 0)
	{

#ifdef DEBUG
	(void)fprintf(outf,"update_inf  : startpos (%d,%d), bufferpos (%d,%d), endpos (%d,%d)\n",
		      YX(startpos), YX(bufferpos), YX(endpos));
	(void)fprintf(outf,"update_inf  : screencount %d, clearsn %d\n",
		      screencount, clearsn);
#endif /* DEBUG */

		if ((screen[prevSF] == prevsf)    &&
		    (bufferpos != startpos))
		{
			if (prevSF == DECR(bufferpos))
			{
				bufferpos = prevSF;
				if (!(--screencount))
				{

#ifdef DEBUG
	(void)fprintf(outf,"update_inf  : field (%d,%d) unchanged, nothing to do\n",
		      YX(prevSF));
#endif /* DEBUG */

					startpos = -1;
					endpos = -1;
					infield = FALSE;
					return;
				}

#ifdef DEBUG
	(void)fprintf(outf,"update_inf  : field (%d,%d) unchanged, reset bufferpos to it)\n",
		      YX(bufferpos));
#endif /* DEBUG */

			}
		}

		endpos = (screencount > screensize ? startpos :
			 (endpos == bufferpos ? INCR(endpos) : bufferpos));

		if (infield)
		{
			checkSF();
			prevSF = -1;
		}

		/* Now update the s_point pointers to show this region */

		if ((s_pfirst < 0) || (s_point[s_pfirst] >= 0))
		{
			startp = &s_pfirst;
			w_endpos = (endpos <= startpos ? endpos + screensize : endpos);
			while ((starti = *startp) >= 0)
			{
				endi = s_point[starti];
				w_endi = (endi <= starti ? endi + screensize : endi);

				/* skip any region which is nowhere near */
				/* where "near" means with a gap of at most NEAR */
				/* so first extend the (starti,endi) area by NEAR */

				if ((w_endi - starti + 2*NEAR) >= screensize)
				{
					starti_ext = endi_ext = starti;
				}
				else
				{
					if ((starti_ext = (starti - NEAR)) < 0)
						starti_ext = 0;
					if ((endi_ext = (endi + NEAR)) >= screensize)
						endi_ext = screensize-1;
				}
				w_endi_ext = (endi_ext <= starti_ext ? endi_ext + screensize : endi_ext);

				if ((starti_ext < startpos) &&
				    (starti_ext > (w_endpos - screensize)) &&
				    (w_endi_ext < startpos))
				{
					/* No overlap : move to the next one */
					startp = s_point + endi;
					continue;
				}
				if ((startpos < starti_ext) &&
				    (startpos > (w_endi_ext - screensize)) &&
				    (w_endpos < starti_ext))
				{
					/* No overlap : are we comparing to an overlap? */
					if (endi < starti)
					{
						startp = s_point + endi;
						continue;
					}
					else
						break;
				}

				/* There is some overlap: merge the two regions */
				/* but first prepare for a continue statement */
				*startp = s_point[endi];

				/* If we already have the whole buffer OK */
				if (endpos == startpos)
					continue;

				/* See if we must move back the startpos */
				if (((startpos > starti) && (startpos <= w_endi_ext)) ||
				    ((startpos < starti) && (startpos <= (w_endi_ext - screensize))))
				{
					startpos = starti;
					if (w_endpos <= starti)
						w_endpos += screensize;
				}
				else
					if (w_endi <= startpos)
						w_endi += screensize;

				/* We may also need to move on the endpos */

				if (w_endpos < w_endi)
					w_endpos = w_endi;

				/* And now we may have too much */
				if ((endpos = w_endpos) >= screensize )
				{
					if ((endpos -= screensize) > startpos)
						endpos = startpos;
				}

				/* the old range is now included in this new one */
			}

			/*
			 * Now insert the new range
			 * If the whole screen then start from the beginning
			 * If it is a range which goes right to the end
			 * then make sure it goes first
			 */

			if (endpos == startpos)
			{
				endpos = startpos = 0;
			}

			if (endpos < startpos)
				startp = &s_pfirst;

			s_point[startpos] = endpos;
			s_point[endpos] = *startp;
			*startp = startpos;
		}

#ifdef LDEBUG
	LOG(log,"update_inf  : range list printout\n");
	startpos = s_pfirst;
	while (startpos >= 0)
	{
		if ((endpos = s_point[startpos]) < 0)
			endpos = startpos;
		LOG(log,"  (%d,%d) to (%d,%d)\n",
			   YX(startpos), YX(endpos));
		startpos = s_point[endpos];
	}
#endif /* LDEBUG */




	/* Check for an unformatted screen (should never happen!) */

	if (FORMATTED && !(screen[firstfield] & FIELDSTART))
	{
		/* Should NEVER happen !!!! */

#ifdef LOGFILE
		logit("Screen has been set unformatted : firstfield was %d !!!",
			 firstfield);
/*                logscreen(); */
#endif /* LOGFILE */

#ifdef DEBUG
	(void)fprintf(outf, "update_cs   : Screen has been set unformatted : firstfield was %d\n",
				firstfield);
#endif /* DEBUG */

#ifdef SHOWFIELDS
	showfields();
#endif /* SHOWFIELDS */

		reset_ff(0);
	}




		screencount = 0;
		startpos = -1;
		endpos = -1;
		infield = FALSE;
	}
}

/* Update the screen at the current (cursorpos) location with the character c.
 * Note that the updating of curscr is NOT done here.
 * Note also that if we are allowing extended attributes then we may
 * need to specify these also for the new character if there has been
 * any Set Attribute setting. In this case the new attribute should be in current_ex.
 */

#ifdef STANDARD_C

void    changechar(unsigned char c)

#else /* STANDARD_C */

void    changechar(c)

unsigned char c;

#endif /* STANDARD_C */

{
	screen[cursorpos] = c;

	if (FORMATTED)
	{
		screen[fieldstart] |= SFmodifiedBITS;
	}

#ifdef EXTENDED
	screen_ex[cursorpos] = current_ex;
#endif /* EXTENDED */

}

/* action routines to respond to various keyboard escape sequences */

/* ARGSUSED */

/* unimplemented action.  The argument is really a char * name of the
 * action, but we just ring the bell.
 */

#ifdef STANDARD_C

void    act_unimpl(int what)

#else /* STANDARD_C */

void    act_unimpl(what)

int     what;

#endif /* STANDARD_C */

{
	slfset("NO: ILLEGAL CHARACTER!", 2);
}

#ifdef STANDARD_C

void    sendnet(unsigned char *buffer, unsigned char *bufend, unsigned char key)

#else /* STANDARD_C */

void    sendnet(buffer, bufend, key)

unsigned char   *buffer;
unsigned char   *bufend;
unsigned char   key;

#endif /* STANDARD_C */

{
	*buffer = key;
	*bufend++ = (char) TELCMD_IAC;
	*bufend++ = (char) TELCMD_EOR;
	writnet(net, buffer, bufend - buffer);

	if (key != NOattentionKEY)
	{
		/* Lock out keyboard input for now */
		unlock = FALSE;
		slputstr( "X" , 9, 0);
		Y = rows-1;
		X = linelen-1;
		writecursor();
	}

#ifdef TONET
	if (write(tonet, (char *) buffer, bufend-buffer) < 0)
		;
#endif /* TONET */

#ifdef DEBUG
	(void)fprintf(outf,"sendnet     :\t[sent %d bytes]\n",bufend-buffer);
#endif /* DEBUG */

	aid = key;

	/* If in insert mode then maybe go back to normal mode */

	if (insert && !keep_insert)
		(void)act_insert(0);

	/* If in 8-bit input mode then go back to 7-bit mode */

	if (SO_imask)
		(void)act_ascii(0);
}

/*
 * Move to the next suitable field.
 * Assume that cursorpos is the current position, plus the associated
 * variables fieldstart and currentstyle.
 * The move can be backward or forward according to the first
 * argument being true or false.
 * In the case of a backward move allow for the case of cursorpos
 * being one beyond the field definition character of the current field.
 * In the case of a forward move cursorpos might already be on an
 * acceptable field.
 * Any autoskip (protected numeric) field will always be skipped.
 * Fields which include the mask bits of the second argument will
 * also be skipped.
 * Fields which are empty will also be skipped.
 * If no suitable field is found then go to the first field.
 * On exit leave cursorpos, fieldstart and currentstyle set, plus
 * X (curscr->_curx) and Y (curscr->_cury) so that writecursor can be called.
 * The exit value is TRUE if a suitable field was found, otherwise FALSE.
 */

#ifdef STANDARD_C

bool    next_field(bool backward, int mask)

#else /* STANDARD_C */

bool    next_field(backward, mask)

bool    backward;
int     mask;

#endif /* STANDARD_C */

{
	register i;
	int     mask_copy = mask;

#ifdef DEBUG
	(void)fprintf(outf, "next_field  : Move %s from (%d,%d) : mask 0x%x\n",
				(backward ? "backward" : "forward"), YX(cursorpos), mask_copy);
#endif /* DEBUG */

	/*
	 * If we are going forwards and the current position is not
	 * a field definition then we must first go back one in case
	 * the previous position was a field definition.
	 * After this possible move w must check if we are on a field
	 * definition: if so then we must go back (another) one.
	 *
	 * If we are going backwards then we need to go backwards one
	 * if we are not on a field definition to cater for the case
	 * where the position is the first of a field.
	 */

	if (!(screen[cursorpos] & FIELDSTART))
		movecursor(-1);
	if ((screen[cursorpos] & FIELDSTART) && !backward)
		movecursor(-1);

#ifdef DEBUG
	(void)fprintf(outf, "next_field  : Start at (%d,%d)\n",
				YX(cursorpos));
#endif /* DEBUG */

	i = cursorpos;  /* remember position */

	/* find a suitable field or do a complete loop */
	do
	{
		if (backward)

		/* Do NOT remove these curly braces !!! */

		{
			DEC(cursorpos);
		}
		else
		{
			INC(cursorpos);
		}
		if (screen[cursorpos] & FIELDSTART)
		{
			if (screen[cursorpos] & mask_copy)
				continue;
			if ((screen[cursorpos] & (SFprotectedBITS + SFnumericBITS)) ==
						 (SFprotectedBITS + SFnumericBITS)     )
			{
				mask_copy |= SFprotectedBITS;
				continue;
			}
			if (!(screen[INCR(cursorpos)] & FIELDSTART))
			{
				fieldstart=cursorpos;
				currentstyle = stylebits((int)screen[cursorpos]);
				movecursor(1);

#ifdef DEBUG
	(void)fprintf(outf, "next_field  : moved cursor to (%d,%d)\n",
				YX(cursorpos));
#endif /* DEBUG */

				return(TRUE);
			}
		}
	} while (i != cursorpos);

	/* did a complete loop and no unprotected field found */

	movecursor(-cursorpos);      /* go to location 0 */

	/* search for the start of the current field */

	while ((screen[i] & FIELDSTART) == 0)
		DEC(i);
	fieldstart = i;
	currentstyle = stylebits((int)screen[i]);

#ifdef DEBUG
	(void)fprintf(outf, "next_field  : Failure : set cursor to (%d,%d)\n", ly+1, lx+1);
#endif /* DEBUG */

	return(FALSE);
}

/*
 * We will want to change the character in the current cursor position.
 * Before that we should check if this is OK.
 */

#ifdef STANDARD_C

bool    ok_write(void)

#else /* STANDARD_C */

bool    ok_write()

#endif /* STANDARD_C */

{
	slfclear();
	if (screen[cursorpos] & FIELDSTART)
	{
		/* trying to write to field definition position */
		slfset("NO: FIELD DEFINITION!", 2);
		return(FALSE);
	}
	if (FORMATTED && (screen[fieldstart] & SFprotectedBITS))
	{
		/* trying to write to field definition position */
		slfset("NO: PROTECTED FIELD!", 2);
		return(FALSE);
	}
	return(TRUE);
}

/* Program attention (PA) or CLEAR key.  CLEAR is treated like PA0  */

#ifdef STANDARD_C

void    act_pa(int key)

#else /* STANDARD_C */

void    act_pa(key)

int     key;

#endif /* STANDARD_C */

{
	unsigned char buf[3];

	slfclear();

	if (!key)
	{
		/* CLEAR key: clear the screen and set cursor to home */

		clearsn = altscreen + 1;
		clscreen();

#ifdef EXTENDED
		/* Reset extended attributes to zero */

		current_ex = 0;
#endif /* EXTENDED */


#ifdef DEBUG
	(void)fprintf(outf,"act_pa      : CLEAR KEY\n");
#endif /* DEBUG */

	}

#ifdef DEBUG
	if (key)
	{
		(void)fprintf(outf,"act_pa      : PA%d KEY\n",key);
	}
#endif /* DEBUG */

	sendnet(&buf[0],&buf[1], PAkey[key]);
}

#ifdef STANDARD_C

void    act_home(int dummy)   /* Home the cursor             (BS) */

#else /* STANDARD_C */

void    act_home(dummy)   /* Home the cursor             (BS) */

int     dummy;

#endif /* STANDARD_C */

{
	slfclear();
        movecursor(-cursorpos);
	(void)next_field(FALSE, SFprotectedBITS);
	writecursor();

#ifdef DEBUG
	(void)fprintf(outf, "act_home    : move to first unprotected field (%d,%d)\n",
					YX(cursorpos));
#endif /* DEBUG */

}

/* ARGSUSED */

#ifdef STANDARD_C

void    act_newline(int dummy)

#else /* STANDARD_C */

void    act_newline(dummy)

int     dummy;

#endif /* STANDARD_C */

{
	slfclear();

#ifdef DEBUG
	(void)fprintf(outf, "act_newline : current position (%d,%d): cursorpos (%d,%d)\n",
		  ly+1, lx+1, YX(cursorpos));
#endif /* DEBUG */

	movecursor(linelen - cursorpos % linelen);

#ifdef DEBUG
	(void)fprintf(outf, "act_newline : movecursor(%d) to next line\n",
		  linelen - cursorpos % linelen);
#endif /* DEBUG */

	if (!FORMATTED)              /* no fields */
	{
		writecursor();

#ifdef DEBUG
	(void)fprintf(outf, "act_newline : has set unformatted screen cursor to (%d,%d)\n",
		  ly+1, lx+1);
#endif /* DEBUG */

		return;
	}

	/* Now fieldstart should be the one applicable to this position */
	/* If the field is unprotected then all is OK */
	if (screen[fieldstart] & SFprotectedBITS)
	{
		(void)next_field(FALSE, SFprotectedBITS);

#ifdef DEBUG
	(void)fprintf(outf, "act_newline : final move to cursorpos (%d,%d)\n",
		  YX(cursorpos));
#endif /* DEBUG */

	}
	if (fieldstart == cursorpos)
		movecursor(1);
	writecursor();

#ifdef DEBUG
	(void)fprintf(outf, "act_newline : has moved cursor to (%d,%d)\n", ly+1, lx+1);
#endif /* DEBUG */

}

/*
 * Move the cursor position.
 * If direction is non-zero then it will indicate one hop in that direction:-
 *
 * 1 : left
 * 2 : down
 * 3 : up
 * 4 : right
 *
 * If direction is zero then the required coordinates must be read in.
 * In this case they should come as a standard VT-type sequence of the form
 *
 * <line>;<column>H
 *
 * Remember, however, that the line and column are counted from 1.
 * Remember also that these coordinates may not yet be read.
 */

#ifdef STANDARD_C

void    act_move(int direction)

#else /* STANDARD_C */

void    act_move(direction)

int     direction;

#endif /* STANDARD_C */

{
	int     line_move = -1;
	int     col_move = -1;
	int     read_value = 0;
	short   read_char;
	int     input_ok = 0;

	slfclear();

#ifdef EXTENDED

	/*
	 * Because of a colour terminal emulator problem
	 * we may want to rewrite the current cursor character.
	 */

	if (!strncmp(term_type, "xterm", 5))
	{
		display_char(cursorpos/linelen, cursorpos%linelen);
	}

#endif /* EXTENDED */

	switch(direction)
	{

	case 1: /* left */

		movecursor(-1);
		break;

	case 2: /* down */

		movecursor(linelen);
		break;

	case 3: /* up */

		movecursor(-linelen);
		break;

	case 4: /* right */

		movecursor(1);
		break;

	case 0: /* general move */
		/* Interpret VT-type cursor move sequence */

		while (!input_ok)
		{
			if ((read_char = kbread_next()) < 0)
			{
				reset_kbread();
				return;
			}
			switch (read_char)
			{
				case ';':
					if (line_move >= 0)
						input_ok = -1;
					else
					{
						line_move = (read_value ? read_value : 1);
						read_value = 0;
					}
					break;

				case 'H':
					if (line_move >= 0)
						col_move = (read_value ? read_value : 1);
					else
					{
						line_move = (read_value ? read_value : 1);
						col_move = 1;
					}
#ifdef DEBUG
	(void)fprintf(outf, "act_move    : want to move cursor to (%d,%d)\n",
		  line_move, col_move);
#endif /* DEBUG */
					cursorpos = linelen * (line_move - 1) + (col_move - 1);
					if (cursorpos >= screensize)
						cursorpos = screensize - 1;
					putcursor();
					if ((screen[fieldstart] & SFprotectedBITS) ||
					    (screen[cursorpos] & FIELDSTART))
						(void)next_field(FALSE, SFprotectedBITS);
					else
						if (fieldstart != DECR(cursorpos))
							(void)next_field(TRUE, SFprotectedBITS);
					input_ok = 1;
					break;

				default:
					if (isdigit((char) read_char))
						read_value = 10 * read_value + (read_char - '0');
					else
						input_ok = -1;
					break;
			}
			if (input_ok < 0)
			{
#ifdef DEBUG
	(void)fprintf(outf, "act_move    : failure to recognise input sequence!\n");
#endif /* DEBUG */
				return;
			}
		}
		break;
	}
	writecursor();

#ifdef DEBUG
	(void)fprintf(outf, "act_move    : moved cursor to (%d,%d)\n",
		  ly+1, lx+1);
#endif /* DEBUG */

}

/* ARGSUSED */

/* Switch to extended/normal ascii character set (input) */

#ifdef STANDARD_C

void    act_ascii(int how)

#else /* STANDARD_C */

void    act_ascii(how)

int     how;

#endif /* STANDARD_C */

{
	if (how)
	{
		SO_imask = 0x80;

#ifdef DEBUG
	(void)fprintf(outf, "act_ascii   : switch to GR codes\n");
#endif /* DEBUG */

		slputstr("8-BIT!", 34, 2);
	}
	else
	{
		SO_imask = 0;

#ifdef DEBUG
	(void)fprintf(outf, "act_ascii   : switch back to GL codes\n");
#endif /* DEBUG */

		slerase(6, 34, 0);
	}
}

/* ARGSUSED */

#ifdef STANDARD_C

void    act_insert(int how)

#else /* STANDARD_C */

void    act_insert(how)

int     how;

#endif /* STANDARD_C */

{
	insert = (how < 2 ? how : 1 - insert);

#ifdef DEBUG
	(void)fprintf(outf, "act_insert  : set insert to %d\n",insert);
#endif /* DEBUG */

	if (insert)
		slputstr( "INSERT MODE" , 43, 1);
	else
		slerase(11, 43, 0);
	if (!sline)
	{
		f_puttcs((insert ? tcsIB : tcsIE), TRUE);
	}

}

/* ARGSUSED */

#ifdef STANDARD_C

void    act_debug(int how)

#else /* STANDARD_C */

void    act_debug(how)

int     how;

#endif /* STANDARD_C */

{
	if (how<2) debug_flag = how;
	else debug_flag = 1 - debug_flag;
}

/* ARGSUSED */

#ifdef STANDARD_C

void    act_erase(int how)   /* Erase to end-of-field */

#else /* STANDARD_C */

void    act_erase(how)   /* Erase to end-of-field */

int     how;

#endif /* STANDARD_C */

{
	if (!ok_write())
		return;

	startpos = cursorpos;
	do
	{

#ifdef EXTENDED
		current_ex = 0;
#endif /* EXTENDED */

		changechar((unsigned char)0);
		INC(cursorpos);
	} 
	while ((screen[cursorpos] & FIELDSTART) == 0);
	endpos = cursorpos;
	cursorpos = startpos;
	update_curscr();
	startpos = -1;
	endpos = -1;
	writecursor();

#ifdef DEBUG
	(void)fprintf(outf, "act_erase   : put cursor to (%d,%d)\n",ly+1, lx+1);
#endif /* DEBUG */

}

/* Erase a character (destructive backspace) (BS) */

#ifdef STANDARD_C

void    act_erbsp(int dummy)

#else /* STANDARD_C */

void    act_erbsp(dummy)

int     dummy;

#endif /* STANDARD_C */

{
	movecursor(-1);
	if (delchar() < 0)
	{
		movecursor(1);               /* FIX (BS) */
		writecursor();
	}
}

/*
 * Create a string for the given directory followed by the given file name
 * and return the address of the string.
 */

#ifdef STANDARD_C

char    *set_fname(char *fdir, char *fname)

#else /* STANDARD_C */

char    *set_fname(fdir, fname)

char    *fdir;
char    *fname;

#endif /* STANDARD_C */

{
	(void)strcpy(&full_fname[0], fdir);
	if (fname)
		(void)strcat(&full_fname[0], fname);
	return (&full_fname[0]);
}

/*
 * This is a routine which gets the user out of his 3270 mode and allows
 * a dialogue so that we can help him.
 * An entry with the prompt parameter set means that there is a prompt
 * string as well as the standard help options.
 * Of course, this routine cannot know what should be the possible answer(s)
 * to this string!
 * If helpchar is zero then ask the user what help he wants.
 * Keep asking him as long as he asks for any of the standard options.
 * If helpchar is non-zero then treat it as if the user had asked for this
 * particular help option.
 * In the case that there is a prompt string AND helpchar is zero then
 * any answer which is NOT one of the standard options causes the
 * routine to return this answer character (zero if the user typed a
 * simple Return) WITHOUT resetting 3270 mode.
 * The same is true (no reset of 3270 mode) if helpchar is non-zero.
 */

#ifdef STANDARD_C

char    dialogue(char *prompt_string, char helpchar)

#else /* STANDARD_C */

char    dialogue(prompt_string, helpchar)

char    *prompt_string;
char    helpchar;

#endif /* STANDARD_C */

{
	int     loop = TRUE;
	int     char_in;

#ifdef DEBUG
	(void)fprintf(outf, "dialogue    : remember cursor at (%d,%d)\n",
		  YX(cursorpos));
#endif /* DEBUG */

	cursorsave = cursorpos;

	/* switch to normal mode */

	clearnormal();
	slsave();
	slclear();
	slputstr("HELP mode : Return[s] to exit", 0, 0);
	reset_tty(SYNCH_DIALOGUE, TRUE);

	while (loop)
	{
		if (helpchar)
		{
			if (prompt_string)
			{
				f_puts(prompt_string, FALSE);
				f_puts("\nWhen you have read this type simply the <Return> key ", FALSE);
				empty_input();
				while (((char_in = f_getc(60)) != EOF) &&
				       (char_in != '\n')               &&
				       (char_in != '\r'))
				       ;
				f_puttcs(tcsCL, FALSE);
			}
			char_in = (int)helpchar;
		}
		else
		{
			f_puts("This is an on-line help dialogue: please choose an option\n\n", FALSE);
			f_puts("  a   Repeat the initial announcements (if there were any)\n", FALSE);
			f_puts("  e   Give information on the 3270 emulation type options\n", FALSE);
			f_puts("  f   Give special help and hints to Falco users\n", FALSE);
			f_puts("  h   Give general help information\n", FALSE);
			f_puts("  k   Show the keyboard mapping for your specific terminal\n", FALSE);
			f_puts("  m   Give general information on the IBM keyboard mapping\n", FALSE);
			f_puts("  o   Show/change the option available settings\n", FALSE);
			f_puts("  p   Give general information on screen saving and printing\n", FALSE);
			f_puts("  s   Show the action of any key or combination\n", FALSE);
			f_puts("  x   Give information on extended character setinput/output\n", FALSE);

			if (prompt_string)
				f_puts(prompt_string, FALSE);
			f_puts("\nTo exit type simply the <Return> key ", FALSE);
			empty_input();
			while ((char_in = f_getc(30)) == (int)' ')
				;

#ifdef DEBUG
	(void)fprintf(outf, "dialogue    : read in character 0x%x\n",
		  char_in);
#endif /* DEBUG */

		}
		if ((char_in == '\n') || (char_in == '\r') || (char_in == EOF))
		{
			char_in = 0;
			loop = FALSE;
		}
		else
		{

#ifdef DEBUG
	(void)fprintf(outf, "dialogue    : read in character %c\n",
		  char_in);
#endif /* DEBUG */

			if (isupper((int)char_in))
				char_in = tolower(char_in);
			switch (char_in)
			{
				case 'a':
					if (fileout(set_fname(LIBDIR, ANNOUNCE), 1) <= 0)
					{
						f_puttcs(tcsCL, FALSE);
						f_puts( "\nSorry, there were no initial announcements\n\n", TRUE);
					}
					break;

				case 'e':
					(void) fileout(set_fname(LIBDIR, HELPEMUL), 1);
					break;

				case 'f':
					(void) fileout(set_fname(LIBDIR, HELPFALCO), 1);
					break;

				case 'h':
					(void) fileout(set_fname(LIBDIR, HELPFILE), 1);
					break;

				case 'k':
					if (fileout(set_fname(LIBDIR, keymap_type), 1) > 0)
						break;
					f_puts( "\nSorry, I don't have a specific help for mapping ", FALSE);
					f_puts( keymap_type, FALSE);
					f_puts( "\nI will try to give you some general information\n\n", TRUE);
					(void)sleep(4);

				case 'm':
					(void) fileout(set_fname(LIBDIR, HELPMAP), 1);
					break;

				case 'o':
					{
						int     ochar_in;

						if (!helpchar)
							(void) fileout(set_fname(LIBDIR, HELPOPT), 1);
				while (TRUE)
						{
							int     opt_ci;
							int     opt_pi;

							f_puts( "\nThe options are as below: currently selected ones are preceded by *\n\n", FALSE);
							for (opt_ci = 0 ; opt_ci < OPT_CSIZE ; opt_ci++)
							{
								char    opt_fstring[10]    = "   %c ";
								opt_pi = opt_cindex[opt_ci];
								if ((opt_pi >= 0) && !opt_smaxlen[opt_pi] &&
								    (*((int *)opt_pointer[opt_pi]) == opt_cival[opt_ci]))
								{ 
							opt_fstring[1] = '*';
								 } else {
								     opt_fstring[1] = ' '; }
								f_putf( opt_fstring, FALSE, (int)opt_cchar[opt_ci]);
								f_putf( opt_cmess[opt_ci], FALSE, 0);

								if ((opt_pi >= 0) && opt_smaxlen[opt_pi])
								{
									f_putc(' ');
									f_putc('[');
									f_putf((char *)opt_pointer[opt_pi], FALSE, 0);
									f_putc(']');
								}
								f_putc('\n');
							}

							f_puts( "\nPlease choose any single letter option: when OK just Return\n\n", FALSE);
							empty_input();

							while ((ochar_in = f_getc(60)) == (int)' ')
								;
							f_puttcs(tcsCL, FALSE);

							if ((ochar_in == '\n') || (ochar_in == EOF))
								break;

							if (isupper((int)ochar_in))
								ochar_in = tolower(ochar_in);

							for (opt_ci = 0 ; opt_ci < OPT_CSIZE ; opt_ci++)
								if (ochar_in == opt_cchar[opt_ci])
									break;
							if (opt_ci < OPT_CSIZE)
							{
								f_putf("\nYou have chosen ", FALSE, 0);
								f_putf(opt_cmess[opt_ci], FALSE, 0);
								f_putc('\n');
								if ((opt_pi = opt_cindex[opt_ci]) < 0)
								{
									/* This must be a special one: check it! */

									if (opt_cchar[opt_ci] == 's')
									{
										/* User wants to save his settings */

#ifdef OPTDIR

											if (set_options(1) < 0)
											{
												f_puts( "\nVery sorry: the settings file is unavailable at the moment\n", TRUE);
												sleep(2);
											}
											else
											{
												f_puts( "\nOK, I have saved these settings in file ", FALSE);
												f_puts( opt_file, FALSE);
												f_putc( '\n');
												f_flush();
											}

#else /* OPTDIR */

											f_puts( "\nVery sorry: this program version does not specify an options directory\n", TRUE);
											sleep(2);

#endif /* OPTDIR */

									}
								}
								else
								{
									if (!opt_smaxlen[opt_pi])
									{
										/* Just an integer value to reset */

										*((int *)opt_pointer[opt_pi]) = opt_cival[opt_ci];
									}
									else
									{
										/* Must be a string value to be read */

										char    instring[50];
										int     inindex;
										int     ichar_in;

										for (inindex = 0 ; ; )
										{
											if (!inindex)
											{
												f_putf( "Please type your new choice [", FALSE, 0);
												f_putf( (char *)opt_pointer[opt_pi], FALSE, 0);
												f_putf( "] ? ", TRUE, 0);
												empty_input();
											}
											if ((ichar_in = f_getc(30)) == '\n')
												break;
											if (ichar_in == EOF)
											{
												if (!inindex)
													break;
												f_putf("\nSorry, you took a bit too long: try again!\n", TRUE, 0);
												inindex = 0;
												continue;
											}
											switch (ichar_in)
											{
												case ' ':
													if (inindex)
														break;
													f_putf("\nSorry, I cannot accept blanks in a name!\n", TRUE, 0);
													inindex = 0;
													break;

												case ':':
													f_putf("\nSorry, I cannot accept a colon in a name!\n", TRUE, 0);
													inindex = 0;
													break;

												default:
													if (isprint(ichar_in))
													{
														if (inindex == (opt_smaxlen[opt_pi] - 1))
														{
															f_putf("\nSorry, this name is too long\n", TRUE, 0);
															inindex = 0;
															break;
														}
														instring[inindex++] = (char) tolower(ichar_in);
														f_putc(ichar_in);
														f_flush();
													}
													else
													{
														f_putf("\nSorry, you just typed some non-ASCII character!\n", TRUE, 0);
														inindex = 0;
														break;
													}
											}
										}

										if (inindex)
										{
											instring[inindex] = '\0';
											strcpy((char *)opt_pointer[opt_pi], instring);
										}
									}
								}
							}
							else
								f_putf("\nSorry, I don't understand option %c!\n", FALSE, ochar_in);
							f_flush();
						}
					}
					if (helpchar)
					{
						slrestore();
						return(char_in);
					}
					break;

				case 'p':
					(void) fileout(set_fname(LIBDIR, HELPPRINT), 1);
					break;

				case 's':
					{
						short   SO_isave;

						setup_tty(TRUE);
						f_puttcs(tcsCL, FALSE);
						f_puts( "OK, you want me to tell you which keys do what\r\n", FALSE);
						f_puts( "Type any key (or combination) and I will tell you about it\r\n", FALSE);
						if (SO_isave = SO_imask)
							f_puts( " (and note that you are currently inputting in extended ASCII!)\r\n", FALSE);
						f_puts( "To end this help just type the Return key\r\n", FALSE);
						empty_input();

						{
							int     res_kb;

							while ((res_kb = read_kbbuf(30)) > 0)
								;
							if (res_kb == EOF)
							{
								f_puttcs(tcsCL, FALSE);
								f_puts("Sorry, you took too long: type faster next time!\r\n", TRUE);
							}
							sleep(3);
						}
						SO_imask = SO_isave;
						clearnormal();
						reset_tty(SYNCH_DIALOGUE, TRUE);
						break;
					}

				case 'x':
					(void) fileout(set_fname(LIBDIR, HELPEXT), 1);
					break;

				default:
					if (prompt_string)
					{
						slrestore();
						return(char_in);
					}
					else
						f_puts( "\nSorry, I don't understand that option!\n\n", FALSE);
			}
		}
		if (helpchar)
		{
			int     c;

			if (prompt_string)
				f_puts(prompt_string, FALSE);
			f_puts("\nTo continue type simply the <Return> key ", FALSE);
			empty_input();
			while ((c = f_getc(30)) != '\n' && c != EOF)
				;
			slrestore();
			return(char_in);
		}
	}

	slrestore();
	mainreset(cursorsave);

	return (char_in);
}

/*
 * A little routine to send back to the IBM the (16-bit) character.
 * The complication is that not only is it to be changed back to EBCDIC
 * but it may also be a graphics character, in which case it must be
 * preceded by the GEchar.
 */

#ifdef STANDARD_C

void    put_ebcdic(int ch_short)

#else /* STANDARD_C */

void    put_ebcdic(ch_short)

int     ch_short;

#endif /* STANDARD_C */

{
	if (ch_short & GE_CHAR)
	{
		put_save((unsigned char)GEchar);
		put_save((ch_short & GE_NBG) ? (unsigned char)(ch_short & 0xff) : atog[ch_short & 0x1f]);
	}
	else
		put_save(atoe[ch_short & 0xff]);
}

#ifdef STANDARD_C

void    act_help(int dummy) /* Request for some on-line help */

#else /* STANDARD_C */

void    act_help(dummy) /* Request for some on-line help */

int     dummy;

#endif /* STANDARD_C */

{
	(void) dialogue((char *)NULL, (char)0);
}

/* Telnet escape character (more or less..)  (BS) */

#ifdef STANDARD_C

void    act_escape(int dummy)

#else /* STANDARD_C */

void    act_escape(dummy)

int     dummy;

#endif /* STANDARD_C */

{
	int     dialans;

	dialans = dialogue("  q   Quit the program completely!!\n", (char)0);
	if ((dialans == 'q') || (dialans == 'Q'))
	{
		end_program(-1 , "OK, bye-bye", 0);
	}
	if (dialans)
		mainreset(cursorsave);
}

#ifdef STANDARD_C

void    reset(int cur_save)

#else /* STANDARD_C */

void    reset(cur_save)

int     cur_save;

#endif /* STANDARD_C */

{
	/* reset should unlock the keyboard */
	/* It should also clear any bad status */
	if (!unlock)
	{
		unlock = TRUE;
		slputstr( " " , 9, 0);
	}
	slfclear();

	/* If in insert mode then go back to normal mode */

	if (insert)
		(void)act_insert(0);

#ifdef DEBUG
	(void)fprintf(outf, "reset       : reset cursorpos to (%d,%d)\n",
		  YX(cur_save));
#endif /* DEBUG */

	cursorpos = cur_save;
	putcursor();
}

#ifdef STANDARD_C

void    redraw(int cur_save)

#else /* STANDARD_C */

void    redraw(cur_save)

int     cur_save;

#endif /* STANDARD_C */

{
	/* Go to synchronous mode to get important stuff out */

	reset_tty(-SYNCH_NORMAL, TRUE);

	clearnormal();

	if (altflip)
		f_puttcs((altscreen ? tcsSW : tcsSN), TRUE);

	/* Can now revert to normal asynch mode */

	setup_tty(FALSE);

	act_refresh(1);

	reset(cur_save);
}

#ifdef STANDARD_C

void    mainreset(int cur_save)

#else /* STANDARD_C */

void    mainreset(cur_save)

int     cur_save;

#endif /* STANDARD_C */

{
	setup_tty(TRUE);
	f_puttcs(tcsIS, TRUE);
	redraw(cur_save);
}

/* simple reset (normally ^t) */

#ifdef STANDARD_C

void    act_reset(int dummy)

#else /* STANDARD_C */

void    act_reset(dummy)

int     dummy;

#endif /* STANDARD_C */

{
	reset(cursorpos);
}

#ifdef STANDARD_C

void    act_mreset(int dummy) /* master reset (normally ^g) */

#else /* STANDARD_C */

void    act_mreset(dummy) /* master reset (normally ^g) */

int     dummy;

#endif /* STANDARD_C */

{
	mainreset(cursorpos);
}

#ifdef STANDARD_C

void    act_tab(int how)

#else /* STANDARD_C */

void    act_tab(how)

int     how;

#endif /* STANDARD_C */

{
	slfclear();
	movecursor(how ? -1 : 1);
	if (!FORMATTED)
	{
		/* no fields */
		writecursor();

#ifdef DEBUG
	(void)fprintf(outf, "act_tab     : set unformatted screen cursor to (%d,%d)\n",
		  ly+1, lx+1);
#endif /* DEBUG */

		return;
	}

	(void)next_field(how, SFprotectedBITS);
	writecursor();
}

#ifdef STANDARD_C

void    act_delete(int how)

#else /* STANDARD_C */

void    act_delete(how)

int     how;

#endif /* STANDARD_C */

{
	(void) delchar();
}

/*
 * Delete the character at the current cursorpos, shifting left everything up to
 * the end of the current field or the end of the current line (whichever
 * comes first).  Insert a null in the newly vacated position.
 * Note that this function is not an inverse to insertchar() if the
 * current field spans multiple lines, but that's the way the 3270 is
 * defined!  Don't move the cursor.
 * screen[] is updated and then update_curscr is called to refresh
 * the user screen (curscr).
 *
 * Return -1 if illegal delete, else 0.
 */

/* ARGSUSED */

#ifdef STANDARD_C

int     delchar(void)

#else /* STANDARD_C */

int     delchar()

#endif /* STANDARD_C */

{
	register int last, i;

	if (!ok_write())
		return(-1);

	if (FORMATTED)
		screen[fieldstart] |= SFmodifiedBITS;
	/* find the end of the region to be shifted */
	last = cursorpos;
	while ((screen[last = INCR(last)] & FIELDSTART) == 0)
	{

/*
 * A real 3270 also looks for the end of the line, but many emulators don't.
 * We may choose not to (by commenting out the following statement.
 * On a 7171-style display we actually don't look here for the end of line.
 *
 */
		if (!emul7171 && !(last % linelen))
			break;
	}
	/* at this point either:
	 *   last is at the start of the next field
	 *   or, if really doing 3270 emulation,
	 *   last is at the start of the next line
	 */

	if (last == 0)
		last = screensize;

#ifdef DEBUG
	(void)fprintf(outf, "delchar     : at (%d,%d): shift prior to (%d,%d)\n",
		    YX(cursorpos), YX(last));
#endif /* DEBUG */

	/* shift everything over (note that wraparound is impossible) */

	i = cursorpos;
	while (++i < last)
	{
		screen[i-1] = screen[i];

#ifdef EXTENDED
		screen_ex[i-1] = screen_ex[i];
#endif /* EXTENDED */

	}

/*      screen[--i] = (emul7171 ? ' ' : 0); */
	screen[--i] = 0;

#ifdef EXTENDED
	screen_ex[i] = 0;
#endif /* EXTENDED */

#ifdef DEBUG
	(void)fprintf(outf, "delchar     : at (%d,%d): now call update_curscr\n",
		    YX(cursorpos));
#endif /* DEBUG */

	startpos = cursorpos;
	endpos = last;
	update_curscr();
	startpos = -1;
	endpos = -1;
	writecursor();
	return(0);
}

/*
 * Put a character in the current place on the screen.
 * Might be a normal character or a special one (Field Mark, Duplicate).
 * In cases like the Dup function the cursor does a tab. To handle this
 * the move parameter controls how the cursor is moved after putting
 * the character on place.
 * In this implementation of place_char this move parameter is simply
 *      0       Move the cursor to the next character (including autoskip).
 *      1       Tab to the next field.
 *
 * If permitting extended character attributes then the character may
 * have to be put on the screen according to an attribute chosen by the
 * user.
 */

#ifdef STANDARD_C

void    place_char(int c, int move)

#else /* STANDARD_C */

void    place_char(c, move)

int     c;

#endif /* STANDARD_C */

{
	if (!ok_write())
		return;

	startpos = cursorpos;
	if (insert)
	{
		if ((endpos = insertchar((unsigned char)c)) < 0)
		{
			startpos = -1;
			return;
		}
	}
	else
	{

#ifdef EXTENDED
		current_ex = (reply_highlight ? (user_highlight | EX_HIGHLIGHT) : 0);
		if (reply_colour)
		{
			current_ex |= (reply_colour ? (user_colour | EX_COLOUR) : 0);
		}
#endif /* EXTENDED */

		changechar((unsigned char)c);
		endpos = startpos;
	}
	INC(endpos);

#ifdef DEBUG
	(void)fprintf(outf, "place_char  : call of update_curscr: (%d,%d) to (%d,%d)\n",
		  YX(startpos), YX(endpos));
#endif /* DEBUG */

	update_curscr();
	startpos = -1;
	endpos = -1;
	if (move)
	{
		(void)next_field(0, SFprotectedBITS);
	}
	else
	{
		movecursor(1);

		/*
		 * Auto-skip feature:  skip field-definition positions.
		 * Normally just skip to the next field.
		 * However, if the next field is autoskip (protected numeric) then
		 * we should skip to the next unprotected field.
		 */

		if (FORMATTED && (screen[cursorpos] & FIELDSTART))
			(void)next_field(FALSE, 0);
	}
	writecursor();

	return;
}

/* ordinary character (not escape sequence) */

#ifdef STANDARD_C

void    act_nil(int c)

#else /* STANDARD_C */

void    act_nil(c)

int     c;

#endif /* STANDARD_C */

{
	if (iscntrl((char)(c & 0x7f)) && c)   /* unknown control character */
	{
		slfset("NO: ILLEGAL CHARACTER!", 2);
		return;
	}

#ifdef EXTENDED

	/*
	 * Special code for picking up the cursor highlight and colour.
	 * If reply_highlight is TRUE then user_highlight can have in the
	 * lower 4 bits any of :-
	 *      0x0             No highlight choice is set
	 *      oxn (0 < n < 8) Colour has been taken from a different position
	 *      0x8             User has set highlight to field/default
	 *      0xn (n > 8)     User has specified a particular highlight
	 * Similarly for reply_colour and user_colour.
	 */

	{
		int     attrib_pos = cursorpos;

		/*
		 * If on a "special" character move back one position.
		 * After that, if in 7171 mode move backwards over any nulls.
		 */

#ifdef DEBUG
	if (reply_highlight || reply_colour)
		(void)fprintf(outf, "act_nil     : at (%d,%d) screen is 0x%x, screen_ex is 0x%x\n",
			  YX(attrib_pos), screen[attrib_pos], screen_ex[attrib_pos]);
#endif /* DEBUG */

		if (((screen[attrib_pos] & FIELDSTART) == 0)          &&
		    (((screen[attrib_pos] & 0xff) == FIELD_MARK)   ||
		     ((screen[attrib_pos] & 0xff) == DUPLICATE)    ||
		     ((screen[attrib_pos] & 0xff) == SUBSTITUTE)))
			DEC(attrib_pos);

		if (!emul7171)
			while (!(screen[attrib_pos]))
				DEC(attrib_pos);

		if (reply_highlight && !(user_highlight & EX_HIGHLIGHT))
			user_highlight = (screen_ex[attrib_pos] & EX_HBITS);

		if (reply_colour && !(user_colour & EX_COLOUR))
			user_colour = (screen_ex[attrib_pos] & EX_CBITS);

#ifdef DEBUG
	if (reply_highlight || reply_colour)
		(void)fprintf(outf, "act_nil     : at (%d,%d) set user_highlight 0x%x, user_colour 0x%x\n",
			  YX(attrib_pos), user_highlight, user_colour);
#endif /* DEBUG */

	}

#endif /* EXTENDED */

	place_char(c, 0);
}

/* User hit "ENTER" or a PF key.  Send all modified fields.
 * If "key" is the special value -1, then we are responding to the "read
 * buffer" command.  Behave like "act_enter(aid)", but send the whole
 * buffer, nulls and all, starting at the beginning.
 * We can use savebuf for sending this data, but make sure it is big enough!
 * Make sure that it is also not being used for reading from the IBM
 * for data coming in more than one data buffer.
 * A real 3270 also turns off insert mode (if on).
 * A 7171 can have this as an option: we have the option also, according
 * to whether keep_insert is non-zero or not.
 */

#ifdef STANDARD_C

void    act_enter(int key)

#else /* STANDARD_C */

void    act_enter(key)

int     key;

#endif /* STANDARD_C */

{
	register int    savep_notnull;
	register int    c,i;

	int     rbflag = 0;
	int     tso_count;

	slfclear();

#ifdef DEBUG
	(void)fprintf(outf,"act_enter   : savebuf is at 0x%lx\n", savebuf);
#endif /* DEBUG */

	if (key == -1)
	{
		key = aid;
		rbflag = 1;
	}

#ifdef DEBUG
	if (key == ENTERkey)
	{
		(void)fprintf(outf,"act_enter   : [ENTER");
	}
	else
	if (key == NOattentionKEY)
	{
		(void)fprintf(outf,"act_enter   : [NOattention");
	}
	else
	if (key & 0x10)
	{
		(void)fprintf(outf,"act_enter   : [PF%d", key & 0xf);
	}
	else
	{
		(void)fprintf(outf,"act_enter   : [PF%d", 12 + (key & 0xf));
	}
#endif /* DEBUG */

	/* now start filling in a command buffer to send to the host */
	/* cursor address */
	savebuf[1] = byte1(cursorpos);
	savebuf[2] = byte2(cursorpos);
	savep = savebuf + 3;

#ifdef DEBUG
	(void)fprintf(outf,"(%d,%d)]\n",YX(cursorpos));
#endif /* DEBUG */

#ifdef EXTENDED
	 /* Flag that we have not yet sent any extended attribute to the IBM. */

	sent_ex = 0;
#endif /* EXTENDED */

	if (rbflag || !FORMATTED)
	{
		/*
		 * Send the whole buffer.
		 * There seems to be a special treatment of nulls sometimes
		 * when sending unformatted screens.
		 * In some emulators trailing nulls are omitted, whilst nulls
		 * occuring internal to non-nulls are converted to blanks.
		 * This might or might not have something to do with the TSO 7171
		 * fix mentioned below. It might or might not depend on being in
		 * 7171 emulation mode (nulls becoming blanks, that is).
		 * Anyway, for the moment I will strip off trailing nulls from
		 * unformatted screens sent with a user AID byte (but not when
		 * answering an IBM READBUFFER request).
		 */

		int     null_count = 0;

#ifdef DEBUG
	(void)fprintf(outf, "act_enter   : %s: send whole screen\n",
				(rbflag ? "READBUFFER" : "unformatted"));
#endif /* DEBUG */

		normal_chars = normal_blanks = TRUE;
		for (i = 0 ; i < screensize ; i++)
		{
			if (rbflag || screen[i])
			{
				/* non-null character: put in any nulls seen just before it */

				while (null_count)
				{
					send_char(i-1);
					null_count--;
				}
				send_char(i);
			}
			else
			{
				/* null character: just count it for now */

				null_count++;
			}
		}

#ifdef DEBUG
	if (null_count)
		(void)fprintf(outf,"act_enter   : omit %d trailing nulls\n", null_count);
#endif /* DEBUG */

	}
	else
	{
		/* send only the modified fields */

		normal_chars = normal_blanks = TRUE;

		while (!(screen[firstfield] & FIELDSTART))
			INC(firstfield);
		i = firstfield;

#ifdef DEBUG
	(void)fprintf(outf,"act_enter   : send modified fields from (%d,%d)\n", YX(i));
#endif /* DEBUG */

#ifdef SHOWFIELDS
	showfields();
#endif /* SHOWFIELDS */

		do
		{
			/* i is at the start of a field */

#ifdef LOGFILE
			if (i == name_field)
			{
				char    login_line[90];
				char    *header = "User login line : ";
				int     j;

				(void)strcpy(login_line, header);

				for (j = strlen(header); j < 89;)
				{
					if (screen[++i] & FIELDSTART)
						break;
					if (!screen[i] || (!j && (screen[i] == ' ')))
						continue;
					login_line[j++] = screen[i];
				}
				login_line[j] = '\0';
				logit (login_line, 0);
				i = name_field;
				name_field = -2;
			}
#endif /* LOGFILE */

			update_normal(i);

			if (screen[i] & SFmodifiedBITS)
			{
				/* transmit the modified field */
				put_save(SBAchar);
				INC(i);
				put_save(byte1(i));
				put_save(byte2(i));

#ifdef DEBUG
	(void)fprintf(outf,"act_enter   :\t<%d,%d>\n<", YX(i));
#endif /* DEBUG */

				savep_notnull = savep - savebuf;

				/*
				 * We must prepare for the infamous TSO fix if in 7171 mode.
				 * This involves not sending nulls when they are between the field
				 * definition and the end of the line unless there are some non-null
				 * characters as well.
				 * Set tso_count as the number of characters between the field
				 * definition and the end of the line, but only if in 7171 mode
				 */
				tso_count = (emul7171 ? linelen - 1 - (i % linelen) : -1);

				for(; ((c=screen[i]) & FIELDSTART) == 0; i = INCR(i))
				{

					if (c)
					{
						/* We can now switch off the infamous TSO fix. */
						tso_count = -1;

						send_char(i);
						savep_notnull = savep - savebuf;

					}
					else
					{

#ifdef DEBUG
		nulls_out++;
#endif /* DEBUG */

						if (emul7171)
						{
							/*
							 * Here we need the infamous TSO fix.
							 * This requires that we do not send (as blanks) any nulls which
							 * are on the same line as the field definition iff there are
							 * only nulls between the field definition and the end of line.
							 * Therefore, if we are now at the end of line and have only
							 * put nulls in, scrap them.
							 */

							if (!(--tso_count))
							{
								savep = savebuf + savep_notnull;

#ifdef DEBUG
		(void)fprintf(outf,"{%d blanks not sent}", nulls_out);
		nulls_out = 0;
#endif /* DEBUG */

							}
							else
								send_char(i);
						}
					}
				}
				savep = savebuf + savep_notnull;

#ifdef DEBUG
	nulls_out = 0;
	(void)fprintf(outf,">\n");
#endif /* DEBUG */

			}
			else
			{
				/* skip over the field */

#ifdef DEBUG
	(void)fprintf(outf,"act_enter   : skip unmodified field (%d,%d)\n", YX(i));
#endif /* DEBUG */

				do
				{
					INC(i);
				} 
				while (!(screen[i] & FIELDSTART));
			}
		} 
		while (i != firstfield);
	}

	sendnet(savebuf, savep, (unsigned char) key);
	savep = savebuf;
}

/*
 * Send the character of position scr_index of the screen
 * into the buffer to go to the IBM.
 * This character may be a field definition or a simple character
 * (maybe a GE character).
 *
 * If running with extended attributes then we may need to send
 * Start Field Extended, rather than Start field, and also the
 * individual characters may have attributes.
 */

/* ARGSUSED */

#ifdef STANDARD_C

void    send_char(int scr_index)

#else /* STANDARD_C */

void    send_char(scr_index)

int     scr_index;

#endif /* STANDARD_C */

{
	register int    c;

#ifdef EXTENDED
	int             count;
#endif /* EXTENDED */

	c = screen[scr_index];

	if (c & FIELDSTART)
	{

#ifdef DEBUG
	if (nulls_out)
		(void)fprintf(outf,"{%d nulls}",nulls_out);
	nulls_out = 0;
#endif /* DEBUG */

#ifdef EXTENDED
		/* May need to put out SFE rather than SF */

		if (screen_ex[scr_index])
		{
			count = 1 + ((screen_ex[scr_index] & EX_HIGHLIGHT) >> 3)
				  + ((screen_ex[scr_index] & EX_COLOUR) >> 7);

			put_save(SFEchar);
			put_save((unsigned char) count);

#ifdef DEBUG
	(void)fprintf(outf,"\nsend_char   :\t[SFE(%d,%d) (0x%x) : %s%s%s%s]",
	YX(scr_index), c,
	c & SFprotectedBITS ? " PROT" : "",
	c & SFnumericBITS ? " NUM" : "",
	(c & SFvisibleBITS)==SFinvisibleBITS ? " INVISIBLE"
	    : (c & SFvisibleBITS)==SFstandoutBITS ? " BRIGHT"
	    : "",
	c & SFmodifiedBITS ? " MDT" : "");
#endif /* DEBUG */

			if (screen_ex[scr_index] & EX_HIGHLIGHT)
			{
				put_save(EX_T_HIGHLIGHT);
				put_save(send_h[screen_ex[scr_index] & EX_HBITS]);

#ifdef DEBUG
	(void)fprintf(outf,"\nsend_char   :\tHighlight pair (ox%x, 0x%x)\n",
			EX_T_HIGHLIGHT, send_h[screen_ex[scr_index] & EX_HBITS]);
#endif /* DEBUG */

			}

			if (screen_ex[scr_index] & EX_COLOUR)
			{
				put_save(EX_T_COLOUR);
				put_save(send_c[(screen_ex[scr_index] & EX_CBITS) >> 4]);

#ifdef DEBUG
	(void)fprintf(outf,"\nsend_char   :\tColour pair (ox%x, 0x%x)\n",
			EX_T_COLOUR, send_c[(screen_ex[scr_index] & EX_HBITS) >> 4]);
#endif /* DEBUG */

			}

			put_save(EX_T_F_ATTRIB);
			put_save(SIXBITtoEBCDIC[c & ATTRMASK]);
		}
		else
		{
			put_save(SFchar);
			put_save(SIXBITtoEBCDIC[c & ATTRMASK]);

#ifdef DEBUG
	(void)fprintf(outf,"\nsend_char   :\t[SFE(%d,%d) (0x%x) : %s%s%s%s]",
	YX(scr_index), c,
	c & SFprotectedBITS ? " PROT" : "",
	c & SFnumericBITS ? " NUM" : "",
	(c & SFvisibleBITS)==SFinvisibleBITS ? " INVISIBLE"
	    : (c & SFvisibleBITS)==SFstandoutBITS ? " BRIGHT"
	    : "",
	c & SFmodifiedBITS ? " MDT" : "");
#endif /* DEBUG */

		}
		sent_ex = screen_ex[scr_index];

#else /* EXTENDED */

			put_save(SFchar);
			put_save(SIXBITtoEBCDIC[c & ATTRMASK]);

#ifdef DEBUG
	(void)fprintf(outf,"\nsend_char   :\t[SF(%d,%d) (0x%x) : %s%s%s%s]",
	YX(scr_index), c,
	c & SFprotectedBITS ? " PROT" : "",
	c & SFnumericBITS ? " NUM" : "",
	(c & SFvisibleBITS)==SFinvisibleBITS ? " INVISIBLE"
	    : (c & SFvisibleBITS)==SFstandoutBITS ? " BRIGHT"
	    : "",
	c & SFmodifiedBITS ? " MDT" : "");
#endif /* DEBUG */

#endif /* EXTENDED */

		update_normal(scr_index);
	}
	else
	{

#ifdef DEBUG
	if (c)
	{
		if (nulls_out)
			(void)fprintf(outf,"{%d nulls}\n",nulls_out);
		nulls_out = 0;
	}
	else
		nulls_out++;
#endif /* DEBUG */

		if (c)
		{

#ifdef EXTENDED
			if ((screen_ex[scr_index] & (EX_HIGHLIGHT | EX_COLOUR)) &&
			    (screen_ex[scr_index] != sent_ex))
			{
				if ((screen_ex[scr_index] & EX_HBITS) != (sent_ex & EX_HBITS))
				{
					put_save(SAchar);
					put_save(EX_T_HIGHLIGHT);
					put_save(send_h[screen_ex[scr_index] & EX_HBITS]);

#ifdef DEBUG
	(void)fprintf(outf,"\nsend_char   :\tHighlight pair (ox%x, 0x%x)\n",
			EX_T_HIGHLIGHT, send_h[screen_ex[scr_index] & EX_HBITS]);
#endif /* DEBUG */

				}

				if ((screen_ex[scr_index] & EX_CBITS) != (sent_ex & EX_CBITS))
				{
					put_save(SAchar);
					put_save(EX_T_COLOUR);
					put_save(send_c[(screen_ex[scr_index] & EX_CBITS) >> 4]);

#ifdef DEBUG
	(void)fprintf(outf,"\nsend_char   :\tColour pair (ox%x, 0x%x)\n",
			EX_T_COLOUR, send_c[(screen_ex[scr_index] & EX_HBITS) >> 4]);
#endif /* DEBUG */

				}
				sent_ex = screen_ex[scr_index];
			}

#endif /* EXTENDED */

			put_ebcdic(c);

#ifdef DEBUG
	(void)fprintf(outf, "%s", print_ascii((unsigned short)c));
#endif /* DEBUG */

		}
		else
		{
			put_ebcdic(' ');
		}
	}
}


/* refresh the screen: reset cursorpos if called with zero parameter */

/* ARGSUSED */

#ifdef STANDARD_C

void    act_refresh(int how)

#else /* STANDARD_C */

void    act_refresh(how)

int     how;

#endif /* STANDARD_C */

{
	clearnormal();
	slfclear();
	slrefresh();
	if ( FORMATTED && !how )
	{
		movecursor(fieldstart+1-cursorpos);
		startpos = endpos = fieldstart;
	}
	else
	{
		startpos = endpos = 0;
	}

#ifdef DEBUG
	(void)fprintf(outf, "act_refresh : call of update_curscr: (%d,%d) to (%d,%d)\n",
		  YX(startpos), YX(endpos));
#endif /* DEBUG */

	update_curscr();
	startpos = -1;
	endpos = -1;
	putcursor();
}

/* Set any user options */

/* ARGSUSED */

#ifdef STANDARD_C

void    act_option(int optval)

#else /* STANDARD_C */

void    act_option(optval)

int     optval;

#endif /* STANDARD_C */

{
	(void) dialogue((char *)NULL, (char) optval);
	mainreset(cursorsave);
}

/*
 * Save screen dump to a file.
 * This may be combined with automatic printing of the file or
 * with sending it via email to the user's specified email address.
 * Such a combination is only done for the general-purpose version,
 * i.e. when symbol TTY is defined. Even there, it will depend on the
 * host system administrator having implemented the necessary procedure,
 * normally via a crontab shell script procedure.
 */

/* ARGSUSED */

#ifdef STANDARD_C

void    act_print(int dummy)

#else /* STANDARD_C */

void    act_print(dummy)

int     dummy;

#endif /* STANDARD_C */

{
	int     randloop;
	int     filenum;
	int     dirstlen;
	bool    e_mail;

#ifdef TTY
	int     pf_basel;
#endif /* TTY */

#ifdef TTY
	while (!strcmp(print_type, "prt"))
	{
		dialogue("You have not yet set a screen print qualifier for logins to this IBM.\nThis is set in the options menu, to which I will now send you.\nWhen you have set it you may wish to save the settings, so that\nthe next time you logon to this IBM system they will be remembered.\n\n", 'o');
		mainreset(cursorsave);
	}
#endif /* TTY */

	e_mail = !strcmp(print_type, "email");

#ifdef TTY
	while (e_mail && !strcmp(email_addr, "none"))
	{
		dialogue("You have not yet set your preferred email address for logins to this IBM.\nThis is set in the options menu, to which I will now send you.\nWhen you have set it you may wish to save the settings, so that\nthe next time you logon to this IBM system they will be remembered.\n\n", 'o');
		mainreset(cursorsave);
	}
#endif /* TTY */

	print_fstr = (FILE *) NULL;
	for (randloop = 0 ; randloop < 10 ; randloop++)
	{

#ifdef PRINTDIR
		dirstlen = strlen(PRINTDIR);
		strncpy(print_file, set_fname(PRINTDIR, (e_mail ? "@" : (char *)NULL)), P_FSIZE - 1);
#else /* PRINTDIR */
		strcpy(print_file, "./");
		dirstlen = 2;
#endif /* PRINTDIR */

		/*
		 * Check that the print_file array length (P_FSIZE) is large enough.
		 * It needs an optional '@', print_user, ':', print_type or email_addr,
		 * ':' and a 4-digit uniqueness identifier.
		 */

		if ((dirstlen + strlen(print_user) + 7 + strlen(e_mail ? email_addr : print_type)) >= P_FSIZE)
			continue;
		strcat(print_file, print_user);
		strcat(print_file, ":");
		strcat(print_file, (e_mail ? email_addr : print_type));

#ifdef TTY
		pf_basel = strlen(print_file);
#endif /* TTY */

		if (randloop)
		{
			strcat(print_file, ":");

#ifdef SYSV

			sprintf(&(print_file[strlen(print_file)]), "%-i", (rand()) & 0xfff);

#else /* SYSV */

			sprintf(&(print_file[strlen(print_file)]), "%-i", (int)(random()) & 0xfff);

#endif /* SYSV */

		}

#ifdef DEBUG
	(void)fprintf(outf, "act_print   : try print_file as %s\n",
			print_file);
#endif /* DEBUG */

		if ((filenum = open(print_file, O_WRONLY + O_CREAT + O_EXCL, 0666)) < 0)
		{
			/* What a shame: we cannot open the file */

#ifdef DEBUG
	(void)fprintf(outf, "act_print   : open print_file %s fails with error %d\n",
			print_file, filenum);
#endif /* DEBUG */

			if (filenum == EEXIST)
				continue;
		}
		else
		{

#ifdef DEBUG
	(void)fprintf(outf, "act_print   : file %s descriptor is %d\n",
			print_file, filenum);
#endif /* DEBUG */

			if ((print_fstr = fdopen(filenum, "w")) != NULL)
				break;

			/* Cannot pick up the file descriptor: how odd! */

#ifdef DEBUG
	(void)fprintf(outf, "act_print   : cannot get file %s open for write\n",
			print_file);
#endif /* DEBUG */

			close(filenum);
			continue;
		}
	}

	if (print_fstr == (FILE *)NULL)
		slfset("Cannot save printfile!", 2);
	else
	{
		/*
		 * Now we have opened the file all we have to do is to write it!
		 * We can do this from either the screen[] array or the pseudo-curses
		 * structure curscr. The latter is easier but less accurate, particularly
		 * for exotic or underscored characters. Since it is so easy, let us
		 * start that way.
		 */

		int             line;
		unsigned char   *cstart;
		unsigned char   *cendp1;

		for (line = 0 ; line < rows ; line++)
		{
			/* Find last non-blank character */

			cstart = char_image[line];
			for (cendp1 = cstart + linelen ; cendp1 != cstart ; cendp1--)
				if (*(cendp1 - 1) != ' ')
					break;
			while (cstart != cendp1)
			{
				putc(*cstart, print_fstr);
				cstart++;
			}
			putc('\n', print_fstr);
		}
		(void)fclose(print_fstr);
		(void)close(filenum);

#ifdef DEBUG
	(void)fprintf(outf, "act_print   : have saved print_file %s\n",
			print_file);
#endif /* DEBUG */

#ifdef TTY
		print_file[pf_basel] = '\0';
#endif /* TTY */

		slfset(&print_file[dirstlen], 1);
	}
}

/* ARGSUSED */

#ifdef STANDARD_C

void    act_fm(int dummy)

#else /* STANDARD_C */

void    act_fm(dummy)

int     dummy;

#endif /* STANDARD_C */

{
	place_special(FIELD_MARK, 0);
	return;
}

#ifdef STANDARD_C

void    act_dup(int dummy)

#else /* STANDARD_C */

void    act_dup(dummy)

int     dummy;

#endif /* STANDARD_C */

{
	place_special(DUPLICATE, 1);
	return;
}


#ifdef STANDARD_C

void    place_special(int c, int move)

#else /* STANDARD_C */

void    place_special(c, move)

int     c;
int     move;

#endif /* STANDARD_C */

{

#ifdef EXTENDED

	/* In extended attribute mode ensure that this goes out normally */

	unsigned char   save_u_h;
	unsigned char   save_u_c;

	save_u_h = user_highlight;
	save_u_c = user_colour;
	user_highlight = 0;
	user_colour = 0;

#endif /* EXTENDED */

	place_char(c, move);

#ifdef EXTENDED

	/* In extended attribute mode restore extended attribute setting */

	user_highlight = save_u_h;
	user_colour = save_u_c;

#endif /* EXTENDED */

	return;
}

/*
 * User has requested to use a particular extended highlighing mode.
 * This is only valid if this program is being compiled for extended
 * attributes: if not then just refuse.
 * If we are allowing extended attributes then it may be permissible,
 * providing that the application program has authorised this extended
 * input.
 * One problem is that we might not be able to show the user the requested
 * highlighting because the termcap entry does not provide terminal
 * control sequences to handle this. For the moment we will assume that
 * this is not a problem (perhaps by having initially told the IBM that
 * there are some highlighting modes which we cannot showm perhaps because
 * we are happy to tell the IBM about this requested highlighting anyway,
 * perhaps simply that virtually all reasonable termcap entries include
 * the required highlighting modes).
 *
 * The entry parameter is an integer value as follows :-
 *
 *   0  Default, i.e. according to context.
 *   8  As specified by the field definition.
 *   9  Blinking.
 *   10 Reverse video
 *   12 Underscore
 *
 * The difference between the first two is that in the default case any
 * text being inserted will take on the characteristics of the place in
 * which it is being inserted.
 */

#ifdef STANDARD_C

void    act_highlight(int input_highlight)

#else /* STANDARD_C */

void    act_highlight(input_highlight)

int     input_highlight;

#endif /* STANDARD_C */

{

#ifdef EXTENDED

	/* Check if IBM has permitted this! */

	if (reply_highlight)
	{
		user_highlight = input_highlight;
		slfset(mode_h[input_highlight], 1);
	}
	else
	{
		slfset("Not permitted now!", 2);
	}

#else /* EXTENDED */

	slfset("unimplemented!", 2);

#endif /* EXTENDED */

}

/*
 * User has requested to use a particular extended colour.
 * This is only valid if this program is being compiled for extended
 * attributes: if not then just refuse.
 * If we are allowing extended attributes then it may be permissible,
 * providing that the application program has authorised this extended
 * input.
 * We assume that we can put out colour: if we cannot then we should
 * have told the IBM ages ago!
 *
 * The entry parameter is an integer value as follows :-
 *
 *   0  Default, i.e. according to context.
 *   8  As specified by the field definition.
 *   9  Blue.
 *   10 Red.
 *   11 Pink.
 *   12 Green.
 *   13 Turquoise.
 *   14 Yellow.
 *   15 Neutral (black on white or white on black).
 *
 * The difference between the first two is that in the default case any
 * text being inserted will take on the characteristics of the place in
 * which it is being inserted.
 *
 */

#ifdef STANDARD_C

void    act_colour(int input_colour)

#else /* STANDARD_C */

void    act_colour(input_colour)

int     input_colour;

#endif /* STANDARD_C */

{

#ifdef EXTENDED

	/* Check if IBM has permitted this! */

	if (reply_colour)
	{
		user_colour = (input_colour << 4);
		slfset(mode_c[input_colour], 1);
	}
	else
	{
		slfset("Not permitted now!", 2);
	}

#else /* EXTENDED */

	slfset("unimplemented!", 2);

#endif /* EXTENDED */

}

/*
 * Insert a character at the current cursorpos, shifting right everything up
 * to the first null (or blank if 7171 emulation) in the current field.
 * Note that the updating of curscr is NOT done here.
 */

#ifdef STANDARD_C

int     insertchar(unsigned char c)

#else /* STANDARD_C */

int     insertchar(c)

unsigned char c;

#endif /* STANDARD_C */

{
	register int last, cp, next;

	/* scan to the right, looking for a null or end-of-field  */
	/* or, in the case of 7171 emulation, the end of the line */
	/* The idea is to find the position last which will be    */
	/* the place to which the last significant character will */
	/* be shifted. This should be either a null if real 3270  */
	/* emulation or the first of the trailing blanks in the   */
	/* line or field if 7171 emulation                        */

	last = next = cursorpos;
	DEC(last);
	while (screen[next])
	{
		if (screen[next] & FIELDSTART)
			break;
		if (emul7171)
		{
			if (screen[next] != ' ')
				last = next;
			if (((next = INCR(next)) % linelen) == 0)
				break;
		}
		else
		{
			last = next;
			if ((next = INCR(next)) == cursorpos)
				break;
		}
	}
	if (((last = INCR(last)) == next) && screen[next])
	{
		slfset("NO MORE INSERTING!", 2);
		return(-1);
	}

	if (FORMATTED)
		screen[fieldstart] |= SFmodifiedBITS;

	/* shift things down, making room */

	cp = cursorpos;
	next = cursorpos = last;
	DEC(next);
	while (cursorpos != cp)
	{
		/* note: cursorpos is an implicit parameter to changechar() */

#ifdef EXTENDED
		current_ex = screen_ex[next];
#endif /* EXTENDED */

		changechar((unsigned char)(screen[next]));

		cursorpos = next;
		DEC(next);
	}
	/* insert the character */

#ifdef EXTENDED
	current_ex = (reply_highlight ? (user_highlight | EX_HIGHLIGHT) : 0);
	if (reply_colour)
	{
		current_ex |= (reply_colour ? (user_colour | EX_COLOUR) : 0);
	}
#endif /* EXTENDED */

	changechar(c);
	return(last);
}

/* routines which will only be called if in action key help mode */

#ifdef STANDARD_C

void    help_act(int act_index)

#else /* STANDARD_C */

void    help_act(act_index)

int     act_index;

#endif /* STANDARD_C */

{
	if (act_index < 0)
	{
		int     c;

		f_puts( "\r\n", FALSE);
		if (iscntrl((c = -act_index) & 0x7f))
		{
			f_puts( print_ascii((unsigned short)c), FALSE);
			if (c & 0x80)
				f_puts( " : an 8-bit control character!\r\n", FALSE);
			else
				f_puts( " : a 7-bit control character!\r\n", FALSE);
		}
		else
		{
			if (c & 0x80)
			{
				f_puts( SO_string[1], FALSE);
				f_putc( (char)c);
				f_puts( SI_string[1], FALSE);
				f_puts( " (", FALSE);
				f_puts( print_ascii((unsigned short)c), FALSE);
				f_puts( ") : an extended ASCII input character\r\n", FALSE);
			}
			else
				f_putf( "%c : a normal input character\r\n", FALSE, c);
		}
	}
	else
	{
		f_puts( "\r\nThis corresponds to <", FALSE);
		f_puts( actions[act_index].act_hmess, FALSE);
		f_puts( ">\r\n", FALSE);
		if (actions[act_index].arg == 999)
			f_puts( "However, it is not (yet) implemented in this emulator\r\n", FALSE);
		else
		{
			if (actions[act_index].immediate < 0)
				f_puts( "It would be obeyed even if the IBM had locked the keyboard input!\r\n", TRUE);
			if (actions[act_index].immediate)
				f_puts( "It would not cause anything to be sent to the IBM\r\n", FALSE);
		}
		if (!strcmp(actions[act_index].name , "normal"))
		{
			f_puts("The help procedure will now assume standard ASCII\r\n", FALSE);
			SO_imask = 0;
		}
		if (!strcmp(actions[act_index].name , "extend"))
		{
			f_puts("The help procedure will now assume extended ASCII\r\n", FALSE);
			SO_imask = 0x80;
		}
	}
	f_flush();
}
