/*
 * refclock_goes - clock driver for the Kinemetrics Truetime GOES
 *	Receiver Version 3.0B - NOT tested with STREAMS or CLKLDISC
 */

#if defined(REFCLOCK) && defined(GOES)

#include <stdio.h>
#include <ctype.h>
#include <sys/time.h>

#include "ntpd.h"
#include "ntp_io.h"
#include "ntp_refclock.h"
#include "ntp_unixtime.h"
#include "ntp_stdlib.h"

/*
 * Support for Kinemetrics Truetime 468-DC GOES Receiver
 *
 * Most of this code is copied from refclock_wwvb.c with thanks.
 *
 * The time code looks like follows; Send the clock a 'R' or 'C' and
 * once per second a timestamp will appear that looks like this:
 *
 * ADDD:HH:MM:SSQCL
 * A - control A
 * Q Quality indication: indicates possible error of
 *     ?     +/- 500 milliseconds	#     +/- 50 milliseconds
 *     *     +/- 5 milliseconds		.     +/- 1 millisecond
 *   space   less than 1 millisecond
 * C - Carriage return
 * L - Line feed
 *
 * The carriage return start bit begins on 0 seconds and extends to 1
 * bit time. Unless you live on 125 degrees west longitude, you can't
 * set your clock propagation delay settings correctly and still use
 * automatic mode. The manual says to use a compromise when setting the
 * switches. This results in significant errors. The solution; use fudge
 * time1 and time2 to incorporate corrections. If your clock is set for
 * 50 and it should be 58 for using the west and 46 for using the east,
 * use the line
 *
 * fudge 127.127.5.0 time1 +0.008 time2 -0.004
 *
 * This corrects the 4 milliseconds advance and 8 milliseconds retard
 * needed. The software will ask the clock which satellite it sees.
 *
 * time1 - offset applied to samples when reading WEST satellite (default = 0)
 * time2 - offset applied to samples when reading EAST satellite (default = 0)
 * val1  - stratum to assign to this clock (default = 0)
 * val2  - refid assigned to this clock (default = "GOES", see below)
 * flag1 - will silence the clock side of xntpd, just reading the clock
 *         without trying to write to it.  (default = 0)
 * flag2 - not assigned
 * flag3 - enable ppsclock streams module
 * flag3 - not assigned
 *
 * This driver is being tested...
 */

/*
 * Definitions
 */
#define	DEVICE	"/dev/goes%d"
#define	SPEED232	B9600	/* 9600 baud */

/*
 * Radio interface parameters
 */
#define	MAXDISPERSE	(FP_SECOND>>1) /* max error for synchronized clock (0.5 s as an u_fp) */
#define	PRECISION	(-10)	/* precision assumed (about 1 ms) */
#define	REFID		"GOES"	/* reference id */
#define	DESCRIPTION	"TrueTime GPS/GOES Receivers" /* WRU */
#define	NSAMPLES	3	/* stages of median filter */
#define	LENGOES0	13	/* Timecode format 0 length */
#define	LENGOES2	21	/* Timecode format 2 length */

/*
 * Tags which station (satellite) we see
 */
#define GOES_WEST   0	/* Default to WEST satellite and apply time1 */
#define GOES_EAST   1	/* until you discover otherwise */

/*
 * Imported from the timer module
 */
extern u_long current_time;

/*
 * Imported from ntpd module
 */
extern int debug;		/* global debug flag */

/*
 * unit control structure
 */
struct goesunit {
	int	pollcnt;	/* poll message counter */
	u_short station;	/* which station we are on */
	u_short polled;		/* Hand in a time sample? */
};

/*
 * Function prototypes
 */
static	int	goes_start	P((int, struct peer *));
static	void	goes_shutdown	P((int, struct peer *));
static	void	goes_receive	P((struct recvbuf *));
static	void	goes_poll	P((int, struct peer *));
static	void	goes_send	P((struct peer *, char *));

/*
 * Transfer vector
 */
struct	refclock refclock_goes = {
	goes_start,		/* start up driver */
	goes_shutdown,		/* shut down driver */
	goes_poll,		/* transmit poll message */
	noentry,		/* not used (old goes_control) */
	noentry,		/* initialize driver (not used) */
	noentry,		/* not used (old goes_buginfo) */
	NOFLAGS			/* not used */
};


/*
 * goes_start - open the devices and initialize data for processing
 */
static int
goes_start(unit, peer)
	int unit;
	struct peer *peer;
{
	register struct goesunit *up;
	struct refclockproc *pp;
	int fd;
	char device[20];

	/*
	 * Open serial port
	 */
	(void)sprintf(device, DEVICE, unit);
	if (!(fd = refclock_open(device, SPEED232, LDISC_CLK)))
		return (0);

	/*
	 * Allocate and initialize unit structure
	 */
	if (!(up = (struct goesunit *)
	    emalloc(sizeof(struct goesunit)))) {
		(void) close(fd);
		return (0);
	}
	memset((char *)up, 0, sizeof(struct goesunit));
	pp = peer->procptr;
	pp->io.clock_recv = goes_receive;
	pp->io.srcclock = (caddr_t)peer;
	pp->io.datalen = 0;
	pp->io.fd = fd;
	if (!io_addclock(&pp->io)) {
		(void) close(fd);
		free(up);
		return (0);
	}
	pp->unitptr = (caddr_t)up;

	/*
	 * Initialize miscellaneous variables
	 */
	peer->precision = PRECISION;
	pp->clockdesc = DESCRIPTION;
	memcpy((char *)&pp->refid, REFID, 4);
	up->pollcnt = 2;
	return (1);
}


/*
 * goes_shutdown - shut down the clock
 */
static void
goes_shutdown(unit, peer)
	int unit;
	struct peer *peer;
{
	register struct goesunit *up;
	struct refclockproc *pp;

	pp = peer->procptr;
	up = (struct goesunit *)pp->unitptr;
	io_closeclock(&pp->io);
	free(up);
}


/*
 * goes_receive - receive data from the serial interface on a
 * Kinimetrics clock
 */
static void
goes_receive(rbufp)
	struct recvbuf *rbufp;
{
	register struct goesunit *up;
	struct refclockproc *pp;
	struct peer *peer;
	l_fp tmp_l_fp;
	u_short new_station;
	char syncchar;
	int loccode;

	/*
	 * Get the clock this applies to and a pointers to the data
	 */
	peer = (struct peer *)rbufp->recv_srcclock;
	pp = peer->procptr;
	up = (struct goesunit *)pp->unitptr;

	/*
	 * pick up STREAMS or CLKLDISC timestamps and strip control chars.
	 */
	pp->lencode = refclock_gtlin(rbufp, pp->lastcode, BMAX,
		&pp->lastrec);
	if (pp->lencode == 0)
		return;

	up->pollcnt = 2;
	record_clock_stats(&peer->srcadr, pp->lastcode);

	/*
	 * We get down to business, check the timecode format and decode
	 * its contents. This code checks for and decodes both format 0
	 * and format 2 and need not be told which in advance.
	 */
	switch (pp->lencode) {

		case LENGOES0:

		/*
	 	 * Timecode format 0: "ddd:hh:mm:ssQ"
	 	 */
		if (sscanf(pp->lastcode, "%3d:%2d:%2d:%2d%c",
		    &pp->day, &pp->hour, &pp->minute,
		    &pp->second, &syncchar) == 5)
			break;

		case LENGOES2:

		/*
		 * Timecode format 2: "NNN.NNNN+N.NNNN+NNN.N"
		 */
		if (sscanf(pp->lastcode,
			"%1d%*2d.%*4d%*c%1*d.%*4d%*c%*3d.%*1d",
			&loccode) == 1) {

			/*
			 * This is less than perfect.  Call the (satellite)
			 * either EAST or WEST and adjust slop accodingly
			 * Perfectionists would recalcuted the exact delay
			 * and adjust accordingly...
			 */
			if (loccode == 0)
				new_station = GOES_EAST;
			else
				new_station = GOES_WEST;
#ifdef DEBUG
			if (debug) {
				if (new_station == GOES_EAST)
					printf("goes: station EAST\n");
				if (new_station == GOES_WEST)
					printf("goes: station WEST\n");
			}
#endif
			if (new_station != up->station) {
				tmp_l_fp = pp->fudgetime1;
				pp->fudgetime1 = pp->fudgetime2;
				pp->fudgetime2 = tmp_l_fp;
				up->station = new_station;
			}

			/*
			 * Switch back to on-second time codes.
			 */
			goes_send(peer, "C");

			/*
			 * Since this is not a time code, just return...
			 */
			return;
		}

		default:

		refclock_report(peer, CEVNT_BADREPLY);
		return;
	}

	/*
	 * Adjust the synchronize indicator according to timecode
	 */
	if (syncchar !=' ' && syncchar !='.' && syncchar !='*')
		pp->leap = LEAP_NOTINSYNC;
	else {
		pp->leap = 0;
		pp->lasttime = current_time;
	}

	/*
	 * The clock will blurt a timecode every second but we only
	 * want one when polled.  If we havn't been polled, bail out.
	 */
	if (!up->polled)
		return;

	/*
	 * After each poll, check the station (satellite)
	 */
	goes_send(peer, "E");

	/*
	 * Process the new sample in the median filter and determine the
	 * reference clock offset and dispersion. We use lastrec as both
	 * the reference time and receive time in order to avoid being
	 * cute, like setting the reference time later than the receive
	 * time, which may cause a paranoid protocol module to chuck out
	 * the data.
 	 */
	if (!refclock_process(pp, NSAMPLES, NSAMPLES)) {
		refclock_report(peer, CEVNT_BADTIME);
		return;
	}
	refclock_receive(peer, &pp->offset, 0, pp->dispersion,
	    &pp->lastrec, &pp->lastrec, pp->leap);

	/*
	 * We have succedded in answering the poll.  Turn off the flag
	 */
	up->polled = 0;
}


/*
 * goes_send - time to send the clock a signal to cough up a time sample
 */
static void
goes_send(peer, cmd)
	struct peer *peer;
	char *cmd;
{
	struct refclockproc *pp;
	register int len = strlen(cmd);

	pp = peer->procptr;
	if (!pp->sloppyclockflag & CLK_FLAG1) {
#ifdef DEBUG
		if (debug)
			printf("goes: Send '%s'\n", cmd);
#endif
		if (write(pp->io.fd, cmd, len) != len) {
			refclock_report(peer, CEVNT_FAULT);
		} else {
			pp->polls++;
		}
	}
}


/*
 * goes_poll - called by the transmit procedure
 */
static void
goes_poll(unit, peer)
	int unit;
	struct peer *peer;
{
	struct goesunit *up;
	struct refclockproc *pp;

	/*
	 * You don't need to poll this clock.  It puts out timecodes
	 * once per second.  If asked for a timestamp, take note.
	 * The next time a timecode comes in, it will be fed back.
	 */
	pp = peer->procptr;
	up = (struct goesunit *)pp->unitptr;
	if (up->pollcnt == 0)
		refclock_report(peer, CEVNT_TIMEOUT);
	else
		up->pollcnt--;

	/*
	 * polled every 64 seconds. Ask goes_receive to hand in a
	 * timestamp.
	 */
	up->polled = 1;
	pp->polls++;

	goes_send(peer, "C");
}

#endif
