/*
 * ntp_loopfilter.c - implements the NTP loop filter algorithm
 */
#include <stdio.h>
#include <ctype.h>
#include <sys/time.h>
#include <signal.h>

#include "ntpd.h"
#include "ntp_io.h"
#include "ntp_unixtime.h"
#include "ntp_stdlib.h"

#ifdef KERNEL_PLL
#include "sys/timex.h"
#define ntp_gettime(t)  syscall(SYS_ntp_gettime, (t))
#define ntp_adjtime(t)  syscall(SYS_ntp_adjtime, (t))
#endif /* KERNEL_PLL */

/*
 * The loop filter is implemented in slavish adherence to the
 * specification (Section 5), except that for consistency we
 * mostly carry the quantities in the same units as appendix G.
 *
 * Note that the long values below are the fractional portion of
 * a long fixed-point value.  This limits these values to +-0.5
 * seconds.  When adjustments are capped inside this range (see
 * CLOCK_MAX_{I,F}) both the clock_adjust and the compliance
 * registers should be fine. (When the compliance is above 16, it
 * will at most accumulate 2 ** CLOCK_MULT times the maximum offset,
 * which means it fits in a s_fp.)
 *
 * The skew compensation is a special case. In version 2, it was
 * kept in ms / 4s (i.e., CLOCK_FREQ was 10). In version 3 (Section 5)
 * it seems to be 2 ** -16ms / 4s in a s_fp for a maximum of +-125ppm
 * (stated maximum 100ppm). Since this seems about to change to a
 * larger range, it will be kept in units of 2 ** -20 (CLOCK_DSCALE)
 * in an s_fp (mainly because that's nearly the same as parts per
 * million). Note that this is ``seconds per second'', whereas a
 * clock adjustment is a 32-bit fraction of a second to be applied
 * every 2 ** CLOCK_ADJ seconds; to find it, shift the drift right by
 * (CLOCK_DSCALE - 16 - CLOCK_ADJ). When updating the drift, on the
 * other hand, the CLOCK_FREQ factor from the spec assumes the value to
 * be in ``seconds per 4 seconds''; to get our units, CLOCK_ADJ must be
 * added to the shift.
 *
 * Kernel PLL/PPS state machine
 *
 * The following state machine is used when the kernel PLL modifications
 * described in the README.kernel file are present. The initial
 * configuration routine loop_config() sets up the initial frequency
 * estimate and tests if the kernel modifications are present. If so and
 * the PLL mode bit 1 (STA_PLL) of the mode word in the configuration
 * file is set, pll_control is set true and the kernel pll is enabled.
 * If the kernel modifications are present and the PLL mode bit 2
 * (STA_PPSFREQ) is set, the kernel PPS frequency discipline is enabled.
 *
 * Each update to a prefer peer sets pps_update to the current time. If
 * the current time is later than pps_update by PPS_MAXUPDATE (600 s),
 * this variable is set to zero. The PPS time discipline is enabled
 * (STA_PPSTIME bit set in the status word) when pps_control is true and
 * pps_update is nonzero. If the kernel reports the PPS time discipline
 * is enabled and a PPS signal is present, the pps_time variable is set
 * to the current time. If the current time is later than pps_time by
 * PPS_MAXAGE (120 s), this variable is set to zero. If both pps_update
 * and pps_time are nonzero, the pps_control is set true; otherwise, it
 * is set false.
 *
 * The pll_enable switch can be set both at configuration time and at run time using xntpdc. If true, the kernel modifications are active as described above; if false the kernel is bypassed entirely (except for the PPS frequency update, if enabled) and the daemon PLL used instead. 
 */
#define RSH_DRIFT_TO_FRAC (CLOCK_DSCALE - 16)
#define RSH_DRIFT_TO_ADJ (RSH_DRIFT_TO_FRAC - CLOCK_ADJ)
#define RSH_FRAC_TO_FREQ (CLOCK_FREQ + CLOCK_ADJ - RSH_DRIFT_TO_FRAC)
#define PPS_MAXAGE 120		/* kernel pps signal timeout (s) */
#define PPS_MAXUPDATE 600	/* prefer peer timeout (s) */
#define	STAB_AVG 2		/* stability time constant */

/*
 * Program variables
 */
	l_fp last_offset;	/* last clock offset */
static	long last_adjustment;	/* last clock adjustment */
	long last_time;		/* time of last clock update (s) */
	u_fp clock_stability;	/* clock stability (ppm) */
 	s_fp clock_frequency;	/* clock frequency error (ppm) */
	s_fp drift_comp;	/* pll frequency (ppm) */
static	long clock_adjust;	/* clock adjust (fraction only) */
static	s_fp max_comp;		/* max frequency offset (ppm) */

	int time_constant;	/* log2 of time constant (0 .. 4) */
static	u_long tcadj_time;	/* last time-constant adjust time */

static	int first_adjustment;	/* true after first adjustment */
static	int tc_counter;		/* time-constant hold counter */

	int pps_control;	/* true if working kernel pps signal */
	int pll_control;	/* true if working kernel pll */
	int pll_enable;		/* true if pll enabled */
	int pll_status;		/* pll status bits */
	u_long pps_time;	/* last pps sample time */
	u_long pps_update;	/* last pps update time */
	int fdpps = -1;		/* pps file descriptor */

/*
 * Imported from the ntp_proto module
 */
extern u_char sys_stratum;
extern s_fp sys_rootdelay;
extern u_fp sys_rootdispersion;
extern u_long sys_refid;
extern s_char sys_precision;

/*
 * Imported from ntp_io.c
 */
extern struct interface *loopback_interface;

/*
 * Imported from ntpd module
 */
extern int debug;		/* global debug flag */

/*
 * Imported from timer module
 */
extern u_long current_time;	/* like it says, in seconds */

/*
 * sys_poll and sys_refskew are set here
 */
extern u_char sys_poll;		/* log2 of system poll interval */
extern u_char sys_leap;		/* system leap bits */
extern l_fp sys_refskew;	/* accumulated skew since last update */
extern u_fp sys_maxd;		/* max dispersion of survivor list */

/*
 * Imported from leap module
 */
extern u_char leapbits;		/* sanitized leap bits */

#if defined(KERNEL_PLL)
#define MOD_BITS (MOD_OFFSET | MOD_MAXERROR | MOD_ESTERROR | \
    MOD_STATUS | MOD_TIMECONST)
extern int sigvec	P((int, struct sigvec *, struct sigvec *));
extern int syscall	P((int, void *, ...));
void pll_trap		P((void));

static struct sigvec sigsys;	/* current sigvec status */
static struct sigvec newsigsys;	/* new sigvec status */
#endif /* KERNEL_PLL */

/*
 * init_loopfilter - initialize loop filter data
 */
void
init_loopfilter()
{
	extern u_long tsf_maxslew;
	u_long tsf_limit;

	drift_comp = 0;
	time_constant = 0;
	tcadj_time = 0;
	tc_counter = 0;

	/*
	 * Limit for drift_comp, minimum of two values. The first is to
	 * avoid signed overflow, the second to keep within 75% of the
	 * maximum adjustment possible in adj_systime().
	 */
	max_comp = 0x7fff0000;
	tsf_limit = ((tsf_maxslew >> 1) + (tsf_maxslew >> 2));
	if ((max_comp >> RSH_DRIFT_TO_ADJ) > tsf_limit)
		max_comp = tsf_limit << RSH_DRIFT_TO_ADJ;

	/*
	 * Reset clockworks
	 */
	clock_adjust = 0;
	L_CLR(&last_offset);
	clock_stability = max_comp;
	first_adjustment = 0;
	pps_update = pps_time = pps_control = 0;
}

/*
 * local_clock - the NTP logical clock loop filter.  Returns 1 if the
 *	clock was stepped, 0 if it was slewed and -1 if it is hopeless.
 */
int
local_clock(fp_offset, peer)
	l_fp *fp_offset;		/* best offset estimate */
	struct peer *peer;		/* from peer - for messages */
{
	register long offset;
	long tmp, ltmp;
	l_fp ftmp;
	s_fp stmp;
	int isneg;
#if defined(KERNEL_PLL)
	struct timex ntv;
#endif /* KERNEL_PLL */

#ifdef DEBUG
	if (debug > 1)
		printf("local_clock(%s, %s)\n", lfptoa(fp_offset, 6),
		    ntoa(&peer->srcadr));
#endif

	/*
	 * Take the absolute value of the offset
	 */
	ftmp = *fp_offset;
	if (L_ISNEG(&ftmp)) {
		L_NEG(&ftmp);
		isneg = 1;
	} else
		isneg = 0;

	/*
	 * If the clock is way off, don't tempt fate by correcting it.
	 */
	if (ftmp.l_ui >= CLOCK_WAYTOOBIG) {
		syslog(LOG_ERR,
		"clock correction %s way too large (set clock manually)",
		   lfptoa(fp_offset, 6));
#ifndef BIGTIMESTEP
		return (-1);
#endif /* BIGTIMESTEP */
	}

	/*
	 * Save this offset for later perusal
	 */
	last_offset = *fp_offset;

	/*
	 * If the magnitude of the offset is greater than CLOCK.MAX,
	 * step the time and reset the clockworks, but only if the
	 * time since the last update is greater than CLOCK_MINSTEP
	 * (900 s). This prevents needless lurches when timekeeping is
	 * spiky. However, resist this coy in the case of the ACTS
	 * driver, since its updates are few and expensive.
	 */
	if (ftmp.l_ui > CLOCK_MAX_I || (ftmp.l_ui == CLOCK_MAX_I
	    && ftmp.l_uf >= CLOCK_MAX_F)) {
		if (current_time - last_time < CLOCK_MINSTEP &&
		    peer->refclktype != REFCLK_NIST_ACTS) {
			syslog(LOG_INFO,
		"clock correction %s too large (ignored)\n",
			    lfptoa(fp_offset, 6));
			return (0);
		}
		syslog(LOG_NOTICE, "clock reset (%s) %s\n",
#ifdef SLEWALWAYS
		    "slew",
#else
		    "step",
#endif
		    lfptoa(fp_offset, 6));
		step_systime(fp_offset);

		/*
		 * Reset clockworks
		 */
		clock_adjust = 0;
		L_CLR(&last_offset);
		clock_stability = max_comp;
		first_adjustment = 0;
		pps_update = pps_time = pps_control = 0;
		return (1);
	}

	/*
	 * The magnitude of the offset is less than CLOCK_MAX. Update
	 * the loop variables. Also, calculate the estimated frequency
	 * offset and frequency stability. This is useful when the
	 * update interval exceeds CLOCK_MAXSEC, like when the ACTS
	 * driver is in charge. We also need the stability for the
	 * multicast stuff. The result is scaled as a s_fp in ppm,
	 * because we know more than we should.
	 */
	offset = last_offset.l_f;
	if (first_adjustment && current_time > last_time + 16) {
		stmp = (offset / (long)(current_time - last_time)) << 4;
		if (-max_comp < stmp && stmp < max_comp) {
			clock_frequency = stmp;
			if (stmp < 0)
				stmp = -stmp;
			stmp -= clock_stability;
			if (stmp < 0)
				clock_stability -= -stmp >> STAB_AVG;
			else
				clock_stability += stmp >> STAB_AVG;
		}
	}

	if (pll_control && pll_enable) {
#if defined(KERNEL_PLL)
		l_fp pps_offset;
		u_fp pps_dispersion;

		/*
		 * This code segment is active when the phase-lock code
		 * is implemented in the kernel, which at present is
		 * only in the (modified) SunOS 4.1, Ultrix 4.3 and
		 * OSF/1 kernels. In the case of the DECstation 5000/240
		 * and Alpha AXP, additional kernel modifications
		 * provide a true microsecond clock. We know the scaling
		 * of the frequency variable (s_fp) is the same as the
		 * kernel variable (1 << SHIFT_USEC = 16).
		 *
		 * We initialize the structure for the ntp_adjtime()
		 * system call. We have to convert everything to
		 * microseconds first. Afterwards, remember the
		 * frequency offset for the drift file.
		 */
		memset((char *)&ntv,  0, sizeof ntv);
		ntv.modes = MOD_BITS;
		if (offset >= 0) {
			TSFTOTVU(offset, ntv.offset);
		} else {
			TSFTOTVU(-offset, ntv.offset);
			ntv.offset = -ntv.offset;
		}
		TSFTOTVU(sys_rootdispersion + sys_rootdelay / 2,
		    ntv.maxerror);
		TSFTOTVU(sys_rootdispersion, ntv.esterror);
		ntv.status = pll_status & (STA_PLL | STA_PPSFREQ);
		if (pps_update)
			ntv.status |= STA_PPSTIME;
		if (sys_leap & LEAP_ADDSECOND &&
		    sys_leap & LEAP_DELSECOND)
			ntv.status |= STA_UNSYNC;
		else if (sys_leap & LEAP_ADDSECOND)
			ntv.status |= STA_INS;
		else if (sys_leap & LEAP_DELSECOND)
			ntv.status |= STA_DEL;
		ntv.constant = time_constant;
		(void)ntp_adjtime(&ntv);
		drift_comp = ntv.freq;
		pll_status = ntv.status;

		/*
		 * If the kernel pps discipline is working, monitor its
		 * performance.
		 */
		if (pll_status & STA_PPSTIME && pll_status &
		    STA_PPSSIGNAL && ntv.shift) {
			if (ntv.offset >= 0)
				TVUTOTSF(ntv.offset, offset);
			else {
				TVUTOTSF(-ntv.offset, offset);
				offset = -offset;
			}
			L_CLR(&pps_offset);
			L_ADDF(&pps_offset, offset);
			TVUTOTSF(ntv.jitter, tmp);
			pps_dispersion = (tmp >> 16) & 0xffff;
			pps_time = current_time;
			record_peer_stats(&loopback_interface->sin,
			    ctlsysstatus(), &last_offset, 0,
			    pps_dispersion);
		}
#endif /* KERNEL_PLL */

	} else if (current_time - last_time < CLOCK_MAXSEC) {

	 	/*
		 * Here the interval between corrections is less than
		 * CLOCK_MAXSEC. In this regime we use a phase-lock loop
		 * to compute new values of phase and frequency. The
		 * bandwidth is controlled by the time constant, which
		 * is adjusted in response to the phase error and
		 * dispersion. Note the frequency is not changed if the
		 * local clock or acts drivers are in
		 * control.
		 */ 
		if (offset < 0)
			clock_adjust = -((-offset) >> time_constant);
		else
			clock_adjust = offset >> time_constant;
		if (first_adjustment && current_time > last_time &&
		    !(peer->refclktype == REFCLK_LOCALCLOCK ||
		    peer->refclktype == REFCLK_NIST_ACTS)) {
			tmp = peer->maxpoll;
			ltmp = current_time - last_time;
			while (ltmp < (1 << peer->maxpoll)) {
				tmp--;
				ltmp <<= 1;
			}
			tmp = (RSH_FRAC_TO_FREQ - tmp) +
			    time_constant + time_constant;
			if (offset < 0)
				tmp = -(-offset >> tmp);
			else
				tmp = offset >> tmp;
			drift_comp += tmp;
			if (drift_comp > max_comp)
				drift_comp = max_comp;
			else if (drift_comp < -max_comp)
				drift_comp = -max_comp;
		}
	} else {

		/*
		 * Here the interval between corrections is greater than
		 * CLOCK_MAXSEC. In this regime we use a frequency-lock
		 * loop to compute new values of phase and frequency.
		 * The following code is based on ideas suggested by
		 * Judah Levine of NIST and used in his lockclock
		 * implementation of ACTS.
		 */
		clock_adjust = offset;
		if (first_adjustment && current_time > last_time) {
			if (clock_frequency < 0)
				drift_comp -= -clock_frequency >>
				    CLOCK_FWGT;
			else
				drift_comp += clock_frequency >>
				    CLOCK_FWGT;
			if (drift_comp > max_comp)
				drift_comp = max_comp;
			else if (drift_comp < -max_comp)
				drift_comp = -max_comp;
		}
	}
	last_adjustment = offset;
	last_time = current_time;
	first_adjustment = 1;

	/*
	 * Determine when to adjust the time constant and poll interval.
	 * We do this regardless of what source controls the loop, since
	 * we might flap back and forth between sources.
	 */
	if (pps_control)
		time_constant = 0;
	else if (current_time - tcadj_time >= (1 << sys_poll)) {
		tcadj_time = current_time;
		tmp = offset;
		if (tmp < 0)
			tmp = -tmp;
		tmp = tmp >> (16 + CLOCK_WEIGHTTC - time_constant);
		if (tmp > sys_maxd) {
			tc_counter = 0;
			time_constant--;
		} else {
			tc_counter++;
			if (tc_counter > CLOCK_HOLDTC) {
				tc_counter = 0;
				time_constant++;
			}
		}
		if (time_constant < (int)(peer->minpoll - NTP_MINPOLL))
			time_constant = peer->minpoll - NTP_MINPOLL;
		if (time_constant > (int)(peer->maxpoll - NTP_MINPOLL))
			time_constant = peer->maxpoll - NTP_MINPOLL;
	}
	sys_poll = (u_char)(NTP_MINPOLL + time_constant);

#ifdef DEBUG
	if (debug > 1)
	    printf("adjust %s, freq %s, tau %3i\n",
	    	mfptoa((clock_adjust < 0 ? -1:0), clock_adjust, 6),
	    	fptoa(drift_comp, 6), time_constant);
#endif /* DEBUG */

	(void) record_loop_stats(&last_offset, &drift_comp,
	     time_constant);
	
	/*
	 * Whew.  I've had enough.
	 */
	return (0);
}


/*
 * adj_host_clock - Called every 2 ** CLOCK_ADJ seconds to update host
 *	 clock
 */
void
adj_host_clock()
{
	register long adjustment;

	/*
	 * Update the dispersion since the last update. Don't allow
	 * frequency measurements over periods longer than NTP_MAXAGE
	 * (86400 s = one day).
	 */
	if (current_time - last_time >= NTP_MAXAGE)
		first_adjustment = 0;
	L_ADDUF(&sys_refskew, NTP_SKEWINC);

	/*
	 * Declare PPS kernel sync if the prefer peer has been heard
	 * during the last five minutes and the kernel pps signal has
	 * been heard during the last two minutes.
	 */
	if (pps_time && current_time - pps_time > PPS_MAXAGE)
		pps_time = 0;
	if (pps_update && current_time - pps_update > PPS_MAXUPDATE)
		pps_update = 0;
	if (pps_time && pps_update) {
		if (!pps_control)
			syslog(LOG_INFO, "kernel pps sync enabled");
		pps_control = 1;
	} else {
		if (pps_control)
			syslog(LOG_INFO, "kernel pps sync disabled");
		pps_control = 0;
	}

	/*
	 * If the phase-lock loop is not implemented in the kernel, we
	 * do it the hard way using incremental adjustments and the
	 * adjtime() system call.
	 */
	if (pll_control && pll_enable)
		return;
	adjustment = clock_adjust;
	if (adjustment < 0)
		adjustment = -(-adjustment >> CLOCK_PHASE);
	else
		adjustment >>= CLOCK_PHASE;

	clock_adjust -= adjustment;
	if (drift_comp < 0)
		adjustment -= -drift_comp >> RSH_DRIFT_TO_ADJ;
	else
		adjustment += drift_comp >> RSH_DRIFT_TO_ADJ;

	{	l_fp offset;
		offset.l_i = 0;
		offset.l_f = adjustment;
		adj_systime(&offset);
	}
}


/*
 * adj_frequency - adjust local clock frequency
 */
void
adj_frequency(freq)
	s_fp freq;		/* frequency (ppm) */
{
#if defined(KERNEL_PLL)
	struct timex ntv;
#endif /* KERNEL_PLL */

	/*
	 * This routine adjusts the frequency offset. It is used by the
	 * local clock driver to adjust frequency when no external
	 * discipline source is available and by the acts driver when
	 * the interval between updates is greater than 1 <<
	 * NTP_MAXPOLL. Note that the maximum offset is limited by
	 * max_comp when the daemon pll is used, but the maximum may be
	 * different when the kernel pll is used.
	 */
	drift_comp += freq;
	if (drift_comp > max_comp)
		drift_comp = max_comp;
	else if (drift_comp < -max_comp)
		drift_comp = -max_comp;
#if defined(KERNEL_PLL)
	/*
	 * If the phase-lock code is implemented in the kernel, set the
	 * kernel frequency as well, but be sure to set drift_comp to
	 * the actual frequency..
	 */
	if (!(pll_control && pll_enable))
		return;
	memset((char *)&ntv,  0, sizeof ntv);
	ntv.modes = MOD_FREQUENCY;
	ntv.freq = freq + drift_comp;
	(void)ntp_adjtime(&ntv);
	drift_comp = ntv.freq;
#endif /* KERNEL_PLL */
}


/*
 * loop_config - configure the loop filter
 */
void
loop_config(item, lfp_value, int_value)
	int item;
	l_fp *lfp_value;
	int int_value;
{
	s_fp tmp;
#if defined(KERNEL_PLL)
	struct timex ntv;
#endif /* KERNEL_PLL */

#ifdef DEBUG
	if (debug) {
		printf("loop_config %d %s %x\n", item,
		    lfptoa(lfp_value, 3), int_value);
	}
#endif
	switch (item) {
	case LOOP_DRIFTCOMP:
		tmp = LFPTOFP(lfp_value);
		if (tmp >= max_comp || tmp <= -max_comp) {
			syslog(LOG_ERR,
	    "loop_config: frequency offset %s in ntp.conf file is too large",
			    fptoa(tmp, 3));
		} else {
			drift_comp = tmp;

#if defined(KERNEL_PLL)
			/*
			 * If the phase-lock code is implemented in the
			 * kernel, give the time_constant and saved
			 * frequency offset to the kernel. If not, no
			 * harm is done.
	 		 */
			pll_status = int_value & (STA_PLL |
			    STA_PPSFREQ);
			if (pll_status & STA_PLL)
				pll_control = 1;
			else
				pll_control = 0;
			ntv.modes = MOD_BITS | MOD_FREQUENCY;
			ntv.offset = 0;
			ntv.freq = drift_comp;
			ntv.maxerror = NTP_MAXDISPERSE;
			ntv.esterror = NTP_MAXDISPERSE;
			ntv.status = pll_status | STA_UNSYNC;
			ntv.constant = time_constant;
			newsigsys.sv_handler = pll_trap;
			newsigsys.sv_mask = 0;
			newsigsys.sv_flags = 0;
			if ((sigvec(SIGSYS, &newsigsys, &sigsys)))
				syslog(LOG_ERR,
			   "sigvec() fails to save SIGSYS trap: %m\n");
			(void)ntp_adjtime(&ntv);
			if ((sigvec(SIGSYS, &sigsys,
			    (struct sigvec *)NULL)))
				syslog(LOG_ERR,
		    "sigvec() fails to restore SIGSYS trap: %m\n");
			if (pll_control)
				syslog(LOG_NOTICE,
			    "using kernel phase-lock loop %04x",
				    ntv.status);
			else
				syslog(LOG_NOTICE,
			    "using xntpd phase-lock loop");
#endif /* KERNEL_PLL */

		}
		break;

	default:
		/* sigh */
		break;
	}
}


#if defined(KERNEL_PLL)
/*
 * _trap - trap processor for undefined syscalls
 *
 * This nugget is called by the kernel when the SYS_ntp_adjtime()
 * syscall bombs because the silly thing has not been implemented in
 * the kernel. In this case the phase-lock loop is emulated by
 * the stock adjtime() syscall and a lot of indelicate abuse.
 */
RETSIGTYPE
pll_trap()
{
	pll_control = 0;
}
#endif /* KERNEL_PLL */

