/*---------------------------------------------------------------------------
 * Copyright (c) 1991 Herb Weiner (herbw@wiskit.rain.com) All rights reserved
 * This software is provided on an AS IS basis without warranty or support of
 * any kind.  Permission is granted to distribute this software provided that
 * this entire copyright notice is retained, and provided that the package is
 * provided (including source code) without charge.  Please email bug reports
 * and enhancements to the author.  Enjoy this software, and please share it.
 *---------------------------------------------------------------------------
 */

/*---------------------------------------------------------------------------
 * slipdaemon is a daemon that manages the client side of a SLIP line.  It
 * performs the following functions:
 * 1. Dials the specified phone number of the SLIP server.
 * 2. Starts the SLIP client (slattconf).
 * 3. Detects loss of connection, and automatically redials and restarts.
 *
 * This program has been tested on a Mac II fx running A/UX version 2.01,
 * using Intel 9600EX modems connected to /dev/tty1.  The slip server is
 * a PC clone running PC-Route or KA9Q.
 *
 * This program is started using the following command line:
 * slipdaemon /dev/tty1 -c -f -h -p -r -s -S -t -w -x 19200 phone local-ip remote-ip netmask
 *
 * This program contains a lot more than I originally intended to include.
 * This was necessitated by omissions and/or bugs in Apple's A/UX 2.01.
 * The following is a list of the problems I encountered on a Macintosh IIfx:
 * 1. Unable to catch SIGHUP.  As far as I can determine, slattconf is
 *    catching the signal.  I'm in my own process group, I'm the first to
 *    open the device, and I place slattach in a separate process group
 *    so that they DON'T catch the signal. (Maybe I should not do this.
 *    Then, maybe we'd both get the signal.)
 * 2. Port left in a funny state.  Possibly Line Discipline 2 still on
 *    file descriptor.  Can not read data from tty.
 * 3. sleep(3) does not work inside SIGALRM handler.  The problem was
 *    solved by linking this module with sleep.c (from 4.3bsd (reno)
 *    distribution on wuarchive).
 * 4. It was necessary to "rm /dev/iw2" and "rm /dev/printer", since
 *    these used the same major and minor device numbers as /dev/tty1.
 *    Doing this is necessary so that (a) ps -ef will list the correct
 *    device number, and so that (b) the protections on /dev/tty1 are
 *    not circumvented by alternate paths to this inode.
 * 5. routed apparently does not work correctly with slip, even if the
 *    patch (supplied by Ron Flax of Apple) is applied to enable routing.
 *
 * Herb Weiner
 * Kitchen Wisdom Publishing
 * herbw@wiskit.rain.com
 *---------------------------------------------------------------------------
 */

#include <stdio.h>
#include <termio.h>
#include <fcntl.h>
#include <signal.h>
#include <compat.h>
#include <string.h>
#include <time.h>
#include <a.out.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/socket.h>

extern int	errno;		/* Error number from last syscall */
extern int	sys_nerr;	/* Size of the sys_errlist array */
extern char	*sys_errlist[];	/* Array of error message text strings */

typedef enum {success, failure, working} response;

struct tm *localtime ();

void dial ();
void catch_hangup ();
char *prefix ();
void check_ipforwarding ();

static int		tty;
static int		my_cflag;
static char		*my_name;
static char		*tty_device;
static char		*baud_rate;
static char		*phone_number;
static char		*local_ip;
static char		*local_ip_orig;
static char		*remote_ip;
static char		*netmask;
static char		**main_argv;
static char		*slip_argv [20];
static char		*slattach_name = "slattach";
static char		*slattconf_name = "slattconf";
static char		*slattach_etc = "/etc/slattach";
static char		*slattconf_etc = "/etc/slattconf";
static int		slip_pid = 0x7FFFFFFF;
static int		alarm_time = 300;
static int		caught_hangup = 0;
static int		opt_attach = 0;
static int		opt_continue = 0;
static int		opt_debug = 0;
static int		opt_superdebug = 0;
static int		opt_fast = 0;
static int		opt_hangup = 0;
static int		opt_kill = 0;
static int		opt_superkill = 0;
static int		opt_limit = 0;
static int		opt_supernbio = 0;
static int		opt_pg = 0;
static int		opt_reopen = 0;
static int		opt_spawn = 0;
static int		opt_superspawn = 0;
static int		opt_timer = 0;
static int		opt_supertty = 0;
static int		opt_write = 0;
static int		opt_extra = 0;
static struct termio	my_termio;

void open_tty ()
{
	int	i;
	int	my_pg;
	int	tty_pg;
	int	my_error;

	if (opt_superdebug)
		fprintf (stderr, "%s: About to open %s\n", prefix (), tty_device);
	tty = open (tty_device, O_NDELAY | O_RDWR);
	if (tty == -1)
	{
		fprintf (stderr, "%s: Unable to open %s <%s>\n", prefix (), tty_device, sys_errlist[errno]);
		exit (-1);
	}
	if (opt_superdebug)
		fprintf (stderr, "%s: Successfully opened %s\n", prefix (), tty_device);

	my_error = ioctl (tty, TCRESET, 0);
	if (my_error == -1)
	{
		fprintf (stderr, "%s: Unable to ioctl %s <%s>\n", prefix (), tty_device, sys_errlist[errno]);
		exit (-1);
	}

	/* (void) ioctl (tty, TCGETA, &my_termio); */
	my_termio.c_iflag = 0;				/* input modes */
	my_termio.c_oflag = 0;				/* output modes */
	my_termio.c_cflag = my_cflag | CS8 | CREAD | CLOCAL;	/* control modes */
	my_termio.c_lflag = 0;				/* local modes */
	my_termio.c_line = 0;				/* line discipline */
/* The following was removed to prevent long delays when reading from the tty */
	for (i = 0; i < NCC; i++)
		my_termio.c_cc[i] = 0x01;		/* control characters */
    	fprintf(stderr, "%s: ioctl TCSETA, iflag %x oflag %x cflag %x lflag %x\n",prefix(), my_termio.c_iflag, my_termio.c_oflag, my_termio.c_cflag, my_termio.c_lflag);

	(void) ioctl (tty, TCSETA, &my_termio);
/* sth
	(void) ioctl (tty, UIOCNOFLOW, 0);
	(void) ioctl (tty, UIOCNOMODEM, 0);*/	/* Turn OFF Modem Control */

	my_pg = getpgrp();
	ioctl (tty, TIOCGPGRP, &tty_pg);
	if (my_pg != tty_pg)
		fprintf (stderr, "%s: ERROR: my process group is %d; tty process group is %d\n",
			prefix (), my_pg, tty_pg);
	if (opt_superdebug)
		fprintf (stderr, "%s: Opened and initialized %s\n", prefix (), tty_device);
}

void catch_alarm ()
{
	int		pid;

	/* fprintf (stderr, "%s: Caught SIGALRM (kill -14)\n", prefix ()); */

	if (opt_fast)
	{
		pid = kill (slip_pid, 0);
		if (pid == -1)
			slip_pid = 0x7FFFFFFF;
		else
			pid = slip_pid;
	}
	else
		pid = find_slattconf_pid (tty_device);

	if (pid == -1)
	{
		fprintf (stderr, "%s: Caught SIGALRM (kill -14)\n", prefix ());
		fprintf (stderr, "%s: %s is not running\n", prefix (), *slip_argv);
		fprintf (stderr, "%s: signalling SIGHUP\n", prefix ());
		if (opt_hangup)
			caught_hangup = 1;
		else
			catch_hangup ();
	}
	else
	{
		/* fprintf (stderr, "%s: %s is running normally (pid %d)\n", prefix (), *slip_argv, pid); */
		/* ping remote_ip 64 1 */
	}

	if (opt_timer)
		alarm (alarm_time);
}

/*
Log a character that was read or written and convert some special characters.
*/
void log_char (str, c)
char *str;
char c;
{
        if (c == '\n') {
                fprintf (stderr, "%s: %s ^J\n", prefix (), str);
        } else if (c == '\r') {
                fprintf (stderr, "%s: %s ^M\n", prefix (), str);
        } else {
                fprintf (stderr, "%s: %s %c\n", prefix (), str, c);
        }
	fflush (stderr);
}

void writex(fildes, buf, nbytes)
int fildes;
char *buf;
unsigned nbytes;
{
	int	i;

	if (opt_superdebug)
		for (i = 0; i < nbytes; i++) log_char ("Write", buf[i]);

	i = write(fildes, buf, nbytes);
	if (i == -1) {
		fprintf (stderr, "%s: Write failure <%s, %d>\n", prefix (),
			sys_errlist[errno], errno);
		fflush (stderr);
	}
}

void catch_hangup ()
{
	int		pid;
	int		i;

	fprintf (stderr, "%s: Caught SIGHUP (kill -1)\n", prefix ());

	(void) ioctl (tty, UIOCNOMODEM, 0);		/* Turn OFF Modem Control */
	(void) fcntl (tty, F_SETFL, O_NDELAY | O_RDWR);	/* Turn ON O_NDELAY */
	stop_slip ();

	if (opt_extra)
	{
		fprintf (stderr, "%s: will close, then open tty\n", prefix ());
		close (tty);
		open_tty ();
		if (opt_write)
		{
			if (opt_superdebug)
				fprintf (stderr, "%s: About to write\n", prefix ());
			sleep (2);
			writex (tty, "+++", 3);
			sleep (2);
/*			writex (tty, "atz\r", 4); */
			writex (tty, "ath0\r", 4);
			sleep (2);
			(void) ioctl (tty, TCFLSH, 0);	/* Flush Input Queue */
		}
		else
		{
			if (opt_superdebug)
				fprintf (stderr, "%s: About to sleep\n", prefix ());
			sleep (1);
		}
	}

	if (opt_spawn)
	{
		fprintf (stderr, "%s: shutting down for auto restart\n", prefix ());
		for  (i = getdtablesize(); i >= 0; i--)
			(void) close (i);
		if (opt_superspawn)
		{
			pid = fork ();
			if (pid > 0)	/* parent process */
			{
				sleep (1);
				_exit (0);
			}
			else
			{
				setpgrp (pid, pid);
				sleep (1);
			}
		}
		(void) execvp (main_argv[0], main_argv);
		fprintf (stderr, "%s: Exec failure <%s>\n", prefix (), sys_errlist[errno]);
		_exit (-1);
	}
	else if (opt_reopen)
	{
		fprintf (stderr, "%s: will close, then open tty\n", prefix ());
		close (tty);
		open_tty ();
	}
	else
	{
		(void) ioctl (tty, TCSETA, &my_termio);		/* Restore sane settings */
		fprintf (stderr, "%s: ioctl OK\n", prefix ());
	}

	start_slip (0);
	pid = find_slattconf_pid (tty_device);
	if (pid != -1)
		fprintf (stderr, "%s: %s restarted normally (pid %d)\n", prefix (), *slip_argv, pid);
	else
		fprintf (stderr, "%s: %s did not restart\n", prefix (), *slip_argv);
/*	fprintf (stderr, "%s: start_slip OK\n", prefix ()); */
	(void) fcntl (tty, F_SETFL, O_RDWR);		/* Turn OFF O_NDELAY */
	(void) ioctl (tty, UIOCMODEM, 0);		/* Turn ON Modem Control */
	fprintf (stderr, "%s: resuming normal operation\n", prefix ());
}

void catch_sigterm ()
{
	int		pid;

	fprintf (stderr, "%s: Caught SIGTERM (kill -15)\n", prefix ());

	if (!opt_continue)
		stop_slip ();

	(void) fcntl (tty, F_SETFL, O_NDELAY | O_RDWR);	/* Turn ON O_NDELAY */
	(void) ioctl (tty, UIOCNOMODEM, 0);
	close (tty);
	fprintf (stderr, "%s: shutdown complete\n", prefix ());
	exit (1);
}

void main (argc, argv)
int	argc;
char	*argv[];
{
	int		i;
	int		pid;
	int		my_pid;
	int		my_pg;
	int		tty_pg;
	int		my_parent;
	int		compatibility_mode = COMPAT_BSDSIGNALS | COMPAT_SYSCALLS;
	int		my_argc = 0;
	char		*my_argv [20];
	char		**argp;
	char		*slashp;
	char		message_buffer [80];
	char		option_buffer [32];

	typedef struct
		{
		char		*name;
		int		*value;
		char		*help;
		}
	option_structure;

	static option_structure option_list [] =
		{
			"-a",	&opt_attach,	"Use slattach rather than slattconf",
			"-c",	&opt_continue,	"Don't stop SLIP when catch SIGTERM",
			"-d",	&opt_debug,	"Run in debug mode (without SLIP)",
			"-D",	&opt_superdebug,"Print extra debugging messages",
			"-f",	&opt_fast,	"Use 3 rather than 300 second alarm",
			"-h",	&opt_hangup,	"Call SIGHUP from main, not alarm",
			"-k",	&opt_kill,	"Kill SLIP if already running",
			"-K",	&opt_superkill,	"Kill slipdaemon if already running",
			"-l",	&opt_limit,	"Start alarm before starting SLIP",
			"-N",	&opt_supernbio,	"setcompat(COMPAT_BSDNBIO)",
			"-p",	&opt_pg,	"Start SLIP with unique process group",
			"-r",	&opt_reopen,	"close and reopen tty on SIGHUP",
			"-s",	&opt_spawn,	"exec fresh slipdaemon on SIGHUP",
			"-S",	&opt_superspawn,"fork fresh slipdaemon on SIGHUP",
			"-t",	&opt_timer,	"Use alarm signal to monitor SLIP",
			"-T",	&opt_supertty,	"setcompat(COMPAT_BSDTTY)",
			"-w",	&opt_write,	"write atz after reopening tty",
			"-x",	&opt_extra,	"Extra close and reopen tty on SIGHUP",
		};
	int	number_of_options = sizeof (option_list) / sizeof (option_structure);

	setbuf (stdout, NULL);
	setbuf (stderr, NULL);

	main_argv = argv;

	slashp = strrchr (argv[0], '/');
	if (slashp == (char *) NULL)
		my_name = argv [0];
	else
		my_name = ++slashp;

	argv++;
	argc--;
	argp = my_argv;

	while (argc > 0)
		if (**argv == '-')
		{
			for (i = 0; i < number_of_options; i++)
				if (strcmp (option_list [i].name, *argv) == 0)
				{
					*(option_list [i].value) ^= 1;
					argv++;
					argc--;
					break;
				}
			if (i == number_of_options)
			{
				fprintf (stderr, "ERROR: unrecognized option %s\n", *argv);
				my_argc = -1;
				break;
			}
		}
		else
		{
			my_argc++;
			if (my_argc > 6)
			{
				fprintf (stderr, "ERROR: too many arguments\n");
				break;
			}
			*argp++ = *argv++;
			argc--;
		}

	if (my_argc != 6)
	{
		fprintf (stderr, "usage: %s tty-device -opts baud-rate phone# local-ip remote-ip netmask\n", my_name);
		fprintf (stderr, "options:\n");
		for (i = 0; i < number_of_options; i++)
			fprintf (stderr, "  %s: %s\n", option_list [i].name, option_list [i].help);
		exit (1);
	}

	if (geteuid() != 0)
	{
		fprintf (stderr, "ERROR: %s requires superuser privileges\n", my_name);
		exit (1);
	}
	
	argp = my_argv;
	tty_device = *argp++;
	baud_rate = *argp++;
	phone_number = *argp++;
	local_ip = *argp++;
	local_ip_orig = local_ip;
	remote_ip = *argp++;
	netmask = *argp++;

	pid = find_slipdaemon_pid (tty_device);
	if (pid != -1)
		if (opt_superkill && (find_slattconf_pid (tty_device) == -1))
		{
			fprintf (stderr, "killing previous slipdaemon (pid = %d)\n", pid);
			kill (pid, SIGTERM);
			sleep (2);
			kill (pid, SIGKILL);
			sleep (2);
		}
		else
		{
			fprintf (stderr, "slipdaemon is already running (pid = %d)\n", pid);
			exit (1);
		}

	if (opt_fast)
		alarm_time = 3;

	argp = slip_argv;

	if (opt_attach)
	{
		*argp++ = slattach_name;
		*argp++ = tty_device;
		*argp++ = local_ip;
		*argp++ = remote_ip;
		*argp++ = baud_rate;
		*argp++	= NULL;
	}
	else
	{
		*argp++ = slattconf_name;
		*argp++ = tty_device;
		*argp++ = baud_rate;
		*argp++ = local_ip;
		*argp++ = remote_ip;
	/*	*argp++ = "netmask";
		*argp++ = netmask; */
		*argp++	= NULL;
	}

	my_cflag = get_my_cflag (baud_rate);

	for  (i = getdtablesize(); i >= 0; i--)
		(void) close (i);
	open ("/dev/null", O_RDONLY);		/* open stdin */
	open ("/usr/spool/sliplog", O_WRONLY | O_APPEND | O_CREAT, 0644); /* open stdout */
	dup2 (1, 2);				/* open stderr */

fprintf(stderr, "%s baud, %d\n", baud_rate, my_cflag);

	my_parent = getppid();
	if (my_parent != 1)
	{
		pid = fork ();
		if (pid > 0)	/* parent process */
			_exit (0);
	}

	if (opt_supertty)
		compatibility_mode |= COMPAT_BSDTTY;
	if (opt_supernbio)
		compatibility_mode |= COMPAT_BSDNBIO;

	setcompat (compatibility_mode); /* for Apple A/UX 2.0 */

	my_pid = getpid();
	my_parent = getppid();

	setpgrp (my_pid, my_pid);

	fprintf (stderr, "%s: started (pid = %d, parent = %d)\n", prefix (),
			my_pid, my_parent);

	sprintf (message_buffer, "%s: ", prefix ());
	for (i = 0; i < number_of_options; i++)
	{
		sprintf (option_buffer, "%s = %d, ", option_list [i].name, *(option_list [i].value));
		if ((strlen(message_buffer) + strlen (option_buffer)) > 80)
		{
			message_buffer [strlen (message_buffer) - 2] = '\0';
			strcat (message_buffer, "\n");
			fputs (message_buffer, stderr);
			sprintf (message_buffer, "%s: ", prefix ());
		}
		strcat (message_buffer, option_buffer);
	}
	message_buffer [strlen (message_buffer) - 2] = '\0';
	strcat (message_buffer, "\n");
	fputs (message_buffer, stderr);

	check_ipforwarding ();

	if (opt_debug || (strcmp (remote_ip, "0.0.0.0") == 0))
	{
		fprintf (stderr, "%s: running in debug mode (without slip)\n", prefix ());
		opt_debug = 1;
	}

	signal (SIGHUP,  catch_hangup);
	signal (SIGTERM, catch_sigterm);
	signal (SIGALRM, catch_alarm);
	signal (SIGINT, SIG_IGN);		/* Ignore Interrupt (Control C) */
	signal (SIGTSTP, SIG_IGN);		/* Ignore Stop (Control Z) */

	if (opt_kill)
		stop_slip ();
	else
	{
		pid = find_slattconf_pid (tty_device);
		if (pid == -1)
			fprintf (stderr, "%s: %s is not running\n", prefix (), *slip_argv);
		else
			fprintf (stderr, "%s: %s is running normally (pid %d)\n", prefix (), *slip_argv, pid);
	}

	open_tty ();

	if (opt_limit)
		alarm (alarm_time);		/* start time limit before start slip */

	pid = find_slattconf_pid (tty_device);
	if (pid == -1)
		start_slip (test_slip ());
		/* start_slip (0); */

	pid = find_slattconf_pid (tty_device);
	if (pid != -1)
		fprintf (stderr, "%s: %s started normally (pid %d)\n", prefix (), *slip_argv, pid);
	else
		fprintf (stderr, "%s: %s did not start\n", prefix (), *slip_argv);

	(void) fcntl (tty, F_SETFL, O_RDWR);	/* Turn off O_NDELAY */
	(void) ioctl (tty, UIOCMODEM, 0);	/* Turn ON Modem Control */
	(void) ioctl (tty, UIOCFLOW, 0);	/* Turn ON Flow Control */

	if (!opt_debug)
	{
		argp = my_argv;
		*argp++ = "route";
		*argp++ = "add";
		*argp++ = "default";
		*argp++ = remote_ip;
		*argp++ = "10";
		*argp++	= NULL;

		pid = fork_and_exec ("/usr/etc/route", my_argv, NULL, NULL, NULL);
		if (pid != -1)
			wait ((int *) NULL);
	}

	if (opt_timer)
		alarm (alarm_time);

	while (1)
	{
		pause ();
		if (caught_hangup)
		{
			alarm (0);
			caught_hangup = 0;
			catch_hangup ();
			alarm (alarm_time);
		}
		/* fprintf (stderr, "%s: a brief pause... and continue\n", prefix ()); */
	}
/*	(void) ioctl (tty, UIOCNOMODEM, 0); */
/*	close (tty); */
}

int test_slip ()
{
	int	number_of_available_characters = 0;

	(void) ioctl (tty, TCFLSH, 0);	/* Flush Input Queue */
	/* sleep (35); */
	sleep (5);
	(void) ioctl (tty, FIONREAD, &number_of_available_characters);
	if (number_of_available_characters > 0)
	{
		fprintf (stderr, "%s: receiving data (modem link established)\n", prefix ());
		return (2);	/* Remote is probably sending RIP packets */
	}

	writex (tty, "at\r", 3);
	sleep (2);
	(void) ioctl (tty, FIONREAD, &number_of_available_characters);
	if (number_of_available_characters == 0)	/* No OK response */
	{

		/*----------------------------------------------------------
		 * Either the modem link is established, or the modem is set
		 * to no echo, or the data is being received and dispatched
		 * to the daemon in the sky.
		 *----------------------------------------------------------
		 */

		writex (tty, "+++", 3);
		sleep (2);
		writex (tty, "at\r", 3);
		sleep (2);
		(void) ioctl (tty, FIONREAD, &number_of_available_characters);
		if (number_of_available_characters == 0)	/* Still No OK response */
		{
			fprintf (stderr, "%s: modem not responding or driver hung\n", prefix ());
			/*----------------------------------------------------
			 * we really need to do something more effective here!
			 *----------------------------------------------------
			 */
			return (0);	/* Assume that connection NOT established */
		}
		else
		{
/*
			writex (tty, "(some kind of continue command here)", 0);
*/
/*			writex (tty, "ato\n", 0); */
			writex (tty, "at\\o\n", 0);		/* try reconnect with AT\O */
			fprintf (stderr, "%s: line appears to be connected\n", prefix ());
			return (1);	/* Assume that connection already established */
		}
	}

	fprintf (stderr, "%s: connection NOT established\n", prefix ());
	return (0);		/* Connection has NOT been established */
}

int start_slip (link_status)
int link_status;
{
	int		pid;
	char	*my_argv [20];
	char	**argp;

	if (link_status == 0)
		dial (tty, phone_number, 3, -1);

	if (opt_debug)
		return;

	if (opt_attach)
		pid = fork_and_exec (slattach_etc, slip_argv, NULL, NULL, NULL);
	else
		pid = fork_and_exec (slattconf_etc, slip_argv, NULL, NULL, NULL);

	sleep (3);

	slip_pid = find_slattconf_pid (tty_device);

	if (opt_attach)
	{
		argp = my_argv;
		*argp++ = "ifconfig";
		*argp++ = "sl0";
		*argp++ = local_ip;
		*argp++ = remote_ip;
		*argp++ = "netmask";
		*argp++ = netmask;
		*argp++ = "up";
		*argp++	= NULL;

		fprintf (stderr, "%s: %s %s %s\n",
				prefix (), my_argv [1], my_argv [2], my_argv [3]);

		pid = fork_and_exec ("/etc/ifconfig", my_argv, NULL, NULL, NULL);
		if (pid != -1)
			wait ((int *) NULL);
	}

	if (pid != -1)
		wait ((int *) NULL);
}

int stop_slip ()
{
	int	pid;
	char	*my_argv [20];
	char	**argp;

	if (!opt_debug)
	{
		if (opt_attach)
		{
			argp = my_argv;
			*argp++ = "ifconfig";
			*argp++ = "sl0";
			*argp++ = "down";
			*argp++	= NULL;

			fprintf (stderr, "%s: %s %s %s\n",
					prefix (), my_argv [1], my_argv [2], my_argv [3]);

			pid = fork_and_exec ("/etc/ifconfig", my_argv, NULL, NULL, NULL);
			if (pid != -1)
				wait ((int *) NULL);
		}
	}

	pid = find_slattconf_pid (tty_device);
	if (pid != -1)
	{
		fprintf (stderr, "%s: killing %s (pid %d)\n", prefix (), *slip_argv, pid);
		kill (pid, SIGHUP);	/* if this doesn't work, try SIGTERM */

		while (find_slattconf_pid (tty_device) != -1)
			sleep (5);
	}
	else
		fprintf (stderr, "%s: %s is not running\n", prefix (), *slip_argv);

	slip_pid = 0x7FFFFFFF;
}

int find_slattconf_pid (tty_device)
char	*tty_device;
{
	int		stdin_pipe;
	static char	*ps_argv [] = { "ps", "-ef", NULL };
	char		my_buffer [256];
	int		slattconf_pid = -1;
	char		search_string1 [64];
	char		search_string2 [64];
	char		*bp;
	int		pid;

	pid = fork_and_exec ("/bin/ps", ps_argv, NULL, &stdin_pipe, NULL);

	if (pid != -1)
	{
		strcpy (search_string1, "slattconf ");
		strcat (search_string1, tty_device);
		strcpy (search_string2, "slattach ");
		strcat (search_string2, tty_device);
		while (read_line (stdin_pipe, my_buffer) != -1)
		{
			bp = strstr (my_buffer, search_string1);
			if (bp != (char *) NULL)
			{
				bp = my_buffer;
				while ((*bp != '\0') && ((*bp < '0') || (*bp > '9')))
					bp++;
				if (*bp == '\0')
					break;
				sscanf (bp, "%d", &slattconf_pid);
				break;
			}

			bp = strstr (my_buffer, search_string2);
			if (bp != (char *) NULL)
			{
				bp = my_buffer;
				while ((*bp != '\0') && ((*bp < '0') || (*bp > '9')))
					bp++;
				if (*bp == '\0')
					break;
				sscanf (bp, "%d", &slattconf_pid);
				break;
			}
		}
		close (stdin_pipe);
		wait ((int *) NULL);
	}
	return (slattconf_pid);
}

int find_slipdaemon_pid (tty_device)
char	*tty_device;
{
	int		stdin_pipe;
	static char	*ps_argv [] = { "ps", "-ef", NULL };
	char		my_buffer [256];
	int		slip_daemon_pid = -1;
	char		search_string1 [64];
	char		*bp;
	int		pid;

	pid = fork_and_exec ("/bin/ps", ps_argv, NULL, &stdin_pipe, NULL);

	if (pid != -1)
	{
		pid = getpid();
		strcpy (search_string1, "slipdaemon ");
		strcat (search_string1, tty_device);
		while (read_line (stdin_pipe, my_buffer) != -1)
		{
			bp = strstr (my_buffer, search_string1);
			if (bp != (char *) NULL)
			{
				bp = my_buffer;
				while ((*bp != '\0') && ((*bp < '0') || (*bp > '9')))
					bp++;
				if (*bp != '\0')
				{
					sscanf (bp, "%d", &slip_daemon_pid);
					if (slip_daemon_pid == pid)
						slip_daemon_pid = -1;
					else
						break;
				}
			}
		}
		close (stdin_pipe);
		wait ((int *) NULL);
	}
	return (slip_daemon_pid);
}

int string_match (string1, string2)
char	*string1;
char	*string2;
{
	int	i;
	int	compare_length = strlen (string1);

	if (compare_length > strlen (string2))
		return (-1);
	for (i = 0; i < compare_length; i++)
		if (toupper (string1[i]) != toupper (string2[i]))
			return (1);
	return (0);
}

int read_line (fd, buffer)
int	fd;
char	*buffer;
{
	int	result;
	char	my_buf;
	char	*bp = buffer;

	*bp = '\0';

	for (;;)
	{
		result = read (fd, &my_buf, 1);
		if (result == -1)
		{
			fprintf (stderr, "%s: read (%d) failed <%s> %d\n",
				prefix (), fd, sys_errlist[errno], errno);
			fflush(stderr);
		}
		if (result == 0)
			return (-1);
		if ((result < 0) || (my_buf == '\n') || (my_buf == '\r'))
		{
			if (bp != buffer)
			{
				*bp++ = '\0';
				break;
			}
		}
		else
			if (((my_buf >= 32) && (my_buf <= 126)) || (my_buf == '\t'))
			{
				*bp++ = my_buf;
				if (opt_superdebug && fd == tty)
					log_char ("char", my_buf);
			}
			else
			{
				fprintf (stderr, "%s: read (%d) unexpected character 0x%02X\n",
					prefix (), fd, my_buf);
				fflush(stderr);
			}
	}

	return (strlen (buffer));
}

response parse_modem_result (result)
char	*result;
{
typedef struct
	{
	char		*message;
	response	meaning;
	}
result_structure;

static result_structure result_list [] =
	{
		"ok",			working,
		"connect",		success,
		"ring",			working,
		"no carrier",	failure,
		"error",		failure,
		"no dialtone",	failure,
		"no dial tone",	failure,
		"busy",			failure,
		"no answer",	failure,
		"carrier",		working,
		"protocol",		working,
		"compression",	working,
		"rring",		working,
		"dialing",		working,
		"at",			working,
		"+++",			working,
	};
int	number_of_results = sizeof (result_list) / sizeof (result_structure);
int	i;

	for (i = 0; i < number_of_results; i++)
		if (string_match (result_list [i].message, result) == 0)
			return (result_list [i].meaning);
	return (failure);
}

int get_my_cflag (baud_rate)
char	*baud_rate;
{
typedef struct
	{
	char		*baud_rate;
	int		my_cflag;
	}
result_structure;

static result_structure result_list [] =
	{
		"0",		B0,
		"50",		B50,
		"75",		B75,
		"110",		B110,
		"134",		B134,
		"150",		B150,
		"200",		B200,
		"300",		B300,
		"600",		B600,
		"1200",		B1200,
		"1800",		B1800,
		"2400",		B2400,
		"4800",		B4800,
		"9600",		B9600,
		"19200",	EXTA,
		"38400",	EXTB,
		"57600",	B50,
		"115200",	B75
	};
int	number_of_results = sizeof (result_list) / sizeof (result_structure);
int	i;

	for (i = 0; i < number_of_results; i++)
		if (string_match (result_list [i].baud_rate, baud_rate) == 0)
			return (result_list [i].my_cflag);
	return (B2400);
}

int read_and_match(fd, str, buffer)
int	fd;
char	*str;
char	*buffer;

{
	int	i;
	int	j;
	int	result;
	char	my_buf;
	char	*bp = buffer;

	fprintf (stderr, "%s: Looking for \"%s\" prompt\n", prefix(), str);
	fflush (stderr);

	i = 0;
	j = 0;
	*bp = '\0';

	for (;;)
	{
		result = read (fd, &my_buf, 1);
		j++;

		if (result == 0) {
			fprintf (stderr, "%s: result == 0\n", prefix());
			fflush(stderr);
			return (0);
		}

		if (opt_superdebug) log_char ("read:", my_buf);

		if (result == -1) {
			fprintf (stderr, "%s: read (%d) failed <%s>\n",
				prefix (), fd, sys_errlist[errno]);
			fflush(stderr);
		}

		if ((my_buf < ' ') & (my_buf != '\n')) continue;

		if (my_buf == '\n') {
			if (bp != buffer) { /* we're done if end of a line */
				*bp = '\0';
				return(0);
			}
		} else {
			if (my_buf == str[i]) {
				i++;
				if (i == strlen(str)) return (1);
			}
			*bp++ = my_buf; /* save message in case no match */
		}
	}
}

int scan_ip(buffer)
char	*buffer;
{
	int	i;
	int	j;
	long	l;

	j = 0;
	for (i = 0; i < strlen(buffer); i++) {
		if (buffer[i] == '.') j++;
		if (j == 3) {
			l = strtol(&buffer[++i], NULL, 10);
			return l;
		}
	}
}

void
wait_for_modem ()
{
	char		buffer [1024];
        response        tty_response;
        static char     *messages [] = {"succeeded", "failed", "in progress"};

	do
	{
		do
			read_line (tty, buffer);
		while (*buffer == '\0');
                fprintf (stderr, "%s: modem: %s\n", prefix(), buffer);
                fflush (stderr);
                tty_response = parse_modem_result (buffer);
                fprintf (stderr, "%s: call %s (%s)\n",
			prefix (), messages [tty_response], buffer);
                fflush (stderr);
	} while (tty_response == working);
}

void dial(tty, number, sleeptime, maxretries)
int	tty;
char	*number;
int	sleeptime;
int	maxretries;
{
	int		flags;
	char		buffer [1024];
	response	tty_response;
	static char	*messages [] = {"succeeded", "failed", "in progress"};
	char		s[10];
	static char	s1[30];
	char		errmsg[1024];
	int		failure_count = 0;
	int		modem_cmd_cnt;
	int		i;

	do
	{
		fprintf (stderr, "%s: initializing modem\n", prefix ());
		fflush (stderr);

		if (ioctl (tty, TCFLSH, 0) == -1)	/* Flush Input Queue */
			fprintf (stderr, "%s: Ioctl failure <%s>\n", prefix (), sys_errlist[errno]);

		sleep (2);
		writex (tty, "+++", 3); /* get the modem's attention */
		sleep (2);
		writex (tty, "ath0\r", 5); /* hang up the phone */
		sleep (2);
		writex (tty, "at&F\r", 5);
		sleep (2);
		writex (tty, "atZ0\r", 5);
		sleep (2);

/*		writex (tty, "ats37=9\r", 8);
		sleep(2);

		writex (tty, "ats50=6\r", 8);
		sleep (2);
*/
		fprintf (stderr, "%s: dialing %s\n", prefix (), number);
		fflush (stderr);

		writex (tty, "atdt", 4);
		writex (tty, number, strlen (number));
		writex (tty, "\r", 1);
		sleep(2);
		modem_cmd_cnt = 5;

/*		fflush(tty); */

		fprintf (stderr, "%s: dialing complete\n", prefix ());
		fflush (stderr);

		flags = fcntl (tty, F_GETFL, 0);
		(void) fcntl (tty, F_SETFL, flags & ~O_NDELAY);

		fprintf (stderr, "%s: preparing to read tty\n", prefix ());
		fflush (stderr);

		do
		{
			do
			{
				read_line (tty, buffer);
				if (*buffer == '\0') sleep(1);
			}
			while (*buffer == '\0');
			fprintf (stderr, "%s: modem: %s\n", prefix(), buffer);
			fflush (stderr);
			tty_response = parse_modem_result (buffer);

			modem_cmd_cnt--;

			if (tty_response == failure && modem_cmd_cnt > 0)
				tty_response = working;

			fprintf (stderr, "%s: call %s (%s)\n",
				prefix (), messages [tty_response], buffer);
			fflush (stderr);
		} while (tty_response == working);

		if (tty_response == success)
		{
			flags = fcntl (tty, F_GETFL, 0);
			(void) fcntl (tty, F_SETFL, flags & ~O_NDELAY);

			sleep (10);
			/* writex(tty, "\n", 1);
			sleep (2);
			writex(tty, "\n", 1); */

			for (i = 11; i > 0; i--) {
				if (read_and_match(tty, "login:", errmsg)) {
					break;
				} else {
					/* writex(tty, "\n", 1); */
					/* sleep(2); */

					fprintf (stderr, "%s: msg: \"%s\"\n", prefix(), errmsg);
					fflush (stderr);
				}
				if (i == 1) {
					fprintf (stderr, "%s: Couldn't find username\n", prefix());
					fflush (stderr);
					tty_response = failure;
					goto login_failed;
				}
			}

			writex(tty, "PUT LOGIN HERE\n", 7);
			sleep (2);

			do /* flush echo */
				read_line(tty, buffer);
			while (*buffer == '\0');

			if (!read_and_match(tty, "Password: ", errmsg))
			{
				fprintf (stderr, "%s: Couldn't find password\n",
					prefix());
				fprintf (stderr, "%s: msg: \"%s\"\n", prefix(), errmsg);
				fflush (stderr);
				tty_response = failure;
				goto login_failed;
			}

			writex(tty, "PUT PASSWORD HERE\n", 8);
			sleep (2);

			if (!read_and_match(tty, "beginning....", errmsg))
			{
				fprintf (stderr, "%s: Couldn't find startup message\n",
					prefix());
				fprintf (stderr, "%s: msg: \"%s\"\n", prefix(), errmsg);
				fflush (stderr);
				tty_response = failure;
			}
		}
		login_failed:

		(void) fcntl (tty, F_SETFL, flags);

		/* If we retry too often the log file will fill up
		   too quickly.  Increase the wait period in small increments
		   up to half an hour.
		*/
		if (tty_response == failure) {
			failure_count++;

			if (sleeptime * failure_count > 1800) {
                       		fprintf (stderr, "%s: waiting %d seconds\n", prefix(), 1800);
				fflush(stderr);
				sleep (1800);
			} else {
                       		fprintf (stderr, "%s: waiting %d seconds\n", prefix(), sleeptime);
                       		/* fprintf (stderr, "%s: waiting %d seconds\n", prefix(), sleeptime * failure_count); */
				fflush(stderr);
				/* sleep (sleeptime * failure_count); */
				sleep (sleeptime);
			}
		}

	} while (tty_response != success);
	(void) ioctl (tty, TCFLSH, 0);	/* Flush Input Queue */
}

void check_ipforwarding ()
{
	char		*filename = "/unix";
	struct nlist	nl[2];
	int		result;
	int		kmem;
	int		ipforwarding;

	nl[0].n_nptr = "ipforwarding";
	nl[0].n_zeroes = 0;
	nl[1].n_nptr = 0;
	nl[1].n_zeroes = 0;

	result = nlist (filename, nl);
	if (result == -1)
		fprintf (stderr, "%s: nlist failed <%s>\n", prefix (), sys_errlist[errno]);
	else if (nl[0].n_value == 0)
		fprintf (stderr, "%s: ipforwarding not in name list\n", prefix ());
	else
	{
		kmem = open("/dev/kmem", O_RDONLY, 0);
		if (kmem < 0)
			fprintf (stderr, "%s: Unable to open /dev/kmem <%s>\n", prefix (), sys_errlist[errno]);
		else
		{
			lseek (kmem, nl[0].n_value, 0);
			read (kmem, &ipforwarding, sizeof(ipforwarding));
			/* if (ipforwarding != 1) WARNING: */
			fprintf (stderr, "%s: ipforwarding = %d\n", prefix (), ipforwarding);
			close (kmem);
		}
	}
}


/*---------------------------------------------------------------------------
 *
 * NAME
 *
 *   fork_and_exec()             - fork and exec a command.
 *
 * SYNOPSIS
 *
 *   int fork_and_exec (command, argv, stdin_to_child, stdout_from_child,
 *                      stderr_from_child)
 *   char    *command;           - Full pathname of command.
 *   char    **argv;             - Argument vector.
 *   int     *stdin_to_child;    - Parent's output pipe or NULL.
 *   int     *stdout_from_child; - Parent's input pipe or NULL.
 *   int     *stderr_from_child; - Parent's input pipe or NULL.
 *
 * RETURN VALUE
 *
 *   pid of child                - Success
 *   -1                          - Failed
 *
 * DESCRIPTION
 *
 *   fork_and_exec () starts the child process.
 *---------------------------------------------------------------------------
 */

int fork_and_exec (command, argv, stdin_to_child, stdout_from_child, stderr_from_child)
char	*command;		/* Full pathname of command */
char	**argv;			/* Argument vector */
int	*stdin_to_child;	/* Parent's output pipe or NULL */
int	*stdout_from_child;	/* Parent's input pipe or NULL */
int	*stderr_from_child;	/* Parent's input pipe or NULL */
{
	int	i;			/* Index variable */
	int	pid;			/* Process id */
	int	pipe_io[2][2];		/* Temporary pipe variable */
	int	stdin_pipe [2];		/* Pipe for stdin */
	int	stdout_pipe [2];	/* Pipe for stdout */
	int	stderr_pipe [2];	/* Pipe for stderr */

	if (stdin_to_child != NULL)
		if  (pipe (stdin_pipe))
		{
			fprintf (stderr, "%s: Pipe failure on stdin <%s>", prefix (), sys_errlist[errno]);
			return (-1);
		}
	if (stdout_from_child != NULL)
		if  (pipe (stdout_pipe))
		{
			if (stdin_to_child != NULL)
			{
				(void) close (stdin_pipe[0]);
				(void) close (stdin_pipe[1]);
			}
			fprintf (stderr, "%s: Pipe failure on stdout <%s>", prefix (), sys_errlist[errno]);
			return  (-1);
		}
	if (stderr_from_child != NULL)
		if  (pipe (stderr_pipe))
		{
			if (stdin_to_child != NULL)
			{
				(void) close (stdin_pipe[0]);
				(void) close (stdin_pipe[1]);
			}
			if (stdout_from_child != NULL)
			{
				(void) close (stdout_pipe[0]);
				(void) close (stdout_pipe[1]);
			}
			fprintf (stderr, "%s: Pipe failure on stderr <%s>", prefix (), sys_errlist[errno]);
			return  (-1);
		}

	switch (pid = fork())
	{
	case -1:				/* Error on fork() call */
		if (stdin_to_child != NULL)
		{
			(void) close (stdin_pipe[0]);
			(void) close (stdin_pipe[1]);
		}
		if (stdout_from_child != NULL)
		{
			(void) close (stdout_pipe[0]);
			(void) close (stdout_pipe[1]);
		}
		if (stderr_from_child != NULL)
		{
			(void) close (stderr_pipe[0]);
			(void) close (stderr_pipe[1]);
		}
		fprintf (stderr, "%s: Fork failure <%s>", prefix (), sys_errlist[errno]);
		return  (-1);
		/* break; */			/* break implied by return() */
	case 0:					/* I'm the child process */
		if (opt_pg)
		{
			pid = getpid();
			setpgrp (pid, pid);	/* escape from parent's process group */
		}

		if (stdin_to_child != NULL)
			if  (dup2 (stdin_pipe[0], 0) < 0)
				fprintf (stderr, "%s: dup2 failure on stdin <%s>", prefix (), sys_errlist[errno]);
		if (stdout_from_child != NULL)
			if  (dup2 (stdout_pipe[1], 1) < 0)
				fprintf (stderr, "%s: dup2 failure on stdout <%s>", prefix (), sys_errlist[errno]);
		if (stderr_from_child != NULL)
			if  (dup2 (stderr_pipe[1], 2) < 0)
				fprintf (stderr, "%s: dup2 failure on stderr <%s>", prefix (), sys_errlist[errno]);

		for  (i = getdtablesize(); i > 2; i--)
		{
			(void) close (i);
		}
		for  (i = 0; i < NSIG; i++)
		{
			(void) signal (i, SIG_DFL);
		}
		(void) execvp (command, argv);
		fprintf (stderr, "%s: Exec failure <%s>", prefix (), sys_errlist[errno]);
		_exit (-1);
		break;
	default:				/* I'm the parent process */
		if (stdin_to_child != NULL)
		{
			(void) close (stdin_pipe[0]);
			*stdin_to_child = stdin_pipe[1];
		}
		if (stdout_from_child != NULL)
		{
			(void) close (stdout_pipe[1]);
			*stdout_from_child = stdout_pipe[0];
		}
		if (stderr_from_child != NULL)
		{
			(void) close (stderr_pipe[1]);
			*stderr_from_child = stderr_pipe[0];
		}
		return (pid);
	}
}

char *prefix ()
{
	static char	my_prefix [64];
	time_t		current_time;
	struct tm	*time_struct;
	static char	*month [] = {"jan", "feb", "mar", "apr", "may", "jun",
					"jul", "aug", "sep", "oct", "nov", "dec"};

	(void) time (&current_time);
	time_struct = localtime (&current_time);

	sprintf (my_prefix, "%02.02d-%s-%02.02d %02.02d:%02.02d:%02.02d %s (%s)",
				time_struct -> tm_mday, month [time_struct -> tm_mon],
				time_struct -> tm_year, time_struct -> tm_hour,
				time_struct -> tm_min, time_struct -> tm_sec,
				my_name, tty_device);
	return (my_prefix);
}
