/*
 * Virtual IP interface module. 
 *
 * Credits where credits are due:
 *
 * John Ioannidis <ji@cs.columbia.edu> came up with the VIF concept,
 * and wrote most of this code.
 *
 * Chuck Smoko <csmoko@relay.nswc.navy.mil> added the IFF_LOOPBACK flag.
 * This causes the route to be installed as a host route directly, and
 * thus removes the need to delete an extraneous network route.
 *
 * Steinar Haug <steinar.haug@runit.sintef.no> ported it to HP-UX 9.0x,
 * and made the code loadable and unloadable on SunOS 4.1.x.
 *
 * Dieter Dworkin Muller <dworkin@village.org> found an mbuf leak. It
 * happened if you had a *network* route to a network with a virtual
 * address, and the virtual address was used. Such a network route could
 * for instance be installed by mistakenly starting /usr/etc/in.routed.
 *
 * Phil Brandenberger <phil@wharton.upenn.edu> ported it to Ultrix.
 * For another Ultrix port, by John Hascall <john@iastate.edu>, check
 * out URL ftp://ftp.iastate.edu/pub/unix/ultrix/vif.
 *
 * Jim Jagielski <jim@jaguNET.com> added support for A/UX as well as
 * the magic that allows you to specify the number of interfaces
 * using the device-file's minor number. Also, we now allocate space
 * in vif_attach(). We do this because, before, we allocated the whole
 * block and if we only had space for 5 vifs, for example, but tried to
 * allocated 10, it would fail and we couldn't use any. Now we individually
 * allocate each vif. This takes a bit more time, and removes the DETACH
 * capability, but it's worth it (IMO). Finally, we do some monkey
 * business on the name of the virtual interface since we support > 10
 * interfaces; most systems don't allow interface names like 'vif10'
 * so we create 10-interface "chunks": vfa, vfb... that can each be
 * numbered from 0 -> 9. Not elegant at all, but it gets the job
 * done.
 *
 * HP-UX note: This code will almost certainly not work on interfaces
 * using checksum offload (eg. FDDI).
 * 
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/mbuf.h>
#include <sys/socket.h>
#include <sys/errno.h>
#include <sys/ioctl.h>
#if !defined(__hpux) && !defined(__ultrix) && !defined(_AUX_SOURCE)
#include <sys/syslog.h>
#endif

#include <net/if.h>
#include <net/netisr.h>
#include <net/route.h>

#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/in_var.h>
#include <netinet/ip.h>
#include <netinet/in_pcb.h>

#if defined(_AUX_SOURCE)
#include <sys/sysmacros.h>
#endif

#define VIFMTU  (1024+512)
#define NVIF_DEF	5
#define NVIF		NVIF_DEF
#define NVIF_MAX	25

int vifs_inited = 0;

int vif_output(), vif_ifioctl();

/*
 * vif_memalloc:
 *      simple front-end for kernel memory allocation
 */
static char *
vif_memalloc(size)
unsigned int size;
{
    char *kmem_alloc();
    register char *p;
    register char *cp;

    p = kmem_alloc(size);

    if (p) {
	for (cp = p; cp < (p + size); cp++) {	/* zero it out */
	    *cp = 0;
	}
	return (p);
    }
    else
	return ((char *) 0);
}

/*
 * vif_attach:
 *      Attach interfaces as required
 */
vif_attach(vifs_num)
register int vifs_num;
{
    register int i, num;
    register struct ifnet *ifp;
    unsigned int vif_size = sizeof(struct ifnet);

    for (i = 0; i < vifs_num; i++) {
	ifp = (struct ifnet *) vif_memalloc(vif_size);
	if (ifp == NULL) {
	    printf("vif: can't allocate %d bytes\n", vif_size);
	    vifs_num = i;
	    break;
	}
	/*
	 * Break the number of interfaces into "chunks" of 10.
	 * This is because most systems don't allow interface
	 * names like 'vif10' (i.e. must be a single digit after
	 * the name
	 */
	if (i>=0 && i<=9) {
	    num = i;
	    ifp->if_name = "vfa";
	}
	else if (i>=10 && i<=19) {
	    num = i - 10;
	    ifp->if_name = "vfb";
	}
	else if (i>=20 && i<=29) {
	    num = i - 20;
	    ifp->if_name = "vfc";
	}
	/*
	 * Punt here... if they have > 30 interfaces then we know
	 * they've hacked this code and can add more of the above
	 * else-if statements
	 */
	else {
	    num = i - 30;
	    ifp->if_name = "vif";
	}
	ifp->if_unit = num;
	ifp->if_mtu = VIFMTU;
#ifdef	MULTICAST
	ifp->if_flags = IFF_LOOPBACK | IFF_NOARP | IFF_MULTICAST;
#else	/* MULTICAST */
	ifp->if_flags = IFF_LOOPBACK | IFF_NOARP;
#endif	/* MULTICAST */
#ifdef _AUX_SOURCE
	ifp->if_flags |= IFF_NOTRAILERS;
#endif  /* _AUX_SOURCE */
	ifp->if_ioctl = vif_ifioctl;
	ifp->if_output = vif_output;
	if_attach(ifp);
    }
    vifs_inited = vifs_num;
}

/*
 * vif_open:
 *      Open entry point. Simply allocate memory and then attach interfaces
 */
vif_open(dev, flag)
int dev, flag;
{
    int unit;
    int vifs_num;

    unit = minor(dev);

    if (!vifs_inited) {
	if (unit > NVIF_MAX) {
	    printf("vif: maximum %d vifs\n", NVIF_MAX);
	    return ENXIO;
	}
	else if (unit <= 0)
	    vifs_num = NVIF_DEF;
	else
	    vifs_num = unit;
	vif_attach(vifs_num);
    }
    return 0;
}

/*
 * vif_close:
 *      close() entry point... just return
 */
vif_close(dev, flag)
int dev, flag;
{
    return 0;
}

/*
 * vif_read:
 *      read() entry point... Not supported
 */
vif_read(dev, puio)
dev_t dev;
struct uio *puio;
{
    return ENXIO;
}

/*
 * vif_close:
 *      close() entry point... Not supported
 */
vif_write(dev, puio)
dev_t dev;
struct uio *puio;
{
    return ENXIO;
}

/*
 * vif_select:
 *      select() entry point... Not supported
 */
vif_select(dev, flag)
dev_t dev;
int flag;
{
    return ENXIO;
}

/*
 * vif_output:
 *      Core of the code... output data in correct type/format
 */
vif_output(ifp, m0, dst)
struct ifnet *ifp;
register struct mbuf *m0;
struct sockaddr *dst;
{
    int s;
    register struct ifqueue *ifq;
    struct mbuf *m;
    struct sockaddr_in *din;

    if (dst->sa_family != AF_INET) {
#if defined(__hpux) || defined(__ultrix) || defined(_AUX_SOURCE)
	printf("%s%d: can't handle af%d\n",
	 ifp->if_name, ifp->if_unit, dst->sa_family);
#else
	log(LOG_ERR, "%s%d: can't handle af%d\n",
	 ifp->if_name, ifp->if_unit, dst->sa_family);
#endif
	m_freem(m0);
	return (EAFNOSUPPORT);
    }

    din = (struct sockaddr_in *) dst;

    if (din->sin_addr.s_addr == IA_SIN(ifp->if_addrlist)->sin_addr.s_addr) {
	/*
	 * Place interface pointer before the data
	 * for the receiving protocol.
	 */
	if (m0->m_off <= MMAXOFF &&
	 m0->m_off >= MMINOFF + sizeof(struct ifnet *)) {
	    m0->m_off -= sizeof(struct ifnet *);
	    m0->m_len += sizeof(struct ifnet *);
	}
	else {
	    MGET(m, M_DONTWAIT, MT_HEADER);
	    if (m == (struct mbuf *) 0)
		return (ENOBUFS);
	    m->m_off = MMINOFF;
	    m->m_len = sizeof(struct ifnet *);
	    m->m_next = m0;
	    m0 = m;
	}
	*(mtod(m0, struct ifnet **)) = ifp;
	s = splimp();
	ifp->if_opackets++;
	ifq = &ipintrq;
	if (IF_QFULL(ifq)) {
	    IF_DROP(ifq);
	    m_freem(m0);
	    splx(s);
	    return (ENOBUFS);
	}
	IF_ENQUEUE(ifq, m0);
	schednetisr(NETISR_IP);
	ifp->if_ipackets++;
	splx(s);
	return (0);
    }
    else {
	m_freem(m0);
    }

    return EHOSTUNREACH;
}

/*
 * vif_ifioctl:
 *      Process an ioctl request.
 */
/*
 * ARGSUSED 
 */
vif_ifioctl(ifp, cmd, data)
register struct ifnet *ifp;
int cmd;
caddr_t data;
{
    int error = 0;

    switch (cmd) {

	case SIOCSIFADDR:
	    ifp->if_flags |= IFF_UP;
	    /*
	     * Everything else is done at a higher level.
	     */
	    break;

#ifdef __hpux
	case SIOCSIFNETMASK:
	    break;
#endif

	default:
	    error = EINVAL;
    }
    return (error);
}

/*
 * vif_ioctl:
 *      ioctl() entry point.... simple
 */
vif_ioctl(dev, cmd, arg, mode)
dev_t dev;
int cmd;
caddr_t arg;
int mode;
{
    int unit;

    unit = minor(dev);
    if ((unit < 0) || (unit >= vifs_inited))
	return ENXIO;

    return EINVAL;
}

#if defined(_AUX_SOURCE)
#define VIF_VERSION	"1.3"
void				/* general initialization */
vif_init()
{
    printf("vif: A/UX 3.x.x virtual interface driver ver jj-%s\n", VIF_VERSION);
}

#endif
