/*
 * Copyright (c) 1988 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by the University of California, Berkeley.  The name of the
 * University may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

/*
 * This program comes from Van Jacobson <van@helios.ee.lbl.gov>
 *
 * Extensively modified by Eric Wassenaar, Nikhef-H, <e07@nikhef.nl>
 *
 * The source of this particular version of the program is available
 * via anonymous ftp from machine 'ftp.nikhef.nl' [192.16.199.1]
 * in the directory '/pub/network' as 'traceroute.tar.Z'
 */

#ifndef lint
static char Version[] = "@(#)traceroute.c	e07@nikhef.nl (Eric Wassenaar) 941215";
#endif

#if defined(apollo) && defined(lint)
#define __attribute(x)
#endif

/*
 * traceroute host  - trace the route ip packets follow going to "host".
 *
 * Attempt to trace the route an ip packet would follow to some
 * internet host.  We find out intermediate hops by launching probe
 * packets with a small ttl (time to live) then listening for an
 * icmp "time exceeded" reply from a gateway.  We start our probes
 * with a ttl of one and increase by one until we get an icmp "port
 * unreachable" (which means we got to "host") or hit a max (which
 * defaults to 30 hops & can be changed with the -m flag).  Three
 * probes (change with -q flag) are sent at each ttl setting and a
 * line is printed showing the ttl, address of the gateway and
 * round trip time of each probe.  If the probe answers come from
 * different gateways, the address of each responding system will
 * be printed.  If there is no response within a 3 sec. timeout
 * interval (changed with the -w flag), a "*" is printed for that
 * probe.
 * If the -l flag is used the ttl from the icmp reply will
 * be printed, otherwise it will be printed only if it has an
 * unexpected value.
 *
 * Probe packets are UDP format.  We don't want the destination
 * host to process them so the destination port is set to an
 * unlikely value (if some clod on the destination is using that
 * value, it can be changed with the -p flag).
 *
 * A sample use might be:
 *
 *     [yak 71]% traceroute nis.nsf.net.
 *     traceroute to nis.nsf.net (35.1.1.48), 30 hops max, 56 byte packet
 *      1  helios.ee.lbl.gov (128.3.112.1)  19 ms  19 ms  0 ms
 *      2  lilac-dmc.Berkeley.EDU (128.32.216.1)  39 ms  39 ms  19 ms
 *      3  lilac-dmc.Berkeley.EDU (128.32.216.1)  39 ms  39 ms  19 ms
 *      4  ccngw-ner-cc.Berkeley.EDU (128.32.136.23)  39 ms  40 ms  39 ms
 *      5  ccn-nerif22.Berkeley.EDU (128.32.168.22)  39 ms  39 ms  39 ms
 *      6  128.32.197.4 (128.32.197.4)  40 ms  59 ms  59 ms
 *      7  131.119.2.5 (131.119.2.5)  59 ms  59 ms  59 ms
 *      8  129.140.70.13 (129.140.70.13)  99 ms  99 ms  80 ms
 *      9  129.140.71.6 (129.140.71.6)  139 ms  239 ms  319 ms
 *     10  129.140.81.7 (129.140.81.7)  220 ms  199 ms  199 ms
 *     11  nic.merit.edu (35.1.1.48)  239 ms  239 ms  239 ms
 *
 * Note that lines 2 & 3 are the same.  This is due to a buggy
 * kernel on the 2nd hop system -- lbl-csam.arpa -- that forwards
 * packets with a zero ttl.
 *
 * A more interesting example is:
 *
 *     [yak 72]% traceroute allspice.lcs.mit.edu.
 *     traceroute to allspice.lcs.mit.edu (18.26.0.115), 30 hops max
 *      1  helios.ee.lbl.gov (128.3.112.1)  0 ms  0 ms  0 ms
 *      2  lilac-dmc.Berkeley.EDU (128.32.216.1)  19 ms  19 ms  19 ms
 *      3  lilac-dmc.Berkeley.EDU (128.32.216.1)  39 ms  19 ms  19 ms
 *      4  ccngw-ner-cc.Berkeley.EDU (128.32.136.23)  19 ms  39 ms  39 ms
 *      5  ccn-nerif22.Berkeley.EDU (128.32.168.22)  20 ms  39 ms  39 ms
 *      6  128.32.197.4 (128.32.197.4)  59 ms  119 ms  39 ms
 *      7  131.119.2.5 (131.119.2.5)  59 ms  59 ms  39 ms
 *      8  129.140.70.13 (129.140.70.13)  80 ms  79 ms  99 ms
 *      9  129.140.71.6 (129.140.71.6)  139 ms  139 ms  159 ms
 *     10  129.140.81.7 (129.140.81.7)  199 ms  180 ms  300 ms
 *     11  129.140.72.17 (129.140.72.17)  300 ms  239 ms  239 ms
 *     12  * * *
 *     13  128.121.54.72 (128.121.54.72)  259 ms  499 ms  279 ms
 *     14  * * *
 *     15  * * *
 *     16  * * *
 *     17  * * *
 *     18  ALLSPICE.LCS.MIT.EDU (18.26.0.115)  339 ms  279 ms  279 ms
 *
 * (I start to see why I'm having so much trouble with mail to
 * MIT.)  Note that the gateways 12, 14, 15, 16 & 17 hops away
 * either don't send ICMP "time exceeded" messages or send them
 * with a ttl too small to reach us.  14 - 17 are running the
 * MIT C Gateway code that doesn't send "time exceeded"s.  God
 * only knows what's going on with 12.
 *
 * The silent gateway 12 in the above may be the result of a bug in
 * the 4.[23]BSD network code (and its derivatives):  4.x (x <= 3)
 * sends an unreachable message using whatever ttl remains in the
 * original datagram.  Since, for gateways, the remaining ttl is
 * zero, the icmp "time exceeded" is guaranteed to not make it back
 * to us.  The behavior of this bug is slightly more interesting
 * when it appears on the destination system:
 *
 *      1  helios.ee.lbl.gov (128.3.112.1)  0 ms  0 ms  0 ms
 *      2  lilac-dmc.Berkeley.EDU (128.32.216.1)  39 ms  19 ms  39 ms
 *      3  lilac-dmc.Berkeley.EDU (128.32.216.1)  19 ms  39 ms  19 ms
 *      4  ccngw-ner-cc.Berkeley.EDU (128.32.136.23)  39 ms  40 ms  19 ms
 *      5  ccn-nerif35.Berkeley.EDU (128.32.168.35)  39 ms  39 ms  39 ms
 *      6  csgw.Berkeley.EDU (128.32.133.254)  39 ms  59 ms  39 ms
 *      7  * * *
 *      8  * * *
 *      9  * * *
 *     10  * * *
 *     11  * * *
 *     12  * * *
 *     13  rip.Berkeley.EDU (128.32.131.22)  59 ms !  39 ms !  39 ms !
 *
 * Notice that there are 12 "gateways" (13 is the final
 * destination) and exactly the last half of them are "missing".
 * What's really happening is that rip (a Sun-3 running Sun OS3.5)
 * is using the ttl from our arriving datagram as the ttl in its
 * icmp reply.  So, the reply will time out on the return path
 * (with no notice sent to anyone since icmp's aren't sent for
 * icmp's) until we probe with a ttl that's at least twice the path
 * length.  I.e., rip is really only 7 hops away.  A reply that
 * returns with a ttl of 1 is a clue this problem exists.
 * Traceroute prints a "!" after the time if the ttl is <= 1.
 * Since vendors ship a lot of obsolete (DEC's Ultrix, Sun 3.x) or
 * non-standard (HPUX) software, expect to see this problem
 * frequently and/or take care picking the target host of your
 * probes.
 *
 * Other possible annotations after the time are !H, !N, !P (got a host,
 * network or protocol unreachable, respectively), !S or !F (source
 * route failed or fragmentation needed -- neither of these should
 * ever occur and the associated gateway is busted if you see one).  If
 * almost all the probes result in some kind of unreachable, traceroute
 * will give up and exit.
 *
 * Still other annotations are (ttl="n") and (ttl="n"!), where "n" is the
 * ttl from the icmp reply packet we received.  The first form will be
 * printed for every packet if you use the -l option.  The second form
 * will be printed (with or without -l) if this ttl is an unexpected value.
 * We expect that the return path from the n'th hop will contain n hops,
 * otherwise we know the reply packet is coming back via a different path than
 * it went out on.  Unfortunately, not everyone uses the same starting ttl
 * value for icmp messages.  The common ones used by routers are 29
 * (Proteon 8.1 and lower software), 59 (Proteon 8.2), 255 (cisco, BSD
 * since 4.3 tahoe).  30 and 60 are also often used by hosts, and probably
 * by some routers, because they were the BSD TCP ttl values.  This makes
 * some "off by one" return paths hard to detect, you might try removing
 * OLD_BSD_TCP and NEW_BSD_TCP from the case statement if this annoys you.
 * If you are using the lsrr (-g) code with the -l code you will see many
 * bogus "!".  
 *
 * With the -l option you will see the TTL value of the ICMP msg that
 * came back to you, as in:
 *
 * nettlerash> traceroute -l rip.berkeley.edu
 * traceroute to rip.berkeley.edu (128.32.131.22), 30 hops max, 40 byte packets
 *  1  ccngw-ner-cc.Berkeley.EDU (128.32.136.23)  0 ms (ttl=59)  10 ms (ttl=59)
 *  2  ccn-nerif35.Berkeley.EDU (128.32.168.35)  10 ms (ttl=58)  10 ms (ttl=58)
 *  3  csgw.Berkeley.EDU (128.32.133.254)  10 ms (ttl=253)  20 ms (ttl=253)
 *  4  rip.Berkeley.EDU (128.32.131.22)  30 ms (ttl=252)  30 ms (ttl=252)
 *
 * This shows that from nettlerash the route goes through two proteons (ttl=
 * 59, then 58), followed by a router using max_ttl to rip which also uses
 * max_ttl.  (-l and printing ttl stuff added by cliff@berkeley.edu.)
 *
 * Notes
 * -----
 * This program must be run by root or be setuid.  (I suggest that
 * you *don't* make it setuid -- casual use could result in a lot
 * of unnecessary traffic on our poor, congested nets.)
 *
 * This program requires a kernel mod that does not appear in any
 * system available from Berkeley:  A raw ip socket using proto
 * IPPROTO_RAW must interpret the data sent as an ip datagram (as
 * opposed to data to be wrapped in a ip datagram).  See the README
 * file that came with the source to this program for a description
 * of the mods I made to /sys/netinet/raw_ip.c.  Your mileage may
 * vary.  But, again, ANY 4.x (x < 4) BSD KERNEL WILL HAVE TO BE
 * MODIFIED TO RUN THIS PROGRAM.
 *
 * The udp port usage may appear bizarre (well, ok, it is bizarre).
 * The problem is that an icmp message only contains 8 bytes of
 * data from the original datagram.  8 bytes is the size of a udp
 * header so, if we want to associate replies with the original
 * datagram, the necessary information must be encoded into the
 * udp header (the ip id could be used but there's no way to
 * interlock with the kernel's assignment of ip id's and, anyway,
 * it would have taken a lot more kernel hacking to allow this
 * code to set the ip id).  So, to allow two or more users to
 * use traceroute simultaneously, we use this task's pid as the
 * source port (the high bit is set to move the port number out
 * of the "likely" range).  To keep track of which probe is being
 * replied to (so times and/or hop counts don't get confused by a
 * reply that was delayed in transit), we increment the destination
 * port number before each probe.
 *
 * Don't use this as a coding example.  I was trying to find a
 * routing problem and this code sort-of popped out after 48 hours
 * without sleep.  I was amazed it ever compiled, much less ran.
 *
 * I stole the idea for this program from Steve Deering.  Since
 * the first release, I've learned that had I attended the right
 * IETF working group meetings, I also could have stolen it from Guy
 * Almes or Matt Mathis.  I don't know (or care) who came up with
 * the idea first.  I envy the originators' perspicacity and I'm
 * glad they didn't keep the idea a secret.
 *
 * Tim Seaver, Ken Adelman and C. Philip Wood provided bug fixes and/or
 * enhancements to the original distribution.
 *
 * I've hacked up a round-trip-route version of this that works by
 * sending a loose-source-routed udp datagram through the destination
 * back to yourself.  Unfortunately, SO many gateways botch source
 * routing, the thing is almost worthless.  Maybe one day...
 *
 *  -- Van Jacobson (van@helios.ee.lbl.gov)
 *     Tue Dec 20 03:50:13 PST 1988
 */

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <signal.h>
#include <signal.h>
#include <setjmp.h>
#include <netdb.h>

#include <sys/types.h>		/* not always automatically included */
#include <sys/param.h>
#include <sys/socket.h>

#include <sys/file.h>
#include <sys/ioctl.h>
#include <sys/time.h>

#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>
#include <netinet/udp.h>

#undef NOERROR			/* in <sys/streams.h> on solaris 2.x */
#include <arpa/nameser.h>
#include <resolv.h>

#include "exit.h"		/* exit codes come from <sysexits.h> */
#include "port.h"		/* various portability definitions */
#include "conf.h"		/* various configuration definitions */

#ifndef MAXDNAME
#define MAXDNAME	256	/* maximum length of domain name */
#endif

typedef int	bool;		/* boolean type */
#define TRUE	1
#define FALSE	0

#ifdef lint
#define EXTERN
#else
#define EXTERN extern
#endif

EXTERN int errno;
EXTERN res_state_t _res;	/* defined in res_init.c */
extern char *version;		/* program version number */

#define IPHDRSZ		20	/* actually sizeof(struct ip) */
#define UDPHDRSZ	8	/* actually sizeof(struct udphdr) */

struct ip ippkt;		/* temporary storage */
struct udphdr udppkt;		/* temporary storage */

struct probe			/* format of a (udp) probe packet */
{
	struct ip ip;
	struct udphdr udp;
	struct timeval tv;	/* time packet left */
	u_char seq;		/* sequence number of this packet */
	u_char ttl;		/* ttl packet left with */
};

#ifdef IP_MAXPACKET
#define MAXPACKET IP_MAXPACKET	/* max ip packet size */
#else
#define MAXPACKET 65535
#endif

#define TIMLEN sizeof(struct timeval)
#define PROBELEN (IPHDRSZ + UDPHDRSZ + TIMLEN + 2)
#define MAXDATA	(MAXPACKET - PROBELEN)

u_char opacket[MAXPACKET];	/* last output (udp) packet */
u_char ipacket[MAXPACKET];	/* last inbound (icmp) packet */

int sendsock;			/* send (udp) socket file descriptor */
int recvsock;			/* receive (icmp) socket file descriptor */

struct sockaddr myaddr;		/* our own source address */
struct sockaddr toaddr;		/* address to try to reach */
struct sockaddr fromaddr;	/* address we got response from */

struct sockaddr_in *me   = (struct sockaddr_in *)&myaddr;
struct sockaddr_in *to   = (struct sockaddr_in *)&toaddr;
struct sockaddr_in *from = (struct sockaddr_in *)&fromaddr;

u_short ident;			/* packet ident */
u_short port = 32768+666;	/* start udp dest port # for probe packets */
int datalen = PROBELEN;		/* how much data in raw ip packet */
int tos = 0;			/* type of service */
int dontfrag = 0;		/* prevent fragmentation if set to IP_DF */
int needfrag = 0;		/* new mtu size if more fragmentation needed */

char *source = NULL;		/* specific source for multi-homed sender */
char *program;			/* name of this program */

int nprobes = 3;		/* number of probe packets to send */
int max_ttl = OLD_BSD_TCP;	/* max number of hops */

int options = 0;		/* socket options */
int waitsecs = 3;		/* time to wait for response (seconds) */
int stopsecs = 0;		/* time to wait between probes (seconds) */
bool verbose = FALSE;		/* print additional information */
bool numeric = FALSE;		/* print addresses numerically */
bool ttlflag = FALSE;		/* print ttl values */
bool alladdr = FALSE;		/* trace all addresses of destination host */

char hostnamebuf[MAXDNAME+1];
char *hostname;			/* remote host to query about */

#define MAXADDRS	35	/* max address count from gethostnamadr.c */

ipaddr_t hostaddr[MAXADDRS];	/* multiple destination addresses */
int naddrs = 0;			/* count of destination addresses */

#ifndef MAX_IPOPTLEN
#define MAX_IPOPTLEN	40
#endif
#define IPOPT_HDRLEN	3	/* actually IPOPT_MINOFF - 1 */

#define MAXIPOPT 9		/* MAX_IPOPTLEN-IPOPT_MINOFF / INADDRSZ */
#define MAXLSRR (MAXIPOPT-1)	/* leave room for destination address */

ipaddr_t lsrraddr[MAXLSRR];	/* loose source route addresses */
int nlsrr = 0;			/* count of loose source route addresses */

#define MAXGATE (MAXLSRR*MAXADDRS)

ipaddr_t gateaddr[MAXGATE];	/* all known gateway addresses */
int ngate = 0;			/* count of all known gateway addresses */

#define NOT_DOTTED_QUAD	((ipaddr_t)-1)

#include "defs.h"		/* declaration of functions */

#ifdef DEBUG
#define assert(condition)\
{\
	if (!(condition))\
	{\
		(void) fprintf(stderr, "assertion botch: ");\
		(void) fprintf(stderr, "%s(%d): ", __FILE__, __LINE__);\
		(void) fprintf(stderr, "%s\n", "condition");\
		exit(EX_SOFTWARE);\
	}\
}
#else
#define assert(condition)
#endif

char Usage[] =
"\
Usage: %s [-g gateway] [-m maxhops] [-q nqueries] host [packetsize]\n\
Flags: [-a] [-f] [-l] [-n] [-r] [-v]\n\
Other: [-c stoptime] [-p port] [-s source_addr] [-t tos] [-w waittime]\
";

/*
** MAIN -- Start of program traceroute
** -----------------------------------
**
**	Exits:
**		Various possibilities from <sysexits.h>
**		EX_OK if the destination host has been reached.
*/

int
main(argc, argv)
int argc;
char *argv[];
{
	bool got_somewhere = FALSE;	/* set if some address was reached */
	ipaddr_t addr;
	struct in_addr inaddr;
	struct hostent *hp;
	register int i;
	register char *option;

#ifdef obsolete
	assert(sizeof(u_int) == 4);	/* probably paranoid */
	assert(sizeof(u_short) == 2);	/* perhaps less paranoid */
	assert(sizeof(ipaddr_t) == 4);	/* but this is critical */
#endif /*obsolete*/

	program = rindex(argv[0], '/');
	if (program++ == NULL)
		program = argv[0];

/*
 * Synchronize stdout and stderr in case output is redirected.
 */
	linebufmode(stdout);

/*
 * Initialize resolver. Shorter timeout values are set later.
 */
	(void) res_init();

/*
 * Check command line options.
 */
	while (argc > 1 && argv[1][0] == '-')
	{
	    for (option = &argv[1][1]; *option != '\0'; option++)
	    {
		switch (*option)
		{
		    case 'a':
			/* probe all of multiple addresses */
			alladdr = TRUE;
			break;

		    case 'c':
			/* time to delay between probes to cisco */
			if (argv[2] == NULL || argv[2][0] == '-')
				fatal("Missing stoptime value");
			stopsecs = atoi(argv[2]);
			if (stopsecs < 1)
				fatal("Invalid stoptime value %s", argv[2]);
			argc--, argv++;
			break;

		    case 'd':
			/* socket level debugging */
			options |= SO_DEBUG;
			break;

		    case 'f':
			/* prevent fragmentation */
			dontfrag = IP_DF;
			break;

		    case 'g':
			/* loose source route gateway address */
			if (argv[2] == NULL || argv[2][0] == '-')
				fatal("Missing gateway name");
			if (nlsrr >= MAXLSRR)
				fatal("Maximum %s gateways", itoa(MAXLSRR));
			lsrraddr[nlsrr] = getaddr(argv[2]);
			nlsrr++;
			argc--, argv++;
			break;

		    case 'l':
			/* list ttl values */
			ttlflag = TRUE;
			break;

		    case 'm':
			/* maximum hop count */
			if (argv[2] == NULL || argv[2][0] == '-')
				fatal("Missing maxhops value");
			max_ttl = atoi(argv[2]);
			if (max_ttl < 1)
				fatal("Invalid maxhops value %s", argv[2]);
			argc--, argv++;
			break;

		    case 'n':
			/* print numeric addresses */
			numeric = TRUE;
			break;

		    case 'p':
			/* specific udp port number */
			if (argv[2] == NULL || argv[2][0] == '-')
				fatal("Missing port value");
			port = atoi(argv[2]);
			if (port < 1)
				fatal("Invalid port value %s", argv[2]);
			argc--, argv++;
			break;

		    case 'q':
			/* number of probe packets per hop */
			if (argv[2] == NULL || argv[2][0] == '-')
				fatal("Missing nqueries value");
			nprobes = atoi(argv[2]);
			if (nprobes < 1)
				fatal("Invalid nqueries value %s", argv[2]);
			argc--, argv++;
			break;

		    case 'r':
			/* bypass normal routing */
			options |= SO_DONTROUTE;
			break;

		    case 's':
			/* explicit own source address */
			if (argv[2] == NULL || argv[2][0] == '-')
				fatal("Missing source address");
			source = argv[2];
			argc--, argv++;
			break;

		    case 't':
			/* type-of-service */
			if (argv[2] == NULL || argv[2][0] == '-')
				fatal("Missing tos value");
			tos = atoi(argv[2]);
			if (tos < 1)
				fatal("Invalid tos value %s", argv[2]);
			if (tos > 255)
				fatal("Maximum tos value %s", itoa(255));
			argc--, argv++;
			break;

		    case 'v':
			/* verbose printout */
			verbose = TRUE;
			break;

		    case 'w':
			/* timeout to wait for replies */
			if (argv[2] == NULL || argv[2][0] == '-')
				fatal("Missing waittime value");
			waitsecs = atoi(argv[2]);
			if (waitsecs < 1)
				fatal("Invalid waittime value %s", argv[2]);
			argc--, argv++;
			break;

		    case 'V':
			printf("Version %s\n", version);
			exit(EX_OK);

		    default:
			fatal(Usage, program);
		}
	    }

	    argc--, argv++;
	}

/*
 * Fetch (mandatory) remote host address(es) to probe.
 */
	if (argc < 2)
		fatal(Usage, program);

	hostname = argv[1];
	addr = inet_addr(hostname);
	inaddr.s_addr = addr;

	if (addr == NOT_DOTTED_QUAD)
	{
		hp = gethostbyname(hostname);
		if (hp == NULL)
		{
			error("Unknown host %s", hostname);
			exit(EX_NOHOST);
		}

		hostname = strcpy(hostnamebuf, hp->h_name);
		for (i = 0; i < MAXADDRS && hp->h_addr_list[i]; i++)
		{
			bcopy(hp->h_addr_list[i], (char *)&inaddr, INADDRSZ);
			hostaddr[i] = inaddr.s_addr;
		}
		naddrs = alladdr ? i : 1;
	}
	else
	{
		hp = gethostbyaddr((char *)&inaddr, INADDRSZ, AF_INET);
		if (hp != NULL)
			hostname = strcpy(hostnamebuf, hp->h_name);
		else
			hostname = strcpy(hostnamebuf, hostname);
		hostaddr[0] = addr;
		naddrs = 1;
	}

/*
 * Check for optional extra data size of output packet.
 */
	if (argc > 2)
	{
		datalen = atoi(argv[2]);
		if (datalen < 1)
			fatal("Invalid packet size %s", argv[2]);
		if (datalen < PROBELEN)
			fatal("Minimum packet size %s", itoa(PROBELEN));
		if (datalen > MAXPACKET)
			fatal("Maximum packet size %s", itoa(MAXPACKET));
	}

	/* rest is undefined */
	if (argc > 3)
		fatal(Usage, program);

/*
 * Miscellaneous initialization.
 */
	/* becomes the udp source port */
	ident = (getpid() & 0xFFFF) | 0x8000;

	/* set shorter nameserver timeout */
	_res.retry = 2;		/* number  of retries, default = 4 */
	_res.retrans = 3;	/* timeout in seconds, default = 5 or 6 */

/*
 * Allocate IP output socket and ICMP input socket.
 */
	check_proto();
	get_socket();

/*
 * Define explicit source address in output packet, if specified.
 */
	/* initialize default source address */
	bzero((char *)&myaddr, sizeof(myaddr));
	me->sin_family = AF_INET;
	me->sin_addr.s_addr = INADDR_ANY;
	me->sin_port = 0;

	if (source)
	{
		addr = inet_addr(source);
		if (addr == NOT_DOTTED_QUAD)
			fatal("Illegal source address %s", source);

		me->sin_addr.s_addr = addr;
#ifndef IP_HDRINCL
		if (bind(sendsock, &myaddr, sizeof(myaddr)) < 0)
		{
			perror("bind");
			exit(EX_OSERR);
		}
#endif
	}

/*
 * Start of main loop.
 */
	/* don't need special privileges any more */
	(void) setuid(getuid());

	/* probe all addresses successively */
	for (i = 0; i < naddrs; i++)
	{
		if (ping(hostaddr[i]))
			got_somewhere = TRUE;
	}

	/* indicate success or failure */
	exit(got_somewhere ? EX_OK : EX_UNAVAILABLE);
	/*NOTREACHED*/
}

/*
** PING -- Probe given address of destination host
** -----------------------------------------------
**
**	Wrapper for trace_route() to hide interrupt handling.
**
**	Returns:
**		TRUE if the destination was reached.
**		FALSE otherwise.
*/

jmp_buf interrupt_buf;

sigtype_t /*ARGSUSED*/
interrupt(sig)
int sig;
{
	longjmp(interrupt_buf, 1);
	/*NOTREACHED*/
}


bool
ping(addr)
ipaddr_t addr;				/* address of destination */
{
	bool got_there;			/* set if destination reached */

	/* abort when interrupted */
	if (setjmp(interrupt_buf) != 0)
	{
		(void) fprintf(stderr, "\n(interrupt)\n");
		(void) signal(SIGINT, SIG_DFL);
		return(FALSE);
	}

	/* allow premature abort */
	(void) signal(SIGINT, interrupt);
	got_there = trace_route(addr);
	(void) signal(SIGINT, SIG_DFL);
	return(got_there);
}

/*
** FATAL -- Abort program when illegal option encountered
** ------------------------------------------------------
**
**	Returns:
**		Aborts after issuing error message.
*/

void /*VARARGS1*/
fatal(fmt, a, b, c, d)
char *fmt;				/* format of message */
char *a, *b, *c, *d;			/* optional arguments */
{
	(void) fprintf(stderr, fmt, a, b, c, d);
	(void) fprintf(stderr, "\n");
	exit(EX_USAGE);
}


void /*VARARGS1*/
error(fmt, a, b, c, d)
char *fmt;				/* format of message */
char *a, *b, *c, *d;			/* optional arguments */
{
	(void) fprintf(stderr, fmt, a, b, c, d);
	(void) fprintf(stderr, "\n");
}

/*
** CHECK_PROTO -- Check protocol numbers
** -------------------------------------
**
**	traceroute uses protocol numbers as defined in <netinet/in.h>.
**	Verify whether they correspond to the values in /etc/protocols.
**	This is probably rather paranoid.
*/

void
check_proto()
{
	struct protoent *proto;

	proto = getprotobyname("ip");
	if (proto == NULL)
	{
		(void) fprintf(stderr, "ip: unknown protocol\n");
		exit(EX_OSFILE);
	}
	if (proto->p_proto != IPPROTO_IP)
	{
		(void) fprintf(stderr, "ip protocol %d should be %d\n",
			proto->p_proto, IPPROTO_IP);
		exit(EX_CONFIG);
	}

	proto = getprotobyname("icmp");
	if (proto == NULL)
	{
		(void) fprintf(stderr, "icmp: unknown protocol\n");
		exit(EX_OSFILE);
	}
	if (proto->p_proto != IPPROTO_ICMP)
	{
		(void) fprintf(stderr, "icmp protocol %d should be %d\n",
			proto->p_proto, IPPROTO_ICMP);
		exit(EX_CONFIG);
	}

	proto = getprotobyname("tcp");
	if (proto == NULL)
	{
		(void) fprintf(stderr, "tcp: unknown protocol\n");
		exit(EX_OSFILE);
	}
	if (proto->p_proto != IPPROTO_TCP)
	{
		(void) fprintf(stderr, "tcp protocol %d should be %d\n",
			proto->p_proto, IPPROTO_TCP);
		exit(EX_CONFIG);
	}

	proto = getprotobyname("udp");
	if (proto == NULL)
	{
		(void) fprintf(stderr, "udp: unknown protocol\n");
		exit(EX_OSFILE);
	}
	if (proto->p_proto != IPPROTO_UDP)
	{
		(void) fprintf(stderr, "udp protocol %d should be %d\n",
			proto->p_proto, IPPROTO_UDP);
		exit(EX_CONFIG);
	}
}

/*
** GET_SOCKET -- Allocate and configure output and input socket
** ------------------------------------------------------------
**
**	A raw ip socket is allocated for sending the probe packets.
**	A raw udp socket is allocated instead on those platforms
**	which do not support raw ip socket manipulation.
**	A raw icmp socket is allocated for receiving icmp messages.
**	These sockets can be allocated only by root.
**	Extra socket options are set as requested on the command line.
**
**	Inputs:
**		Global ``datalen'' contains the ip packet size.
**		This is used to set the size of the socket buffer.
**
**	Outputs:
**		Global ``recvsock'' is the input  socket descriptor.
**		Global ``sendsock'' is the output socket descriptor.
*/

void
get_socket()
{
	int on = 1;

/*
 * Allocate input socket to receive replies.
 */
	recvsock = socket(AF_INET, SOCK_RAW, IPPROTO_ICMP);
	if (recvsock < 0)
	{
		perror("icmp socket");
		exit(EX_OSERR);
	}

	if (options & SO_DEBUG)
		(void) setsockopt(recvsock, SOL_SOCKET, SO_DEBUG,
			(char *)&on, sizeof(on));

	if (options & SO_DONTROUTE)
		(void) setsockopt(recvsock, SOL_SOCKET, SO_DONTROUTE,
			(char *)&on, sizeof(on));

/*
 * Allocate output socket to send probe packets.
 */
#ifdef SYSV_UDPPROTO
	sendsock = socket(AF_INET, SOCK_RAW, IPPROTO_UDP);
	if (sendsock < 0)
	{
		perror("udp socket");
		exit(EX_OSERR);
	}
#else
	sendsock = socket(AF_INET, SOCK_RAW, IPPROTO_RAW);
	if (sendsock < 0)
	{
		perror("raw socket");
		exit(EX_OSERR);
	}
#endif /*SYSV_UDPPROTO*/

#ifdef SO_SNDBUF
	if (setsockopt(sendsock, SOL_SOCKET, SO_SNDBUF,
		(char *)&datalen, sizeof(datalen)) < 0)
	{
		perror("setsockopt: SO_SNDBUF");
		exit(EX_OSERR);
	}
#endif

#ifdef IP_HDRINCL
	if (setsockopt(sendsock, IPPROTO_IP, IP_HDRINCL,
		(char *)&on, sizeof(on)) < 0)
	{
		perror("setsockopt: IP_HDRINCL");
		exit(EX_OSERR);
	}
#endif

	if (options & SO_DEBUG)
		(void) setsockopt(sendsock, SOL_SOCKET, SO_DEBUG,
			(char *)&on, sizeof(on));

	if (options & SO_DONTROUTE)
		(void) setsockopt(sendsock, SOL_SOCKET, SO_DONTROUTE,
			(char *)&on, sizeof(on));
}

/*
** SET_OPTIONS -- Initialize IP options
** ------------------------------------
**
**	Special IP options are setup as requested on the command line.
**	This will extend the IP header with the IP options buffer.
**	Not all platforms may support this, even if IP_OPTIONS exists.
**
**	Inputs:
**		``nlsrr'' intermediate gateway addresses are stored in
**		``lsrraddr[]''. Final destination address is ``toaddr''.
*/

#ifdef IP_OPTIONS

void
set_options()
{
	register int i;
	u_char ipoptbuf[MAX_IPOPTLEN];
	u_char *ipopt = ipoptbuf;

	bzero((char *)ipoptbuf, sizeof(ipoptbuf));

	ipopt[IPOPT_OPTVAL] = IPOPT_NOP;
	ipopt++;

	ipopt[IPOPT_OPTVAL] = IPOPT_LSRR;
	ipopt[IPOPT_OLEN]   = IPOPT_HDRLEN + (nlsrr + 1)*INADDRSZ;
	ipopt[IPOPT_OFFSET] = IPOPT_MINOFF;
	ipopt += IPOPT_HDRLEN;

	for (i = 0; i < nlsrr; i++)
	{
		struct in_addr inaddr;

		inaddr.s_addr = lsrraddr[i];
		bcopy((char *)&inaddr, (char *)ipopt, INADDRSZ);
		ipopt += INADDRSZ;
	}

	bcopy((char *)&to->sin_addr, (char *)ipopt, INADDRSZ);

	if (setsockopt(sendsock, IPPROTO_IP, IP_OPTIONS,
		(char *)ipoptbuf, sizeof(ipoptbuf)) < 0)
	{
		perror("setsockopt: IP_OPTIONS");
		exit(EX_OSERR);
	}
}

#endif /*IP_OPTIONS*/

/*
** TRACE_ROUTE -- Probe given address of destination host
** ------------------------------------------------------
**
**	Returns:
**		TRUE if the destination was reached.
**		FALSE otherwise.
*/

bool
trace_route(addr)
ipaddr_t addr;				/* address of destination */
{
	int ttl;			/* current hop count */
	int probe;			/* current probe number */
	int seq = 0;			/* packet sequence number */
	int len = datalen;		/* total ip packet size */
	bool got_there = FALSE;		/* set if destination reached */

	bzero((char *)&toaddr, sizeof(toaddr));
	to->sin_family = AF_INET;
	to->sin_addr.s_addr = addr;
	to->sin_port = 0;

	printf("traceroute to %s (%s)", hostname, inet_ntoa(to->sin_addr));
	if (source)
		printf(" from %s", source);
	printf(" %d hops max, %d byte packets\n", max_ttl, datalen);

/*
 * Set ip options if loose source routing addresses were specified.
 */
#ifdef IP_OPTIONS

	if (nlsrr > 0)
		set_options();

#else /*IP_OPTIONS*/

	if (nlsrr > 0)
		error("gateway source routing not supported");

#endif /*IP_OPTIONS*/

/*
 * Start emitting probe packets.
 */
	for (ttl = 1; ttl <= max_ttl; ttl++)
	{
		ipaddr_t lastaddr = 0;
		int timeouts = 0;
		int unreachable = 0;
		int mtu = 0;
		int star = 0;

		printf("%2d ", ttl);

		for (probe = 1; probe <= nprobes; probe++)
		{
			register int cc;
			int icmpcode;
			struct ip *ip;
			struct timeval sendtime, recvtime;
			struct timeval waittime, nexttime;

			(void) fflush(stdout);

			/* initialize maximum timeout */
			waittime.tv_sec = waitsecs;
			waittime.tv_usec = 0;

		/*
		 * Send probe packet and wait for a valid reply packet.
		 * Decrement waittime in case we are catching packets that
		 * do not belong to us, e.g. because somebody is ping-ing.
		 */
			(void) gettimeofday(&sendtime, (struct timezone *)NULL);

			cc = send_probe(sendsock, ++seq, ttl, len);
			if (cc == 0)
				mtu = select_mtu(len);
retry:
			(void) gettimeofday(&nexttime, (struct timezone *)NULL);

			if (cc > 0)
				cc = recv_reply(recvsock, &waittime);

			if (cc <= 0)
			{
				star++;
				timeouts++;
				continue;	/* next probe */
			}

			(void) gettimeofday(&recvtime, (struct timezone *)NULL);

			icmpcode = check_reply(ipacket, cc, seq);
			if (icmpcode < 0)
			{
				(void) tvdelta(&recvtime, &nexttime);
				(void) tvdelta(&waittime, &recvtime);
				goto retry;	/* not our reply */
			}

		/*
		 * Valid reply packet received. Print remote address
		 * (if different from previous case) and rtt value.
		 */
			if (from->sin_addr.s_addr != lastaddr)
			{
				print_from(ipacket, cc, from->sin_addr);
				lastaddr = from->sin_addr.s_addr;
			}

			for (; star > 0; star--)
				printf(" *");

			printf("  %d ms", tvdelta(&recvtime, &sendtime));

		/*
		 * Check ttl for unexpected values. Print if necessary.
		 */
			ip = (struct ip *)ipacket;
			switch (ip->ip_ttl + (ttl - 1))
			{
			    case MAX_START_TTL:
			    case NETWARE_TTL:
			    case OLD_PROTEON_TTL:
			    case NEW_PROTEON_TTL:
			    case OLD_BSD_TCP:
			    case NEW_BSD_TCP:
				if (ttlflag)
					printf(" (ttl=%d)", (int)ip->ip_ttl);
				break;

			    default:
				if (ttlflag || (nlsrr == 0))
					printf(" (ttl=%d!)", (int)ip->ip_ttl);
				break;
			}

		/*
		 * Certain routers will erroneously report ICMP_UNREACH_PORT
		 * if they are specified as loose source route gateway hosts.
		 */
			if (nlsrr > 0 && icmpcode == ICMP_UNREACH_PORT)
			{
				if (gatewayaddr(from->sin_addr))
				{
					printf(" !G");
					icmpcode = ICMP_TIMXCEED;
				}
			}

		/*
		 * Check icmp reply code for special cases.
		 */
			switch (icmpcode)
			{
			    case ICMP_TIMXCEED:
				/* intermediate host -- nothing to do */
				break;

			    case ICMP_UNREACH_PORT:
				/* destination host -- normal case */
				if (ip->ip_ttl <= 1)
					printf(" !");
				got_there = TRUE;
				break;

			    case ICMP_UNREACH_PROTOCOL:
				printf(" !P");
				got_there = TRUE;
				break;

			    case ICMP_UNREACH_NET:
				printf(" !N");
				unreachable++;
				break;

			    case ICMP_UNREACH_HOST:
				printf(" !H");
				unreachable++;
				break;

			    case ICMP_UNREACH_SRCFAIL:
				printf(" !S");
				unreachable++;
				break;

			    case ICMP_UNREACH_NEEDFRAG:
				mtu = needfrag;
				if (mtu > 0)
					printf(" !F=%d", mtu);
				else
					printf(" !F");
				if (mtu == 0)
					mtu = select_mtu(len);
				unreachable++;
				break;

			    default:
				printf(" ?");
				break;
			}

			(void) fflush(stdout);

			/* optionally pause between probes */
			if (stopsecs != 0)
				(void) sleep((unsigned)stopsecs);

		} /* for probe <= nprobes */

		for (; star > 0; star--)
			printf(" *");

		printf("\n");

		/* terminate if destination reached */
		if (got_there)
			break;

		/* retry same ttl with shorter mtu size */
		if (mtu >= PROBELEN && mtu < len)
		{
			printf("(switching to new packet size %d)\n", mtu);
			len = mtu;
			ttl--;
			continue;
		}

		/* abort if all probes yielded unreachable */
		if (unreachable && (unreachable + timeouts >= nprobes))
			break;

	} /* for ttl <= max_ttl */

	/* indicate success or failure */
	return(got_there);
}

/*
** SEND_PROBE -- Send out probe packet
** -----------------------------------
**
**	Inputs:
**		Global ``ident''    contains our identification.
**		Global ``port''     contains the base port offset.
**		Global ``tos''      contains the type of service.
**		Global ``dontfrag'' contains the don't fragment flag.
**		Global ``toaddr''   contains destination address.
**		Global ``myaddr''   contains own source address.
**
**	Returns:
**		Number of bytes sent.
**		0 if packet size is too large.
**		-1 in case of other errors.
*/

int
send_probe(sock, seq, ttl, len)
int sock;				/* output socket descriptor */
int seq;				/* sequence no. determines port */
int ttl;				/* current ttl value */
int len;				/* size of ip packet */
{
	u_char *pkt = opacket;
	struct ip *ip = &ippkt;		/* temporary ip header */
	struct udphdr *udp = &udppkt;	/* temporary udp header */
	struct timeval tv;
	int udplen = len - IPHDRSZ;	/* size of udp packet */
	register int n;

/*
 * Fill in ip header.
 */
	/* ip->ip_v   = (u_char)  IPVERSION;	/* version */
	/* ip->ip_hl  = (u_char)  0;		/* header length */
	   ip->ip_tos = (u_char)  tos;		/* type of service */
	   ip->ip_len = (short)   len;		/* total size */
	/* ip->ip_id  = (u_short) 0;		/* identification */
	   ip->ip_off = (short)   dontfrag;	/* fragment offset */
	   ip->ip_ttl = (u_char)  ttl;		/* time-to-live */
	   ip->ip_p   = (u_char)  IPPROTO_UDP;	/* protocol */
	   ip->ip_sum = (u_short) 0;		/* checksum */
	   ip->ip_src = me->sin_addr;		/* source address */
	   ip->ip_dst = to->sin_addr;		/* destination address */

/*
 * Fill in udp header.
 */
	udp->uh_sport = htons(ident);		/* source port */
	udp->uh_dport = htons(port+seq);	/* destination port */
	udp->uh_ulen  = htons((u_short)udplen);	/* udp size */
	udp->uh_sum   = 0;			/* checksum */

/*
 * Construct the output packet.
 */
	bcopy((char *)ip, (char *)pkt, IPHDRSZ);
	pkt += IPHDRSZ;

	bcopy((char *)udp, (char *)pkt, UDPHDRSZ);
	pkt += UDPHDRSZ;

/*
 * Fill in remainder of output packet (actually unused).
 */
	(void) gettimeofday(&tv, (struct timezone *)NULL);
	bcopy((char *)&tv, (char *)pkt, TIMLEN);
	pkt += TIMLEN;

	pkt[0] = (u_char)seq;
	pkt[1] = (u_char)ttl;

/*
 * If we are sending only the udp packet instead of the ip packet,
 * we request an explicit ttl value for the resulting ip packet.
 */
#ifdef SYSV_UDPPROTO

	if (setsockopt(sock, IPPROTO_IP, IP_TTL,
		(char *)&ttl, sizeof(ttl)) < 0)
	{
		perror("setsockopt: IP_TTL");
		exit(EX_OSERR);
	}

#endif /*SYSV_UDPPROTO*/

/*
 * Send out the probe packet.
 */
#ifdef SYSV_UDPPROTO

	/* skip the ip header */
	udp = (struct udphdr *)&opacket[IPHDRSZ];
	n = sendto(sock, (char *)udp, udplen, 0, &toaddr, sizeof(toaddr));
	if (n > 0)
		n += IPHDRSZ;

#else /*SYSV_UDPPROTO*/

	/* send raw ip packet */
	ip = (struct ip *)&opacket[0];
	n = sendto(sock, (char *)ip, len, 0, &toaddr, sizeof(toaddr));

#endif /*SYSV_UDPPROTO*/

	if (n < 0 || n != len)
	{
#ifdef EMSGSIZE
		/* message too long */
		if (n < 0 && errno == EMSGSIZE)
			return(0);
#endif /*EMSGSIZE*/
#if defined(apollo)
		/* message too long */
		if (n < 0 && errno == EIO && dontfrag)
			return(0);
#endif
		(void) fprintf(stderr, "\n");
		if (n < 0)
			perror("sendto");
		else
			error("sendto: truncated packet to %s", hostname);

		/* failure */
		return(-1);
	}

	/* successfully sent */
	return(n);
}

/*
** RECV_REPLY -- Wait for arrival of input packet
** ----------------------------------------------
**
**	Outputs:
**		Input packet  is stored in global ``ipacket''.
**		Reply address is stored in global ``fromaddr''.
**
**	Returns:
**		Number of bytes received.
**		-1 in case of error or timeout.
*/

int
recv_reply(sock, wait)
int sock;				/* input socket descriptor */
struct timeval *wait;			/* timeout value */
{
	fd_set fds;
	int fromlen;
	int len;
	register int n;

	/* FD_ZERO(&fds); */
	bzero((char *)&fds, sizeof(fds));
	FD_SET(sock, &fds);

/*
 * Wait for arrival of input packet, or timeout.
 */
	n = select(FD_SETSIZE, &fds, (fd_set *)0, (fd_set *)0, wait);
	if (n <= 0)
	{
		if (n == 0)
			errno = ETIMEDOUT;
		return(-1);
	}

/*
 * Fake an error if nothing was actually read.
 */
	len = sizeof(ipacket);
	fromlen = sizeof(fromaddr);
	n = recvfrom(sock, (char *)ipacket, len, 0, &fromaddr, &fromlen);
	if (n == 0)
		errno = ECONNRESET;
	return(n);
}

/*
** CHECK_REPLY -- Check validity of received input packet
** ------------------------------------------------------
**
**	Inputs:
**		Global ``ident'' contains our identification.
**		Global ``port''  contains the base port offset.
**
**	Returns:
**		ICMP_TIMXCEED if ttl dropped to zero (intermediate host).
**		Subcode of ICMP_UNREACH if destination unreachable.
**		Got to the final destination if ICMP_UNREACH_PORT.
**		-1 if not one of the above.
**
**	Side effects:
**		Sets ``needfrag'' if further fragmentation is needed
**		and the new mtu size to use has been told.
*/

int
check_reply(buf, cc, seq)
u_char *buf;				/* address of input packet */
int cc;					/* total size of input packet */
int seq;				/* sequence no. determines port */
{
	struct ip *ip;
	struct icmp *icp;
	struct udphdr *udp;
	int iphdrlen;
	int type;			/* type of icmp input packet */
	int code;			/* subcode of type */

/*
 * The input packet contains the ip header in front of the icmp packet.
 * Make sure the packet contains the icmp header after the ip header.
 */
	ip = (struct ip *)buf;
	iphdrlen = ip->ip_hl << 2;

	if (ip->ip_p != IPPROTO_ICMP)
	{
		if (verbose)
			printf("\n%d bytes from %s: (not icmp)\n",
				cc, inet_ntoa(from->sin_addr));
		return(-1);
	}

	if (cc < iphdrlen + ICMP_MINLEN)
	{
		if (verbose)
			printf("\n%d bytes from %s: (too short)\n",
				cc, inet_ntoa(from->sin_addr));
		return(-1);
	}

	/* move to the icmp packet */
	icp = (struct icmp *)(buf + iphdrlen);
	cc -= iphdrlen;

	/* pickup type and its subcode */
	type = icp->icmp_type;
	code = icp->icmp_code;

/*
 * Make sure this is really our response packet.
 * The original ip packet is stored in the icmp message after the
 * icmp header, with the original udp header after the ip header.
 */
	ip = &icp->icmp_ip;	/* original ip packet */
	iphdrlen = ip->ip_hl << 2;
	udp = (struct udphdr *)((u_char *)ip + iphdrlen);

	if (cc < (ICMP_MINLEN + iphdrlen + 2*INT16SZ))
		goto bad;
	if (ip->ip_p != IPPROTO_UDP)
		goto bad;
	if (udp->uh_sport != htons(ident))
		goto bad;
	if (udp->uh_dport != htons(port+seq))
		goto bad;

/*
 * Check for fragmentation problems.
 * This should happen only if the dontfrag option is set, and
 * we hit a machine which cannot handle the current mtu size.
 * Some of the more recent routers return the new mtu size.
 */
	needfrag = 0;
	if (type == ICMP_UNREACH && code == ICMP_UNREACH_NEEDFRAG)
		needfrag = ntohl(icp->icmp_void) & 0xFFFF;

/*
 * Check the type of the icmp packet.
 * We should get ICMP_TIMXCEED if we got to an intermediate host.
 * We get ICMP_UNREACH if the desired host is currently unreachable,
 * or if we got to the final destination itself.
 */

	if (type == ICMP_TIMXCEED && code == ICMP_TIMXCEED_INTRANS)
		return(type);

	if (type == ICMP_UNREACH)
		return(code);

/*
 * Useless packet.
 * No detailed printout is done. See ping for more elaborate version.
 */
bad:
	if (verbose)
	{
		ip = (struct ip *)buf;
		printf("\n%d bytes from %s to %s: ",
			cc, inet_ntoa(from->sin_addr), inet_ntoa(ip->ip_dst));
		printf("icmp type %d (%s) code %d\n",
			type, pr_type(type), code);
		printf("\n");
	}

	return(-1);
}

/*
** PR_TYPE -- Return type of icmp input packet
** -------------------------------------------
**
**	Returns:
**		Pointer to name of packet type.
*/

char *
pr_type(type)
int type;				/* type value of icmp packet */
{
	static char *ttab[] =
	{
	/*  0 */	"echo reply",
	/*  1 */	"ICMP 1",
	/*  2 */	"ICMP 2",
	/*  3 */	"dest unreachable",
	/*  4 */	"source quench",
	/*  5 */	"redirect",
	/*  6 */	"ICMP 6",
	/*  7 */	"ICMP 7",
	/*  8 */	"echo request",
	/*  9 */	"ICMP 9",
	/* 10 */	"ICMP 10",
	/* 11 */	"time exceeded",
	/* 12 */	"parameter problem",
	/* 13 */	"timestamp request",
	/* 14 */	"timestamp reply",
	/* 15 */	"info request",
	/* 16 */	"info reply",
	/* 17 */	"mask request",
	/* 18 */	"mask reply"
	};

	if (type < 0 || type > 18)
		return("unknown icmp type");

	return(ttab[type]);
}

/*
** PRINT_FROM -- Print the address from which we got an input packet
** -----------------------------------------------------------------
*/

void
print_from(buf, cc, inaddr)
u_char *buf;				/* address of input packet */
int cc;					/* number of bytes received */
struct in_addr inaddr;			/* internet address */
{
	int iphdrlen;
	struct ip *ip;

	ip = (struct ip *)buf;
	iphdrlen = ip->ip_hl << 2;
	cc -= iphdrlen;

	if (numeric)
		printf(" %s", inet_ntoa(inaddr));
	else
		printf(" %s (%s)", inetname(inaddr), inet_ntoa(inaddr));

	if (verbose)
		printf (" %d bytes to %s", cc, inet_ntoa(ip->ip_dst));
}

/*
** INETNAME -- Construct internet address representation
** -----------------------------------------------------
**
**	Returns:
**		Pointer to string containing address representation.
*/

char *
inetname(inaddr)
struct in_addr inaddr;			/* internet address to convert */
{
	static char buf[MAXDNAME+1];	/* buffer to store representation */
	struct hostent *hp;

/*
 * Look up the hostname corresponding with the given address.
 */
	if (!numeric && (inaddr.s_addr != INADDR_ANY))
		hp = gethostbyaddr((char *)&inaddr, INADDRSZ, AF_INET);
	else
		hp = NULL;

/*
 * Construct the proper address representation.
 */
	if (hp != NULL)
		(void) strcpy(buf, hp->h_name);
	else
		(void) sprintf(buf, "%s", inet_ntoa(inaddr));
	return(buf);
}

/*
** GETADDR -- Fetch internet address of host
** -----------------------------------------
**
**	Returns:
**		Internet address of given host.
**		Aborts if address could not be determined.
**
**	Only the first address as returned by the resolver is used.
**	This address is used as loose source route gateway address.
**
**	As a side effect, we will try to determine all its addresses
**	and add them to the global address list ``gateaddr[]'' which
**	has ``ngate'' entries.
*/

ipaddr_t
getaddr(host)
char *host;				/* host name or dotted quad */
{
	char hostbuf[MAXDNAME+1];
	register int i;
	ipaddr_t addr;
	struct in_addr inaddr;
	struct hostent *hp;

	addr = inet_addr(host);
	inaddr.s_addr = addr;

	if (addr == NOT_DOTTED_QUAD)
	{
		hp = gethostbyname(host);
		if (hp == NULL)
		{
			error("Unknown host %s", host);
			exit(EX_NOHOST);
		}

		bcopy(hp->h_addr, (char *)&inaddr, INADDRSZ);
		addr = inaddr.s_addr;
	}
	else
	{
		hp = gethostbyaddr((char *)&inaddr, INADDRSZ, AF_INET);
		if (hp != NULL)
		{
			host = strcpy(hostbuf, hp->h_name);
			hp = gethostbyname(host);
		}
	}

/*
 * Append all found addresses to the global address list.
 */
	for (i = 0; hp != NULL && hp->h_addr_list[i] && ngate < MAXGATE; i++)
	{
		bcopy(hp->h_addr_list[i], (char *)&inaddr, INADDRSZ);
		if (!gatewayaddr(inaddr))
		{
			gateaddr[ngate] = inaddr.s_addr;
			ngate++;
		}
	}

	if (hp == NULL && ngate < MAXGATE)
	{
		if (!gatewayaddr(inaddr))
		{
			gateaddr[ngate] = inaddr.s_addr;
			ngate++;
		}
	}

	return(addr);
}

/*
** GATEWAYADDR -- Check whether address belongs to known gateways
** --------------------------------------------------------------
**
**	Returns:
**		TRUE if the address is listed in the global table.
**		FALSE otherwise.
*/

bool
gatewayaddr(inaddr)
struct in_addr inaddr;			/* internet address to check */
{
	register int i;

	for (i = 0; i < ngate; i++)
	{
		if (inaddr.s_addr == gateaddr[i])
			break;
	}

	return(i < ngate ? TRUE : FALSE);
}

/*
** IN_CKSUM -- Compute checksum for IP packets
** -------------------------------------------
**
**	The complete packet must have been constructed.
**	The checksum field to be computed must be zero.
**
**	Returns:
**		Computed checksum.
*/

#ifdef obsolete

u_short
in_cksum(buf, len)
u_short *buf;				/* start of packet */
int len;				/* length of packet in bytes */
{
	register u_short *w = buf;	/* address of next 16-bit word */
	register int nleft = len;	/* remaining 16-bit words */
	register int sum = 0;		/* 32-bit accumulator */
	u_short answer = 0;

/*
 * Our algorithm is simple, using a 32 bit accumulator (sum),
 * we add sequential 16 bit words to it, and at the end, fold back
 * all the carry bits from the top 16 bits into the lower 16 bits.
 */
	while (nleft > 1)
	{
		sum += *w++;
		nleft -= 2;
	}

	if (nleft == 1)
	{
		*(u_char *)(&answer) = *(u_char *)w;
		sum += answer;
	}

/*
 * Add back carry outs from top 16 bits to low 16 bits.
 */
	sum = (sum >> 16) + (sum & 0xFFFF);	/* add hi 16 to low 16 */
	sum += (sum >> 16);			/* add carry */
	answer = ~sum;				/* truncate to 16 bits */
	return(answer);
}

#endif /*obsolete*/

/*
** IN_CKSUM -- Compute checksum for IP packets
** -------------------------------------------
**
**	The complete packet must have been constructed.
**	The checksum field to be computed must be zero.
**
**	Returns:
**		Computed checksum.
*/

#ifdef notyet

u_short
in_cksum(buf, len)
u_short *buf;				/* start of packet */
int len;				/* length of packet in bytes */
{
	register u_char *b = (u_char *)buf;
	register int n = len;		/* remaining 16-bit words */
	register int sum = 0;		/* 32-bit accumulator */
	u_short answer;

/*
 * Our algorithm is simple, using a 32 bit accumulator (sum),
 * we add sequential 16 bit words to it, and at the end, fold back
 * all the carry bits from the top 16 bits into the lower 16 bits.
 */
	while (n > 1)
	{
		answer = (b[0] << 8) | b[1];
		sum += answer;
		b += 2; n -= 2;
	}

	if (n == 1)
	{
		answer = (b[0] << 8);
		sum += answer;
	}

/*
 * Add back carry outs from top 16 bits to low 16 bits.
 */
	sum = (sum >> 16) + (sum & 0xFFFF);	/* add hi 16 to low 16 */
	sum += (sum >> 16);			/* add carry */
	answer = ~sum & 0xFFFF;			/* truncate to 16 bits */
	answer = htons(answer);			/* correct order */
	return(answer);
}

#endif /*notyet*/

/*
** TVDELTA -- Subtract two timeval structs
** ---------------------------------------
**
**	Returns:
**		Value of time difference in milliseconds.
**
**	The difference of the two values is stored back in the first.
*/

int
tvdelta(t2, t1)
struct timeval *t2;			/* latest timeval */
struct timeval *t1;			/* oldest timeval */
{
	register int millisec;

	t2->tv_usec -= t1->tv_usec;
	if (t2->tv_usec < 0)
	{
		t2->tv_sec--;
		t2->tv_usec += 1000000;
	}

	if (t2->tv_sec < t1->tv_sec)
	{
		t2->tv_sec = 0;
		t2->tv_usec = 0;
	}
	else
		t2->tv_sec -= t1->tv_sec;

	millisec = t2->tv_sec*1000 + (t2->tv_usec + 500)/1000;
	return(millisec);
}

/*
** SELECT_MTU -- Select next lower mtu size from table
** ---------------------------------------------------
**
**	Returns:
**		New mtu size, if available.
**		Old mtu size otherwise.
*/

int
select_mtu(len)
int len;				/* current ip packet size */
{
	register int i;
	register int mtu;

	for (i = 0; (mtu = mtusize[i]) > 0; i++)
	{
		if (mtu < len)
			break;
	}

	if (mtu > 0)
		len = mtu;

	return(len);
}

/*
** ITOA -- Convert integer value to ascii string
** ---------------------------------------------
**
**	Returns:
**		Pointer to string.
*/

char *
itoa(n)
int n;					/* value to convert */
{
	static char buf[30];

	(void) sprintf(buf, "%d", n);
	return(buf);
}
