/*
 * macenv.c = The MacOS environment setter-upper-function :)
 *
 * Copyright 1994, Jim Jagielski (jim@jagubox.gsfc.nasa.gov)
 * All rights reserved.
 *
 * Permission is granted to copy and create derivative works for any
 * non-commercial purpose, provided this copyright notice is preserved
 * in all copies of source code, or included in human readable form
 * and conspicuously displayed on all copies of object code or
 * distribution media.
 *
 * This software is provided on an AS-IS basis and the author makes
 * no warrantee of any kind.
 */

#include "ngroups.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <utmp.h>

#include <stdio.h>
#include <grp.h>

#ifdef BSD
#include <strings.h>
#define strchr index
#else
#include <string.h>
#include <memory.h>
#endif

/*
 * Additional stuff to check SCSI devices
 */
#include <sys/ssioctl.h>
#include <sys/gdisk.h>
#include <fcntl.h>

#include "config.h"
#include "pwd.h"

#ifdef USE_SYSLOG
#include <syslog.h>

#ifndef LOG_WARN
#define LOG_WARN	LOG_WARNING
#endif
#endif

#ifndef lint
static char sccsid[] = "@(#)macenv.	1.1	08:07:13	15 Dec 1994";
#endif

extern	struct utmp utent;

long	strtol();
#ifdef HAVE_ULIMIT
long	ulimit();
#endif

void	addenv();
extern	char *getdef_str();
extern	int getdef_bool();
extern	int getdef_num();

struct	stat statbuf;
int	makedir = 0;
void	check_dir();

#define MAC_INTERFACE	"/dev/uinter0"
#define DESKTOP		"/Desktop Folder"
#define THE_CONSOLE	"/dev/console"
#define CDROM		0x5

/*
 * macenv - setup MacOS environment for user.
 *
 * This section of code is called to setup the MacOS environment for
 * the current user. In particular it adjusts:
 *
 *   /dev/uinter0                                      owner|group|600
 *   /dev/ttyC*                                        owner|group|622
 *   /dev/dsk/c[123456]d0s[25,26,27,28,29,30,31]       owner|group|600
 *   /dev/rdsk/c[123456]d0s[25,26,27,28,29,30,31]      owner|group|600
 *   /Desktop Folder -> $HOME/.mac/hostname/Desktop Folder
 *
 * It also will create DeskTop Folder (and Trash) if need be
 */

void 
macenv(info)
struct passwd *info;
{
    extern	int errno;
    char	buf[BUFSIZ];
    char	tty[sizeof utent.ut_line + 8];
    char	hostname[MAXHOSTNAMELEN];
    char	device[BUFSIZ];
    int 	i;
    int 	j;
    uid_t	usr;
    gid_t	grp;
    struct scsiinquiry	ddat;
    int		fd;

    extern	char *ttyname();

    /*
     * Determine the name of the TTY device which the user is logging
     * into.
     */

    if (utent.ut_line[0] != '/')
	(void) strcat(strcpy(tty, "/dev/"), utent.ut_line);
    else
	(void) strcpy(tty, utent.ut_line);

    /*
     * Only setup the MacOS if the TTY device we are logging into is
     * /dev/console. Check both the entry in ut_line as well as
     * the device that stdin is connected to
     */

    if (strcmp(tty, THE_CONSOLE) || strcmp(ttyname(0), THE_CONSOLE))
	return;

    /*
     * At this point, we know that we'll need to do some stuff that
     * only root should be able to. Make sure we are root and complain
     * (but continue) if not
     */

    if (geteuid() != 0) {
	(void) sprintf(buf,
	"Not root! May not correctly setup Mac environment...\n");
#ifdef USE_SYSLOG
	syslog(LOG_WARN, buf);
#endif
	fprintf(stderr, "%s", buf);
	fflush(stderr);
    }

    /*
     * See if the user's home exists... Just return if not, since
     * setup() will catch this. We do this, since below we check for
     * $HOME/.mac/hostname/Desktop Folder and create one if it doesn't
     * exist... well, we should be sure that $HOME exists 1st
     */

    if (stat(info->pw_dir, &statbuf))
	return;

    /*
     * Reduce typing
     */

    usr = info->pw_uid;
    grp = info->pw_gid;

    /*
     * First handle the interface device. We report errors in
     * setting this up, since it's pretty important
     */

    if (chown(MAC_INTERFACE, usr, grp)) {
	(void) sprintf(buf, "unable to chown %s for user `%s'\n",
	 MAC_INTERFACE, info->pw_name);
#ifdef USE_SYSLOG
	syslog(LOG_WARN, buf);
#endif
	perror(buf);
    }

    if (chmod(MAC_INTERFACE, 0600)) {
	(void) sprintf(buf, "unable to chmod %s for user `%s'\n",
	 MAC_INTERFACE, info->pw_name);
#ifdef USE_SYSLOG
	syslog(LOG_WARN, buf);
#endif
	perror(buf);
    }

    /*
     * Now we handle the MacOS-partition device files... Older versions
     * lacked s25, s26... s29, so we don't bother reporting errors.
     */

    for (i = 0; i <= 6; i++) {
	for (j = 25; j <= 30; j++) {
	    (void) sprintf(device, "/dev/dsk/c%dd0s%d", i, j);
	    (void) chown(device, usr, grp);
	    (void) chmod(device, 0600);
	    (void) sprintf(device, "/dev/rdsk/c%dd0s%d", i, j);
	    (void) chown(device, usr, grp);
	    (void) chmod(device, 0600);
	}
    }

    /*
     * Now we do the CommandShell ttys
     */

    for (i = 0; i <= 15; i++) {
	(void) sprintf(device, "/dev/ttyC%x", i);
	(void) chown(device, usr, grp);
	(void) chmod(device, 0622);
    }

    /*
     * Now we check for any connected CD-ROMS. If they exist, we must own
     * them to access it from the MacOS interface
     */

#ifdef GD_DRIVEINQUIRY
    for (i = 0; i <= 6; i++) {
	(void) sprintf(device, "/dev/rdsk/c%dd0s31", i);
	if ((fd = open(device, O_RDONLY)) != -1) {
	    if (ioctl(fd, GD_DRIVEINQUIRY, (u_char *)&ddat) != -1) {
		if ( (ddat.devtype == CDROM) && (ddat.removable == 1) &&
		 (ddat.qualifier == 0) ) {
		    (void) chown(device, usr, grp);
		    (void) chmod(device, 0600);
		    (void) sprintf(device, "/dev/dsk/c%dd0s31", i);
		    (void) chown(device, usr, grp);
		    (void) chmod(device, 0600);
		}
	    }
	    close(fd);
	}
    }
#endif

    /*
     * We now handle the symlink to "our" Desktop folder. For
     * completeness, we see if the danged thing even exists. If not,
     * then we make it. We also check for Trash, although maybe we
     * shouldn't care... What the heck!
     */

    if (gethostname(hostname, sizeof(hostname) - 1))
	strcpy(hostname, "notnamed");
    hostname[MAXHOSTNAMELEN - 1] = '\0';	/* just in case */

    /*
     * Do a quick check to see if our Desktop Folder exists.
     * If so, we are golden, if not, then step through and create
     * it as needed.
     */

    (void) sprintf(device, "%s/.mac/%s/Desktop Folder", info->pw_dir,
     hostname);
    if (stat(device, &statbuf)) {

	/*
	 * Ugg, it doesn't exist... Let's find out what does.
	 *
	 * 1st check for $HOME/.mac (we know that $HOME exists)
	 */

	(void) sprintf(device, "%s/.mac", info->pw_dir);
	check_dir(device, usr, grp, 0755);

	/*
	 *  Now look for $HOME/.mac/hostname
	 */

	(void) strcat(device, "/");
	(void) strcat(device, hostname);
	check_dir(device, usr, grp, 0755);

	/*
	 * Okay... now we create $HOME/.mac/hostname/Desktop Folder
	 */

	makedir = 1;		/* we _know_ we have to make this */
	(void) strcat(device, "/Desktop Folder");
	check_dir(device, usr, grp, 0777);
    }

    /*
     * Now that we know that our Desktop Folder exists, we make
     * the required symlink
     */

    (void) unlink(DESKTOP);
    if (symlink(device, DESKTOP)) {
	(void) sprintf(buf, "unable to create symlink: %s -> %s\n", DESKTOP,
	 device);
#ifdef USE_SYSLOG
	syslog(LOG_WARN, buf);
#endif
	perror(buf);
    }
    (void) chown(DESKTOP, usr, grp);

    /*
     * Check for Trash as well, while we're here
     */

    (void) sprintf(device, "%s/.mac/%s/Trash", info->pw_dir, hostname);
    check_dir(device, usr, grp, 0755);

    /*
     * And we are _done_!
     */
}

/*
 * check_dir - See if the directory exists. If not, create it
 */

void
check_dir(path, usr, grp, mode)
char	*path;
uid_t	usr;
gid_t	grp;
int	mode;
{
    char	buf[BUFSIZ];

    if (makedir || stat(path, &statbuf)) {
	/*
	 * Nope... gotta make it. Report errors!
	 */
	if (mkdir(path, mode)) {
	    (void) sprintf(buf, "unable to create dir `%s'\n", path);
#ifdef USE_SYSLOG
	    syslog(LOG_WARN, buf);
#endif
	    perror(buf);
	}
	if (chown(path, usr, grp)) {
	    (void) sprintf(buf, "unable to chown dir `%s'\n", path);
#ifdef USE_SYSLOG
	    syslog(LOG_WARN, buf);
#endif
	    perror(buf);
	}
	makedir = 1;		/* save some unneeded stat()s */
    }
}
