// Robin Hood Web Server - A web server for BeOS
// Copyright (C) 1999 Joe Kloss

// This program is free software; you can redistribute it and/or 
// modify it under the terms of the GNU General Public License 
// as published by the Free Software Foundation; either version 2 
// of the License, or (at your option) any later version. 

// This program is distributed in the hope that it will be useful, 
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
// GNU General Public License for more details. 

// You should have received a copy of the GNU General Public License 
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

// Contact Info:
// Author: Joe Kloss
// E-mail: axly@deltanet.com
// Postal Address: 25002 Ravenswood, Lake Forest, CA 92630, USA

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <time.h>
#include "RobinHoodServer.h"
#include "RHCGI.h"
#include "RHGet.h"
#include "RHLogger.h"

static const char *kDATE_FORMAT = "%a, %d %b %Y %H:%M:%S";
static const char *kDEFAULT_HOST = "Default-Host";

RHServer::RHServer( FieldList *virtualHosts, const char *defaultFileName )
	: HTTPFileServer( defaultFileName )
{
	static int32 lastsn = 0;
	sn = atomic_add( &lastsn, 1 );
	
	this->virtualHosts = virtualHosts;
}

RHServer::~RHServer( void )
{
	
}

HTTPHandler *RHServer::NewCopy( void )
{
	HTTPHandler *handler;
	handler = new RHServer( virtualHosts, defaultFileName );
	return handler;
}

bool RHServer::MessageReceived( HTTPRequest *request )
{
	log_printf( "%ld Request-Line: %s\n", sn, request->GetRequestLine() );
	
	const char *header;
	for ( int32 i = 0; (header = request->HeaderAt(i)); i++ )
		log_printf( "%ld Header: %s\n", sn, header );
	
	return HTTPFileServer::MessageReceived( request );
}

void RHServer::ConnectionOpened( const char *remote_addr )
{
	char			date[256];
	time_t			now;
	tm				*brokentime;
		
	now = time( NULL );
	brTimeLock.Lock();
	brokentime = localtime( &now );
	strftime (date, 256, kDATE_FORMAT, brokentime);
	brTimeLock.Unlock();
	
	log_printf( "%ld Open: %s %s\n", sn, remote_addr, date );
}

void RHServer::ConnectionClosed( status_t status )
{
	char			date[256];
	time_t			now;
	tm				*brokentime;
	
	now = time( NULL );
	brTimeLock.Lock();
	brokentime = localtime( &now );
	strftime (date, 256, kDATE_FORMAT, brokentime);
	brTimeLock.Unlock();
	
	log_printf( "%ld Close: %s\n", sn, date );
}

status_t RHServer::FindWebDirectory( void )
{
	// ****
	// Set Web Directory from host name and virtual hosts table.
	// ****
	
	int32			fieldSize = 1024;
	char			fieldValue[1024];
	
	// Get host_name
	// If hostName was found in URI, don't bother
	if( brURI->host[0] == 0 )
	{
		if( !request->FindHeader( kHEAD_HOST, brURI->host, 64 ) )
			strcpy( brURI->host, kDEFAULT_HOST ); // Set to defualt host
	}
	
	if( !virtualHosts->FindField( brURI->host, fieldValue, fieldSize ) )
	{
		if( !virtualHosts->FindField( kDEFAULT_HOST, fieldValue, fieldSize ) )
		{
			// Could not find a default web directory
			response->SetHTMLMessage( 500, "500 No default web directory!" ); // Internal Server Error
			request->SendReply( response );
			return B_ERROR;
		}
	}
	// Set web directory
	if( SetWebDirectory( fieldValue ) != B_OK )
	{
		// Error setting directory
		response->SetHTMLMessage( 500, "500 Web directory not found!" ); // Internal Server Error
		request->SendReply( response );
		return B_ERROR;
	}
	return B_OK;
}

bool RHServer::HandlePost( void )
{
	// Set Web Directory from virtual hosts table
	if( FindWebDirectory() != B_NO_ERROR )
		return true;
	
	// Is it the CGI directory?
	if( strncmp( brURI->path, "cgi-bin/", 8 ) == 0 )
		return InvokeCGI();
	else
	{
		response->SetHTMLMessage( 404 ); // Not Found
		request->SendReply( response );
		return true;
	}
}

bool RHServer::InvokeCGI( void )
{
	return invoke_cgi( request, response, brURI, &webDirectory, sn );
}

bool RHServer::HandleGet( void )
{
	// Set Web Directory from virtual hosts table
	if( FindWebDirectory() != B_NO_ERROR )
		return true;
	return http_get( request, response, brURI, &webDirectory, sn );
}