// Robin Hood Web Server - A web server for BeOS
// Copyright (C) 1999 Joe Kloss

// This program is free software; you can redistribute it and/or 
// modify it under the terms of the GNU General Public License 
// as published by the Free Software Foundation; either version 2 
// of the License, or (at your option) any later version. 

// This program is distributed in the hope that it will be useful, 
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
// GNU General Public License for more details. 

// You should have received a copy of the GNU General Public License 
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

// Contact Info:
// Author: Joe Kloss
// E-mail: axly@deltanet.com
// Postal Address: 25002 Ravenswood, Lake Forest, CA 92630, USA

#include <string.h>
#include <stdio.h>
#include <File.h>
#include <Directory.h>
#include <NodeInfo.h>
#include <stdio.h>
#include "RHGet.h"
#include "RHCGI.h"
#include "RHSSI.h"
#include "parsedate.h"
#include "ByteRanges.h"
#include "RHAuthenticate.h"
#include "RHLogger.h"

bool http_get( HTTPRequest *request, HTTPResponse *response, brokenURI *brURI, BPath *webDirectory, int32 sn )

{
	// Is it the CGI directory? If so, invoke the CGI insted of getting a file.
	if( strncmp( brURI->path, "cgi-bin/", 8 ) == 0 )
		return invoke_cgi( request, response, brURI, webDirectory, sn );
	
	// Temporary buffers
	int32			fieldSize = 1024;
	char			fieldValue[1024];
	char			headBuffer[1024];
	
	BFile			theFile, gzFile;
	BPath			absPath;
	bool			gz = false;
	
	// Is gzip allowed? Then see if a gziped file is available.
	if( request->FindHeader( kHEAD_ACCEPT_ENCODING, fieldValue, fieldSize ) )
	{
		if( strstr( fieldValue, "gzip" )||strstr( fieldValue, "x-gzip" ) )
		{
			char	gzPath[2051];
			strcpy( gzPath, brURI->path );
			strcat( gzPath, ".gz" );
			absPath.SetTo( webDirectory->Path(), gzPath );
			
			// Was gz file found? If not, send normal file.
			if( gzFile.SetTo( absPath.Path(), B_READ_ONLY ) == B_NO_ERROR )
				gz = true;
		}
	}
	
	absPath.SetTo( webDirectory->Path(), brURI->path );
	// Check Authorization
	if( !http_authenticate( request, response, &absPath, S_IROTH ) )
	{
		log_status_line( sn, response->GetStatusLine() );
		return true;
	}
	
	// Was the file found?
	if( theFile.SetTo( absPath.Path(), B_READ_ONLY ) == B_NO_ERROR )
	{
		int32		statusCode = 200;
		
		time_t		modtime;
		theFile.GetModificationTime( &modtime );
		
		// Get file length and mime type
		int32		entityLength, contentLength;
		BNodeInfo	theNode( &theFile );
		char		mimeType[1024];
		bool		isSSI;
		
		theNode.GetType( mimeType );
		
		// Detect SSI file
		isSSI = (strcmp( mimeType, "text/x-server-parsed-html" ) == 0);
		
		if( !isSSI )
		{
			// Conditional Get by modifiaction date Header?
			if( request->FindHeader( kHEAD_IF_MODIFIED, fieldValue, fieldSize ) )
			{
				time_t ifmod = parsedate(fieldValue, -1);
				
				if( (ifmod != -1)&&(ifmod >= modtime) )
				{
					response->SetHTMLMessage( 304 ); // Not Modified
					request->SendReply( response );
					return true;
				}
			}
			
			// Conditional Get by modifiaction date Header?
			if( request->FindHeader( kHEAD_IF_UNMODIFIED, fieldValue, fieldSize ) )
			{
				time_t ifmod = parsedate(fieldValue, -1);
				if( (ifmod != -1)&&(ifmod <= modtime) )
				{
					response->SetHTMLMessage( 412 ); // Pre-condition failed
					request->SendReply( response );
					return true;
				}
			}
		}
		off_t size;
		
		if( gz )
			gzFile.GetSize( &size );
		else
			theFile.GetSize( &size );
		
		entityLength= size;
		
		
		// Add Date header
		time_t			now;
		struct tm 		*brokentime;
		
		now = time( NULL );
		brTimeLock.Lock();
		brokentime = gmtime( &now );
		strftime (fieldValue, 256, kHTTP_DATE, brokentime);
		brTimeLock.Unlock();
		
		response->AddHeader( kHEAD_DATE, fieldValue );
		
		// Add last modified header
		brTimeLock.Lock();
		brokentime = gmtime (&modtime);
		strftime (fieldValue, fieldSize, kHTTP_DATE, brokentime);
		brTimeLock.Unlock();
		
		response->AddHeader( kHEAD_LAST_MODIFIED, fieldValue );
		
		// Add server name header
		response->AddHeader( kHEAD_SERVER, "RobinHood" );
		
		// Should we invoke SSI?
		if( isSSI )
		{
			BMallocIO		output;
			
			BPath			CWD( absPath );
			CWD.GetParent( &CWD );
			invoke_ssi( &theFile, &output, webDirectory, &CWD, sn );
			contentLength = output.BufferLength();
			response->SetContentLength( contentLength );
			
			// Tell client not to cache SSI generated response
			response->AddHeader( "Cache-Control: no-cache" );
			response->AddHeader( "Pragma: no-cache" );
			
			// Add content length header
			sprintf( headBuffer, "%s: %ld", kHEAD_LENGTH, contentLength );
			response->AddHeader( headBuffer );
			
			// Add mime type header
			response->AddHeader( kHEAD_TYPE, "text/html" );
			if( request->GetMethod() == METHOD_GET )
				response->SetMessageBody( &output );
			output.Seek( 0, SEEK_SET );
			response->SetStatusLine( statusCode );
			
			log_status_line( sn, response->GetStatusLine() );
			log_sending( sn, contentLength, absPath.Path() );
			
			bigtime_t 		startTime, deltaTime;
			startTime = system_time();
			
			size = request->SendReply( response );
			
			deltaTime = system_time() - startTime;
			int32 		bytesSeconds = (size*1000000)/deltaTime;
			log_sent( sn, size, deltaTime/1000, bytesSeconds );
		}
		else
		{
			// If "Range:" header, parse range
			ByteRangeSet	rangeSet;
			
			if( request->FindHeader( kHEAD_RANGE, fieldValue, fieldSize ) )
			{
				rangeSet.AddByteRange( fieldValue );
				statusCode = 206;
			}
			
			// Add content encoding header if gzFile
			if( gz )
				response->AddHeader( kHEAD_ENCODING, "gzip" );
			
			if( statusCode == 206 ) // If partial content
			{
				contentLength = rangeSet.ContentLength( entityLength );
				response->SetContentLength( contentLength );
				if( rangeSet.CountRanges() == 1 )
					response->AddHeader( kHEAD_CONTENT_RANGE, 
						rangeSet.ContentRangeString( headBuffer, 0, entityLength ) );
			}
			else // Set content-length to entity-length
				contentLength = entityLength;
		
		
			// Set content-length
			response->SetContentLength( contentLength );
			
			// Add content length header
			sprintf( headBuffer, "%s: %ld", kHEAD_LENGTH, contentLength );
			response->AddHeader( headBuffer );
			
			// Add mime type header
			response->AddHeader( kHEAD_TYPE, mimeType );
			
			// Set message body
			BFile		*bodyFile;
			
			if( request->GetMethod() == METHOD_GET )
			{
				if( gz )
				{
					response->SetMessageBody( &gzFile );
					bodyFile = &gzFile;
				}
				else
				{
					response->SetMessageBody( &theFile );
					bodyFile = &theFile;
				}
			}
			else // Don't send body on HEAD
				bodyFile = NULL;
			
			// Set status line
			response->SetStatusLine( statusCode );
			log_status_line( sn, response->GetStatusLine() );
			log_sending( sn, contentLength, absPath.Path() );
			
			bigtime_t 		startTime, deltaTime;
			startTime = system_time();
			if( rangeSet.CountRanges() <= 1 ) // if less <= one byte range
			{
				// If partial content, seek to offset
				if( (statusCode == 206)&&(bodyFile != NULL) )
				{
					FileOffsetSpec		offset;
					if( rangeSet.GetFileOffset( &offset, entityLength, 0 ) )
						bodyFile->Seek( offset.offset, SEEK_SET );
				}
				size = request->SendReply( response ); // Send standard reply
			}
			else // Send multipart MIME message
			{
				// Not implemented yet... send status 200 OK and full message instead
				statusCode = 200;
				response->SetStatusLine( statusCode );
				size = request->SendReply( response );
			}
			deltaTime = system_time() - startTime;
			int32 		bytesSeconds = (size*1000000)/deltaTime;
			log_sent( sn, size, deltaTime/1000, bytesSeconds );
		}
	}
	else
	{
		response->SetHTMLMessage( 404 ); // Not Found
		log_status_line( sn, response->GetStatusLine() );
		request->SendReply( response );
	}
	return true;
}