// Robin Hood Web Server - A web server for BeOS
// Copyright (C) 1999 Joe Kloss

// This program is free software; you can redistribute it and/or 
// modify it under the terms of the GNU General Public License 
// as published by the Free Software Foundation; either version 2 
// of the License, or (at your option) any later version. 

// This program is distributed in the hope that it will be useful, 
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
// GNU General Public License for more details. 

// You should have received a copy of the GNU General Public License 
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

// Contact Info:
// Author: Joe Kloss
// E-mail: axly@deltanet.com
// Postal Address: 25002 Ravenswood, Lake Forest, CA 92630, USA

#include <stdio.h>
#include <File.h>
#include <Path.h>
#include <FindDirectory.h>
#include <stdlib.h>
#include <string.h>
#include "RobinHoodServer.h"
#include "TCP_IO.h"
#include "StringUtils.h"
#include "DataIOUtils.h"
#include "HTTPListener.h"
#include "RHApplication.h"
#include "RHMessages.h"

struct ServerInfo
{
	HTTPListener 	*listener;
	RHServer 		*handler;
	FieldList 		*virtualHosts;
	thread_id		thread;
};

static const char *kDefaultHost = "Default-Host /boot/home/public_html/";
static const char *kDefaultHostFile = 
"# Robin Hood Web Server Virtual Hosts\n"
"# Changes to this file will take effect when the server is restarted.\n"
"# Format:\n"
"# 1*(server-Line 1*host-line)\n"
"# server-line = \"Server\" \":\" [*server-field] NL\n"
"# server-field = field-name \"=\" field-value\n"
"# host-line = \"Host\" \":\" VHost-Name SP VWeb-Root NL\n"
"# Example:\n"
"# Server: port=80 maxcon=32\n"
"#	Host: Default-Host /boot/home/public_html/default/\n"
"#	Host: www.my-domain.com /boot/home/public_html/my domain/\n"
"#	Host: www.foo-bar.org /boot/home/foo/bar/\n"
"#\n"
"# Server: port=225 maxcon=100\n"
"#	Host: Default-Host /boot/home/stuff/\n"
"\n"
"Server: port=80 maxcon=32\n"
"	Host: Default-Host /boot/home/public_html/\n";

RHApp::RHApp( void )
	: BApplication( "application/x-vnd.KS-RH" )
{
	
}

RHApp::~RHApp( void )
{
	
}

void RHApp::MessageReceived( BMessage *message )
{
	switch( message->what )
	{
		case MSG_LOG: // Forward message to all log servers
			{
				BMessenger		*msgr;
				
				for( int32 i=0; (msgr=(BMessenger *)logMsgrList.ItemAt(i)); i++ )
				{
					if( msgr->IsValid() )
						msgr->SendMessage( message );
					else // Remove from list if not valid
					{
						logMsgrList.RemoveItem( i );
						i--;
					}
				}
			}
			break;
			
		case MSG_ADD_LOG_SERVER:
			{
				BMessenger		*msgr = new BMessenger;
				if( message->FindMessenger( kMSG_LOG_MESSENGER, msgr ) != B_NO_ERROR )
					delete msgr;
				else
					logMsgrList.AddItem( msgr );
			}
			break;
			
		default:
			BApplication::MessageReceived( message );
			break;
	}
}


bool RHApp::QuitRequested( void )
{
	StopAllServers();
	return BApplication::QuitRequested();
}

void RHApp::ReadyToRun( void )
{
	StartServers();
}

status_t RHApp::StartServer( uint16 port, FieldList *virtualHosts, int32 maxCon )
{
	ServerInfo 		*info = new ServerInfo();
	info->virtualHosts = virtualHosts;
	info->handler = new RHServer( info->virtualHosts );
	info->listener = new HTTPListener( info->handler, port, maxCon );
	if( (info->thread = info->listener->Run()) < 0 )
	{
		delete info->listener;
		delete info->handler;
		delete info;
		return B_ERROR;
	}
	else
	{
		serverList.AddItem( info );
		return B_OK;
	}
}

status_t RHApp::StopServer( int32 index, bool now )
{
	ServerInfo 		*info;
	
	if( !(info = ServerAt( index )) )
		return B_ERROR;
	info->listener->Quit();
	delete info->listener;
	delete info->handler;
	delete info->virtualHosts;
	delete info;
	serverList.RemoveItem( info );
	return B_OK;
}

status_t RHApp::StopAllServers( void )
{
	while( StopServer( 0 ) == B_OK ) {  }
	return B_OK;
}

int32 RHApp::CountServers( void )
{
	return serverList.CountItems();
}

ServerInfo *RHApp::ServerAt( int32 index )
{
	return (ServerInfo *)serverList.ItemAt( index );
}

void RHApp::StartServers( void )
{
	BPath				path;
	FieldList			*virtualHosts;
	virtualHosts = new FieldList();
	
	// Load Virtual Hosts Data
	if( find_directory( B_USER_SETTINGS_DIRECTORY, &path) != B_NO_ERROR )
		virtualHosts->AddField( kDefaultHost );
	else
	{
		BFile vHostsFile;
		path.Append( "RHVirtualHosts" );
		
		if( vHostsFile.SetTo( path.Path(), B_READ_ONLY ) != B_NO_ERROR )
		{
			if( vHostsFile.SetTo( path.Path(), B_READ_WRITE | B_CREATE_FILE ) == B_NO_ERROR )
			{
				io_printf( &vHostsFile, "%s", kDefaultHostFile );
			}
			virtualHosts->AddField( kDefaultHost );
			StartServer( 80, virtualHosts, 32 );
		}
		else
		{
			char		line[4096];
			char		lineType[32];
			char		lineText[1024];
			
			char		field[64];
			char		fieldName[32];
			char		fieldValueQ[32];
			char		fieldValue[32];
			
			int32		servers = 0;
			uint16		port = 80;
			int32		maxConn = 32;
			const char	*sPtr, *sPtr2;
			
			// Read lines until eof
			while( io_getline( &vHostsFile, line, 4096 ) )
			{
				// If not a comment or an empty line, interpret the line
				if( (*line != '#')&&(*line != 0) )
				{
					// break line into lineType and lineText
					sPtr = line;
					sPtr = get_next_token( lineType, sPtr, 32, ":" );
					sPtr = get_next_token( lineText, sPtr, 1024, ":" );
					
					// LineType: Server
					if( strcasecmp( "Server", lineType ) == 0 )
					{
						// Start the previous server if there is one
						if( servers++ > 0 )
						{
							StartServer( port, virtualHosts, maxConn );
							virtualHosts = new FieldList();
						}
						
						port = 80;
						maxConn = 32;
						sPtr = lineText;
						
						// Break lineText into fields until \0
						while( *sPtr && (sPtr = get_next_token( field, sPtr, 64, " ", "\"" )) )
						{
							// break field into fieldName and fieldValue
							sPtr2 = field;
							sPtr2 = get_next_token( fieldName, sPtr2, 32, "=" );
							sPtr2 = get_next_token( fieldValueQ, sPtr2, 32, "=" );
							// remove any quotes from fieldValue
							get_next_token( fieldValue, fieldValueQ, 32, "\"" );
							
							if( strcasecmp( "port", fieldName ) == 0 )
								port = strtol ( fieldValue, NULL, 10 );
							else if( strcasecmp( "maxcon", fieldName ) == 0 )
								maxConn = strtol ( fieldValue, NULL, 10 );
						}
					} // End if( strcasecmp( "Server", lineType ) == 0 )
					else if( strcasecmp( "Host", lineType ) == 0 )
					{
						virtualHosts->AddField( lineText );
					}
				} // End if( (*hostLine != '#')&&(*hostLine != '\n')&&(*hostLine != '\r') )
			} // End while( io_getline( &vHostsFile, line, 4096 ) )
			
			// Start the last server
			StartServer( port, virtualHosts, maxConn );
		} // End else
	} // End else
}

