/*
 * DiTerm Copyright (c)1997-98 by Dianne Hackborn.
 * All rights reserved.
 *
 * Based (increasingly loosly) on WebTerm, a Web-based terminal
 * applet written in Java, which is
 * Copyright (C)1996 by the National Alliance for Computational
 * Science and Engineering (NACSE).
 * See <URL:http://www.nacse.org/> for more information.
 *
 * This code is not public domain, nor freely distributable.
 * Please direct any questions or requests to Dianne Hackborn,
 * at <hackbod@lucent.com> or <hackbod@angryredplanet.com>.
 *
 * ----------------------------------------------------------------------
 *
 * ArpVT200.h
 *
 * VT200 emulator class: a subclass of ArpEmulator that implements
 * standard XTerm, VT200, VT100, and VT52 emulations.
 *
 * ----------------------------------------------------------------------
 *
 * Known Bugs
 * ~~~~~~~~~~
 *
 *	• Does not archive its state.
 *
 * ----------------------------------------------------------------------
 *
 * To Do
 * ~~~~~
 *
 * • Create an abstract ArpANSIEmulator class from this, which
 *   can be subclassed to implement particular ANSI-style
 *   terminal emulators.
 * • Move Command into the ArpVT200 class, so that it's not
 *   sitting out in the global name space.
 *
 * ----------------------------------------------------------------------
 *
 * History
 * ~~~~~~~
 *
 * 7/29/97:
 *	• Correct cursor keys were not being sent in VT52 mode.
 *	• Cursor position report was one row and column too short.
 *	• Cursor movement sequences should have been transforming
 *	  0-position movements into 1 character position.
 *	• The ANSI terminator character (ST) was not being processed,
 *	  due to some signed character comparisons that should have
 *	  been unsigned.  This caused the X-term sequence changing
 *	  the window title to eat all incoming characters until the
 *	  terminal was reset.
 *
 * 7/26/97:
 *	• IND and RI did not work correctly when terminal's region
 *	  was smaller than terminal size.
 *
 * 7/25/97:
 *	• Implemented insert mode (IRM) and made RETURN key send
 *	  out correct value (CR instead of LF).
 *	• Now ignores mouse events when SHIFT is held down, so they
 *	  can be used for highlighting.
 *
 * 0.1: Created this file from WebTerm's VT200.java.
 *
 */

/* VT Emulation info:

	 One cost-effective document is "VT220 Programmer Pocket Guide", part
	 number EK-VT220-HR-001, $21.00 (US). From the 48-state United States,
	 call 1-800/DIGITAL to order this. From elsewhere, dial +1 603/884-6660.
*/

#pragma once

#ifndef ARPVT200_H
#define ARPVT200_H

#ifndef ARPCOMMON_ARPEMULATOR_H
#include <ArpCommon/ArpEmulator.h>
#endif

/* This class encapsulates all of the information about a parsed
   command, and some basic methods to help the parser.  See the
   ArpVT200 class for what these values mean and details on
   the format of commands. */

class Command {

public:
	// Types of commands.
	enum Type {
		TYPE_NONE = 0,    // Non-command.
		TYPE_ESC =  1,    // ESC-term
		TYPE_CSI =  2,    // ESC-[-params-inter-term
		TYPE_DCS =  3,    // ESC-P-params-inter-term-data
		TYPE_OSC =  4,    // ESC-]-params-inter-term-data
		TYPE_PM =   5,    // ESC-^-params-inter-term-data
		TYPE_APC =  6,    // ESC-_-params-inter-term-data
		
		// This one is used for constructing commands that don't follow
		// a standard ASCII command structure.
		TYPE_RAW = 7,     // Just use raw data string.
	};
	
	// Flags encountered during parameter parsing.
	// These are bit values.
	enum Flag {
		FLAG_LT = 0,    // '<' encountered in params.
		FLAG_LT_MASK = (1<<FLAG_LT),
		FLAG_EQ = 1,    // '=' encountered in params.
		FLAG_EQ_MASK = (1<<FLAG_EQ),
		FLAG_GT = 2,    // '>' encountered in params.
		FLAG_GT_MASK = (1<<FLAG_GT),
		FLAG_QUES = 3,  // '?' encountered in params.
		FLAG_QUES_MASK = (1<<FLAG_QUES),
	};
	
	// Other constants.
	enum {
		// The maximum number of parameters we can handle at one time.
		MAX_PARAMS = 64,

		// The maximum number of intermediate characters we can handle at one time.
		MAX_INTERS = 16,
	};
	
	Command(void) : cur_param(0), data("") { init(); }

	// Reset the command.
	void init(void);

	// Set the command type, e.g. TYPE_CSI.
	bool setType(int type);

	/* This is the programmic interface to the command: it allows commands
	   to be constructed in their logical units. */

	// Directly add a paramter to the command.
	bool addParam(int param,int flags);

	// Skip [set to default] next parameter in command.
	bool skipParam(void);

	// Directly add intermediate characters to the command.
	bool addIntermediates(const ArpString& inter);

	// Directly add string data to the command.
	bool addData(const ArpString& data);
	bool addData(ichar data);

	// Directly set the terminating character
	bool setTerminator(ichar t);

	/* This is the parser interface to the command: it allows commands
	   to be constructed one character at a time, as they occur in the
	   command sequence. */

	// Parse the next character in a parameter stream or flag character.
	bool parseParamChar(ichar c);

	// Add the next intermediate character.
	bool parseInterChar(ichar c);

	// Add the next data character.
	bool parseDataChar(ichar c);

	// Set the terminating character.
	bool parseTermChar(ichar c);

	// Conversions.
	const ArpString paramsToString(void);
	const ArpString toSequence(void);
	const ArpString toString(void);

	int type;                     // One of ArpVT200 emu's state codes.

	int cur_param;                // Current parameters
	int params[MAX_PARAMS];       // Parsed parameter values
	bool has_params[MAX_PARAMS];  // Was a value supplied?
	int flags;                    // Param flags: <, =, >, ?

	int cur_inter;                // Current intermediate character
	ichar inters[MAX_INTERS];     // Intermediate characters supplied

	ArpString data;               // Data string

	ichar term;                   // Terminating character
};

class ArpVT200 : public ArpEmulator {
private:
	typedef ArpEmulator inherited;

	/* ------------------------------------------------------------
	   CONSTRUCTOR AND EMULATOR METHODS WE OVERRIDE.
	   ------------------------------------------------------------ */

public:

	/* Types of emulations available.  These coorespond to indices
	   of the ArpTerminalType with the preferred name. */
	enum EmulationType {
		EMU_XTERM = 0,
		EMU_VT220 = 1,
		EMU_VT100 = 3,
		EMU_VT52 = 5,
		
		EMU_NUM = 7,
	};

	ArpVT200(ArpTerminalInterface& myterm, int32 type=EMU_XTERM);

	static ArpEmulatorAddon AddonInterface;
	
	virtual	status_t Archive(BMessage *into, bool deep = true) const;
	virtual void AttachTerminal(ArpTerminalInterface& terminal);
	
	virtual void Reset(bool hard);
	
	virtual const ArpEmulationType* EmulationType(void) const;
	
protected:

	/* ------------------------------------------------------------
	   CONSTANT VALUES SECTION
	   ------------------------------------------------------------ */

	/* Basic types of characters:
		 
	   A "control character" is of the range 0x00-0x1f or 0x80-0x9f;
	   the DEC control characters are defined below.

	   An "intermediate character" is in the range 0x20-0x2f; they
	   may occur multiple times before the last character of an
	   escape sequence or CSI command.

	   A "parameter character" is in the range 0x30-0x39 or 0x3b; it
	   appears as "23;7;45".

	   A "flag character" is in the range 0x3c-0x3f; it sets a
	   boolean flag if it appears anywhere.

	   A "final character" is in the range 0x40-0x7e; it terminates
	   an escape sequence or CSI command.
	 */

	// State of character parser.
	enum ParserState {
	
		/* In the normal state, non-control characters is simply
		   rendered to the screen. */
		STATE_ASCII = Command::TYPE_NONE, // Rendering characters

		/* An escape sequence has the following form:
		   1. ESC
		   2. Zero or more intermediate characters
		   3. Final character */
		STATE_ESC = Command::TYPE_ESC,    // In escape sequence

		/* A control sequence has the following form:
		   1. CSI (or ESC-[)
		   2. Zero or more parameter characters
		   3. Zero or more intermediate characters
		   4. Final character */
		STATE_CSI = Command::TYPE_CSI,    // In control seqeuence

		/* A device control sequence has the following form:
		   1. DCS (or ESC-P)
		   2. Zero or more parameter characters
		   3. Zero or more intermediate characters
		   4. Final character
		   5. Data string
		   6. ST (or ESC-\) */
		STATE_DCS = Command::TYPE_DCS,  // In device control sequence

		/* An operating system command has the following form:
		   1. OSC (or ESC-])
		   2. Data string
		   3. ST (or ESC-\) */
		STATE_OSC = Command::TYPE_OSC, // In operating system command

		/* A privacy message has the following form:
		   1. PM (or ESC-^)
		   2. Data string
		   3. ST (or ESC-\) */
		STATE_PM = Command::TYPE_PM,      // In privacy message

		/* An application program command has the following form:
		   1. APC (or ESC-_)
		   2. Data string
		   3. ST (or ESC-\) */
		STATE_APC = Command::TYPE_APC,    // In app. program command

		/* The state when processing a command's final data string */
		STATE_DATA = STATE_APC+1,        // Ends with ST.

		/* The first state in a VT52 direct cursor address; the next
		   character received is the line, with ' ' being the top line. */
		STATE_DCA1 = STATE_APC+2,

		/* The second state in a VT52 direct cursor address; the next
		   character received is the column, with ' ' being the far left. */
		STATE_DCA2 = STATE_APC+3,
	};
	
	/* Operating modes.  We assign an ID to each mode we understand,
	   then create arrays to help us map these IDs to particular command
	   sequences and values. */

	enum OperatingMode {
		MODE_KAM = 0,      //  2  Keyboard locked / unlocked
		MODE_IRM = 1,      //  4  Character insert / replace mode
		MODE_SRM = 2,      //  12 Send-receive on / off
		MODE_LNM = 3,      //  20 LF is new-line / linefeed
		MODE_DECCKM = 4,   // ?1  Cursor mode: application / cursor
		MODE_DECANM = 5,   // ?2  Emulation: (NA) / VT52
		MODE_DECCOLM = 6,  // ?3  132 column / 80 column
		MODE_DECSCLM = 7,  // ?4  Scrolling smooth / jump
		MODE_DECSCNM = 8,  // ?5  Screen reverse / normal
		MODE_DECOM = 9,    // ?6  Origin mode
		MODE_DECAWM = 10,  // ?7  Autowrap on / off
		MODE_DECARM = 11,  // ?8  Autorepeat on / off
		MODE_MITMRM = 12,  // ?9  Send MIT mouse rep on / off
		MODE_DECPFF = 13,  // ?18 Print form feed on / off
		MODE_DECPEX = 14,  // ?19 Print extent screen / region
		MODE_DECTCEM = 15, // ?25 Text cursor on / off
		MODE_DECTEK = 16,  // ?38 Enter Tektronix mode
		MODE_XTMACOL = 17, // ?40 Allow 80<->132 switch on / off
		MODE_XTMCFIX = 18, // ?41 Curses(5) fix on / off
		MODE_DECNRCM = 19, // ?42 Char set national / multinational
		MODE_XTMMBM = 20,  // ?44 Margin bell on / off
		MODE_XTMRWM = 21,  // ?45 Reverse-wrap on / off
		MODE_XTMLOGM = 22, // ?46 Logging on / off
		MODE_XTMASM = 23,  // ?47 Alternate / normal screen buffer
		MODE_XTMMRM = 24,  // ?1000 VT200 mouse report on / off
		MODE_XTMHMRM = 25, // ?1001 VT200 hilight mouse rep on / off
		// The rest are not actually mapped to CSI h and CSI l values.
		MODE_DECKPAM = 26, //     Keypad application / numeric
		MODE_DECELR = 27,  //     Enable DEC locator reports
		MODE_DECOLR = 28,  //     This is a one-shot locator report
		MODE_DECPLR = 29,  //     Locator reports in pixels / chars
		_NUM_MODE = 30,
	};
	
	// Flag status of each mode's command sequence.
	static const int mode_flags[];

	// Parameter value of each mode's command sequence.
	static const int mode_values[];

	// Initial values of modes.
	static const bool mode_inits[];

	// Names of the modes
	static const char* mode_names[];

	// Abbreviations of mode names
	static const char* mode_abbrvs[];

	// Mode operations that can be performed
	enum ModeOperation {
		MOP_RESET = 0,    // CSI Pn l
		MOP_SET = 1,      // CSI Pn h
		MOP_SAVE = 2,     // CSI Pn r
		MOP_RESTORE = 3,  // CSI Pn s
	};
	
	// Character codes that need to be directly handled by emulator.
	static const ichar special_chars[];

	/* ------------------------------------------------------------
	   EMULATION ENTRY POINTS.
	   ------------------------------------------------------------ */

	bool EmuTTYKeyPress(BMessage* msg, int32 key, int32 mods);
	bool EmuTTYMouseDown(BMessage* msg, BPoint pos, int32 buttons);
	bool EmuTTYMouseUp(BMessage* msg, BPoint pos, int32 buttons);
	bool EmuTTYMouseGen(BMessage* msg, BPoint pos, int32 buttons);
	
	// Need our own special version to deal with insert mode.
	virtual void EmulateToTTY(const ichar* d, size_t len);
	void EmulateToTTY(const ArpString& str)
		{ EmulateToTTY((const ichar*)str,str.Length()); }

	bool EmuRemoteChar(ichar c);
	virtual const ichar* EmuRemoteNextSpecial(const ichar* str,
												size_t len);
	
	/* ------------------------------------------------------------
	   INITIALIZATION AND GLOBAL CONTROL.
	   ------------------------------------------------------------ */

	// Set modes to initial states.
	void init_modes(void);

	// Set tabs to defaults -- every eight characters.
	void init_tabs(void);

	// Reset the state machine back to displaying ASCII characters.
	void reset_state(void);

	// Reflect the emulation mode back into the Terminal's settings.
	void update_terminal_mode(void);

	// Handling of complex (non-character) types of events.
	void build_lr(Command& cmd, BMessage* msg);

	// Do a Reset() of the emulation, with the option to not
	// initialize the subclass/terminal.
	void do_reset(bool hard, bool subclass);
	
	/* ------------------------------------------------------------
	   ESC PARSING
	   ------------------------------------------------------------ */

	void doESC(Command& mycmd);

	/* ------------------------------------------------------------
	   CSI PARSING
	   ------------------------------------------------------------ */

	void doCSI(Command& mycmd);

	/* ------------------------------------------------------------
	   DSR PARSING.
	   ------------------------------------------------------------ */

	bool doDSR(Command& mycmd);

	/* ------------------------------------------------------------
	   DCS PARSING.
	   ------------------------------------------------------------ */

	void doDCS(Command& mycmd);

	/* ------------------------------------------------------------
	   OSC PARSING.
	   ------------------------------------------------------------ */

	void doOSC(Command mycmd);

	/* ------------------------------------------------------------
	   PM PARSING.
	   ------------------------------------------------------------ */

	void doPM(Command& mycmd);

	/* ------------------------------------------------------------
	   APC PARSING.
	   ------------------------------------------------------------ */

	void doAPC(Command& mycmd);

	/* ------------------------------------------------------------
	   EMULATION SUPPORT.
	   ------------------------------------------------------------ */

	/* Send a Device Attribute string, of the given type, to the remote
	   host. */
	enum DeviceAttribute {
		DA_PRIMARY = 0,
		DA_SECONDARY = 1,
		DA_VT52 = 2,
	};

	bool sendDA(DeviceAttribute type);

	/* Do the given operation 'op' on the mode state 'mode'. */
	void doMode(OperatingMode mode, ModeOperation op);

	// Small convenience functions.
	void SendToRemote(const ichar* str, size_t len)
		{ terminal.TermSendRemote(str,len); }
	void SendToRemote(ichar c)
		{ terminal.TermSendRemote(&c,1); }
	void SendToRemote(const ArpString& str)
		{ terminal.TermSendRemote((const ichar*)str,str.Length()); }
	
	/* ------------------------------------------------------------
	   INSTANCE DATA SECTION
	   ------------------------------------------------------------ */

	// The current parsing state.
	ParserState state;

	// Current command being parsed.
	Command cmd;

	/* Sub-command being executed (e.g., IND within CSI).
	   We need this because we transform 8-bit codes to their
	   ESC-Fe counterpart, and call doESC to execute them. */
	Command sub_cmd ;

	/* 'true' for characters we need to directly handle.
	   (this is constructed from the 'special_chars' array.) */
	enum { _NUM_CHAR_FLAGS = 256 };
	static bool char_flags[_NUM_CHAR_FLAGS];

	// Current state of modes.
	bool mode_states[_NUM_MODE];

	// Saved state of modes.
	bool mode_saves[_NUM_MODE];

	// Current emulation type
	int emulation;

	// Current columns where there are tabstops -- true is a tabstop.
	enum { _NUM_TABS = 512 };
	bool tabs[_NUM_TABS];  // lots of room!

	// The line value processed in a VT52 direct cursor address.
	int dca_line;

	// ** NOTE: We also need to save the rendering mode and some flags.
	int32 saved_row;
	int32 saved_col;
	int saved_style;

private:
	// Don't allow default copy constructor and assignment
	//ArpVT200(const ArpVT200&) { }
	ArpVT200& operator=(const ArpVT200&) { return *this; }
};

#endif
