/*
 * Copyright (c)1997-98 by Dianne Hackborn.
 * All rights reserved.
 *
 * Based (increasingly loosly) on WebTerm, a Web-based terminal
 * applet written in Java, which is
 * Copyright (C)1996 by the National Alliance for Computational
 * Science and Engineering (NACSE).
 * See <URL:http://www.nacse.org/> for more information.
 *
 * This code is not public domain, nor freely distributable.
 * Please direct any questions or requests to Dianne Hackborn,
 * at <hackbod@lucent.com> or <hackbod@angryredplanet.com>.
 *
 */

//#define DEBUG 1

#ifndef ARPVT200_H
#include "ArpVT200.h"
#endif

#ifndef ARPCOMMON_ARPDEBUG_H
#include <ArpCommon/ArpDebug.h>
#endif

#ifndef _MESSAGE_H
#include <app/Message.h>
#endif

#ifndef _INTERFACE_DEFS_H
#include <interface/InterfaceDefs.h>
#endif

#ifndef _POINT_H
#include <interface/Point.h>
#endif

#include <string.h>

/* ----------------------------------------------------------
	"Command" class implementation.
   ---------------------------------------------------------- */
   
#if defined(DEBUG) && defined(DEBUG_COMMAND)
#define DBCMD(level,x) { if(level) { x; } cdb.flush(); DB_WAIT(DB_DELAY) }
#else
#define DBCMD(level,x)
#endif

void Command::init(void)
{
	DBCMD(DBALL, cdb << "Command::init()" << endl);
	type = 0;
	for(int i=0; i<=cur_param && i <MAX_PARAMS; i++ ) {
		params[i] = 0;
		has_params[i] = false;
	}
	cur_param = 0;
	flags = 0;
	cur_inter = 0;
	data.SetLength(0);
	term = 0;
}

bool Command::setType(int type)
{
	DBCMD(DBALL, cdb << "Command::setType(" << type << ")" << endl);
	this->type = type;
	return true;
}

bool Command::addParam(int param,int flags)
{
	DBCMD(DBALL, cdb << "Command::addParam(" << param
			<< ", " << flags << ")" << endl);
	if( cur_param < MAX_PARAMS ) {
		params[cur_param] = param;
		has_params[cur_param] = true;
		this->flags |= flags;
		cur_param++;
		return true;
	}
	return false;
}

bool Command::skipParam(void)
{
	DBCMD(DBALL, cdb << "skipParam()" << endl);
	if( cur_param < MAX_PARAMS ) {
		params[cur_param] = 0;
		has_params[cur_param] = false;
		cur_param++;
		return true;
	}
	return false;
}

bool Command::addIntermediates(const ArpString& inter)
{
	DBCMD(DBALL, cdb << "ohhhhh noooooo...!!!" << endl);
	DBCMD(DBALL, cdb << "addIntermediates(" << inter << ")" << endl);
	int i=0;
	while( i < inter.Length() && cur_inter < MAX_INTERS ) {
		parseInterChar(inter[i]);
		i++;
	}
	if( i >= inter.Length() ) return true;
	return false;
}

bool Command::addData(const ArpString& data)
{
	DBCMD(DBALL, cdb << "addData(" << data << ")" << endl);
	this->data += data;
	return true;
}

bool Command::addData(ichar data)
{
	DBCMD(DBALL, cdb << "addData(" << data << ")" << endl);
	this->data += data;
	return true;
}

bool Command::setTerminator(ichar t)
{
	DBCMD(DBALL, cdb << "setTerminator(" << t << ")" << endl);
	term = t;
	return true;
}

bool Command::parseParamChar(ichar c)
{
	DBCMD(DBALL,cdb << "ArpVT200: Parsing seq char " << c << endl);
	if( c >= '0' && c <= '9' ) {
		if( cur_param < MAX_PARAMS ) {
			params[cur_param] *= 10;
			params[cur_param] += (int)(c-'0');
			has_params[cur_param] = true;
			DB(DBALL,cdb << "ArpVT200: Param #" << cur_param <<
							 " now " << params[cur_param] << endl);
		}
		return true;
	} else if( c == ';' || c == ':' ) {
		cur_param++;
		if( cur_param > MAX_PARAMS ) cur_param = MAX_PARAMS;
		if( cur_param < MAX_PARAMS ) has_params[cur_param] = true;
		DBCMD(DBALL,cdb << "ArpVT200: Switch to param #"
					<< cur_param << endl);
		return true;
	} else if( c >= '<' && c <= '?' ) {
		flags |= 1<<((int)(c-'<'));
		return true;
	}
	return false;
}

bool Command::parseInterChar(ichar c)
{
	if( cur_inter < MAX_INTERS ) {
		inters[cur_inter] = c;
		cur_inter++;
	}
	return true;
}

bool Command::parseDataChar(ichar c)
{
	data += c;
	return true;
}

bool Command::parseTermChar(ichar c)
{
	DBCMD(DBALL,cdb << "ArpVT200: Last param: " << cur_param << endl);
	if( cur_param >= MAX_PARAMS ) cur_param = MAX_PARAMS-1;
	if( cur_param < MAX_PARAMS ) {
		if( has_params[cur_param] ) cur_param++;
	}
	term = c;
	return true;
}

const ArpString Command::paramsToString(void)
{
	int i;
	ArpString str;
	str = "Params:";
	for(i=0; i<cur_param; i++) {
		if( has_params[i] ) {
			str += " ";
			str += (long)params[i];
		} else {
			str += " <DEF>";
		}
	}
	return str;
}

const ArpString Command::toSequence(void)
{
	ArpString str;
	switch( type ) {
	case TYPE_ESC:
		str += "\033";
		break;
	case TYPE_CSI:
		str += "\033[";
		break;
	case TYPE_DCS:
		str += "\033P";
		break;
	case TYPE_OSC:
		str += "\033]";
		break;
	case TYPE_PM:
		str += "\033^";
		break;
	case TYPE_APC:
		str += "\033_";
		break;
	case TYPE_RAW:
		return data;
	}
	
	if( (flags&FLAG_LT_MASK) != 0 ) str += (ichar)'<';
	if( (flags&FLAG_EQ_MASK) != 0 ) str += (ichar)'=';
	if( (flags&FLAG_GT_MASK) != 0 ) str += (ichar)'>';
	if( (flags&FLAG_QUES_MASK) != 0 ) str += (ichar)'?';

	for( int i=0; i<cur_param; i++ ) {
		if( has_params[i] ) {
			str += (long)params[i];
		}
		if( i < (cur_param-1) ) str += (ichar)';';
	}

	if( cur_inter > 0 ) {
		str.Append(inters,0,cur_inter);
	}

	str += term;

	if( data.Length() > 0 ) {
		str += data;
		str += "\033\\";
	}

	return str;
}

const ArpString Command::toString(void)
{
	return ArpEmulator::sequenceToString(toSequence());
}

/* ----------------------------------------------------------
	"ArpVT200" class tables and other static data.
   ---------------------------------------------------------- */
   
// Flag status of each mode's command sequence.
const int ArpVT200::mode_flags[_NUM_MODE] = {
	0, 0, 0, 0,                                       // KAM, IRM, SRM, LNM
	Command::FLAG_QUES_MASK, Command::FLAG_QUES_MASK, // DECCKM, DECANM
	Command::FLAG_QUES_MASK, Command::FLAG_QUES_MASK, // DECCOLM, DECSCLM
	Command::FLAG_QUES_MASK, Command::FLAG_QUES_MASK, // DECSCNM, DECOM
	Command::FLAG_QUES_MASK, Command::FLAG_QUES_MASK, // DECAWM, DECARM
	Command::FLAG_QUES_MASK, Command::FLAG_QUES_MASK, // MITMRM, DECPFF
	Command::FLAG_QUES_MASK, Command::FLAG_QUES_MASK, // DECPEX, DECTCEM
	Command::FLAG_QUES_MASK, Command::FLAG_QUES_MASK, // DECTEX, XTMACOL
	Command::FLAG_QUES_MASK, Command::FLAG_QUES_MASK, // XTMCFIX, DECNRCM
	Command::FLAG_QUES_MASK, Command::FLAG_QUES_MASK, // XTMMBM, XTMRWM
	Command::FLAG_QUES_MASK, Command::FLAG_QUES_MASK, // XTMLOGM, XTMASM
	Command::FLAG_QUES_MASK, Command::FLAG_QUES_MASK, // XTMMRM, XTMHMRM
	0, 0,                                             // DECKPAM, DECELR
	0, 0                                              // DECOLR, DECPLR
};

// Parameter value of each mode's command sequence.
const int ArpVT200::mode_values[_NUM_MODE] = {
	 2,  4, 12, 20,    // KAM, IRM, SRM, LNM
	 1,  2,  3,  4,    // DECCKM, DECANM, DECCOLM, DECSCLM
	 5,  6,  7,  8,    // DECSCNM, DECOM, DECAWM, DECARM
	 9, 18, 19, 25,    // MITMRM, DECPFF, DECPEX, DECTCEM
	38, 40, 41, 42,    // DECTEX, XTMACOL, XTMCFIX, DECNRCM
	44, 45, 46, 47,    // XTMMBM, XTMRWM, XTMLOGM, XTMASM,
	1000, 1001,        // XTMMRM, XTMHMRM
	 // These are not standard mode syntax.
	 2,  2,  2,  2     // DECKPAM, DECELR, DECOLR, DECPLR
};

// Initial values of modes.
const bool ArpVT200::mode_inits[_NUM_MODE] = {
	false, false, false, false, // KAM, IRM, SRM, LNM
	false, true,  false, false, // DECCKM, DECANM, DECCOLM, DECSCLM
	false, false, true,  true,  // DECSCNM, DECOM, DECAWM, DECARM
	false, false, true,  true,  // MITMRM, DECPFF, DECPEX, DECTCEM
	false, false, false, true,  // DECTEX, XTMACOL, XTMCFIX, DECNRCM
	false, false, false, false, // XTMMBM, XTMRWM, XTMLOGM, XTMASM,
	false, false,               // XTMMRM, XTMHMRM
	false, false, false, false  // DECKPAM, DECELR, DECOLR, DECPLR
};

// Names of the modes
const char* ArpVT200::mode_names[_NUM_MODE] = {
	"Keyboard action", "Insertion", "Send-receive",
	"Linefeed", "Cursor key", "ANSI/VT52", "Column",
	"Scrolling", "Screen", "Origin", "Wraparound",
	"Auto-repeat", "Send MIT mouse", "Print formfeed",
	"Print extent", "Text cursor", "Tektronix",
	"Allow column switch", "Curses fix", "Character set",
	"Margin bell", "Reverse-wraparound", "Logging",
	"Alternate screen", "VT200 mouse report",
	"VT200 hilight mouse report",
	"Keypad", "Enable locator reports", "One-shot locator report",
	"Pixel-unit locator reports"
};

// Abbreviations of mode names
const char* ArpVT200::mode_abbrvs[_NUM_MODE] = {
	"KAM", "IRM", "SRM", "LNM",
	"DECCKM", "DECANM", "DECCOLM", "DECSCLM",
	"DECSCNM", "DECOM", "DECAWM", "DECARM",
	"MITMRM", "DECPFF", "DECPEX", "DECTCEM",
	"DECTEX", "XTMACOL", "XTMCFIX", "DECNRCM",
	"XTMMBM", "XTMRWM", "XTMLOGM", "XTMASM",
	"XTMMWM", "XTMHMRM",
	"DECKPAM", "DECELR", "DECOLR", "DECPLR"
};

// Character codes that need to be directly handled by emulator.
const int _NUM_SPECIAL_CHARS = 26;
const ichar ArpVT200::special_chars[_NUM_SPECIAL_CHARS] = {
	ANSI_ENQ, ANSI_BS, ANSI_HT, ANSI_LF, ANSI_VT, ANSI_FF,
	ANSI_SO, ANSI_SI, ANSI_DC1, ANSI_DC3, ANSI_CAN, ANSI_SUB,
	ANSI_ESC,
	ANSI_DEL,
	ANSI_IND, ANSI_NEL, ANSI_HTS, ANSI_RI, ANSI_SS2, ANSI_SS3,
	ANSI_DCS, ANSI_CSI, ANSI_ST, ANSI_OSC, ANSI_PM, ANSI_APC };

bool ArpVT200::char_flags[_NUM_CHAR_FLAGS] = { false };

/* ----------------------------------------------------------
	"ArpVT200" class add-on interface.
   ---------------------------------------------------------- */
   
static const ArpEmulationType type_info[] = {
	{ sizeof(ArpEmulationType), "XTERM", "X Terminal", NULL },
	
	{ sizeof(ArpEmulationType), "DEC-VT220", "DEC VT220", NULL },
	{ sizeof(ArpEmulationType), "VT220", "DEC VT220", "DEC-VT220" },
	
	{ sizeof(ArpEmulationType), "DEC-VT100", "DEC VT100", NULL },
	{ sizeof(ArpEmulationType), "VT100", "DEC VT100", "DEC-VT100" },
	
	{ sizeof(ArpEmulationType), "DEC-VT52", "DEC VT52", NULL },
	{ sizeof(ArpEmulationType), "VT52", "DEC VT52", "DEC-VT52" },
};

static int32 DoCountEmulations(void)
{
	return ArpVT200::EMU_NUM;
}

static const ArpEmulationType* DoEmulationType(int32 idx)
{
	if( idx < 0 ) idx = 0;
	if( idx >= ArpVT200::EMU_NUM ) idx = ArpVT200::EMU_NUM-1;
	return &type_info[idx];
}

static ArpEmulatorInterface* DoAllocEmulator(const char* name,
											ArpTerminalInterface& terminal)
{
	int emulation=-1;
	if( !strcmp(name,type_info[ArpVT200::EMU_XTERM].Name) ) {
		emulation = ArpVT200::EMU_XTERM;
	} else if( !strcmp(name,type_info[ArpVT200::EMU_VT220].Name)
				|| !strcmp(name,type_info[ArpVT200::EMU_VT220+1].Name) ) {
		emulation = ArpVT200::EMU_VT220;
	} else if( !strcmp(name,type_info[ArpVT200::EMU_VT100].Name)
				|| !strcmp(name,type_info[ArpVT200::EMU_VT100+1].Name) ) {
		emulation = ArpVT200::EMU_VT100;
	} else if( !strcmp(name,type_info[ArpVT200::EMU_VT52].Name)
				|| !strcmp(name,type_info[ArpVT200::EMU_VT52+1].Name) ) {
		emulation = ArpVT200::EMU_VT52;
	} else return NULL;
	return new ArpVT200(terminal,emulation);
}

ArpEmulatorAddon ArpVT200::AddonInterface = {
	ArpEmulator_Current,
	
	"VT200/XTerm Emulator",
	"Angry Red Planet",
	B_UTF8_COPYRIGHT "1997-98 Dianne Hackborn",
	__DATE__, __TIME__,
	PROGRAM_VERSION_CODE,
	
	"This emulator implements the basic protocol used by "
	"DEC VT200, VT100, VT52 emulators.  In addition, it "
	"implements a fairly complete X Terminal emulatiom, "
	"which is very similar to the VT200 and VT100.  "
	"All of these terminals (except the VT52) are based "
	"on the standard ANSI terminal control sequences.",
	
	&DoCountEmulations,
	&DoEmulationType,
	&DoAllocEmulator,
};

/* ----------------------------------------------------------
	"ArpVT200" class constructor and high-level interface.
   ---------------------------------------------------------- */

ArpVT200::ArpVT200(ArpTerminalInterface& myterm, int32 type)
		: ArpEmulator(myterm), emulation(type),
		  state(STATE_ASCII), dca_line(0),
		  saved_row(0), saved_col(0),
		  saved_style(ArpTerminalInterface::TERM_STYLEPLAIN)
{
	// Flag the ASCII characters we need to directly deal with.
	//for( int i=0; i<_NUM_CHAR_FLAGS; i++ ) {
	//	char_flags[i] = false;
	//}
	for( int i=0; i<_NUM_SPECIAL_CHARS; i++ ) {
		char_flags[(int)special_chars[i]] = true;
	}

	do_reset(true,false);
}

status_t ArpVT200::Archive(BMessage* into, bool deep) const
{
	status_t status = inherited::Archive(into,deep);
	if( status == B_NO_ERROR )
		status = into->AddString("class","ArpVT200");
	return status;
}

void ArpVT200::AttachTerminal(ArpTerminalInterface& terminal)
{
	inherited::AttachTerminal(terminal);
}

/* Do a reset of the entire terminal.  If "hard" is true, this is a
   full hard reset. */

void ArpVT200::Reset(bool hard)
{
	do_reset(hard,true);
}

const ArpEmulationType* ArpVT200::EmulationType(void) const
{
	if( emulation < 0 ) return &type_info[0];
	else if( emulation >= EMU_NUM ) return &type_info[EMU_NUM-1];
	return &type_info[emulation];
}

/* ----------------------------------------------------------
	"ArpVT200" interface from terminal display to remote device.
   ---------------------------------------------------------- */
   
/* Handling of complex (non-character) types of events. */

void ArpVT200::build_lr(Command& /*cmd*/, BMessage* /*msg*/)
{
	// XXX not implemented.
}

bool ArpVT200::EmuTTYKeyPress(BMessage* /*msg*/, int32 key, int32 mods)
{
	if( (mods&(B_SHIFT_KEY|B_CONTROL_KEY|B_OPTION_KEY)) != 0 ) return false;
	
	Command scmd;
	DB(DBALL,cdb << "ArpVT200: EmuTTYKeyPress key=" << key << " mods="
			<< mods << endl);
	
	switch( key ) {
	case KEY_RETURN: {
		SendToRemote((ichar)'\r');
	} break;
	case KEY_CURSOR_LEFT: {
		if( mode_states[MODE_DECCKM] && mode_states[MODE_DECANM] ) {
			scmd.setType(Command::TYPE_ESC);
			scmd.addIntermediates("O");
		} else if( mode_states[MODE_DECANM] ) {
			scmd.setType(Command::TYPE_CSI);
		} else {
			scmd.setType(Command::TYPE_ESC);
		}
		scmd.setTerminator('D');
	} break;
	case KEY_CURSOR_RIGHT: {
		if( mode_states[MODE_DECCKM] && mode_states[MODE_DECANM] ) {
			scmd.setType(Command::TYPE_ESC);
			scmd.addIntermediates("O");
		} else if( mode_states[MODE_DECANM] ) {
			scmd.setType(Command::TYPE_CSI);
		} else {
			scmd.setType(Command::TYPE_ESC);
		}
		scmd.setTerminator('C');
	} break;
	case KEY_CURSOR_UP: {
		if( mode_states[MODE_DECCKM] && mode_states[MODE_DECANM] ) {
			scmd.setType(Command::TYPE_ESC);
			scmd.addIntermediates("O");
		} else if( mode_states[MODE_DECANM] ) {
			scmd.setType(Command::TYPE_CSI);
		} else {
			scmd.setType(Command::TYPE_ESC);
		}
		scmd.setTerminator('A');
	} break;
	case KEY_CURSOR_DOWN: {
		if( mode_states[MODE_DECCKM] && mode_states[MODE_DECANM] ) {
			scmd.setType(Command::TYPE_ESC);
			scmd.addIntermediates("O");
		} else if( mode_states[MODE_DECANM] ) {
			scmd.setType(Command::TYPE_CSI);
		} else {
			scmd.setType(Command::TYPE_ESC);
		}
		scmd.setTerminator('B');
	} break;
	case KEY_HOME: {
		// XXX currently ignore it.
		return false;
	}
	case KEY_END: {
		// XXX currently ignore it.
		return false;
	} break;
	default: {
		scmd.setType(Command::TYPE_CSI);
		switch( key ) {	
			case KEY_PAGE_UP: scmd.addParam(5,0); break;
			case KEY_PAGE_DOWN: scmd.addParam(6,0); break;
			case KEY_F1: scmd.addParam(11,0); break;
			case KEY_F2: scmd.addParam(12,0); break;
			case KEY_F3: scmd.addParam(13,0); break;
			case KEY_F4: scmd.addParam(14,0); break;
			case KEY_F5: scmd.addParam(15,0); break;
			case KEY_F6: scmd.addParam(17,0); break;
			case KEY_F7: scmd.addParam(18,0); break;
			case KEY_F8: scmd.addParam(19,0); break;
			case KEY_F9: scmd.addParam(20,0); break;
			case KEY_F10: scmd.addParam(21,0); break;
			case KEY_F11: scmd.addParam(23,0); break;
			case KEY_F12: scmd.addParam(24,0); break;
			default: return false;
		}
		scmd.setTerminator('~');
	} break;
	}

	if( scmd.type != Command::TYPE_NONE ) {
		DB(DBALL,cdb << "ArpVT200: Sending event: "
						 << scmd.toString() << endl);
		SendToRemote(scmd.toSequence());
		return true;
	}
	return false;
}

bool ArpVT200::EmuTTYMouseGen(BMessage* msg, BPoint pos, int32 /*buttons*/)
{
	Command scmd;
	int32 mod = msg->FindInt32("modifiers");
	if( (mod&B_SHIFT_KEY) != 0 ) return false;

	if( mode_states[MODE_DECELR] ) {
		build_lr(scmd,msg);
	} else if( mode_states[MODE_MITMRM] || mode_states[MODE_XTMMRM]
				|| mode_states[MODE_XTMHMRM] ) {
		char button = ' ';
		if( (mod&B_OPTION_KEY) != 0 ) {
			button = '"';
		} else if( (mod&B_CONTROL_KEY) != 0 ) {
			button = '!';
		}
		if( msg->what == B_MOUSE_UP ) {
			if( mode_states[MODE_MITMRM] ) return false;
			button = '#';
		}
		ichar x = (ichar)(terminal.TermXToCol(pos.x)+'!');
		ichar y = (ichar)(terminal.TermYToRow(pos.y)+'!');
		scmd.setType(Command::TYPE_RAW);
		scmd.addData(ANSI_ESC);
		scmd.addData('[');
		scmd.addData('M');
		scmd.addData(button);
		scmd.addData(x);
		scmd.addData(y);
	}

	if( scmd.type != Command::TYPE_NONE ) {
		DB(DBALL,cdb << "ArpVT200: Sending event: "
						 << scmd.toString() << endl);
		SendToRemote(scmd.toSequence());
		return true;
	}
	return false;
}

bool ArpVT200::EmuTTYMouseDown(BMessage* msg, BPoint pos, int32 buttons)
{
	return EmuTTYMouseGen(msg,pos,buttons);
}

bool ArpVT200::EmuTTYMouseUp(BMessage* msg, BPoint pos, int32 buttons)
{
	return EmuTTYMouseGen(msg,pos,buttons);
}

/* ----------------------------------------------------------
	"ArpVT200" interface from remote device to terminal display.
   ---------------------------------------------------------- */
   
void ArpVT200::EmulateToTTY(const ichar* d, size_t len)
{
	while( len > 0 ) {
		const ichar* spec = EmuRemoteNextSpecial(d,len);
		if( spec > d ) {
			size_t cnt = (spec-d)/sizeof(ichar);
			if( cnt > 0 ) {
				DB(DBALL,cdb << "ArpVT200: Show Text '"
							<< sequenceToString(ArpString(d,0,cnt))
							<< "'" << endl);
				if( mode_states[MODE_IRM] ) {
					int32 col=0;
					terminal.TermGetCursorPos(NULL,&col);
					terminal.TermInsertChars(col,col+cnt-1);
				}
				terminal.TermSendTTY(d,cnt,terminal.TERM_OUTPARTIAL);
			}
			d = spec;
			if( cnt < len ) len -= cnt;
			else len = 0;
		}
		if( len > 0 ) {
			DB(DBALL,cdb << "ArpVT200: Process char '"
						<< sequenceToString(ArpString(d,0,1))
						<< "'" << endl);
			if( EmuRemoteChar(*d) ) {
				if( mode_states[MODE_IRM] ) {
					int32 col=0;
					terminal.TermGetCursorPos(NULL,&col);
					terminal.TermInsertChars(col,col);
				}
				terminal.TermSendTTY(d,1,terminal.TERM_OUTPARTIAL);
			}
			d++;
			len--;
		}
	}
	DB(DBALL,cdb << "ArpVT200: Cleanup terminal." << endl);
	terminal.TermClean();
}

const ichar*
ArpVT200::EmuRemoteNextSpecial(const ichar* str, size_t len)
{
	DB(DBALL,cdb << "ArpVT200:EmuRemoteNextSpecial(), state=" << state << endl);
	//	<< ", string=" << sequenceToString(ArpString(str,0,len)) << endl);
	if( state == STATE_ASCII ) {
		while( len > 0 && int(*str) < _NUM_CHAR_FLAGS
				&& !char_flags[int(*str)] ) {
			str++;
			len--;
		}
	}
	DB(DBALL,if( len > 0 && *str == ANSI_HT )
				cdb << "ArpVT200: SPEC Char: HT" << endl);
	return str;
}

/* Main character parsing function: handles control characters and
   the main state machine; dispatches command sequences to
   approriate methods for executing them. */

bool ArpVT200::EmuRemoteChar(ichar c)
{

	// Handle an Xterm non-ANSIness in the DATA state --
	// the data is terminated by -any- non-printing character.
	if( emulation == EMU_XTERM && state == STATE_DATA ) {
		if( c <= (ichar)0x1f ||
				( c >= (ichar)0x80 && c <= (ichar)0x9f) ) {
			DB(DBALL, cerr << "ArpVT200: Faking XTerm end of data."
								<< endl);
			c = ANSI_ST;
		}
	}

	// First, process control characters.

	switch( c ) {

	// These first two cancel the current command.
	case ANSI_CAN: {
		DB(DBALL,cdb << "ArpVT200: CAN (Cancel)" << endl);
		reset_state();
		return false;
	} break;
	case ANSI_SUB: {
		DB(DBALL,cdb << "ArpVT200: SUB (Substitute)" << endl);
		reset_state();
		terminal.TermSetStyle(terminal.TermStyle()
							^ terminal.TERM_STYLEINVERSE);
		terminal.TermSendTTY((const ichar*)"?",1,
								terminal.TERM_OUTPARTIAL);
		terminal.TermSetStyle(terminal.TermStyle()
							^ terminal.TERM_STYLEINVERSE);
		return false;
	} break;

	// ENQ immediately sends an ACK.
	case ANSI_ENQ: {
		DB(DBALL,cdb << "ArpVT200: ENQ (Enquire)" << endl);
		SendToRemote(ANSI_ACK);
		return false;
	} break;
		
	// BS moves cursor to the left.
	case ANSI_BS: {
		DB(DBALL,cdb << "ArpVT200: BS (Backspace)" << endl);
		int32 row,col;
		terminal.TermGetCursorPos(&row,&col);
		if( col > 0 ) col--;
		terminal.TermSetCursorPos(row,col);

		// An Xterm non-ANSIness.
		if( emulation == EMU_XTERM ) {
			DB(DBALL, cerr << "ArpVT200: XTerm BS ends sequence."
								<< endl);
			reset_state();
		}
		return false;
	} break;

	// HT moves to next horizontal tab stop.
	// XXX tabs aren't fully implemented!
	case ANSI_HT: {
		DB(DBALL,cdb << "ArpVT200: HT (Horizontal Tab)" << endl);
		DB(DBALL,cdb << "ArpVT200: Doing HT..." << endl);
		int32 row,col;
		terminal.TermGetCursorPos(&row,&col);
		col++;
		int32 last;
		terminal.TermGetRegion(NULL,NULL,NULL,&last);
		while( col < _NUM_TABS && col <= last && !tabs[col] ) col++;
		if( col > last ) col = last;
		terminal.TermSetCursorPos(row,col);

		// An Xterm non-ANSIness.
		if( emulation == EMU_XTERM ) reset_state();
		return false;
	} break;

	// LF, VT, and FF all move to next line.
	case ANSI_LF:
	case ANSI_VT:
	case ANSI_FF: {
		//DB(DBALL,cdb << "ArpVT200: LF, VT, or FF" << endl);
		terminal.TermSendTTY((const ichar*)"\n",1,
								terminal.TERM_OUTPARTIAL);

		// An Xterm non-ANSIness.
		if( emulation == EMU_XTERM ) reset_state();
		return false;
	} break;

	// Character sets are not currently implented.  Awww...
	case ANSI_SO:
	case ANSI_SI: {
		DB(DBALL,cdb << "ArpVT200: SO or SI" << endl);

		// An Xterm non-ANSIness.
		if( emulation == EMU_XTERM ) reset_state();
		return false;
	} break;

		// DC1 and DC3 are ignored.
	case ANSI_DC1:
	case ANSI_DC3: {
		DB(DBALL,cdb << "ArpVT200: DC1 or DC3" << endl);
		return false;
	} break;

	// Escape cancels the current command, and starts a new one.
	// However, be aware that ESC-\ is ST, the command data terminator!
	// We thus can't just initialize our state right now...
	case ANSI_ESC: {
		DB(DBALL,cdb << "ArpVT200: Entered ESC" << endl);
		state = STATE_ESC;
		return false;
	} break;

	// Transform 8-bit escapes into their respective escape sequence.
	default: {
		if( c >= (ichar)0x80 && c <= (ichar)0x9f ) {
			DB(DBALL,cdb << "ArpVT200: 8bit: " << charToString(c) << endl);
			sub_cmd.init();
			sub_cmd.type = STATE_ESC;
			sub_cmd.term = (ichar)( c - ( (ichar)0x80 - '@' ) );
			doESC(sub_cmd);
			return false;
		}
	} break;
	}

	switch( state ) {

	/* Process normally -- display the characters. */
	case STATE_ASCII:
		return true;

	/* Process a basic escape sequence -- intermediate characters
	   followed by a terminal character. */
	case STATE_ESC:
		if( c == '\\' ) {
			sub_cmd.init();
			sub_cmd.type = STATE_ESC;
			sub_cmd.term = '\\';
			doESC(sub_cmd);
			return false;
		} else if( mode_states[MODE_DECANM] &&
		 			c >= (ichar)0x20 && c <= (ichar)0x2f ) {
			cmd.parseInterChar(c);
			return false;
		} else if( (c >= (ichar)0x20 && c <= (ichar)0x7e)
					|| c >= (ichar)0xa0 ) {
			cmd.parseTermChar(c);
			cmd.type = Command::TYPE_ESC;
			doESC(cmd);
			if( state == STATE_ESC ) {
				reset_state();
			}
			return false;
		}
		return true;

	/* Process control sequence -- parameters followed by
	   intermediate characters followed by terminal character. */
	case STATE_CSI:
		if( c >= (ichar)0x30 && c <= (ichar)0x3f ) {
			cmd.parseParamChar(c);
			return false;
		} else if( c >= (ichar)0x20 && c <= (ichar)0x2f ) {
			cmd.parseInterChar(c);
			return false;
		} else if( (c >= (ichar)0x20 && c <= (ichar)0x7e)
					|| c >= (ichar)0xa0 ) {
			// We let all printable characters terminate the control
			// sequence, to handle some non-ANSI sequences.
			cmd.parseTermChar(c);
			doCSI(cmd);
			reset_state();
			return false;
		}
		return true;

	/* Process other sequence -- parameters followed by intermediate
	   characters followed by terminal character, then go into
	   data processing state. */
	case STATE_DCS:
	case STATE_OSC:
	case STATE_PM:
	case STATE_APC:
		// Another Xterm non-ANSIness.  *sigh*
		if( emulation == EMU_XTERM ) {
			if( c < '0' || c > '9' ) {
				cmd.parseTermChar(c);
				state = STATE_DATA;
				return false;
			}
		}
		if( c >= (ichar)0x30 && c <= (ichar)0x3f ) {
			cmd.parseParamChar(c);
			return false;
		} else if( c >= (ichar)0x20 && c <= (ichar)0x2f ) {
			cmd.parseInterChar(c);
			return false;
		} else if( (c >= (ichar)0x20 && c <= (ichar)0x7e)
					|| c >= (ichar)0xa0 ) {
			cmd.parseTermChar(c);
			state = STATE_DATA;
			return false;
		}

		return true;

	/* Collect command data.  ST will be intercepted above, to cause
	   the final command to actually be executed. */
	case STATE_DATA:
		if( (c >= (ichar)0x20 && c <= (ichar)0x7e)
				|| c >= (ichar)0xa0 ) {
			cmd.parseDataChar(c);
			return false;
		}

		return true;

	/* Process a VT52 direct-cursor address command. */
	case STATE_DCA1:
		dca_line = (int)(c-' ');
		state = STATE_DCA2;
		return false;
	case STATE_DCA2:
		terminal.TermSetCursorPos(dca_line,(int)(c-' '));
		reset_state();
		return false;

	default:
		state = STATE_ASCII;
		RPTERR(("ArpVT200: Unknown emulator state!\n"));
		return false;
	}
}

/* ----------------------------------------------------------
	"ArpVT200" class initialization and global control.
   ---------------------------------------------------------- */
   
/* Set modes to initial states. */

void ArpVT200::init_modes(void)
{
	for( int i=0; i<_NUM_MODE; i++ ) {
		mode_states[i] = mode_saves[i] = mode_inits[i];
	}
}

/* Set tabs to defaults -- every eight characters. */

void ArpVT200::init_tabs(void)
{
	for( int i=0; i<_NUM_TABS; i+=1 ) {
		tabs[i] = (i%8)==0;
	}
}

/* Reset the state machine back to displaying ASCII characters. */

void ArpVT200::reset_state(void)
{
	if( state != STATE_ASCII ) {
		cmd.init();
	}
	state = STATE_ASCII;
}

/* Reflect the emulation mode back into the Terminal's settings. */

void ArpVT200::update_terminal_mode(void)
{
	int mode = 0;
	if( mode_states[MODE_DECSCLM] ) mode |= terminal.TERM_MODESMOOTH;
	if( !mode_states[MODE_DECAWM] ) mode |= terminal.TERM_MODENOWRAP;
	if( mode_states[MODE_DECSCNM] ) mode |= terminal.TERM_MODEINVERSE;
	if( mode_states[MODE_LNM] ) mode |= terminal.TERM_MODENEWLINE;
	if( !mode_states[MODE_DECTCEM] ) mode |= terminal.TERM_MODEHIDECURSOR;
	terminal.TermSetMode(mode);
}

/* Do a reset of the emulator.  If "hard" is true, this is a
   full hard reset.  If 'subclass' is true, the subclass/terminal
   is also reset. */

void ArpVT200::do_reset(bool hard, bool subclass)
{
	init_modes();
	init_tabs();
	reset_state();
	if( subclass ) inherited::Reset(hard);
	update_terminal_mode();
	if( emulation < EMU_VT220 ) {
		mode_states[MODE_DECANM] = true;
		emulation = EMU_XTERM;
	} else if( emulation < EMU_VT100 ) {
		mode_states[MODE_DECANM] = true;
		emulation = EMU_VT220;
	} else if( emulation < EMU_VT52 ) {
		mode_states[MODE_DECANM] = true;
		emulation = EMU_VT100;
	} else if( emulation < EMU_NUM ) {
		mode_states[MODE_DECANM] = false;
		emulation = EMU_VT52;
	}
}

/* ----------------------------------------------------------
	"ArpVT200" class "ESC" sequence execution.
   ---------------------------------------------------------- */
   
void ArpVT200::doESC(Command& mycmd)
{

	if( mycmd.cur_inter <= 0 ) {
		switch( mycmd.term ) {

		// VT52 cursor up sequence
		case 'A': {
			DB(DBALL,cdb << "ArpVT200: (VT52) Cursor up." << endl);
			int32 row,col;
			terminal.TermGetCursorPos(&row,&col);
			if( --row < 0 ) row = 0;
			terminal.TermSetCursorPos(row,col);
			return;
		} break;
		
		// VT52 cursor down sequence
		case 'B': {
			DB(DBALL,cdb << "ArpVT200: (VT52) Cursor down." << endl);
			int32 row,col;
			terminal.TermGetCursorPos(&row,&col);
			int32 num_rows;
			terminal.TermGetSize(&num_rows,NULL);
			if( ++row >= num_rows ) row = num_rows-1;
			terminal.TermSetCursorPos(row,col);
			return;
		} break;
		
		// VT52 cursor right sequence
		case 'C': {
			DB(DBALL,cdb << "ArpVT200: (VT52) Cursor right." << endl);
			int32 row,col;
			terminal.TermGetCursorPos(&row,&col);
			int32 num_cols;
			terminal.TermGetSize(NULL,&num_cols);
			if( ++col >= num_cols ) col = num_cols-1;
			terminal.TermSetCursorPos(row,col);
			return;
		} break;
		
		// IND moves cursor down, possibly scrolling.
		// In VT52 mode, this is cursor left.
		case 'D': {
			if( mode_states[MODE_DECANM] ) {
				DB(DBALL,cdb << "ArpVT200: IND (Index)" << endl);
#if 0
				terminal.TermSendTTY((const ichar*)"\n",1,
										terminal.TERM_OUTPARTIAL);
#endif
				int32 row,col;
				terminal.TermGetCursorPos(&row,&col);
				int32 bottom;
				terminal.TermGetRegion(NULL,&bottom,NULL,NULL);
				if( row < bottom ) row++;
				else terminal.TermScroll(-1);
				terminal.TermSetCursorPos(row,col);
			} else {
				DB(DBALL,cdb << "ArpVT200: (VT52) Cursor left." << endl);
				int32 row,col;
				terminal.TermGetCursorPos(&row,&col);
				if( --col < 0 ) col = 0;
				terminal.TermSetCursorPos(row,col);
			}
			return;
		} break;
		
		// NEL moves cursor down and to column zero, possibly scrolling.
		case 'E': {
			DB(DBALL,cdb << "ArpVT200: NEL (Next Line)" << endl);
			int32 row,col;
			terminal.TermGetCursorPos(&row,&col);
			int32 bottom;
			terminal.TermGetRegion(NULL,&bottom,NULL,NULL);
			if( row < bottom ) row++;
			else terminal.TermScroll(-1);
			terminal.TermSetCursorPos(row,0);
			return;
		} break;
		
		// HTS sets a tab stop.
		// In VT52 mode, this is cursor home.
		case 'H': {
			if( mode_states[MODE_DECANM] ) {
				DB(DBALL,cdb << "ArpVT200: HTS (Horizontal Tab Set)" << endl);
				int32 col;
				terminal.TermGetCursorPos(NULL,&col);
				if( col < _NUM_TABS ) tabs[col] = true;
			} else {
				DB(DBALL,cdb << "ArpVT200: (VT52) Cursor home" << endl);
				terminal.TermSetCursorPos(0,0);
			}
			return;
		} break;
		
		// HTJ.  Not a VT220 code.
		// In VT52 mode, this is reverse line feed.
		case 'I': {
			if( mode_states[MODE_DECANM] ) {
				DB(DBALL,cdb << "ArpVT200: HTJ (Horizontal Tab W/Justification)" << endl);
			} else {
				DB(DBALL,cdb << "ArpVT200: (VT52) Reverse line feed" << endl);
				int32 row,col;
				terminal.TermGetCursorPos(&row,&col);
				int32 top;
				terminal.TermGetRegion(&top,NULL,NULL,NULL);
				if( row > top ) row--;
				else terminal.TermScroll(1);
				terminal.TermSetCursorPos(row,col);
			}
			return;
		} break;
		
		// VT52 Erase to end of screen sequence
		case 'J': {
			DB(DBALL,cdb << "ArpVT200: (VT52) Erase to end of screen." << endl);
			terminal.TermClearEOD(' ',terminal.TermStyle());
			return;
		} break;
		
		// PLD and PLU.  Not a VT220 code.
		// In VT52 mode, this is erase to end of line.
		case 'K': {
			if( mode_states[MODE_DECANM] ) {
				DB(DBALL,cdb << "ArpVT200: PLD (Partial Line Down)" << endl);
			} else {
				DB(DBALL,cdb << "ArpVT200: (VT52) Erase to end of line." << endl);
				terminal.TermClearEOL(' ', terminal.TermStyle());
			}
			return;
		} break;
		
		case 'L': {
			DB(DBALL,cdb << "ArpVT200: PLU (Partial Line Up)" << endl);
			return;
		} break;
		
		// RI moves cursor up, possibly scrolling
		case 'M': {
			DB(DBALL,cdb << "ArpVT200: RI (Reverse Index)" << endl);
			int32 row,col;
			terminal.TermGetCursorPos(&row,&col);
			int32 top;
			terminal.TermGetRegion(&top,NULL,NULL,NULL);
			if( row > top ) row--;
			else terminal.TermScroll(1);
			terminal.TermSetCursorPos(row,col);
			return;
		} break;
		
		// SS2 and SS3.  XXX Character sets are not implemented!
		case 'N': {
			DB(DBALL,cdb << "ArpVT200: SS2 (Single Shift G2)" << endl);
			return;
		} break;
		
		case 'O': {
			DB(DBALL,cdb << "ArpVT200: SS3 (Single Shift G3)" << endl);
			return;
		} break;
		
		// DCS cancels the current command and starts a new
		// device control string.
		case 'P': {
			reset_state();
			DB(DBALL,cdb << "ArpVT200: Entered DCS" << endl);
			cmd.type = state = STATE_DCS;
			return;
		} break;
		
		// VT52 print cursor line: a do-nothing for this emulation.
		case 'V': {
			DB(DBALL,cdb << "ArpVT200: (VT52) Print cursor line" << endl);
			return;
		} break;
		
		// VT52 printer controller modes: a do-nothing for this emulation.
		case 'W': {
			DB(DBALL,cdb << "ArpVT200: (VT52) Enter printer controller mode" << endl);
			return;
		} break;
		case 'X': {
			DB(DBALL,cdb << "ArpVT200: (VT52) Exit printer controller mode" << endl);
			return;
		} break;
		
		// VT52 direct cursor address sequence.
		case 'Y': {
			DB(DBALL,cdb << "ArpVT200: (VT52) Direct cursor address" << endl);
			state = STATE_DCA1;
			return;
		} break;
		
		// VT52 identify sequence.  XXX Not implemented!
		case 'Z': {
			DB(DBALL,cdb << "ArpVT200: (VT52) Identify" << endl);
			sendDA(DA_VT52);
			return;
		} break;
		
		// VT52 enter ANSI sequence.
		case '<': {
			doMode(MODE_DECANM,MOP_SET);
			return;
		} break;
		
		// VT52 enter autoprint mode: a do-nothing for this emulation.
		case ' ': {
			DB(DBALL,cdb << "ArpVT200: (VT52) Enter autoprint mode" << endl);
			return;
		} break;
		
		// RIS resets the terminal.
		case 'c': {
			DB(DBALL,cdb << "ArpVT200: RIS (Reset to Initial State)" << endl);
			reset_state();
			Reset(true);
			return;
		} break;
		
		// CSI cancels the current command and starts a new
		// control sequence.
		case '[': {
			reset_state();
			DB(DBALL,cdb << "ArpVT200: Entered CSI" << endl);
			cmd.type = state = STATE_CSI;
			return;
		} break;
		
		// OSC cancels the current command and starts a new
		// operating system command.
		// In VT52 mode, this is print screen: a do-nothing for this emulation.
		case ']': {
			if( mode_states[MODE_DECANM] ) {
				reset_state();
				DB(DBALL,cdb << "ArpVT200: Entered OSC" << endl);
				cmd.type = state = STATE_OSC;
			} else {
				DB(DBALL,cdb << "ArpVT200: (VT52) Print screen" << endl);
			}
			return;
		} break;
		
		// PM cancels the current command and starts a new
		// privacy message.
		case '^': {
			reset_state();
			DB(DBALL,cdb << "ArpVT200: Entered PM" << endl);
			cmd.type = state = STATE_PM;
			return;
		} break;
		
		// APC cancels the current command and starts a new
		// application program command.
		// In VT52 mode, this is the exit autoprint sequence.
		case '_': {
			if( mode_states[MODE_DECANM] ) {
				reset_state();
				DB(DBALL,cdb << "ArpVT200: Entered APC" << endl);
				cmd.type = state = STATE_APC;
			} else {
				DB(DBALL,cdb << "ArpVT200: (VT52) Exit autoprint mode" << endl);
			}
			return;
		} break;
		
		// ST terminates the collection of a command's data,
		// and executes the final command.
		case '\\': {
			DB(DBALL,cdb << "ArpVT200: ST (String Terminator)" << endl);
			if( state == STATE_DATA ) {
				switch(cmd.type) {
				case STATE_DCS:
					doDCS(cmd);
					reset_state();
					return;
				case STATE_OSC:
					doOSC(cmd);
					reset_state();
					return;
				case STATE_PM:
					doPM(cmd);
					reset_state();
					return;
				case STATE_APC:
					doAPC(cmd);
					reset_state();
					return;
				default:
					RPTERR(("ArpVT200: Unknown state at string terminator.\n"));
					reset_state();
					return;
				}
			}
			RPTERR(("ArpVT200: String terminator outside of string.\n"));
			return;
		} break;

		// The rest are non-ANSI escape sequences.

		case '>': {       // Normal keypad mode
			doMode(MODE_DECKPAM,MOP_RESET);
			return;
		} break;
		case '=': {       // Application keypad mode
			doMode(MODE_DECKPAM,MOP_SET);
			return;
		} break;
		case '8': {       // Restore cursor position
			DB(DBALL,cdb << "ArpVT200: DECRC: Restore cursor." << endl);
			terminal.TermSetCursorPos(saved_row,saved_col);
			terminal.TermSetStyle(saved_style);
			return;
		} break;
		case '7': {       // Save cursor position
			DB(DBALL,cdb << "ArpVT200: DECSC: Save cursor." << endl);
			terminal.TermGetCursorPos(&saved_row,&saved_col);
			saved_style = terminal.TermStyle();
			return;
		} break;
		}

	}

	// Commands with "#" intermediate character.
	if( mycmd.cur_inter == 1 && mycmd.inters[0] == '#' ) {
		switch( mycmd.term ) {
		case '3':
			DB(DBALL,cdb << "ArpVT200: DECDHL: Double-height (top)" << endl);
			return;
		case '4':
			DB(DBALL,cdb << "ArpVT200: DECDHL: Double-height (bottom)" << endl);
			return;
		case '5':
			DB(DBALL,cdb << "ArpVT200: DECSWL: Single-width" << endl);
			return;
		case '6':
			DB(DBALL,cdb << "ArpVT200: DECDWL: Double-width" << endl);
			return;
		}
	}

	mycmd.setType(Command::TYPE_ESC);
	RPTERR(("ArpVT200: Unknown ESC \"%s\"\n",
			(const char*)(mycmd.toString())));
}

/* ----------------------------------------------------------
	"ArpVT200" class "CSI" sequence execution.
   ---------------------------------------------------------- */
   
void ArpVT200::doCSI(Command& mycmd)
{

	int mop = -1;

	switch( mycmd.term ) {
	case 'h':
		mop = MOP_SET;
		break;
	case 'l':
		mop = MOP_RESET;
		break;
	case 's':
		if( mycmd.flags == Command::FLAG_QUES_MASK ) mop = MOP_SAVE;
		break;
	case 'r':
		if( mycmd.flags == Command::FLAG_QUES_MASK ) mop = MOP_RESTORE;
		break;
	}

	if( mop >= 0 ) {
		int flags = mycmd.flags;

		// Cycle through each parameter
		for( int i=0; i<mycmd.cur_param; i++ ) {
			int param = mycmd.params[i];

			// Look for a mode matching the parameter.
			int j = 0;
			while( j<_NUM_MODE ) {
				if( mode_flags[j] == flags && mode_values[j] == param ) {
					doMode((OperatingMode)j,(ModeOperation)mop);
					break;
				}
				j++;
			}

			if( j >= _NUM_MODE ) {
				RPTERR(("ArpVT200: Unknown Mode: %d (Flags=%d, Op=%d)\n",
						param, flags, mop));
			}

		}
		return;
	}

	// Standard commands.
	if( mycmd.flags == 0 && mycmd.cur_inter <= 0 ) {
		int cnt = mycmd.cur_param;
		int* args = mycmd.params;
		bool* given = mycmd.has_params;

		switch( mycmd.term ) {
		case '@': {                    // ICH Insert blanks
			DB(DBALL,cdb << "ArpVT200: ICH: Insert characters." << endl);
			DB(DBALL,cdb << mycmd.paramsToString() << endl);
			int num = args[0];
			if( !given[0] ) num = 1;
			if( num <= 0 ) num = 1;
			int32 row,col;
			terminal.TermGetCursorPos(&row,&col);
			terminal.TermInsertChars(col,col+num-1,
						 ' ', terminal.TERM_STYLEPLAIN);
		} break;
		case 'A': {                    // CUU: Cursor up
			DB(DBALL,cdb << "ArpVT200: CUU: Cursor up." << endl);
			DB(DBALL,cdb << mycmd.paramsToString() << endl);
			int num = args[0];
			if( !given[0] ) num = 1;
			if( num <= 0 ) num = 1;
			int32 row,col;
			terminal.TermGetCursorPos(&row,&col);
			terminal.TermSetCursorPos(row-num,col);
		} break;
		case 'B': {                    // CUD: Cursor down
			DB(DBALL,cdb << "ArpVT200: CUD: Cursor down." << endl);
			DB(DBALL,cdb << mycmd.paramsToString() << endl);
			int num = args[0];
			if( !given[0] ) num = 1;
			if( num <= 0 ) num = 1;
			int32 row,col;
			terminal.TermGetCursorPos(&row,&col);
			terminal.TermSetCursorPos(row+num,col);
		} break;
		case 'C': {                    // CUF: Cursor right
			DB(DBALL,cdb << "ArpVT200: CUF: Cursor forward." << endl);
			DB(DBALL,cdb << mycmd.paramsToString() << endl);
			int num = args[0];
			if( !given[0] ) num = 1;
			if( num <= 0 ) num = 1;
			int32 row,col;
			terminal.TermGetCursorPos(&row,&col);
			terminal.TermSetCursorPos(row,col+num);
		} break;
		case 'D': {                    // CUB: Cursor left
			DB(DBALL,cdb << "ArpVT200: CUB: Cursor backward." << endl);
			DB(DBALL,cdb << mycmd.paramsToString() << endl);
			int num = args[0];
			if( !given[0] ) num = 1;
			if( num <= 0 ) num = 1;
			int32 row,col;
			terminal.TermGetCursorPos(&row,&col);
			terminal.TermSetCursorPos(row,col-num);
		} break;
		case 'E': {                    // CNL: Cursor to next line
			DB(DBALL,cdb << "ArpVT200: CNL: Cursor next line." << endl);
			DB(DBALL,cdb << mycmd.paramsToString() << endl);
			// XXX Don't know the correct implementation...
			int num = args[0];
			if( !given[0] ) num = 1;
			if( num <= 0 ) num = 1;
			int32 row;
			terminal.TermGetCursorPos(&row,NULL);
			row += num;
			int32 bottom;
			terminal.TermGetRegion(NULL,&bottom,NULL,NULL);
			if( row > bottom ) {
				terminal.TermScroll(bottom-row);
				row = bottom;
			}
			terminal.TermSetCursorPos(row,0);
		} break;
		case 'F': {                    // CPL: Cursor to previous line
			DB(DBALL,cdb << "ArpVT200: CPL: Cursor previous line." << endl);
			DB(DBALL,cdb << mycmd.paramsToString() << endl);
			// XXX Don't know the correct implementation...
			int num = args[0];
			if( !given[0] ) num = 1;
			if( num <= 0 ) num = 1;
			int32 row;
			terminal.TermGetCursorPos(&row,NULL);
			row -= num;
			int32 top;
			terminal.TermGetRegion(&top,NULL,NULL,NULL);
			if( row < top ) {
				terminal.TermScroll(top-row);
				row = top;
			}
			terminal.TermSetCursorPos(row,0);
		} break;
		case 'H':                     // CUP or HVP: Set cursor pos
		case 'f': {
#ifdef DEBUG
			if( mycmd.term == 'H' ) {
				cdb << "ArpVT200: CUP: Cursor position." << endl;
			} else {
				cdb << "ArpVT200: HVP: Horiz/vert position." << endl;
			}
#endif
			DB(DBALL,cdb << mycmd.paramsToString() << endl);
			int32 row,col;
			terminal.TermGetCursorPos(&row,&col);
			int32 top,bottom,left,right;
			terminal.TermGetRegion(&top,&bottom,&left,&right);
			// Yeach, this is not written well!
			if( cnt == 0 ) {
				if( mode_states[MODE_DECOM] ) {
					row = top;
					col = left;
				} else {
					row = col = 0;
				}
			} else {
				if( given[0] && cnt >= 1 ) {
					row = args[0]-1;
					if( mode_states[MODE_DECOM] ) {
						row += top;
					}
				}
				if( given[1] && cnt >= 2 ) {
					col = args[1]-1;
					if( mode_states[MODE_DECOM] ) {
						col += left;
					}
				}
			}
			if( mode_states[MODE_DECOM] ) {
				if( row > bottom ) {
					row = bottom;
				}
				if( col > right ) {
					col = right;
				}
			}
			terminal.TermSetCursorPos(row,col);
		} break;
		case 'I': {                    // CHT: Move to tabstop N
			DB(DBALL,cdb << "ArpVT200: CHT: Cursor horizontal tab." << endl);
			DB(DBALL,cdb << mycmd.paramsToString() << endl);
			// XXX this may not be completely right...
			int i=0;
			int32 right;
			terminal.TermGetRegion(NULL,NULL,NULL,&right);
			if( right >= _NUM_TABS ) right = _NUM_TABS-1;
			for( int num=0; i<=right && num<=mycmd.cur_param; i++ ) {
				if( tabs[i] ) num++;
			}
			int32 row;
			terminal.TermGetCursorPos(&row,NULL);
			terminal.TermSetCursorPos(row,i);
		} break;
		case 'J': {                    // ED: Erase in display
			DB(DBALL,cdb << "ArpVT200: ED: Erase in display." << endl);
			DB(DBALL,cdb << mycmd.paramsToString() << endl);
			if( !given[0] || args[0] == 0 )
				terminal.TermClearEOD(' ',terminal.TermStyle());
			else if( args[0] == 1 )
				terminal.TermClearBOD(' ',terminal.TermStyle());
			else if( args[0] == 2 ) {
				int32 top,bottom;
				terminal.TermGetRegion(&top,&bottom,NULL,NULL);
				terminal.TermScroll(-(bottom-top+1));
			}
		} break;
		case 'K': {                    // EL: Erase in line
			DB(DBALL,cdb << "ArpVT200: EL: Erase in line." << endl);
			DB(DBALL,cdb << mycmd.paramsToString() << endl);
			if( !given[0] || args[0] == 0 )
				terminal.TermClearEOL(' ',terminal.TermStyle());
			else if( args[0] == 1 )
				terminal.TermClearBOL(' ',terminal.TermStyle());
			else if( args[0] == 2 )
				terminal.TermClearLine(' ',terminal.TermStyle());
		} break;
		case 'L': {                    // IL: Insert lines at cursor
			DB(DBALL,cdb << "ArpVT200: IL: Insert line." << endl);
			DB(DBALL,cdb << mycmd.paramsToString() << endl);
			int num = args[0];
			if( !given[0] ) num = 1;
			int32 row;
			terminal.TermGetCursorPos(&row,NULL);
			int32 bottom;
			terminal.TermGetRegion(NULL,&bottom,NULL,NULL);
			if( row < 0 ) row = 0;
			terminal.TermScrollRegion(row,bottom,num);
		} break;
		case 'M': {                    // DL: Delete lines at cursor
			DB(DBALL,cdb << "ArpVT200: DL: Delete line." << endl);
			DB(DBALL,cdb << mycmd.paramsToString() << endl);
			int num = args[0];
			if( !given[0] ) num = 1;
			int32 row;
			terminal.TermGetCursorPos(&row,NULL);
			int32 bottom;
			terminal.TermGetRegion(NULL,&bottom,NULL,NULL);
			terminal.TermScrollRegion(row,bottom,-num);
		} break;
		case 'P': {                    // DCH: Delete character
			DB(DBALL,cdb << "ArpVT200: DCH: Delete character" << endl);
			DB(DBALL,cdb << mycmd.paramsToString() << endl);
			int num = args[0]-1;
			if( !given[0] ) num = 0;
			if( num < 0 ) num = 0;
			int32 col;
			terminal.TermGetCursorPos(NULL,&col);
			terminal.TermDeleteChars(col,col+num,
						 ' ', terminal.TERM_STYLEPLAIN);
		} break;
		case 'X': {                    // ECH: Erase character
			DB(DBALL,cdb << "ArpVT200: ECH: Erase character" << endl);
			DB(DBALL,cdb << mycmd.paramsToString() << endl);
			int num = args[0]-1;
			if( !given[0] ) num = 0;
			if( num < 0 ) num = 0;
			int32 col;
			terminal.TermGetCursorPos(NULL,&col);
			terminal.TermSetChars(col,col+num,
						' ', terminal.TERM_STYLEPLAIN);
		} break;
		case 'c': {                    // DA: Device attr report
			if( cnt <= 0 || (cnt == 1 && args[0] == 0) ) {
				sendDA(DA_PRIMARY);
			} else {
				RPTERR(("ArpVT200: Unknown DA: %s\n",
						(const char*)(mycmd.toString())));
			}
		} break;
		case 'g': {
			DB(DBALL,cdb << "ArpVT200: TBC: Tabulation clear." << endl);
			DB(DBALL,cdb << mycmd.paramsToString() << endl);
			if( mycmd.cur_param <= 0 || mycmd.params[0] == 0 ) {
				for( int i=0; i<_NUM_TABS; i++ ) tabs[i] = false;
			} else if( mycmd.params[0] == 3 ) {
				int32 col;
				terminal.TermGetCursorPos(NULL,&col);
				if( col < _NUM_TABS ) tabs[col] = false;
			}
		} break;
		case 'm': {
			DB(DBALL,cdb << "ArpVT200: SGR: Set graphic rendition." << endl);
			DB(DBALL,cdb << mycmd.paramsToString() << endl);
			if( cnt <= 0 ) {
				DB(DBALL,cdb << "ArpVT200: Plain style." << endl);
				terminal.TermSetStyle(terminal.TERM_STYLEPLAIN);
				terminal.TermSetForeColor(0);
				terminal.TermSetBackColor(0);
			} else {
				for( int i=0; i<cnt; i++ ) {
					switch( args[i] ) {
					case 0:
						DB(DBALL,cdb << "ArpVT200: Plain style." << endl);
						terminal.TermSetStyle(terminal.TERM_STYLEPLAIN);
						terminal.TermSetForeColor(0);
						terminal.TermSetBackColor(0);
						break;
					case 1:
						DB(DBALL,cdb << "ArpVT200: Bold style." << endl);
						terminal.TermSetStyle(terminal.TermStyle() | terminal.TERM_STYLEBOLD);
						break;
					case 4:
						DB(DBALL,cdb << "ArpVT200: Underscore style." << endl);
						terminal.TermSetStyle(terminal.TermStyle() | terminal.TERM_STYLEUNDERSCORE);
						break;
					case 5:
						DB(DBALL,cdb << "ArpVT200: Italic style." << endl);
						terminal.TermSetStyle(terminal.TermStyle() | terminal.TERM_STYLEITALIC);
						break;
					case 7:
						DB(DBALL,cdb << "ArpVT200: Inverse style." << endl);
						terminal.TermSetStyle(terminal.TermStyle() | terminal.TERM_STYLEINVERSE);
						break;
						/* VT220 control code */
					case 22:
						DB(DBALL,cdb << "ArpVT200: Bold style off." << endl);
						terminal.TermSetStyle(terminal.TermStyle() & ~terminal.TERM_STYLEBOLD);
						break;
					case 24:
						DB(DBALL,cdb << "ArpVT200: Underscore style off." << endl);
						terminal.TermSetStyle(terminal.TermStyle()
						& ~terminal.TERM_STYLEUNDERSCORE);
						break;
					case 25:
						DB(DBALL,cdb << "ArpVT200: Italic style off." << endl);
						terminal.TermSetStyle(terminal.TermStyle() & ~terminal.TERM_STYLEITALIC);
						break;
					case 27:
						DB(DBALL,cdb << "ArpVT200: Inverse style off." << endl);
						terminal.TermSetStyle(terminal.TermStyle() & ~terminal.TERM_STYLEINVERSE);
						break;
					default:
						if( args[i] >= 30 && args[i] <= 39 )
							terminal.TermSetForeColor(args[i]-30+1);
						else if( args[i] >= 40 && args[i] <= 49 )
							terminal.TermSetBackColor(args[i]-40+1);
						else
							DB(DBALL,cdb <<
								"ArpVT200: Unknown style: " << args[i]
								<< endl);
					break;
					}
				}
			}
		} break;
		case 'n': {                    // Device status report
			doDSR(mycmd);
		} break;
		case 'q': {                    // DECLL: Load LEDs
			DB(DBALL,cdb << "ArpVT200: DECLL: Load LEDs." << endl);
			DB(DBALL,cdb << mycmd.paramsToString() << endl);
#ifdef DEBUG
			for( int p=0; p<mycmd.cur_param; p++ ) {
				ArpString str("       ");
				for( int i=0; i<4; i++ ) {
					if( (mycmd.params[p]&(1<<i)) != 0 )
						str = str + "*** ";
					else str = str + "--- ";
				}
				cdb << str << endl;
			}
#endif
		} break;
		case 'r': {                    // DECSTBM: Set scroll region
			DB(DBALL,cdb << "ArpVT200: DECSTBM: Set scroll region." << endl);
			DB(DBALL,cdb << mycmd.paramsToString() << endl);
			int32 top = 0;
			int32 bottom = 0;
			terminal.TermGetSize(&bottom,NULL);
			bottom--;
			if( given[0] && cnt >= 1 ) top = args[0]-1;
			if( given[1] && cnt >= 2 ) bottom = args[1]-1;
			terminal.TermSetRegion(top,bottom);
			if( mode_states[MODE_DECOM] ) {
				int32 top,left;
				terminal.TermGetRegion(&top,NULL,&left,NULL);
				terminal.TermSetCursorPos(top,left);
			} else {
				terminal.TermSetCursorPos(0,0);
			}
		} break;
		case 'x': {                    // DECREQTPARM: Request term params
			DB(DBALL,cdb << "ArpVT200: DECREQTPARM: Request term params." << endl);
			DB(DBALL,cdb << mycmd.paramsToString() << endl);
			Command scmd;
			scmd.setType(Command::TYPE_CSI);
			scmd.addParam(2,0);   // Only reporting on request
			scmd.addParam(1,0);   // No parity set
			scmd.addParam(1,0);   // 8 bits per character
			scmd.addParam(120,0); // 19200 bps transmit speed
			scmd.addParam(120,0); // 19200 bps receive speed
			scmd.addParam(1,0);   // Bit rate multiplier is 16
			scmd.addParam(0,0);   // Switch flags.   XXX not implemented.
			scmd.setTerminator('x');
			DB(DBALL,cdb << "ArpVT200: Sending DECREPTPARM: "
						<< scmd.toString() << endl);
			SendToRemote(scmd.toSequence());
		} break;
		case 'y': {                    // DECTST: Invoke confidence test
			DB(DBALL,cdb << "ArpVT200: DECTST: Invoke confidence test." << endl);
			DB(DBALL,cdb << mycmd.paramsToString() << endl);
			if( mycmd.cur_param == 2 && mycmd.params[0] == 2
					&& mycmd.params[1] == 0 ) {
				// This one does a reset.
				reset_state();
				Reset(true);
			}
		} break;
		default:
			RPTERR(("ArpVT200: Unknown CSI: %s\n",
					(const char*)(mycmd.toString())));
			break;
		}
		return;
	}

	// Command with intermediate sequence '' and '?' flag
	if( mycmd.flags == Command::FLAG_QUES_MASK && mycmd.cur_inter <= 0 ) {
		switch( mycmd.term ) {

		// DSR - device status report.
		case 'n': {
			doDSR(cmd);
			return;
		} break;
		}
	}

	// Command with intermediate sequence '' and '>' flag
	if( mycmd.flags == Command::FLAG_GT_MASK && mycmd.cur_inter <= 0 ) {
		switch( mycmd.term ) {

		// DA - secondary device attribute report.
		case 'c': {
			if( mycmd.cur_param <= 0 ||
					(mycmd.cur_param == 1 && mycmd.params[0] == 0) ) {
				sendDA(DA_SECONDARY);
			} else {
				RPTERR(("ArpVT200: Unknown DA: %s\n",
						(const char*)(mycmd.toString())));
			}
			return;
		} break;
		}
	}

	// Command with intermediate sequence '!' and no flags
	if( mycmd.flags == 0 && mycmd.cur_inter == 1 && mycmd.inters[0] == '!' ) {
		switch( mycmd.term ) {

		// DECSTR does a soft-reset of the terminal
		case 'p': {
			DB(DBALL,cdb << "ArpVT200: DECSTR (Soft reset)" << endl);
			reset_state();
			Reset(false);
			return;
		} break;
		}
	}

	// Command with intermediate sequence ''' and no flags
	if( mycmd.flags == 0 && mycmd.cur_inter == 1 && mycmd.inters[0] == '\'' ) {
		switch( mycmd.term ) {

		/* These aren't implemented yet...

		// DECELR - enable locator reporting.
		case 'z': {
			DB(DBALL,cdb << "ArpVT200: DECELR (Enable Locator Reports)" << endl);
			mode_states[MODE_DECELR] = false;
			mode_states[MODE_DECOLR] = false;
			mode_states[MODE_DECPLR] = false;
			if( mycmd.cur_param > 0 && mycmd.params[0] == 1 ) {
				mode_states[MODE_DECELR] = true;
			} else if( mycmd.cur_param > 0 && mycmd.params[0] == 2 ) {
				mode_states[MODE_DECELR] = true;
				mode_states[MODE_DECOLR] = true;
			}
			if( mycmd.cur_param > 1 && mycmd.params[1] == 1 ) {
				mode_states[MODE_DECPLR] = true;
			}
			return;
		} break;
		
		// DECRLP - request current locator position.
		case '|': {
			DB(DBALL,cdb << "ArpVT200: DECRLP (Request Locator Position)" << endl);
			if( mycmd.cur_param <= 0 || mycmd.params[0] <= 1 ) {
				Command scmd = new Command();
				build_lr(scmd,NULL);
				DB(DBALL,cdb << "ArpVT200: Sending event: " <<
							 scmd.toString() << endl);
				send(scmd.toSequence());
				return;
			}
		} break;
		*/
		}
	}

	RPTERR(("ArpVT200: Unknown CSI: %s\n",
			(const char*)(mycmd.toString())));
}

/* ----------------------------------------------------------
	"ArpVT200" class "DSR" sequence execution.
   ---------------------------------------------------------- */
   
bool ArpVT200::doDSR(Command& mycmd)
{
	Command scmd;
	if( mycmd.cur_param >= 1 && mycmd.cur_inter <= 0 ) {
		if( mycmd.flags == 0 && mycmd.params[0] == 6 ) {
			DB(DBALL,cdb << "ArpVT200: DSR - cursor position." << endl);
			scmd.setType(Command::TYPE_CSI);
			int32 row,col;
			terminal.TermGetCursorPos(&row,&col);
			if( mode_states[MODE_DECOM] ) {
				int32 top,left;
				terminal.TermGetRegion(&top,NULL,&left,NULL);
				row -= top;
				col -= left;
			}
			row++;
			col++;
			if( row < 1 ) row = 1;
			if( col < 1 ) col = 1;
			scmd.addParam(row,0);
			scmd.addParam(col,0);
			scmd.setTerminator('R');
		} else if( mycmd.flags == 0 && mycmd.params[0] == 5 ) {
			DB(DBALL,cdb << "ArpVT200: DSR - terminal status." << endl);
			scmd.setType(Command::TYPE_CSI);
			scmd.addParam(0,0);
			scmd.setTerminator('n');
		} else if( mycmd.flags == Command::FLAG_QUES_MASK &&
				 mycmd.params[0] == 15 ) {
			DB(DBALL,cdb << "ArpVT200: DSR - printer status." << endl);
			scmd.setType(Command::TYPE_CSI);
			scmd.addParam(13,Command::FLAG_QUES_MASK);  // No printer.
			scmd.setTerminator('n');
		} else if( mycmd.flags == Command::FLAG_QUES_MASK &&
				 mycmd.params[0] == 25 ) {
			DB(DBALL,cdb << "ArpVT200: DSR - user defined keys." << endl);
			scmd.setType(Command::TYPE_CSI);
			scmd.addParam(21,Command::FLAG_QUES_MASK);  // Locked
			scmd.setTerminator('n');
		} else {
			RPTERR(("ArpVT200: Unknown DSR: %s\n",
					(const char*)(mycmd.toString())));
		}
	} else {
		RPTERR(("ArpVT200: Unknown DSR: %s\n",
				(const char*)(mycmd.toString())));
	}

	if( scmd.type != Command::TYPE_NONE ) {
		DB(DBALL,cdb << "ArpVT200: Sending DSR report: "
					<< scmd.toString() << endl);
		SendToRemote(scmd.toSequence());
		return true;
	}
	return false;
}

/* ----------------------------------------------------------
	"ArpVT200" class "DCS" sequence execution.
   ---------------------------------------------------------- */
   
void ArpVT200::doDCS(Command& mycmd)
{

	// Standard commands.
	if( mycmd.flags == 0 && mycmd.cur_inter <= 0 ) {
		switch( mycmd.term ) {
		default: {
			RPTERR(("ArpVT200: Unknown DCS: %s\n",
					(const char*)(mycmd.toString())));
		} break;
		}
		return;
	}

	RPTERR(("ArpVT200: Unknown DCS: %s\n",
			(const char*)(mycmd.toString())));
}

/* ----------------------------------------------------------
	"ArpVT200" class "OSC" sequence execution.
   ---------------------------------------------------------- */
   
void ArpVT200::doOSC(Command mycmd)
{

	// Standard commands.
	if( mycmd.flags == 0 && mycmd.cur_inter <= 0 ) {
		switch( mycmd.term ) {
		default: {
			RPTERR(("ArpVT200: Unknown OSC: %s\n",
					(const char*)(mycmd.toString())));
		} break;
		}
		return;
	}

	RPTERR(("ArpVT200: Unknown OSC: %s\n",
			(const char*)(mycmd.toString())));
}

/* ----------------------------------------------------------
	"ArpVT200" class "PM" sequence execution.
   ---------------------------------------------------------- */
   
void ArpVT200::doPM(Command& mycmd)
{

	// Standard commands.
	if( mycmd.flags == 0 && mycmd.cur_inter <= 0 ) {
		switch( mycmd.term ) {
		default: {
			RPTERR(("ArpVT200: Unknown PM: %s\n",
					(const char*)(mycmd.toString())));
		} break;
		}
		return;
	}

	RPTERR(("ArpVT200: Unknown PM: %s\n",
			(const char*)(mycmd.toString())));
}

/* ----------------------------------------------------------
	"ArpVT200" class "APC" sequence execution.
   ---------------------------------------------------------- */
   
void ArpVT200::doAPC(Command& mycmd)
{

	// Standard commands.
	if( mycmd.flags == 0 && mycmd.cur_inter <= 0 ) {
		switch( mycmd.term ) {
		default: {
			RPTERR(("ArpVT200: Unknown APC: %s\n",
					(const char*)(mycmd.toString())));
		};
		}
		return;
	}

	RPTERR(("ArpVT200: Unknown APC: %s\n",
			(const char*)(mycmd.toString())));
}

/* ----------------------------------------------------------
	"ArpVT200" class emulation support.
   ---------------------------------------------------------- */
   
/* Send a Device Attribute string, of the given type, to the remote
   host. */

bool ArpVT200::sendDA(DeviceAttribute type)
{
	Command scmd;
	scmd.setType(Command::TYPE_CSI);
	scmd.setTerminator('c');

	if( type == DA_PRIMARY || type == DA_VT52 ) {
		if( emulation == EMU_VT220 ) {
			scmd.addParam(62,Command::FLAG_QUES_MASK);  // VT 220
			int32 num_cols;
			terminal.TermGetSize(NULL,&num_cols);
			if( num_cols >= 132 ) {
				scmd.addParam(1,Command::FLAG_QUES_MASK);   // 132 columns
			}
			//scmd.addParam(2,Command::FLAG_QUES_MASK);   // printer port
			//scmd.addParam(6,Command::FLAG_QUES_MASK);   // selective erase
			//scmd.addParam(7,Command::FLAG_QUES_MASK);   // DRCS
			//scmd.addParam(8,Command::FLAG_QUES_MASK);   // UDK
			//scmd.addParam(9,Command::FLAG_QUES_MASK);   // 7-bit nat. repl. chars
		} else if( type == DA_VT52 ) {
			scmd.setType(Command::TYPE_ESC);
			scmd.addIntermediates("/");
			scmd.setTerminator('Z');
		} else {
			scmd.addParam(6,Command::FLAG_QUES_MASK);     // VT 102
			// Others:
			// ESC [ ? 1; 2 c     -- VT 100 with AVO
			// ESC [ ? 1; 0 c     -- VT 101
			//            ^--------- 1 = STP,      2 = AVO,  4 = GO
			//                       processor op, video op, graphics op
		}

	} else if( type == DA_SECONDARY ) {

		scmd.addParam(1,Command::FLAG_GT_MASK);    // VT 220
		scmd.addParam(10,Command::FLAG_GT_MASK);   // Firmware version 1.0
		scmd.addParam(0,Command::FLAG_GT_MASK);    // No options

	}
		
	if( scmd.type != Command::TYPE_NONE ) {
		DB(DBALL,cdb << "ArpVT200: Sending DA report: " <<
						 scmd.toString() << endl);
		SendToRemote(scmd.toSequence());
		return true;
	}
	return false;
}

/* Do the given operation 'op' on the mode state 'mode'. */

void ArpVT200::doMode(OperatingMode mode, ModeOperation op)
{
	DB(DBALL,cdb << mode_names[mode] << " mode (" <<
					 mode_abbrvs[mode] << "): Op=" << op << endl);
	switch( op ) {
	case MOP_RESET:
		mode_states[mode] = false;
		break;
	case MOP_SET:
		mode_states[mode] = true;
		break;
	case MOP_SAVE:
		mode_saves[mode] = mode_states[mode];
		return;
	case MOP_RESTORE:
		mode_states[mode] = mode_saves[mode];
		break;
	default:
		RPTERR(("ArpVT200: Unknown mode operation: %d\n", op));
	}

	if( mode == MODE_DECOM ) {
		if( mode_states[MODE_DECOM] ) {
			int32 top,left;
			terminal.TermGetRegion(&top,NULL,&left,NULL);
			terminal.TermSetCursorPos(top,left);
		} else {
			terminal.TermSetCursorPos(0,0);
		}
	} else if( mode == MODE_MITMRM && op == MOP_SET ) {
		mode_states[MODE_MITMRM] = false;
		mode_states[MODE_XTMHMRM] = false;
	} else if( mode == MODE_XTMMRM && op == MOP_SET ) {
		mode_states[MODE_MITMRM] = false;
		mode_states[MODE_XTMHMRM] = false;
	} else if( mode == MODE_XTMHMRM && op == MOP_SET ) {
		mode_states[MODE_MITMRM] = false;
		mode_states[MODE_XTMMRM] = false;
	} else if( mode == MODE_DECSCLM || mode == MODE_DECAWM ||
				 mode == MODE_DECSCNM || mode == MODE_LNM ||
				 mode == MODE_DECTCEM ) {
		update_terminal_mode();
	}
}
