//	img2rsrc.cc

#include "img2rsrc.h"
#include "Datatypes.h"
#include "GetBitmap.h"

void usage(void);
BBitmap *bitmap_from_filename(char *);

Img2Rsrc::Img2Rsrc()
	: BApplication(I2R_SIGNATURE)
{
	fileName = NULL;
	iconBmap = NULL;
	micnBmap = NULL;
	append = false;
	force = false;
}

void Img2Rsrc::ArgvReceived(int32 argc, char **argv)
{
	int i;
	status_t err;
	
	if (!DATAVersion)
	{
		fprintf(stderr,"img2rsrc:  libdatatypes.so not found\n");
		exit(1);
	}
	err = DATAInit(I2R_SIGNATURE);
	if (err != B_OK)
	{
		fprintf(stderr,"img2rsrc:  error initializing Datatypes\n");
		exit(1);
	}
	
	for (i = 1; i < argc; i++)
	{
		if (argv[i][0] == '-')
		{
			err = process_option(&argv[i][1],&i,argv);
			if (err)
				exit(1);
		}
		else if (fileName)
		{
			fprintf(stderr,"img2rsrc:  superfluous argument %s\n",argv[i]);
			exit(1);
		}
		else
			fileName = strdup(argv[i]);
	}
}

void Img2Rsrc::ReadyToRun(void)
{
	int i;
	status_t err;
	BFile file;
	size_t length;
	uint32 flags;
	BResources rsrc;
	BNodeInfo nodeInfo;
	uint8 *addr, *bits;
	uint8 buffer[1024];
	BBitmap *bitmap;
	BView *iconView, *micnView;
	BBitmap oldIcon(BRect(0,0,31,31),B_COLOR_8_BIT);
	BBitmap oldMicn(BRect(0,0,15,15),B_COLOR_8_BIT);
	BBitmap iconBitmap(BRect(0,0,31,31),B_COLOR_8_BIT,true);
	BBitmap micnBitmap(BRect(0,0,15,15),B_COLOR_8_BIT,true);
		
	if (!fileName)
	{
		fprintf(stderr,"img2rsrc:  output file not specified\n");
		exit(1);
	}
	else if (!iconBmap && !micnBmap)
	{
		fprintf(stderr,"img2rsrc:  no image files specified\n");
		exit(1);
	}
	flags = B_READ_WRITE | B_CREATE_FILE;
	if (!append)
		flags |= (force ? B_ERASE_FILE : B_FAIL_IF_EXISTS);
	err = file.SetTo(fileName,flags);
	if (err != B_OK)
	{
		switch (err)
		{
			case B_FILE_EXISTS:
				fprintf(stderr,"img2rsrc:  file \"%s\" already exists\n",fileName);
				break;
			case B_PERMISSION_DENIED:
				fprintf(stderr,"img2rsrc:  permission to write to file \"%s\" denied\n",fileName);
				break;
			default:
				fprintf(stderr,"img2rsrc:  error opening file \"%s\"\n",fileName);
		}
		exit(1);
	}
	err = nodeInfo.SetTo(&file);
	if (err != B_OK)
	{
		fprintf(stderr,"img2rsrc:  error getting node info on file \"%s\"\n",fileName);
		exit(1);
	}
	if (!append)
		nodeInfo.SetType("application/x-be-resource");
	err = rsrc.SetTo(&file);
	if (err != B_OK)
	{
		err = rsrc.SetTo(&file,true);
		if (err != B_OK)
		{
			fprintf(stderr,"img2rsrc:  error opening resource fork of file \"%s\"\n",fileName);
			exit(1);
		}
	}
	
	if (!force)
	{
		err = nodeInfo.GetIcon(&oldIcon,B_LARGE_ICON);
		if (err == B_OK)
		{
			fprintf(stderr,"img2rsrc:  'ICON' attribute already exists in file \"%s\"\n",fileName);
			exit(1);
		}
		err = nodeInfo.GetIcon(&oldMicn,B_MINI_ICON);
		if (err == B_OK)
		{
			fprintf(stderr,"img2rsrc:  'MICN' attribute already exists in file \"%s\"\n",fileName);
			exit(1);
		}
		if (rsrc.HasResource('ICON',0L))
		{
			fprintf(stderr,"img2rsrc:  'ICON' resource already exists in file \"%s\"\n",fileName);
			exit(1);
		}
		if (rsrc.HasResource('MICN',0L))
		{
			fprintf(stderr,"img2rsrc:  'MICN' resource already exists in file \"%s\"\n",fileName);
			exit(1);
		}
	}
	
	bitmap = (iconBmap ? iconBmap : micnBmap);
	iconView = new BView(iconBitmap.Bounds(),NULL,B_FOLLOW_ALL_SIDES,B_WILL_DRAW);
	iconBitmap.Lock();
	memset(iconBitmap.Bits(),B_TRANSPARENT_8_BIT,iconBitmap.BitsLength());
	iconBitmap.AddChild(iconView);
	iconView->SetDrawingMode(B_OP_OVER);
	iconView->DrawBitmap(bitmap,iconView->Bounds());
	iconView->Sync();
	iconBitmap.Unlock();
	err = nodeInfo.SetIcon(&iconBitmap,B_LARGE_ICON);
	if (err != B_OK)
	{
		fprintf(stderr,"img2rsrc:  failed to add 'ICON' attribute to file \"%s\"\n",fileName);
		exit(1);
	}
	bits = (uint8 *)iconBitmap.Bits();
	addr = buffer;
	for (i = 0; i < 32; i++)
	{
		memcpy(addr,bits,32);
		addr += 32;
		bits += iconBitmap.BytesPerRow();
	}
	err = rsrc.AddResource('ICON',0,buffer,1024,"BEOS:L:STD_ICON");
	if (err != B_OK)
	{
		fprintf(stderr,"img2rsrc:  failed to add 'ICON' resource to file \"%s\"\n",fileName);
		exit(1);
	}
	
	bitmap = (micnBmap ? micnBmap : iconBmap);
	micnView = new BView(micnBitmap.Bounds(),NULL,B_FOLLOW_ALL_SIDES,B_WILL_DRAW);
	micnBitmap.Lock();
	memset(micnBitmap.Bits(),B_TRANSPARENT_8_BIT,micnBitmap.BitsLength());
	micnBitmap.AddChild(micnView);
	micnView->SetDrawingMode(B_OP_OVER);
	micnView->DrawBitmap(bitmap,micnView->Bounds());
	micnView->Sync();
	micnBitmap.Unlock();
	err = nodeInfo.SetIcon(&micnBitmap,B_MINI_ICON);
	if (err != B_OK)
	{
		fprintf(stderr,"img2rsrc:  failed to add 'MICN' resource to file \"%s\"\n",fileName);
		exit(1);
	}
	bits = (uint8 *)micnBitmap.Bits();
	addr = buffer;
	for (i = 0; i < 16; i++)
	{
		memcpy(addr,bits,16);
		addr += 16;
		bits += micnBitmap.BytesPerRow();
	}
	err = rsrc.AddResource('MICN',0,buffer,256,"BEOS:M:STD_ICON");
	if (err != B_OK)
	{
		fprintf(stderr,"img2rsrc:  failed to add 'MICN' resource to file \"%s\"\n",fileName);
		exit(1);
	}
	
	PostMessage(B_QUIT_REQUESTED);
}
	
status_t Img2Rsrc::process_option(char *option, int *index, char **argv)
{
	if (!strcmp(option,"?"))
	{
		usage();
		return B_ERROR;
	}
	else if (!strcmp(option,"a"))
	{
		append = true;
		return B_OK;
	}
	else if (!strcmp(option,"f"))
	{
		force = true;
		return B_OK;
	}
	else
	{
		(*index)++;
		if (!argv[*index])
		{
			fprintf(stderr,"img2rsrc:  option -%s requires argument\n",option);
			return B_ERROR;
		}
		if (!strcmp(option,"mf"))
		{
			micnBmap = bitmap_from_filename(argv[*index]);
			if (!micnBmap)
				return B_ERROR;
		}
		else if (!strcmp(option,"if"))
		{
			iconBmap = bitmap_from_filename(argv[*index]);
			if (!iconBmap)
				return B_ERROR;
		}
		else
		{
			fprintf(stderr,"img2rsrc:  unknown option -%s\n",option);
			return B_ERROR;
		}
		return B_OK;
	}
}

BBitmap *bitmap_from_filename(char *name)
{
	BBitmap *bmap = GetBitmap(name);
	if (!bmap)
	{
		fprintf(stderr,"img2rsrc:  could not read image from file \"%s\"\n",name);
		return NULL;
	}
	else
		return bmap;
}

void usage(void)
{
	fprintf(stderr,"img2rsrc, version 1.0 by E. Tomlinson\n");
	fprintf(stderr,"Usage:  img2rsrc [flags] <output file>\n");
	fprintf(stderr,"\t-?\tthis help message\n");
	fprintf(stderr,"\t-mf\timage file for MICN resource\n");
	fprintf(stderr,"\t-if\timage file for ICON resource\n");
	fprintf(stderr,"\t-a\tadd resources to existing file\n");
	fprintf(stderr,"\t-f\tforce:\n");
	fprintf(stderr,"\t\tif -a specified, force overwrite of existing resources\n");
	fprintf(stderr,"\t\totherwise, force clobbering of existing file.\n");
}
