MODULE POPM;  (* RC 6.3.89 / 19.10.92, mmb 4.3.91 / 30.10.92 *)
(* Machine dependent constants needed before code generation *)
(* Host interface, IBM RS/6000 version *)
(* modifications HM: *)
(* 94-05-09 MaxPtr and MaxGPtr smaller *)
(* 94-05-24 Sysflag 1 for records => 68K alignment in records (MaxSysFlag = 1 instead of 0) *)

  IMPORT
    Out, Time, Files, br:=BinaryRider, OakFiles, SYSTEM;

  CONST (* IBM RS/6000 *)
    
    (* basic type sizes *)
    ByteSize* = 1;  (* SYSTEM.BYTE *)
    CharSize* = 1;  (* CHAR *)
    BoolSize* = 1;  (* BOOLEAN *)
    SetSize* = 4;  (* SET *)
    SIntSize* = 1;  (* SHORTINT *)
    IntSize* = 2;  (* INTEGER *)
    LIntSize* = 4;  (* LONGINT *)
    RealSize* = 4;  (* REAL *)
    LRealSize* = 8;  (* LONGREAL *)
    ProcSize* = 8;  (* PROCEDURE type *)
    PointerSize* = 4;  (* POINTER type *)

    (* value of constant NIL *)
    nilval* = 0;
    
    (* target machine minimum values of basic types expressed in host machine format: *)
    MinSInt* = -80H;
    MinInt* = -8000H;
    MinLInt* =  80000000H;  (*-2147483648*)
    MinRealPat = 0FF7FFFFFH;  (* most  negative, 32-bit pattern *)
    MinLRealPatL = 0FFEFFFFFH;  (* most  negative, lower 32-bit pattern *)
    MinLRealPatH = 0FFFFFFFFH;  (* most  negative, higher 32-bit pattern *)
    
    (* target machine maximum values of basic types expressed in host machine format: *)
    MaxSInt* = 7FH;
    MaxInt* = 7FFFH;
    MaxLInt* = 7FFFFFFFH;  (*2147483647*)
    MaxSet* = 31;  (* must be >= 15, else the bootstraped compiler cannot run (IN-tests) *)
    MaxRealPat = 7F7FFFFFH;  (* most positive, 32-bit pattern *)
    MaxLRealPatL = 7FEFFFFFH;  (* most positive, lower 32-bit pattern *)
    MaxLRealPatH = 0FFFFFFFFH;    (* most positive, higher 32-bit pattern *)
    
    (* maximal index value for array declaration: *)
    MaxIndex* = MaxLInt;
    
    (* parametrization of numeric scanner: *)
    MaxHDig* = 8;  (* maximal hexadecimal longint length *)
    MaxRExp* = 38;  (* maximal real exponent *)
    MaxLExp* = 308;  (* maximal longreal exponent *)
    
    (* inclusive range of parameter of standard procedure HALT: *)
    MinHaltNr* = 20;
    MaxHaltNr* = 255;
    
    (* inclusive range of register number of procedures SYSTEM.GETREG and SYSTEM.PUTREG: *)
    MinRegNr* = 0;
    MaxRegNr* = 66;  (* 0..31: Rx or FPRx, depending on second operand, 32..66: control registers *)
    (* encoding: code = 32+reg
      MQ = 0; XER = 1; fromRTCU = 4; fromRTCL = 5; fromDEC = 6; LR = 8; CTR = 9;
      CR = 32; MSR = 33; FPSCR = 34;
      others are privileged
    *)
    
    (* maximal value of flag used to mark interface structures: *)
    MaxSysFlag* = 1;  (* IBM RS/6000: only 0 is valid, not used *)
    
    (* maximal condition value of parameter of SYSTEM.CC: *)
    MaxCC* = -1;  (* IBM RS/6000: not used *)
    
    (* initialization of linkadr field in ObjDesc, must be different from any valid link address: *)
    LANotAlloc* = -1;
    
    (* initialization of constant address, must be different from any valid constant address: *)
    ConstNotAlloc* = -1;  (* IBM RS/6000: only strings are allocated *)
    
    (* initialization of tdadr field in StrDesc, must be different from any valid address: *)
    TDAdrUndef* = -1;
    
    (* maximal number of cases in a case statement: *)
    MaxCases* = 128;
    
    (* maximal range of a case statement (higher label - lower label ~ jump table size): *)
    MaxCaseRange* = 512;
    
    (* maximal number of exit statements within a (nested) loop statement: *)
    MaxExit* = 16;
        
    (* whether hidden pointer fields have to be nevertheless exported: *)
    ExpHdPtrFld* = TRUE;
    HdPtrName* = "@ptr";

    (* whether hidden procedure fields have to be nevertheless exported (may be used for System.Free): *)
    ExpHdProcFld* = FALSE;
    HdProcName* = "@proc";
    
    (* whether hidden bound procedures have to be nevertheless exported: *)
    ExpHdTProc* = FALSE;
    HdTProcName* = "@tproc";

    (* maximal number of hidden fields in an exported record: *)
    MaxHdFld* = 512;

    (* whether addresses of formal parameters are exported: *)
    ExpParAdr* = TRUE;

    (* whether addresses or entry numbers are exported for global variables: *)
    ExpVarAdr* = TRUE;

    (* maximal number of exported stuctures: *)
    MaxStruct* = 255;  (* must be < 256 *)
    
    (* maximal number of pointer fields in a record: *)
    MaxPtr* = (*16384*) 1024;

    (* maximal number of global pointers: *)
    MaxGPtr* = (*16384*) 1024;
    
    (* whether field leaf of pointer variable p has to be set to FALSE, when NEW(p) or SYSTEM.NEW(p, n) is used: *)
    NEWusingAdr* = FALSE;
    
    (* special character (< " ") returned by procedure Get, if end of text reached *)
    Eot* = 0X;
    
    (* version flag *)
    CeresVersion* = FALSE;
    
  VAR
    MinReal*, MaxReal*: REAL;
    MinLReal*, MaxLReal*: LONGREAL;
    noerr*: BOOLEAN;  (* no error found until now *)
    curpos*, errpos*: LONGINT;  (* character and error position in source file *)
    breakpc*: LONGINT;  (* set by OPV.Init *)
    
  CONST
    SFext  = ".Sym";
    TSFext = ".TSym"; (* temporary new symbol file *)
    SFtag  = 0F7X;    (* symbol file tag *)
    OFext  = ".Obj";
    OFtag  = 0F8X;    (* object file tag *)

  TYPE
    FileName = ARRAY 32 OF CHAR;
    
  VAR
    LRealPat: RECORD L, H: LONGINT END;
    lastpos, pat: LONGINT;  (* last position error in source file *)
    oldSF, inR: br.Reader;
    ObjF, newSF, RefF: br.Writer;
    oldSFile, newSFile, ObjFile, RefFile: Files.File;
    now301: BOOLEAN;
    
  PROCEDURE FlipBits* (i: LONGINT): LONGINT;
    VAR s, d: SET;
  BEGIN
    IF CeresVersion THEN
      s := SYSTEM.VAL(SET, i); d := {}; i := 0;
      WHILE i < 32 DO IF i IN s THEN INCL(d, 31-i) END; INC(i) END;
      RETURN SYSTEM.VAL(LONGINT, d)
    ELSE
      RETURN i
    END
  END FlipBits;
   
  PROCEDURE Init* (source: br.Reader);
  BEGIN inR := source;  (* log is via Out module -- opened automatically *)
    noerr := TRUE; curpos := inR.Pos(); errpos := curpos; lastpos := curpos-10; now301 := FALSE
  END Init;
  
  PROCEDURE Get* (VAR ch: CHAR);  (* read next character from source text, Eot if no more *)
  BEGIN inR.ReadChar(ch); INC(curpos)
  END Get;
  
  PROCEDURE NewKey* (): LONGINT;
    VAR time: Time.TimeStamp;
  BEGIN Time.GetTime(time); RETURN time.msecs
  END NewKey;
  
  PROCEDURE MakeFileName (VAR name, FName: ARRAY OF CHAR; ext: ARRAY OF CHAR);
    VAR i, j: INTEGER; ch: CHAR;
  BEGIN i := 0;
    LOOP ch := name[i];
      IF ch = 0X THEN EXIT END ;
      FName[i] := ch; INC(i);
    END ;
    j := 0;
    REPEAT ch := ext[j]; FName[i] := ch; INC(i); INC(j)
    UNTIL ch = 0X
  END MakeFileName;
  
  (* ------------------------- Log Output ------------------------- *)
  
  PROCEDURE LogW* (ch: CHAR);
  BEGIN
    Out.Char(ch)
  END LogW;
  
  PROCEDURE LogWStr* (s: ARRAY OF CHAR);
  BEGIN
    Out.String(s)
  END LogWStr;
  
  PROCEDURE LogWNum* (i, len: LONGINT);
  BEGIN
    Out.LongInt(i, len)
  END LogWNum;

  PROCEDURE LogWHex (i: LONGINT);
  BEGIN
    Out.Hex(i, 8); Out.Char("H")
  END LogWHex;

  PROCEDURE LogWLn*;
  BEGIN
    Out.Ln
  END LogWLn;
  
  PROCEDURE Mark* (n: INTEGER; pos: LONGINT);
  BEGIN
    IF n >= 0 THEN
      noerr := FALSE;
      IF (pos < lastpos) OR (lastpos + 9 < pos) THEN lastpos := pos;
        LogWLn; LogWStr("  pos"); LogWNum(pos, 6);
        IF n = 255 THEN LogWStr("  pc "); LogWHex(breakpc)
        ELSIF n = 254 THEN LogWStr("  pc not found")
        ELSE LogWStr("  err"); LogWNum(n, 4)
        END
      END
    ELSE
      LogWLn; LogWStr("  pos"); LogWNum(pos, 6); LogWStr("  warning"); LogWNum(-n, 4)
    END
  END Mark;
  
  PROCEDURE err* (n: INTEGER);
  BEGIN
    IF n = -10000 THEN now301 := TRUE; RETURN END;
    IF (n = -301) & now301 THEN RETURN END;
    Mark(n, errpos)
  END err;

  (* ------------------------- Read Symbol File ------------------------- *)
  
  PROCEDURE SymRCh* (VAR b: CHAR);
  BEGIN oldSF.ReadChar(b)
  END SymRCh;

  PROCEDURE SymRTag* (VAR k: INTEGER);
    VAR i: LONGINT;
  BEGIN oldSF.ReadNum(i); k := SHORT(i)
  END SymRTag;

  PROCEDURE SymRInt* (VAR k: LONGINT);
  BEGIN oldSF.ReadNum(k)
  END SymRInt;
  
  PROCEDURE SymRLInt* (VAR k: LONGINT);
  BEGIN oldSF.ReadNum(k)
  END SymRLInt;
  
  PROCEDURE SymRSet* (VAR s: SET);
    VAR j: LONGINT;
  BEGIN oldSF.ReadNum(j);
    IF CeresVersion THEN j := FlipBits(j) END;
    s := SYSTEM.VAL(SET, j)
  END SymRSet;

  PROCEDURE SymRReal* (VAR r: REAL);
  BEGIN oldSF.ReadReal(r)
  END SymRReal;
  
  PROCEDURE SymRLReal* (VAR lr: LONGREAL);
  BEGIN oldSF.ReadLReal(lr)
  END SymRLReal;
  
  PROCEDURE CloseOldSym*;
  (* called only if OldSym previously returned done = TRUE *)
  END CloseOldSym;

  PROCEDURE OldSym* (VAR modName: ARRAY OF CHAR; self: BOOLEAN; VAR done: BOOLEAN);
  (* open file in read mode *)
    VAR ch: CHAR; fileName: FileName; res: INTEGER;
  BEGIN MakeFileName(modName, fileName, SFext);
    oldSFile := Files.Old(fileName, {Files.read}, res); done := res=Files.done;
    IF done THEN
      oldSF:=br.ConnectReader(oldSFile); SymRCh(ch);
      IF ch # SFtag THEN err(151);  (*not a symbol file*)
        CloseOldSym; done := FALSE
      END
    ELSIF ~self THEN err(152)   (*sym file not found*)
    END
  END OldSym;
  
  PROCEDURE eofSF* (): BOOLEAN;
  (* = TRUE if end of old file reached *)
  BEGIN RETURN oldSF.Res()#br.done
  END eofSF;
  
  (* ------------------------- Write Symbol File ------------------------- *)
  
  PROCEDURE SymWCh* (ch: CHAR);
  BEGIN newSF.WriteChar(ch)
  END SymWCh;

  PROCEDURE SymWTag* (k: INTEGER);
  BEGIN newSF.WriteNum(k)
  END SymWTag;

  PROCEDURE SymWInt* (i: LONGINT);
  BEGIN newSF.WriteNum(i)
  END SymWInt;

  PROCEDURE SymWLInt* (k: LONGINT);
  BEGIN newSF.WriteNum(k)
  END SymWLInt;

  PROCEDURE SymWSet* (s: SET);
  BEGIN
    IF CeresVersion THEN
      newSF.WriteNum(FlipBits(SYSTEM.VAL(LONGINT, s)))
    ELSE
      newSF.WriteNum(SYSTEM.VAL(LONGINT, s))
    END
  END SymWSet;

  PROCEDURE SymWReal* (r: REAL);
  BEGIN newSF.WriteReal(r)
  END SymWReal;
  
  PROCEDURE SymWLReal* (lr: LONGREAL);
  BEGIN newSF.WriteLReal(lr)
  END SymWLReal;
  
  PROCEDURE RegisterNewSym* (VAR modName: ARRAY OF CHAR);
  (* delete possibly already existing file with same name, register new created file *)
  VAR tempName, fileName: FileName; res: INTEGER;
  BEGIN 
    newSFile.Register();
    
    (* we need to rename temporary file with symbol extension *)
    MakeFileName(modName, fileName, SFext);
    MakeFileName(modName, tempName, TSFext);
    OakFiles.Rename(tempName, fileName, res);
  END RegisterNewSym;
  
  PROCEDURE DeleteNewSym* (VAR modName: ARRAY OF CHAR);
  (* delete new created file, don't touch possibly already existing file with same name *)
  VAR
    res: INTEGER; fileName: FileName;
  BEGIN
    MakeFileName(modName, fileName, TSFext);
    OakFiles.Delete(fileName, res)
  END DeleteNewSym;

  PROCEDURE NewSym* (VAR modName: ARRAY OF CHAR; VAR done: BOOLEAN);
  (* open new file in write mode, don't touch possibly already existing file with same name *)
    VAR fileName: FileName; res: INTEGER;
  BEGIN MakeFileName(modName, fileName, TSFext);
    newSFile := Files.New(fileName, {Files.write, Files.read}, res); 
    done := newSFile # NIL;
    IF done THEN newSF:=br.ConnectWriter(newSFile);
      SymWCh(SFtag)
    ELSE err(153)
    END
  END NewSym;

  PROCEDURE EqualSym* (VAR oldkey: LONGINT): BOOLEAN;
  (* compare old and new Symbol File, close old file, return TRUE if equal *)
    VAR ch0, ch1: CHAR; equal: BOOLEAN; newkey: LONGINT;
    newSF: br.Reader;
  BEGIN
    oldSF:=br.ConnectReader(oldSFile);
    oldSF.SetPos(2); oldSF.ReadNum(oldkey);
    newSF:=br.ConnectReader(newSFile);
    newSF.SetPos(2); newSF.ReadNum(newkey);
    REPEAT oldSF.ReadChar(ch0); newSF.ReadChar(ch1)
    UNTIL (ch0 # ch1) OR (newSF.Res()#br.done);
    equal := (oldSF.Res()#br.done) & (newSF.Res()#br.done); 
    CloseOldSym;
    RETURN equal
  END EqualSym;

  (* ------------------------- Write Reference & Object Files ------------------------- *)

  PROCEDURE RefW* (ch: CHAR);
  BEGIN RefF.WriteChar(ch)
  END RefW;

  PROCEDURE RefWNum* (i: LONGINT);
  BEGIN RefF.WriteNum(i)
  END RefWNum;

  PROCEDURE RefWBytes* (VAR bytes: ARRAY OF SYSTEM.BYTE; n: LONGINT);  (* MK *)
  BEGIN RefF.WriteBytes(bytes, 0, n)
  END RefWBytes;

  PROCEDURE RefPos* (): LONGINT;     (* MK *)
  BEGIN RETURN RefF.Pos()
  END RefPos;
  
  PROCEDURE ObjW* (ch: CHAR);
  BEGIN ObjF.WriteChar(ch)
  END ObjW;

  PROCEDURE ObjWInt* (i: INTEGER);
  BEGIN
    ObjF.WriteInt(i)
  END ObjWInt;

  PROCEDURE ObjWLInt* (i: LONGINT);
  BEGIN
    ObjF.WriteLInt(i)
  END ObjWLInt;

  PROCEDURE ObjWBytes* (VAR bytes: ARRAY OF SYSTEM.BYTE; n: LONGINT);
  BEGIN ObjF.WriteBytes(bytes, 0, n)
  END ObjWBytes;

  PROCEDURE OpenRefObj* (VAR modName: ARRAY OF CHAR);
    VAR FName: ARRAY 32 OF CHAR; res: INTEGER;
  BEGIN
    RefFile := Files.Tmp("", {Files.write, Files.read}, res); 
    RefF:=br.ConnectWriter(RefFile);
    MakeFileName(modName, FName, OFext);
    ObjFile := Files.New(FName, {Files.write}, res);
    IF ObjFile # NIL THEN
      ObjF:=br.ConnectWriter(ObjFile);
      ObjW(OFtag); ObjW("6"); ObjWInt(0); ObjWInt(0);
    ELSE err(153)
    END
  END OpenRefObj;

  PROCEDURE CloseRefObj*;
    VAR refsize: LONGINT; ch: CHAR; ref: br.Reader;
  BEGIN (*ref block*)
    refsize := 0; ObjW(8BX);   (* MG 28.2.96 Files.Length didn't work? *)
    ref:=br.ConnectReader(RefFile); ref.ReadChar(ch);
    WHILE ref.Res()=br.done DO INC(refsize); ObjW(ch); ref.ReadChar(ch) END;
    ObjF.SetPos(2); ObjWLInt(refsize); (*ObjWBytes(refsize, 4);*)
    ObjFile.Register()
  END CloseRefObj;

BEGIN
  pat := MinRealPat; SYSTEM.MOVE(SYSTEM.ADR(pat), SYSTEM.ADR(MinReal), 4);  (*-3.40282346E38*)
  pat := MaxRealPat; SYSTEM.MOVE(SYSTEM.ADR(pat), SYSTEM.ADR(MaxReal), 4);  (*3.40282346E38*)
  LRealPat.L := MinLRealPatL; LRealPat.H := MinLRealPatH;
  SYSTEM.MOVE(SYSTEM.ADR(LRealPat), SYSTEM.ADR(MinLReal), 8);  (*-1.7976931348623157D308*)
  LRealPat.L := MaxLRealPatL; LRealPat.H := MaxLRealPatH;
  SYSTEM.MOVE(SYSTEM.ADR(LRealPat), SYSTEM.ADR(MaxLReal), 8)   (*1.7976931348623157D308*)
END POPM.
