MODULE Decoder;  (* mmb 25.12.90 / 31.5.94 *)(* MAH 25 Jul 94  Byte ordering changed *)
  (* MK 24 0ct 95 extended reference information *)
  (* MG 2 Aug 98 ported to OOC *) 

  IMPORT
    Files, SYSTEM, Args:=ProgramArgs, tr:=TextRider, br:=BinaryRider, Out;

  CONST
    (* formats *)
    LIxAAxLK = 0; BOxBIxBDxAAxLK = 1; BOxBIxLK = 2; LKu = 3; Rcu = 4; None = 6; BFxBFAxLKu = 7; BTxBAxBBxLKu = 8;
    RTxDxRA = 9; RSxDxRA = 11; RSxRAxRBxRcu = 12; RTxRAxNBxRcu = 13; RSxRAxNBxRcu = 14;
    RTxRAxUI = 15; RTxRAxSI = 16; RTxRAxRBxRc = 17; RTxRAxRBxRcu = 18; RTxRAxRc = 19; RTxRAxRcu = 20; 
    BFxRAxSI = 21; BFxRAxUI = 22; BFxRAxRBxRcu = 23; TOxRAxSI = 24; TOxRAxRBxRcu = 25; RAxRSxUI = 26; 
    RAxRSxRBxRc = 27; RAxRSxRc = 28; RAxRSxSHxMBxMExRc = 29; RAxRSxRBxMBxMExRc = 30; RAxRSxSHxRc = 31;
    SPRxRSxRcu = 32; RTxSPRxRcu = 33; FXMxRSxRcu = 34; BFxRcu = 35; RTxRcu = 36; RSxRcu = 37; FRTxDxRA = 38;
    FRTxRAxRBxRcu = 39; FRSxDxRA = 40; FRSxRAxRBxRcu = 41; FRTxFRBxRc = 42; FRTxFRAxFRBxRc = 43;
    FRTxFRAxFRCxRc = 44; FRTxFRAxFRCxFRBxRc = 45; BFxFRAxFRBxRcu = 46; FRTxRc = 47; BFxBFAxRcu = 48;
    FLMxFRBxRc = 49; BFxIxRc = 50; BTxRc = 51; RAxRBxRc = 52; RAxRBxRcu = 53; SRxRSxRcu = 54; RTxSRxRcu = 55;
    RTxRA = 56; RTxSI = 57;
    (* misc *)
    tabs = "  ";   (* space equivalent for some tabs *)
    Pointer = 13; ProcTyp = 14;
    
  VAR
    curInstr, pc: LONGINT;

  PROCEDURE ReadInt (VAR r: br.Reader; VAR x: INTEGER);
  BEGIN
    r.ReadInt(x)
  END ReadInt;

  PROCEDURE ReadLInt (VAR r: br.Reader; VAR x: LONGINT);
  BEGIN
    r.ReadLInt(x)
  END ReadLInt;

  PROCEDURE Instr (mnemonic: ARRAY OF CHAR; fields: SHORTINT);

    PROCEDURE WriteCB (i: LONGINT);
    BEGIN
      Out.String("cf"); Out.LongInt(i DIV 4, 0); 
      CASE i MOD 4 OF 
        0: Out.String(".lt")
      |  1: Out.String(".gt")
      |  2: Out.String(".eq")
      |  3: Out.String(".so")
      END
    END WriteCB;

    PROCEDURE AA;
    BEGIN IF 30 IN SYSTEM.VAL(SET, curInstr) THEN Out.Char("a") END
    END AA;
  
    PROCEDURE BA;
    BEGIN WriteCB(curInstr DIV 65536 MOD 32)
    END BA;
  
    PROCEDURE BB;
    BEGIN WriteCB(curInstr DIV 2048 MOD 32)
    END BB;
  
    PROCEDURE BD;
    BEGIN Out.Hex(ASH(SYSTEM.LSH(curInstr MOD 65536 DIV 4, 18), -16)+pc, 8); Out.Char("H")
    END BD;
  
    PROCEDURE BF;
    BEGIN Out.String("cf"); Out.LongInt(curInstr DIV 8388608 MOD 8, 0)
    END BF;
  
    PROCEDURE BFA;
    BEGIN Out.String("cf"); Out.LongInt(curInstr DIV 262144 MOD 8, 0)
    END BFA;
  
    PROCEDURE BI;
    BEGIN WriteCB(curInstr DIV 65536 MOD 32)
    END BI;
  
    PROCEDURE BO;
    BEGIN
      CASE curInstr DIV 2097152 MOD 32 OF
        0, 1: Out.String("(--CTR#0)&~CB")
      |  2, 3: Out.String("(--CTR=0)&~CB")
      |  4..7: Out.String("~CB")
      |  8, 9: Out.String("(--CTR#0)&~CB")
      |  10, 11: Out.String("(--CTR=0)&~CB")
      |  12..15: Out.String("CB")
      |  16, 17, 24, 25: Out.String("--CTR#0")
      |  18, 19, 26, 27: Out.String("--CTR=0")
      |  20..23, 28..31: Out.String("ALWAYS")
      END
    END BO;
  
    PROCEDURE BT;
    BEGIN WriteCB(curInstr DIV 2097152 MOD 32)
    END BT;
  
    PROCEDURE D;
    BEGIN Out.LongInt(ASH(SYSTEM.LSH(curInstr MOD 65536, 16), -16), 0)
    END D;
  
    PROCEDURE FXM;
      VAR s: SET; i: INTEGER; first: BOOLEAN;
    BEGIN s := SYSTEM.VAL(SET, curInstr DIV 4096 MOD 256); first := TRUE; i := 24; Out.String("cf[");
      WHILE i < 32 DO
        IF i IN s THEN
          IF ~first THEN Out.Char(",") ELSE first := FALSE END; 
          Out.Int(i-24, 0)
        END;
        INC(i)
      END;
      Out.Char("]")
    END FXM;
  
    PROCEDURE FLM;    
      VAR s: SET; i: INTEGER; first: BOOLEAN;
    BEGIN s := SYSTEM.VAL(SET, curInstr DIV 131072 MOD 256); first := TRUE; i := 24; Out.String("fpscr[");
      WHILE i < 32 DO
        IF i IN s THEN
          IF ~first THEN Out.Char(",") ELSE first := FALSE END; 
          Out.Int(i-24, 0)
        END;
        INC(i)
      END;
      Out.Char("]")
    END FLM;
  
    PROCEDURE FRA;
    BEGIN Out.Char("f"); Out.LongInt(curInstr DIV 65536 MOD 32, 0)
    END FRA;
  
    PROCEDURE FRB;
    BEGIN Out.Char("f"); Out.LongInt(curInstr DIV 2048 MOD 32, 0)
    END FRB;
  
    PROCEDURE FRC;
    BEGIN Out.Char("f"); Out.LongInt(curInstr DIV 64 MOD 32, 0)
    END FRC;
  
    PROCEDURE FRS;
    BEGIN Out.Char("f"); Out.LongInt(curInstr DIV 2097152 MOD 32, 0)
    END FRS;
  
    PROCEDURE FRT;
    BEGIN Out.Char("f"); Out.LongInt(curInstr DIV 2097152 MOD 32, 0)
    END FRT;
  
    PROCEDURE I;
      VAR s: SET; i: INTEGER;
    BEGIN
      s := SYSTEM.VAL(SET, curInstr DIV 4096 MOD 16); i := 28;
      WHILE i < 32 DO 
        IF i IN s THEN Out.Char("1") ELSE Out.Char("0") END; 
        INC(i) 
      END;
      Out.Char("B")
    END I;
  
    PROCEDURE LI;
    BEGIN
      Out.Hex(pc+ASH(SYSTEM.LSH(curInstr MOD 67108864 DIV 4, 8), -6), 8); Out.Char("H");
      Out.String(" ["); Out.Hex(-ASH(SYSTEM.LSH(curInstr MOD 67108864 DIV 4, 8), -6), 8);
      Out.String("H]")
    END LI;
  
    PROCEDURE LK;
    BEGIN IF curInstr MOD 2 = 1 THEN Out.Char("l") END
    END LK;

    PROCEDURE LKU;
    BEGIN IF curInstr MOD 2 = 1 THEN Out.Char(9X); Out.String("# invalidate lr") END
    END LKU;
  
    PROCEDURE MB;
    BEGIN Out.LongInt(curInstr DIV 64 MOD 32, 0)
    END MB;
  
    PROCEDURE ME;
    BEGIN Out.LongInt(curInstr DIV 2 MOD 32, 0)
    END ME;
  
    PROCEDURE NB;
    BEGIN Out.LongInt(curInstr DIV 2048 MOD 32, 0)
    END NB;
   
    PROCEDURE RA;
    BEGIN Out.Char("r"); Out.LongInt(curInstr DIV 65536 MOD 32, 0)
    END RA;
  
    PROCEDURE RB;
    BEGIN Out.Char("r"); Out.LongInt(curInstr DIV 2048 MOD 32, 0)
    END RB;

    PROCEDURE RC;
    BEGIN IF curInstr MOD 2 = 1 THEN Out.Char(".") END
    END RC;

    PROCEDURE RCU;
    BEGIN IF curInstr MOD 2 = 1 THEN Out.Char(9X); Out.String("# invalidate CR") END
    END RCU;
  
    PROCEDURE RS;
    BEGIN Out.Char("r"); Out.LongInt(curInstr DIV 2097152 MOD 32, 0)
    END RS;
  
    PROCEDURE RT;
    BEGIN Out.Char("r"); Out.LongInt(curInstr DIV 2097152 MOD 32, 0)
    END RT;
  
    PROCEDURE SH;
    BEGIN Out.LongInt(curInstr DIV 2048 MOD 32, 0)
    END SH;

    PROCEDURE SI;
    BEGIN Out.LongInt(ASH(SYSTEM.LSH(curInstr, 16), -16), 0)
    END SI;
  
    PROCEDURE SPR;
    BEGIN
      CASE curInstr DIV 65536 MOD 32 OF
        0: Out.String("mq")
      |  1: Out.String("xer")
      |  4: Out.String("to rtcu")
      |  5: Out.String("to rtcl")
      |  6: Out.String("to dec")
      |  8: Out.String("lr")
      |  9: Out.String("ctr")
      |  17: Out.String("tid")
      |  18: Out.String("dsisr")
      |  19: Out.String("dar")
      |  20: Out.String("to rtcu")
      |  21: Out.String("to rtcl")
      |  22: Out.String("to dec")
      |  24: Out.String("sdr0")
      |  25: Out.String("sdr1")
      |  26: Out.String("srr0")
      |  27: Out.String("srr1")
      |  2, 3, 7, 10..16, 23, 28..31: Out.String("INVALID")
      END
    END SPR;
  
    PROCEDURE SR;
    BEGIN Out.Char("s"); Out.LongInt(curInstr DIV 65536 MOD 16, 0)
    END SR;
  
    PROCEDURE TOf;
    BEGIN
      CASE curInstr DIV 2097152 MOD 32 OF
        0: Out.String("NEVER")
      |  1: Out.String(">u")
      |  2: Out.String("<u")
      |  3, 11, 19, 24, 25, 26, 27: Out.Char("#")
      |  4: Out.Char("=")
      |  5: Out.String(">=u")
      |  6: Out.String("<=u")
      |  7, 15, 23, 28, 29, 30, 31: Out.String("ALWAYS")
      |  8: Out.String(">s")
      |  9: Out.String(">s OR >u")
      |  10: Out.String(">s OR <u")
      |  12: Out.String(">=s")
      |  13: Out.String(">=s OR >u")
      |  14: Out.String(">=s OR <u")
      |  16: Out.String("<s")
      |  17: Out.String("<s OR >u")
      |  18: Out.String("<s OR <u")
      |  20: Out.String("<=s")
      |  21: Out.String("<=s OR >u")
      |  22: Out.String("<=s OR <u")
      END
    END TOf;
  
    PROCEDURE UI;
    BEGIN Out.Hex(curInstr MOD 65536, 8); Out.Char("H")
    END UI;

    PROCEDURE tab;
    BEGIN Out.Char(9X)
    END tab;

    PROCEDURE comma;
    BEGIN Out.String(", ")
    END comma;

    PROCEDURE op;
    BEGIN Out.Char("(")
    END op;

    PROCEDURE cp;
    BEGIN Out.Char(")")
    END cp;
  
  BEGIN
    Out.String(mnemonic);
    CASE fields OF
      LIxAAxLK: LK; AA; tab; LI
    |  BOxBIxBDxAAxLK: LK; AA; tab; BO; comma; BI; comma; BD
    |  BOxBIxLK: LK; tab; BO; comma; BI
    |  LKu: tab; LKU
    |  Rcu: tab; LKU
    |  None:
    |  BFxBFAxLKu: tab; BF; comma; BFA; LKU
    |  BTxBAxBBxLKu: tab; BT; comma; BA; comma; BB; LKU
    |  RTxDxRA: tab; RT; comma; D; op; RA; cp
    |  RSxDxRA: tab; RS; comma; D; op; RA; cp
    |  RSxRAxRBxRcu: tab; RS; comma; RA; comma; RB; RCU
    |  RTxRAxNBxRcu: tab; RT; comma; RA; comma; NB; RCU
    |  RSxRAxNBxRcu: tab; RS; comma; RA; comma; NB; RCU
    |  RTxRAxUI: tab; RT; comma; RA; comma; UI
    |  RTxRAxSI: tab; RT; comma; RA; comma; SI
    |  RTxRAxRBxRc: RC; tab; RT; comma; RA; comma; RB
    |  RTxRAxRBxRcu: tab; RT; comma; RA; comma; RB; RCU
    |  RTxRAxRc: RC; tab; RT; comma; RA
    |  RTxRAxRcu: tab; RT; comma; RA; RCU
    |  BFxRAxSI: tab; BF; comma; RA; comma; SI
    |  BFxRAxUI: tab; BF; comma; RA; comma; UI
    |  BFxRAxRBxRcu: tab; BF; comma; RA; comma; RB; RCU
    |  TOxRAxSI: tab; TOf; comma; RA; comma; SI
    |  TOxRAxRBxRcu: tab; TOf; comma; RA; comma; RB; RCU
    |  RAxRSxUI: tab; RA; comma; RS; comma; UI
    |  RAxRSxRBxRc: RC; tab; RA; comma; RS; comma; RB
    |  RAxRSxRc: RC; tab; RA; comma; RS
    |  RAxRSxSHxMBxMExRc: RC; tab; RA; comma; RS; comma; SH; comma; MB; comma; ME
    |  RAxRSxRBxMBxMExRc: RC; tab; RA; comma; RS; comma; RB; comma; MB; comma; ME
    |  RAxRSxSHxRc: RC; tab; RA; comma; RS; comma; SH
    |  SPRxRSxRcu: tab; SPR; comma; RS; RCU
    |  RTxSPRxRcu: tab; RT; comma; SPR; RCU
    |  FXMxRSxRcu: tab; FXM; comma; RS; RCU
    |  BFxRcu: tab; BF; RCU
    |  RTxRcu: tab; RT; RCU
    |  RSxRcu: tab; RS; RCU
    |  FRTxDxRA: tab; FRT; comma; D; op; RA; cp
    |  FRTxRAxRBxRcu: tab; FRT; comma; RA; comma; RB; RCU
    |  FRSxDxRA: tab; FRS; comma; D; op; RA; cp
    |  FRSxRAxRBxRcu: tab; FRS; comma; RA; comma; RB; RCU
    |  FRTxFRBxRc: RC; tab; FRT; comma; FRB
    |  FRTxFRAxFRBxRc: RC; tab; FRT; comma; FRA; comma; FRB
    |  FRTxFRAxFRCxRc: RC; tab; FRT; comma; FRA; comma; FRC
    |  FRTxFRAxFRCxFRBxRc: RC; tab; FRT; comma; FRA; comma; FRC; comma; FRB
    |  BFxFRAxFRBxRcu: tab; BF; comma; FRA; comma; FRB; RCU
    |  FRTxRc: RC; tab; FRT
    |  BFxBFAxRcu: tab; BF; comma; BFA; RCU
    |  FLMxFRBxRc: RC; tab; FLM; comma; FRB
    |  BFxIxRc: RC; tab; BF; comma; I
    |  BTxRc: RC; tab; BT
    |  RAxRBxRc: RC; tab; RA; comma; RB
    |  RAxRBxRcu: tab; RA; comma; RB; RCU
    |  SRxRSxRcu: tab; SR; comma; RS; RCU
    |  RTxSRxRcu: tab; RT; comma; SR; RCU
    |  RTxRA: tab; RT; comma; RA
    |  RTxSI: tab; RT; comma; SI
    END;
  END Instr;

  PROCEDURE error;
  BEGIN Out.String("invalid instruction "); Out.Hex(curInstr, 8); Out.Char("H"); Out.Ln
  END error;

  PROCEDURE DecodeXL (instr: LONGINT);
    VAR extOp: LONGINT;
  BEGIN
    extOp := ASH(instr MOD 2048, -1);
    IF extOp <= 150 THEN
      IF extOp <= 50 THEN
        IF extOp <= 16 THEN
          IF extOp = 0 THEN Instr("mcrf", BFxBFAxLKu) ELSIF extOp = 16 THEN Instr("bclr", BOxBIxLK) 
          ELSE error END
        ELSE
          IF extOp = 33 THEN Instr("crnor", BTxBAxBBxLKu) ELSIF extOp = 50 THEN Instr("rfi", LKu) 
          ELSE error END
        END
      ELSE
        IF extOp <= 129 THEN
          IF extOp = 82 THEN Instr("rfsvc", LKu) ELSIF extOp = 129 THEN Instr("crandc", BTxBAxBBxLKu) 
          ELSE error END
        ELSE
          IF extOp = 150 THEN Instr("isync", LKu) ELSE error END
        END
      END
    ELSE
      IF extOp <= 289 THEN
        IF extOp <= 225 THEN
          IF extOp = 193 THEN Instr("crxor", BTxBAxBBxLKu) ELSIF extOp = 225 THEN Instr("crnand", BTxBAxBBxLKu) 
          ELSE error END
        ELSE
          IF extOp = 257 THEN Instr("crand", BTxBAxBBxLKu) ELSIF extOp = 289 THEN Instr("creqv", BTxBAxBBxLKu) 
          ELSE error END
        END
      ELSE
        IF extOp <= 449 THEN
          IF extOp = 417 THEN Instr("crorc", BTxBAxBBxLKu) ELSIF extOp = 449 THEN Instr("cror", BTxBAxBBxLKu) 
          ELSE error END
        ELSE
          IF extOp = 528 THEN Instr("bcctr", BOxBIxLK) ELSE error END
        END
      END
    END
  END DecodeXL;

  PROCEDURE DecodeX (instr: LONGINT);
    VAR extOp: LONGINT;
  BEGIN
    extOp := ASH(instr MOD 2048, -1);
    IF extOp <= 467 THEN
      IF extOp <= 200 THEN
        IF extOp <= 87 THEN
          IF extOp <= 26 THEN
            IF extOp <= 10 THEN
              IF extOp <= 4 THEN
                IF extOp = 0 THEN Instr("cmp", BFxRAxRBxRcu) 
                ELSIF extOp = 4 THEN Instr("tw", TOxRAxRBxRcu) 
                ELSE error END
              ELSE
                IF extOp = 8 THEN Instr("subfc", RTxRAxRBxRc) 
                ELSIF extOp = 10 THEN Instr("addc", RTxRAxRBxRc) 
                ELSE error END
              END
            ELSE
              IF extOp <= 23 THEN
                IF extOp = 19 THEN Instr("mfcr", RTxRcu) ELSIF extOp = 23 THEN Instr("lwzx", RTxRAxRBxRcu) 
                ELSE error END
              ELSE
                IF extOp = 24 THEN Instr("slw", RAxRSxRBxRc) ELSIF extOp = 26 THEN Instr("cntlzw", RAxRSxRc) 
                ELSE error END
              END
            END
          ELSE
            IF extOp <= 55 THEN
              IF extOp <= 29 THEN
                IF extOp = 28 THEN Instr("and", RAxRSxRBxRc) 
                ELSIF extOp = 29 THEN Instr("maskg", RAxRSxRBxRc) 
                ELSE error END
              ELSE
                IF extOp = 32 THEN Instr("cmpl", BFxRAxRBxRcu) ELSIF extOp = 55 THEN Instr("lwzux", RTxRAxRBxRcu) 
                ELSE error END
              END
            ELSE
              IF extOp <= 83 THEN
                IF extOp = 60 THEN Instr("andc", RAxRSxRBxRc) ELSIF extOp = 83 THEN Instr("mfmsr", RTxRcu) 
                ELSE error END
              ELSE
                IF extOp = 87 THEN Instr("lbzx", RTxRAxRBxRcu) ELSE error END
              END
            END
          END
        ELSE
          IF extOp <= 144 THEN
            IF extOp <= 119 THEN
              IF extOp <= 107 THEN
                IF extOp = 104 THEN Instr("neg", RTxRAxRc) ELSIF extOp = 107 THEN Instr("mul", RTxRAxRBxRc) 
                ELSE error END
              ELSE
                IF extOp = 118 THEN Instr("clf", RAxRBxRcu) ELSIF extOp = 119 THEN Instr("lbzux", RTxRAxRBxRcu) 
                ELSE error END
              END
            ELSE
              IF extOp <= 136 THEN
                IF extOp = 124 THEN Instr("nor", RAxRSxRBxRc) ELSIF extOp = 136 THEN Instr("subfe", RTxRAxRBxRc) 
                ELSE error END
              ELSE
                IF extOp = 138 THEN Instr("adde", RTxRAxRBxRc) ELSIF extOp = 144 THEN Instr("mtcrf", FXMxRSxRcu) 
                ELSE error END
              END
            END
          ELSE
            IF extOp <= 153 THEN
              IF extOp <= 151 THEN
                IF extOp = 146 THEN Instr("mtmsr", RSxRcu) ELSIF extOp = 151 THEN Instr("stwx", RSxRAxRBxRcu) 
                ELSE error END
              ELSE
                IF extOp = 152 THEN Instr("slq", RAxRSxRBxRc) ELSIF extOp = 153 THEN Instr("sle", RAxRSxRBxRc) 
                ELSE error END
              END
            ELSE
              IF extOp <= 184 THEN
                IF extOp = 183 THEN Instr("stwux", RSxRAxRBxRcu) 
                ELSIF extOp = 184 THEN Instr("sliq", RAxRSxSHxRc) 
                ELSE error END
              ELSE
                IF extOp = 200 THEN Instr("subfze", RTxRAxRc) ELSE error END
              END
            END
          END
        END
      ELSE
        IF extOp <= 279 THEN
          IF extOp <= 235 THEN
            IF extOp <= 216 THEN
              IF extOp <= 210 THEN
                IF extOp = 202 THEN Instr("addze", RTxRAxRc) ELSIF extOp = 210 THEN Instr("mtsr", SRxRSxRcu) 
                ELSE error END
              ELSE
                IF extOp = 215 THEN Instr("stbx", RSxRAxRBxRcu) 
                ELSIF extOp = 216 THEN Instr("sllq", RAxRSxRBxRc) 
                ELSE error END
              END
            ELSE
              IF extOp <= 232 THEN
                IF extOp = 217 THEN Instr("sleq", RAxRSxRBxRc) ELSIF extOp = 232 THEN Instr("subfme", RTxRAxRc) 
                ELSE error END
              ELSE
                IF extOp = 234 THEN Instr("addme", RTxRAxRc) ELSIF extOp = 235 THEN Instr("mullw", RTxRAxRBxRc) 
                ELSE error END
              END
            END
          ELSE
            IF extOp <= 264 THEN
              IF extOp <= 247 THEN
                IF extOp = 242 THEN Instr("mtsrin", RSxRAxRBxRcu) 
                ELSIF extOp = 247 THEN Instr("stbux", RSxRAxRBxRcu) 
                ELSE error END
              ELSE
                IF extOp = 248 THEN Instr("slliq", RAxRSxSHxRc) 
                ELSIF extOp = 264 THEN Instr("doz", RTxRAxRBxRc) 
                ELSE error END
              END
            ELSE
              IF extOp <= 277 THEN
                IF extOp = 266 THEN Instr("add", RTxRAxRBxRc) 
                ELSIF extOp = 277 THEN Instr("lscbx", RTxRAxRBxRc) 
                ELSE error END
              ELSE
                IF extOp = 279 THEN Instr("lhzx", RTxRAxRBxRcu) ELSE error END
              END
            END
          END
        ELSE
          IF extOp <= 360 THEN
            IF extOp <= 316 THEN
              IF extOp <= 306 THEN
                IF extOp = 284 THEN Instr("eqv", RAxRSxRBxRc) ELSIF extOp = 306 THEN Instr("tlbie", RAxRBxRc) 
                ELSE error END
              ELSE
                IF extOp = 311 THEN Instr("lhzux", RTxRAxRBxRcu) 
                ELSIF extOp = 316 THEN Instr("xor", RAxRSxRBxRc) 
                ELSE error END
              END
            ELSE
              IF extOp <= 339 THEN
                IF extOp = 331 THEN Instr("div", RTxRAxRBxRc) ELSIF extOp = 339 THEN Instr("mfspr", RTxSPRxRcu) 
                ELSE error END
              ELSE
                IF extOp = 343 THEN Instr("lhax", RTxRAxRBxRcu) ELSIF extOp = 360 THEN Instr("abs", RTxRAxRc) 
                ELSE error END
              END
            END
          ELSE
            IF extOp <= 412 THEN
              IF extOp <= 375 THEN
                IF extOp = 363 THEN Instr("divs", RTxRAxRBxRc) 
                ELSIF extOp = 375 THEN Instr("lhaux", RTxRAxRBxRcu) 
                ELSE error END
              ELSE
                IF extOp = 407 THEN Instr("sthx", RSxRAxRBxRcu) 
                ELSIF extOp = 412 THEN Instr("orc", RAxRSxRBxRc) 
                ELSE error END
              END
            ELSE
              IF extOp <= 444 THEN
                IF extOp = 439 THEN Instr("sthux", RSxRAxRBxRcu) 
                ELSIF extOp = 444 THEN Instr("or", RAxRSxRBxRc) 
                ELSE error END
              ELSE
                IF extOp = 467 THEN Instr("mtspr", SPRxRSxRcu) ELSE error END
              END
            END
          END
        END
      END
    ELSE
      IF extOp <= 665 THEN
        IF extOp <= 595 THEN
          IF extOp <= 533 THEN
            IF extOp <= 512 THEN
              IF extOp <= 488 THEN
                IF extOp = 476 THEN Instr("nand", RAxRSxRBxRc) ELSIF extOp = 488 THEN Instr("nabs", RTxRAxRc) 
                ELSE error END
              ELSE
                IF extOp = 502 THEN Instr("cli", RAxRBxRcu) ELSIF extOp = 512 THEN Instr("mcrxr", BFxRcu) 
                ELSE error END
              END
            ELSE
              IF extOp <= 522 THEN
                IF extOp = 520 THEN Instr("subfco", RTxRAxRBxRc) ELSIF extOp = 522 THEN Instr("addco", RTxRAxRBxRc) 
                ELSE error END
              ELSE
                IF extOp = 531 THEN Instr("clcs", RTxRAxRcu) ELSIF extOp = 533 THEN Instr("lswx", RTxRAxRBxRcu) 
                ELSE error END
              END
            END
          ELSE
            IF extOp <= 537 THEN
              IF extOp <= 535 THEN
                IF extOp = 534 THEN Instr("lwbrx", RTxRAxRBxRcu) 
                ELSIF extOp = 535 THEN Instr("lfsx", FRTxRAxRBxRcu) 
                ELSE error END
              ELSE
                IF extOp = 536 THEN Instr("srw", RAxRSxRBxRc) ELSIF extOp = 537 THEN Instr("rrib", RAxRSxRBxRc) 
                ELSE error END
              END
            ELSE
              IF extOp <= 567 THEN
                IF extOp = 541 THEN Instr("maskir", RAxRSxRBxRc) 
                ELSIF extOp = 567 THEN Instr("lfsux", FRTxRAxRBxRcu) 
                ELSE error END
              ELSE
                IF extOp = 595 THEN Instr("mfsr", RTxSRxRcu) ELSE error END
              END
            END
          END
        ELSE
          IF extOp <= 631 THEN
            IF extOp <= 616 THEN
              IF extOp <= 598 THEN
                IF extOp = 597 THEN Instr("lswi", RTxRAxNBxRcu) ELSIF extOp = 598 THEN Instr("sync", Rcu) 
                ELSE error END
              ELSE
                IF extOp = 599 THEN Instr("lfdx", FRTxRAxRBxRcu) 
                ELSIF extOp = 616 THEN Instr("nego", RTxRAxRc) 
                ELSE error END
              END
            ELSE
              IF extOp <= 627 THEN
                IF extOp = 619 THEN Instr("mulo", RTxRAxRBxRc) 
                ELSIF extOp = 627 THEN Instr("mfsri", RTxRAxRBxRcu) 
                ELSE error END
              ELSE
                IF extOp = 630 THEN Instr("dclst", RAxRBxRcu) 
                ELSIF extOp = 631 THEN Instr("lfdux", FRTxRAxRBxRcu) 
                ELSE error END
              END
            END
          ELSE
            IF extOp <= 662 THEN
              IF extOp <= 650 THEN
                IF extOp = 648 THEN Instr("subfeo", RTxRAxRBxRc) 
                ELSIF extOp = 650 THEN Instr("addeo", RTxRAxRBxRc) 
                ELSE error END
              ELSE
                IF extOp = 661 THEN Instr("stswx", RSxRAxRBxRcu) 
                ELSIF extOp = 662 THEN Instr("stwbrx", RSxRAxRBxRcu) 
                ELSE error END
              END
            ELSE
              IF extOp <= 664 THEN
                IF extOp = 663 THEN Instr("stfsx", FRSxRAxRBxRcu) 
                ELSIF extOp = 664 THEN Instr("srq", RAxRSxRBxRc) 
                ELSE error END
              ELSE
                IF extOp = 665 THEN Instr("sre", RAxRSxRBxRc) ELSE error END
              END
            END
          END
        END
      ELSE
        IF extOp <= 778 THEN
          IF extOp <= 729 THEN
            IF extOp <= 714 THEN
              IF extOp <= 696 THEN
                IF extOp = 695 THEN Instr("stfsux", FRSxRAxRBxRcu) 
                ELSIF extOp = 696 THEN Instr("sriq", RAxRSxSHxRc) 
                ELSE error END
              ELSE
                IF extOp = 712 THEN Instr("subfzeo", RTxRAxRc) ELSIF extOp = 714 THEN Instr("addzeo", RTxRAxRc) 
                ELSE error END
              END
            ELSE
              IF extOp <= 727 THEN
                IF extOp = 725 THEN Instr("stswi", RSxRAxNBxRcu) 
                ELSIF extOp = 727 THEN Instr("stfdx", FRSxRAxRBxRcu) 
                ELSE error END
              ELSE
                IF extOp = 728 THEN Instr("srlq", RAxRSxRBxRc) 
                ELSIF extOp = 729 THEN Instr("sreq", RAxRSxRBxRc) 
                ELSE error END
              END
            END
          ELSE
            IF extOp <= 759 THEN
              IF extOp <= 746 THEN
                IF extOp = 744 THEN Instr("subfmeo", RTxRAxRc) ELSIF extOp = 746 THEN Instr("addmeo", RTxRAxRc) 
                ELSE error END
              ELSE
                IF extOp = 747 THEN Instr("mullwo", RTxRAxRBxRc) 
                ELSIF extOp = 759 THEN Instr("stfdux", FRSxRAxRBxRcu) 
                ELSE error END
              END
            ELSE
              IF extOp <= 776 THEN
                IF extOp = 760 THEN Instr("srliq", RAxRSxSHxRc) 
                ELSIF extOp = 776 THEN Instr("dozo", RTxRAxRBxRc) 
                ELSE error END
              ELSE
                IF extOp = 778 THEN Instr("addo", RTxRAxRBxRc) ELSE error END
              END
            END
          END
        ELSE
          IF extOp <= 875 THEN
            IF extOp <= 824 THEN
              IF extOp <= 792 THEN
                IF extOp = 790 THEN Instr("lhbrx", RTxRAxRBxRcu) 
                ELSIF extOp = 792 THEN Instr("sraw", RAxRSxRBxRc) 
                ELSE error END
              ELSE
                IF extOp = 818 THEN Instr("rac", RTxRAxRBxRc) ELSIF extOp = 824 THEN Instr("srawi", RAxRSxSHxRc) 
                ELSE error END
              END
            ELSE
              IF extOp <= 872 THEN
                IF extOp = 843 THEN Instr("divo", RTxRAxRBxRc) ELSIF extOp = 872 THEN Instr("abso", RTxRAxRc) 
                ELSE error END
              ELSE
                IF extOp = 875 THEN Instr("divso", RTxRAxRBxRc) ELSE error END
              END
            END
          ELSE
            IF extOp <= 921 THEN
              IF extOp <= 918 THEN
                IF extOp = 900 THEN Instr("nabso", RTxRAxRc) 
                ELSIF extOp = 918 THEN Instr("sthbrx", RSxRAxRBxRcu) 
                ELSE error END
              ELSE
                IF extOp = 920 THEN Instr("sraq", RAxRSxRBxRc) 
                ELSIF extOp = 921 THEN Instr("srea", RAxRSxRBxRc) 
                ELSE error END
              END
            ELSE
              IF extOp <= 952 THEN
                IF extOp = 922 THEN Instr("extsh", RAxRSxRc) ELSIF extOp = 952 THEN Instr("sraiq", RAxRSxSHxRc) 
                ELSE error END
              ELSE
                IF extOp = 954 THEN Instr("extsb", RAxRSxRc) ELSIF extOp = 1014 THEN Instr("dcbz", RAxRBxRcu) ELSE error END
              END
            END
          END
        END
      END
    END
  END DecodeX;

  PROCEDURE DecodeX2 (instr: LONGINT);
    VAR extOp: LONGINT;
  BEGIN
    extOp := ASH(instr MOD 64, -1);  (* for A-form instructions *)
    IF (extOp < 32) & (extOp IN {18, 20, 21, 25, 28, 29, 30, 31}) THEN
      CASE extOp OF
        18: Instr("fdiv", FRTxFRAxFRBxRc)
      |  20: Instr("fsub", FRTxFRAxFRBxRc)
      |  21: Instr("fadd", FRTxFRAxFRBxRc)
      |  25: Instr("fmul", FRTxFRAxFRCxRc)
      |  28: Instr("fmsub", FRTxFRAxFRCxFRBxRc)
      |  29: Instr("fmadd", FRTxFRAxFRCxFRBxRc)
      |  30: Instr("fnmsub", FRTxFRAxFRCxFRBxRc)
      |  31: Instr("fnmadd", FRTxFRAxFRCxFRBxRc)
      END
    ELSE extOp := ASH(instr MOD 2048, -1);  (* for X-form instructions *)
      IF extOp <= 70 THEN
        IF extOp <= 38 THEN
          IF extOp <= 12 THEN
            IF extOp = 0 THEN Instr("fcmpu", BFxFRAxFRBxRcu) ELSIF extOp = 12 THEN Instr("frsp", FRTxFRBxRc) 
            ELSE error END
          ELSE
            IF extOp = 32 THEN Instr("fcmpo", BFxFRAxFRBxRcu) ELSIF extOp = 38 THEN Instr("mtfsb1", BTxRc) 
            ELSE error END
          END
        ELSE
          IF extOp <= 64 THEN
            IF extOp = 40 THEN Instr("fneg", FRTxFRBxRc) ELSIF extOp = 64 THEN Instr("mcrfs", BFxBFAxRcu) 
            ELSE error END
          ELSE
            IF extOp = 70 THEN Instr("mtfsb0", BTxRc) ELSE error END
          END
        END
      ELSE
        IF extOp <= 136 THEN
          IF extOp <= 134 THEN
            IF extOp = 72 THEN Instr("fmr", FRTxFRBxRc) ELSIF extOp = 134 THEN Instr("mtfsfi", BFxIxRc) 
            ELSE error END
          ELSE
            IF extOp = 136 THEN Instr("fnabs", FRTxFRBxRc) ELSE error END
          END
        ELSE
          IF extOp <= 583 THEN
            IF extOp = 264 THEN Instr("fabs", FRTxFRBxRc) ELSIF extOp = 583 THEN Instr("mffs", FRTxRc) 
            ELSE error END
          ELSE
            IF extOp = 711 THEN Instr("mtfsf", FLMxFRBxRc) ELSE error END
          END
        END
      END
    END
  END DecodeX2;
    
  PROCEDURE DecodeInstr (instr: LONGINT);
    VAR pOpcode: LONGINT; s: ARRAY 2 OF CHAR;
  BEGIN curInstr := instr;
    pOpcode := SYSTEM.LSH(instr, -26);
    CASE pOpcode OF
      3: Instr("twi", TOxRAxSI)
    |  7: Instr("mulli", RTxRAxSI)
    |  8: Instr("subfic", RTxRAxSI)
    |  9: Instr("dozi", RTxRAxSI)
    |  10: Instr("cmpli", BFxRAxUI)
    |  11: Instr("cmpi", BFxRAxSI)
    |  12: Instr("addic", RTxRAxSI)
    |  13: Instr("addic.", RTxRAxSI)
    |  14:
        IF instr DIV 65536 MOD 32 = 0 THEN Instr("li", RTxSI)
        ELSIF instr MOD 65536 = 0 THEN Instr("lr", RTxRA)
        ELSE Instr("addi", RTxDxRA)
        END
    |  15: Instr("addis", RTxRAxUI)
    |  16: Instr("bc", BOxBIxBDxAAxLK)
    |  17: Instr("svc", None)  (* special handling to be added later *)
    |  18: s[0] := "b"; s[1] := 0X; Instr(s, LIxAAxLK)
    |  19: DecodeXL(instr)
    |  20: Instr("rlwimi", RAxRSxSHxMBxMExRc)
    |  21: Instr("rlwinm", RAxRSxSHxMBxMExRc)
    |  22: Instr("rlmi", RAxRSxSHxMBxMExRc)
    |  23: Instr("rlwnm", RAxRSxRBxMBxMExRc)
    |  24: Instr("ori", RAxRSxUI)
    |  25: Instr("oris", RAxRSxUI)
    |  26: Instr("xori", RAxRSxUI)
    |  27: Instr("xoris", RAxRSxUI)
    |  28: Instr("andi.", RAxRSxUI)
    |  29: Instr("andis.", RAxRSxUI)
    |  31: DecodeX(instr)
    |  32: Instr("lwz", RTxDxRA)
    |  33: Instr("lwzu", RTxDxRA)
    |  34: Instr("lbz", RTxDxRA)
    |  35: Instr("lbzu", RTxDxRA)
    |  36: Instr("stw", RSxDxRA)
    |  37: Instr("stwu", RSxDxRA)
    |  38: Instr("stb", RSxDxRA)
    |  39: Instr("stbu", RSxDxRA)
    |  40: Instr("lhz", RTxDxRA)
    |  41: Instr("lhzu", RTxDxRA)
    |  42: Instr("lha", RTxDxRA)
    |  43: Instr("lhau", RTxDxRA)
    |  44: Instr("sth", RSxDxRA)
    |  45: Instr("sthu", RSxDxRA)
    |  46: Instr("lmw", RTxDxRA)
    |  47: Instr("stmw", RSxDxRA)
    |  48: Instr("lfs", FRTxDxRA)
    |  49: Instr("lfsu", FRTxDxRA)
    |  50: Instr("lfd", FRTxDxRA)
    |  51: Instr("lfdu", FRTxDxRA)
    |  52: Instr("stfs", FRSxDxRA)
    |  53: Instr("stfsu", FRSxDxRA)
    |  54: Instr("stfd", FRSxDxRA)
    |  55: Instr("stfdu", FRSxDxRA)
    |  59:  (* PowerPC single precision FP *)
        CASE ASH(instr MOD 64, -1) OF
           18: Instr("fdivs", FRTxFRAxFRBxRc)
        |  20: Instr("fsubs", FRTxFRAxFRBxRc)
        |  21: Instr("fadds", FRTxFRAxFRBxRc)
        |  25: Instr("fmuls", FRTxFRAxFRCxRc)
        |  28: Instr("fmsubs", FRTxFRAxFRCxFRBxRc)
        |  29: Instr("fmadds", FRTxFRAxFRCxFRBxRc)
        |  30: Instr("fnmsubs", FRTxFRAxFRCxFRBxRc)
        |  31: Instr("fnmadds", FRTxFRAxFRCxFRBxRc)
        ELSE
        END
    |  63: DecodeX2(instr)
    |  0: error
    ELSE error
    END
  END DecodeInstr;

  PROCEDURE DumpInt (VAR R: br.Reader; comment: ARRAY OF CHAR); 
    VAR i: INTEGER;
  BEGIN
    R.ReadInt(i); Out.String(tabs); Out.String(comment);
    Out.String(": "); Out.Int(i, 0); Out.Ln
  END DumpInt;

  PROCEDURE DumpIntS (VAR R: br.Reader; comment: ARRAY OF CHAR; VAR i: INTEGER); 
  BEGIN
    R.ReadInt(i); Out.String(tabs); Out.String(comment);
    Out.String(": "); Out.Int(i, 0); Out.Ln
  END DumpIntS;

  PROCEDURE DumpLongInt (VAR R: br.Reader; comment: ARRAY OF CHAR);
    VAR i: LONGINT;
  BEGIN
    R.ReadLInt(i); Out.String(tabs); Out.String(comment);
    Out.String(": "); Out.LongInt(i, 0); Out.Ln;
  END DumpLongInt;

  PROCEDURE DumpHex (VAR R: br.Reader; comment: ARRAY OF CHAR);
    VAR i: LONGINT;
  BEGIN
    R.ReadLInt(i); Out.String(tabs); Out.String(comment);
    Out.String(": "); Out.Hex(i, 8); Out.Char("H"); Out.Ln
  END DumpHex;

  PROCEDURE WriteName (VAR R: br.Reader);
      VAR s: ARRAY 64 OF CHAR; ch: CHAR; i: INTEGER;
  BEGIN
    i := 0; REPEAT R.ReadChar(ch); s[i] := ch; INC(i) UNTIL ch = 0X;
    Out.String(s)
  END WriteName;

  PROCEDURE WriteInt (VAR R: br.Reader);
    VAR i: INTEGER;
  BEGIN
    R.ReadInt(i); Out.Int(i, 0)
  END WriteInt;

  PROCEDURE WriteLongInt (VAR R: br.Reader);
    VAR i: LONGINT;
  BEGIN
    R.ReadLInt(i); Out.LongInt(i, 0)
  END WriteLongInt;

  PROCEDURE WriteLongHex (VAR R: br.Reader);
    VAR i: LONGINT;
  BEGIN
    R.ReadLInt(i); Out.Hex(i, 8)
  END WriteLongHex;

  PROCEDURE DumpName (VAR R: br.Reader; comment: ARRAY OF CHAR);
    VAR s: ARRAY 64 OF CHAR; ch: CHAR; i: INTEGER;
  BEGIN
    i := 0; REPEAT R.ReadChar(ch); s[i] := ch; INC(i) UNTIL ch = 0X;
    Out.String(tabs); Out.String(comment); Out.String(": "); 
    Out.String(s); Out.Ln
  END DumpName;

  PROCEDURE ReadCompInt (VAR R: br.Reader; VAR i: LONGINT);
    VAR n: LONGINT; s: SHORTINT; x: CHAR;
  BEGIN s := 0; n := 0; R.ReadChar(x);
    WHILE ORD(x) >= 128 DO INC(n, ASH(ORD(x) - 128, s)); INC(s, 7); R.ReadChar(x) END;
    i := n + ASH(ORD(x) MOD 64 - ORD(x) DIV 64 * 64, s)
  END ReadCompInt;

  PROCEDURE OverReadTypes (VAR r: br.Reader);   (* MK *)
    VAR n: LONGINT; ch: CHAR;
  BEGIN
    r.ReadChar (ch);
    IF ch = CHR (ProcTyp) THEN ReadCompInt (r, n)
    ELSIF ch = 0FX THEN ReadCompInt (r, n); ReadCompInt (r, n); OverReadTypes (r)
    ELSIF ch = 10X THEN r.ReadChar (ch); ReadCompInt (r, n)
    ELSIF ch = 11X THEN ReadCompInt (r, n); OverReadTypes (r)
    ELSIF ch = CHR (Pointer) THEN OverReadTypes (r)
    END;
  END OverReadTypes;
  
  PROCEDURE DumpRefs (VAR R: br.Reader; VAR nextProc: LONGINT);
    VAR ch: CHAR; fsize, psize, ralloc, calloc, falloc: LONGINT; leaf: BOOLEAN;
  BEGIN
    ReadCompInt(R, nextProc); nextProc := nextProc*4;
    IF nextProc # 0 THEN  (* fix, if there are no references *)
      Out.Ln; Out.String("PROCEDURE ");
      ReadCompInt(R, fsize); ReadCompInt(R, psize); ReadCompInt(R, ralloc);
      ReadCompInt(R, falloc); ReadCompInt(R, calloc); 
      R.ReadBool(leaf);
      R.ReadChar(ch);
      REPEAT Out.Char(ch); R.ReadChar(ch) UNTIL ch = 0X;
      R.ReadChar(ch);
      WHILE (ch # 0F8X) & (R.Res()=br.done) DO (* MK extended reference info *)
        R.ReadChar(ch);
        REPEAT R.ReadChar(ch) UNTIL ch = 0X;
        ReadCompInt(R, fsize);
        OverReadTypes (R);
        R.ReadChar (ch)
      END;
      Out.String(": fsize = "); Out.LongInt(fsize, 0);
      Out.String(", psize = "); Out.LongInt(psize, 0);
      Out.String(", ralloc = "); Out.LongInt(ralloc, 0);
      Out.String(", falloc = "); Out.LongInt(falloc, 0);
      Out.String(", calloc = "); Out.LongInt(calloc, 0);
      IF leaf THEN Out.String(", LEAF") ELSE Out.String(", NON-LEAF") END;
      Out.Ln
    END
  END DumpRefs;

  PROCEDURE Decode*; 
    VAR
      f: Files.File; R, Ref: br.Reader;
      i, lastProc: LONGINT;
      x, entno, nofcom, nofrec, nofptrs, nofGmod, noflk, code, const, noftraps: INTEGER; ch: CHAR;
      invRef: BOOLEAN;
      nofmth, nofnewmth, nofinhmth, res: INTEGER;
      name: ARRAY 64 OF CHAR;
      cli: tr.Reader;
  BEGIN
    f := NIL;
    IF Args.args.ArgNumber() > 0 THEN
      cli:=tr.ConnectReader(Args.args);
      i := 0; cli.ReadLine(name); cli.ReadLine(name);
      WHILE (name[i] # 0X) & (name[i] # ".") DO INC(i) END;
      IF name[i] = 0X THEN name[i] := "." END;
      name[i+1] := "O"; name[i+2] := "b"; name[i+3] := "j"; name[i+4] := 0X;
      f := Files.Old(name, {Files.read}, res);
      IF f # NIL THEN
        R:=br.ConnectReader(f); pc := 0; R.ReadChar(ch);
        (* header *)
        IF ch = 0F8X THEN
          Out.String("HEADER "); R.ReadChar(ch); Out.Char(ch); Out.Ln;
          ReadLInt(R, i); i := R.Available()-i-1; 
          Ref:=br.ConnectReader(f); Ref.SetPos(i);
          Ref.ReadChar(ch); invRef := ch # 8BX;
          IF invRef THEN Out.String("invalid REFERENCES"); Out.Ln END;
          Ref.ReadChar(ch);
          DumpIntS(R, "Entries", entno); DumpIntS(R, "Commands", nofcom);
          DumpIntS(R, "Pointers", nofptrs); DumpIntS(R, "Records", nofrec); DumpIntS(R, "Modules", nofGmod);
          DumpInt(R, "Linktable"); DumpIntS(R, "Links", noflk);
          DumpLongInt(R, "Datasize"); DumpIntS(R, "Constsize", const); DumpIntS(R, "Codesize (words)", code);
          DumpIntS(R, "Traps", noftraps);
          DumpHex(R, "Key"); DumpName(R, "Modulename"); R.ReadChar(ch)
        ELSE
          Out.String("invalid HEADER"); Out.Ln;
          WHILE (ch # 82X) & (R.Res()=br.done) DO R.ReadChar(ch) END
        END;
        Out.Ln;
        (* entry block *)
        IF ch = 82X THEN
          Out.String("ENTRY"); Out.Ln; i := 0;
          WHILE i < entno DO
            Out.String(tabs); Out.LongInt(i, 0); Out.String(": ");
            ReadInt(R, x); Out.Hex(LONG(x)*4, 8); Out.Char("H");
            Out.Ln; INC(i)
          END;
          R.ReadChar(ch)
        ELSE
          Out.String("invalid ENTRY"); Out.Ln;
          WHILE (ch # 83X) & (R.Res()=br.done) DO R.ReadChar(ch) END
        END;
        Out.Ln;
        (* command block *)
        IF ch = 83X THEN
          Out.String("COMMAND"); Out.Ln; i := 0;
          WHILE i < nofcom DO
            Out.String(tabs); WriteName(R); Out.String(": "); ReadInt(R, x);
            Out.Hex(LONG(x)*4, 8); Out.Char("H");
            Out.Ln; INC(i)
          END;
          R.ReadChar(ch)
        ELSE
          Out.String("invalid COMMAND"); Out.Ln;
          WHILE (ch # 84X) & (R.Res()=br.done) DO R.ReadChar(ch) END
        END;
        Out.Ln;
        (* pointer block *)
        IF ch = 84X THEN
          Out.String("POINTERS"); i := 0;
          WHILE i < nofptrs DO
            IF i MOD 8 = 0 THEN Out.Ln; Out.String(tabs) END;
            WriteLongInt(R); Out.Char(" "); INC(i)
          END;
          Out.Ln; R.ReadChar(ch)
        ELSE
          Out.String("invalid POINTERS"); Out.Ln;
          WHILE (ch # 85X) & (R.Res()=br.done) DO R.ReadChar(ch) END
        END;
        Out.Ln;
        (* import block *)
        IF ch = 85X THEN
          Out.String("IMPORT"); Out.Ln; i := 0;
          WHILE i < nofGmod DO
            Out.String(tabs); WriteLongHex(R); Out.String("H "); WriteName(R);
            Out.Ln; INC(i)
          END;
          R.ReadChar(ch)
        ELSE
          Out.String("invalid IMPORT"); Out.Ln;
          WHILE (ch # 86X) & (R.Res()=br.done) DO R.ReadChar(ch) END
        END;
        Out.Ln;
        (* link entries *)
        IF ch = 86X THEN
          Out.String("LINKS"); Out.Ln; i := 0;
          WHILE i < noflk DO
            Out.String(tabs); R.ReadChar(ch); Out.Char("("); Out.Int(ORD(ch), 0);
            R.ReadChar(ch); Out.Char(","); Out.Int(ORD(ch), 0); Out.String("): ");
            ReadInt(R, x); x := -x; Out.Hex(LONG(x)*4, 8); Out.Char("H");
            Out.Ln; INC(i)
          END;
          R.ReadChar(ch)
        ELSE
          Out.String("invalid LINKS"); Out.Ln;
          WHILE (ch # 87X) & (R.Res()=br.done) DO R.ReadChar(ch) END
        END;
        Out.Ln;
        (* const block *)
        IF ch = 87X THEN
          Out.String("CONST (not dumped)"); R.SetPos(R.Pos()+const); Out.Ln
        ELSE Out.String("invalid CONST"); Out.Ln
        END;
        Out.Ln;
        WHILE (ch # 88X) & (R.Res()=br.done) DO R.ReadChar(ch) END;
        (* code block *)
        IF ch = 88X THEN
          Out.String("CODE"); Out.Ln; pc := 0; lastProc := 0;
          WHILE pc < LONG(code)*4 DO
            IF (pc >= lastProc) & ~invRef THEN DumpRefs(Ref, lastProc) END;
            ReadLInt(R, i);
            Out.String(tabs); Out.Hex(pc, 8); Out.String(tabs); Out.String(tabs);
            Out.Hex(i, 8); Out.Char("H"); Out.String(tabs); DecodeInstr(i); Out.Ln;
            INC(pc, 4)
          END;
          Out.Ln; R.ReadChar(ch)
        ELSE
          Out.String("invalid CODE"); Out.Ln;
          WHILE (ch # 89X) & (R.Res()=br.done) DO R.ReadChar(ch) END
        END;
        Out.Ln;
        (* type descriptors *)
        IF ch = 89X THEN
          Out.String("TYPES"); Out.Ln; i := 0;
          WHILE i < nofrec DO
            Out.String(tabs); Out.String("size="); WriteLongInt(R);
            Out.String(", addr="); WriteInt(R);
            Out.String(", base=("); WriteInt(R); Out.Char(","); WriteInt(R); Out.Char(")");
            R.ReadBytes(nofmth, 0, 2); R.ReadBytes(nofinhmth, 0, 2); R.ReadBytes(nofnewmth, 0, 2);
            R.ReadBytes(nofptrs, 0, 2);
            Out.Ln; Out.String(tabs); Out.String("name="); WriteName(R);
            Out.Ln; Out.String(tabs); Out.Int(nofmth, 0);
            Out.String(" methods, "); Out.Int(nofinhmth, 0);
            Out.String(" inherited, new=(");
            x := 0;
            WHILE x < nofnewmth DO
              WriteInt(R); Out.Char(":"); WriteInt(R); Out.Char(" "); INC(x)
            END;
            Out.Char(")"); Out.Ln; Out.String(tabs); Out.String("ptrtab=");
            x := 0;
            WHILE x < nofptrs DO
              IF x MOD 8 = 7 THEN Out.Ln; Out.String(tabs) END;
              WriteLongInt(R); Out.Char(" "); INC(x)
            END;
            Out.Ln; INC(i)
          END;
          Out.Ln; R.ReadChar(ch)
        ELSE 
          Out.String("invalid TYPES"); Out.Ln;
          WHILE (ch # 8AX) & (R.Res()=br.done) DO R.ReadChar(ch) END
        END;
        Out.Ln;
        IF ch = 8AX THEN
          Out.String("TRAPS"); Out.Ln; i := 0;
          WHILE i < noftraps DO
            Out.String(tabs); ReadInt(R, x); Out.Hex(LONG(x)*4, 8); 
            Out.String("H: "); WriteInt(R); Out.Ln; INC(i)
          END;
          Out.Ln
        ELSE
          Out.String("invalid TRAPS"); Out.Ln
        END
      ELSE
        Out.String(name); Out.String(" not found"); Out.Ln
      END
    ELSE
      Out.String("Error: Nothing to decode."); Out.Ln
    END
  END Decode;

BEGIN
  Decode
END Decoder.

