/* ------------------------------------------------------------------ 

	Title: WGGraphAxis

	Description:  A class to be used in constructing graph axis.
		It provides base operations for the abscissa & ordinate.

	Author :
		Stephen Wardlaw, M.D.
		Yale University School of Medicine
		20 York St.
		New Haven, CT  06504

	Edit History:
		18 March, 1998
			Removed all structs to make it "endian-safe"
		09 Dec 97
			Adapted for BeOS
		09 Feb 94
			Split again from 'Abscissa' & 'Ordinate'
		24 Jan 94
			Split from the old Module Graph in MW M-2 and
			re-written for MetroWerks C++
		16 May 90
			Modified for MetroWerks Modula-2 on Macintosh
		06 Nov 87
			Last IBM Version 6.0
		?? Aug 80
			First version for Apple IIe
		
------------------------------------------------------------------ */ 

#include "WGGraphAxis.h"
#include "UFloating_Util.h"

// -------------------------------------------------------------------
// *  Constructors & Destructor             
// -------------------------------------------------------------------
WGGraphAxis::WGGraphAxis()
{
	SetDefaults();
}
// -------------------------------------------------------------------
WGGraphAxis::WGGraphAxis(const WGGraphAxis &theAxis)
{
	mMax = theAxis.mMax;
	mMin = theAxis.mMin;
	mInterval = theAxis.mInterval;
	mDivs = theAxis.mDivs;
	mScale = theAxis.mScale;
	mNumFmt = theAxis.mNumFmt;
	mNumDPs = theAxis.mNumDPs;
	mLogBase = theAxis.mLogBase;
}
// -------------------------------------------------------------------
// *  Public Methods             
// -------------------------------------------------------------------
EAxisError
WGGraphAxis::SetInterval(double interval)
{
	mInterval = interval;
	return TestValues();
}
// -------------------------------------------------------------------
EAxisError
WGGraphAxis::SetLabelFormat(ERealFormat format, int16 numDPs)
{
	mNumFmt = format;
	mNumDPs = numDPs;
	return axErr_noErr;
}
// -------------------------------------------------------------------
EAxisError
WGGraphAxis::SetMinorDivisions(int16 divs)
{
	if(divs >= 1 && divs <= 9) {
		mDivs = divs;
		return axErr_noErr;
	} else {
		return axErr_interval;
	}
}
// -------------------------------------------------------------------
EAxisError
WGGraphAxis::SetScale(EAxisScale newScale)
{
	if(newScale >= scale_linear && newScale <= scale_log10) {
		mScale = newScale;
		return TestValues();
	} else {
		return axErr_badScale;
	}
}
// -------------------------------------------------------------------
EAxisError
WGGraphAxis::SetSpan(double min, double max)
{
	// Make sure max > min
	if(max > min) {
		mMax = max;
		mMin = min;
	} else {
		mMax = min;
		mMin = max;
	}
	return TestValues();
}
// -------------------------------------------------------------------
// *  Protected Methods             
// -------------------------------------------------------------------
// This method sets the axis and checks for errors. If errors are found,
//	it corrects them and returns false.
EAxisError
WGGraphAxis::TestValues()
{
	if(mScale != scale_linear && mMin <= 0.0) {
		// Axis Minimum Must Be > 0 With Log Scale
		mMin = 1;
		if(mMax <= mMin) {
			mMax = mMin + 1.0;
		}
		return axErr_logZero;
		}
		
	switch (mScale) {
		case scale_log2:
			mLogBase = 2.0;
			mDivs = 2;
			break;
		
		case scale_log10:
			mLogBase = 10;
			mDivs = 9;
			break;
		
		default :
			;// No default state
	}
	
	// Ensure max > min
	if(mMax == mMin) {
		mMax = mMin + 1.0;
		return axErr_badScale;
	}
	
	// Check for too small a tic interval
	if(mInterval <= 0.0 || ((mMax - mMin)/mInterval) > 20.0) {
		// Tic Interval Too Small;
		mInterval = (mMax - mMin)/20.0;
		return axErr_interval;
	}
	
	return axErr_noErr;
}
// -------------------------------------------------------------------
// Transforms numbers from graph coord #s to orig input values
double
WGGraphAxis::XformFrom(double num) 
{
	mMathErr = 0;
	switch (mScale) {
	
		case scale_log2:
		case scale_log10:
			if(num < log(HUGE_VAL)) {
				return Power(mLogBase,num);
			} else {
				mMathErr = -1;
				return HUGE_VAL;
			}
			
		default:
			return num;
	}
}	
// -------------------------------------------------------------------
// Transforms numbers from input values to the graph coord #s
double
WGGraphAxis::XformTo(double num) 
{
	mMathErr = 0;
	
	switch (mScale) {
	
		case scale_log2:
		case scale_log10:
			if(num > 0.0) {
				return LogOfBase(num,mLogBase);
			} else {
				mMathErr = -1;
				return min_num;
			}

		default:
			return num;
	}
}	
// -------------------------------------------------------------------
// *  Protected Methods             
// -------------------------------------------------------------------
void
WGGraphAxis::SetDefaults()
{
	mMax = 100.0;
	mMin = 0.0;
	mInterval = 10.0;
	mDivs = 2;
	mScale = scale_linear;
	mNumFmt = realFmt_fixed;
	mNumDPs = 0;
	mLogBase = 10;
}
// -------------------------------------------------------------------
