//======================================================================
//
// FILEMON.c - main module for VxD FILEMON
//
// By Mark Russinovich and Bryce Cogswell
//
//======================================================================
#define   DEVICE_MAIN
#include  <vtoolsc.h>
#include  "ioctlcmd.h"
#include  "filemon.h"
#undef    DEVICE_MAIN

#if DEBUG
#define dprint(arg) dprintf arg
#else
#define dprint(arg)
#endif

//----------------------------------------------------------------------
//                     G L O B A L   D A T A 
//----------------------------------------------------------------------

//
// Indicates if the GUI wants activity to be logged
//
BOOLEAN                 FilterOn = FALSE;

//
// Global filter (sent to us by the GUI)
//
FILTER                  FilterDef;

//
// Real service pointers with the hook thunks
//
ppIFSFileHookFunc       PrevIFSHookProc;

//
// Hash table data 
//
PHASH_ENTRY		HashTable[NUMHASH];

//
// List of freed entries (to save some allocation calls)
//
PHASH_ENTRY		FreeEntries = NULL;

//
// Buffer data
//
PSTORE_BUF		Store 		= NULL;
ULONG			Sequence 	= 0;

//
// Maximum amount of buffers we will grab for buffered unread data
//
ULONG			NumStore 	= 0;
ULONG			MaxStore 	= 5;

//
// Semaphore for critical sections
//
SEMHANDLE               StoreMutex, HashMutex;

//
// Unknown error string
//
CHAR                    errstring[32];


//----------------------------------------------------------------------
//                   V X D  C O N T R O L
//----------------------------------------------------------------------

//
// Device declaration
//
Declare_Virtual_Device(FILEMON)

//
// Message handlers - we only care about dynamic loading and unloading
//
DefineControlHandler(SYS_DYNAMIC_DEVICE_INIT, OnSysDynamicDeviceInit);
DefineControlHandler(SYS_DYNAMIC_DEVICE_EXIT, OnSysDynamicDeviceExit);
DefineControlHandler(W32_DEVICEIOCONTROL, OnW32Deviceiocontrol);

//----------------------------------------------------------------------
// 
// ControlDispatcher
//
// Multiplexes incoming VxD messages from Windows to their handlers.
//
//----------------------------------------------------------------------
BOOL __cdecl ControlDispatcher(
	DWORD dwControlMessage,
	DWORD EBX,
	DWORD EDX,
	DWORD ESI,
	DWORD EDI,
	DWORD ECX)
{
  START_CONTROL_DISPATCH

    ON_SYS_DYNAMIC_DEVICE_INIT(OnSysDynamicDeviceInit);
    ON_SYS_DYNAMIC_DEVICE_EXIT(OnSysDynamicDeviceExit);
    ON_W32_DEVICEIOCONTROL(OnW32Deviceiocontrol);

  END_CONTROL_DISPATCH

  return TRUE;
}

//----------------------------------------------------------------------
//      P A T T E R N   M A T C H I N G   R O U T I N E S
//----------------------------------------------------------------------


//----------------------------------------------------------------------
//
// MatchOkay
//
// Only thing left after compare is more mask. This routine makes
// sure that its a valid wild card ending so that its really a match.
//
//----------------------------------------------------------------------
BOOLEAN MatchOkay( PCHAR Pattern )
{
    //
    // If pattern isn't empty, it must be a wildcard
    //
    if( *Pattern && *Pattern != '*' ) {
 
       return FALSE;
    }

    //
    // Matched
    //
    return TRUE;
}


//----------------------------------------------------------------------
//
// MatchWithPatternCore
//
// Performs nifty wildcard comparison.
//
//----------------------------------------------------------------------
BOOLEAN MatchWithPatternCore( PCHAR Pattern, PCHAR Name )
{
    //
    // End of pattern?
    //
    if( !*Pattern ) {

        return FALSE;
    }

    //
    // If we hit a wild card, do recursion
    //
    if( *Pattern == '*' ) {

        Pattern++;

        while( *Name && *Pattern ) {

            //
            // See if this substring matches
            //
	    if( *Pattern == *Name || *Name == '*' ) {

  	        if( MatchWithPatternCore( Pattern+1, Name+1 )) {

                    return TRUE;
                }
            }

            //
            // Try the next substring
            //
            Name++;
        }

        //
        // See if match condition was met
        //
        return MatchOkay( Pattern );
    } 

    //
    // Do straight compare until we hit a wild card
    //
    while( *Name && *Pattern != '*' ) {

        if( *Pattern == *Name ) {

            Pattern++;
            Name++;

        } else {

            return FALSE;
	}
    }

    //
    // If not done, recurse
    //
    if( *Name ) {

        return MatchWithPatternCore( Pattern, Name );
    }

    //
    // Make sure its a match
    //
    return MatchOkay( Pattern );
}


//----------------------------------------------------------------------
//
// MatchWithPattern
//
// Converts strings to upper-case before calling core comparison routine.
//
//----------------------------------------------------------------------
BOOLEAN MatchWithPattern( PCHAR Pattern, PCHAR Name )
{
    CHAR       Upname[1024];
    int        i;
    BOOLEAN    isMatch;

    //
    // Make the name upcased
    //
    i = 0;
    while( *Name ) {
        if( *Name >= 'a' && *Name <= 'z' )
            Upname[i++] = *Name - 'a' + 'A';
        else
            Upname[i++] = *Name;
        Name++;
    }
    Upname[i] = 0;

    //
    // Now do the comparison
    //
    isMatch = MatchWithPatternCore( Pattern, Upname );
    return isMatch;
}


//----------------------------------------------------------------------
//            B U F F E R   M A N A G E M E N T
//----------------------------------------------------------------------

//----------------------------------------------------------------------
//
// FilemonFreeStore
//
// Frees all the data output buffers that we have currently allocated.
//
//----------------------------------------------------------------------
VOID FilemonFreeStore()
{
    PSTORE_BUF 	prev;
    
    //
    // Just traverse the list of allocated output buffers
    //
    while( Store ) {

	prev = Store->Next;
	PageFree( (MEMHANDLE) Store, 0 );
	Store = prev;
    }
}	


//----------------------------------------------------------------------
//
// FilemonNewStore
//
// Called when the current buffer has filled up. This moves us to the
// pre-allocated buffer and then allocates another buffer.
//
//----------------------------------------------------------------------
void FilemonNewStore( void )
{
    PSTORE_BUF prev = Store, newstore;

    //
    // If we have maxed out or haven't accessed the current store
    // just return.
    //
    if( MaxStore == NumStore ) {

	Store->Len = 0;
	return;	
    }

    //
    // If the output buffer we currently are using is empty, just
    // use it.
    //
    if( !Store->Len ) 
	return;

    //
    // Allocate a new output buffer. Release lock to prevent deadlock
    // on reentrance (allocating memory can result in file I/O
    //
    Signal_Semaphore( StoreMutex );
    PageAllocate(STORESIZE, PG_SYS, 0, 0, 0, 0, NULL, PAGELOCKED, 
                  (PMEMHANDLE) &newstore, (PVOID) &newstore );
    Wait_Semaphore( StoreMutex, BLOCK_SVC_INTS );

    if( newstore ) { 

        //
        // Allocation was successful so add the buffer to the list
        // of allocated buffers and increment the buffer count.
        //
        Store   = newstore;
        Store->Len  = 0;
        Store->Next = prev;
        NumStore++;

    } else {

        //
        // The allocation failed - just reuse the current buffer
        //
        Store->Len = 0;
    }
    Signal_Semaphore( StoreMutex );
}


//----------------------------------------------------------------------
//
// FilemonOldestStore
//
// Goes through the linked list of storage buffers and returns the 
// oldest one.
//
//----------------------------------------------------------------------
PSTORE_BUF FilemonOldestStore( void )
{
    PSTORE_BUF  ptr = Store, prev = NULL;

    //
    // Traverse the list
    //    
    while ( ptr->Next ) {

        ptr = (prev = ptr)->Next;
    }

    //
    // Remove the buffer from the list
    //
    if ( prev ) {

        prev->Next = NULL;    
    }
    NumStore--;
    return ptr;
}


//----------------------------------------------------------------------
//
// FilemonResetStore
//
// When a GUI is no longer communicating with us, but we can't unload,
// we reset the storage buffers.
//
//----------------------------------------------------------------------
VOID FilemonResetStore()
{
    PSTORE_BUF  current, next;

    //
    // Traverse the list of output buffers
    //
    current = Store->Next;
    while( current ) {

        //
        // Free the buffer
        //
        next = current->Next;

        PageFree( (MEMHANDLE) current, 0 );

        current = next;
    }

    // 
    // Move the output pointer in the buffer that's being kept
    // the start of the buffer.
    // 
    Store->Len = 0;
    Store->Next = NULL;
}


//----------------------------------------------------------------------
//
// UpdateStore
//
// Add a new string to Store, if it fits.
//
//----------------------------------------------------------------------
void UpdateStore( ULONG seq, const char * format, ... )
{	
    PENTRY		Entry;
    int			len;
    va_list		arg_ptr;
    CHAR                text[1024];

    //
    // If no filtering is desired, don't bother
    //
    if( !FilterOn ) {
 
         return;
    }

    //
    // Lock the output buffer.
    //
    Wait_Semaphore( StoreMutex, BLOCK_SVC_INTS );

    //
    // Vsprintf to determin length of the buffer
    //
    va_start( arg_ptr, format );
    len = vsprintf( text, format, arg_ptr );
    va_end( arg_ptr );

    //
    // If the current output buffer is near capacity, move to a new
    // output buffer
    //
    if ( Store->Len > MAX_STORE-(len+sizeof(ENTRY) )) {
	
        FilemonNewStore();
    }

    //
    // This test is reduntant, but keeps us on the safe side
    //
    if ( Store->Len <= MAX_STORE-(len+sizeof(ENTRY)) )  { 

        //
        // Extract the sequence number and store it
        //
        Entry = (void *)(Store->Data+Store->Len);

        Entry->seq = seq;

        strcpy( Entry->text, text );
 
        //
        // Store the length of the string, plus 1 for the terminating
        // NULL  
        //   
        Store->Len += sizeof(Entry->seq)+len+1;
    }

    //
    // Release the output buffer lock
    //
    Signal_Semaphore( StoreMutex );
}

//----------------------------------------------------------------------
//       H A S H   T A B L E   M A N A G E M E N T
//----------------------------------------------------------------------

//----------------------------------------------------------------------
//
// FilemonHashCleanup
//
// Called when we are unloading to free any memory that we have 
// in our possession.
//
//----------------------------------------------------------------------
VOID FilemonHashCleanup()
{
    PHASH_ENTRY		hashEntry, nextEntry;
    ULONG		i;
  
    //
    // First, free the hash table entries
    //
    for( i = 0; i < NUMHASH; i++ ) {

	hashEntry = HashTable[i];

	while( hashEntry ) {
		nextEntry = hashEntry->Next;
		HeapFree( hashEntry->FullName, 0 );
		HeapFree( hashEntry, 0 );
		hashEntry = nextEntry;
	}
    }

    //
    // Now, free structures on our free list
    //
    while( FreeEntries ) {
	nextEntry = FreeEntries->Next;
	HeapFree( FreeEntries, 0 );
	FreeEntries = nextEntry;
    }
}

//----------------------------------------------------------------------
//
// FilemonStoreHash
//
// Stores the key and associated fullpath in the hash table.
//
//----------------------------------------------------------------------
VOID FilemonStoreHash( sfn_t filenumber, PCHAR fullname )
{
    PHASH_ENTRY     newEntry;

    Wait_Semaphore( HashMutex, BLOCK_SVC_INTS );

    //
    // Find or allocate an entry to use.
    //
    if( FreeEntries ) {

        newEntry    = FreeEntries;
        FreeEntries = FreeEntries->Next;
        Signal_Semaphore( HashMutex );

    } else {
    
        //
        // Have to release the lock, because memory allocation can 
        // cause re-entrancy.
        //
        Signal_Semaphore( HashMutex );
        newEntry = HeapAllocate( sizeof(HASH_ENTRY), 0 );
    }

    //
    // Initialize the new entry.
    //
    newEntry->filenumber 	        = filenumber;
    newEntry->FullName   	        = HeapAllocate( strlen(fullname)+1, 0);
    strcpy( newEntry->FullName, fullname );

    //
    // Lock the hash table and insert the new entry.
    //
    Wait_Semaphore( HashMutex, BLOCK_SVC_INTS );
    newEntry->Next		= HashTable[ HASHOBJECT(filenumber) ];
    HashTable[ HASHOBJECT(filenumber) ] = newEntry;	
    Signal_Semaphore( HashMutex );
}

//----------------------------------------------------------------------
//
// FilemonFreeHashEntry
//
// When we see a file close, we can free the string we had associated
// with the fileobject being closed since we know it won't be used
// again.
//
//----------------------------------------------------------------------
VOID FilemonFreeHashEntry( sfn_t filenumber )
{
    PHASH_ENTRY		hashEntry, prevEntry;

    Wait_Semaphore( HashMutex, BLOCK_SVC_INTS );

    //
    // Look-up the entry.
    //
    hashEntry = HashTable[ HASHOBJECT( filenumber ) ];
    prevEntry = NULL;

    while( hashEntry && hashEntry->filenumber != filenumber ) {

	prevEntry = hashEntry;
	hashEntry = hashEntry->Next;
    }

    //  
    // If we fall of the hash list without finding what we're looking
    // for, just return.
    //
    if( !hashEntry ) {

 	Signal_Semaphore( HashMutex );
	return;
    }

    //
    // Got it! Remove it from the list
    //
    if( prevEntry ) {

	prevEntry->Next = hashEntry->Next;
    } else {

	HashTable[ HASHOBJECT( filenumber )] = hashEntry->Next;
    }

    //
    // Free the memory associated with the name of the free entry.
    //
    hashEntry->Next 	= FreeEntries;
    FreeEntries 		= hashEntry;

    Signal_Semaphore( HashMutex );

    //
    // Free the memory associated with it
    //
    HeapFree( hashEntry->FullName, 0 );
}

//----------------------------------------------------------------------
//       P A T H  A N D  P R O C E S S  N A M E  R O U T I N E S
//----------------------------------------------------------------------

//----------------------------------------------------------------------
//
// ErrorString
//
// Returns the string form of an error code.
//
//----------------------------------------------------------------------
PCHAR ErrorString( DWORD retval )
{
  switch( retval ) {
  case ERROR_SUCCESS:
    return "SUCCESS";
  case ERROR_OUTOFMEMORY:
    return "OUTOFMEM";
  case ERROR_ACCESS_DENIED:
    return "ACCDENIED";
  case ERROR_PATH_NOT_FOUND:
    return "NOTFOUND";
  case ERROR_TOO_MANY_OPEN_FILES:
    return "TOOMANYOPEN";
  case ERROR_FILE_NOT_FOUND:
    return "NOTFOUND";
  case ERROR_NO_MORE_ITEMS:
    return "NOMORE";
  case ERROR_MORE_DATA:
    return "MOREDATA";
  case ERROR_INVALID_DRIVE:
    return "INVALIDDRIVE";
  case ERROR_NOT_SAME_DEVICE:
    return "DIFFERENTDEVICE";
  case ERROR_WRITE_PROTECT:
    return "WRITEPROTECTED";
  case ERROR_BAD_UNIT:
    return "BADUNIT";
  case ERROR_NOT_READY:
    return "NOTREADY";
  case ERROR_NO_MORE_FILES:
    return "NOMORE";
  default:
    sprintf(errstring, "0x%x", retval );
    return errstring;
  }
}

//----------------------------------------------------------------------
//
// wstrlen
//
// Determine the length of a wide-character string.
//
//----------------------------------------------------------------------
int wstrlen( unsigned short *unistring )
{
    int i = 0;
    int len = 0;
  
    while( unistring[i++] != 0 ) len+=2;
    return len;
}


//----------------------------------------------------------------------
//
// FilmonGetProcess
//
// Retrieves the process name.
//
//----------------------------------------------------------------------
PCHAR FilemonGetProcess( PCHAR ProcessName )
{
    PVOID       CurProc;
    PVOID       ring3proc;
    char        *name;

    //
    // Get the ring0 process pointer.
    //
    CurProc = VWIN32_GetCurrentProcessHandle();
  
    //
    // Now, map the ring3 PCB 
    //
    ring3proc = (PVOID) SelectorMapFlat( Get_Sys_VM_Handle(), 
			       (DWORD) (*(PDWORD) ((char *) CurProc + 0x38)) | 0x7, 0 );

    if( ring3proc == (PVOID) -1 ) {
        strcpy( ProcessName, "???");
        return ProcessName;
    }

    //
    // copy out the process name (max 8 characters)
    //
    name = ((char *)ring3proc) + 0xF2;
    if( name[0] >= 'A' && name[0] < 'z' ) {

        strcpy( ProcessName, name );
        ProcessName[8] = 0;
    } else {

        strcpy( ProcessName, "???" );
    }

    //
    // Apply process name filter
    //
    if( MatchWithPattern( FilterDef.processfilter, ProcessName ) ) {

       return ProcessName;

    } else {

       return NULL;
    }
}

//----------------------------------------------------------------------
//
// ApplyNameFilter
//
// If the name matches the exclusion mask, we do not log it. Else if
// it doesn't match the inclusion mask we do not log it. 
//
//----------------------------------------------------------------------
BOOLEAN ApplyNameFilter( PCHAR fullname )
{
    //   
    // If it matches the exclusion string, do not log it
    //
    if( MatchWithPattern( FilterDef.excludefilter, fullname ) ) {

        return FALSE;
    }
 
    //
    // If it *doesn't* match the path filter, do not log it
    //
    if( !MatchWithPattern( FilterDef.pathfilter, fullname )) {

        return FALSE;
    }

    //
    // It passed the filter so it will be logged
    //
    return TRUE;
}

//----------------------------------------------------------------------
//
// FilemonGetFullPath
//
// Returns the full pathname of a file, if we can obtain one, else
// returns a handle.
//
//----------------------------------------------------------------------
PCHAR FilemonGetFullPath(  sfn_t filenumber, PCHAR fullname )
{
  PHASH_ENTRY		hashEntry;

  //
  // See if we find the key in the hash table.
  //
  Wait_Semaphore( HashMutex, BLOCK_SVC_INTS );

  hashEntry = HashTable[ HASHOBJECT( filenumber ) ];

  while( hashEntry && hashEntry->filenumber != filenumber ) {

      hashEntry = hashEntry->Next;
  }

  Signal_Semaphore( HashMutex );

  fullname[0] = 0;
  if( hashEntry ) {

      strcpy( fullname, hashEntry->FullName );

  } else {

      sprintf( fullname, "0x%X", filenumber );
  }
  return fullname;
}


//----------------------------------------------------------------------
//
// FilemonConvertPath
//
// Converts a unicode path name to ANSI.
//
//----------------------------------------------------------------------
PCHAR FilemonConvertPath( int drive, path_t ppath, PCHAR fullpathname )
{
    int  i = 0;
    _QWORD result;

    //
    // Stick on the drive letter if we know it.
    //
    if( drive != 0xFF ) {
 
        fullpathname[0] = drive+'A'-1;
        fullpathname[1] = ':';
        i = 2;
    } 
    UniToBCSPath( &fullpathname[i], ppath->pp_elements, 256, BCS_WANSI, &result );
    return( fullpathname );
}


//----------------------------------------------------------------------
//
// FilemonHookProc
//
// All (most) IFS functions come through this routine for us to look
// at.
//
//----------------------------------------------------------------------
int _cdecl FilemonHookProc(pIFSFunc pfn, int fn, int Drive, int ResType,
			   int CodePage, pioreq pir)
{
    int                retval;
    char               fullpathname[256];
    char               processname[256];
    char               data[256];
    char               drivestring[4];
    ioreq              origir;
    _WIN32_FIND_DATA   *finddata;
    struct srch_entry  *search;
    _QWORD             result;
    int                i, j;

    // 
    // Inititlize default data.
    //
    data[0] = 0;

    //
    // Save original iorequest because some entries get modified.
    //
    origir = *pir;

    //
    // Get the current process name.
    //
    FilemonGetProcess( processname );

    //
    // Call the previous hooker first, to get the return code.
    //
    retval = (*PrevIFSHookProc)(pfn, fn, Drive, ResType, CodePage, pir);

    //
    // Now extract parameters based on the function type.
    //
    switch( fn ) {

    case IFSFN_OPEN:

        FilemonConvertPath( Drive, pir->ir_ppath, fullpathname );
        if( FilemonGetProcess(processname) && ApplyNameFilter( fullpathname )) {
            sprintf(data,"");
            if( origir.ir_options & ACTION_CREATENEW ) strcat(data,"CREATENEW ");
            if( origir.ir_options & ACTION_CREATEALWAYS ) strcat(data,"CREATEALWAYS ");
            if( origir.ir_options & ACTION_OPENEXISTING ) strcat(data,"OPENEXISTING ");
            if( origir.ir_options & ACTION_OPENALWAYS ) strcat(data,"OPENALWAYS ");
            if( origir.ir_options & ACTION_REPLACEEXISTING ) strcat(data,"REPLACEEXISTING ");

            UpdateStore( Sequence++, "%s\tOpen\t%s\t%s\t%s", 
		 processname, fullpathname,
		 data, ErrorString( retval ));
	}
        FilemonStoreHash( pir->ir_sfn, fullpathname );
        break;

    case IFSFN_READ:
    case IFSFN_WRITE:
        FilemonGetFullPath( pir->ir_sfn, fullpathname );
        if( ((fn == IFSFN_READ && FilterDef.logreads ) ||
	     (fn == IFSFN_WRITE && FilterDef.logwrites )) &&
	    FilemonGetProcess(processname) && ApplyNameFilter( fullpathname )) {

             sprintf( data, "Offset: %ld Length: %ld", origir.ir_pos, origir.ir_length );
             UpdateStore( Sequence++, "%s\t%s\t%s\t%s\t%s", 
		 processname, fn == IFSFN_READ? "Read" : "Write", 
		 fullpathname,
		 data, ErrorString( retval ));
	}
        break;

    case IFSFN_CLOSE:
        FilemonGetFullPath( pir->ir_sfn, fullpathname );      
        if( FilemonGetProcess(processname) && ApplyNameFilter( fullpathname )) {
            UpdateStore( Sequence++, "%s\tClose\t%s\t\t%s", 
		 processname, fullpathname,
		 ErrorString( retval ));
        }
        FilemonFreeHashEntry( pir->ir_sfn );
        break;

    case IFSFN_DIR:
	FilemonConvertPath( Drive, pir->ir_ppath, fullpathname );
        if( FilemonGetProcess(processname) && ApplyNameFilter( fullpathname )) {    
            switch( pir->ir_flags ) {
            case CREATE_DIR:
                sprintf(data, "CREATE");
                break;
            case DELETE_DIR:
                sprintf(data,"DELETE");
                break;
            case CHECK_DIR:
                sprintf(data,"CHECK");
                break;
            default:
                sprintf(data,"QUERY");
                break;
            }
    
            UpdateStore( Sequence++, "%s\tDirectory\t%s\t%s\t%s", 
		 processname, 
		 fullpathname,
		 data, ErrorString( retval ));
	}
        break;

    case IFSFN_SEEK:
	FilemonGetFullPath( pir->ir_sfn, fullpathname );
        if( FilemonGetProcess(processname) && ApplyNameFilter( fullpathname )) {
            sprintf(data, "%s Offset: %ld",
                pir->ir_flags == FILE_BEGIN ? "Beginning" : "End",
	        origir.ir_pos );
            UpdateStore( Sequence++, "%s\tSeek\t%s\t%s\t%s", 
		 processname, fullpathname,
		 data, ErrorString( retval ));
	}
        break;

    case IFSFN_COMMIT:
	FilemonGetFullPath( pir->ir_sfn, fullpathname );
        if( FilemonGetProcess(processname) && ApplyNameFilter( fullpathname )) {
            sprintf(data, "%s", pir->ir_options == FILE_COMMIT_ASYNC ? 
   	         "ASYNC" : "NOACCESSUPDATE" );
            UpdateStore( Sequence++, "%s\tCommit\t%s\t%s\t%s", 
		 processname, fullpathname,
		 data, ErrorString( retval ));
	}
        break;

    case IFSFN_FILELOCKS:
	FilemonGetFullPath( pir->ir_sfn, fullpathname );
        if( FilemonGetProcess(processname) && ApplyNameFilter( fullpathname )) {
            sprintf(data, "Offset: %ld Length:%ld", origir.ir_pos, origir.ir_locklen );
            UpdateStore( Sequence++, "%s\t%s\t%s\t%s\t%s", 
		 processname, origir.ir_flags == LOCK_REGION ? "Lock" : "Unlock",
		 fullpathname,
		 data, ErrorString( retval ));
	}
        break;

    case IFSFN_FINDOPEN:

	FilemonConvertPath( Drive, pir->ir_ppath, fullpathname ); 
        if( FilemonGetProcess(processname) && ApplyNameFilter( fullpathname )) {	
            if( !retval ) {

                finddata = (_WIN32_FIND_DATA *) pir->ir_data;
                UniToBCS( data, finddata->cFileName, 256, wstrlen(finddata->cFileName),BCS_WANSI, &result );
                data[wstrlen(finddata->cFileName)/2] = 0;
            }
            UpdateStore( Sequence++, "%s\tFindOpen\t%s\t%s\t%s", 
		 processname, fullpathname,
		 data, ErrorString( retval ));
	}
        FilemonStoreHash( pir->ir_sfn, fullpathname );
        break;

    case IFSFN_FINDNEXT:

	FilemonGetFullPath( pir->ir_sfn, fullpathname );
        if( FilemonGetProcess(processname) && ApplyNameFilter( fullpathname )) {	
            if( !retval ) {
                finddata = (_WIN32_FIND_DATA *) pir->ir_data;
                UniToBCS( data, finddata->cFileName, 256, wstrlen(finddata->cFileName),BCS_WANSI, &result );
                data[wstrlen(finddata->cFileName)/2] = 0;
            }

            UpdateStore( Sequence++, "%s\tFindNext\t%s\t%s\t%s", 
		 processname, fullpathname,
		 data, ErrorString( retval ));
	}
        break;

    case IFSFN_FINDCLOSE:
	FilemonGetFullPath( pir->ir_sfn, fullpathname );
        if( FilemonGetProcess(processname) && ApplyNameFilter( fullpathname )) {	
            UpdateStore( Sequence++, "%s\tFindClose\t%s\t\t%s", 
		 processname, fullpathname,
		 ErrorString( retval ));
	}
        FilemonFreeHashEntry( pir->ir_sfn );
        break;

    case IFSFN_FILEATTRIB:
	FilemonConvertPath( Drive, pir->ir_ppath, fullpathname );
        if( FilemonGetProcess(processname) && ApplyNameFilter( fullpathname )) {    
            switch(origir.ir_flags ) {
            case GET_ATTRIBUTES:
                sprintf(data,"GetAttributes");
                break;
            case SET_ATTRIBUTES:
                sprintf(data, "SetAttributes" );
                break;
            case GET_ATTRIB_COMP_FILESIZE:
                sprintf(data, "GET_ATTRIB_COMP_FILESIZE" );
                break;
            case SET_ATTRIB_MODIFY_DATETIME:
                sprintf(data, "SET_ATTRIB_MODIFY_DATETIME");
                break;
            case SET_ATTRIB_LAST_ACCESS_DATETIME:
                sprintf(data, "SET_ATTRIB_LAST_ACCESS_DATETIME");
                break;
            case GET_ATTRIB_LAST_ACCESS_DATETIME:
                sprintf(data, "GET_ATTRIB_LAST_ACCESS_DATETIME");
                break;
            case SET_ATTRIB_CREATION_DATETIME:
                sprintf(data, "SET_ATTRIB_CREATION_DATETIME");
                break;
            case GET_ATTRIB_CREATION_DATETIME:
                sprintf(data, "GET_ATTRIB_CREATION_DATETIME");
                break;
            }
            UpdateStore( Sequence++, "%s\tAttributes\t%s\t%s\t%s", 
		 processname, fullpathname,
		 data, ErrorString( retval ));
	}
        break;

    case IFSFN_FILETIMES:
	FilemonGetFullPath( pir->ir_sfn, fullpathname );
        if( FilemonGetProcess(processname) && ApplyNameFilter( fullpathname )) {	
            switch( pir->ir_flags ) {
            case GET_MODIFY_DATETIME:
                sprintf(data, "Get Modify");
                break;
            case SET_MODIFY_DATETIME:
                sprintf(data, "Set Modify");
                break;
            case GET_LAST_ACCESS_DATETIME:
                sprintf(data, "Get Access");
                break;
            case SET_LAST_ACCESS_DATETIME:
                sprintf(data, "Set Access");
                break;
            case GET_CREATION_DATETIME:
                sprintf(data, "Get Creation");
                break;
            case SET_CREATION_DATETIME:
                sprintf(data, "Set Creation");
                break;
            }
            UpdateStore( Sequence++, "%s\tAttributes\t%s\t%s\t%s", 
		 processname, fullpathname,
		 data, ErrorString( retval ));
	}
        break;

    case IFSFN_FLUSH:
        if( FilemonGetProcess(processname) ) {
            UpdateStore( Sequence++, "%s\tFlushVolume\t\t\t%s",
		 processname, ErrorString( retval ));
	}
        break;

    case IFSFN_DELETE:
	FilemonConvertPath( Drive, pir->ir_ppath, fullpathname );
        if( FilemonGetProcess(processname) && ApplyNameFilter( fullpathname )) {    
            UpdateStore( Sequence++, "%s\tDelete\t%s\t\t%s", 
		 processname, fullpathname,
		 ErrorString( retval ));
	}
        FilemonFreeHashEntry( pir->ir_sfn );
        break;

    case IFSFN_SEARCH:
        if( pir->ir_flags == SEARCH_FIRST ) 
            FilemonConvertPath( Drive, pir->ir_ppath, fullpathname );
        else
            sprintf(fullpathname, "SearchNext" );
        if( FilemonGetProcess(processname) && ApplyNameFilter( fullpathname ) ) {
            if( !retval ) {
                j = 0;
                if( origir.ir_attr & FILE_ATTRIBUTE_LABEL ) {
                     sprintf(data, "VolumeLabel: " );
                     j = strlen( data );
                }
                search = (struct srch_entry *) origir.ir_data;
                for( i = 0; i < 13; i++ ) 
                     if( search->se_name[i] != ' ' ) data[j++] = search->se_name[i];
                data[j] = 0;
             }
             UpdateStore( Sequence++, "%s\tSearch\t%s\t%s\t%s", 
		 processname, fullpathname, data, ErrorString( retval ));    
	}
        break;
    
    case IFSFN_GETDISKINFO:

        if( FilemonGetProcess( processname ) ) {
            if( !retval ) sprintf(data, "Free Space: %ld bytes", 
			  pir->ir_length * pir->ir_numfree );
            drivestring[0] = Drive+'A'-1;
            drivestring[1] = ':';
            drivestring[2] = 0;
            UpdateStore( Sequence++, "%s\tGetDiskInfo\t%s\t%s\t%s",
		 processname, drivestring, data, ErrorString( retval ));
	}
        break;

    case IFSFN_RENAME:
	FilemonConvertPath( Drive, pir->ir_ppath, fullpathname );
        if( FilemonGetProcess(processname) && ApplyNameFilter( fullpathname )) {          
            UpdateStore( Sequence++, "%s\tRename\t%s\t%s\t%s",
		 processname, fullpathname,
		 FilemonConvertPath( Drive, pir->ir_ppath2, data ),
		 ErrorString( retval ));		 
	}
        break;
    }
    dprintf("==>%d\n", fn );
    return retval;
}


//----------------------------------------------------------------------
//
// OnSysDynamicDeviceInit
//
// Dynamic init. Hook all registry related VxD APIs.
//
//----------------------------------------------------------------------
BOOL OnSysDynamicDeviceInit()
{
    int i;

    //
    // Initialize the locks.
    //
    StoreMutex = Create_Semaphore(1);
    HashMutex  = Create_Semaphore(1);

    // 
    // Zero hash table.
    //
    for(i = 0; i < NUMHASH; i++ ) HashTable[i] = NULL;

    //
    // Allocate the initial output buffer.
    //
    PageAllocate(STORESIZE, PG_SYS, 0, 0, 0, 0, NULL, PAGELOCKED, 
	       (PMEMHANDLE) &Store, (PVOID) &Store );
    Store->Len = 0;
    Store->Next = NULL;
    NumStore = 1;

    //
    // Hook IFS functions.
    //
    PrevIFSHookProc = IFSMgr_InstallFileSystemApiHook(FilemonHookProc);

    return TRUE;
}

//----------------------------------------------------------------------
//
// OnSysDynamicDeviceExit
//
// Dynamic exit. Unhook everything.
//
//----------------------------------------------------------------------
BOOL OnSysDynamicDeviceExit()
{
    //
    // Unhook IFS functions.
    //
    IFSMgr_RemoveFileSystemApiHook(FilemonHookProc);

    //
    // Free all memory.
    //
    FilemonHashCleanup();
    FilemonFreeStore();
    return TRUE;
}

//----------------------------------------------------------------------
//
// OnW32Deviceiocontrol
//
// Interface with the GUI.
//
//----------------------------------------------------------------------
DWORD OnW32Deviceiocontrol(PIOCTLPARAMS p)
{
    PSTORE_BUF      old;

    switch( p->dioc_IOCtlCode ) {
    case 0:
        return 0;

    case FILEMON_zerostats:

        Wait_Semaphore( StoreMutex, BLOCK_SVC_INTS );
        while ( Store->Next )  {
 
            //
            // Release the next entry.
            //
            old = Store->Next;
            Store->Next = old->Next;
            Signal_Semaphore( StoreMutex );
            PageFree( (MEMHANDLE) old, 0 );
            Wait_Semaphore( StoreMutex, BLOCK_SVC_INTS );
            NumStore--;
        }
        Store->Len = 0;
        Signal_Semaphore( StoreMutex );
        Sequence = 0;
        return 0;

    case FILEMON_getstats:

        //
        // Copy buffer into user space.
        Wait_Semaphore( StoreMutex, BLOCK_SVC_INTS );
        if ( MAX_STORE > p->dioc_cbOutBuf ) {

           //
           // Buffer is too small. Return error.
           //
           Signal_Semaphore( StoreMutex );

           return 1;

        } else if ( Store->Len  ||  Store->Next ) {

           //
           // Switch to a new buffer.
           //
           FilemonNewStore();

           //
           // Fetch the oldest buffer to give to caller.
           //
           old = FilemonOldestStore();
           Signal_Semaphore( StoreMutex );

           //
           // Copy it into the caller's buffer.
           //
           memcpy( p->dioc_OutBuf, old->Data, old->Len );

           //
           // Return length of copied info.
           //
           *p->dioc_bytesret = old->Len;

           //   
           // Deallocate the buffer.
           //
           PageFree( (MEMHANDLE) old, 0 );
           NumStore--;

       } else {

           //
           // There is no unread data.
           //
           Signal_Semaphore( StoreMutex );
           *p->dioc_bytesret = 0;
       }
       return 0;

    case FILEMON_stopfilter:

       FilterOn = FALSE;
       return 0;

    case FILEMON_startfilter:

       FilterOn = TRUE;
       return 0;

    case FILEMON_setfilter:

       FilterDef = * (PFILTER) p->dioc_InBuf;
       return 0;

    default:
       return 1;
    }
}
