open Py_types
open Py_pretok

let lookup_path = [
  "."; 
  "/usr/local/lib/python1.5"; 
  "/mnt/user2/work/py/iscr"
];;

Gtk.gtk_init();;

let sys = Py_sys.mk_sys lookup_path;;

let interpreter = new Py_interp.py_interpreter sys;;
let version = match sys#get_dictionary#get_item (PyString "version")
with Some x -> x | None -> (PyString "Unknown!");;

let copyright = match sys#get_dictionary#get_item (PyString "copyright")
with Some x -> x | None -> (PyString "Unknown");;

print_endline ("Viperi " ^ (Py_functions.str version ));;
print_string ("Copyright " ^ (Py_functions.str copyright));;
print_newline ();;

let top_level_handler x = 
  try
    begin match x with
    | Py_exceptions.PyException y ->
      print_endline ""; 
      print_endline "Uncaught Python Exception at top level!!";

      begin match y with
      | PyStringException (s1, s2) -> 
        print_endline "  .. Kind: String";
        print_endline ("  .. Arguments: " ^ s1 ^ ": " ^ (Py_functions.str s2));
        interpreter#print_tb      

      | PyInstanceException i ->
        print_string "  .. Kind: Instance of ";
        print_endline i#get_class#get_name;
        let d = i#get_dictionary in
        if d#len > 0 then begin
          print_endline "  .. Attributes:";
          i#get_dictionary#iter
          begin fun k v ->
            print_string "      ";
            print_string (Py_functions.str k);
            print_string " --> ";
            match v with
            | PyTuple x ->
              for i=0 to (List.length x) - 1 do
                print_endline (Py_functions.str (List.nth x i))
              done
            | _ -> print_string (Py_functions.repr v);
            print_endline ""
          end;
          print_endline ""
        end;
        interpreter#print_exc_tb
      end (* match y *)

    | _ -> 
      print_endline "Unexpected System Exception at toplevel";
      print_endline (Printexc.to_string x)
    end (* match x *);
    flush stdout
  with _ -> 
    print_endline "DOUBLE FAULT DURING TOP LEVEL EXCEPTION HANDLING";
    flush stdout; 
    exit 1
;;

let interactive_input () =
  let ps1 = 
    match sys#get_attr (PyString "ps1") with
    | Some (PyString x) -> x
    | _ -> ">>>"
  and ps2 = 
    match sys#get_attr (PyString "ps2") with
    | Some (PyString x) -> x
    | _ -> "..."
  in
  let src = ref "" 
  and line = ref (Gnu_readline.readline(ps1)) 
  in while (String.length !line) > 1 do
    src := !src ^ !line;
    line := Gnu_readline.readline(ps2)
  done;
  !src
;;

exception Done;;
let interactive_process () =
  try 
    while true do
      try 
        let text = interactive_input() 
        in
          if String.length text = 0
          then raise Done
          else ignore (interpreter#run_string text)
      with 
        | Done -> raise Done
        | x -> top_level_handler x
    done
    ;
    0 (* dummy *)
  with Done -> 
    flush stdout; 
    0

let batch_process filename =
  try interpreter#run_file filename
  with 
  | Py_exceptions.PyException (PyInstanceException i) as x ->
    if i#get_class#get_name = "SystemExit"
    then match i#get_attr (PyString "code") with
    | Some (PyInt i) -> 
      print_endline ("SystemExit" ^ (string_of_int i)); 
      i
    | _ -> 
      print_endline "SystemExit exception has no code attribute (returning -1)!";
      flush stdout;
      -1
    else begin (* instance exception other than system exit *)
      top_level_handler x;
      raise x;
      -1 (* dummy *)
    end
  | x -> (* any exception other than an instance *)
    top_level_handler x;
    raise x;
    -1 (* dummy *)

let return_code = 
  if Array.length (Sys.argv) <= 1
  then interactive_process ()
  else batch_process (Sys.argv.(1))

let _ = print_endline ("DONE: returning code " ^ (string_of_int return_code));;

(* exit return_code ;; *)

(* we should now drop thru into GTK window! *)

